"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbPostInstallSetup = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../lambdas/lambdaLayerVersionArns");
/**
 * This construct performs post-installation setup on a MongoDB database by logging into the database, and
 * executing commands against it. To provide this functionality, this construct will create an AWS Lambda function
 * that is granted the ability to connect to the given MongoDB using its administrator credentials. This lambda
 * is run automatically when you deploy or update the stack containing this construct. Logs for all AWS Lambdas are
 * automatically recorded in Amazon CloudWatch.
 *
 * Presently, the only post-installation action that this construct can perform is creating users. There are two types
 * of users that it can create:
 * 1. Password-authenticated users -- these users will be created within the 'admin' database.
 * 2. X.509-authenticated users -- these users will be created within the '$external' database.
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the MongoDB application, and perform post-installation tasks.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 *
 * @ResourcesDeployed
 */
class MongoDbPostInstallSetup extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        (_a = props.users.x509AuthUsers) === null || _a === void 0 ? void 0 : _a.forEach(user => {
            try {
                JSON.parse(user.roles);
            }
            catch (e) {
                throw new Error(`MongoDbPostInstallSetup: Could not parse JSON role for x509 user: ${user.roles}`);
            }
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lamdbaFunc = new aws_lambda_1.Function(this, 'Lambda', {
            vpc: props.vpc,
            vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            description: `Used by a MongoDbPostInstallSetup ${this.node.uniqueId} to perform post-installation setup on a MongoDB`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambdas', 'nodejs'), {
            // Exclude commented out, for now, as a work-around for a CDK bug with at least CDK v1.49.1.
            // If we exclude files, then the asset hash is not calculated correctly and can result in updates to these
            // files not being picked up by the live system.
            // exclude: [
            //   '**/*',
            //   '!mongodb', '!mongodb/*',
            //   '!lib',
            //   '!lib/custom-resource', '!lib/custom-resource/*',
            //   '!lib/aws-lambda', '!lib/aws-lambda/*',
            //   '!lib/secrets-manager', '!lib/secrets-manager/*',
            //   '**/test',
            // ],
            }),
            environment: {
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'mongodb.configureMongo',
            layers: [openSslLayer],
            timeout: core_1.Duration.minutes(2),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        lamdbaFunc.connections.allowTo(props.mongoDb, aws_ec2_1.Port.tcp(props.mongoDb.port));
        props.mongoDb.certificateChain.grantRead(lamdbaFunc.grantPrincipal);
        props.mongoDb.adminUser.grantRead(lamdbaFunc.grantPrincipal);
        (_c = props.users.passwordAuthUsers) === null || _c === void 0 ? void 0 : _c.forEach(secret => secret.grantRead(lamdbaFunc));
        (_d = props.users.x509AuthUsers) === null || _d === void 0 ? void 0 : _d.forEach(user => user.certificate.grantRead(lamdbaFunc));
        const properties = {
            Connection: {
                Hostname: props.mongoDb.fullHostname,
                Port: props.mongoDb.port.toString(),
                CaCertificate: props.mongoDb.certificateChain.secretArn,
                Credentials: props.mongoDb.adminUser.secretArn,
            },
            PasswordAuthUsers: (_e = props.users.passwordAuthUsers) === null || _e === void 0 ? void 0 : _e.map(secret => secret.secretArn),
            X509AuthUsers: (_f = props.users.x509AuthUsers) === null || _f === void 0 ? void 0 : _f.map(user => ({ Certificate: user.certificate.secretArn, Roles: user.roles })),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lamdbaFunc.functionArn,
            resourceType: 'Custom::RFDK_MongoDbPostInstallSetup',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(lamdbaFunc.role);
        /* istanbul ignore next */
        if (props.mongoDb.node.defaultChild) {
            // Add a dependency on the ASG within the StaticPrivateIpServer to ensure that
            // mongo is running before we try to login to it.
            resource.node.addDependency(props.mongoDb.node.defaultChild.node.defaultChild);
        }
        this.node.defaultChild = resource;
    }
}
exports.MongoDbPostInstallSetup = MongoDbPostInstallSetup;
//# sourceMappingURL=data:application/json;base64,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