/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { IVpc, SubnetSelection } from '@aws-cdk/aws-ec2';
import { ISecret } from '@aws-cdk/aws-secretsmanager';
import { Construct } from '@aws-cdk/core';
import { IMongoDb } from '.';
/**
 * User added to the $external admin database.
 * Referencing: https://docs.mongodb.com/v3.6/core/security-x.509/#member-certificate-requirements
 */
export interface IMongoDbX509User {
    /**
     * The certificate of the user that they will use for authentication. This must be a secret
     * containing the plaintext string contents of the certificate in PEM format. For example,
     * the cert property of {@link IX509CertificatePem} is compatible with this.
     *
     * Some important notes:
     * 1. MongoDB **requires** that this username differ from the MongoDB server certificate
     * in at least one of: Organization (O), Organizational Unit (OU), or Domain Component (DC).
     * See: https://docs.mongodb.com/manual/tutorial/configure-x509-client-authentication/
     *
     * 2. The client certificate must be signed by the same Certificate Authority (CA) as the
     * server certificate that is being used by the MongoDB application.
     */
    readonly certificate: ISecret;
    /**
     * JSON-encoded string with the roles this user should be given.
     */
    readonly roles: string;
}
/**
 * This interface is for defining a set of users to pass to MongoDbPostInstallSetup so that it can
 * create them in the MongoDB.
 */
export interface IMongoDbUsers {
    /**
     * Zero or more secrets containing credentials, and specification for users to be created in the
     * admin database for authentication using SCRAM. See: https://docs.mongodb.com/v3.6/core/security-scram/
     *
     * Each secret must be a JSON document with the following structure:
     *     {
     *         "username": <username of the user>,
     *         "password": <password of the user>,
     *         "roles": <a list of roles that the user is being given>
     *     }
     *
     * For examples of the roles list, see the MongoDB user creation documentation. For example,
     * https://docs.mongodb.com/manual/tutorial/create-users/
     *
     * @default No password-authenticated users are created.
     */
    readonly passwordAuthUsers?: ISecret[];
    /**
     * Information on the X.509-authenticated users that should be created.
     * See: https://docs.mongodb.com/v3.6/core/security-x.509/
     *
     * @default No x.509 authenticated users are created.
     */
    readonly x509AuthUsers?: IMongoDbX509User[];
}
/**
 * Input properties for MongoDbPostInstallSetup.
 */
export interface MongoDbPostInstallSetupProps {
    /**
     * The VPC in which to create the network endpoint for the lambda function that is
     * created by this construct.
     */
    readonly vpc: IVpc;
    /**
     * Where within the VPC to place the lambda function's endpoint.
     *
     * @default The instance is placed within a Private subnet.
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * The MongoDB that we will connect to to perform the post-installation steps upon.
     */
    readonly mongoDb: IMongoDb;
    /**
     * The Users that should be created in the MongoDB database. This construct will create these
     * users only if they do not already exist. If a user does already exist, then it will not be modified.
     */
    readonly users: IMongoDbUsers;
}
/**
 * This construct performs post-installation setup on a MongoDB database by logging into the database, and
 * executing commands against it. To provide this functionality, this construct will create an AWS Lambda function
 * that is granted the ability to connect to the given MongoDB using its administrator credentials. This lambda
 * is run automatically when you deploy or update the stack containing this construct. Logs for all AWS Lambdas are
 * automatically recorded in Amazon CloudWatch.
 *
 * Presently, the only post-installation action that this construct can perform is creating users. There are two types
 * of users that it can create:
 * 1. Password-authenticated users -- these users will be created within the 'admin' database.
 * 2. X.509-authenticated users -- these users will be created within the '$external' database.
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the MongoDB application, and perform post-installation tasks.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 *
 * @ResourcesDeployed
 */
export declare class MongoDbPostInstallSetup extends Construct {
    constructor(scope: Construct, id: string, props: MongoDbPostInstallSetupProps);
}
