"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstance = void 0;
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const _1 = require("./");
/**
 * This construct provides a {@link StaticPrivateIpServer} that is hosting MongoDB. The data for this MongoDB database
 * is stored in an Amazon Elastic Block Storage (EBS) Volume that is automatically attached to the instance when it is
 * launched, and is separate from the instance's root volume; it is recommended that you set up a backup schedule for
 * this volume.
 *
 * When this instance is first launched, or relaunched after an instance replacement, it will:
 * 1. Attach an EBS volume to /var/lib/mongo upon which the MongoDB data is stored;
 * 2. Automatically install the specified version of MongoDB, from the official Mongo Inc. sources;
 * 3. Create an admin user in that database if one has not yet been created -- the credentials for this user
 * can be provided by you, or randomly generated;
 * 4. Configure MongoDB to require authentication, and only allow encrypted connections over TLS.
 *
 * The instance's launch logs and MongoDB logs will be automatically stored in Amazon CloudWatch logs; the
 * default log group name is: /renderfarm/<this construct ID>
 *
 * Resources Deployed
 * ------------------------
 * - {@link StaticPrivateIpServer} that hosts MongoDB.
 * - An A Record in the provided PrivateHostedZone to create a DNS entry for this server's static private IP.
 * - A Secret containing the administrator credentials for MongoDB.
 * - An encrypted EBS Volume on which the MongoDB data is stored.
 * - Amazon CloudWatch log group that contains instance-launch and MongoDB application logs.
 *
 * @ResourcesDeployed
 */
class MongoDbInstance extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.version = props.mongoDb.version;
        // Select the subnet for this instance.
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        this.server = new _1.StaticPrivateIpServer(this, 'Server', {
            vpc: props.vpc,
            vpcSubnets: { subnets: [subnet] },
            instanceType: (_a = props.instanceType) !== null && _a !== void 0 ? _a : new aws_ec2_1.InstanceType('r5.large'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            blockDevices: [
                {
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(MongoDbInstance.ROOT_DEVICE_SIZE.toGibibytes(), { encrypted: true }),
                },
            ],
            keyName: props.keyName,
            resourceSignalTimeout: core_1.Duration.minutes(5),
            role: props.role,
            securityGroup: props.securityGroup,
        });
        new aws_route53_1.ARecord(this, 'ARecord', {
            target: aws_route53_1.RecordTarget.fromIpAddresses(this.server.privateIpAddress),
            zone: props.mongoDb.dnsZone,
            recordName: props.mongoDb.hostname,
        });
        this.adminUser = (_b = props.mongoDb.adminUser) !== null && _b !== void 0 ? _b : new aws_secretsmanager_1.Secret(this, 'AdminUser', {
            description: `Admin credentials for the MongoDB database ${this.node.uniqueId}`,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        this.mongoDataVolume = (_d = (_c = props.mongoDb.mongoDataVolume) === null || _c === void 0 ? void 0 : _c.volume) !== null && _d !== void 0 ? _d : new aws_ec2_1.Volume(this, 'MongoDbData', {
            size: MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE,
            ...(_e = props.mongoDb.mongoDataVolume) === null || _e === void 0 ? void 0 : _e.volumeProps,
            availabilityZone: subnet.availabilityZone,
            encrypted: true,
        });
        const volumeMount = new _1.MountableBlockVolume(this, {
            blockVolume: this.mongoDataVolume,
            volumeFormat: _1.BlockVolumeFormat.XFS,
        });
        const mongoInstaller = new _1.MongoDbInstaller(this, {
            version: props.mongoDb.version,
            userSsplAcceptance: props.mongoDb.userSsplAcceptance,
        });
        // Set up the server's UserData.
        this.server.userData.addCommands('set -xefuo pipefail');
        this.server.userData.addSignalOnExitCommand(this.server.autoscalingGroup);
        this.configureCloudWatchLogStreams(this.server, id, props.logGroupProps); // MUST BE FIRST
        volumeMount.mountToLinuxInstance(this.server, {
            location: MongoDbInstance.MONGO_DEVICE_MOUNT_POINT,
        });
        mongoInstaller.installOnLinuxInstance(this.server);
        this.configureMongoDb(this.server, props.mongoDb);
        this.certificateChain = props.mongoDb.serverCertificate.certChain;
        this.connections = this.server.connections;
        this.grantPrincipal = this.server.grantPrincipal;
        this.port = 27017;
        this.role = this.server.role;
        this.userData = this.server.userData;
        this.fullHostname = `${props.mongoDb.hostname}.${props.mongoDb.dnsZone.zoneName}`;
        this.node.defaultChild = this.server;
    }
    /**
     * Adds UserData commands to install & configure the CloudWatch Agent onto the instance.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *     - The cloud-init log
     *     - The MongoDB application log.
     *
     * @param host The instance/host to setup the CloudWatchAgent upon.
     * @param groupName Name to append to the log group prefix when forming the log group name.
     * @param logGroupProps Properties for the log group
     */
    configureCloudWatchLogStreams(host, groupName, logGroupProps) {
        var _a;
        const prefix = (_a = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _a !== void 0 ? _a : MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = _1.LogGroupFactory.createOrFetch(this, 'MongoDbInstanceLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(host.grantPrincipal);
        const cloudWatchConfigurationBuilder = new _1.CloudWatchConfigBuilder(MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'MongoDB', '/var/log/mongodb/mongod.log');
        new _1.CloudWatchAgent(this, 'MongoDbInstanceLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host,
        });
    }
    /**
     * Adds commands to the userData of the instance to install MongoDB, create an admin user if one does not exist, and
     * to to start mongod running.
     */
    configureMongoDb(instance, settings) {
        const scriptsAsset = new aws_s3_assets_1.Asset(this, 'MongoSetup', {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', settings.version),
        });
        scriptsAsset.grantRead(instance.grantPrincipal);
        const scriptZipfile = instance.userData.addS3DownloadCommand({
            bucket: scriptsAsset.bucket,
            bucketKey: scriptsAsset.s3ObjectKey,
        });
        instance.userData.addCommands(
        // Ensure mongod is installed and stopped before we go any further
        'which mongod && test -f /etc/mongod.conf', 'sudo service mongod stop', 
        // We're going to make a temporary RAM filesystem for the mongo setup files.
        // This will let us write sensitive data to "disk" without worrying about it
        // being persisted in any physical disk, even temporarily.
        'MONGO_SETUP_DIR=$(mktemp -d)', 'mkdir -p "${MONGO_SETUP_DIR}"', 'sudo mount -t tmpfs -o size=50M tmpfs "${MONGO_SETUP_DIR}"', 'pushd "${MONGO_SETUP_DIR}"', `unzip ${scriptZipfile}`, 
        // Backup mongod.conf for now
        'cp /etc/mongod.conf .');
        const cert = settings.serverCertificate;
        instance.userData.addCommands(`bash serverCertFromSecrets.sh "${cert.cert.secretArn}" "${cert.certChain.secretArn}" "${cert.key.secretArn}" "${cert.passphrase.secretArn}"`);
        cert.cert.grantRead(instance.grantPrincipal);
        cert.certChain.grantRead(instance.grantPrincipal);
        cert.key.grantRead(instance.grantPrincipal);
        cert.passphrase.grantRead(instance.grantPrincipal);
        const certsDirectory = '/etc/mongod_certs';
        instance.userData.addCommands(
        // Move the certificates into place
        `sudo mkdir -p ${certsDirectory}`, `sudo mv ./ca.crt ./key.pem ${certsDirectory}`, 'sudo chown root.mongod -R /etc/mongod_certs/', // Something weird about shell interpretation. Can't use '*' on this or next line.
        'sudo chmod 640 -R /etc/mongod_certs/', 'sudo chmod 750 /etc/mongod_certs/', // Directory needs to be executable.
        // mongod user id might, potentially change on reboot. Make sure we own all mongo data
        `sudo chown mongod.mongod -R ${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}`, 
        // Configure mongod
        'bash ./setMongoLimits.sh', `bash ./setStoragePath.sh "${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}"`, 'bash ./setMongoNoAuth.sh', 'sudo service mongod start', `bash ./setAdminCredentials.sh "${this.adminUser.secretArn}"`);
        this.adminUser.grantRead(instance.grantPrincipal);
        instance.userData.addCommands(
        // Setup for live deployment, and start mongod
        'sudo service mongod stop', 'bash ./setLiveConfiguration.sh', 'sudo systemctl enable mongod', // Enable restart on reboot
        'sudo service mongod start', 'popd');
        instance.userData.addOnExitCommands(
        // Clean up the temporary RAM filesystem
        'test "${MONGO_SETUP_DIR} != "" && sudo umount "${MONGO_SETUP_DIR}');
    }
}
exports.MongoDbInstance = MongoDbInstance;
// How often Cloudwatch logs will be flushed.
MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
// Default prefix for a LogGroup if one isn't provided in the props.
MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
// Size of the EBS volume for MongoDB data, if we create one.
MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE = core_1.Size.gibibytes(20);
// Mount point for the MongoDB data volume.
MongoDbInstance.MONGO_DEVICE_MOUNT_POINT = '/var/lib/mongo';
// Size of the root device volume on the instance.
MongoDbInstance.ROOT_DEVICE_SIZE = core_1.Size.gibibytes(10);
//# sourceMappingURL=data:application/json;base64,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