"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AWSLimitExhaustedError = exports.LoadBalancerFactory = void 0;
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const health_monitor_1 = require("./health-monitor");
/**
 * This class is responsible for managing the statistics for all the
 * load balancers created in this construct. It is also responsible to search
 * for the finding the first Load balancer/Listener which can accomodate the
 * worker-fleet based on its size.
 *
 * A typical load balancer hierarchy looks like following:
 *  |__ Load Balancer 1
 *  |         |____________Listener 1
 *  |         |                 |_______Target Group 1 ------- Target/Fleet
 *  |         |                 |_______Target Group 2 ------- Target/Fleet
 *  |         |
 *  |         |____________Listener 2
 *  |                           |_______Target Group 1 ------- Target/Fleet
 *  |                           |_______Target Group 2 ------- Target/Fleet
 *  |
 *  |__ Load Balancer 2
 *            |____________Listener 1
 *            |                 |_______Target Group 1 ------- Target/Fleet
 *            |                 |_______Target Group 2 ------- Target/Fleet
 *            |
 *            |____________Listener 2
 *                              |_______Target Group 1 ------- Target/Fleet
 *                              |_______Target Group 2 ------- Target/Fleet
 *
 *  Components:
 *  1. LoadBalancerFactory: This is the root node of the tree. It contains the
 *     map of load balancer to its managers. It is responsible for creating a
 *     new load balancer if required. It delegates the registerFleet calls to
 *     downstream and returns parent load balancer, listener and target group
 *     of the registered fleet if the registration was successful
 *
 *  2. LoadBalancerManager: This class manages a single load balancer. It
 *     contains a map of all the listeners->manager. It also contains the component
 *     counts like listener, target group and target count. It delegates the
 *     registration call to downstream listeners and updates the stats when
 *     the registration is successful. It returns the parent listener and
 *     target group on successful registration.
 *
 *  3. ListenerManager: This class managers a single Listener. It contains a map
 *     of all of its target groups to its associated fleet. It also contains the
 *     component counts. It returns the target group on registration.
 */
class LoadBalancerFactory {
    constructor(healthMonitorScope, vpc) {
        this.loadBalancerMap = new Map();
        this.healthMonitorScope = healthMonitorScope;
        this.vpc = vpc;
    }
    static getAccountLimit(limitName, defaultValue, elbAccountLimits) {
        if (!elbAccountLimits) {
            return defaultValue;
        }
        const foundLimit = elbAccountLimits.find(limit => limit.name === limitName);
        if (!foundLimit) {
            return defaultValue;
        }
        return foundLimit.max;
    }
    /**
     * This method scans all the load balancers and its listeners and registers the fleet
     * to the load balancer and/or listener which can accommodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the load balancer, listener
     * and target group to which the fleet was registered.
     *
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(fleet, healthCheckConfig, healthMonitorProps) {
        let loadBalancerParent = null;
        let listenerParent = null;
        let targetGroupParent = null;
        // iterate through each load balancer and try registering to each one.
        for (const [loadBalancer, loadBalancerMeta] of this.loadBalancerMap.entries()) {
            try {
                const { listener, targetGroup } = loadBalancerMeta.registerWorkerFleet(loadBalancer, fleet, healthCheckConfig, healthMonitorProps);
                loadBalancerParent = loadBalancer;
                listenerParent = listener;
                targetGroupParent = targetGroup;
                break;
            }
            catch (e) {
                // suppress all AWSLimitExhaustedError, we will scale in case of this error
                /* istanbul ignore next */
                if (!(e instanceof AWSLimitExhaustedError)) {
                    /* istanbul ignore next */
                    throw e;
                }
            }
        }
        // Check if fleet was not registered.
        if (!loadBalancerParent) {
            // If this section is reached, no load balancer was found which could
            // accommodate fleet, create a new one and register
            loadBalancerParent = this.createLoadBalancer(this.healthMonitorScope, this.loadBalancerMap.size, healthMonitorProps);
            const loadBalancerManager = new LoadBalancerManager();
            // Add it to the map
            this.loadBalancerMap.set(loadBalancerParent, loadBalancerManager);
            // try registering the fleet to the new load balancer
            try {
                const { listener, targetGroup } = loadBalancerManager.registerWorkerFleet(loadBalancerParent, fleet, healthCheckConfig, healthMonitorProps);
                listenerParent = listener;
                targetGroupParent = targetGroup;
            }
            catch (e) {
                throw e;
            }
        }
        /* istanbul ignore next */
        if (!loadBalancerParent || !listenerParent || !targetGroupParent) {
            /* istanbul ignore next */
            throw new Error('Fleet registered successfully but a parent was found null');
        }
        return {
            loadBalancer: loadBalancerParent,
            listener: listenerParent,
            targetGroup: targetGroupParent,
        };
    }
    /**
     * Following method creates a new load balancer within the given scope.
     *
     * @param scope
     * @param loadBalancerindex
     */
    createLoadBalancer(scope, loadBalancerindex, healthMonitorProps) {
        var _a;
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(scope, `ALB_${loadBalancerindex}`, {
            vpc: this.vpc,
            internetFacing: false,
            deletionProtection: (_a = healthMonitorProps.deletionProtection) !== null && _a !== void 0 ? _a : true,
        });
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        return loadBalancer;
    }
}
exports.LoadBalancerFactory = LoadBalancerFactory;
LoadBalancerFactory.DEFAULT_LISTENERS_PER_APPLICATION_LOAD_BALANCER = 50;
LoadBalancerFactory.DEFAULT_TARGETS_PER_APPLICATION_LOAD_BALANCER = 1000;
LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_ACTION_ON_APPLICATION_LOAD_BALANCER = 5;
LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_APPLICATION_LOAD_BALANCER = 100;
/**
 * This class manages the properties of a single load balancer and its statistics.
 * It is also responsible to scan through all the listeners registered under it
 * and register the given fleet.
 */
class LoadBalancerManager {
    constructor() {
        this.listenerMap = new Map();
        this.loadBalancerComponentCount = new LoadBalancerComponentStats();
    }
    /**
     * This method scans all the listeners of this load balancer and registers the fleet
     * to one which can accomodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the listener
     * and target group to which the fleet was registered.
     *
     * @param loadBalancer
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(loadBalancer, fleet, healthCheckConfig, healthMonitorProps) {
        // this initializes with 0 and keeps the track of all components
        // newly added down the hierarchy.
        const statsDelta = new LoadBalancerComponentStats();
        // Do all the load balancer level service limit checks first
        // check for target limit in load balancer
        const targetPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('targets-per-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGETS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.loadBalancerComponentCount.targetCount + fleet.targetCapacity) > targetPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "targets-per-application-load-balancer" reached. Limit: ' +
                targetPerLoadBalancerLimit);
        }
        // check for target group limit in load balancer
        const targetGroupsPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('target-groups-per-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.loadBalancerComponentCount.targetGroupCount + 1) > targetGroupsPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "target-groups-per-application-load-balancer" reached. Limit: ' +
                targetGroupsPerLoadBalancerLimit);
        }
        let listenerParent = null;
        let targetGroupParent = null;
        // try registering to each listener.
        for (const [listener, listenerMeta] of this.listenerMap.entries()) {
            try {
                const { componentsAdded, targetGroup } = listenerMeta.registerWorkerFleet(loadBalancer, listener, fleet, healthCheckConfig, healthMonitorProps);
                statsDelta.add(componentsAdded);
                listenerParent = listener;
                targetGroupParent = targetGroup;
                break;
            }
            catch (e) {
                // suppress all AWSLimitExhaustedError, we will scale in case of this error
                /* istanbul ignore next */
                if (!(e instanceof AWSLimitExhaustedError)) {
                    /* istanbul ignore next */
                    throw e;
                }
            }
        }
        /* istanbul ignore next */
        if (!listenerParent) {
            // If this section is reached, no listener was found which could accommodate fleet
            // create new listener and register
            const listenersPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('listeners-per-application-load-balancer', LoadBalancerFactory.DEFAULT_LISTENERS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
            if ((this.loadBalancerComponentCount.listenerCount + 1) > listenersPerLoadBalancerLimit) {
                throw new AWSLimitExhaustedError('AWS service limit "listeners-per-application-load-balancer" reached. Limit: ' +
                    listenersPerLoadBalancerLimit);
            }
            listenerParent = this.createListener(fleet.targetScope, loadBalancer);
            const listenerManager = new ListenerManager();
            this.listenerMap.set(listenerParent, listenerManager);
            statsDelta.add(new LoadBalancerComponentStats(1, 0, 0));
            try {
                const { componentsAdded, targetGroup } = listenerManager.registerWorkerFleet(loadBalancer, listenerParent, fleet, healthCheckConfig, healthMonitorProps);
                targetGroupParent = targetGroup;
                statsDelta.add(componentsAdded);
            }
            catch (e) {
                throw e;
            }
        }
        // update the current load balancer's stats
        this.loadBalancerComponentCount.add(statsDelta);
        return {
            componentsAdded: statsDelta,
            listener: listenerParent,
            targetGroup: targetGroupParent,
        };
    }
    /**
     * Following method creates a new listener in the fleet's scope and
     * registers it to the given load balancer.
     *
     * @param scope
     * @param loadBalancer
     */
    createListener(scope, loadBalancer) {
        return new aws_elasticloadbalancingv2_1.ApplicationListener(scope, 'Listener', {
            port: health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT + this.listenerMap.size,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            loadBalancer,
            open: false,
        });
    }
}
/**
 * This class manages the properties of a single listener and all the components
 * under its hierarchy.
 * It is also responsible to create a new target group and register the given fleet.
 */
class ListenerManager {
    constructor() {
        this.targetMap = new Map();
        this.listenerComponentCount = new LoadBalancerComponentStats();
    }
    /**
     * This method scans all the listeners of this load balancer and registers the fleet
     * to one which can accommodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the target group
     * to which the fleet was registered.
     *
     * @param loadBalancer
     * @param listener
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(loadBalancer, listener, fleet, healthCheckConfig, healthMonitorProps) {
        const componentsAdded = new LoadBalancerComponentStats();
        // Do all listener level service limit checks
        // check for target limit in listener
        const targetGroupPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('target-groups-per-action-on-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_ACTION_ON_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.listenerComponentCount.targetGroupCount + 1) > targetGroupPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "target-groups-per-action-on-application-load-balancer" reached. Limit: ' +
                targetGroupPerLoadBalancerLimit);
        }
        // latest version of CDK does not support 'forwardConfig' in listener rule yet. This means
        // we cannot add multiple target groups to a single listener. Adding this check till this
        // feature is supported.
        if (this.listenerComponentCount.targetGroupCount > 0) {
            throw new AWSLimitExhaustedError('Unable to add more than 1 Target Group to Listener.');
        }
        // Create a new target group
        const targetGroup = this.createTargetGroup(fleet.targetScope, loadBalancer, listener, fleet, healthCheckConfig);
        this.targetMap.set(targetGroup, fleet);
        // update the listener stats
        componentsAdded.targetGroupCount++;
        componentsAdded.targetCount += fleet.targetCapacity;
        // update the current listener's stats
        this.listenerComponentCount.add(componentsAdded);
        return {
            componentsAdded,
            targetGroup,
        };
    }
    /**
     * Following method creates a new new target group in the fleet's scope and
     * registers it to the given listener.
     *
     * @param scope
     * @param loadBalancer
     * @param listener
     * @param monitorableFleet
     * @param healthCheckConfig
     */
    createTargetGroup(scope, loadBalancer, listener, monitorableFleet, healthCheckConfig) {
        const targetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(scope, 'TargetGroup', {
            port: health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            targets: [monitorableFleet.targetToMonitor],
            healthCheck: {
                port: healthCheckConfig.port ? healthCheckConfig.port.toString() : health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT.toString(),
                interval: healthCheckConfig.interval || health_monitor_1.HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL,
                healthyThresholdCount: healthCheckConfig.instanceHealthyThresholdCount || health_monitor_1.HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD,
                unhealthyThresholdCount: healthCheckConfig.instanceUnhealthyThresholdCount || health_monitor_1.HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD,
                protocol: aws_elasticloadbalancingv2_1.Protocol.HTTP,
            },
            vpc: loadBalancer.vpc,
        });
        listener.addTargetGroups('TargetGroup', {
            targetGroups: [targetGroup],
        });
        return targetGroup;
    }
}
/**
 * This class contains the statistics of all the nested load balancer
 * components like listener count, target group count and target count.
 * This statistics object will be associated with each load balancer
 * and listener for tracking the count of components.
 */
class LoadBalancerComponentStats {
    constructor(listenerCount = 0, targetGroupCount = 0, targetCount = 0) {
        this.listenerCount = listenerCount;
        this.targetGroupCount = targetGroupCount;
        this.targetCount = targetCount;
    }
    add(operand) {
        this.listenerCount += operand.listenerCount;
        this.targetGroupCount += operand.targetGroupCount;
        this.targetCount += operand.targetCount;
    }
}
class AWSLimitExhaustedError extends Error {
    constructor(message) {
        super(message);
    }
}
exports.AWSLimitExhaustedError = AWSLimitExhaustedError;
//# sourceMappingURL=data:application/json;base64,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