"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImportedAcmCertificate = void 0;
const crypto = require("crypto");
const path_1 = require("path");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../lambdas/lambdaLayerVersionArns");
/**
 * A Construct that holds a Custom Resource modelling a certificate that was imported into ACM. It uses a Lambda
 * Function to extract the certificate from Secrets and then import it into ACM. It is intended to be used with the
 * X509CertificatePem Construct.
 *
 * Resources Deployed
 * ------------------------
 * 1) DynamoDB Table - Used for tracking resources created by the CustomResource.
 * 2) Lambda Function, with Role - Used to create/update/delete the CustomResource.
 * 3) ACM Certificate - Created by the CustomResource.
 *
 * @ResourcesDeployed
 */
class ImportedAcmCertificate extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.stack = core_1.Stack.of(this);
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            serverSideEncryption: true,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lambda = new aws_lambda_1.SingletonFunction(this, 'AcmImporter', {
            uuid: ImportedAcmCertificate.IMPORTER_UUID,
            code: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', 'lambdas', 'nodejs')),
            handler: 'x509-certificate.importCert',
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            layers: [openSslLayer],
            retryAttempts: 0,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            timeout: core_1.Duration.seconds(30),
        });
        this.database.grantReadWriteData(lambda);
        this.database.grant(lambda, 'dynamodb:DescribeTable');
        props.cert.grantRead(lambda);
        props.key.grantRead(lambda);
        props.passphrase.grantRead(lambda);
        (_a = props.certChain) === null || _a === void 0 ? void 0 : _a.grantRead(lambda);
        (_b = props.encryptionKey) === null || _b === void 0 ? void 0 : _b.grantDecrypt(lambda);
        const uniqueValue = crypto.createHash('md5').update(this.node.uniqueId).digest('hex');
        this.uniqueTag = new core_1.Tag(`AcmCertImport-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`aws:RequestTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:AddTagsToCertificate',
                'acm:ImportCertificate',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // GetCertificate and DeleteCertificate don't currently support and conditions, so we have to give a broader policy
        // on them for now.
        // See https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html#awscertificatemanager-aws_TagKeys
        // for the condition keys currently available on ACM actions.
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:DeleteCertificate',
                'acm:GetCertificate',
            ],
            resources: ['*'],
        }));
        const properties = {
            X509CertificatePem: {
                Cert: props.cert.secretArn,
                Key: props.key.secretArn,
                Passphrase: props.passphrase.secretArn,
                CertChain: props.certChain ? props.certChain.secretArn : '',
            },
            Tags: [
                {
                    Key: this.uniqueTag.key,
                    Value: this.uniqueTag.value,
                },
                { Key: 'Name',
                    Value: this.uniqueTag.value,
                },
            ],
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lambda.functionArn,
            properties,
            resourceType: 'Custom::RFDK_AcmImportedCertificate',
        });
        this.certificateArn = core_1.Token.asString(resource.getAtt('CertificateArn'));
    }
}
exports.ImportedAcmCertificate = ImportedAcmCertificate;
ImportedAcmCertificate.IMPORTER_UUID = '2d20d8f2-7b84-444e-b738-c75b499a9eaa';
//# sourceMappingURL=data:application/json;base64,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