"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthMonitor = void 0;
const path = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_cloudwatch_actions_1 = require("@aws-cdk/aws-cloudwatch-actions");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
const load_balancer_manager_1 = require("./load-balancer-manager");
/**
 *  A new or imported Health Monitor.
 */
class HealthMonitorBase extends core_1.Construct {
}
/**
 * This construct is responsible for the deep health checks of compute instances.
 * It also replaces unhealthy instances and suspends unhealthy fleets.
 * Although, using this constructs adds up additional costs for monitoring,
 * it is highly recommended using this construct to help avoid / minimize runaway costs for compute instances.
 *
 * An instance is considered to be unhealthy when:
 *   1) Deadline client is not installed on it;
 *   2) Deadline client is installed but not running on it;
 *   3) RCS is not configured correctly for Deadline client;
 *   4) it is unable to connect to RCS due to any infrastructure issues;
 *   5) the health monitor is unable to reach it because of some infrastructure issues.
 *
 * A fleet is considered to be unhealthy when:
 *   1) at least 1 instance is unhealthy for the configured grace period;
 *   2) a percentage of unhealthy instances in the fleet is above a threshold at any given point of time.
 *
 * This internally creates an array of application load balancers and attaches
 * the worker-fleet (which internally is implemented as an Auto Scaling Group) to its listeners.
 * There is no load-balancing traffic on the load balancers,
 * it is only used for health checks.
 * Intention is to use the default properties of laod balancer health
 * checks which does HTTP pings at frequent intervals to all the
 * instances in the fleet and determines its health. If any of the
 * instance is found unhealthy, it is replaced. The target group
 * also publishes the unhealthy target count metric which is used
 * to identify the unhealthy fleet.
 *
 * Other than the default instance level protection, it also creates a lambda
 * which is responsible to set the fleet size to 0 in the event of a fleet
 * being sufficiently unhealthy to warrant termination.
 * This lambda is triggered by CloudWatch alarms via SNS (Simple Notification Service).
 *
 * Resources Deployed
 * ------------------------
 * 1) Application Load Balancer(s) doing frequent pings to the workers;
 * 2) KMS Key to encrypt SNS messages - If no encryption key is provided;
 * 3) SNS topic for all unhealthy fleet notifications;
 * 4) The Alarm if the node is unhealthy for a long period;
 * 5) The Alarm if the healthy host percentage if lower than allowed;
 * 4) A single Lambda function that sets fleet size to 0 when triggered.
 *
 * @ResourcesDeployed
 */
class HealthMonitor extends HealthMonitorBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.props = props;
        this.lbFactory = new load_balancer_manager_1.LoadBalancerFactory(this, props.vpc);
        const topicEncryptKey = props.encryptionKey || new aws_kms_1.Key(this, 'SNSEncryptionKey', {
            description: `This key is used to encrypt SNS messages for ${this.node.uniqueId}.`,
            enableKeyRotation: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            trustAccountIdentities: true,
        });
        // allow cloudwatch service to send encrypted messages
        topicEncryptKey.grant(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'), 'kms:Decrypt', 'kms:GenerateDataKey');
        this.unhealthyFleetActionTopic = new aws_sns_1.Topic(this, 'UnhealthyFleetTopic', {
            masterKey: topicEncryptKey,
        });
        this.unhealthyFleetActionTopic.grantPublish(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'));
        this.alarmTopicAction = new aws_cloudwatch_actions_1.SnsAction(this.unhealthyFleetActionTopic);
        this.unhealthyFleetActionLambda = new aws_lambda_1.SingletonFunction(this, 'UnhealthyFleetAction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../lambdas/nodejs/unhealthyFleetAction')),
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            lambdaPurpose: 'unhealthyFleetTermination',
            timeout: core_1.Duration.seconds(300),
            uuid: '28bccf6a-aa76-478c-9239-e2f5bcc0254c',
        });
        this.unhealthyFleetActionTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.unhealthyFleetActionLambda));
    }
    /**
     * Attaches the load-balancing target to the ELB for instance-level
     * monitoring. The ELB does frequent pings to the workers and determines
     * if a worker node is unhealthy. If so, it replaces the instance.
     *
     * It also creates an Alarm for healthy host percent and suspends the
     * fleet if the given alarm is breaching. It sets the maxCapacity
     * property of the auto-scaling group to 0. This should be
     * reset manually after fixing the issue.
     *
     * @param monitorableFleet
     * @param healthCheckConfig
     */
    registerFleet(monitorableFleet, healthCheckConfig) {
        const { loadBalancer, targetGroup } = this.lbFactory.registerWorkerFleet(monitorableFleet, healthCheckConfig, this.props);
        this.createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup);
    }
    createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup) {
        monitorableFleet.connections.allowFrom(loadBalancer, aws_ec2_1.Port.tcp(healthCheckConfig.port || HealthMonitor.LOAD_BALANCER_LISTENING_PORT));
        const percentMetric = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD,
        });
        // When unhealthy fleet is more than healthyFleetThresholdPercent or 35% at any given period of 5 minutes
        const immediateTerminationAlarm = percentMetric.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetTermination', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: 100 - (healthCheckConfig.healthyFleetThresholdPercent || HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            actionsEnabled: true,
        });
        immediateTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        // When at least one node is unhealthy over a period of 2 hours
        const percentMetricGracePeriod = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE,
        });
        const gracePeriodTerminationAlarm = percentMetricGracePeriod.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetGracePeriod', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            actionsEnabled: true,
        });
        gracePeriodTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        monitorableFleet.targetUpdatePolicy.attachToRole(this.unhealthyFleetActionLambda.role);
    }
}
exports.HealthMonitor = HealthMonitor;
/**
 * Default health check listening port
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_PORT = 63415;
/**
 * Resource Tracker in Deadline currently publish health status every 5 min, hence keeping this same
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(5);
/**
 * Resource Tracker in Deadline currently determines host unhealthy in 15 min, hence keeping this count
 */
HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD = 3;
/**
 * This is the minimum possible value of ALB health-check config, we want to mark worker healthy ASAP
 */
HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD = 2;
/**
 * Since we are not doing any load balancing, this port is just an arbitrary port.
 */
HealthMonitor.LOAD_BALANCER_LISTENING_PORT = 8081;
/**
 * This number is taken from Resource Tracker implementation. If a fleet's healthy percent
 * is less than this threshold at any given point of time, it is suspended.
 */
HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD = 65;
/**
 * This number is taken from Resource Tracker implementation. If a fleet has at least 1
 * unhealthy host for a period of 2 hours, it is suspended.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE = 0;
/**
 * This number is taken from Resource Tracker implementation. We monitor unhealthy fleet for immediate
 * termination for a period fo 5 minutes.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD = core_1.Duration.minutes(5);
/**
 * In Resource Tracker, we evaluate the fleet's health for determining the grace period over a period
 * of 5 minutes. For the first unhealthy signal, a instance can take upto 10min (max), hence we are
 * setting this period to be 15.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE = core_1.Duration.minutes(15);
/**
 * This number is taken from Resource Tracker implementation. Fleet is terminated immediately if it
 * has unhealthy host percent above the hard limit.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD = 1;
/**
 * This number is taken from Resource Tracker implementation. The grace period duration is 2 hours,
 * since the grace period is 15 minutes, we need continuous 8 data points crossing the threshold.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE = 8;
//# sourceMappingURL=data:application/json;base64,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