"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExportingLogGroup = void 0;
const path = require("path");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
/**
 * This construct takes the name of a CloudWatch LogGroup and will either create it if it doesn't already exist,
 * or reuse the existing one. It also creates a regularly scheduled lambda that will export LogEvents to S3
 * before they expire in CloudWatch.
 *
 * It's used for cost-reduction, as it is more economical to archive logs in S3 than CloudWatch when
 * retaining them for more than a week.
 * Note, that it isn't economical to export logs to S3 if you plan on storing them for less than
 * 7 days total (CloudWatch and S3 combined).
 *
 * Resources Deployed
 * ------------------------
 * 1) The Lambda SingletonFunction that checks for the existence of the LogGroup;
 * 2) The CloudWatch LogGroup (if it didn't exist already);
 * 3) The CloudWatch Alarm watching log exportation failures;
 * 4) The CloudWatch Event Rule to schedule log exportation;
 * 5) The Lambda SingletonFunction, with role, to export log groups to S3 by schedule.
 *
 * @ResourcesDeployed
 */
class ExportingLogGroup extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * UUID needed to identify the SingletonFunction for the log exporter.
         */
        this.LOG_EXPORTER_UUID = '6382448c-e4b2-42e9-b14f-a0a9ccdb198e';
        /**
         * Duration of time between export task Lambda runs.
         */
        this.EXPORT_TASK_FREQUENCY = core_1.Duration.hours(1);
        /**
         * Default value for the number of days to retain logs in CloudWatch for.
         */
        this.RETENTION_DEFAULT = aws_logs_1.RetentionDays.THREE_DAYS;
        const retentionInDays = props.retention ? props.retention : this.RETENTION_DEFAULT;
        const exportLogsFunction = this.setupLogsExporter();
        this.exportErrorAlarm = exportLogsFunction.metricErrors().createAlarm(this, 'LogExporterFailureAlarm', {
            evaluationPeriods: 1,
            threshold: 1,
        });
        const logRetention = new aws_lambda_1.LogRetention(this, 'LogRetention', {
            logGroupName: props.logGroupName,
            retention: retentionInDays,
        });
        this.logGroup = aws_logs_1.LogGroup.fromLogGroupArn(scope, `${props.logGroupName}LogGroup`, logRetention.logGroupArn);
        this.logGroup.grant(exportLogsFunction, 'logs:CreateExportTask');
        const scheduledLogExportation = new aws_events_1.Rule(this, 'LogExporterRule', {
            schedule: aws_events_1.Schedule.rate(this.EXPORT_TASK_FREQUENCY),
        });
        scheduledLogExportation.addTarget(new aws_events_targets_1.LambdaFunction(exportLogsFunction, {
            event: aws_events_1.RuleTargetInput.fromObject({
                BucketName: props.bucketName,
                ExportFrequencyInHours: this.EXPORT_TASK_FREQUENCY.toHours(),
                LogGroupName: props.logGroupName,
                RetentionInHours: retentionInDays.valueOf() * 24,
            }),
        }));
    }
    setupLogsExporter() {
        const exportLogsFunction = new aws_lambda_1.SingletonFunction(this, 'LogExporterFunction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../lambdas/nodejs/export-logs')),
            handler: 'index.handler',
            lambdaPurpose: 'LogGroupExporter',
            logRetention: aws_logs_1.RetentionDays.ONE_DAY,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            uuid: this.LOG_EXPORTER_UUID,
        });
        exportLogsFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['logs:DescribeExportTasks'],
            effect: aws_iam_1.Effect.ALLOW,
            resources: ['*'],
        }));
        return exportLogsFunction;
    }
}
exports.ExportingLogGroup = ExportingLogGroup;
//# sourceMappingURL=data:application/json;base64,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