/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { Alarm } from '@aws-cdk/aws-cloudwatch';
import { ILogGroup, RetentionDays } from '@aws-cdk/aws-logs';
import { Construct } from '@aws-cdk/core';
/**
 * Properties for setting up an {@link ExportingLogGroup}.
 */
export interface ExportingLogGroupProps {
    /**
     * The S3 bucket's name to export the logs to. Bucket must already exist and have read/write privilidges enabled for
     * logs.amazonaws.com.
     */
    readonly bucketName: string;
    /**
     * The log group name.
     */
    readonly logGroupName: string;
    /**
     * The number of days log events are kept in CloudWatch Logs. Exportation to S3 will happen the hour before
     * they expire in CloudWatch. Retention in S3 must be configured on the S3 Bucket provided.
     * @default - 3 days
     */
    readonly retention?: RetentionDays;
}
/**
 * This construct takes the name of a CloudWatch LogGroup and will either create it if it doesn't already exist,
 * or reuse the existing one. It also creates a regularly scheduled lambda that will export LogEvents to S3
 * before they expire in CloudWatch.
 *
 * It's used for cost-reduction, as it is more economical to archive logs in S3 than CloudWatch when
 * retaining them for more than a week.
 * Note, that it isn't economical to export logs to S3 if you plan on storing them for less than
 * 7 days total (CloudWatch and S3 combined).
 *
 * Resources Deployed
 * ------------------------
 * 1) The Lambda SingletonFunction that checks for the existence of the LogGroup;
 * 2) The CloudWatch LogGroup (if it didn't exist already);
 * 3) The CloudWatch Alarm watching log exportation failures;
 * 4) The CloudWatch Event Rule to schedule log exportation;
 * 5) The Lambda SingletonFunction, with role, to export log groups to S3 by schedule.
 *
 * @ResourcesDeployed
 */
export declare class ExportingLogGroup extends Construct {
    /**
     * The LogGroup created or fetched for the given name.
     */
    readonly logGroup: ILogGroup;
    /**
     * CloudWatch alarm on the error metric of the export LogGroup task Lambda.
     */
    readonly exportErrorAlarm: Alarm;
    /**
     * UUID needed to identify the SingletonFunction for the log exporter.
     */
    private readonly LOG_EXPORTER_UUID;
    /**
     * Duration of time between export task Lambda runs.
     */
    private readonly EXPORT_TASK_FREQUENCY;
    /**
     * Default value for the number of days to retain logs in CloudWatch for.
     */
    private readonly RETENTION_DEFAULT;
    constructor(scope: Construct, id: string, props: ExportingLogGroupProps);
    private setupLogsExporter;
}
