"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectableApplicationEndpoint = exports.ApplicationEndpoint = exports.Endpoint = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const core_1 = require("@aws-cdk/core");
/**
 * Connection endpoint
 *
 * Consists of a combination of hostname, port, and transport protocol.
 */
class Endpoint {
    /**
     * Constructs an Endpoint instance.
     *
     * @param props The properties for the endpoint
     */
    constructor(props) {
        var _a;
        const { address, port } = props;
        const protocol = (_a = props.protocol) !== null && _a !== void 0 ? _a : aws_ec2_1.Protocol.TCP;
        if (!core_1.Token.isUnresolved(port) && !Endpoint.isValidPort(port)) {
            throw new Error(`Port must be an integer between [${Endpoint.MIN_PORT}, ${Endpoint.MAX_PORT}] but got: ${port}`);
        }
        this.hostname = address;
        this.portNumber = port;
        this.protocol = protocol;
        this.port = new aws_ec2_1.Port({
            protocol: this.protocol,
            fromPort: port,
            toPort: port,
            stringRepresentation: this.renderPort(port),
        });
        this.socketAddress = `${address}:${this.portAsString()}`;
    }
    /**
     * Determines if a port is valid
     *
     * @param port: The port number
     * @returns boolean whether the port is valid
     */
    static isValidPort(port) {
        return Number.isInteger(port) && port >= Endpoint.MIN_PORT && port <= Endpoint.MAX_PORT;
    }
    /**
     * Returns the port number as a string representation that can be used for embedding within other strings.
     *
     * This is intended to deal with CDK's token system. Numeric CDK tokens are not expanded when their string
     * representation is embedded in a string. This function returns the port either as an unresolved string token or
     * as a resolved string representation of the port value.
     *
     * @returns {string} An (un)resolved string representation of the endpoint's port number
     */
    portAsString() {
        if (core_1.Token.isUnresolved(this.portNumber)) {
            return core_1.Token.asString(this.portNumber);
        }
        else {
            return this.portNumber.toString();
        }
    }
    renderPort(port) {
        return core_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
    }
}
exports.Endpoint = Endpoint;
/**
 * The minimum port value
 */
Endpoint.MIN_PORT = 1;
/**
 * The maximum port value
 */
Endpoint.MAX_PORT = 65535;
/**
 * An endpoint serving http or https for an application.
 */
class ApplicationEndpoint extends Endpoint {
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint
     */
    constructor(props) {
        var _a;
        super({
            address: props.address,
            port: props.port,
            protocol: aws_ec2_1.Protocol.TCP,
        });
        this.applicationProtocol = (_a = props.protocol) !== null && _a !== void 0 ? _a : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
    }
}
exports.ApplicationEndpoint = ApplicationEndpoint;
/**
 * An endpoint serving http or https for an application.
 */
class ConnectableApplicationEndpoint extends ApplicationEndpoint {
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint
     */
    constructor(props) {
        super(props);
        this.connections = props.connections;
    }
}
exports.ConnectableApplicationEndpoint = ConnectableApplicationEndpoint;
//# sourceMappingURL=data:application/json;base64,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