"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudWatchConfigBuilder = exports.TimeZone = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * Enum to describe the time zone property
 */
var TimeZone;
(function (TimeZone) {
    /**
     * The Local time zone
     */
    TimeZone["LOCAL"] = "Local";
    /**
     * The UTC time zone
     */
    TimeZone["UTC"] = "UTC";
})(TimeZone = exports.TimeZone || (exports.TimeZone = {}));
/**
 * Class that can build a CloudWatch Agent configuration.
 */
class CloudWatchConfigBuilder {
    /**
     * Constructs
     */
    constructor(logFlushInterval = CloudWatchConfigBuilder.DEFAULT_FLUSH_INTERVAL) {
        this.cloudWatchFileList = [];
        this.logFlushInterval = logFlushInterval;
    }
    /**
     * This method adds the log file path and its associated log group and log stream properties to the list
     * of files which needs to be streamed to cloud watch logs.
     *
     * @param logGroupName - string for the log group name
     * @param logStreamPrefix - string for the log stream prefix. The actual stream name will be appended by instance id
     * @param logFilePath - local file path which needs to be streamed
     * @param timeZone -  the time zone to use when putting timestamps on log events
     */
    addLogsCollectList(logGroupName, logStreamPrefix, logFilePath, timeZone = CloudWatchConfigBuilder.DEFAULT_LOG_TIMEZONE) {
        this.cloudWatchFileList.push({
            log_group_name: logGroupName,
            log_stream_name: logStreamPrefix + '-' + CloudWatchConfigBuilder.CLOUDWATCH_CONFIG_INSTANCE_ID_VARIABLE,
            file_path: logFilePath,
            timezone: timeZone,
        });
    }
    /**
     * The method generates the configuration for log file streaming to be added
     * to CloudWatch Agent Configuration File.
     */
    generateCloudWatchConfiguration() {
        const cloudWatchConfig = {
            logs: {
                logs_collected: {
                    files: {
                        collect_list: this.cloudWatchFileList,
                    },
                },
                log_stream_name: CloudWatchConfigBuilder.DEFAULT_STREAM_NAME
                    + '-'
                    + CloudWatchConfigBuilder.CLOUDWATCH_CONFIG_INSTANCE_ID_VARIABLE,
                force_flush_interval: this.logFlushInterval.toSeconds(),
            },
        };
        return JSON.stringify(cloudWatchConfig);
    }
}
exports.CloudWatchConfigBuilder = CloudWatchConfigBuilder;
CloudWatchConfigBuilder.DEFAULT_STREAM_NAME = 'DefaultLogStream';
CloudWatchConfigBuilder.DEFAULT_FLUSH_INTERVAL = core_1.Duration.seconds(60);
CloudWatchConfigBuilder.DEFAULT_LOG_TIMEZONE = TimeZone.LOCAL;
CloudWatchConfigBuilder.CLOUDWATCH_CONFIG_INSTANCE_ID_VARIABLE = '{instance_id}';
//# sourceMappingURL=data:application/json;base64,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