"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const types_1 = require("../types");
test.each([
    // Valid
    [
        [{ Key: 'key', Value: 'val' }],
        true,
    ],
    // Valid Multiple
    [
        [
            { Key: 'key1', Value: 'val1' },
            { Key: 'key2', Value: 'val2' },
        ],
        true,
    ],
    // Valid Empty
    [
        [],
        true,
    ],
    // Not array
    [
        'notArray',
        false,
    ],
    // Array doesn't contain objects
    [
        ['notTag'],
        false,
    ],
    // Tag array object missing Key
    [
        [{ Value: 'val' }],
        false,
    ],
    // Tag array object missing value
    [
        [{ Key: 'key' }],
        false,
    ],
])('implementsTag: %p returns %p', (value, doesImplement) => {
    expect(types_1.implementsTag(value)).toEqual(doesImplement);
});
test.each([
    // Valid no CertChain
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
        },
        true,
    ],
    // Valid with CertChain
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            CertChain: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CertChain',
        },
        true,
    ],
    // Valid, extra field
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Extra: 'test',
        },
        true,
    ],
    // Value undefined
    [undefined, false],
    // Value not object
    ['test', false],
    // No Cert
    [
        {
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
        },
        false,
    ],
    // Cert not arn
    [
        {
            Cert: 'test',
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
        },
        false,
    ],
    // Cert not string
    [
        {
            Cert: false,
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
        },
        false,
    ],
    // No Key
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
        },
        false,
    ],
    // Key not arn
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Key: 'test',
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
        },
        false,
    ],
    // Key not string
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Key: false,
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
        },
        false,
    ],
    // No Passphrase
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
        },
        false,
    ],
    // Passphrase not arn
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
            Passphrase: 'test',
        },
        false,
    ],
    // Passphrase not string
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
            Passphrase: false,
        },
        false,
    ],
    // CertChain not arn
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            CertChain: 'test',
        },
        false,
    ],
    // CertChain not string
    [
        {
            Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
            Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            CertChain: true,
        },
        false,
    ],
])('implementsISecretCertificate: %p returns %p', (value, doesImplement) => {
    expect(types_1.implementsISecretCertificate(value)).toEqual(doesImplement);
});
test.each([
    // Valid
    [
        {
            Description: 'Test Desc',
            NamePrefix: 'prefix',
            Tags: [{ Key: 'key', Value: 'val' }],
        },
        true,
    ],
    // Valid with encryption key
    [
        {
            Description: 'Test Desc',
            NamePrefix: 'prefix',
            Tags: [{ Key: 'key', Value: 'val' }],
            EncryptionKey: 'arn:aws:kms:abc123:1234:key/12ab',
        },
        true,
    ],
    // Value not defined
    [undefined, false],
    // Value not an object
    ['test', false],
    // No description
    [
        {
            NamePrefix: 'prefix',
            Tags: [{ Key: 'key', Value: 'val' }],
        },
        false,
    ],
    // Description not string
    [
        {
            Description: false,
            NamePrefix: 'prefix',
            Tags: [{ Key: 'key', Value: 'val' }],
        },
        false,
    ],
    // No NamePrefix
    [
        {
            Description: 'Test Desc',
            Tags: [{ Key: 'key', Value: 'val' }],
        },
        false,
    ],
    // NamePrefix not string
    [
        {
            Description: 'Test Desc',
            NamePrefix: false,
            Tags: [{ Key: 'key', Value: 'val' }],
        },
        false,
    ],
    // No Tags
    [
        {
            Description: 'Test Desc',
            NamePrefix: 'prefix',
        },
        false,
    ],
    // Tags not array
    [
        {
            Description: 'Test Desc',
            NamePrefix: 'prefix',
            Tags: 'notArray',
        },
        false,
    ],
    // EncrpytionKey not ARN
    [
        {
            Description: 'Test Desc',
            NamePrefix: 'prefix',
            Tags: [{ Key: 'key', Value: 'val' }],
            EncryptionKey: 'test',
        },
        false,
    ],
    // EncryptionKey not string
    [
        {
            Description: 'Test Desc',
            NamePrefix: 'prefix',
            Tags: [{ Key: 'key', Value: 'val' }],
            EncryptionKey: {},
        },
        false,
    ],
])('implementsINewSecretProps: %p returns %p', (value, doesImplement) => {
    expect(types_1.implementsINewSecretProps(value)).toEqual(doesImplement);
});
test.each([
    // Valid
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Secret: {
                Description: 'Test Desc',
                NamePrefix: 'prefix',
                Tags: [{ Key: 'key', Value: 'val' }],
            },
        },
        true,
    ],
    // Value not defined
    [undefined, false],
    // Value not object
    ['test', false],
    // No Passphrase
    [
        {
            Secret: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Secret',
        },
        false,
    ],
    // Non ARN Passphrase
    [
        {
            Passphrase: 'badArn',
            Secret: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Secret',
        },
        false,
    ],
    // Non string Passphrase
    [
        {
            Passphrase: {},
            Secret: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Secret',
        },
        false,
    ],
    // No Secret
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
        },
        false,
    ],
    // Non ARN Secret
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Secret: 'badArn',
        },
        false,
    ],
    // Non string Secret
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Secret: {},
        },
        false,
    ],
    // Extra field
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Secret: {
                Description: 'Test Desc',
                NamePrefix: 'prefix',
                Tags: [{ Key: 'key', Value: 'val' }],
            },
            Extra: 'test',
        },
        true,
    ],
])('implementsIX509ResourceProperties: %p returns %p', (value, doesImplement) => {
    expect(types_1.implementsIX509ResourceProperties(value)).toEqual(doesImplement);
});
test.each([
    // Valid
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Secret: {
                Description: 'Test Desc',
                NamePrefix: 'prefix',
                Tags: [{ Key: 'key', Value: 'val' }],
            },
            DistinguishedName: { CN: 'test' },
            SigningCertificate: {
                Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
                Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
                Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            },
        },
        true,
    ],
    // Valid, no SigningCertificate
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Secret: {
                Description: 'Test Desc',
                NamePrefix: 'prefix',
                Tags: [{ Key: 'key', Value: 'val' }],
            },
            DistinguishedName: { CN: 'test' },
        },
        true,
    ],
    // Value not defined
    [undefined, false],
    // Value not object
    ['test', false],
    // Bad IX509ResourceProperties
    [
        {
            Passphrase: '',
            Secret: {},
            DistinguishedName: { CN: 'test' },
            SigningCertificate: {
                Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
                Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
                Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            },
        },
        false,
    ],
    // Bad DistinguishedName
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Secret: {
                Description: 'Test Desc',
                NamePrefix: 'prefix',
                Tags: [{ Key: 'key', Value: 'val' }],
            },
            DistinguishedName: {},
            SigningCertificate: {
                Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
                Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
                Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            },
        },
        false,
    ],
    // Bad SigningCertificate
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Secret: {
                Description: 'Test Desc',
                NamePrefix: 'prefix',
                Tags: [{ Key: 'key', Value: 'val' }],
            },
            DistinguishedName: { CN: 'test' },
            SigningCertificate: {
                Cert: '',
                Key: '',
                Passphrase: '',
            },
        },
        false,
    ],
])('implementsIX509CertificateGenerate: %p returns %p', (value, doesImplement) => {
    expect(types_1.implementsIX509CertificateGenerate(value)).toEqual(doesImplement);
});
test.each([
    // Valid
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Secret: {
                Description: 'Test Desc',
                NamePrefix: 'prefix',
                Tags: [{ Key: 'key', Value: 'val' }],
            },
            Certificate: {
                Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
                Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
                Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            },
        },
        true,
    ],
    // Value not defined
    [undefined, false],
    // Value not object
    ['test', false],
    // Bad IX509ResourceProperties
    [
        {
            Passphrase: '',
            Secret: {},
            Certificate: {
                Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
                Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
                Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            },
        },
        false,
    ],
    // Bad Certificate
    [
        {
            Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            Secret: {
                Description: 'Test Desc',
                NamePrefix: 'prefix',
                Tags: [{ Key: 'key', Value: 'val' }],
            },
            Certificate: {
                Cert: '',
                Key: '',
                Passphrase: '',
            },
        },
        false,
    ],
])('implementsIX509CertificateEncodePkcs12: %p returns %p', (value, doesImplement) => {
    expect(types_1.implementsIX509CertificateEncodePkcs12(value)).toEqual(doesImplement);
});
test.each([
    // Valid
    [
        {
            Tags: [{ Key: 'key', Value: 'val' }],
            X509CertificatePem: {
                Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
                Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
                Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            },
        },
        true,
    ],
    // Value not defined
    [undefined, false],
    // Value not object
    ['test', false],
    // Bad X509CertificatePem
    [
        {
            Tags: [{ Key: 'key', Value: 'val' }],
            X509CertificatePem: {
                Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
                Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            },
        },
        false,
    ],
    // Bad Tags
    [
        {
            Tags: [{ Key: 'key' }],
            X509CertificatePem: {
                Cert: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert',
                Key: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Key',
                Passphrase: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Passphrase',
            },
        },
        false,
    ],
])('implementsIAcmImportCertProps: %p returns %p', (value, doesImplement) => {
    expect(types_1.implementsIAcmImportCertProps(value)).toEqual(doesImplement);
});
//# sourceMappingURL=data:application/json;base64,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