"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.importCert = exports.AcmCertificateImporter = void 0;
/* eslint-disable no-console */
const crypto = require("crypto");
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const custom_resource_1 = require("../lib/custom-resource");
const x509_certs_1 = require("../lib/x509-certs");
const types_1 = require("./types");
const ACM_VERSION = '2015-12-08';
const DYNAMODB_VERSION = '2012-08-10';
const SECRETS_MANAGER_VERSION = '2017-10-17';
class AcmCertificateImporter extends custom_resource_1.DynamoBackedCustomResource {
    constructor(acmClient, dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient);
        this.acmClient = acmClient;
        this.secretsManagerClient = secretsManagerClient;
    }
    validateInput(data) {
        return types_1.implementsIAcmImportCertProps(data);
    }
    async doCreate(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
        ]);
        const cert = await this.getSecretString(resourceProperties.X509CertificatePem.Cert);
        const certChainArn = resourceProperties.X509CertificatePem.CertChain;
        const certChain = certChainArn.length > 0 ? await this.getSecretString(certChainArn) : undefined;
        const key = await this.getSecretString(resourceProperties.X509CertificatePem.Key);
        const passphrase = await this.getSecretString(resourceProperties.X509CertificatePem.Passphrase);
        const decryptedKey = await x509_certs_1.Certificate.decryptKey(key, passphrase);
        const tags = resourceProperties.Tags;
        const certificateArn = await this.importAndStoreCertificate({
            resourceTable,
            key: decryptedKey,
            cert,
            certChain,
            physicalId,
            tags,
        });
        return { CertificateArn: certificateArn };
    }
    async doDelete(physicalId) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
        ]);
        const resources = await resourceTable.query(physicalId);
        for (const [key, resource] of Object.entries(resources)) {
            console.log(`Deleting resource for '${key}'`);
            const arn = resource.ARN;
            try {
                await this.acmClient.deleteCertificate({ CertificateArn: arn }).promise();
            }
            catch (e) {
                // AccessDeniedException can happen if either:
                //  a) We do not have the required permission to delete the Certificate (unlikely)
                //  b) The Certificate has already been deleted (more likely)
                if (e.message.indexOf('AccessDeniedException')) {
                    console.warn(`Could not delete Certificate ${arn}. Please ensure it has been deleted.`);
                }
                throw e; // Rethrow so the custom resource handler will error-out.
            }
            await resourceTable.deleteItem({
                primaryKeyValue: physicalId,
                sortKeyValue: key,
            });
        }
    }
    async importAndStoreCertificate(args) {
        let certificateArn;
        const sortKey = crypto.createHash('md5').update(args.cert).digest('hex');
        const existingItem = await args.resourceTable.getItem({
            primaryKeyValue: args.physicalId,
            sortKeyValue: sortKey,
        });
        if (existingItem) {
            if (!existingItem.ARN) {
                throw Error("Database Item missing 'ARN' attribute");
            }
            certificateArn = existingItem.ARN;
            const certificate = await this.acmClient.getCertificate({ CertificateArn: certificateArn }).promise();
            // If the cert already existed, we will updating it by performing an import again, with the new values.
            if (certificate.Certificate) {
                const importCertRequest = {
                    CertificateArn: certificateArn,
                    Certificate: args.cert,
                    CertificateChain: args.certChain,
                    PrivateKey: args.key,
                    Tags: args.tags,
                };
                await this.acmClient.importCertificate(importCertRequest).promise();
            }
            else {
                throw Error(`Database entry ${existingItem.ARN} could not be found in ACM.`);
            }
        }
        else {
            const importCertRequest = {
                Certificate: args.cert,
                CertificateChain: args.certChain,
                PrivateKey: args.key,
                Tags: args.tags,
            };
            const resp = await this.acmClient.importCertificate(importCertRequest).promise();
            if (!resp.CertificateArn) {
                throw new Error(`CertificateArn was not properly populated after attempt to import ${args.cert}`);
            }
            certificateArn = resp.CertificateArn;
            await args.resourceTable.putItem({
                primaryKeyValue: args.physicalId,
                sortKeyValue: sortKey,
                attributes: {
                    ARN: certificateArn,
                },
                allow_overwrite: false,
            });
        }
        return certificateArn;
    }
    async getSecretString(SecretId) {
        console.debug(`Retrieving secret: ${SecretId}`);
        const resp = await this.secretsManagerClient.getSecretValue({ SecretId }).promise();
        if (!resp.SecretString) {
            throw new Error(`Secret ${SecretId} did not contain a SecretString as expected`);
        }
        return resp.SecretString;
    }
}
exports.AcmCertificateImporter = AcmCertificateImporter;
/**
 * The handler used to import an X.509 certificate to ACM from a Secret
 */
async function importCert(event, context) {
    const handler = new AcmCertificateImporter(new aws_sdk_1.ACM({ apiVersion: ACM_VERSION }), new aws_sdk_1.DynamoDB({ apiVersion: DYNAMODB_VERSION }), new aws_sdk_1.SecretsManager({ apiVersion: SECRETS_MANAGER_VERSION }));
    return await handler.handler(event, context);
}
exports.importCert = importCert;
//# sourceMappingURL=data:application/json;base64,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