"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const types_1 = require("../types");
test.each([
    [
        {
            Hostname: 'foo.bar',
            Port: '1234',
            Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
            CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
        },
        true,
    ],
    [
        {
            Port: '1234',
            Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
            CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
        },
        false,
    ],
    [
        {
            Hostname: 'foo.bar',
            Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
            CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
        },
        false,
    ],
    [
        {
            Hostname: 'foo.bar',
            Port: '1234',
            CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
        },
        false,
    ],
    [
        {
            Hostname: 'foo.bar',
            Port: '1234',
            Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
        },
        false,
    ],
    [
        {
            Hostname: 'foo.bar',
            Port: 'foo',
            Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
            CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
        },
        false,
    ],
    [
        {
            Hostname: 'foo.bar',
            Port: '0',
            Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
            CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
        },
        false,
    ],
    [
        {
            Hostname: 'foo.bar',
            Port: '100000',
            Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
            CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
        },
        false,
    ],
    [
        {
            Hostname: 'foo.bar',
            Port: '1234',
            Credentials: 'not a secret',
            CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
        },
        false,
    ],
    [
        {
            Hostname: 'foo.bar',
            Port: 'foo',
            Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
            CaCertificate: 'not a secret',
        },
        false,
    ],
    [
        'not an object',
        false,
    ],
    [
        undefined,
        false,
    ],
])('implementsIConnectionOptions: %p returns %p', (value, expected) => {
    expect(types_1.implementsIConnectionOptions(value)).toStrictEqual(expected);
});
const goodRoles = JSON.stringify([
    { role: 'readWrite', db: 'testdb' },
]);
test.each([
    [
        {
            Certificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
            Roles: goodRoles,
        },
        true,
    ],
    [
        {
            Roles: goodRoles,
        },
        false,
    ],
    [
        {
            Certificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
        },
        false,
    ],
    [
        {
            Certificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
            Roles: '}{',
        },
        false,
    ],
])('implementsIX509AuthenticatedUser: %p returns %p', (value, expected) => {
    expect(types_1.implementsIX509AuthenticatedUser(value)).toStrictEqual(expected);
});
test.each([
    [
        {
            Connection: {
                Hostname: 'foo.bar',
                Port: '1234',
                Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
            },
        },
        true,
    ],
    [
        {
            Connection: {
                Hostname: 'foo.bar',
                Port: '1234',
                Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
            },
            PasswordAuthUsers: [
                'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User1',
                'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User2',
            ],
        },
        true,
    ],
    [
        {
            Connection: {
                Hostname: 'foo.bar',
                Port: '1234',
                Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
            },
            X509AuthUsers: [
                {
                    Certificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                    Roles: goodRoles,
                },
                {
                    Certificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials2',
                    Roles: goodRoles,
                },
            ],
        },
        true,
    ],
    [
        {},
        false,
    ],
    [
        {
            Connection: {
                Hostname: 'foo.bar',
                Port: 'bad port',
                Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
            },
        },
        false,
    ],
    [
        {
            Connection: {
                Hostname: 'foo.bar',
                Port: '1234',
                Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
            },
            PasswordAuthUsers: 'not array',
        },
        false,
    ],
    [
        {
            Connection: {
                Hostname: 'foo.bar',
                Port: '1234',
                Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
            },
            PasswordAuthUsers: [
                'not a secret',
                'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User2',
            ],
        },
        false,
    ],
    [
        {
            Connection: {
                Hostname: 'foo.bar',
                Port: '1234',
                Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
            },
            PasswordAuthUsers: [
                'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User1',
                'not a secret',
            ],
        },
        false,
    ],
    [
        {
            Connection: {
                Hostname: 'foo.bar',
                Port: '1234',
                Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
            },
            X509AuthUsers: 'not array',
        },
        false,
    ],
    [
        {
            Connection: {
                Hostname: 'foo.bar',
                Port: '1234',
                Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
            },
            X509AuthUsers: [
                {
                    Certificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                    Roles: '}{',
                },
                {
                    Certificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                    Roles: goodRoles,
                },
            ],
        },
        false,
    ],
    [
        {
            Connection: {
                Hostname: 'foo.bar',
                Port: '1234',
                Credentials: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                CaCertificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/CA',
            },
            X509AuthUsers: [
                {
                    Certificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials',
                    Roles: goodRoles,
                },
                {
                    Certificate: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Credentials2',
                    Roles: '}{',
                },
            ],
        },
        false,
    ],
    [
        'not an object',
        false,
    ],
    [
        undefined,
        false,
    ],
])('implementsIMongoDbConfigureResource: %p returns %p', (value, expected) => {
    expect(types_1.implementsIMongoDbConfigureResource(value)).toStrictEqual(expected);
});
//# sourceMappingURL=data:application/json;base64,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