"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const distinguished_name_1 = require("../distinguished-name");
test('toString only CN', () => {
    // GIVEN
    const name = {
        CN: 'Test CN',
    };
    // WHEN
    const dname = new distinguished_name_1.DistinguishedName(name);
    // THEN
    expect(dname.toString()).toBe('/CN=Test CN');
});
test('toString only CN+OU', () => {
    // GIVEN
    const name = {
        CN: 'Test CN',
        OU: 'Test OU',
    };
    // WHEN
    const dname = new distinguished_name_1.DistinguishedName(name);
    // THEN
    expect(dname.toString()).toBe('/CN=Test CN/OU=Test OU');
});
test('toString only CN+O', () => {
    // GIVEN
    const name = {
        CN: 'Test CN',
        O: 'Test O',
    };
    // WHEN
    const dname = new distinguished_name_1.DistinguishedName(name);
    // THEN
    expect(dname.toString()).toBe('/CN=Test CN/O=Test O');
});
test('toString CN+OU+O', () => {
    // GIVEN
    const name = {
        CN: 'Test CN',
        OU: 'Test OU',
        O: 'Test O',
    };
    // WHEN
    const dname = new distinguished_name_1.DistinguishedName(name);
    // THEN
    expect(dname.toString()).toBe('/CN=Test CN/O=Test O/OU=Test OU');
});
test('validation pass', () => {
    // GIVEN
    const name = {
        CN: 'Test CN',
        OU: 'Test OU',
        O: 'Test O',
    };
    // WHEN
    const dname = new distinguished_name_1.DistinguishedName(name);
    // THEN
    expect(dname.isValid()).toBe(true);
});
test('validation fail CN', () => {
    // GIVEN
    const name = {
        CN: 'Test/CN',
        OU: 'Test OU',
        O: 'Test O',
    };
    // WHEN
    const dname = new distinguished_name_1.DistinguishedName(name);
    // THEN
    expect(dname.isValid()).toBe(false);
});
test('validation fail OU', () => {
    // GIVEN
    const name = {
        CN: 'Test CN',
        OU: 'Test/OU',
        O: 'Test O',
    };
    // WHEN
    const dname = new distinguished_name_1.DistinguishedName(name);
    // THEN
    expect(dname.isValid()).toBe(false);
});
test('validation fail O', () => {
    // GIVEN
    const name = {
        CN: 'Test CN',
        OU: 'Test OU',
        O: 'Test/O',
    };
    // WHEN
    const dname = new distinguished_name_1.DistinguishedName(name);
    // THEN
    expect(dname.isValid()).toBe(false);
});
test.each([
    [undefined, false],
    ['string', false],
    [{}, false],
    [{ CN: {} }, false],
    [{ CN: 'string' }, true],
    [{ CN: 'string', O: {} }, false],
    [{ CN: 'string', O: 'string' }, true],
    [{ CN: 'string', OU: {} }, false],
    [{ CN: 'string', OU: 'string' }, true],
    [{ CN: 'string', O: {}, OU: 'string' }, false],
    [{ CN: 'string', O: 'string', OU: {} }, false],
    [{ CN: 'string', O: 'string', OU: 'string' }, true],
    [{ CN: 'string', O: 'string', OU: 'string', RANDOM: {} }, true],
    [{ CN: 'string', O: 'string', OU: 'string', RANDOM: 'string' }, true],
])('implementsDistinguishedNameProps: %p is %p', (value, expected) => {
    expect(distinguished_name_1.implementsDistinguishedNameProps(value)).toBe(expected);
});
//# sourceMappingURL=data:application/json;base64,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