"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Certificate = void 0;
/* eslint-disable no-console */
const child_process = require("child_process");
const fs = require("fs");
const os = require("os");
const path = require("path");
const util_1 = require("util");
const filesystem_1 = require("../filesystem");
const exec = util_1.promisify(child_process.exec);
class Certificate {
    constructor(cert, key, passphrase, certChain) {
        this.cert = cert;
        this.key = key;
        this.passphrase = passphrase;
        this.certChain = certChain;
    }
    static async fromGenerated(subject, passphrase, signingCertificate) {
        const tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
        try {
            let cert;
            let key;
            let certChain;
            if (!signingCertificate) {
                [cert, key] = await Certificate.generateSelfSigned(tmpDir, subject, passphrase);
                // certChain cannot be left undefined. CFN expects that attributes will *always* have values.
                certChain = '';
            }
            else {
                [cert, key, certChain] = await Certificate.generateSigned(tmpDir, subject, passphrase, signingCertificate);
            }
            return new Certificate(cert, key, passphrase, certChain);
        }
        finally {
            const unlinks = [];
            const filenames = await fs.promises.readdir(tmpDir);
            for (const file of filenames) {
                unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
            }
            await Promise.all(unlinks);
            await fs.promises.rmdir(tmpDir);
        }
    }
    static async decryptKey(key, passphrase) {
        const tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
        try {
            const encrypedKeyFile = path.join(tmpDir, 'encrypted.key');
            await filesystem_1.writeAsciiFile(encrypedKeyFile, key);
            const decryptedKeyFile = path.join(tmpDir, 'decrypted.key');
            const cmd = 'openssl rsa ' +
                `-in ${encrypedKeyFile} ` +
                '-passin env:CERT_PASSPHRASE ' +
                `-out ${decryptedKeyFile}`;
            console.debug(`Running: ${cmd}`);
            await exec(cmd, { env: { CERT_PASSPHRASE: passphrase, PATH: process.env.PATH } });
            const keyDecrypted = await filesystem_1.readAsciiFile(decryptedKeyFile);
            return keyDecrypted;
        }
        finally {
            const unlinks = [];
            const filenames = await fs.promises.readdir(tmpDir);
            for (const file of filenames) {
                unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
            }
            await Promise.all(unlinks);
            await fs.promises.rmdir(tmpDir);
        }
    }
    static async generateSelfSigned(tmpDir, subject, passphrase) {
        const crtFile = path.join(tmpDir, 'crt');
        const keyFile = path.join(tmpDir, 'key');
        const cmd = 'openssl req -x509 ' +
            '-passout env:CERT_PASSPHRASE ' +
            '-newkey rsa:2048 ' +
            '-days 1095 ' +
            '-extensions v3_ca ' +
            `-keyout ${keyFile} -out ${crtFile} ` +
            `-subj ${subject.toString()}`;
        console.debug(`Running: ${cmd}`);
        await exec(cmd, { env: { CERT_PASSPHRASE: passphrase, PATH: process.env.PATH } });
        const cert = await filesystem_1.readAsciiFile(crtFile);
        const key = await filesystem_1.readAsciiFile(keyFile);
        return [cert, key];
    }
    static async generateSigned(tmpDir, subject, passphrase, signingCertificate) {
        const signingCertFile = path.join(tmpDir, 'signing.crt');
        const signingKeyFile = path.join(tmpDir, 'signing.key');
        const caChain = signingCertificate.cert + signingCertificate.certChain;
        await filesystem_1.writeAsciiFile(signingCertFile, caChain);
        await filesystem_1.writeAsciiFile(signingKeyFile, signingCertificate.key);
        const csrFile = path.join(tmpDir, 'cert.csr');
        const crtFile = path.join(tmpDir, 'cert.crt');
        const keyFile = path.join(tmpDir, 'cert.key');
        const certSigningRequest = 'openssl req ' +
            '-passout env:CERT_PASSPHRASE ' +
            '-newkey rsa:2048 ' +
            '-days 1095 ' +
            `-out ${csrFile} -keyout ${keyFile} ` +
            `-subj ${subject.toString()}`;
        const crtCreate = 'openssl x509 -sha256 -req ' +
            '-passin env:SIGNING_PASSPHRASE ' +
            '-days 1095 ' +
            `-in ${csrFile} ` +
            `-CA ${signingCertFile} -CAkey ${signingKeyFile} -CAcreateserial ` +
            `-out ${crtFile}`;
        console.debug(`Running: ${certSigningRequest}`);
        await exec(certSigningRequest, { env: { CERT_PASSPHRASE: passphrase, PATH: process.env.PATH } });
        console.debug(`Running: ${crtCreate}`);
        await exec(crtCreate, { env: { PATH: process.env.PATH, SIGNING_PASSPHRASE: signingCertificate.passphrase } });
        const cert = await filesystem_1.readAsciiFile(crtFile);
        const key = await filesystem_1.readAsciiFile(keyFile);
        // Return the certificate, private key, and certificate chain. The certificate chain is the signing certificate
        // prepended to its own certificate chain.
        return [cert, key, caChain];
    }
    async toPkcs12(passphrase) {
        const tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
        try {
            const crtFileName = path.join(tmpDir, 'input.crt');
            const keyFileName = path.join(tmpDir, 'input.key');
            const caCert = this.certChain ? this.cert + this.certChain : this.cert;
            await filesystem_1.writeAsciiFile(crtFileName, caCert);
            await filesystem_1.writeAsciiFile(keyFileName, this.key);
            const pkcs12FileName = path.join(tmpDir, 'cert.p12');
            const command = 'openssl pkcs12 -export -nodes -passin env:PASSIN -passout env:PASSOUT ' +
                `-out ${pkcs12FileName} -inkey ${keyFileName} -in ${crtFileName}`;
            await exec(command, { env: {
                    PASSIN: this.passphrase,
                    PASSOUT: passphrase,
                    PATH: process.env.PATH,
                } });
            const pkcs12Data = await filesystem_1.readBinaryFile(pkcs12FileName);
            return pkcs12Data;
        }
        finally {
            const unlinks = [];
            const filenames = await fs.promises.readdir(tmpDir);
            for (const file of filenames) {
                unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
            }
            await Promise.all(unlinks);
            await fs.promises.rmdir(tmpDir);
        }
    }
}
exports.Certificate = Certificate;
//# sourceMappingURL=data:application/json;base64,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