"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const crypto_1 = require("crypto");
const AWS = require("aws-sdk");
const aws_sdk_mock_1 = require("aws-sdk-mock");
const sinon_1 = require("sinon");
const secret_1 = require("../secret");
// Enable/disable debugging statements.
const DEBUG = false;
if (!DEBUG) {
    console.debug = () => { };
}
describe('Secret class', () => {
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
    });
    describe('create', () => {
        test('success', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            aws_sdk_mock_1.mock('SecretsManager', 'createSecret', sinon_1.fake.resolves({ ARN: arn }));
            const name = 'SecretName';
            const client = new AWS.SecretsManager();
            const secret = await secret_1.Secret.create({ name, client });
            expect(secret).toEqual(secret_1.Secret.fromArn(arn, client));
        });
        test('success - all options + string', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            aws_sdk_mock_1.mock('SecretsManager', 'createSecret', sinon_1.fake.resolves({ ARN: arn }));
            const name = 'SecretName';
            const client = new AWS.SecretsManager();
            const secret = await secret_1.Secret.create({
                name,
                client,
                encryptionKey: { arn: 'testArn' },
                description: 'test desc',
                data: 'test data',
                tags: [{ Key: 'key', Value: 'value' }],
            });
            expect(secret).toEqual(secret_1.Secret.fromArn(arn, client));
        });
        test('success - all options + binary', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            aws_sdk_mock_1.mock('SecretsManager', 'createSecret', sinon_1.fake.resolves({ ARN: arn }));
            const name = 'SecretName';
            const client = new AWS.SecretsManager();
            const secret = await secret_1.Secret.create({
                name,
                client,
                encryptionKey: { arn: 'testArn' },
                description: 'test desc',
                data: Buffer.from(crypto_1.randomBytes(512)),
                tags: [{ Key: 'key', Value: 'value' }],
            });
            expect(secret).toEqual(secret_1.Secret.fromArn(arn, client));
        });
        test('missing response', async () => {
            aws_sdk_mock_1.mock('SecretsManager', 'createSecret', sinon_1.fake.resolves({}));
            const name = 'SecretName';
            const client = new AWS.SecretsManager();
            const secret = await secret_1.Secret.create({ name, client });
            expect(secret).toBeUndefined();
        });
        test('SecretsManager error', async () => {
            aws_sdk_mock_1.mock('SecretsManager', 'createSecret', sinon_1.fake.rejects({}));
            const name = 'SecretName';
            const client = new AWS.SecretsManager();
            await expect(secret_1.Secret.create({ name, client })).rejects.toThrowError();
        });
    });
    describe('delete', () => {
        test('success', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeDeleteSecret = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('SecretsManager', 'deleteSecret', fakeDeleteSecret);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            await secret.delete();
            expect(fakeDeleteSecret.callCount).toEqual(1);
        });
        test('secret already deleted', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeDeleteSecret = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('SecretsManager', 'deleteSecret', fakeDeleteSecret);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            await secret.delete();
            await expect(() => secret.delete()).rejects.toThrowError('Secret has already been deleted');
            expect(fakeDeleteSecret.callCount).toEqual(1);
        });
        test('SecretManager error', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeDeleteSecret = sinon_1.fake.rejects({});
            aws_sdk_mock_1.mock('SecretsManager', 'deleteSecret', fakeDeleteSecret);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            await expect(() => secret.delete()).rejects.toThrowError();
        });
    });
    describe('putValue', () => {
        test('string success', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakePutSecretValue = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('SecretsManager', 'putSecretValue', fakePutSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            const value = 'Super secret value'.toString();
            await secret.putValue(value);
            expect(fakePutSecretValue.callCount).toEqual(1);
            expect(fakePutSecretValue.calledWith({
                SecretId: arn,
                SecretBinary: undefined,
                SecretString: value,
            })).toBeTruthy();
        });
        test('Buffer success', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakePutSecretValue = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('SecretsManager', 'putSecretValue', fakePutSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            const value = Buffer.from(crypto_1.randomBytes(512));
            await secret.putValue(value);
            expect(fakePutSecretValue.callCount).toEqual(1);
            expect(fakePutSecretValue.calledWith({
                SecretId: arn,
                SecretBinary: value,
                SecretString: undefined,
            })).toBeTruthy();
        });
        test('already deleted', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeDeleteSecret = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('SecretsManager', 'deleteSecret', fakeDeleteSecret);
            const fakePutSecretValue = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('SecretsManager', 'putSecretValue', fakePutSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            const value = 'value';
            await secret.delete();
            await expect(() => secret.putValue(value)).rejects.toThrowError('Secret has been deleted');
            expect(fakePutSecretValue.callCount).toEqual(0);
        });
        test('SecretManager error', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakePutSecretValue = sinon_1.fake.rejects({});
            aws_sdk_mock_1.mock('SecretsManager', 'putSecretValue', fakePutSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            const value = 'Super secret value';
            await expect(() => secret.putValue(value)).rejects.toThrowError();
            expect(fakePutSecretValue.callCount).toEqual(1);
        });
    });
    describe('getValue', () => {
        test('SecretString success', async () => {
            const value = 'Super secret value'.toString();
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeGetSecretValue = sinon_1.fake.resolves({
                SecretString: value,
            });
            aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', fakeGetSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            await secret.getValue();
            expect(fakeGetSecretValue.callCount).toEqual(1);
        });
        test('SecrectBinary string success', async () => {
            const value = 'Super secret value'.toString();
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeGetSecretValue = sinon_1.fake.resolves({
                SecretBinary: value,
            });
            aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', fakeGetSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            await expect(secret.getValue()).resolves.toEqual(Buffer.from(value));
            expect(fakeGetSecretValue.callCount).toEqual(1);
        });
        test('SecretBinary Buffer success', async () => {
            const value = Buffer.from(crypto_1.randomBytes(512));
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeGetSecretValue = sinon_1.fake.resolves({
                SecretBinary: value,
            });
            aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', fakeGetSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            await expect(secret.getValue()).resolves.toEqual(value);
            expect(fakeGetSecretValue.callCount).toEqual(1);
        });
        test('SecretBinary ArrayBuffer success', async () => {
            const value = new Int32Array();
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeGetSecretValue = sinon_1.fake.resolves({
                SecretBinary: value,
            });
            aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', fakeGetSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            await expect(secret.getValue()).resolves.toEqual(Buffer.from(value.buffer));
            expect(fakeGetSecretValue.callCount).toEqual(1);
        });
        test('SecretBinary unknown type error', async () => {
            const value = new ArrayBuffer(0);
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeGetSecretValue = sinon_1.fake.resolves({
                SecretBinary: value,
            });
            aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', fakeGetSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            await expect(() => secret.getValue()).rejects.toThrowError('Unknown type for SecretBinary data');
            expect(fakeGetSecretValue.callCount).toEqual(1);
        });
        test('already deleted', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeDeleteSecret = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('SecretsManager', 'deleteSecret', fakeDeleteSecret);
            const fakeGetSecretValue = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', fakeGetSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            await secret.delete();
            await expect(() => secret.getValue()).rejects.toThrowError('Secret has been deleted');
            expect(fakeGetSecretValue.callCount).toEqual(0);
        });
        test('SecretManager error', async () => {
            const arn = 'arn:aws:secretsmanager:fake0secret1:123:secret:1a2b/';
            const fakeGetSecretValue = sinon_1.fake.rejects({});
            aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', fakeGetSecretValue);
            const client = new AWS.SecretsManager();
            const secret = secret_1.Secret.fromArn(arn, client);
            await expect(() => secret.getValue()).rejects.toThrowError();
            expect(fakeGetSecretValue.callCount).toEqual(1);
        });
    });
});
test('fromArn invalid ARN', async () => {
    const invalidArn = 'notAnArn';
    const client = new AWS.SecretsManager();
    expect(() => secret_1.Secret.fromArn(invalidArn, client)).toThrowError(`Not a Secret ARN: ${invalidArn}`);
});
test.each([
    ['test', 'test'],
    ['Test', 'Test'],
    ['test_test', 'test_test'],
    ['test+test', 'test+test'],
    ['test.test', 'test.test'],
    ['test@test', 'test@test'],
    ['test-test', 'test-test'],
    ['test-test-', 'test-test-'],
    ['test-----test', 'test-----test'],
    ['t t', 'tt'],
    ['t~t', 'tt'],
    ['t`t', 'tt'],
    ['t     t', 'tt'],
    ['t ~ t', 'tt'],
])('sanitizeSecretName', (name, nameSanitized) => {
    expect(secret_1.sanitizeSecretName(name)).toEqual(nameSanitized);
});
//# sourceMappingURL=data:application/json;base64,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