"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Secret = exports.sanitizeSecretName = void 0;
const validation_1 = require("./validation");
function sanitizeSecretName(name) {
    // The secret name must be ASCII letters, digits, or the following characters : /_+=.@-
    return name.replace(/[^A-Za-z0-9/_+=.@-]/g, '');
}
exports.sanitizeSecretName = sanitizeSecretName;
class Secret {
    constructor(arn, client) {
        this.client = client;
        this.arn = arn;
    }
    static fromArn(arn, client) {
        if (!validation_1.isArn(arn)) {
            throw Error(`Not a Secret ARN: ${arn}`);
        }
        return new Secret(arn, client);
    }
    /**
     * Creates a Secret with the given name and, optionally, containing the given 'data'. Tags the
     * secret with the given 'tags' if provided.
     * @param args
     * @throws Error if the request fails.
     * @returns A new Secret object for the newly created secret.
     */
    static async create(args) {
        var _a;
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/SecretsManager.html#createSecret-property
        const request = {
            Name: args.name,
            Description: args.description,
            KmsKeyId: (_a = args.encryptionKey) === null || _a === void 0 ? void 0 : _a.arn,
            Tags: args.tags,
            SecretString: (typeof args.data === 'string') ? args.data : undefined,
            SecretBinary: Buffer.isBuffer(args.data) ? args.data : undefined,
        };
        try {
            const response = await args.client.createSecret(request).promise();
            console.debug(`CreateSecret response: ${JSON.stringify(response)}`);
            if (response.ARN) {
                return Secret.fromArn(response.ARN, args.client);
            }
            return undefined;
        }
        catch (e) {
            throw new Error(`CreateSecret '${args.name}' failed in region '${args.client.config.region}': ` +
                `${e.code} -- ${e.message}`);
        }
    }
    /**
     * Deletes this Secret in AWS SecretsManager
     * @param force If true, then force the delete with no recovery.
     * @throws Error if the request fails.
     */
    async delete(force) {
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/SecretsManager.html#deleteSecret-property
        if (!this.arn) {
            throw Error('Secret has already been deleted');
        }
        const request = {
            SecretId: this.arn,
            ForceDeleteWithoutRecovery: force,
        };
        try {
            console.debug(`Deleting Secret: ${this.arn}`);
            const response = await this.client.deleteSecret(request).promise();
            console.debug(`DeleteSecret response: ${JSON.stringify(response)}`);
            this.arn = undefined;
        }
        catch (e) {
            throw new Error(`DeleteSecret '${this.arn}' failed in region '${this.client.config.region}':` +
                `${e.code} -- ${e.message}`);
        }
    }
    /**
     * Store the given 'data' in the Secret. Text is stored in the 'SecretString'
     * of the Secret, whereas bytes are stored base64-encoded in the 'SecretBinary'
     * of the Secret.
     * @param data
     * @throws Error if the request fails.
     */
    async putValue(data) {
        if (!this.arn) {
            throw Error('Secret has been deleted');
        }
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/SecretsManager.html#putSecretValue-property
        const request = {
            SecretId: this.arn,
            SecretString: (typeof data === 'string') ? data : undefined,
            SecretBinary: Buffer.isBuffer(data) ? data : undefined,
        };
        try {
            const response = await this.client.putSecretValue(request).promise();
            console.debug(`PutSecret response: ${JSON.stringify(response)}`);
        }
        catch (e) {
            throw new Error(`PutSecret '${this.arn}' failed in region '${this.client.config.region}':` +
                `${e.code} -- ${e.message}`);
        }
    }
    /**
     * Get the latest version of the data stored in the secret.
     */
    async getValue() {
        if (!this.arn) {
            throw Error('Secret has been deleted');
        }
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/SecretsManager.html#getSecretValue-property
        const request = {
            SecretId: this.arn,
        };
        try {
            const response = await this.client.getSecretValue(request).promise();
            if (response.SecretBinary) {
                // SecretBinary can be: Buffer|Uint8Array|Blob|string
                const data = response.SecretBinary;
                if (Buffer.isBuffer(data)) {
                    return data;
                }
                else if (typeof data === 'string') {
                    return Buffer.from(data, 'binary');
                }
                else if (ArrayBuffer.isView(data)) {
                    return Buffer.from(data);
                }
                else {
                    throw new Error('Unknown type for SecretBinary data');
                }
            }
            return response.SecretString;
        }
        catch (e) {
            throw new Error(`GetSecret '${this.arn}' failed in region '${this.client.config.region}':` +
                `${e.code} -- ${e.message}`);
        }
    }
}
exports.Secret = Secret;
Secret.API_VERSION = '2017-10-17';
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VjcmV0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2VjcmV0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7O0dBR0c7OztBQVFILDZDQUFxQztBQUVyQyxTQUFnQixrQkFBa0IsQ0FBQyxJQUFZO0lBQzdDLHVGQUF1RjtJQUN2RixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsc0JBQXNCLEVBQUUsRUFBRSxDQUFDLENBQUM7QUFDbEQsQ0FBQztBQUhELGdEQUdDO0FBRUQsTUFBYSxNQUFNO0lBbURqQixZQUFzQixHQUFXLEVBQUUsTUFBc0I7UUFDdkQsSUFBSSxDQUFDLE1BQU0sR0FBRyxNQUFNLENBQUM7UUFDckIsSUFBSSxDQUFDLEdBQUcsR0FBRyxHQUFHLENBQUM7SUFDakIsQ0FBQztJQW5ETSxNQUFNLENBQUMsT0FBTyxDQUFDLEdBQVcsRUFBRSxNQUFzQjtRQUN2RCxJQUFJLENBQUMsa0JBQUssQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUNmLE1BQU0sS0FBSyxDQUFDLHFCQUFxQixHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQ3pDO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxHQUFHLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLElBTzFCOztRQUNDLHlHQUF5RztRQUN6RyxNQUFNLE9BQU8sR0FBdUM7WUFDbEQsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO1lBQ2YsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXO1lBQzdCLFFBQVEsUUFBRSxJQUFJLENBQUMsYUFBYSwwQ0FBRSxHQUFHO1lBQ2pDLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtZQUNmLFlBQVksRUFBRSxDQUFDLE9BQU8sSUFBSSxDQUFDLElBQUksS0FBSyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNyRSxZQUFZLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDakUsQ0FBQztRQUNGLElBQUk7WUFDRixNQUFNLFFBQVEsR0FBd0MsTUFBTSxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUN4RyxPQUFPLENBQUMsS0FBSyxDQUFDLDBCQUEwQixJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNwRSxJQUFJLFFBQVEsQ0FBQyxHQUFHLEVBQUU7Z0JBQ2hCLE9BQU8sTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQzthQUNsRDtZQUNELE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBQUMsT0FBTyxDQUFDLEVBQUU7WUFDVixNQUFNLElBQUksS0FBSyxDQUFDLGlCQUFpQixJQUFJLENBQUMsSUFBSSx1QkFBdUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxLQUFLO2dCQUM3RixHQUFHLENBQUMsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7U0FDaEM7SUFDSCxDQUFDO0lBV0Q7Ozs7T0FJRztJQUNJLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBZTtRQUNqQyx5R0FBeUc7UUFDekcsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDYixNQUFNLEtBQUssQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1NBQ2hEO1FBQ0QsTUFBTSxPQUFPLEdBQXVDO1lBQ2xELFFBQVEsRUFBRSxJQUFJLENBQUMsR0FBRztZQUNsQiwwQkFBMEIsRUFBRSxLQUFLO1NBQ2xDLENBQUM7UUFDRixJQUFJO1lBQ0YsT0FBTyxDQUFDLEtBQUssQ0FBQyxvQkFBb0IsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFDOUMsTUFBTSxRQUFRLEdBQ0osTUFBTSxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUM1RCxPQUFPLENBQUMsS0FBSyxDQUFDLDBCQUEwQixJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNwRSxJQUFJLENBQUMsR0FBRyxHQUFHLFNBQVMsQ0FBQztTQUN0QjtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxDQUFDLEdBQUcsdUJBQXVCLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sSUFBSTtnQkFDM0YsR0FBRyxDQUFDLENBQUMsSUFBSSxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1NBQ2hDO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBcUI7UUFDekMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDYixNQUFNLEtBQUssQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1NBQ3hDO1FBQ0QsMkdBQTJHO1FBQzNHLE1BQU0sT0FBTyxHQUF5QztZQUNwRCxRQUFRLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDbEIsWUFBWSxFQUFFLENBQUMsT0FBTyxJQUFJLEtBQUssUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMzRCxZQUFZLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ3ZELENBQUM7UUFDRixJQUFJO1lBQ0YsTUFBTSxRQUFRLEdBQ0osTUFBTSxJQUFJLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUM5RCxPQUFPLENBQUMsS0FBSyxDQUFDLHVCQUF1QixJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUNsRTtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLElBQUksQ0FBQyxHQUFHLHVCQUF1QixJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLElBQUk7Z0JBQ3hGLEdBQUcsQ0FBQyxDQUFDLElBQUksT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztTQUNoQztJQUNILENBQUM7SUFFRDs7T0FFRztJQUNJLEtBQUssQ0FBQyxRQUFRO1FBQ25CLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ2IsTUFBTSxLQUFLLENBQUMseUJBQXlCLENBQUMsQ0FBQztTQUN4QztRQUNELDJHQUEyRztRQUMzRyxNQUFNLE9BQU8sR0FBeUM7WUFDcEQsUUFBUSxFQUFFLElBQUksQ0FBQyxHQUFHO1NBQ25CLENBQUM7UUFDRixJQUFJO1lBQ0YsTUFBTSxRQUFRLEdBQ0osTUFBTSxJQUFJLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUM5RCxJQUFJLFFBQVEsQ0FBQyxZQUFZLEVBQUU7Z0JBQ3pCLHFEQUFxRDtnQkFDckQsTUFBTSxJQUFJLEdBQUcsUUFBUSxDQUFDLFlBQVksQ0FBQztnQkFDbkMsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUN6QixPQUFPLElBQUksQ0FBQztpQkFDYjtxQkFBTSxJQUFJLE9BQU8sSUFBSSxLQUFLLFFBQVEsRUFBRTtvQkFDbkMsT0FBTyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztpQkFDcEM7cUJBQU0sSUFBSSxXQUFXLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUNuQyxPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7aUJBQzFCO3FCQUFNO29CQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsb0NBQW9DLENBQUMsQ0FBQztpQkFDdkQ7YUFDRjtZQUNELE9BQU8sUUFBUSxDQUFDLFlBQVksQ0FBQztTQUM5QjtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLElBQUksQ0FBQyxHQUFHLHVCQUF1QixJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLElBQUk7Z0JBQ3hGLEdBQUcsQ0FBQyxDQUFDLElBQUksT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztTQUNoQztJQUNILENBQUM7O0FBN0lILHdCQThJQztBQTdJd0Isa0JBQVcsR0FBRyxZQUFZLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG4vKiBlc2xpbnQtZGlzYWJsZSBuby1jb25zb2xlICovXG5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBpbXBvcnQvbm8tZXh0cmFuZW91cy1kZXBlbmRlbmNpZXNcbmltcG9ydCB7IFNlY3JldHNNYW5hZ2VyIH0gZnJvbSAnYXdzLXNkayc7XG5cbmltcG9ydCB7IEtleSB9IGZyb20gJy4uL2ttcyc7XG5pbXBvcnQgeyBpc0FybiB9IGZyb20gJy4vdmFsaWRhdGlvbic7XG5cbmV4cG9ydCBmdW5jdGlvbiBzYW5pdGl6ZVNlY3JldE5hbWUobmFtZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgLy8gVGhlIHNlY3JldCBuYW1lIG11c3QgYmUgQVNDSUkgbGV0dGVycywgZGlnaXRzLCBvciB0aGUgZm9sbG93aW5nIGNoYXJhY3RlcnMgOiAvXys9LkAtXG4gIHJldHVybiBuYW1lLnJlcGxhY2UoL1teQS1aYS16MC05L18rPS5ALV0vZywgJycpO1xufVxuXG5leHBvcnQgY2xhc3MgU2VjcmV0IHtcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBBUElfVkVSU0lPTiA9ICcyMDE3LTEwLTE3JztcblxuICBwdWJsaWMgc3RhdGljIGZyb21Bcm4oYXJuOiBzdHJpbmcsIGNsaWVudDogU2VjcmV0c01hbmFnZXIpIHtcbiAgICBpZiAoIWlzQXJuKGFybikpIHtcbiAgICAgIHRocm93IEVycm9yKGBOb3QgYSBTZWNyZXQgQVJOOiAke2Fybn1gKTtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBTZWNyZXQoYXJuLCBjbGllbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBTZWNyZXQgd2l0aCB0aGUgZ2l2ZW4gbmFtZSBhbmQsIG9wdGlvbmFsbHksIGNvbnRhaW5pbmcgdGhlIGdpdmVuICdkYXRhJy4gVGFncyB0aGVcbiAgICogc2VjcmV0IHdpdGggdGhlIGdpdmVuICd0YWdzJyBpZiBwcm92aWRlZC5cbiAgICogQHBhcmFtIGFyZ3NcbiAgICogQHRocm93cyBFcnJvciBpZiB0aGUgcmVxdWVzdCBmYWlscy5cbiAgICogQHJldHVybnMgQSBuZXcgU2VjcmV0IG9iamVjdCBmb3IgdGhlIG5ld2x5IGNyZWF0ZWQgc2VjcmV0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhc3luYyBjcmVhdGUoYXJnczoge1xuICAgIG5hbWU6IHN0cmluZyxcbiAgICBjbGllbnQ6IFNlY3JldHNNYW5hZ2VyLFxuICAgIGVuY3J5cHRpb25LZXk/OiBLZXksXG4gICAgZGVzY3JpcHRpb24/OiBzdHJpbmcsXG4gICAgZGF0YT86IEJ1ZmZlciB8IHN0cmluZyxcbiAgICB0YWdzPzogQXJyYXk8eyBLZXk6IHN0cmluZywgVmFsdWU6IHN0cmluZyB9PlxuICB9KTogUHJvbWlzZTxTZWNyZXQgfCB1bmRlZmluZWQ+IHtcbiAgICAvLyBTZWU6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NKYXZhU2NyaXB0U0RLL2xhdGVzdC9BV1MvU2VjcmV0c01hbmFnZXIuaHRtbCNjcmVhdGVTZWNyZXQtcHJvcGVydHlcbiAgICBjb25zdCByZXF1ZXN0OiBTZWNyZXRzTWFuYWdlci5DcmVhdGVTZWNyZXRSZXF1ZXN0ID0ge1xuICAgICAgTmFtZTogYXJncy5uYW1lLFxuICAgICAgRGVzY3JpcHRpb246IGFyZ3MuZGVzY3JpcHRpb24sXG4gICAgICBLbXNLZXlJZDogYXJncy5lbmNyeXB0aW9uS2V5Py5hcm4sXG4gICAgICBUYWdzOiBhcmdzLnRhZ3MsXG4gICAgICBTZWNyZXRTdHJpbmc6ICh0eXBlb2YgYXJncy5kYXRhID09PSAnc3RyaW5nJykgPyBhcmdzLmRhdGEgOiB1bmRlZmluZWQsXG4gICAgICBTZWNyZXRCaW5hcnk6IEJ1ZmZlci5pc0J1ZmZlcihhcmdzLmRhdGEpID8gYXJncy5kYXRhIDogdW5kZWZpbmVkLFxuICAgIH07XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHJlc3BvbnNlOiBTZWNyZXRzTWFuYWdlci5DcmVhdGVTZWNyZXRSZXNwb25zZSA9IGF3YWl0IGFyZ3MuY2xpZW50LmNyZWF0ZVNlY3JldChyZXF1ZXN0KS5wcm9taXNlKCk7XG4gICAgICBjb25zb2xlLmRlYnVnKGBDcmVhdGVTZWNyZXQgcmVzcG9uc2U6ICR7SlNPTi5zdHJpbmdpZnkocmVzcG9uc2UpfWApO1xuICAgICAgaWYgKHJlc3BvbnNlLkFSTikge1xuICAgICAgICByZXR1cm4gU2VjcmV0LmZyb21Bcm4ocmVzcG9uc2UuQVJOLCBhcmdzLmNsaWVudCk7XG4gICAgICB9XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ3JlYXRlU2VjcmV0ICcke2FyZ3MubmFtZX0nIGZhaWxlZCBpbiByZWdpb24gJyR7YXJncy5jbGllbnQuY29uZmlnLnJlZ2lvbn0nOiBgICtcbiAgICAgICAgYCR7ZS5jb2RlfSAtLSAke2UubWVzc2FnZX1gKTtcbiAgICB9XG4gIH1cblxuICAvLyBVbmRlZmluZWQgb25seSBpZiB0aGUgU2VjcmV0IGhhcyBiZWVuIGRlbGV0ZWQuXG4gIHB1YmxpYyBhcm46IHN0cmluZyB8IHVuZGVmaW5lZDtcbiAgcHJvdGVjdGVkIHJlYWRvbmx5IGNsaWVudDogU2VjcmV0c01hbmFnZXI7XG5cbiAgcHJvdGVjdGVkIGNvbnN0cnVjdG9yKGFybjogc3RyaW5nLCBjbGllbnQ6IFNlY3JldHNNYW5hZ2VyKSB7XG4gICAgdGhpcy5jbGllbnQgPSBjbGllbnQ7XG4gICAgdGhpcy5hcm4gPSBhcm47XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyB0aGlzIFNlY3JldCBpbiBBV1MgU2VjcmV0c01hbmFnZXJcbiAgICogQHBhcmFtIGZvcmNlIElmIHRydWUsIHRoZW4gZm9yY2UgdGhlIGRlbGV0ZSB3aXRoIG5vIHJlY292ZXJ5LlxuICAgKiBAdGhyb3dzIEVycm9yIGlmIHRoZSByZXF1ZXN0IGZhaWxzLlxuICAgKi9cbiAgcHVibGljIGFzeW5jIGRlbGV0ZShmb3JjZT86IGJvb2xlYW4pOiBQcm9taXNlPHZvaWQ+IHtcbiAgICAvLyBTZWU6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NKYXZhU2NyaXB0U0RLL2xhdGVzdC9BV1MvU2VjcmV0c01hbmFnZXIuaHRtbCNkZWxldGVTZWNyZXQtcHJvcGVydHlcbiAgICBpZiAoIXRoaXMuYXJuKSB7XG4gICAgICB0aHJvdyBFcnJvcignU2VjcmV0IGhhcyBhbHJlYWR5IGJlZW4gZGVsZXRlZCcpO1xuICAgIH1cbiAgICBjb25zdCByZXF1ZXN0OiBTZWNyZXRzTWFuYWdlci5EZWxldGVTZWNyZXRSZXF1ZXN0ID0ge1xuICAgICAgU2VjcmV0SWQ6IHRoaXMuYXJuLFxuICAgICAgRm9yY2VEZWxldGVXaXRob3V0UmVjb3Zlcnk6IGZvcmNlLFxuICAgIH07XG4gICAgdHJ5IHtcbiAgICAgIGNvbnNvbGUuZGVidWcoYERlbGV0aW5nIFNlY3JldDogJHt0aGlzLmFybn1gKTtcbiAgICAgIGNvbnN0IHJlc3BvbnNlOiBTZWNyZXRzTWFuYWdlci5EZWxldGVTZWNyZXRSZXNwb25zZSA9XG4gICAgICAgICAgICAgICAgYXdhaXQgdGhpcy5jbGllbnQuZGVsZXRlU2VjcmV0KHJlcXVlc3QpLnByb21pc2UoKTtcbiAgICAgIGNvbnNvbGUuZGVidWcoYERlbGV0ZVNlY3JldCByZXNwb25zZTogJHtKU09OLnN0cmluZ2lmeShyZXNwb25zZSl9YCk7XG4gICAgICB0aGlzLmFybiA9IHVuZGVmaW5lZDtcbiAgICB9IGNhdGNoIChlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYERlbGV0ZVNlY3JldCAnJHt0aGlzLmFybn0nIGZhaWxlZCBpbiByZWdpb24gJyR7dGhpcy5jbGllbnQuY29uZmlnLnJlZ2lvbn0nOmAgK1xuICAgICAgICBgJHtlLmNvZGV9IC0tICR7ZS5tZXNzYWdlfWApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBTdG9yZSB0aGUgZ2l2ZW4gJ2RhdGEnIGluIHRoZSBTZWNyZXQuIFRleHQgaXMgc3RvcmVkIGluIHRoZSAnU2VjcmV0U3RyaW5nJ1xuICAgKiBvZiB0aGUgU2VjcmV0LCB3aGVyZWFzIGJ5dGVzIGFyZSBzdG9yZWQgYmFzZTY0LWVuY29kZWQgaW4gdGhlICdTZWNyZXRCaW5hcnknXG4gICAqIG9mIHRoZSBTZWNyZXQuXG4gICAqIEBwYXJhbSBkYXRhXG4gICAqIEB0aHJvd3MgRXJyb3IgaWYgdGhlIHJlcXVlc3QgZmFpbHMuXG4gICAqL1xuICBwdWJsaWMgYXN5bmMgcHV0VmFsdWUoZGF0YTogc3RyaW5nIHwgQnVmZmVyKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgaWYgKCF0aGlzLmFybikge1xuICAgICAgdGhyb3cgRXJyb3IoJ1NlY3JldCBoYXMgYmVlbiBkZWxldGVkJyk7XG4gICAgfVxuICAgIC8vIFNlZTogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0phdmFTY3JpcHRTREsvbGF0ZXN0L0FXUy9TZWNyZXRzTWFuYWdlci5odG1sI3B1dFNlY3JldFZhbHVlLXByb3BlcnR5XG4gICAgY29uc3QgcmVxdWVzdDogU2VjcmV0c01hbmFnZXIuUHV0U2VjcmV0VmFsdWVSZXF1ZXN0ID0ge1xuICAgICAgU2VjcmV0SWQ6IHRoaXMuYXJuLFxuICAgICAgU2VjcmV0U3RyaW5nOiAodHlwZW9mIGRhdGEgPT09ICdzdHJpbmcnKSA/IGRhdGEgOiB1bmRlZmluZWQsXG4gICAgICBTZWNyZXRCaW5hcnk6IEJ1ZmZlci5pc0J1ZmZlcihkYXRhKSA/IGRhdGEgOiB1bmRlZmluZWQsXG4gICAgfTtcbiAgICB0cnkge1xuICAgICAgY29uc3QgcmVzcG9uc2U6IFNlY3JldHNNYW5hZ2VyLlB1dFNlY3JldFZhbHVlUmVzcG9uc2UgPVxuICAgICAgICAgICAgICAgIGF3YWl0IHRoaXMuY2xpZW50LnB1dFNlY3JldFZhbHVlKHJlcXVlc3QpLnByb21pc2UoKTtcbiAgICAgIGNvbnNvbGUuZGVidWcoYFB1dFNlY3JldCByZXNwb25zZTogJHtKU09OLnN0cmluZ2lmeShyZXNwb25zZSl9YCk7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBQdXRTZWNyZXQgJyR7dGhpcy5hcm59JyBmYWlsZWQgaW4gcmVnaW9uICcke3RoaXMuY2xpZW50LmNvbmZpZy5yZWdpb259JzpgICtcbiAgICAgICAgYCR7ZS5jb2RlfSAtLSAke2UubWVzc2FnZX1gKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogR2V0IHRoZSBsYXRlc3QgdmVyc2lvbiBvZiB0aGUgZGF0YSBzdG9yZWQgaW4gdGhlIHNlY3JldC5cbiAgICovXG4gIHB1YmxpYyBhc3luYyBnZXRWYWx1ZSgpOiBQcm9taXNlPHN0cmluZyB8IEJ1ZmZlciB8IHVuZGVmaW5lZD4ge1xuICAgIGlmICghdGhpcy5hcm4pIHtcbiAgICAgIHRocm93IEVycm9yKCdTZWNyZXQgaGFzIGJlZW4gZGVsZXRlZCcpO1xuICAgIH1cbiAgICAvLyBTZWU6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NKYXZhU2NyaXB0U0RLL2xhdGVzdC9BV1MvU2VjcmV0c01hbmFnZXIuaHRtbCNnZXRTZWNyZXRWYWx1ZS1wcm9wZXJ0eVxuICAgIGNvbnN0IHJlcXVlc3Q6IFNlY3JldHNNYW5hZ2VyLkdldFNlY3JldFZhbHVlUmVxdWVzdCA9IHtcbiAgICAgIFNlY3JldElkOiB0aGlzLmFybixcbiAgICB9O1xuICAgIHRyeSB7XG4gICAgICBjb25zdCByZXNwb25zZTogU2VjcmV0c01hbmFnZXIuR2V0U2VjcmV0VmFsdWVSZXNwb25zZSA9XG4gICAgICAgICAgICAgICAgYXdhaXQgdGhpcy5jbGllbnQuZ2V0U2VjcmV0VmFsdWUocmVxdWVzdCkucHJvbWlzZSgpO1xuICAgICAgaWYgKHJlc3BvbnNlLlNlY3JldEJpbmFyeSkge1xuICAgICAgICAvLyBTZWNyZXRCaW5hcnkgY2FuIGJlOiBCdWZmZXJ8VWludDhBcnJheXxCbG9ifHN0cmluZ1xuICAgICAgICBjb25zdCBkYXRhID0gcmVzcG9uc2UuU2VjcmV0QmluYXJ5O1xuICAgICAgICBpZiAoQnVmZmVyLmlzQnVmZmVyKGRhdGEpKSB7XG4gICAgICAgICAgcmV0dXJuIGRhdGE7XG4gICAgICAgIH0gZWxzZSBpZiAodHlwZW9mIGRhdGEgPT09ICdzdHJpbmcnKSB7XG4gICAgICAgICAgcmV0dXJuIEJ1ZmZlci5mcm9tKGRhdGEsICdiaW5hcnknKTtcbiAgICAgICAgfSBlbHNlIGlmIChBcnJheUJ1ZmZlci5pc1ZpZXcoZGF0YSkpIHtcbiAgICAgICAgICByZXR1cm4gQnVmZmVyLmZyb20oZGF0YSk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdVbmtub3duIHR5cGUgZm9yIFNlY3JldEJpbmFyeSBkYXRhJyk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIHJldHVybiByZXNwb25zZS5TZWNyZXRTdHJpbmc7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBHZXRTZWNyZXQgJyR7dGhpcy5hcm59JyBmYWlsZWQgaW4gcmVnaW9uICcke3RoaXMuY2xpZW50LmNvbmZpZy5yZWdpb259JzpgICtcbiAgICAgICAgYCR7ZS5jb2RlfSAtLSAke2UubWVzc2FnZX1gKTtcbiAgICB9XG4gIH1cbn0iXX0=