"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
function sleep(timeout) {
    return new Promise((resolve) => {
        setTimeout(() => { resolve(); }, timeout);
    });
}
/**
 * Make sure that a task completes successfully. This does increase the length of time the Lambda runs by
 * quite a bit, but Lambdas are cheap and we want to know if our logs are exporting properly.
 */
async function confirmTaskCompletion(taskId) {
    var _a, _b;
    const cloudwatchlogs = new aws_sdk_1.CloudWatchLogs({ apiVersion: '2014-03-28' });
    let errorCount = 0;
    let complete = false;
    while (!complete) {
        try {
            const response = await cloudwatchlogs.describeExportTasks({ taskId }).promise();
            if (response.$response.error) {
                throw new Error(`Task ${taskId} failed with message: ${response.$response.error.message}`);
            }
            if (((_a = response.exportTasks) === null || _a === void 0 ? void 0 : _a.length) !== 1) {
                throw new Error(`Received ${(_b = response.exportTasks) === null || _b === void 0 ? void 0 : _b.length} export tasks from DescribeExportTasks for task ${taskId}.`);
            }
            const task = response.exportTasks[0];
            if (!task.status || !task.status.code) {
                throw new Error(`Task ${taskId} did not return a status code.`);
            }
            const taskStatus = task.status.code;
            if (taskStatus === 'RUNNING' || taskStatus.indexOf('PENDING') !== -1) {
                await sleep(500);
            }
            else if (taskStatus === 'FAILED' || taskStatus === 'CANCELLED') {
                throw new Error(`Task ${taskId} failed with status code: ${taskStatus}`);
            }
            else {
                console.log(`${taskId}: Task has completed successfully!`);
                complete = true;
            }
        }
        catch (e) {
            // Retry 3 times before giving up
            if (errorCount < 3) {
                console.error(`${taskId}: Encountered failure #${errorCount} with message: ${e.message}`);
                errorCount++;
            }
            else {
                throw e;
            }
        }
    }
}
function getDatePath(dateObj) {
    const year = dateObj.getFullYear();
    const month = dateObj.getMonth() + 1;
    const date = dateObj.getDate();
    return `${year}/${month}/${date}`;
}
async function exportToS3Task(bucketName, exportFrequencyInHours, logGroupName, retentionInHours) {
    const cloudwatchlogs = new aws_sdk_1.CloudWatchLogs({ apiVersion: '2014-03-28' });
    // End time is now minus the retention period in CloudWatch plus one hour. This creates an extra hour buffer to
    // make sure no logs expire before they get exported.
    const endTime = new Date();
    endTime.setHours(endTime.getHours() - retentionInHours + 1);
    // Start time is the end time minus the frequency that the Lambda is run, with an extra minute taken off to account
    // for any drift in Lambda execution times between runs.
    const startTime = new Date();
    startTime.setHours(endTime.getHours() - exportFrequencyInHours);
    startTime.setMinutes(startTime.getMinutes() - 1);
    const destinationPrefix = `${logGroupName}/${getDatePath(new Date())}`;
    const params = {
        destination: bucketName,
        destinationPrefix,
        from: startTime.getTime(),
        logGroupName,
        to: endTime.getTime(),
    };
    const response = await cloudwatchlogs.createExportTask(params).promise();
    if (response.$response.error) {
        throw new Error(response.$response.error.message);
    }
    if (response.taskId) {
        console.log(`${response.taskId}: Successfully created export task for ${logGroupName}.`);
        console.log(`Exporting into ${bucketName} from ${startTime} to ${endTime}.`);
        await confirmTaskCompletion(response.taskId);
    }
    else {
        throw new Error(`For logGroup ${logGroupName}, No error thrown for CreateExportTask, but no task ID was received.`);
    }
}
/**
 * Lambda for exporting logs from CloudWatch to S3.
 *
 * @param event.BucketName The name of the S3 bucket to export the logs into
 * @param event.ExportFrequencyInHours How often this Lambda runs
 * @param event.LogGroupName The name of the LogGroup to export the logs from
 * @param event.RetentionInHours The retention period for logs in CloudWatch
 */
async function handler(event) {
    await exportToS3Task(event.BucketName, event.ExportFrequencyInHours, event.LogGroupName, event.RetentionInHours);
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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