"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
/**
 * Send the completeLifecycleAction() to signal that the next stage for the lifecycle action.
 * Note: This **must** always be sent; failing to do so will leave the ASG instance stuck in
 * a 'Pending: Wait' state.
 */
async function completeLifecycle(success, message) {
    // References:
    //  - https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_CompleteLifecycleAction.html
    //  - https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/AutoScaling.html#completeLifecycleAction-property
    const autoscaling = new aws_sdk_1.AutoScaling();
    try {
        const request = {
            AutoScalingGroupName: message.AutoScalingGroupName,
            LifecycleActionResult: success ? 'CONTINUE' : 'ABANDON',
            LifecycleHookName: message.LifecycleHookName,
            InstanceId: message.EC2InstanceId,
            LifecycleActionToken: message.LifecycleActionToken,
        };
        console.log('Sending CompleteLifecycleAction request: ' + JSON.stringify(request));
        const response = await autoscaling.completeLifecycleAction(request).promise();
        console.log('Got response: ' + JSON.stringify(response));
    }
    catch (e) {
        throw new Error(`Error sending completeLifecycleAction: ${e.code} -- ${e.message}`);
    }
}
async function attachEniToInstance(instanceId, eniId) {
    // References:
    //  - https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachNetworkInterface.html
    //  - https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/EC2.html#attachNetworkInterface-property
    const ec2 = new aws_sdk_1.EC2();
    try {
        const request = {
            DeviceIndex: 1,
            InstanceId: instanceId,
            NetworkInterfaceId: eniId,
        };
        console.log('Sending AttachNetworkInterface request: ' + JSON.stringify(request));
        const response = await ec2.attachNetworkInterface(request).promise();
        console.log('Got response: ' + JSON.stringify(response));
    }
    catch (e) {
        throw new Error(`Error attaching network interface to instance: ${e.code} -- ${e.message}`);
    }
}
async function handler(event) {
    console.log(`Got Event: ${JSON.stringify(event)}`);
    for (const record of event.Records) {
        try {
            console.log(`Processing record: ${JSON.stringify(record)}`);
            const message = JSON.parse(record.Sns.Message);
            // A test event is sent by Lifecycle hooks to ensure the permissions are set up correctly, so
            // only act on actual EC2 Instance Launches.
            if (message.LifecycleTransition === 'autoscaling:EC2_INSTANCE_LAUNCHING') {
                // Get the id of the ENI that we're attaching from the NotificationMetadata in the message.
                const eniId = JSON.parse(message.NotificationMetadata).eniId;
                let success = false;
                try {
                    await attachEniToInstance(message.EC2InstanceId, eniId);
                    success = true;
                }
                catch (e) {
                    console.error(e.toString());
                    console.error(e.stack);
                }
                finally {
                    // Note: Instance stays in 'Pending: Wait' state unless this lambda signals a lifecycle transition, so we must **always** send one.
                    //  https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_CompleteLifecycleAction.html
                    await completeLifecycle(success, message);
                }
            }
        }
        catch (e) {
            console.error(e.toString());
            console.error(e.stack);
        }
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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