"""
Main module for the L3 wrapper.
"""
import logging
from os.path import isdir, join, exists
from os import rename, remove
from glob import glob
import subprocess
import secrets
from l3wrapper import l3wrapper_data_path
from l3wrapper.dictionary import build_class_dict, \
                                 build_item_dictionaries, \
                                 parse_raw_rules, \
                                 write_human_readable, \
                                 build_columns_dictionary, \
                                 Transaction
from l3wrapper.validation import check_column_names, check_dtype
from joblib import Parallel, delayed
import time
from collections import Counter
from operator import itemgetter
import numpy as np
from sklearn.base import BaseEstimator, ClassifierMixin
from sklearn.utils.validation import check_X_y, check_array, check_is_fitted
from sklearn.utils.multiclass import unique_labels, check_classification_targets
import warnings


BIN_DIR = "bin"
TRAIN_BIN = "L3CFiltriItemTrain"
CLASSIFY_BIN = "L3CFiltriItemClassifica"
CLASSIFICATION_RESULTS = "classificati.txt"
LEVEL1_FILE = "livelloI.txt"
LEVEL2_FILE = "livelloII.txt"
LEVEL1_FILE_READABLE = "lvl1_R.txt"
LEVEL2_FILE_READABLE = "lvl2_R.txt"
FILTER_LEVEL1 = ''
FILTER_LEVEL2 = ''
FILTER_BOTH = ''


def _create_column_names(X):
    # Keep L3 1-based indexing naming
    return [f"{i}" for i in range(1, X.shape[1] + 1)]


def _dump_array_to_file(X, filestem, ext):
    with open(f"{filestem}.{ext}", "w") as fp:
        for row in X:
            row = [f"{i}" for i in row]
            fp.write(f"{','.join(row)}\n")


def _remove_fit_files(filestem):
    """Remove the files generated by the fit method.
    
    Retain the .cls and .diz files. These are required by the classification module of L3.
    """
    [remove(f"{filestem}{f}") for f in [
        f"_{LEVEL1_FILE}",
        f"_{LEVEL2_FILE}",
        ".cls",
        ".diz",
        "_stdout.txt",
        ".bin",
        ".data"]
    ]


def _get_matching_rules(transaction, rules, max_matching):
    if max_matching < 1:
        raise ValueError("'max_matching' must be at least 1")

    matching_rules = list()
    rule_iter = (r for r in rules if r.match(transaction)) 
    count = 0
    while count < max_matching:
        try:
            matching_rules.append(next(rule_iter))
            count += 1
        except StopIteration:
            break

    return matching_rules


def _get_majority_class(y):
    """Get the majority class.

    Note that if two or more classes have the same occurrences, one of them is returned arbitrarily.
    """
    from collections import Counter
    mc = Counter(y).most_common()
    return mc[0][0]


class L3Classifier(BaseEstimator, ClassifierMixin):
    """ Main class used as wrapper estimator around L3 binaries. 
    
    The model generation relies on L3 binaries strictly. While the generated rule sets are used
    by the wrapper in the inference phase.  
    
    Parameters
    ----------
    demo_param : str, default='demo'
        A parameter used for demonstation of how to pass and store paramters.
    Attributes
    ----------
    X_ : ndarray, shape (n_samples, n_features)
        The input passed during :meth:`fit`.
    y_ : ndarray, shape (n_samples,)
        The labels passed during :meth:`fit`.
    classes_ : ndarray, shape (n_classes,)
        The classes seen at :meth:`fit`.
    """
    def __init__(self, min_sup=0.01, min_conf=0.5,
                 l3_root=l3wrapper_data_path,
                 assign_unlabeled='majority_class',
                 match_strategy='majority_voting',
                 max_matching=1,
                 specialistic_rules=True,
                 max_length=0):
        self.min_sup = min_sup
        self.min_conf = min_conf
        self.l3_root = l3_root
        self.assign_unlabeled = assign_unlabeled
        self.match_strategy = match_strategy
        self.max_matching = max_matching
        self.specialistic_rules = specialistic_rules
        self.max_length = max_length

    def _more_tags(self):
        return {
            'requires_fit': True,
            'allow_nan': False, 
            'X_types': ['2darray', 'string']
        }

    def _get_class_label(self, matching_rules: list):
        """TODO Important method to weight majority voting"""
        class_ids = list()
        class_priority = {class_id : 0 for (class_id, _) in self._class_dict.items()}
        for rule in matching_rules:
            class_ids.append(rule.class_id)
            class_priority[rule.class_id] += rule.rank

        most_common = Counter(class_ids).most_common()
        most_common = sorted(most_common, key=lambda x: class_priority[x[0]])   # ascending by class priority
        most_common = sorted(most_common, key=itemgetter(1), reverse=True)      # descending by matched count

        return self._class_dict[most_common[0][0]]

    def fit(self, X, y, column_names=None, save_human_readable=False, remove_files=True):
        """A reference implementation of a fitting function for a classifier.
        Parameters
        ----------
        X : array-like, shape (n_samples, n_features)
            The training input samples.
        y : array-like, shape (n_samples,)
            The target values. An array of int.
        Returns
        -------
        self : object
            Returns self.
        """
        self._train_bin_path = join(self.l3_root, BIN_DIR, TRAIN_BIN)
        self._classify_bin_path = join(self.l3_root, BIN_DIR, CLASSIFY_BIN)
        self._logger = logging.getLogger(__name__)

        X = check_dtype(X)
        check_classification_targets(y)

        # Check that X and y have correct shape
        X, y = check_X_y(X, y, dtype=np.unicode_)

        # Store the classes seen during fit
        self.classes_ = unique_labels(y)

        # Define the label when no rule matches
        if self.assign_unlabeled == 'majority_class':
            self.unlabeled_class_ = _get_majority_class(y)
        else:
            self.unlabeled_class_ = self.assign_unlabeled

        self.X_ = X
        self.y_ = y

        token = secrets.token_hex(4)
        filestem = f"{token}"

        # TODO the character ':' is not allowed in any column name, enforce this.
        # Create column names if not provided
        if not column_names:
            column_names = _create_column_names(X)
        check_column_names(X, column_names)
        self._column_id_to_name = build_columns_dictionary(column_names)

        # Dump X and y in a single .data (csv) file. "y" target labels are inserted as the last column
        X_todump = np.hstack([X, y.reshape(-1, 1)])
        _dump_array_to_file(X_todump, filestem, "data")

        # Invoke the training module of L3.
        if self.specialistic_rules:
            specialistic_flag = "0"
        else:
            specialistic_flag = "1"

        with open(f"{filestem}_stdout.txt", "w") as stdout:
            subprocess.run(
                [
                    self._train_bin_path,
                    filestem,                       # training file filestem
                    f"{self.min_sup * 100:.2f}",    # min sup
                    f"{self.min_conf * 100:.2f}",   # min conf
                    "nofiltro",                     # filtering measure for items (DEPRECATED)
                    "0",                            # filtering threshold (DEPRECATED)
                    specialistic_flag,              # specialistic/general rules (TO VERIFY)
                    f"{self.max_length}",           # max length allowed for rules
                    self.l3_root                    # L3 root containing the 'bin' directory with binaries
                ],
                stdout=stdout
            )
        
        # rename useful (lvl1) and sparse (lvl2) rule files
        rename(LEVEL1_FILE, f"{token}_{LEVEL1_FILE}")
        rename(LEVEL2_FILE, f"{token}_{LEVEL2_FILE}")

        # read the mappings of classification labels
        self._class_dict = build_class_dict(filestem)

        # read the mappings item->"column_name","value"
        self._item_id_to_item, self._item_to_item_id = build_item_dictionaries(filestem)
        self.n_items_used_ = len(self._item_id_to_item)

        # parse the two rule sets and store them
        self.lvl1_rules_ = parse_raw_rules(f"{token}_{LEVEL1_FILE}")
        self.lvl2_rules_ = parse_raw_rules(f"{token}_{LEVEL2_FILE}")
        self.n_lvl1_rules_ = len(self.lvl1_rules_)
        self.n_lvl2_rules_ = len(self.lvl2_rules_)

        # translate the model to human readable format
        if save_human_readable:
            write_human_readable(f"{token}_{LEVEL1_FILE_READABLE}", self.lvl1_rules_,
                                 self._item_id_to_item, self._column_id_to_name, self._class_dict)
            write_human_readable(f"{token}_{LEVEL2_FILE_READABLE}", self.lvl2_rules_,
                                 self._item_id_to_item, self._column_id_to_name, self._class_dict)

        if remove_files:
            _remove_fit_files(token)

        self.current_token_ = token # keep track of the latest token generated by the fit method
        
        return self

    def predict(self, X):
        """ A reference implementation of a prediction for a classifier.
        Parameters
        ----------
        X : array-like, shape (n_samples, n_features)
            The input samples.
        Returns
        -------
        y : ndarray, shape (n_samples,)
            The label for each sample is the label of the closest sample
            seen during fit.
        """
        # Check is fit had been called
        check_is_fitted(self, ['X_', 'y_'])

        # Input validation
        X = check_array(X, dtype=np.unicode_)

        self.labeled_transactions_ = list()
        y_pred = list()
        for X_row in X:
            tr = Transaction(X_row, self._item_to_item_id)
            class_priority = {k: 0 for k, v in self._class_dict.items()} # used the majority voting is even
            used_level = None

            # match against level 1
            matching_rules = _get_matching_rules(tr, self.lvl1_rules_, self.max_matching)

            # if level 1 was not used, match against level 2
            if not matching_rules:
                matching_rules = _get_matching_rules(tr, self.lvl2_rules_, self.max_matching)
                if matching_rules:
                    used_level = 2
            else:
                used_level = 1
            
            if not matching_rules:
                y_pred.append(self.unlabeled_class_)
            else:
                y_pred.append(self._get_class_label(matching_rules))

            tr.used_level = used_level
            self.labeled_transactions_.append(tr)        # keep track of labeled transaction

        return np.array(y_pred)