import random
from collections import defaultdict

from corpus_builder.text_augmentation import eda

class CorpusBuilder:
    """Automatically build a corpus defined by a Grammar
    """

    def __init__(self, grammar, classes_labels, entities_labels):
        """Initializes builder with grammar and annotation scheme

        Args:
            grammar (Grammar): Definition of the domain corpus
            classes_labels ({str: [str]}): Classifications and labels to be generated by the grammar.
                The key should be the same key as one Rule. The list of strings are the labels of the
                classification, that should be in the same order as the associated rule.
            entities_labels ({str: str}): Entities annotations, key should be the key of the Rule
                that generates one entity, and the value is the label associated with that entity
        """
        self.grammar = grammar
        self.classes_labels = classes_labels
        self.entities_labels = entities_labels

    def create_sentence(self):
        """Create one sentence in natural language and its annotations

        Returns:
            str: Generated natural language sentence
            dict: Annotations of the generated sentence
        """
        sentence = [self.grammar.root]
        annotations = {'entities': []}
        
        while self._exists_non_terminal(sentence):
            non_terminal_idx = self._choose_non_terminal(sentence)
            non_terminal = sentence[non_terminal_idx]

            if self._is_classifiable(non_terminal):
                rule = self.grammar.get_rule(non_terminal)
                idx = random.randint(0, len(rule.derivation)-1)
                expansion = [rule.derivation[idx]]
                class_label = self.classes_labels[non_terminal][idx]
                annotations[non_terminal] = class_label
            elif self._is_entity_marked(non_terminal):
                expansion = self.fully_expand([non_terminal])
                label = self.entities_labels[non_terminal]
                value =  ' '.join(clean_numbers(expansion))
                annotations['entities'].append((value, label))
            else:
                expansion = self.grammar.get_rule(non_terminal).expand()
                expansion = separate_tokens(expansion)

            sentence = sentence[:non_terminal_idx] + expansion + sentence[non_terminal_idx+1:]
            
        sentence = clean_numbers(sentence, annotations['entities'])
        sentence = ' '.join(sentence)
            
        return (sentence, annotations)

    def create_corpus(self, n_base_sentences, n_aug):
        """Create a complete corpus with generation of base examples and data augmenation

        Args:
            n_base_sentences (int): number of sentences to be produced by the grammar
            n_aug (int): number of augmentation operations to be done for each base sentence

        Returns:
            [(str, dict)]: list of examples and annotations with length (n_aug + 1) * n_base_sentences
        """
        base_corpus = [self.create_sentence() for _ in range(n_base_sentences)]
        corpus = []
        if n_aug == 0:
            return base_corpus
        else:
            for sen, anot in base_corpus:
                augmented_sentences = eda(sen, num_aug=n_aug)
                for aug_sen in augmented_sentences:
                    aug_annotation = annotate_augmentent_sentence(aug_sen, anot)
                    corpus.append((aug_sen, aug_annotation))
            return corpus

    def _exists_non_terminal(self, s):
        return any(self.grammar.is_non_terminal(token) for token in s)

    def _choose_non_terminal(self, s):
        non_terminals = [token for token in s if self.grammar.is_non_terminal(token)]
        return s.index(random.choice(non_terminals))

    def _is_classifiable(self, non_terminal):
        return non_terminal in self.classes_labels.keys()

    def _is_entity_marked(self, non_terminal):
        return non_terminal in self.entities_labels.keys()

    def fully_expand(self, sentence):
        while self._exists_non_terminal(sentence):
            non_terminal_idx = self._choose_non_terminal(sentence)
            non_terminal = sentence[non_terminal_idx]
            expansion = self.grammar.get_rule(non_terminal).expand()
            expansion = separate_tokens(expansion)
            sentence = sentence[:non_terminal_idx] + expansion + sentence[non_terminal_idx+1:]
        return sentence

def annotate_augmentent_sentence(sentence, base_annotation):
    anot = base_annotation.copy()
    anot['entities'] = []
    for value, label in base_annotation['entities']:
        if value in sentence:
            anot['entities'].append((value, label))
    return anot     

def separate_tokens(s):
    return s.replace('><', ' ').split()

def clean_numbers(s, entities=[]):
    entity_mask = get_entity_mask(s, entities)

    i = 0
    while i < len(s) - 1:
        if s[i].isnumeric() and s[i+1].isnumeric() and (entity_mask[i] == entity_mask[i+1]):
            s[i] = ''.join(s[i:i+2])
            s.pop(i+1)
            entity_mask.pop(i+1)
        else:
            i += 1
    return s

def get_entity_mask(s, entities):
    entity_mask = ['' for _ in range(len(s))]
    ent_ranges = {}
    labels_freqs = defaultdict(int)
    entities.sort(key=lambda ent : len(ent[0]), reverse=True)
    for value, label in entities:
        if not value:
            continue
        ent_range = get_ent_range(value, s)
        if ent_range is not None:
            ent_ranges[label] = ent_range
            labels_freqs[label] += 1
    
    for label in ent_ranges:
        b, e = ent_ranges[label]
        for i in range(b, e+1):
            entity_mask[i] = label + str(labels_freqs[label])
        
    return entity_mask

def get_ent_range(value, sen):
    # possible positions where the entity can begin
    possible_begins = [i for i in range(len(sen)) if sen[i] == value[0]]
    # check if the whole entity is after the possible begin,
    # if so then there is the true position of the entity
    for i in possible_begins:
        num_match = 0
        for j in range(len(value)):
            if sen[i + j] == value[j]:
                # match ok so far
                num_match += 1
                if value[j] == value[-1] and num_match == len(value):
                    # end of entity, then is the correct place
                    return (i, i+j)
            else:
                # wrong position, test next possible start
                break
    return None