"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('create HTTPS redirect', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    // WHEN
    new lib_1.HttpsRedirect(stack, 'Redirect', {
        recordNames: ['foo.example.com', 'baz.example.com'],
        targetDomain: 'bar.example.com',
        zone: aws_route53_1.HostedZone.fromHostedZoneAttributes(stack, 'HostedZone', {
            hostedZoneId: 'ID',
            zoneName: 'example.com',
        })
    });
    // THEN
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            RedirectAllRequestsTo: {
                HostName: 'bar.example.com',
                Protocol: 'https'
            }
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: ['foo.example.com', 'baz.example.com']
        }
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Name: 'foo.example.com.',
        HostedZoneId: 'ID',
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Name: 'baz.example.com.',
        HostedZoneId: 'ID',
    });
});
test('create HTTPS redirect for apex', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    // WHEN
    new lib_1.HttpsRedirect(stack, 'Redirect', {
        targetDomain: 'bar.example.com',
        zone: aws_route53_1.HostedZone.fromHostedZoneAttributes(stack, 'HostedZone', {
            hostedZoneId: 'ID',
            zoneName: 'example.com',
        })
    });
    // THEN
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            RedirectAllRequestsTo: {
                HostName: 'bar.example.com',
                Protocol: 'https'
            }
        }
    });
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        Name: 'example.com.',
    });
});
test('create HTTPS redirect with existing cert', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    // WHEN
    new lib_1.HttpsRedirect(stack, 'Redirect', {
        recordNames: ['foo.example.com'],
        certificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
        targetDomain: 'bar.example.com',
        zone: aws_route53_1.HostedZone.fromHostedZoneAttributes(stack, 'HostedZone', {
            hostedZoneId: 'ID',
            zoneName: 'example.com',
        })
    });
    // THEN
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            RedirectAllRequestsTo: {
                HostName: 'bar.example.com',
                Protocol: 'https'
            }
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            ViewerCertificate: {
                AcmCertificateArn: 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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