from datasets import load_dataset

from minieval.datatypes import Instance, LMOutput, Task, TaskConfig
from minieval.extract import qa


class MMLU(Task):
    # fmt: off
    SUBSETS = [
        "abstract_algebra", "anatomy", "astronomy", "business_ethics", "clinical_knowledge", "college_biology", 
        "college_chemistry", "college_computer_science", "college_mathematics", "college_medicine", "college_physics", 
        "computer_security", "conceptual_physics", "econometrics", "electrical_engineering", "elementary_mathematics", 
        "formal_logic", "global_facts", "high_school_biology", "high_school_chemistry", "high_school_computer_science", "high_school_european_history", "high_school_geography", "high_school_government_and_politics", 
        "high_school_macroeconomics", "high_school_mathematics", "high_school_microeconomics", "high_school_physics", 
        "high_school_psychology", "high_school_statistics", "high_school_us_history", "high_school_world_history", "human_aging", 
        "human_sexuality", "international_law", "jurisprudence", "logical_fallacies", "machine_learning", "management", "marketing", 
        "medical_genetics", "miscellaneous", "moral_disputes", "moral_scenarios", "nutrition", "philosophy", "prehistory", 
        "professional_accounting", "professional_law", "professional_medicine", "professional_psychology", "public_relations", 
        "security_studies", "sociology", "us_foreign_policy", "virology", "world_religions"
    ]
    # fmt: on
    HF_PATH = "cais/mmlu"

    def __init__(self, config: TaskConfig):
        self.config = config
        requests = []
        for subset in self.SUBSETS:
            dataset = load_dataset(path=self.HF_PATH, name=subset, split="test")
            requests += list(map(self._process_instance, dataset, subset))
        self.requests = requests
        self.FEW_SHOT = self._construct_few_shot()

    def _process_instance(self, doc, subset):
        gold_idx = doc["answer"]
        choices = doc["choices"]

        return Instance(
            question=doc["question"],
            gold_completion=choices[gold_idx],
            choices=choices,
            solution=gold_idx,
            metadata={"subset": subset},
        )
    
    def _extract_answer(self, generation: LMOutput) -> int:
        answer = qa.extract_mcqa_answer(
            generation.text, 
            answer_regexes=["\\(?([A-D])\\)?"] # both "(A)" and "A"
        )
        if answer in ["A", "B", "C", "D"]:
            return ["A", "B", "C", "D"].index(answer)
        return None

    def _construct_few_shot(self, k):
        # Sample few shot examples from the "dev" set, following
        # https://github.com/hendrycks/test/blob/master/evaluate.py#L28
        dataset = load_dataset(path=self.HF_PATH, split="dev")
        few_shot_docs = list(map(self._process_instance, dataset))

        # TODO: the few shot examples need to match the subset!

        return few_shot_docs[:k]


class MMLUPro(Task):
    # fmt: off
    SUBSETS = [
        "math", "health", "physics", "business", "biology", "chemistry", "computer science", 
        "economics", "engineering", "philosophy", "other", "history", "psychology", "law"
    ]
    # fmt: on
    HF_PATH = "TIGER-Lab/MMLU-Pro"

    def __init__(self, config):
        self.config = config
        dataset = load_dataset(path=self.HF_PATH, split="test")
        self.requests = map(self._process_instance, dataset)
        self.FEW_SHOT = self._construct_few_shot()

    def _process_instance(self, doc):
        gold_idx = doc["answer_index"]
        choices = doc["options"]

        return Instance(
            question=doc["question"],
            gold_completion=choices[gold_idx],
            choices=choices,
            solution=gold_idx,
            subset=doc["category"],
            metadata={"id": doc["question_id"], "src": doc["src"]},
        )
    
    def _extract_answer(self, generation: LMOutput) -> int:
        answer = qa.extract_mcqa_answer(
            generation.text, 
            answer_regexes=["\\(?([A-J])\\)?"] # both "(A)" and "A"
        )
        if answer in ["A", "B", "C", "D", "E", "F", "G", "H", "I", "J"]:
            return ["A", "B", "C", "D", "E", "F", "G", "H", "I", "J"].index(answer)
        return None

    def _construct_few_shot(self, k):
        dataset = load_dataset(path=self.HF_PATH, split="validation")
        few_shot_docs = list(map(self._process_instance, dataset))

        # TODO: the few shot examples need to match the subset!

        return few_shot_docs[:k]


# TODO: Grab prompts from https://github.com/openai/simple-evals/blob/main/mgsm_eval.py
class MultilingualMMLU(Task):
    HF_PATH = "openai/MMMLU"
    # fmt: off
    SUBSETS = [
        "AR-XY", "BN-BD", "DE-DE", "ES-LA", "FR-FR", "HI-IN", "ID-ID", 
        "IT-IT", "JA-JP", "KO-KR", "PT-BR", "SW-KE", "YO-NG", "ZH-CN",
    ]
    # fmt: on

    def __init__(self, config: TaskConfig):
        self.config = config
        requests = []
        for subset in self.SUBSETS:
            dataset = load_dataset(path=self.HF_PATH, split=subset)
            requests += map(self._process_instance, dataset, subset)
        self.requests = requests

    def _process_instance(self, doc, subset):
        gold_idx = ord(doc["Answer"]) - ord("A")
        choices = [doc["A"], doc["B"], doc["C"], doc["D"]]

        return Instance(
            question=doc["Question"],
            gold_completion=choices[gold_idx],
            choices=choices,
            solution=gold_idx,
            subset=subset,
            metadata={"id": doc["Unnamed: 0"], "src": doc["src"], "mmlu_subset": doc["Subject"]},
        )
