from datasets import load_dataset

from minieval.datatypes import Instance, LMOutput, Task, TaskConfig
from minieval.extract import math_latex
from minieval.few_shot import FewShot


class MinervaMath(Task):
    SUBSETS = [
        "algebra",
        "counting_and_probability",
        "geometry",
        "intermediate_algebra",
        "number_theory",
        "prealgebra",
        "precalculus",
    ]
    HF_PATH = "EleutherAI/hendrycks_math"
    FEW_SHOT = FewShot.get("minerva::oe_eval")

    def __init__(self, config: TaskConfig):
        self.config = config
        requests = []
        for subset in self.SUBSETS:
            dataset = load_dataset(path=self.HF_PATH, name=subset, split="test")
            requests += list(map(self._process_instance, dataset, subset))
        self.requests = requests

    def _process_instance(self, doc: dict, subset: str = None) -> Instance:
        solution = math_latex.extract_math_answer(doc["solution"])[0]  # get primary extracted answer

        return Instance(
            question=doc["problem"],
            gold_completion=doc["solution"],
            solution=solution,
            subset=subset,
            metadata={"level": doc.get("level"), "type": doc.get("type")},
        )
    
    def _extract_answer(self, generation: LMOutput) -> list[str]:
        return math_latex.extract_math_answer(generation.text)


class Math500(MinervaMath):
    HF_PATH = "HuggingFaceH4/MATH-500"

    def __init__(self, config: TaskConfig):
        self.config = config
        dataset = load_dataset(path=self.HF_PATH, split="test")
        self.requests = list(map(self._process_instance, dataset))
