import re
from typing import List

from datasets import load_dataset

from minieval.datatypes import Instance, LMOutput, Task, TaskConfig
from minieval.extract import math_latex
from minieval.few_shot import FewShot


class GSM8K(Task):
    HF_PATH = "gsm8k"
    FEW_SHOT = FewShot.get("gsm::oe_eval")

    def __init__(self, config: TaskConfig):
        self.config = config
        requests = []
        for subset in ["train", "test"]:
            dataset = load_dataset(path=self.HF_PATH, name="main", split=subset)
            requests += list(map(self._process_instance, dataset))
        self.requests = requests

    def _process_instance(self, doc):
        short_answer = doc["answer"].split("####")[-1].strip()
        gold_cot = self._cleanup_answer_str(doc, doc["answer"])
        return Instance(
            question=doc["question"],
            gold_completion=gold_cot,
            solution=short_answer,
            metadata={
                "short_answer": short_answer,
            },
        )
    
    def _extract_answer(self, generation: LMOutput) -> list[str]:
        # @davidh TODO: Review the oe-eval GSM implementation
        return math_latex.extract_math_answer(generation.text)

    def _cleanup_answer_str(self, doc: dict, answer: str) -> str:
        """
        Convert the gold CoT to a more natural-appearing string to improve bpb calculation. E.g.:

        Question: Janet’s ducks lay 16 eggs per day. She eats three for breakfast every morning and bakes muffins for her friends every day with four. She sells the remainder at the farmers' market daily for $2 per fresh duck egg. How much in dollars does she make every day at the farmers' market?

        Original Answer: Janet has 16 eggs and uses 4 for baking and sells 3 for breakfast. Therefore, she makes 16 - 3 - 4 = <<16-3-4=9>>9 eggs sold, leading to a daily income of 9 * 2 = $<<9*2=22>>22.\n#### 22

        New Answer: Janet sells 16 - 3 - 4 = 9 duck eggs a day. She makes 9 * 2 = $18 every day at the farmer’s market. So the answer is 18.
        """

        def _add_spaces_around_operators_no_regex(_str):
            """Add spacing around special operators if it does not exist"""
            operators = {"+", "-", "*", "/", "="}
            result: List[str] = []
            for char in _str:
                if char in operators:
                    if result and result[-1] != " ":
                        result.append(" ")
                    result.append(char)
                    result.append(" ")
                else:
                    result.append(char)

            # Join the list and replace double spaces with single spaces
            return "".join(result).replace("  ", " ")

        answer = re.sub(r"<<.*?>>", "", answer)
        answer = re.sub(r"\s+", " ", answer).strip()
        answer = re.split(r"####", answer)[0]
        answer = answer[0].capitalize() + answer[1:] if answer else answer
        answer = answer.strip()
        if not answer.endswith("."):
            answer += "."
        answer = answer + f" So the answer is {doc['answer'].split('####')[-1].strip()}."
        answer = _add_spaces_around_operators_no_regex(answer)
        return answer

    def _clean_short_answer(self, continuation: str):
        output = re.sub(r"(\d),(\d)", r"\1\2", continuation)
        numbers = re.findall(r"[-+]?\d*\.\d+|\d+", output)
        if numbers:
            return numbers[-1]
        else:
            return output


class GSMPlus(GSM8K):
    HF_PATH = "qintongli/GSM-Plus"

    def __init__(self, config: TaskConfig):
        self.config = config
        dataset = load_dataset(path=self.HF_PATH, name="main", split="test")
        self.requests = map(self._process_instance, dataset)


class GSMSymbolic(GSM8K):
    HF_PATH = "apple/GSM-Symbolic"

    def __init__(self, split):
        if type(self) is GSMSymbolic:
            raise TypeError("GSM Symbolic is an abstract class. Please use a child!")

        dataset = load_dataset(path=self.HF_PATH, name="main", split=split)
        self.requests = map(self._process_instance, dataset)


class GSMSymbolicMain(GSMSymbolic):
    def __init__(self, config: TaskConfig):
        self.config = config
        super().__init__(split="main")


class GSMSymbolicP1(GSMSymbolic):
    def __init__(self, config: TaskConfig):
        self.config = config
        super().__init__(split="p1")


class GSMSymbolicP2(GSMSymbolic):
    def __init__(self, config: TaskConfig):
        self.config = config
        super().__init__(split="p2")


# TODO: Add MGSM https://github.com/openai/simple-evals/blob/main/mgsm_eval.py
