import random
import re

from datasets import load_dataset

from minieval.datatypes import Instance, LMOutput, Task, TaskConfig
from minieval.extract import qa
from minieval.few_shot import FewShot


class GPQA(Task):
    HF_PATH = "Idavidrein/gpqa"
    FEW_SHOT = FewShot.get("gpqa::original")
    SEED = 111

    def __init__(self, config: TaskConfig):
        self.config = config
        dataset = load_dataset(path=self.HF_PATH, name="gpqa_main", split="train")
        self.requests = list(map(self._process_instance, dataset))

    def _process_instance(self, doc):
        gold_answer = self._preprocess(doc["Correct Answer"])
        choices = [
            self._preprocess(doc["Incorrect Answer 1"]),
            self._preprocess(doc["Incorrect Answer 2"]),
            self._preprocess(doc["Incorrect Answer 3"]),
            gold_answer,
        ]

        random.Random(self.SEED + hash(doc["Record ID"])).shuffle(choices)
        correct_answer_index = choices.index(gold_answer)

        return Instance(
            question=doc["Question"],
            gold_completion=choices[correct_answer_index],
            choices=choices,
            solution=correct_answer_index,
            metadata={
                "id": doc["Record ID"],
                "canary_string": doc["Canary String"],
                "explanation": doc["Explanation"],
            },
        )
    
    def _extract_answer(self, generation: LMOutput) -> int:
        answer = qa.extract_mcqa_answer(
            generation.text, 
            answer_regexes=["\\(?([A-D])\\)?"] # both "(A)" and "A"
        )
        if answer in ["A", "B", "C", "D"]:
            return ["A", "B", "C", "D"].index(answer)
        return None

    def _preprocess(self, text):
        if text is None:
            return " "
        text = text.strip()
        text = text.replace(" [title]", ". ")
        text = re.sub("\\[.*?\\]", "", text)
        text = text.replace("  ", " ")
        return text


class GPQADiamond(GPQA):
    def __init__(self, config: TaskConfig):
        self.config = config
        dataset = load_dataset(path=self.HF_PATH, name="gpqa_diamond", split="train")
        self.requests = map(self._process_instance, dataset)


class SuperGPQA(GPQA):
    HF_PATH = "m-a-p/SuperGPQA"
    FEW_SHOT = FewShot.get("gpqa::original")

    def __init__(self, config: TaskConfig):
        self.config = config
        dataset = load_dataset(path=self.HF_PATH, split="train")
        self.requests = map(self._process_instance, dataset)

    def _process_instance(self, doc):
        gold_answer = doc["answer"]
        choices: list[str] = doc["options"]

        correct_answer_index = choices.index(gold_answer)

        return Instance(
            question=doc["question"],
            gold_completion=gold_answer,
            choices=doc["options"],
            solution=correct_answer_index,
            metadata={
                "id": doc["uuid"],
                "discipline": doc["discipline"],
                "field": doc["field"],
                "subfield": doc["subfield"],
                "difficulty": doc["difficulty"],
                "is_calculation": doc["is_calculation"],
            },
        )
