from datasets import load_dataset

from minieval.datatypes import Instance, LMOutput, Task, TaskConfig
from minieval.extract import qa
from minieval.few_shot import FewShot


class ARC(Task):
    HF_PATH = "ai2_arc"

    def __init__(self, dataset_name):
        if type(self) is ARC:
            raise TypeError("ARC is an abstract class. Please use a child!")

        requests = []
        for subset in ["train", "validation", "test"]:
            dataset = load_dataset(path=self.HF_PATH, split=subset, name=dataset_name)
            requests += list(map(self._process_instance, dataset))
        self.requests = requests

    def _process_instance(self, doc):
        if doc["answerKey"].isdigit():
            doc["answerKey"] = chr(ord("A") + int(doc["answerKey"]) - 1)

        gold_idx = ["A", "B", "C", "D", "E"].index(doc["answerKey"])
        choices = doc["choices"]["text"]

        return Instance(
            question=doc["question"],
            gold_completion=choices[gold_idx],
            choices=choices,
            solution=gold_idx,
            metadata={"id": doc["id"]},
        )
    
    def _extract_answer(self, generation: LMOutput) -> int:
        answer = qa.extract_mcqa_answer(
            generation.text, 
            answer_regexes=["\\(?([A-D])\\)?"] # both "(A)" and "A"
        )
        if answer in ["A", "B", "C", "D"]:
            return ["A", "B", "C", "D"].index(answer)
        return None


class ARCChallenge(ARC):
    FEW_SHOT = FewShot.get("arc_challenge::olmes")

    def __init__(self, config: TaskConfig):
        self.config = config
        super().__init__(dataset_name="ARC-Challenge")


class ARCEasy(ARC):
    FEW_SHOT = FewShot.get("arc_easy::olmes")

    def __init__(self, config: TaskConfig):
        self.config = config
        super().__init__(dataset_name="ARC-Easy")
