from minieval.datatypes import Metric, Response


class PassAtK(Metric):
    k: int

    def __init__(self, k: int = 1):
        super().__init__()
        self.k = k
        self.name = f'pass@{k}'

    def _compute_metric(self, response: Response) -> bool:
        assert len(response.output) >= self.k, \
            f"Cannot compute pass@k when n < k. n={len(response.output)}, k={self.k}"

        scorers = response.output[0].score.keys()
        response.scores[self.name] = {}

        for scorer in scorers:
            scores = [output.score[scorer] for output in response.output]
            pass_at_k = any(scores[:self.k])
            response.scores[self.name][scorer] = pass_at_k

        return response


class Top1(Metric):
    name = 'top1'

    def __init__(self):
        super().__init__()

    def _compute_metric(self, response: Response) -> bool:
        assert len(response.output) == 1, \
            f"Accuracy only supports single generations"
        
        response.scores[self.name] = {}
        
        scorers = response.output[0].score.keys()
        for scorer in scorers:
            scores = [output.score[scorer] for output in response.output]

            response.scores[self.name][scorer] = scores[0]
        
        return response


class MajAtK(Metric):
    def __init__(self, k: int = 1):
        super().__init__()
        self.k = k
        self.name = f'maj@{k}'
    
    def _compute_metric(self, response: Response) -> Response:
        assert len(response.output) >= self.k, \
            f"Cannot compute pass@k when n < k. n={len(response.output)}, k={self.k}"

        scorers = response.output[0].score.keys()
        response.scores[self.name] = {}

        for scorer in scorers:
            scores = [output.score[scorer] for output in response.output]

            # Get all extracted answers and their counts for this scorer
            answer_counts = {}
            for output in response.output[:self.k]:
                ans = str(output.extracted_answer)
                if ans not in answer_counts:
                    answer_counts[ans] = 0
                answer_counts[ans] += 1

            # Find most common answer
            majority_answer = max(answer_counts.items(), key=lambda x: x[1])[0]

            # Get score for majority answer
            majority_outputs = [
                output for output in response.output[:self.k]
                if str(output.extracted_answer) == majority_answer
            ]
            
            # All outputs with same answer should have same score
            scores = [output.score[scorer] for output in majority_outputs]
            assert len(set(scores)) == 1, "Scores differ for same extracted answer"
            
            response.scores[self.name][scorer] = scores[0]

        return response
