from typing import Callable, ClassVar, Type, TypeVar, Union
from dataclasses import dataclass, field
from typing import Optional, List, Dict, Any


T = TypeVar("T", bound=Type["TaskConfig"])


@dataclass
class Instance:
    """ A single unit of work """
    question: str
    gold_completion: Optional[str] = None
    choices: Optional[list[str]] = None
    solution: Optional[str] = None
    subset: Optional[str] = None
    metadata: dict = field(default_factory=dict)


@dataclass
class LMOutput:
    text: str
    logprobs: Optional[List[Dict[str, Any]]] = None
    extracted_answer: Optional[list] = None
    score: dict = field(default_factory=dict)


@dataclass
class Response:
    input: Instance
    messages: dict
    request: str
    output: list[LMOutput]
    scores: dict = field(default_factory=dict)


@dataclass
class Formatter:
    def build_messages(self, instances):
        return list(map(self._build_message, instances))
    
    def build_requests(self, template_func, messages):
        return list(map(
            lambda msg: self._build_request(template_func, msg), messages
        ))

    def _build_message(self, instance: Instance) -> dict:
        messages = self._build_message(instance)
        return messages
    
    def _build_request(self, template_func: callable, messages: dict) -> str:
        request = template_func(messages, tokenize=False)
        return request

    def _build_message(self):
        raise NotImplementedError()


@dataclass
class Scorer:
    def score_responses(self, responses: list[Response]) -> list[Response]:
        return list(map(self._score_response, responses))
    
    def _score_response(self, response: Response) -> Response:
        input: Instance = response.input
        outputs: list[LMOutput] = response.output

        for output in outputs:
            _ = self._score_response_single(input, output)

        return response
    
    def _score_response_single(self, input: Instance, output: LMOutput) -> float:
        raise NotImplementedError()


@dataclass
class Metric:
    """ Collates scores into metrics. E.g., accuracy, pass@k """
    def compute_metrics(self, responses: list[Response]) -> list[Response]:
        return list(map(self._compute_metric, responses))

    def _compute_metric(self, response: Response) -> bool:
        raise NotImplementedError


@dataclass
class LauncherConfig:
    save_path: str


@dataclass
class ModelConfig:
    name: str
    revision: Optional[str] = None


@dataclass
class TaskConfig:
    alias: str
    formatter: Formatter
    scorer: list[Scorer]
    metric: list[Metric]
    generation_kwargs: Optional[dict] = None
    limit: Optional[int] = None


class Task:
    config: TaskConfig

    HF_PATH: str
    SUBSETS: Optional[list[str]] = None
    INSTRUCTION: Optional[str] = None
    FEW_SHOT: Optional[list] = None
    SEED: Optional[int] = None

    def __init__(self, config: TaskConfig):
        raise NotImplementedError()

    def _process_instance(self, doc: dict) -> Instance:
        raise NotImplementedError()

    def extract_answers(self, generations: list[list[LMOutput]]) -> list[list[LMOutput]]:
        for gen_set in generations:
            for gen in gen_set:
                gen.extracted_answer = self._extract_answer(gen)
        return generations
    
    def _extract_answer(self, gen: LMOutput) -> list[str]:
        return gen.text


class TaskRegistry:
    """ Registry for task aliases. """

    _instance: ClassVar[Union["TaskRegistry", None]] = None
    _named_tasks: ClassVar[dict[str, Type["TaskConfig"]]] = {}
    _task_mapping: ClassVar[dict[str, Type["Task"]]] = {}

    def __new__(cls, *args, **kwargs):
        # singleton pattern
        if cls._instance is None:
            cls._instance = super().__new__(cls)
        return cls._instance

    @classmethod
    def register(cls, task_alias: str, task: Task) -> Callable[[T], T]:
        # instantiate the singleton instance here; it won't get instantiated
        # twice cuz it's a singleton, after all.
        instance = cls()

        def decorator(task_config: T, _task_alias: str = task_alias, _class: Task = task, _instance: "TaskRegistry" = instance) -> T:
            # add to the registry
            _instance._named_tasks[_task_alias] = task_config

            _instance._task_mapping[task_alias] = _class

            # a little bit of a Python crime, but when registering a group,
            # we replace the class name with the task name for the `.name` property.
            task_config.alias = _task_alias  # pyright: ignore
            return task_config

        return decorator

    @classmethod
    def names(cls) -> list[str]:
        return list(cls._named_tasks.keys())

    @classmethod
    def exists(cls, task_alias: str) -> bool:
        return any(cls.search(task_alias))

    @classmethod
    def get_config(cls, task_alias: str) -> "TaskConfig":
        assert cls._instance is not None, "TaskRegistry is not initialized"

        if task_alias not in cls._named_tasks:
            raise ValueError(f"Task {task_alias} not found in the Task Registry!")
        
        task_config_class = cls._named_tasks[task_alias]
        
        kwargs = {"alias": task_alias}

        # Get default values from class attributes if they exist
        for attr_key, attr_val in task_config_class.__dict__.items():
            # Don't include class attributes (like __module__)
            if attr_key.startswith('__'):
                continue
            
            kwargs.update({attr_key: attr_val})
            
        config = task_config_class(**kwargs)
        return config
        
    @classmethod
    def get_task(cls, task_alias: str) -> Type["Task"]:
        if task_alias not in cls._task_mapping:
            raise ValueError(f"Task class for {task_alias} not found in the Task Registry!")
        
        return cls._task_mapping[task_alias]