import os
from typing import Optional, List, Dict, Any

try:
    from vllm import LLM, SamplingParams
    from vllm.outputs import CompletionOutput, RequestOutput
except ImportError:
    raise ImportError("vLLM is required for VLLMBackend. Install with: pip install minieval[vllm]")

from ..datatypes import LMOutput


class VLLMBackend:
    def __init__(self, model_path: str, **kwargs):
        self.model_path = model_path
            
        self.llm = LLM(model=model_path, **kwargs)
        self.quiet_vllm_logger()

    def quiet_vllm_logger():
        os.environ["VLLM_LOGGING_LEVEL"] = "WARNING"
        os.environ["VLLM_LOG_LEVEL"] = "WARNING"
    
    def generate(
        self, 
        requests: List[str], 
        sampling_params: Optional[Dict[str, Any]] = {}
    ) -> List[LMOutput]:        
        sampling_params = sampling_params.copy()
        sampling_params.setdefault('logprobs', 5)
        
        vllm_sampling_params = SamplingParams(**sampling_params)
        outputs: list[RequestOutput] \
            = self.llm.generate(requests, vllm_sampling_params)
        
        results = []
        for output in outputs:
            output_results = []
            for completion in output.outputs:
                output_results.append(LMOutput(
                    text=completion.text,
                    logprobs=completion.logprobs
                ))
            results.append(output_results)
        
        return results
    
    def logprobs(
        self, 
        requests: List[str], 
        continuations: List[str],
        sampling_params: Optional[Dict[str, Any]] = {}
    ) -> List[List[Dict[str, Any]]]: 
        sampling_params = sampling_params.copy()
        sampling_params.setdefault('logprobs', 5)
        sampling_params['max_tokens'] = 1  # Just want logprobs, not generation
        
        results = []
        for request, continuation in zip(requests, continuations):
            
            # Create full prompt with continuation
            full_prompt = request + continuation
            
            vllm_sampling_params = SamplingParams(**sampling_params)
            outputs: list[RequestOutput] \
                = self.llm.generate([full_prompt], vllm_sampling_params)
            
            results = []
            for output in outputs:
                output_results = []
                for completion in output.outputs:
                    output_results.append(LMOutput(
                        text=completion.text,
                        logprobs=completion.logprobs
                    ))
                results.append(output_results)
        
        return results