from typing import Optional, List, Dict, Any
import os
from concurrent.futures import ThreadPoolExecutor
from tqdm import tqdm

try:
    import litellm
except ImportError:
    raise ImportError("litellm is required for LiteLLMBackend. Install with: pip install minieval[litellm]")

from ..datatypes import LMOutput


class LiteLLMBackend:
    API_KEY_MAPPINGS = {
        'OPENAI_API_KEY': 'openai_api_key',
        'ANTHROPIC_API_KEY': 'anthropic_api_key',
        'COHERE_API_KEY': 'cohere_api_key',
        'REPLICATE_API_TOKEN': 'replicate_api_token',
        'HUGGINGFACE_API_KEY': 'huggingface_api_key',
        'TOGETHER_API_KEY': 'together_api_key',
        'AZURE_API_KEY': 'azure_api_key',
        'AZURE_API_BASE': 'azure_api_base',
        'AZURE_API_VERSION': 'azure_api_version',
        'BEDROCK_AWS_ACCESS_KEY_ID': 'aws_access_key_id',
        'BEDROCK_AWS_SECRET_ACCESS_KEY': 'aws_secret_access_key',
        'BEDROCK_AWS_REGION_NAME': 'aws_region_name',
    }

    def __init__(self, model_name: str, **kwargs):
        self.model_name = model_name
        self.kwargs = kwargs
        self._setup_api_keys()
    
    def _setup_api_keys(self):
        for env_var, litellm_key in self.API_KEY_MAPPINGS.items():
            value = os.getenv(env_var)
            if value:
                setattr(litellm, litellm_key, value)
    
    def generate(
        self, 
        requests: List[List[Dict[str, str]]], 
        sampling_params: Optional[Dict[str, Any]] = {}
    ) -> List[LMOutput]:
        sampling_params = sampling_params.copy()
        sampling_params.setdefault('logprobs', True)
        n = sampling_params.pop('repeats', 1)
        max_completion_tokens = sampling_params.pop('max_gen_tokens', None)
        temperature = sampling_params.pop('temperature', None)

        def _generate_single(messages):
            try:
                response = litellm.completion(
                    model=self.model_name,
                    messages=messages,
                    n=n,
                    max_completion_tokens=max_completion_tokens,
                    temperature=temperature,
                    **self.kwargs
                )
                
                results = []
                for choice in response.choices:
                    text = choice.message.content if choice and choice.message else ""
                    logprobs = getattr(choice, 'logprobs', None)
                    results.append(LMOutput(text=text or "", logprobs=logprobs))
                return results
                
            except Exception as e:
                return [RuntimeError(f"LightLLM failed to generate: {e}")]
        
        with ThreadPoolExecutor() as executor:
            results = list(tqdm(
                executor.map(_generate_single, requests),
                total=len(requests),
                desc="Generating responses"
            ))
        
        return results
    
    def logprobs(
        self, 
        requests: List[List[Dict[str, str]]], 
        continuations: List[str],
        sampling_params: Optional[Dict[str, Any]] = {}
    ) -> List[List[Dict[str, Any]]]:
        sampling_params = sampling_params.copy()
        sampling_params.setdefault('logprobs', True)
        sampling_params['max_tokens'] = 1  # Just want logprobs, not generation
        n = sampling_params.pop('repeats', 1)
        max_completion_tokens = sampling_params.pop('max_gen_tokens', None)
        temperature = sampling_params.pop('temperature', None)

        def _logprobs_single(args):
            messages, continuation = args
            try:
                messages_with_continuation = messages.copy()
                if messages_with_continuation:
                    messages_with_continuation[-1] = {
                        "role": messages_with_continuation[-1]["role"],
                        "content": messages_with_continuation[-1]["content"] + continuation
                    }
                
                response = litellm.completion(
                    model=self.model_name,
                    messages=messages_with_continuation,
                    n=n,
                    max_completion_tokens=max_completion_tokens,
                    temperature=temperature,
                    **self.kwargs
                )
                
                results = []
                for choice in response.choices:
                    logprobs = getattr(choice, 'logprobs', None)
                    if logprobs and hasattr(logprobs, 'content'):
                        results.append(logprobs.content if logprobs.content else [])
                    else:
                        results.append([])
                return results
                
            except Exception as e:
                return [RuntimeError(f"LightLLM failed to generate: {e}")]
        
        args = list(zip(requests, continuations))
        
        with ThreadPoolExecutor() as executor:
            results = list(tqdm(
                executor.map(_logprobs_single, args),
                total=len(requests),
                desc="Generating logprobs"
            ))
        
        return results