from enum import Enum
from minieval.formatters import base_model_chat_template


class Backend(str, Enum):
    mock = "mock"
    vllm = "vllm"
    litellm = "litellm"


def init_backend(backend_type, model_name):
    match backend_type:
        case "mock":
            from .mock import MockLLM
            return MockLLM()
        case "vllm":
            from .vllm import VLLMBackend
            return VLLMBackend(model_path=model_name)
        case "litellm":
            from .litellm import LiteLLMBackend
            return LiteLLMBackend(model_name=model_name)
        case _:
            raise ValueError(f"Unknown backend type: {backend_type}")


def init_template(backend_type, model_name):
    match backend_type:
        case Backend.litellm:
            # LiteLLM uses the messages dict directly
            return None
        case Backend.vllm:
            from transformers import AutoTokenizer
            tokenizer = AutoTokenizer.from_pretrained(model_name)
            if hasattr(tokenizer, 'apply_chat_template'):
                return tokenizer.apply_chat_template
            else:
                return base_model_chat_template
        case Backend.mock:
            return base_model_chat_template
        case _:
            raise ValueError(f"No tokenizer logic for backend: {backend_type}")