A deviously simple eval library. Supports all your [favorite tasks](./minieval/tasks/).

### Quick Start

```sh
pip install minieval # we 🤍 uv
```

**CLI Usage**

```sh
# eval with a random solver
minieval -t minerva:cot --tasks.limit 10

# eval with vLLM, like Qwen
minieval -m Qwen/Qwen3-4B -t minerva:cot --tasks.limit 10 --backend vllm

# eval with an API, like GPT-4o
minieval -m gpt-4.1-nano -t gpqa:cot --tasks.limit 10 --backend litellm
```

---

### Usage

**Basic Usage**
```sh
# utility commands
minieval --list

# override args using olmo-core style
minieval -t minerva:selfc \
    --tasks.metric.ks [1, 2] \
    --tasks.generation_kwargs.max_gen_toks 10 \
    --tasks.generation_kwargs.repeats 2 \
    --tasks.generation_kwargs.temperature 0.6
```

**Beaker Usage**

Internally at AI2, `minieval` supports Beaker using Gantry.

```sh
# eval with vLLM on Beaker
minieval \
    --model allenai/OLMo-2-0425-1B \
    --model.revision stage2-ingredient2-step23852-tokens51B \
    --task arc_challenge \
    --task.limit 10 \
    --beaker.workspace ai2/lm-eval \
    --beaker.budget ai2/oe-eval \
    --beaker.priority normal \
    --beaker.retries 3 \
    --beaker.image XXXX \
    --beaker.secrets.hf DAVIDH_HF_TOKEN \
    --beaker.secrets.openai DAVIDH_OPENAI_API_KEY \
    --beaker.secrets.aws_lambda_access_key_id LAMBDA_AWS_ACCESS_KEY_ID \
    --beaker.secrets.aws_lambda_access_key_secret LAMBDA_AWS_ACCESS_KEY_SECRET \
    --beaker.follow
```

**Python Usage**

```python
from minieval.tasks.minerva import Math500
from minieval.formats import CoT
from minieval.metrics import passAtK

from vllm import LLM, CompletionOutput, RequestOutput, SamplingParams

llm = LLM(model_path)
sampling_params = SamplingParams(
    temperature=0, 
    max_tokens=1024,
)

task = Math500() # e.g., MMLU, HumanEval
formatter = CoT() # e.g., RC, MC, BPB, CoT, Gen
metric = passAtK(ks=[1, 2, 4]) # e.g., majAtK

# build hf dataset into standard instance format
instances = task.requests

# apply chat template
messages = formatter.build_messages(instances)
requests = formatter.build_requests(messages)

# generate responses
generations = llm.generate(requests, sampling_params)

# extract answers (if applicable, e.g., CoT)
generations = task.extract_outputs(generations)

# compile responses
responses = []
for inst, msg, req, gen in zip(instances, messages, requests, generations):
    responses += [Response(input=inst, messages=msg, request=req, output=gen)]

# grade respones
instance_scores = metric.grade_responses(responses)
dataset_scores  = metric.compute_metric()
```

**Local Install**

```sh
pip install -e ".[all]"
```

```sh
# For development, run without a model to use a random solver
minieval -t minerva
```

---

### About

Design principles are based on OAI's [nanoeval](https://github.com/openai/preparedness/tree/main/project/nanoeval):

- **Minimal indirection.** You should be able to implement and understand an eval in 100 lines.
- **Separation of concerns.** Keep data loading away from completions/parsing/different ways of running an eval.
- **Fast iteration and testability.** minievals should import in less than a second and be testable without a live LLM backend.
- **High performance.** Minieval should max out the compute resources available to it.

Primitives:

- `Config` - A set of task aliases and runtime characteristics of how to run it (i.e. concurrency, recording, other administrivia)
- `TaskRegistry` - Enumerates the set of task aliases, which contains information for a task, formatting and scoring. Task aliases can be configured in code or on the CLI using config overrides.
- `Task` - An eval to run and the characteristics of extracting and scoring a result.
- `Instance` - A single scoreable unit of work.
- `Formatter` - Converts an `Instance` (the question, choices, metadata) to a `Request` (the text input to a model) in a particular eval format. For example, there may be different ways to prompt a model to answer a multiple-choice question (i.e. looking at logits, few-shot prompting, etc)
- `Backend` - A strategy (usually involving sampling a model) to go from a `Request` to a `Result` that can be scored. 