from typing import List, Optional, Sequence
import argparse
import os
import shutil
import sys
from vanillaplusjs.build.graph import FileDependencyGraph


def main(args: Sequence[str]):
    argparser = argparse.ArgumentParser(
        prog="vanillajsplus clean",
        description="Removes the generated files (out/, artifacts/)",
    )
    argparser.add_argument(
        "--folder",
        type=str,
        default=".",
        help="The folder containing vanillaplusjs.json",
    )
    argparser.add_argument(
        "--placeholders",
        action="store_true",
        help="Removes any generated placeholders without confirmation",
    )
    argparser.add_argument(
        "--no-placeholders",
        action="store_true",
        help="Disables the removal of generated placeholders without confirmation",
    )

    args = argparser.parse_args(args)

    placeholders: Optional[bool] = None
    if args.placeholders:
        placeholders = True
    elif args.no_placeholders:
        placeholders = False

    if placeholders is None:
        placeholders = prompt_placeholders(args.folder)

    clean(args.folder, placeholders)


def clean(folder: str, placeholders: bool) -> None:
    """Removes the generated files in the vanillaplusjs project at the given
    folder.

    Args:
        folder (str): The folder containing vanillaplusjs.json
        placeholders (bool): Whether to remove generated placeholders or not
    """
    if not os.path.exists(os.path.join(folder, "vanillaplusjs.json")):
        print("vanillaplusjs.json not found. Call 'vanillaplusjs init' to create it.")
        sys.exit(1)

    if placeholders:
        placeholder_graph_path = os.path.join(folder, "out", "placeholder_graph.json")
        if os.path.exists(placeholder_graph_path):
            with open(placeholder_graph_path, "r") as f:
                placeholder_graph = FileDependencyGraph.load(f)

            for file, node in placeholder_graph.nodes.items():
                if node.parents:
                    os.remove(os.path.join(folder, file))

    shutil.rmtree(os.path.join(folder, "out"), ignore_errors=True)
    shutil.rmtree(os.path.join(folder, "artifacts"), ignore_errors=True)


def prompt_placeholders(folder: str) -> bool:
    """Prompts the user if they want to remove the generated placeholders.
    This will intelligently skip prompting if the choice is irrelevant, and
    will provide relevant information to the user.

    Args:
        folder (str): The folder to clean

    Returns:
        bool: The users choice, if it matters, otherwise an arbitrary value
    """
    placeholder_graph_path = os.path.join(folder, "out", "placeholder_graph.json")
    if not os.path.exists(placeholder_graph_path):
        return False

    with open(placeholder_graph_path, "r") as f:
        placeholder_graph = FileDependencyGraph.load(f)

    if len(placeholder_graph.nodes) == 0:
        return False

    placeholders: List[str] = []
    for file, node in placeholder_graph.nodes.items():
        if node.parents:
            placeholders.append(file)

    if len(placeholders) == 0:
        return False

    print("This prompt can be suppressed by --placeholders or --no-placeholders.")
    print("The following placeholders would be removed:")
    for idx, placeholder in enumerate(placeholders):
        print(
            f"{idx + 1}. {placeholder} (generated by {placeholder_graph.get_parents(placeholder)})"
        )

    print("Do you want to remove them? (y/n) ", end="")

    while True:
        choice = input().lower()
        if choice == "y":
            return True
        elif choice == "n":
            return False
        else:
            print("Please enter y or n: ", end="")
