"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
require("@aws-cdk/assert/jest");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const mediastore = require("aws-cdk-lib/aws-mediastore");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const s3 = require("aws-cdk-lib/aws-s3");
const lib_1 = require("../lib");
// --------------------------------------------------------------
// Test the default deployment pattern variables
// --------------------------------------------------------------
test('Test the default deployment pattern variables', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {});
    // Assertion
    expect(cloudFrontToMediaStore.cloudFrontWebDistribution).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.mediaStoreContainer).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontLoggingBucket).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontOriginRequestPolicy).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontFunction).not.toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment without HTTP security headers
// --------------------------------------------------------------
test('Test the deployment without HTTP security headers', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        insertHttpSecurityHeaders: false
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'testcloudfrontmediastoreCloudFrontOriginAccessIdentity966405A0'
                            }
                        }
                    ]
                }
            ]
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontFunction).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with existing MediaStore container
// --------------------------------------------------------------
test('Test the deployment with existing MediaStore container', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MyMediaStoreContainer', {
        containerName: 'MyMediaStoreContainer'
    });
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        existingMediaStoreContainerObj: mediaStoreContainer
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'MyMediaStoreContainer',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        ContainerName: 'MyMediaStoreContainer'
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with the user provided MediaStore properties
// --------------------------------------------------------------
test('Test the deployment with the user provided MediaStore properties', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        mediaStoreContainerProps: {
            containerName: 'MyMediaStoreContainer',
            policy: '{}',
            lifecyclePolicy: '{}',
            corsPolicy: [],
            metricPolicy: {
                containerLevelMetrics: 'DISABLED'
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        ContainerName: 'MyMediaStoreContainer',
        Policy: '{}',
        LifecyclePolicy: '{}',
        CorsPolicy: [],
        MetricPolicy: {
            ContainerLevelMetrics: 'DISABLED'
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with the user provided CloudFront properties
// --------------------------------------------------------------
test('Test the deployment with the user provided CloudFront properties', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        cloudFrontDistributionProps: {
            defaultBehavior: {
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.HTTPS_ONLY,
                allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
                cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS',
                    'PUT',
                    'PATCH',
                    'POST',
                    'DELETE'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'https-only'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'testcloudfrontmediastoreCloudFrontOriginAccessIdentity966405A0'
                            }
                        }
                    ]
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        AccessLoggingEnabled: true,
        ContainerName: {
            Ref: 'AWS::StackName'
        },
        CorsPolicy: [
            {
                AllowedHeaders: ['*'],
                AllowedMethods: ['GET'],
                AllowedOrigins: ['*'],
                ExposeHeaders: ["*"],
                MaxAgeSeconds: 3000
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
        CloudFrontOriginAccessIdentityConfig: {
            Comment: {
                'Fn::Join': [
                    '',
                    [
                        'access-identity-',
                        {
                            Ref: 'AWS::Region',
                        },
                        '-',
                        {
                            Ref: 'AWS::StackName',
                        }
                    ]
                ]
            }
        }
    });
});
// --------------------------------------------------------------
// Cloudfront logging bucket with destroy removal policy and auto delete objects
// --------------------------------------------------------------
test('Cloudfront logging bucket with destroy removal policy and auto delete objects', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.CloudFrontToMediaStore(stack, 'cloudfront-mediatstore', {
        cloudFrontLoggingBucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    });
    expect(stack).toHaveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "cloudfrontmediatstoreCloudfrontLoggingBucket2565C68A"
        }
    });
});
// --------------------------------------------------------------
// Cloudfront logging bucket error providing existing log bucket and logBucketProps
// --------------------------------------------------------------
test('Cloudfront logging bucket error when providing existing log bucket and logBucketProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const logBucket = new s3.Bucket(stack, 'cloudfront-log-bucket', {});
    const app = () => {
        new lib_1.CloudFrontToMediaStore(stack, 'cloudfront-s3', {
            cloudFrontDistributionProps: {
                logBucket
            },
            cloudFrontLoggingBucketProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                autoDeleteObjects: true
            }
        });
    };
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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