"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const secretsmanager = require("../lib");
module.exports = {
    'secret rotation single user'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const target = new ec2.Connections({
            defaultPort: ec2.Port.tcp(3306),
            securityGroups: [new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc })],
        });
        const excludeCharacters = ' ;+%{}' + '@\'"`/\\#'; // DMS and BASH problem chars
        // WHEN
        new secretsmanager.SecretRotation(stack, 'SecretRotation', {
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            secret,
            target,
            vpc,
            excludeCharacters: excludeCharacters,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            Description: 'from SecretRotationSecurityGroupAEC520AB:3306',
            FromPort: 3306,
            GroupId: {
                'Fn::GetAtt': [
                    'SecurityGroupDD263621',
                    'GroupId',
                ],
            },
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SecretRotationSecurityGroup9985012B',
                    'GroupId',
                ],
            },
            ToPort: 3306,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::RotationSchedule', {
            SecretId: {
                Ref: 'SecretA720EF05',
            },
            RotationLambdaARN: {
                'Fn::GetAtt': [
                    'SecretRotationA9FFCFA9',
                    'Outputs.RotationLambdaARN',
                ],
            },
            RotationRules: {
                AutomaticallyAfterDays: 30,
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: 'Default/SecretRotation/SecurityGroup',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Serverless::Application', {
            Location: {
                ApplicationId: 'arn:aws:serverlessrepo:us-east-1:297356227824:applications/SecretsManagerRDSMySQLRotationSingleUser',
                SemanticVersion: '1.1.60',
            },
            Parameters: {
                endpoint: {
                    'Fn::Join': [
                        '',
                        [
                            'https://secretsmanager.',
                            {
                                Ref: 'AWS::Region',
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix',
                            },
                        ],
                    ],
                },
                functionName: 'SecretRotation',
                excludeCharacters: excludeCharacters,
                vpcSecurityGroupIds: {
                    'Fn::GetAtt': [
                        'SecretRotationSecurityGroup9985012B',
                        'GroupId',
                    ],
                },
                vpcSubnetIds: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                            },
                            ',',
                            {
                                Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                            },
                        ],
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::ResourcePolicy', {
            ResourcePolicy: {
                Statement: [
                    {
                        Action: 'secretsmanager:DeleteSecret',
                        Effect: 'Deny',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
            SecretId: {
                Ref: 'SecretA720EF05',
            },
        }));
        test.done();
    },
    'secret rotation multi user'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const masterSecret = new secretsmanager.Secret(stack, 'MasterSecret');
        const target = new ec2.Connections({
            defaultPort: ec2.Port.tcp(3306),
            securityGroups: [new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc })],
        });
        // WHEN
        new secretsmanager.SecretRotation(stack, 'SecretRotation', {
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            secret,
            masterSecret,
            target,
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Serverless::Application', {
            Parameters: {
                endpoint: {
                    'Fn::Join': [
                        '',
                        [
                            'https://secretsmanager.',
                            {
                                Ref: 'AWS::Region',
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix',
                            },
                        ],
                    ],
                },
                functionName: 'SecretRotation',
                vpcSecurityGroupIds: {
                    'Fn::GetAtt': [
                        'SecretRotationSecurityGroup9985012B',
                        'GroupId',
                    ],
                },
                vpcSubnetIds: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                            },
                            ',',
                            {
                                Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                            },
                        ],
                    ],
                },
                masterSecretArn: {
                    Ref: 'MasterSecretA11BF785',
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::ResourcePolicy', {
            ResourcePolicy: {
                Statement: [
                    {
                        Action: 'secretsmanager:DeleteSecret',
                        Effect: 'Deny',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
            SecretId: {
                Ref: 'MasterSecretA11BF785',
            },
        }));
        test.done();
    },
    'throws when connections object has no default port range'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', {
            vpc,
        });
        // WHEN
        const target = new ec2.Connections({
            securityGroups: [securityGroup],
        });
        // THEN
        test.throws(() => new secretsmanager.SecretRotation(stack, 'Rotation', {
            secret,
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            vpc,
            target,
        }), /`target`.+default port range/);
        test.done();
    },
    'throws when master secret is missing for a multi user application'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const target = new ec2.Connections({
            defaultPort: ec2.Port.tcp(3306),
            securityGroups: [new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc })],
        });
        // THEN
        test.throws(() => new secretsmanager.SecretRotation(stack, 'Rotation', {
            secret,
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            vpc,
            target,
        }), /The `masterSecret` must be specified for application using the multi user scheme/);
        test.done();
    },
    'rotation function name does not exceed 64 chars'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const target = new ec2.Connections({
            defaultPort: ec2.Port.tcp(3306),
            securityGroups: [new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc })],
        });
        // WHEN
        const id = 'SecretRotation'.repeat(5);
        new secretsmanager.SecretRotation(stack, id, {
            application: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            secret,
            target,
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Serverless::Application', {
            Parameters: {
                endpoint: {
                    'Fn::Join': [
                        '',
                        [
                            'https://secretsmanager.',
                            {
                                Ref: 'AWS::Region',
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix',
                            },
                        ],
                    ],
                },
                functionName: 'RotationSecretRotationSecretRotationSecretRotationSecretRotation',
                vpcSecurityGroupIds: {
                    'Fn::GetAtt': [
                        'SecretRotationSecretRotationSecretRotationSecretRotationSecretRotationSecurityGroupBFCB171A',
                        'GroupId',
                    ],
                },
                vpcSubnetIds: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                            },
                            ',',
                            {
                                Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                            },
                        ],
                    ],
                },
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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