#!/usr/bin/env python3
"""
Command-line interface for Terminal Chat Server.
"""

import argparse
import sys
from .server import TerminalChatServer

def main():
    """Main CLI entry point."""
    parser = argparse.ArgumentParser(
        description="Terminal Chat Server - A terminal-based chat server with room access codes",
        epilog="Example: terminal-chat-server --port 8888 --host 0.0.0.0"
    )
    
    parser.add_argument(
        '--host', 
        default='0.0.0.0',
        help='Host address to bind the server (default: 0.0.0.0)'
    )
    
    parser.add_argument(
        '--port', 
        type=int,
        default=9999,
        help='Port to bind the server (default: 9999)'
    )
    
    parser.add_argument(
        '--version',
        action='store_true',
        help='Show version information and exit'
    )
    
    args = parser.parse_args()
    
    if args.version:
        from terminal_chat_server import __version__, __author__
        print(f"Terminal Chat Server v{__version__}")
        print(f"Author: {__author__}")
        sys.exit(0)
    
    try:
        server = TerminalChatServer(host=args.host, port=args.port)
        server.start_server()
    except KeyboardInterrupt:
        print("\n🛑 Server stopped by user")
        sys.exit(0)
    except Exception as e:
        print(f"❌ Failed to start server: {e}")
        sys.exit(1)

if __name__ == "__main__":
    main()
