from __future__ import annotations

from dataclasses import dataclass, field
from enum import Enum
from typing import List, Dict, Optional, Any

import dacite


class Provider(Enum):
    OPENAI = "OPENAI"
    COHERE = "COHERE"
    UNKNOWN = "UNKNOWN"


class FinishReason(Enum):
    LENGTH = "LENGTH"
    STOP = "STOP"
    UNKNOWN = "UNKNOWN"


@dataclass
class TokenLogProbs:
    token: str
    logprob: Optional[float]
    top_logprobs: Optional[Dict[str, float]] = field(repr=False)
    text_offset: int = field(repr=False)


@dataclass
class LogProbs:
    tokens: List[TokenLogProbs] = field(repr=False)
    likelihood: float


@dataclass
class Completion:
    id: str = field(repr=False)
    external_id: Optional[str] = field(repr=False)
    model_version_id: str = field(repr=False)
    text: str
    finish_reason: FinishReason = field(repr=False)
    logprobs: Optional[LogProbs] = field(repr=False)


@dataclass
class CompletionResultData:
    completions: List[Completion]


@dataclass
class CompletionResultError:
    message: str


@dataclass
class CompletionResult:
    data: CompletionResultData
    error: Optional[CompletionResultError]


@dataclass
class GenerateResult:
    # The completions generated by the LLM provider, normalized to Vellum's schema.
    results: List[CompletionResult]

    @classmethod
    def from_raw(cls, raw_result: dict) -> GenerateResult:
        return dacite.from_dict(data_class=cls, data=raw_result, config=dacite.Config(cast=[Provider, FinishReason]))

    @property
    def texts(self) -> List[str]:
        return [completion.text for result in self.results for completion in result.data.completions]

    @property
    def text(self) -> str:
        if len(self.texts) != 1:
            raise ValueError(f"Expected exactly one completion, but got {len(self.texts)}")

        return self.texts[0]


@dataclass
class GenerateRequest:
    # A mapping between prompt template variables and the values they should be substituted with
    input_values: Dict[str, Any]
    # Optionally provide an external ID for each prompt. This ID will be returned in the result
    # and can later be used to uniquely identify the generation request that produced the result.
    external_ids: Optional[List[str]] = None
