from ..value_set import ValueSet


class AnaphylacticReactionToCommonBakersYeast(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  an anaphylactic reaction to baker's yeast.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of an anaphylactic reaction to baker's yeast.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = "Anaphylactic Reaction to Common Baker's Yeast"
    OID = '2.16.840.1.113883.3.464.1003.199.12.1032'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    SNOMEDCT = {
        '34015007',  # Bakers' asthma (disorder)
        '419447004',  # Allergic reaction caused by flour dust (disorder)
    }


class AnaphylacticReactionToDtapVaccine(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  an anaphylactic reaction to DTaP (diphtheria, tetanus, and acellular pertussis) vaccine.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of an anaphylactic reaction to DTaP vaccine.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = 'Anaphylactic Reaction to DTaP Vaccine'
    OID = '2.16.840.1.113883.3.464.1003.199.12.1031'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    SNOMEDCT = {
        '219084006',  # Adverse reaction to vaccine product containing Clostridium tetani antigen (disorder)
        '293108006',  # Adverse reaction to vaccine product containing Clostridium tetani and Corynebacterium diphtheriae antigens (disorder)
        '428281000124107',  # Anaphylaxis due to diphtheria and tetanus vaccine (disorder)
        '428291000124105',  # Anaphylaxis due to tetanus, diphtheria and acellular pertussis vaccine (disorder)
    }


class DisordersOfTheImmuneSystem(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of conditions affecting the immune system.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of disorders of the immune system including disorders of the immunoglobulins and the complement system.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = 'Disorders of the Immune System'
    OID = '2.16.840.1.113883.3.464.1003.120.12.1001'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'D80.0',  # Hereditary hypogammaglobulinemia
        'D80.1',  # Nonfamilial hypogammaglobulinemia
        'D80.2',  # Selective deficiency of immunoglobulin A [IgA]
        'D80.3',  # Selective deficiency of immunoglobulin G [IgG] subclasses
        'D80.4',  # Selective deficiency of immunoglobulin M [IgM]
        'D80.5',  # Immunodeficiency with increased immunoglobulin M [IgM]
        'D80.6',  # Antibody deficiency with near-normal immunoglobulins or with hyperimmunoglobulinemia
        'D80.7',  # Transient hypogammaglobulinemia of infancy
        'D80.8',  # Other immunodeficiencies with predominantly antibody defects
        'D80.9',  # Immunodeficiency with predominantly antibody defects, unspecified
        'D81.0',  # Severe combined immunodeficiency [SCID] with reticular dysgenesis
        'D81.1',  # Severe combined immunodeficiency [SCID] with low T- and B-cell numbers
        'D81.2',  # Severe combined immunodeficiency [SCID] with low or normal B-cell numbers
        'D81.4',  # Nezelof's syndrome
        'D81.6',  # Major histocompatibility complex class I deficiency
        'D81.7',  # Major histocompatibility complex class II deficiency
        'D81.89',  # Other combined immunodeficiencies
        'D81.9',  # Combined immunodeficiency, unspecified
        'D82.0',  # Wiskott-Aldrich syndrome
        'D82.1',  # Di George's syndrome
        'D82.2',  # Immunodeficiency with short-limbed stature
        'D82.3',  # Immunodeficiency following hereditary defective response to Epstein-Barr virus
        'D82.4',  # Hyperimmunoglobulin E [IgE] syndrome
        'D82.8',  # Immunodeficiency associated with other specified major defects
        'D82.9',  # Immunodeficiency associated with major defect, unspecified
        'D83.0',  # Common variable immunodeficiency with predominant abnormalities of B-cell numbers and function
        'D83.1',  # Common variable immunodeficiency with predominant immunoregulatory T-cell disorders
        'D83.2',  # Common variable immunodeficiency with autoantibodies to B- or T-cells
        'D83.8',  # Other common variable immunodeficiencies
        'D83.9',  # Common variable immunodeficiency, unspecified
        'D84.0',  # Lymphocyte function antigen-1 [LFA-1] defect
        'D84.1',  # Defects in the complement system
        'D84.8',  # Other specified immunodeficiencies
        'D84.81',  # Immunodeficiency due to conditions classified elsewhere
        'D84.821',  # Immunodeficiency due to drugs
        'D84.822',  # Immunodeficiency due to external causes
        'D84.89',  # Other immunodeficiencies
        'D84.9',  # Immunodeficiency, unspecified
        'D89.3',  # Immune reconstitution syndrome
        'D89.810',  # Acute graft-versus-host disease
        'D89.811',  # Chronic graft-versus-host disease
        'D89.812',  # Acute on chronic graft-versus-host disease
        'D89.813',  # Graft-versus-host disease, unspecified
        'D89.82',  # Autoimmune lymphoproliferative syndrome [ALPS]
        'D89.831',  # Cytokine release syndrome, grade 1
        'D89.832',  # Cytokine release syndrome, grade 2
        'D89.833',  # Cytokine release syndrome, grade 3
        'D89.834',  # Cytokine release syndrome, grade 4
        'D89.835',  # Cytokine release syndrome, grade 5
        'D89.839',  # Cytokine release syndrome, grade unspecified
        'D89.89',  # Other specified disorders involving the immune mechanism, not elsewhere classified
        'D89.9',  # Disorder involving the immune mechanism, unspecified
    }
    SNOMEDCT = {
        '18827005',  # Complement abnormality (disorder)
        '23238000',  # Common variable agammaglobulinemia (disorder)
        '24419001',  # Disorder of complement (disorder)
        '24743004',  # Complement deficiency disease (disorder)
        '36070007',  # Wiskott-Aldrich syndrome (disorder)
        '39674000',  # Familial C3B inhibitor deficiency syndrome (disorder)
        '55602000',  # Nezelof's syndrome (disorder)
        '81166004',  # Properdin deficiency disease (disorder)
        '82286005',  # Hyperimmunoglobulin M syndrome (disorder)
        '82966003',  # Hereditary angioedema (disorder)
        '88714009',  # Transient hypogammaglobulinemia of infancy (disorder)
        '190979003',  # Selective immunoglobulin A deficiency (disorder)
        '190980000',  # Selective immunoglobulin M deficiency (disorder)
        '190981001',  # Selective immunoglobulin G deficiency (disorder)
        '190996002',  # Severe combined immunodeficiency with reticular dysgenesis (disorder)
        '190997006',  # Severe combined immunodeficiency with low T- and B-cell numbers (disorder)
        '190998001',  # Severe combined immunodeficiency with low or normal B-cell numbers (disorder)
        '191001007',  # Major histocompatibility complex class I deficiency (disorder)
        '191002000',  # Major histocompatibility complex class II deficiency (disorder)
        '191011000',  # Common variable immunodeficiency with predominant abnormalities of B-cell numbers and functions (disorder)
        '191012007',  # Common variable immunodeficiency with predominant immunoregulatory T-cell disorders (disorder)
        '191013002',  # Common variable immunodeficiency with autoantibodies to B- or T-cells (disorder)
        '191018006',  # Lymphocyte function antigen-1 defect (disorder)
        '234416002',  # X-linked hypogammaglobulinemia (disorder)
        '234532001',  # Immunodeficiency disorder (disorder)
        '234562007',  # Anti-protein antibody deficiency (disorder)
        '234593008',  # Classical complement pathway abnormality (disorder)
        '234594002',  # Complement 1q deficiency (disorder)
        '234595001',  # Complement 1q beta chain deficiency (disorder)
        '234596000',  # Complement 1q dysfunction (disorder)
        '234597009',  # Complement 1r deficiency (disorder)
        '234598004',  # Complement 1s deficiency (disorder)
        '234599007',  # Complement 2 deficiency (disorder)
        '234600005',  # Complement 4 deficiency (disorder)
        '234601009',  # Complement 4A deficiency (disorder)
        '234602002',  # Complement 4B deficiency (disorder)
        '234604001',  # Alternative pathway deficiency (disorder)
        '234605000',  # Factor B deficiency (disorder)
        '234607008',  # Factor D deficiency (disorder)
        '234608003',  # Terminal component deficiency (disorder)
        '234609006',  # Complement 5 deficiency (disorder)
        '234611002',  # Complement 6 deficiency (disorder)
        '234612009',  # Complement 7 deficiency (disorder)
        '234613004',  # Combined complement 6 and 7 deficiencies (disorder)
        '234614005',  # Complement 8 beta chain deficiency (disorder)
        '234615006',  # Complement 8 beta chain dysfunction (disorder)
        '234616007',  # Complement 8 alpha-gamma deficiency (disorder)
        '234617003',  # Complement 9 deficiency (disorder)
        '234618008',  # Complement regulatory factor defect (disorder)
        '234619000',  # Hereditary C1 esterase inhibitor deficiency - deficient factor (disorder)
        '234620006',  # Hereditary C1 esterase inhibitor deficiency - dysfunctional factor (disorder)
        '234621005',  # Factor I deficiency (disorder)
        '234622003',  # Factor H deficiency (disorder)
        '234623008',  # Complement 4 binding protein deficiency (disorder)
        '234624002',  # Decay accelerating factor deficiency (disorder)
        '234625001',  # Homologous restriction factor deficiency (disorder)
        '234626000',  # Complement 5a inhibitor deficiency (disorder)
        '234627009',  # Anaphylotoxin inactivator deficiency (disorder)
        '234628004',  # Complement receptor deficiency (disorder)
        '234629007',  # Complement receptor 1 deficiency (disorder)
        '234630002',  # Complement receptor 3 deficiency (disorder)
        '234631003',  # Immunodeficiency with major anomalies (disorder)
        '234646005',  # Graft-versus-host disease (disorder)
        '263661007',  # Complement 5 dysfunction (disorder)
        '402355000',  # Acute graft-versus-host disease (disorder)
        '402356004',  # Chronic graft-versus-host disease (disorder)
        '402357008',  # Lichenoid chronic graft-versus-host disease (disorder)
        '402358003',  # Sclerodermoid chronic graft-versus-host disease (disorder)
        '402360001',  # Materno-fetal graft-versus-host disease (disorder)
        '403835002',  # X-linked hyper-immunoglobulin M syndrome (disorder)
        '403836001',  # Autosomal recessive hyperimmunoglobulin M syndrome (disorder)
        '403837005',  # Wiskott-Aldrich autosomal dominant variant syndrome (disorder)
        '414029004',  # Disorder of immune function (disorder)
        '426202004',  # Immune reconstitution syndrome (disorder)
        '427167008',  # Hereditary angioedema with normal C1 esterase inhibitor activity (disorder)
        '442459007',  # Combined immunodeficiency disease (disorder)
        '444547006',  # Graft versus host disease of skin (disorder)
        '449187006',  # Kappa light chain deficiency (disorder)
        '702444009',  # Autoimmune lymphoproliferative syndrome (disorder)
        '719156006',  # X-linked intellectual disability with hypogammaglobulinemia and progressive neurological deterioration syndrome (disorder)
        '720853005',  # Severe combined immunodeficiency, microcephaly, growth retardation, sensitivity to ionizing radiation syndrome (disorder)
        '722067005',  # Severe combined immunodeficiency with hypereosinophilia (disorder)
        '763623001',  # Severe combined immunodeficiency due to cytidine 5-prime triphosphate synthetase 1 deficiency (disorder)
        '765188009',  # Severe combined immunodeficiency due to complete recombination-activating gene 1 and/or recombination-activating gene 2 deficiency (disorder)
        '771078002',  # Immunodeficiency due to mannan binding lectin serine peptidase 2 deficiency (disorder)
        '771443008',  # Complement component 3 deficiency (disorder)
        '778027003',  # Primary CD59 deficiency (disorder)
        '783007005',  # Recurrent Neisseria infection due to factor D deficiency (disorder)
        '783248004',  # Hyperimmunoglobulin M syndrome with susceptibility to opportunistic infection (disorder)
        '783249007',  # Hyperimmunoglobulin M syndrome without susceptibility to opportunistic infection (disorder)
        '783617001',  # Severe combined immunodeficiency due to lymphocyte-specific protein-tyrosine kinase deficiency (disorder)
        '783621008',  # Immunodeficiency with factor I anomaly (disorder)
        '789777007',  # Short-limb skeletal dysplasia with severe combined immunodeficiency (disorder)
        '846606007',  # Graft versus host disease of liver (disorder)
        '121121000119106',  # Acute graft-versus-host reaction following bone marrow transplant (disorder)
        '121131000119109',  # Chronic graft versus host disease after transplantation of bone marrow (disorder)
        '128631000119109',  # Chronic graft-versus-host disease following kidney transplant (disorder)
    }


class EncephalopathyDueToChildhoodVaccination(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  encephalopathy due to childhood vaccination.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of encephalopathy due to childhood vaccination.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = 'Encephalopathy due to Childhood Vaccination'
    OID = '2.16.840.1.113883.3.464.1003.114.12.1007'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'G04.32',  # Postimmunization acute necrotizing hemorrhagic encephalopathy
    }
    SNOMEDCT = {
        '192704009',  # Post-immunization encephalitis (disorder)
        '192705005',  # Post-bacillus Calmette-Guerin vaccination encephalitis (disorder)
        '192706006',  # Post typhoid vaccination encephalitis (disorder)
        '192707002',  # Post paratyphoid vaccination encephalitis (disorder)
        '192708007',  # Post cholera vaccination encephalitis (disorder)
        '192709004',  # Post plague vaccination encephalitis (disorder)
        '192710009',  # Post tetanus vaccination encephalitis (disorder)
        '192711008',  # Post diphtheria vaccination encephalitis (disorder)
        '192712001',  # Post pertussis vaccination encephalitis (disorder)
        '192713006',  # Post smallpox vaccination encephalitis (disorder)
        '192714000',  # Post rabies vaccination encephalitis (disorder)
        '192715004',  # Post typhus vaccination encephalitis (disorder)
        '192716003',  # Post yellow fever vaccination encephalitis (disorder)
        '192717007',  # Post measles vaccination encephalitis (disorder)
        '192718002',  # Post polio vaccination encephalitis (disorder)
        '192719005',  # Post mumps vaccination encephalitis (disorder)
        '192720004',  # Post rubella vaccination encephalitis (disorder)
        '192721000',  # Post influenza vaccination encephalitis (disorder)
        '192722007',  # Post hepatitis A vaccination encephalitis (disorder)
        '192723002',  # Post hepatitis B vaccination encephalitis (disorder)
        '192724008',  # Post mixed vaccination encephalitis (disorder)
    }


class Hiv(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of human immunodeficiency virus (HIV) disease.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of HIV and acquired immunodeficiency syndrome.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS153v10, CMS117v10, CMS646v2
    """

    VALUE_SET_NAME = 'HIV'
    OID = '2.16.840.1.113883.3.464.1003.120.12.1003'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'B20',  # Human immunodeficiency virus [HIV] disease
        'B97.35',  # Human immunodeficiency virus, type 2 [HIV 2] as the cause of diseases classified elsewhere
        'Z21',  # Asymptomatic human immunodeficiency virus [HIV] infection status
    }
    ICD9CM = {
        '042',  # Human immunodeficiency virus [HIV] disease
        '079.53',  # Human immunodeficiency virus, type 2 [HIV-2]
        'V08',  # Asymptomatic human immunodeficiency virus [HIV] infection status
    }
    SNOMEDCT = {
        '40780007',  # Human immunodeficiency virus I infection (disorder)
        '48794007',  # Human immunodeficiency virus infection with infectious mononucleosis-like syndrome (disorder)
        '52079000',  # Congenital human immunodeficiency virus infection (disorder)
        '62479008',  # Acquired immune deficiency syndrome (disorder)
        '79019005',  # Human immunodeficiency virus II infection (disorder)
        '86406008',  # Human immunodeficiency virus infection (disorder)
        '87117006',  # Human immunodeficiency virus infection with acute lymphadenitis (disorder)
        '91923005',  # Acquired immunodeficiency syndrome virus infection associated with pregnancy (disorder)
        '91947003',  # Asymptomatic human immunodeficiency virus infection (disorder)
        '91948008',  # Asymptomatic human immunodeficiency virus infection in pregnancy (disorder)
        '111880001',  # Acute human immunodeficiency virus infection (disorder)
        '165816005',  # Human immunodeficiency virus positive (finding)
        '186706006',  # Human immunodeficiency virus infection constitutional disease (disorder)
        '186707002',  # Human immunodeficiency virus infection with neurological disease (disorder)
        '186708007',  # Human immunodeficiency virus infection with secondary clinical infectious disease (disorder)
        '230180003',  # Human immunodeficiency virus leukoencephalopathy (disorder)
        '230202002',  # Vacuolar myelopathy (disorder)
        '230598008',  # Neuropathy caused by human immunodeficiency virus (disorder)
        '235009000',  # Human immunodeficiency virus-associated periodontitis (disorder)
        '235726002',  # Human immunodeficiency virus enteropathy (disorder)
        '236406007',  # Acquired immune deficiency syndrome-related nephropathy (disorder)
        '240103002',  # Human immunodeficiency virus myopathy (disorder)
        '276665006',  # Congenital acquired immune deficiency syndrome (disorder)
        '276666007',  # Congenital human immunodeficiency virus positive status syndrome (disorder)
        '315019000',  # Human immunodeficiency virus infection with aseptic meningitis (disorder)
        '397763006',  # Human immunodeficiency virus encephalopathy (disorder)
        '398329009',  # Human immunodeficiency virus encephalitis (disorder)
        '402901009',  # Oral hairy leukoplakia associated with human immunodeficiency virus disease (disorder)
        '405631006',  # Pediatric human immunodeficiency virus infection (disorder)
        '406109008',  # Human immunodeficiency virus enzyme-linked immunosorbent assay test positive (finding)
        '414604009',  # Leukoplakia of tongue associated with human immunodeficiency virus disease (disorder)
        '416729007',  # Neutropenia associated with acquired immunodeficiency syndrome (disorder)
        '420244003',  # Encephalitis associated with acquired immunodeficiency syndrome (disorder)
        '420281004',  # Skin rash associated with acquired immunodeficiency syndrome (disorder)
        '420302007',  # Reticulosarcoma associated with acquired immunodeficiency syndrome (disorder)
        '420308006',  # Retinal vascular changes associated with acquired immunodeficiency syndrome (disorder)
        '420321004',  # Intestinal malabsorption associated with acquired immunodeficiency syndrome (disorder)
        '420384005',  # Hematopoietic system disease associated with acquired immunodeficiency syndrome (disorder)
        '420395004',  # Acute endocarditis associated with acquired immunodeficiency syndrome (disorder)
        '420403001',  # Pneumocystosis associated with acquired immunodeficiency syndrome (disorder)
        '420452002',  # Myelopathy associated with acquired immunodeficiency syndrome (disorder)
        '420524008',  # Kaposi's sarcoma associated with acquired immunodeficiency syndrome (disorder)
        '420543008',  # Anemia associated with acquired immunodeficiency syndrome (disorder)
        '420544002',  # Bacterial pneumonia associated with acquired immunodeficiency syndrome (disorder)
        '420549007',  # Salivary gland disease associated with acquired immunodeficiency syndrome (disorder)
        '420554003',  # Progressive multifocal leukoencephalopathy associated with acquired immunodeficiency syndrome (disorder)
        '420614009',  # Organic dementia associated with acquired immunodeficiency syndrome (disorder)
        '420658009',  # Radiculitis associated with acquired immunodeficiency syndrome (disorder)
        '420687005',  # Ill-defined intestinal infection associated with acquired immunodeficiency syndrome (disorder)
        '420691000',  # Nutritional deficiency associated with acquired immunodeficiency syndrome (disorder)
        '420718004',  # Central nervous system demyelinating disease associated with acquired immunodeficiency syndrome (disorder)
        '420721002',  # Acquired immunodeficiency syndrome-associated disorder (disorder)
        '420764009',  # Salmonella infection associated with acquired immunodeficiency syndrome (disorder)
        '420774007',  # Organic brain syndrome associated with acquired immunodeficiency syndrome (disorder)
        '420787001',  # Pneumococcal pneumonia associated with acquired immunodeficiency syndrome (disorder)
        '420801006',  # Malaise associated with acquired immunodeficiency syndrome (disorder)
        '420818005',  # Mycobacteriosis associated with acquired immunodeficiency syndrome (disorder)
        '420877009',  # Dermatomycosis associated with acquired immunodeficiency syndrome (disorder)
        '420900006',  # Fatigue associated with acquired immunodeficiency syndrome (disorder)
        '420938005',  # Subacute endocarditis associated with acquired immunodeficiency syndrome (disorder)
        '420945005',  # Histoplasmosis associated with acquired immunodeficiency syndrome (disorder)
        '421020000',  # Microsporidiosis associated with acquired immunodeficiency syndrome (disorder)
        '421023003',  # Presenile dementia associated with acquired immunodeficiency syndrome (disorder)
        '421047005',  # Candidiasis of lung associated with acquired immunodeficiency syndrome (disorder)
        '421077004',  # Disseminated candidiasis associated with acquired immunodeficiency syndrome (disorder)
        '421102007',  # Aplastic anemia associated with acquired immunodeficiency syndrome (disorder)
        '421230000',  # Hepatomegaly associated with acquired immunodeficiency syndrome (disorder)
        '421272004',  # Subacute myocarditis associated with acquired immunodeficiency syndrome (disorder)
        '421283008',  # Primary lymphoma of brain associated with acquired immunodeficiency syndrome (disorder)
        '421312009',  # Agranulocytosis associated with acquired immunodeficiency syndrome (disorder)
        '421315006',  # Myelitis associated with acquired immunodeficiency syndrome (disorder)
        '421394009',  # Skin disorder associated with acquired immunodeficiency syndrome (disorder)
        '421403008',  # Cryptococcosis associated with acquired immunodeficiency syndrome (disorder)
        '421415007',  # Subacute adenoviral encephalitis associated with acquired immunodeficiency syndrome (disorder)
        '421431004',  # Nocardiosis associated with acquired immunodeficiency syndrome (disorder)
        '421454008',  # Infectious gastroenteritis associated with acquired immunodeficiency syndrome (disorder)
        '421460008',  # Retinopathy associated with acquired immunodeficiency syndrome (disorder)
        '421508002',  # Viral pneumonia associated with acquired immunodeficiency syndrome (disorder)
        '421510000',  # Isosporiasis associated with acquired immunodeficiency syndrome (disorder)
        '421529006',  # Dementia associated with acquired immunodeficiency syndrome (disorder)
        '421571007',  # Tuberculosis associated with acquired immunodeficiency syndrome (disorder)
        '421597001',  # Polyneuropathy associated with acquired immunodeficiency syndrome (disorder)
        '421660003',  # Failure to thrive in infant associated with acquired immunodeficiency syndrome (disorder)
        '421666009',  # Toxoplasmosis associated with acquired immunodeficiency syndrome (disorder)
        '421671002',  # Pneumonia associated with acquired immunodeficiency syndrome (disorder)
        '421695000',  # Abnormal weight loss associated with acquired immunodeficiency syndrome (finding)
        '421706001',  # Blindness associated with acquired immunodeficiency syndrome (disorder)
        '421708000',  # Infective arthritis associated with acquired immunodeficiency syndrome (disorder)
        '421710003',  # Candidiasis of mouth associated with acquired immunodeficiency syndrome (disorder)
        '421766003',  # Thrombocytopenia associated with acquired immunodeficiency syndrome (disorder)
        '421827003',  # Encephalopathy associated with acquired immunodeficiency syndrome (disorder)
        '421851008',  # Acquired hemolytic anemia associated with acquired immunodeficiency syndrome (disorder)
        '421874007',  # Respiratory disorder associated with acquired immunodeficiency syndrome (disorder)
        '421883002',  # Strongyloidiasis associated with acquired immunodeficiency syndrome (disorder)
        '421929001',  # Myocarditis associated with acquired immunodeficiency syndrome (disorder)
        '421983003',  # Noninfectious gastroenteritis associated with acquired immunodeficiency syndrome (disorder)
        '421998001',  # Central nervous disorder associated with acquired immunodeficiency syndrome (disorder)
        '422003001',  # Cachexia associated with acquired immunodeficiency syndrome (finding)
        '422012004',  # Neuritis associated with acquired immunodeficiency syndrome (disorder)
        '422074008',  # Coccidiosis associated with acquired immunodeficiency syndrome (disorder)
        '422089004',  # Encephalomyelitis associated with acquired immunodeficiency syndrome (disorder)
        '422127002',  # Herpes zoster associated with acquired immunodeficiency syndrome (disorder)
        '422136003',  # Neuralgia associated with acquired immunodeficiency syndrome (disorder)
        '422177004',  # Dyspnea associated with acquired immunodeficiency syndrome (disorder)
        '422189002',  # Low vision associated with acquired immunodeficiency syndrome (disorder)
        '422194002',  # Hyperhidrosis associated with acquired immunodeficiency syndrome (disorder)
        '422282000',  # Malignant neoplasm associated with acquired immunodeficiency syndrome (disorder)
        '422337001',  # Coccidioidomycosis associated with acquired immunodeficiency syndrome (disorder)
        '432218001',  # History of asymptomatic human immunodeficiency virus infection (situation)
        '442537007',  # Non-Hodgkin lymphoma associated with Human immunodeficiency virus infection (disorder)
        '442662004',  # Copies of human immunodeficiency virus ribonucleic acid above reference range (finding)
        '445945000',  # Infectious disease associated with acquired immune deficiency syndrome (disorder)
        '697904001',  # Pulmonary arterial hypertension associated with human immunodeficiency virus infection (disorder)
        '697965002',  # Cholangitis associated with acquired immunodeficiency syndrome (disorder)
        '699433000',  # Human immunodeficiency virus carrier (finding)
        '700053002',  # Sepsis associated with acquired immunodeficiency syndrome (disorder)
        '713260006',  # Subacute adenoviral encephalitis co-occurrent with human immunodeficiency virus infection (disorder)
        '713275003',  # Splenomegaly co-occurrent with human immunodeficiency virus infection (disorder)
        '713278001',  # Neuralgia co-occurrent with human immunodeficiency virus infection (disorder)
        '713297001',  # Candidiasis of esophagus co-occurrent with human immunodeficiency virus infection (disorder)
        '713298006',  # Heart disease co-occurrent with human immunodeficiency virus infection (disorder)
        '713299003',  # Disorder of eye proper co-occurrent with human immunodeficiency virus infection (disorder)
        '713300006',  # Disorder of gastrointestinal tract co-occurrent with human immunodeficiency virus infection (disorder)
        '713316008',  # Eruption of skin co-occurrent with human immunodeficiency virus infection (disorder)
        '713318009',  # Myocarditis co-occurrent with human immunodeficiency virus infection (disorder)
        '713320007',  # Radiculitis co-occurrent with human immunodeficiency virus infection (disorder)
        '713325002',  # Primary cerebral lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        '713339002',  # Infection caused by Strongyloides co-occurrent with human immunodeficiency virus infection (disorder)
        '713340000',  # Disorder of skin co-occurrent with human immunodeficiency virus infection (disorder)
        '713341001',  # Myelitis co-occurrent with human immunodeficiency virus infection (disorder)
        '713342008',  # Infection caused by Salmonella co-occurrent with human immunodeficiency virus infection (disorder)
        '713349004',  # Anemia co-occurrent with human immunodeficiency virus infection (disorder)
        '713444005',  # Hemophagocytic syndrome co-occurrent with human immunodeficiency virus infection (disorder)
        '713445006',  # Disseminated infection caused by Strongyloides co-occurrent with human immunodeficiency virus infection (disorder)
        '713446007',  # Chronic infection caused by herpes simplex virus co-occurrent with human immunodeficiency virus infection (disorder)
        '713483007',  # Reticulosarcoma co-occurrent with human immunodeficiency virus infection (disorder)
        '713484001',  # Disorder of respiratory system co-occurrent with human immunodeficiency virus infection (disorder)
        '713487008',  # Progressive multifocal leukoencephalopathy co-occurrent with human immunodeficiency virus infection (disorder)
        '713488003',  # Presenile dementia co-occurrent with human immunodeficiency virus infection (disorder)
        '713489006',  # Polyneuropathy co-occurrent with human immunodeficiency virus infection (disorder)
        '713490002',  # Infection caused by Pneumocystis co-occurrent with human immunodeficiency virus infection (disorder)
        '713491003',  # Organic brain syndrome co-occurrent with human immunodeficiency virus infection (disorder)
        '713497004',  # Candidiasis of mouth co-occurrent with human immunodeficiency virus infection (disorder)
        '713503007',  # Disorder of spinal cord co-occurrent with human immunodeficiency virus infection (disorder)
        '713504001',  # Disorder of kidney co-occurrent with human immunodeficiency virus infection (disorder)
        '713505000',  # Gastrointestinal malabsorption syndrome co-occurrent with human immunodeficiency virus infection (disorder)
        '713506004',  # Neuritis co-occurrent with human immunodeficiency virus infection (disorder)
        '713507008',  # Lymphadenopathy co-occurrent with human immunodeficiency virus infection (disorder)
        '713508003',  # Aplastic anemia co-occurrent with human immunodeficiency virus infection (disorder)
        '713510001',  # Enlargement of liver co-occurrent with human immunodeficiency virus infection (disorder)
        '713511002',  # Acute endocarditis co-occurrent with human immunodeficiency virus infection (disorder)
        '713523008',  # Cardiomyopathy co-occurrent with human immunodeficiency virus infection (disorder)
        '713526000',  # Recurrent bacterial pneumonia co-occurrent with human immunodeficiency virus infection (disorder)
        '713527009',  # Disorder of peripheral nervous system co-occurrent with human immunodeficiency virus infection (disorder)
        '713530002',  # Agranulocytosis co-occurrent with human immunodeficiency virus infection (disorder)
        '713531003',  # Visual impairment co-occurrent with human immunodeficiency virus infection (disorder)
        '713532005',  # Infective arthritis co-occurrent with human immunodeficiency virus infection (disorder)
        '713533000',  # Acquired hemolytic anemia co-occurrent with human immunodeficiency virus infection (disorder)
        '713543002',  # Demyelinating disease of central nervous system co-occurrent with human immunodeficiency virus infection (disorder)
        '713544008',  # Bacterial pneumonia co-occurrent with human immunodeficiency virus infection (disorder)
        '713545009',  # Infection caused by Nocardia co-occurrent with human immunodeficiency virus infection (disorder)
        '713546005',  # Isosporiasis co-occurrent with human immunodeficiency virus infection (disorder)
        '713570009',  # Infectious gastroenteritis co-occurrent with human immunodeficiency virus infection (disorder)
        '713571008',  # Disorder of central nervous system co-occurrent with human immunodeficiency virus infection (disorder)
        '713572001',  # Malignant neoplastic disease co-occurrent with human immunodeficiency virus infection (disorder)
        '713695001',  # Nephrotic syndrome co-occurrent with human immunodeficiency virus infection (disorder)
        '713696000',  # Renal failure syndrome co-occurrent with human immunodeficiency virus infection (disorder)
        '713718006',  # Diffuse non-Hodgkin immunoblastic lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        '713722001',  # Infection caused by Cytomegalovirus co-occurrent with human immunodeficiency virus infection (disorder)
        '713729005',  # Infection caused by Coccidia co-occurrent with human immunodeficiency virus infection (disorder)
        '713730000',  # Infection caused by herpes simplex virus co-occurrent with human immunodeficiency virus infection (disorder)
        '713731001',  # Pyrexia of unknown origin co-occurrent with human immunodeficiency virus infection (disorder)
        '713732008',  # Infection caused by Aspergillus co-occurrent with human immunodeficiency virus infection (disorder)
        '713733003',  # Infection caused by herpes zoster virus co-occurrent with human immunodeficiency virus infection (disorder)
        '713734009',  # Infection caused by Dermatophyte co-occurrent with human immunodeficiency virus infection (disorder)
        '713742005',  # Human immunodeficiency virus antibody positive (finding)
        '713844000',  # Dementia co-occurrent with human immunodeficiency virus infection (disorder)
        '713845004',  # Infection caused by Cryptosporidium co-occurrent with human immunodeficiency virus infection (disorder)
        '713880000',  # Opportunistic mycosis co-occurrent with human immunodeficiency virus infection (disorder)
        '713881001',  # Infection caused by Microsporidia co-occurrent with human immunodeficiency virus infection (disorder)
        '713887002',  # Focal segmental glomerulosclerosis co-occurrent with human immunodeficiency virus infection (disorder)
        '713897006',  # Burkitt lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        '713964006',  # Multidermatomal infection caused by Herpes zoster co-occurrent with human immunodeficiency virus infection (disorder)
        '713967004',  # Disseminated atypical infection caused by Mycobacterium co-occurrent with human immunodeficiency virus infection (disorder)
        '714083007',  # Recurrent salmonella sepsis co-occurrent with human immunodeficiency virus infection (disorder)
        '714464009',  # Immune reconstitution inflammatory syndrome caused by human immunodeficiency virus infection (disorder)
        '719522009',  # Infection of upper respiratory tract caused by Candida co-occurrent with human immunodeficiency virus infection (disorder)
        '721166000',  # Human immunodeficiency virus complicating pregnancy childbirth and the puerperium (disorder)
        '722557007',  # Parkinsonism due to human immunodeficiency virus infection (disorder)
        '733834006',  # Invasive carcinoma of uterine cervix co-occurrent with human immunodeficiency virus infection (disorder)
        '733835007',  # Extrapulmonary tuberculosis co-occurrent with human immunodeficiency virus infection (disorder)
        '735521001',  # Human immunodeficiency virus World Health Organization 2007 stage 1 co-occurrent with tuberculosis (disorder)
        '735522008',  # Human immunodeficiency virus World Health Organization 2007 stage 1 co-occurrent with malaria (disorder)
        '735523003',  # Human immunodeficiency virus World Health Organization 2007 stage 2 co-occurrent with tuberculosis (disorder)
        '735524009',  # Human immunodeficiency virus World Health Organization 2007 stage 2 co-occurrent with malaria (disorder)
        '735525005',  # Human immunodeficiency virus World Health Organization 2007 stage 3 co-occurrent with tuberculosis (disorder)
        '735526006',  # Human immunodeficiency virus World Health Organization 2007 stage 3 co-occurrent with malaria (disorder)
        '735527002',  # Human immunodeficiency virus World Health Organization 2007 stage 4 co-occurrent with tuberculosis (disorder)
        '735528007',  # Human immunodeficiency virus World Health Organization 2007 stage 4 co-occurrent with malaria (disorder)
        '771119002',  # Infection caused by Coccidia co-occurrent with acquired immunodeficiency syndrome (disorder)
        '771126002',  # Infection caused by Toxoplasma gondii co-occurrent with acquired immunodeficiency syndrome (disorder)
        '771127006',  # Infection caused by Isospora co-occurrent with acquired immunodeficiency syndrome (disorder)
        '838377003',  # Chronic hepatitis C co-occurrent with human immunodeficiency virus infection (disorder)
        '840442003',  # Encephalitis caused by human immunodeficiency virus type 2 (disorder)
        '840498003',  # Encephalitis caused by human immunodeficiency virus type 1 (disorder)
        '81000119104',  # Symptomatic human immunodeficiency virus infection (disorder)
        '72621000119104',  # Human immunodeficiency virus (HIV) II infection category B1 (disorder)
        '72631000119101',  # Human immunodeficiency virus (HIV) II infection category B2 (disorder)
        '76981000119106',  # Human immunodeficiency virus (HIV) infection category B1 (disorder)
        '76991000119109',  # Human immunodeficiency virus (HIV) infection category B2 (disorder)
        '80191000119101',  # Symptomatic human immunodeficiency virus I infection (disorder)
        '90681000119107',  # Asymptomatic human immunodeficiency virus A1 infection (disorder)
        '90691000119105',  # Asymptomatic human immunodeficiency virus A2 infection (disorder)
        '10746341000119109',  # Acquired immune deficiency syndrome complicating childbirth (disorder)
        '10755671000119100',  # Human immunodeficiency virus in mother complicating childbirth (disorder)
        '15928141000119107',  # Cognitive impairment co-occurrent and due to human immunodeficiency virus infection (disorder)
    }


class HepatitisA(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of Hepatitis A infection.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of Hepatitis A.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS347v5, CMS117v10
    """

    VALUE_SET_NAME = 'Hepatitis A'
    OID = '2.16.840.1.113883.3.464.1003.110.12.1024'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'B15.0',  # Hepatitis A with hepatic coma
        'B15.9',  # Hepatitis A without hepatic coma
    }
    SNOMEDCT = {
        '16060001',  # Hepatic coma due to viral hepatitis A (disorder)
        '18917003',  # Acute fulminating type A viral hepatitis (disorder)
        '25102003',  # Acute type A viral hepatitis (disorder)
        '40468003',  # Viral hepatitis, type A (disorder)
        '43634002',  # Relapsing type A viral hepatitis (disorder)
        '79031007',  # Anicteric type A viral hepatitis (disorder)
        '111879004',  # Viral hepatitis A without hepatic coma (disorder)
        '165997004',  # Hepatitis A test positive (finding)
        '206373002',  # Congenital hepatitis A infection (disorder)
        '278971009',  # Hepatitis A immune (finding)
        '310875001',  # Hepatitis A - current infection (finding)
        '424758008',  # Viral hepatitis A without hepatic coma, without hepatitis delta (disorder)
        '428030001',  # History of hepatitis A (situation)
        '105801000119103',  # Immunity to hepatitis A by positive serology (finding)
    }


class HepatitisB(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for acute or chronic Hepatitis B infections.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of acute or chronic Hepatitis B.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS347v5, CMS117v10
    """

    VALUE_SET_NAME = 'Hepatitis B'
    OID = '2.16.840.1.113883.3.67.1.101.1.269'
    DEFINITION_VERSION = '20170825'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'B16.0',  # Acute hepatitis B with delta-agent with hepatic coma
        'B16.1',  # Acute hepatitis B with delta-agent without hepatic coma
        'B16.2',  # Acute hepatitis B without delta-agent with hepatic coma
        'B16.9',  # Acute hepatitis B without delta-agent and without hepatic coma
        'B17.0',  # Acute delta-(super) infection of hepatitis B carrier
        'B18.0',  # Chronic viral hepatitis B with delta-agent
        'B18.1',  # Chronic viral hepatitis B without delta-agent
        'B19.10',  # Unspecified viral hepatitis B without hepatic coma
        'B19.11',  # Unspecified viral hepatitis B with hepatic coma
    }
    SNOMEDCT = {
        '1116000',  # Chronic aggressive type B viral hepatitis (disorder)
        '13265006',  # Acute fulminating type B viral hepatitis (disorder)
        '26206000',  # Hepatic coma due to viral hepatitis B (disorder)
        '38662009',  # Chronic persistent type B viral hepatitis (disorder)
        '50167007',  # Chronic active type B viral hepatitis (disorder)
        '53425008',  # Anicteric type B viral hepatitis (disorder)
        '60498001',  # Congenital viral hepatitis B infection (disorder)
        '61977001',  # Chronic type B viral hepatitis (disorder)
        '66071002',  # Viral hepatitis type B (disorder)
        '76795007',  # Acute type B viral hepatitis (disorder)
        '111891008',  # Viral hepatitis B without hepatic coma (disorder)
        '165806002',  # Hepatitis B surface antigen positive (finding)
        '186624004',  # Hepatic coma due to acute hepatitis B with delta agent (disorder)
        '186626002',  # Acute hepatitis B with delta-agent (coinfection) without hepatic coma (disorder)
        '186639003',  # Chronic viral hepatitis B without delta-agent (disorder)
        '235864009',  # Acute hepatitis B with hepatitis D (disorder)
        '235865005',  # Hepatitis D superinfection of hepatitis B carrier (disorder)
        '235869004',  # Chronic viral hepatitis B with hepatitis D (disorder)
        '235871004',  # Hepatitis B carrier (finding)
        '271511000',  # Hepatitis B immune (finding)
        '313234004',  # Hepatitis B antibody present (finding)
        '406117000',  # Hepatitis B surface antigen enzyme-linked immunosorbent assay positive (finding)
        '424099008',  # Hepatic coma due to acute hepatitis B (disorder)
        '424340000',  # Hepatic coma due to chronic hepatitis B (disorder)
        '442134007',  # Hepatitis B associated with Human immunodeficiency virus infection (disorder)
        '442374005',  # Hepatitis B and hepatitis C (disorder)
        '446698005',  # Reactivation of hepatitis B viral hepatitis (disorder)
        '838380002',  # Chronic hepatitis B co-occurrent with hepatitis C and hepatitis D (disorder)
        '153091000119109',  # Hepatic coma due to chronic hepatitis B with delta agent (disorder)
    }


class Intussusception(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of intussusception.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of intussusception.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = 'Intussusception'
    OID = '2.16.840.1.113883.3.464.1003.199.12.1056'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'K56.1',  # Intussusception
    }
    SNOMEDCT = {
        '11578004',  # Intussusception of rectum (disorder)
        '17186003',  # Ileocolic intussusception (disorder)
        '24610009',  # Sigmoidorectal intussusception (disorder)
        '26316005',  # Internal complete rectal prolapse with intussusception of rectosigmoid (disorder)
        '27673007',  # Intussusception of colon (disorder)
        '38397000',  # Gastroesophageal intussusception (disorder)
        '49723003',  # Intussusception of intestine (disorder)
        '64976006',  # Cecocolic intussusception (disorder)
        '91446004',  # Intussusception of appendix (disorder)
        '111361008',  # Intussusception of cecum (disorder)
        '197046003',  # Gastroduodenal intussusception (disorder)
        '197047007',  # Jejunogastric intussusception (disorder)
        '197049005',  # Ileocecal intussusception (disorder)
        '197050005',  # Chronic intussusception (disorder)
        '197051009',  # Compound intussusception (disorder)
        '197052002',  # Multiple intussusception (disorder)
        '197053007',  # Retrograde intussusception (disorder)
        '197054001',  # Secondary intussusception (disorder)
        '313326008',  # Internal intussusception of rectum (disorder)
        '313327004',  # Inflammatory cloacogenic polyp (disorder)
        '405558008',  # Ileal intussusception (disorder)
        '721651002',  # Intussusception of small intestine (disorder)
        '721676001',  # Intussusception of large intestine (disorder)
    }


class MalignantNeoplasmOfLymphaticAndHematopoieticTissue(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of lymphoma and other cancers of the lymph and hematologic systems.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of lymphoma, leukemia and other malignant cancers of the lymphatic and hematologic systems.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = 'Malignant Neoplasm of Lymphatic and Hematopoietic Tissue'
    OID = '2.16.840.1.113883.3.464.1003.108.12.1009'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'C81.00',  # Nodular lymphocyte predominant Hodgkin lymphoma, unspecified site
        'C81.01',  # Nodular lymphocyte predominant Hodgkin lymphoma, lymph nodes of head, face, and neck
        'C81.02',  # Nodular lymphocyte predominant Hodgkin lymphoma, intrathoracic lymph nodes
        'C81.03',  # Nodular lymphocyte predominant Hodgkin lymphoma, intra-abdominal lymph nodes
        'C81.04',  # Nodular lymphocyte predominant Hodgkin lymphoma, lymph nodes of axilla and upper limb
        'C81.05',  # Nodular lymphocyte predominant Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        'C81.06',  # Nodular lymphocyte predominant Hodgkin lymphoma, intrapelvic lymph nodes
        'C81.07',  # Nodular lymphocyte predominant Hodgkin lymphoma, spleen
        'C81.08',  # Nodular lymphocyte predominant Hodgkin lymphoma, lymph nodes of multiple sites
        'C81.09',  # Nodular lymphocyte predominant Hodgkin lymphoma, extranodal and solid organ sites
        'C81.10',  # Nodular sclerosis Hodgkin lymphoma, unspecified site
        'C81.11',  # Nodular sclerosis Hodgkin lymphoma, lymph nodes of head, face, and neck
        'C81.12',  # Nodular sclerosis Hodgkin lymphoma, intrathoracic lymph nodes
        'C81.13',  # Nodular sclerosis Hodgkin lymphoma, intra-abdominal lymph nodes
        'C81.14',  # Nodular sclerosis Hodgkin lymphoma, lymph nodes of axilla and upper limb
        'C81.15',  # Nodular sclerosis Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        'C81.16',  # Nodular sclerosis Hodgkin lymphoma, intrapelvic lymph nodes
        'C81.17',  # Nodular sclerosis Hodgkin lymphoma, spleen
        'C81.18',  # Nodular sclerosis Hodgkin lymphoma, lymph nodes of multiple sites
        'C81.19',  # Nodular sclerosis Hodgkin lymphoma, extranodal and solid organ sites
        'C81.20',  # Mixed cellularity Hodgkin lymphoma, unspecified site
        'C81.21',  # Mixed cellularity Hodgkin lymphoma, lymph nodes of head, face, and neck
        'C81.22',  # Mixed cellularity Hodgkin lymphoma, intrathoracic lymph nodes
        'C81.23',  # Mixed cellularity Hodgkin lymphoma, intra-abdominal lymph nodes
        'C81.24',  # Mixed cellularity Hodgkin lymphoma, lymph nodes of axilla and upper limb
        'C81.25',  # Mixed cellularity Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        'C81.26',  # Mixed cellularity Hodgkin lymphoma, intrapelvic lymph nodes
        'C81.27',  # Mixed cellularity Hodgkin lymphoma, spleen
        'C81.28',  # Mixed cellularity Hodgkin lymphoma, lymph nodes of multiple sites
        'C81.29',  # Mixed cellularity Hodgkin lymphoma, extranodal and solid organ sites
        'C81.30',  # Lymphocyte depleted Hodgkin lymphoma, unspecified site
        'C81.31',  # Lymphocyte depleted Hodgkin lymphoma, lymph nodes of head, face, and neck
        'C81.32',  # Lymphocyte depleted Hodgkin lymphoma, intrathoracic lymph nodes
        'C81.33',  # Lymphocyte depleted Hodgkin lymphoma, intra-abdominal lymph nodes
        'C81.34',  # Lymphocyte depleted Hodgkin lymphoma, lymph nodes of axilla and upper limb
        'C81.35',  # Lymphocyte depleted Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        'C81.36',  # Lymphocyte depleted Hodgkin lymphoma, intrapelvic lymph nodes
        'C81.37',  # Lymphocyte depleted Hodgkin lymphoma, spleen
        'C81.38',  # Lymphocyte depleted Hodgkin lymphoma, lymph nodes of multiple sites
        'C81.39',  # Lymphocyte depleted Hodgkin lymphoma, extranodal and solid organ sites
        'C81.40',  # Lymphocyte-rich Hodgkin lymphoma, unspecified site
        'C81.41',  # Lymphocyte-rich Hodgkin lymphoma, lymph nodes of head, face, and neck
        'C81.42',  # Lymphocyte-rich Hodgkin lymphoma, intrathoracic lymph nodes
        'C81.43',  # Lymphocyte-rich Hodgkin lymphoma, intra-abdominal lymph nodes
        'C81.44',  # Lymphocyte-rich Hodgkin lymphoma, lymph nodes of axilla and upper limb
        'C81.45',  # Lymphocyte-rich Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        'C81.46',  # Lymphocyte-rich Hodgkin lymphoma, intrapelvic lymph nodes
        'C81.47',  # Lymphocyte-rich Hodgkin lymphoma, spleen
        'C81.48',  # Lymphocyte-rich Hodgkin lymphoma, lymph nodes of multiple sites
        'C81.49',  # Lymphocyte-rich Hodgkin lymphoma, extranodal and solid organ sites
        'C81.70',  # Other Hodgkin lymphoma, unspecified site
        'C81.71',  # Other Hodgkin lymphoma, lymph nodes of head, face, and neck
        'C81.72',  # Other Hodgkin lymphoma, intrathoracic lymph nodes
        'C81.73',  # Other Hodgkin lymphoma, intra-abdominal lymph nodes
        'C81.74',  # Other Hodgkin lymphoma, lymph nodes of axilla and upper limb
        'C81.75',  # Other Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        'C81.76',  # Other Hodgkin lymphoma, intrapelvic lymph nodes
        'C81.77',  # Other Hodgkin lymphoma, spleen
        'C81.78',  # Other Hodgkin lymphoma, lymph nodes of multiple sites
        'C81.79',  # Other Hodgkin lymphoma, extranodal and solid organ sites
        'C81.90',  # Hodgkin lymphoma, unspecified, unspecified site
        'C81.91',  # Hodgkin lymphoma, unspecified, lymph nodes of head, face, and neck
        'C81.92',  # Hodgkin lymphoma, unspecified, intrathoracic lymph nodes
        'C81.93',  # Hodgkin lymphoma, unspecified, intra-abdominal lymph nodes
        'C81.94',  # Hodgkin lymphoma, unspecified, lymph nodes of axilla and upper limb
        'C81.95',  # Hodgkin lymphoma, unspecified, lymph nodes of inguinal region and lower limb
        'C81.96',  # Hodgkin lymphoma, unspecified, intrapelvic lymph nodes
        'C81.97',  # Hodgkin lymphoma, unspecified, spleen
        'C81.98',  # Hodgkin lymphoma, unspecified, lymph nodes of multiple sites
        'C81.99',  # Hodgkin lymphoma, unspecified, extranodal and solid organ sites
        'C82.00',  # Follicular lymphoma grade I, unspecified site
        'C82.01',  # Follicular lymphoma grade I, lymph nodes of head, face, and neck
        'C82.02',  # Follicular lymphoma grade I, intrathoracic lymph nodes
        'C82.03',  # Follicular lymphoma grade I, intra-abdominal lymph nodes
        'C82.04',  # Follicular lymphoma grade I, lymph nodes of axilla and upper limb
        'C82.05',  # Follicular lymphoma grade I, lymph nodes of inguinal region and lower limb
        'C82.06',  # Follicular lymphoma grade I, intrapelvic lymph nodes
        'C82.07',  # Follicular lymphoma grade I, spleen
        'C82.08',  # Follicular lymphoma grade I, lymph nodes of multiple sites
        'C82.09',  # Follicular lymphoma grade I, extranodal and solid organ sites
        'C82.10',  # Follicular lymphoma grade II, unspecified site
        'C82.11',  # Follicular lymphoma grade II, lymph nodes of head, face, and neck
        'C82.12',  # Follicular lymphoma grade II, intrathoracic lymph nodes
        'C82.13',  # Follicular lymphoma grade II, intra-abdominal lymph nodes
        'C82.14',  # Follicular lymphoma grade II, lymph nodes of axilla and upper limb
        'C82.15',  # Follicular lymphoma grade II, lymph nodes of inguinal region and lower limb
        'C82.16',  # Follicular lymphoma grade II, intrapelvic lymph nodes
        'C82.17',  # Follicular lymphoma grade II, spleen
        'C82.18',  # Follicular lymphoma grade II, lymph nodes of multiple sites
        'C82.19',  # Follicular lymphoma grade II, extranodal and solid organ sites
        'C82.20',  # Follicular lymphoma grade III, unspecified, unspecified site
        'C82.21',  # Follicular lymphoma grade III, unspecified, lymph nodes of head, face, and neck
        'C82.22',  # Follicular lymphoma grade III, unspecified, intrathoracic lymph nodes
        'C82.23',  # Follicular lymphoma grade III, unspecified, intra-abdominal lymph nodes
        'C82.24',  # Follicular lymphoma grade III, unspecified, lymph nodes of axilla and upper limb
        'C82.25',  # Follicular lymphoma grade III, unspecified, lymph nodes of inguinal region and lower limb
        'C82.26',  # Follicular lymphoma grade III, unspecified, intrapelvic lymph nodes
        'C82.27',  # Follicular lymphoma grade III, unspecified, spleen
        'C82.28',  # Follicular lymphoma grade III, unspecified, lymph nodes of multiple sites
        'C82.29',  # Follicular lymphoma grade III, unspecified, extranodal and solid organ sites
        'C82.30',  # Follicular lymphoma grade IIIa, unspecified site
        'C82.31',  # Follicular lymphoma grade IIIa, lymph nodes of head, face, and neck
        'C82.32',  # Follicular lymphoma grade IIIa, intrathoracic lymph nodes
        'C82.33',  # Follicular lymphoma grade IIIa, intra-abdominal lymph nodes
        'C82.34',  # Follicular lymphoma grade IIIa, lymph nodes of axilla and upper limb
        'C82.35',  # Follicular lymphoma grade IIIa, lymph nodes of inguinal region and lower limb
        'C82.36',  # Follicular lymphoma grade IIIa, intrapelvic lymph nodes
        'C82.37',  # Follicular lymphoma grade IIIa, spleen
        'C82.38',  # Follicular lymphoma grade IIIa, lymph nodes of multiple sites
        'C82.39',  # Follicular lymphoma grade IIIa, extranodal and solid organ sites
        'C82.40',  # Follicular lymphoma grade IIIb, unspecified site
        'C82.41',  # Follicular lymphoma grade IIIb, lymph nodes of head, face, and neck
        'C82.42',  # Follicular lymphoma grade IIIb, intrathoracic lymph nodes
        'C82.43',  # Follicular lymphoma grade IIIb, intra-abdominal lymph nodes
        'C82.44',  # Follicular lymphoma grade IIIb, lymph nodes of axilla and upper limb
        'C82.45',  # Follicular lymphoma grade IIIb, lymph nodes of inguinal region and lower limb
        'C82.46',  # Follicular lymphoma grade IIIb, intrapelvic lymph nodes
        'C82.47',  # Follicular lymphoma grade IIIb, spleen
        'C82.48',  # Follicular lymphoma grade IIIb, lymph nodes of multiple sites
        'C82.49',  # Follicular lymphoma grade IIIb, extranodal and solid organ sites
        'C82.50',  # Diffuse follicle center lymphoma, unspecified site
        'C82.51',  # Diffuse follicle center lymphoma, lymph nodes of head, face, and neck
        'C82.52',  # Diffuse follicle center lymphoma, intrathoracic lymph nodes
        'C82.53',  # Diffuse follicle center lymphoma, intra-abdominal lymph nodes
        'C82.54',  # Diffuse follicle center lymphoma, lymph nodes of axilla and upper limb
        'C82.55',  # Diffuse follicle center lymphoma, lymph nodes of inguinal region and lower limb
        'C82.56',  # Diffuse follicle center lymphoma, intrapelvic lymph nodes
        'C82.57',  # Diffuse follicle center lymphoma, spleen
        'C82.58',  # Diffuse follicle center lymphoma, lymph nodes of multiple sites
        'C82.59',  # Diffuse follicle center lymphoma, extranodal and solid organ sites
        'C82.60',  # Cutaneous follicle center lymphoma, unspecified site
        'C82.61',  # Cutaneous follicle center lymphoma, lymph nodes of head, face, and neck
        'C82.62',  # Cutaneous follicle center lymphoma, intrathoracic lymph nodes
        'C82.63',  # Cutaneous follicle center lymphoma, intra-abdominal lymph nodes
        'C82.64',  # Cutaneous follicle center lymphoma, lymph nodes of axilla and upper limb
        'C82.65',  # Cutaneous follicle center lymphoma, lymph nodes of inguinal region and lower limb
        'C82.66',  # Cutaneous follicle center lymphoma, intrapelvic lymph nodes
        'C82.67',  # Cutaneous follicle center lymphoma, spleen
        'C82.68',  # Cutaneous follicle center lymphoma, lymph nodes of multiple sites
        'C82.69',  # Cutaneous follicle center lymphoma, extranodal and solid organ sites
        'C82.80',  # Other types of follicular lymphoma, unspecified site
        'C82.81',  # Other types of follicular lymphoma, lymph nodes of head, face, and neck
        'C82.82',  # Other types of follicular lymphoma, intrathoracic lymph nodes
        'C82.83',  # Other types of follicular lymphoma, intra-abdominal lymph nodes
        'C82.84',  # Other types of follicular lymphoma, lymph nodes of axilla and upper limb
        'C82.85',  # Other types of follicular lymphoma, lymph nodes of inguinal region and lower limb
        'C82.86',  # Other types of follicular lymphoma, intrapelvic lymph nodes
        'C82.87',  # Other types of follicular lymphoma, spleen
        'C82.88',  # Other types of follicular lymphoma, lymph nodes of multiple sites
        'C82.89',  # Other types of follicular lymphoma, extranodal and solid organ sites
        'C82.90',  # Follicular lymphoma, unspecified, unspecified site
        'C82.91',  # Follicular lymphoma, unspecified, lymph nodes of head, face, and neck
        'C82.92',  # Follicular lymphoma, unspecified, intrathoracic lymph nodes
        'C82.93',  # Follicular lymphoma, unspecified, intra-abdominal lymph nodes
        'C82.94',  # Follicular lymphoma, unspecified, lymph nodes of axilla and upper limb
        'C82.95',  # Follicular lymphoma, unspecified, lymph nodes of inguinal region and lower limb
        'C82.96',  # Follicular lymphoma, unspecified, intrapelvic lymph nodes
        'C82.97',  # Follicular lymphoma, unspecified, spleen
        'C82.98',  # Follicular lymphoma, unspecified, lymph nodes of multiple sites
        'C82.99',  # Follicular lymphoma, unspecified, extranodal and solid organ sites
        'C83.00',  # Small cell B-cell lymphoma, unspecified site
        'C83.01',  # Small cell B-cell lymphoma, lymph nodes of head, face, and neck
        'C83.02',  # Small cell B-cell lymphoma, intrathoracic lymph nodes
        'C83.03',  # Small cell B-cell lymphoma, intra-abdominal lymph nodes
        'C83.04',  # Small cell B-cell lymphoma, lymph nodes of axilla and upper limb
        'C83.05',  # Small cell B-cell lymphoma, lymph nodes of inguinal region and lower limb
        'C83.06',  # Small cell B-cell lymphoma, intrapelvic lymph nodes
        'C83.07',  # Small cell B-cell lymphoma, spleen
        'C83.08',  # Small cell B-cell lymphoma, lymph nodes of multiple sites
        'C83.09',  # Small cell B-cell lymphoma, extranodal and solid organ sites
        'C83.10',  # Mantle cell lymphoma, unspecified site
        'C83.11',  # Mantle cell lymphoma, lymph nodes of head, face, and neck
        'C83.12',  # Mantle cell lymphoma, intrathoracic lymph nodes
        'C83.13',  # Mantle cell lymphoma, intra-abdominal lymph nodes
        'C83.14',  # Mantle cell lymphoma, lymph nodes of axilla and upper limb
        'C83.15',  # Mantle cell lymphoma, lymph nodes of inguinal region and lower limb
        'C83.16',  # Mantle cell lymphoma, intrapelvic lymph nodes
        'C83.17',  # Mantle cell lymphoma, spleen
        'C83.18',  # Mantle cell lymphoma, lymph nodes of multiple sites
        'C83.19',  # Mantle cell lymphoma, extranodal and solid organ sites
        'C83.30',  # Diffuse large B-cell lymphoma, unspecified site
        'C83.31',  # Diffuse large B-cell lymphoma, lymph nodes of head, face, and neck
        'C83.32',  # Diffuse large B-cell lymphoma, intrathoracic lymph nodes
        'C83.33',  # Diffuse large B-cell lymphoma, intra-abdominal lymph nodes
        'C83.34',  # Diffuse large B-cell lymphoma, lymph nodes of axilla and upper limb
        'C83.35',  # Diffuse large B-cell lymphoma, lymph nodes of inguinal region and lower limb
        'C83.36',  # Diffuse large B-cell lymphoma, intrapelvic lymph nodes
        'C83.37',  # Diffuse large B-cell lymphoma, spleen
        'C83.38',  # Diffuse large B-cell lymphoma, lymph nodes of multiple sites
        'C83.39',  # Diffuse large B-cell lymphoma, extranodal and solid organ sites
        'C83.50',  # Lymphoblastic (diffuse) lymphoma, unspecified site
        'C83.51',  # Lymphoblastic (diffuse) lymphoma, lymph nodes of head, face, and neck
        'C83.52',  # Lymphoblastic (diffuse) lymphoma, intrathoracic lymph nodes
        'C83.53',  # Lymphoblastic (diffuse) lymphoma, intra-abdominal lymph nodes
        'C83.54',  # Lymphoblastic (diffuse) lymphoma, lymph nodes of axilla and upper limb
        'C83.55',  # Lymphoblastic (diffuse) lymphoma, lymph nodes of inguinal region and lower limb
        'C83.56',  # Lymphoblastic (diffuse) lymphoma, intrapelvic lymph nodes
        'C83.57',  # Lymphoblastic (diffuse) lymphoma, spleen
        'C83.58',  # Lymphoblastic (diffuse) lymphoma, lymph nodes of multiple sites
        'C83.59',  # Lymphoblastic (diffuse) lymphoma, extranodal and solid organ sites
        'C83.70',  # Burkitt lymphoma, unspecified site
        'C83.71',  # Burkitt lymphoma, lymph nodes of head, face, and neck
        'C83.72',  # Burkitt lymphoma, intrathoracic lymph nodes
        'C83.73',  # Burkitt lymphoma, intra-abdominal lymph nodes
        'C83.74',  # Burkitt lymphoma, lymph nodes of axilla and upper limb
        'C83.75',  # Burkitt lymphoma, lymph nodes of inguinal region and lower limb
        'C83.76',  # Burkitt lymphoma, intrapelvic lymph nodes
        'C83.77',  # Burkitt lymphoma, spleen
        'C83.78',  # Burkitt lymphoma, lymph nodes of multiple sites
        'C83.79',  # Burkitt lymphoma, extranodal and solid organ sites
        'C83.80',  # Other non-follicular lymphoma, unspecified site
        'C83.81',  # Other non-follicular lymphoma, lymph nodes of head, face, and neck
        'C83.82',  # Other non-follicular lymphoma, intrathoracic lymph nodes
        'C83.83',  # Other non-follicular lymphoma, intra-abdominal lymph nodes
        'C83.84',  # Other non-follicular lymphoma, lymph nodes of axilla and upper limb
        'C83.85',  # Other non-follicular lymphoma, lymph nodes of inguinal region and lower limb
        'C83.86',  # Other non-follicular lymphoma, intrapelvic lymph nodes
        'C83.87',  # Other non-follicular lymphoma, spleen
        'C83.88',  # Other non-follicular lymphoma, lymph nodes of multiple sites
        'C83.89',  # Other non-follicular lymphoma, extranodal and solid organ sites
        'C83.90',  # Non-follicular (diffuse) lymphoma, unspecified, unspecified site
        'C83.91',  # Non-follicular (diffuse) lymphoma, unspecified, lymph nodes of head, face, and neck
        'C83.92',  # Non-follicular (diffuse) lymphoma, unspecified, intrathoracic lymph nodes
        'C83.93',  # Non-follicular (diffuse) lymphoma, unspecified, intra-abdominal lymph nodes
        'C83.94',  # Non-follicular (diffuse) lymphoma, unspecified, lymph nodes of axilla and upper limb
        'C83.95',  # Non-follicular (diffuse) lymphoma, unspecified, lymph nodes of inguinal region and lower limb
        'C83.96',  # Non-follicular (diffuse) lymphoma, unspecified, intrapelvic lymph nodes
        'C83.97',  # Non-follicular (diffuse) lymphoma, unspecified, spleen
        'C83.98',  # Non-follicular (diffuse) lymphoma, unspecified, lymph nodes of multiple sites
        'C83.99',  # Non-follicular (diffuse) lymphoma, unspecified, extranodal and solid organ sites
        'C84.00',  # Mycosis fungoides, unspecified site
        'C84.01',  # Mycosis fungoides, lymph nodes of head, face, and neck
        'C84.02',  # Mycosis fungoides, intrathoracic lymph nodes
        'C84.03',  # Mycosis fungoides, intra-abdominal lymph nodes
        'C84.04',  # Mycosis fungoides, lymph nodes of axilla and upper limb
        'C84.05',  # Mycosis fungoides, lymph nodes of inguinal region and lower limb
        'C84.06',  # Mycosis fungoides, intrapelvic lymph nodes
        'C84.07',  # Mycosis fungoides, spleen
        'C84.08',  # Mycosis fungoides, lymph nodes of multiple sites
        'C84.09',  # Mycosis fungoides, extranodal and solid organ sites
        'C84.10',  # Sézary disease, unspecified site
        'C84.11',  # Sézary disease, lymph nodes of head, face, and neck
        'C84.12',  # Sézary disease, intrathoracic lymph nodes
        'C84.13',  # Sézary disease, intra-abdominal lymph nodes
        'C84.14',  # Sézary disease, lymph nodes of axilla and upper limb
        'C84.15',  # Sézary disease, lymph nodes of inguinal region and lower limb
        'C84.16',  # Sézary disease, intrapelvic lymph nodes
        'C84.17',  # Sézary disease, spleen
        'C84.18',  # Sézary disease, lymph nodes of multiple sites
        'C84.19',  # Sézary disease, extranodal and solid organ sites
        'C84.40',  # Peripheral T-cell lymphoma, not classified, unspecified site
        'C84.41',  # Peripheral T-cell lymphoma, not classified, lymph nodes of head, face, and neck
        'C84.42',  # Peripheral T-cell lymphoma, not classified, intrathoracic lymph nodes
        'C84.43',  # Peripheral T-cell lymphoma, not classified, intra-abdominal lymph nodes
        'C84.44',  # Peripheral T-cell lymphoma, not classified, lymph nodes of axilla and upper limb
        'C84.45',  # Peripheral T-cell lymphoma, not classified, lymph nodes of inguinal region and lower limb
        'C84.46',  # Peripheral T-cell lymphoma, not classified, intrapelvic lymph nodes
        'C84.47',  # Peripheral T-cell lymphoma, not classified, spleen
        'C84.48',  # Peripheral T-cell lymphoma, not classified, lymph nodes of multiple sites
        'C84.49',  # Peripheral T-cell lymphoma, not classified, extranodal and solid organ sites
        'C84.60',  # Anaplastic large cell lymphoma, ALK-positive, unspecified site
        'C84.61',  # Anaplastic large cell lymphoma, ALK-positive, lymph nodes of head, face, and neck
        'C84.62',  # Anaplastic large cell lymphoma, ALK-positive, intrathoracic lymph nodes
        'C84.63',  # Anaplastic large cell lymphoma, ALK-positive, intra-abdominal lymph nodes
        'C84.64',  # Anaplastic large cell lymphoma, ALK-positive, lymph nodes of axilla and upper limb
        'C84.65',  # Anaplastic large cell lymphoma, ALK-positive, lymph nodes of inguinal region and lower limb
        'C84.66',  # Anaplastic large cell lymphoma, ALK-positive, intrapelvic lymph nodes
        'C84.67',  # Anaplastic large cell lymphoma, ALK-positive, spleen
        'C84.68',  # Anaplastic large cell lymphoma, ALK-positive, lymph nodes of multiple sites
        'C84.69',  # Anaplastic large cell lymphoma, ALK-positive, extranodal and solid organ sites
        'C84.70',  # Anaplastic large cell lymphoma, ALK-negative, unspecified site
        'C84.71',  # Anaplastic large cell lymphoma, ALK-negative, lymph nodes of head, face, and neck
        'C84.72',  # Anaplastic large cell lymphoma, ALK-negative, intrathoracic lymph nodes
        'C84.73',  # Anaplastic large cell lymphoma, ALK-negative, intra-abdominal lymph nodes
        'C84.74',  # Anaplastic large cell lymphoma, ALK-negative, lymph nodes of axilla and upper limb
        'C84.75',  # Anaplastic large cell lymphoma, ALK-negative, lymph nodes of inguinal region and lower limb
        'C84.76',  # Anaplastic large cell lymphoma, ALK-negative, intrapelvic lymph nodes
        'C84.77',  # Anaplastic large cell lymphoma, ALK-negative, spleen
        'C84.78',  # Anaplastic large cell lymphoma, ALK-negative, lymph nodes of multiple sites
        'C84.79',  # Anaplastic large cell lymphoma, ALK-negative, extranodal and solid organ sites
        'C84.90',  # Mature T/NK-cell lymphomas, unspecified, unspecified site
        'C84.91',  # Mature T/NK-cell lymphomas, unspecified, lymph nodes of head, face, and neck
        'C84.92',  # Mature T/NK-cell lymphomas, unspecified, intrathoracic lymph nodes
        'C84.93',  # Mature T/NK-cell lymphomas, unspecified, intra-abdominal lymph nodes
        'C84.94',  # Mature T/NK-cell lymphomas, unspecified, lymph nodes of axilla and upper limb
        'C84.95',  # Mature T/NK-cell lymphomas, unspecified, lymph nodes of inguinal region and lower limb
        'C84.96',  # Mature T/NK-cell lymphomas, unspecified, intrapelvic lymph nodes
        'C84.97',  # Mature T/NK-cell lymphomas, unspecified, spleen
        'C84.98',  # Mature T/NK-cell lymphomas, unspecified, lymph nodes of multiple sites
        'C84.99',  # Mature T/NK-cell lymphomas, unspecified, extranodal and solid organ sites
        'C84.A0',  # Cutaneous T-cell lymphoma, unspecified, unspecified site
        'C84.A1',  # Cutaneous T-cell lymphoma, unspecified lymph nodes of head, face, and neck
        'C84.A2',  # Cutaneous T-cell lymphoma, unspecified, intrathoracic lymph nodes
        'C84.A3',  # Cutaneous T-cell lymphoma, unspecified, intra-abdominal lymph nodes
        'C84.A4',  # Cutaneous T-cell lymphoma, unspecified, lymph nodes of axilla and upper limb
        'C84.A5',  # Cutaneous T-cell lymphoma, unspecified, lymph nodes of inguinal region and lower limb
        'C84.A6',  # Cutaneous T-cell lymphoma, unspecified, intrapelvic lymph nodes
        'C84.A7',  # Cutaneous T-cell lymphoma, unspecified, spleen
        'C84.A8',  # Cutaneous T-cell lymphoma, unspecified, lymph nodes of multiple sites
        'C84.A9',  # Cutaneous T-cell lymphoma, unspecified, extranodal and solid organ sites
        'C84.Z0',  # Other mature T/NK-cell lymphomas, unspecified site
        'C84.Z1',  # Other mature T/NK-cell lymphomas, lymph nodes of head, face, and neck
        'C84.Z2',  # Other mature T/NK-cell lymphomas, intrathoracic lymph nodes
        'C84.Z3',  # Other mature T/NK-cell lymphomas, intra-abdominal lymph nodes
        'C84.Z4',  # Other mature T/NK-cell lymphomas, lymph nodes of axilla and upper limb
        'C84.Z5',  # Other mature T/NK-cell lymphomas, lymph nodes of inguinal region and lower limb
        'C84.Z6',  # Other mature T/NK-cell lymphomas, intrapelvic lymph nodes
        'C84.Z7',  # Other mature T/NK-cell lymphomas, spleen
        'C84.Z8',  # Other mature T/NK-cell lymphomas, lymph nodes of multiple sites
        'C84.Z9',  # Other mature T/NK-cell lymphomas, extranodal and solid organ sites
        'C85.10',  # Unspecified B-cell lymphoma, unspecified site
        'C85.11',  # Unspecified B-cell lymphoma, lymph nodes of head, face, and neck
        'C85.12',  # Unspecified B-cell lymphoma, intrathoracic lymph nodes
        'C85.13',  # Unspecified B-cell lymphoma, intra-abdominal lymph nodes
        'C85.14',  # Unspecified B-cell lymphoma, lymph nodes of axilla and upper limb
        'C85.15',  # Unspecified B-cell lymphoma, lymph nodes of inguinal region and lower limb
        'C85.16',  # Unspecified B-cell lymphoma, intrapelvic lymph nodes
        'C85.17',  # Unspecified B-cell lymphoma, spleen
        'C85.18',  # Unspecified B-cell lymphoma, lymph nodes of multiple sites
        'C85.19',  # Unspecified B-cell lymphoma, extranodal and solid organ sites
        'C85.20',  # Mediastinal (thymic) large B-cell lymphoma, unspecified site
        'C85.21',  # Mediastinal (thymic) large B-cell lymphoma, lymph nodes of head, face, and neck
        'C85.22',  # Mediastinal (thymic) large B-cell lymphoma, intrathoracic lymph nodes
        'C85.23',  # Mediastinal (thymic) large B-cell lymphoma, intra-abdominal lymph nodes
        'C85.24',  # Mediastinal (thymic) large B-cell lymphoma, lymph nodes of axilla and upper limb
        'C85.25',  # Mediastinal (thymic) large B-cell lymphoma, lymph nodes of inguinal region and lower limb
        'C85.26',  # Mediastinal (thymic) large B-cell lymphoma, intrapelvic lymph nodes
        'C85.27',  # Mediastinal (thymic) large B-cell lymphoma, spleen
        'C85.28',  # Mediastinal (thymic) large B-cell lymphoma, lymph nodes of multiple sites
        'C85.29',  # Mediastinal (thymic) large B-cell lymphoma, extranodal and solid organ sites
        'C85.80',  # Other specified types of non-Hodgkin lymphoma, unspecified site
        'C85.81',  # Other specified types of non-Hodgkin lymphoma, lymph nodes of head, face, and neck
        'C85.82',  # Other specified types of non-Hodgkin lymphoma, intrathoracic lymph nodes
        'C85.83',  # Other specified types of non-Hodgkin lymphoma, intra-abdominal lymph nodes
        'C85.84',  # Other specified types of non-Hodgkin lymphoma, lymph nodes of axilla and upper limb
        'C85.85',  # Other specified types of non-Hodgkin lymphoma, lymph nodes of inguinal region and lower limb
        'C85.86',  # Other specified types of non-Hodgkin lymphoma, intrapelvic lymph nodes
        'C85.87',  # Other specified types of non-Hodgkin lymphoma, spleen
        'C85.88',  # Other specified types of non-Hodgkin lymphoma, lymph nodes of multiple sites
        'C85.89',  # Other specified types of non-Hodgkin lymphoma, extranodal and solid organ sites
        'C85.90',  # Non-Hodgkin lymphoma, unspecified, unspecified site
        'C85.91',  # Non-Hodgkin lymphoma, unspecified, lymph nodes of head, face, and neck
        'C85.92',  # Non-Hodgkin lymphoma, unspecified, intrathoracic lymph nodes
        'C85.93',  # Non-Hodgkin lymphoma, unspecified, intra-abdominal lymph nodes
        'C85.94',  # Non-Hodgkin lymphoma, unspecified, lymph nodes of axilla and upper limb
        'C85.95',  # Non-Hodgkin lymphoma, unspecified, lymph nodes of inguinal region and lower limb
        'C85.96',  # Non-Hodgkin lymphoma, unspecified, intrapelvic lymph nodes
        'C85.97',  # Non-Hodgkin lymphoma, unspecified, spleen
        'C85.98',  # Non-Hodgkin lymphoma, unspecified, lymph nodes of multiple sites
        'C85.99',  # Non-Hodgkin lymphoma, unspecified, extranodal and solid organ sites
        'C86.0',  # Extranodal NK/T-cell lymphoma, nasal type
        'C86.1',  # Hepatosplenic T-cell lymphoma
        'C86.2',  # Enteropathy-type (intestinal) T-cell lymphoma
        'C86.3',  # Subcutaneous panniculitis-like T-cell lymphoma
        'C86.4',  # Blastic NK-cell lymphoma
        'C86.5',  # Angioimmunoblastic T-cell lymphoma
        'C86.6',  # Primary cutaneous CD30-positive T-cell proliferations
        'C88.2',  # Heavy chain disease
        'C88.3',  # Immunoproliferative small intestinal disease
        'C88.4',  # Extranodal marginal zone B-cell lymphoma of mucosa-associated lymphoid tissue [MALT-lymphoma]
        'C88.8',  # Other malignant immunoproliferative diseases
        'C88.9',  # Malignant immunoproliferative disease, unspecified
        'C90.00',  # Multiple myeloma not having achieved remission
        'C90.01',  # Multiple myeloma in remission
        'C90.02',  # Multiple myeloma in relapse
        'C90.10',  # Plasma cell leukemia not having achieved remission
        'C90.11',  # Plasma cell leukemia in remission
        'C90.12',  # Plasma cell leukemia in relapse
        'C90.20',  # Extramedullary plasmacytoma not having achieved remission
        'C90.21',  # Extramedullary plasmacytoma in remission
        'C90.22',  # Extramedullary plasmacytoma in relapse
        'C90.30',  # Solitary plasmacytoma not having achieved remission
        'C90.31',  # Solitary plasmacytoma in remission
        'C90.32',  # Solitary plasmacytoma in relapse
        'C91.00',  # Acute lymphoblastic leukemia not having achieved remission
        'C91.01',  # Acute lymphoblastic leukemia, in remission
        'C91.02',  # Acute lymphoblastic leukemia, in relapse
        'C91.10',  # Chronic lymphocytic leukemia of B-cell type not having achieved remission
        'C91.11',  # Chronic lymphocytic leukemia of B-cell type in remission
        'C91.12',  # Chronic lymphocytic leukemia of B-cell type in relapse
        'C91.30',  # Prolymphocytic leukemia of B-cell type not having achieved remission
        'C91.31',  # Prolymphocytic leukemia of B-cell type, in remission
        'C91.32',  # Prolymphocytic leukemia of B-cell type, in relapse
        'C91.40',  # Hairy cell leukemia not having achieved remission
        'C91.41',  # Hairy cell leukemia, in remission
        'C91.42',  # Hairy cell leukemia, in relapse
        'C91.50',  # Adult T-cell lymphoma/leukemia (HTLV-1-associated) not having achieved remission
        'C91.51',  # Adult T-cell lymphoma/leukemia (HTLV-1-associated), in remission
        'C91.52',  # Adult T-cell lymphoma/leukemia (HTLV-1-associated), in relapse
        'C91.60',  # Prolymphocytic leukemia of T-cell type not having achieved remission
        'C91.61',  # Prolymphocytic leukemia of T-cell type, in remission
        'C91.62',  # Prolymphocytic leukemia of T-cell type, in relapse
        'C91.90',  # Lymphoid leukemia, unspecified not having achieved remission
        'C91.91',  # Lymphoid leukemia, unspecified, in remission
        'C91.92',  # Lymphoid leukemia, unspecified, in relapse
        'C91.A0',  # Mature B-cell leukemia Burkitt-type not having achieved remission
        'C91.A1',  # Mature B-cell leukemia Burkitt-type, in remission
        'C91.A2',  # Mature B-cell leukemia Burkitt-type, in relapse
        'C91.Z0',  # Other lymphoid leukemia not having achieved remission
        'C91.Z1',  # Other lymphoid leukemia, in remission
        'C91.Z2',  # Other lymphoid leukemia, in relapse
        'C92.00',  # Acute myeloblastic leukemia, not having achieved remission
        'C92.01',  # Acute myeloblastic leukemia, in remission
        'C92.02',  # Acute myeloblastic leukemia, in relapse
        'C92.10',  # Chronic myeloid leukemia, BCR/ABL-positive, not having achieved remission
        'C92.11',  # Chronic myeloid leukemia, BCR/ABL-positive, in remission
        'C92.12',  # Chronic myeloid leukemia, BCR/ABL-positive, in relapse
        'C92.20',  # Atypical chronic myeloid leukemia, BCR/ABL-negative, not having achieved remission
        'C92.21',  # Atypical chronic myeloid leukemia, BCR/ABL-negative, in remission
        'C92.22',  # Atypical chronic myeloid leukemia, BCR/ABL-negative, in relapse
        'C92.30',  # Myeloid sarcoma, not having achieved remission
        'C92.31',  # Myeloid sarcoma, in remission
        'C92.32',  # Myeloid sarcoma, in relapse
        'C92.40',  # Acute promyelocytic leukemia, not having achieved remission
        'C92.41',  # Acute promyelocytic leukemia, in remission
        'C92.42',  # Acute promyelocytic leukemia, in relapse
        'C92.50',  # Acute myelomonocytic leukemia, not having achieved remission
        'C92.51',  # Acute myelomonocytic leukemia, in remission
        'C92.52',  # Acute myelomonocytic leukemia, in relapse
        'C92.60',  # Acute myeloid leukemia with 11q23-abnormality not having achieved remission
        'C92.61',  # Acute myeloid leukemia with 11q23-abnormality in remission
        'C92.62',  # Acute myeloid leukemia with 11q23-abnormality in relapse
        'C92.90',  # Myeloid leukemia, unspecified, not having achieved remission
        'C92.91',  # Myeloid leukemia, unspecified in remission
        'C92.92',  # Myeloid leukemia, unspecified in relapse
        'C92.A0',  # Acute myeloid leukemia with multilineage dysplasia, not having achieved remission
        'C92.A1',  # Acute myeloid leukemia with multilineage dysplasia, in remission
        'C92.A2',  # Acute myeloid leukemia with multilineage dysplasia, in relapse
        'C92.Z0',  # Other myeloid leukemia not having achieved remission
        'C92.Z1',  # Other myeloid leukemia, in remission
        'C92.Z2',  # Other myeloid leukemia, in relapse
        'C93.00',  # Acute monoblastic/monocytic leukemia, not having achieved remission
        'C93.01',  # Acute monoblastic/monocytic leukemia, in remission
        'C93.02',  # Acute monoblastic/monocytic leukemia, in relapse
        'C93.10',  # Chronic myelomonocytic leukemia not having achieved remission
        'C93.11',  # Chronic myelomonocytic leukemia, in remission
        'C93.12',  # Chronic myelomonocytic leukemia, in relapse
        'C93.30',  # Juvenile myelomonocytic leukemia, not having achieved remission
        'C93.31',  # Juvenile myelomonocytic leukemia, in remission
        'C93.32',  # Juvenile myelomonocytic leukemia, in relapse
        'C93.90',  # Monocytic leukemia, unspecified, not having achieved remission
        'C93.91',  # Monocytic leukemia, unspecified in remission
        'C93.92',  # Monocytic leukemia, unspecified in relapse
        'C93.Z0',  # Other monocytic leukemia, not having achieved remission
        'C93.Z1',  # Other monocytic leukemia, in remission
        'C93.Z2',  # Other monocytic leukemia, in relapse
        'C94.00',  # Acute erythroid leukemia, not having achieved remission
        'C94.01',  # Acute erythroid leukemia, in remission
        'C94.02',  # Acute erythroid leukemia, in relapse
        'C94.20',  # Acute megakaryoblastic leukemia not having achieved remission
        'C94.21',  # Acute megakaryoblastic leukemia, in remission
        'C94.22',  # Acute megakaryoblastic leukemia, in relapse
        'C94.30',  # Mast cell leukemia not having achieved remission
        'C94.31',  # Mast cell leukemia, in remission
        'C94.32',  # Mast cell leukemia, in relapse
        'C94.80',  # Other specified leukemias not having achieved remission
        'C94.81',  # Other specified leukemias, in remission
        'C94.82',  # Other specified leukemias, in relapse
        'C95.00',  # Acute leukemia of unspecified cell type not having achieved remission
        'C95.01',  # Acute leukemia of unspecified cell type, in remission
        'C95.02',  # Acute leukemia of unspecified cell type, in relapse
        'C95.10',  # Chronic leukemia of unspecified cell type not having achieved remission
        'C95.11',  # Chronic leukemia of unspecified cell type, in remission
        'C95.12',  # Chronic leukemia of unspecified cell type, in relapse
        'C95.90',  # Leukemia, unspecified not having achieved remission
        'C95.91',  # Leukemia, unspecified, in remission
        'C95.92',  # Leukemia, unspecified, in relapse
        'C96.0',  # Multifocal and multisystemic (disseminated) Langerhans-cell histiocytosis
        'C96.2',  # Malignant mast cell neoplasm
        'C96.20',  # Malignant mast cell neoplasm, unspecified
        'C96.21',  # Aggressive systemic mastocytosis
        'C96.22',  # Mast cell sarcoma
        'C96.29',  # Other malignant mast cell neoplasm
        'C96.4',  # Sarcoma of dendritic cells (accessory cells)
        'C96.9',  # Malignant neoplasm of lymphoid, hematopoietic and related tissue, unspecified
        'C96.A',  # Histiocytic sarcoma
        'C96.Z',  # Other specified malignant neoplasms of lymphoid, hematopoietic and related tissue
    }
    SNOMEDCT = {
        '13048006',  # Orbital lymphoma (disorder)
        '31047003',  # Lymphomatoid papulosis (disorder)
        '58961005',  # Lethal midline granuloma (disorder)
        '91854005',  # Acute leukemia in remission (disorder)
        '91855006',  # Acute leukemia, disease (disorder)
        '91856007',  # Acute lymphoid leukemia in remission (disorder)
        '91857003',  # Acute lymphoid leukemia, disease (disorder)
        '91858008',  # Acute monocytic leukemia in remission (disorder)
        '91860005',  # Acute myeloid leukemia in remission (disorder)
        '91861009',  # Acute myeloid leukemia, disease (disorder)
        '92516002',  # Burkitt's tumor of extranodal AND/OR solid organ site (disorder)
        '92811003',  # Chronic leukemia in remission (disorder)
        '92812005',  # Chronic leukemia, disease (disorder)
        '92813000',  # Chronic lymphoid leukemia in remission (disorder)
        '92814006',  # Chronic lymphoid leukemia, disease (disorder)
        '92817004',  # Chronic myeloid leukemia in remission (disorder)
        '92818009',  # Chronic myeloid leukemia, disease (disorder)
        '93133006',  # Letterer-Siwe disease of intra-abdominal lymph nodes (disorder)
        '93134000',  # Letterer-Siwe disease of intrapelvic lymph nodes (disorder)
        '93135004',  # Letterer-Siwe disease of intrathoracic lymph nodes (disorder)
        '93136003',  # Letterer-Siwe disease of lymph nodes of axilla AND/OR upper limb (disorder)
        '93137007',  # Letterer-Siwe disease of lymph nodes of head, face AND/OR neck (disorder)
        '93138002',  # Letterer-Siwe disease of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93139005',  # Letterer-Siwe disease of lymph nodes of multiple sites (disorder)
        '93140007',  # Letterer-Siwe disease of spleen (disorder)
        '93141006',  # Letterer-Siwe disease of extranodal AND/OR solid organ site (disorder)
        '93142004',  # Leukemia in remission (disorder)
        '93143009',  # Leukemia, disease (disorder)
        '93144003',  # Leukemic reticuloendotheliosis of intra-abdominal lymph nodes (disorder)
        '93145002',  # Leukemic reticuloendotheliosis of intrapelvic lymph nodes (disorder)
        '93146001',  # Leukemic reticuloendotheliosis of intrathoracic lymph nodes (disorder)
        '93150008',  # Leukemic reticuloendotheliosis of lymph nodes of multiple sites (disorder)
        '93151007',  # Hairy cell leukemia of spleen (disorder)
        '93152000',  # Leukemic reticuloendotheliosis of extranodal AND/OR solid organ site (disorder)
        '93169003',  # Lymphoid leukemia in remission (disorder)
        '93182006',  # Malignant histiocytosis of intra-abdominal lymph nodes (disorder)
        '93183001',  # Malignant histiocytosis of intrapelvic lymph nodes (disorder)
        '93184007',  # Malignant histiocytosis of intrathoracic lymph nodes (disorder)
        '93185008',  # Malignant histiocytosis of lymph nodes of axilla AND/OR upper limb (disorder)
        '93186009',  # Malignant histiocytosis of lymph nodes of head, face AND/OR neck (disorder)
        '93187000',  # Malignant histiocytosis of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93188005',  # Malignant histiocytosis of lymph nodes of multiple sites (disorder)
        '93189002',  # Malignant histiocytosis of spleen (disorder)
        '93190006',  # Malignant histiocytosis of extranodal AND/OR solid organ site (disorder)
        '93191005',  # Malignant lymphoma of intra-abdominal lymph nodes (disorder)
        '93192003',  # Malignant lymphoma of intrapelvic lymph nodes (disorder)
        '93193008',  # Malignant lymphoma of intrathoracic lymph nodes (disorder)
        '93194002',  # Malignant lymphoma of lymph nodes of axilla AND/OR upper limb (disorder)
        '93195001',  # Malignant lymphoma of lymph nodes of head, face AND/OR neck (disorder)
        '93196000',  # Malignant lymphoma of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93197009',  # Malignant lymphoma of lymph nodes of multiple sites (disorder)
        '93198004',  # Malignant lymphoma of spleen (disorder)
        '93199007',  # Malignant lymphoma of extranodal AND/OR solid organ site (disorder)
        '93200005',  # Malignant mast cell tumor of intra-abdominal lymph nodes (disorder)
        '93201009',  # Malignant mast cell tumor of intrapelvic lymph nodes (disorder)
        '93202002',  # Malignant mast cell tumor of intrathoracic lymph nodes (disorder)
        '93203007',  # Malignant mast cell tumor of lymph nodes of axilla AND/OR upper limb (disorder)
        '93204001',  # Malignant mast cell tumor of lymph nodes of head, face AND/OR neck (disorder)
        '93205000',  # Malignant mast cell tumor of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93206004',  # Malignant mast cell tumor of lymph nodes of multiple sites (disorder)
        '93207008',  # Malignant mast cell tumor of spleen (disorder)
        '93208003',  # Malignant mast cell tumor of extranodal AND/OR solid organ site (disorder)
        '93451002',  # Erythroleukemia, FAB M6 (disorder)
        '93487009',  # Hodgkin's disease, lymphocytic depletion of lymph nodes of axilla AND/OR upper limb (disorder)
        '93488004',  # Hodgkin's disease, lymphocytic depletion of lymph nodes of head, face AND/OR neck (disorder)
        '93489007',  # Hodgkin's disease, lymphocytic depletion of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93492006',  # Hodgkin's disease, lymphocytic depletion of extranodal AND/OR solid organ site (disorder)
        '93493001',  # Hodgkin's disease, lymphocytic-histiocytic predominance of intra-abdominal lymph nodes (disorder)
        '93494007',  # Hodgkin's disease, lymphocytic-histiocytic predominance of intrapelvic lymph nodes (disorder)
        '93495008',  # Hodgkin's disease, lymphocytic-histiocytic predominance of intrathoracic lymph nodes (disorder)
        '93496009',  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of axilla AND/OR upper limb (disorder)
        '93497000',  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of head, face AND/OR neck (disorder)
        '93498005',  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93500006',  # Hodgkin's disease, lymphocytic-histiocytic predominance of spleen (disorder)
        '93501005',  # Hodgkin's disease, lymphocytic-histiocytic predominance of extranodal AND/OR solid organ site (disorder)
        '93505001',  # Hodgkin's disease, mixed cellularity of lymph nodes of axilla AND/OR upper limb (disorder)
        '93506000',  # Hodgkin's disease, mixed cellularity of lymph nodes of head, face AND/OR neck (disorder)
        '93507009',  # Hodgkin's disease, mixed cellularity of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93509007',  # Hodgkin's disease, mixed cellularity of spleen (disorder)
        '93510002',  # Hodgkin's disease, mixed cellularity of extranodal AND/OR solid organ site (disorder)
        '93514006',  # Hodgkin's disease, nodular sclerosis of lymph nodes of axilla AND/OR upper limb (disorder)
        '93515007',  # Hodgkin's disease, nodular sclerosis of lymph nodes of head, face AND/OR neck (disorder)
        '93516008',  # Hodgkin's disease, nodular sclerosis of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93518009',  # Hodgkin's disease, nodular sclerosis of spleen (disorder)
        '93519001',  # Hodgkin's disease, nodular sclerosis of extranodal AND/OR solid organ site (disorder)
        '93520007',  # Hodgkin's disease of intra-abdominal lymph nodes (disorder)
        '93521006',  # Hodgkin's disease of intrapelvic lymph nodes (disorder)
        '93522004',  # Hodgkin's disease of intrathoracic lymph nodes (disorder)
        '93523009',  # Hodgkin's disease of lymph nodes of axilla AND/OR upper limb (disorder)
        '93524003',  # Hodgkin's disease of lymph nodes of head, face AND/OR neck (disorder)
        '93525002',  # Hodgkin's disease of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93526001',  # Hodgkin's disease of lymph nodes of multiple sites (disorder)
        '93527005',  # Hodgkin's disease of spleen (disorder)
        '93528000',  # Hodgkin's disease of extranodal AND/OR solid organ site (disorder)
        '93530003',  # Hodgkin's granuloma of intrapelvic lymph nodes (disorder)
        '93531004',  # Hodgkin's granuloma of intrathoracic lymph nodes (disorder)
        '93532006',  # Hodgkin's granuloma of lymph nodes of axilla AND/OR upper limb (disorder)
        '93533001',  # Hodgkin's granuloma of lymph nodes of head, face AND/OR neck (disorder)
        '93534007',  # Hodgkin's granuloma of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93536009',  # Hodgkin's granuloma of spleen (disorder)
        '93537000',  # Hodgkin's granuloma of extranodal AND/OR solid organ site (disorder)
        '93541001',  # Hodgkin's paragranuloma of lymph nodes of axilla AND/OR upper limb (disorder)
        '93542008',  # Hodgkin's paragranuloma of lymph nodes of head, face AND/OR neck (disorder)
        '93543003',  # Hodgkin's paragranuloma of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93546006',  # Hodgkin's paragranuloma of extranodal AND/OR solid organ site (disorder)
        '93547002',  # Hodgkin's sarcoma of intra-abdominal lymph nodes (disorder)
        '93548007',  # Hodgkin's sarcoma of intrapelvic lymph nodes (disorder)
        '93549004',  # Hodgkin's sarcoma of intrathoracic lymph nodes (disorder)
        '93550004',  # Hodgkin's sarcoma of lymph nodes of axilla AND/OR upper limb (disorder)
        '93551000',  # Hodgkin's sarcoma of lymph nodes of head, face AND/OR neck (disorder)
        '93552007',  # Hodgkin's sarcoma of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '93554008',  # Hodgkin's sarcoma of spleen (disorder)
        '93555009',  # Hodgkin's sarcoma of extranodal AND/OR solid organ site (disorder)
        '94071006',  # Primary malignant neoplasm of spleen (disorder)
        '94148006',  # Megakaryocytic leukemia in remission (disorder)
        '94686001',  # Mixed cell type lymphosarcoma of intra-abdominal lymph nodes (disorder)
        '94687005',  # Mixed cell type lymphosarcoma of intrapelvic lymph nodes (disorder)
        '94688000',  # Mixed cell type lymphosarcoma of intrathoracic lymph nodes (disorder)
        '94690004',  # Mixed cell type lymphosarcoma of lymph nodes of head, face, and neck (disorder)
        '94707004',  # Mycosis fungoides of intra-abdominal lymph nodes (disorder)
        '94708009',  # Mycosis fungoides of intrapelvic lymph nodes (disorder)
        '94709001',  # Mycosis fungoides of intrathoracic lymph nodes (disorder)
        '94710006',  # Mycosis fungoides of lymph nodes of axilla AND/OR upper limb (disorder)
        '94711005',  # Mycosis fungoides of lymph nodes of head, face AND/OR neck (disorder)
        '94712003',  # Mycosis fungoides of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '94714002',  # Mycosis fungoides of spleen (disorder)
        '94715001',  # Mycosis fungoides of extranodal AND/OR solid organ site (disorder)
        '94716000',  # Myeloid leukemia in remission (disorder)
        '94718004',  # Myeloid sarcoma in remission (disorder)
        '94719007',  # Myeloid sarcoma, disease (disorder)
        '95186006',  # Nodular lymphoma of intra-abdominal lymph nodes (disorder)
        '95187002',  # Nodular lymphoma of intrapelvic lymph nodes (disorder)
        '95188007',  # Nodular lymphoma of intrathoracic lymph nodes (disorder)
        '95192000',  # Nodular lymphoma of lymph nodes of multiple sites (disorder)
        '95193005',  # Nodular lymphoma of spleen (disorder)
        '95194004',  # Nodular lymphoma of extranodal AND/OR solid organ site (disorder)
        '95209008',  # Plasma cell leukemia in remission (disorder)
        '95210003',  # Plasma cell leukemia, disease (disorder)
        '95224004',  # Reticulosarcoma of intra-abdominal lymph nodes (disorder)
        '95225003',  # Reticulosarcoma of intrapelvic lymph nodes (disorder)
        '95226002',  # Reticulosarcoma of intrathoracic lymph nodes (disorder)
        '95230004',  # Reticulosarcoma of lymph nodes of multiple sites (disorder)
        '95231000',  # Reticulosarcoma of spleen (disorder)
        '95260009',  # Sézary's disease of lymph nodes of head, face AND/OR neck (disorder)
        '95261008',  # Sézary's disease of lymph nodes of inguinal region AND/OR lower limb (disorder)
        '95263006',  # Sézary's disease of spleen (disorder)
        '95264000',  # Sézary's disease of extranodal AND/OR solid organ site (disorder)
        '109962001',  # Diffuse non-Hodgkin's lymphoma (disorder)
        '109964000',  # Diffuse non-Hodgkin's lymphoma, undifferentiated (disorder)
        '109965004',  # Diffuse non-Hodgkin's lymphoma, lymphoblastic (disorder)
        '109966003',  # Diffuse non-Hodgkin's lymphoma, immunoblastic (disorder)
        '109967007',  # Diffuse non-Hodgkin's lymphoma, small cleaved cell (disorder)
        '109968002',  # Diffuse non-Hodgkin's lymphoma, small cell (disorder)
        '109969005',  # Diffuse non-Hodgkin's lymphoma, large cell (disorder)
        '109970006',  # Follicular non-Hodgkin's lymphoma, small cleaved cell (disorder)
        '109971005',  # Follicular non-Hodgkin's lymphoma, mixed small cleaved cell and large cell (disorder)
        '109972003',  # Follicular non-Hodgkin's lymphoma, large cell (disorder)
        '109975001',  # T-zone lymphoma (disorder)
        '109976000',  # Lymphoepithelioid lymphoma (disorder)
        '109977009',  # Peripheral T-cell lymphoma (disorder)
        '109978004',  # T-cell lymphoma (disorder)
        '109979007',  # B-cell lymphoma (disorder)
        '109980005',  # Malignant immunoproliferative disease (disorder)
        '109988003',  # Histiocytic sarcoma (disorder)
        '109994006',  # Essential thrombocythemia (disorder)
        '109995007',  # Myelodysplastic syndrome (disorder)
        '109996008',  # Myelodysplastic syndrome: Refractory anemia, without ringed sideroblasts, without excess blasts (disorder)
        '109998009',  # Refractory anemia with ringed sideroblasts (disorder)
        '110000005',  # Refractory anemia with excess blasts in transformation (disorder)
        '110002002',  # Mast cell leukemia (disorder)
        '110004001',  # Acute promyelocytic leukemia, FAB M3 (disorder)
        '110005000',  # Acute myelomonocytic leukemia, FAB M4 (disorder)
        '110006004',  # Prolymphocytic leukemia (disorder)
        '110007008',  # Adult T-cell leukemia/lymphoma (disorder)
        '118599009',  # Hodgkin's disease (disorder)
        '118600007',  # Malignant lymphoma (disorder)
        '118601006',  # Non-Hodgkin's lymphoma (disorder)
        '118602004',  # Hodgkin's granuloma (disorder)
        '118605002',  # Hodgkin lymphoma, nodular lymphocyte predominance (disorder)
        '118606001',  # Hodgkin's sarcoma (disorder)
        '118607005',  # Hodgkin lymphoma, lymphocyte-rich (disorder)
        '118608000',  # Hodgkin's disease, nodular sclerosis (disorder)
        '118609008',  # Hodgkin's disease, mixed cellularity (disorder)
        '118610003',  # Hodgkin's disease, lymphocytic depletion (disorder)
        '118611004',  # Sézary's disease (disorder)
        '118612006',  # Malignant histiocytosis (disorder)
        '118613001',  # Hairy cell leukemia (disorder)
        '118614007',  # Langerhans cell histiocytosis, disseminated (disorder)
        '118615008',  # Malignant mast cell tumor (disorder)
        '118617000',  # Burkitt's lymphoma (disorder)
        '118618005',  # Mycosis fungoides (disorder)
        '127070008',  # Malignant histiocytic disorder (disorder)
        '127220001',  # Malignant lymphoma of lymph nodes (disorder)
        '127225006',  # Chronic myelomonocytic leukemia (disorder)
        '128874001',  # Cutaneous CD30+ lymphoproliferative disorder (disorder)
        '128875000',  # Primary cutaneous CD30 antigen positive large T-cell lymphoma (disorder)
        '187822008',  # Fibrosarcoma of spleen (disorder)
        '188487008',  # Lymphosarcoma and reticulosarcoma (disorder)
        '188489006',  # Reticulosarcoma of lymph nodes of head, face and neck (disorder)
        '188492005',  # Reticulosarcoma of lymph nodes of axilla and upper limb (disorder)
        '188493000',  # Reticulosarcoma of lymph nodes of inguinal region and lower limb (disorder)
        '188498009',  # Lymphosarcoma (disorder)
        '188500005',  # Lymphosarcoma of lymph nodes of head, face and neck (disorder)
        '188501009',  # Lymphosarcoma of intrathoracic lymph nodes (disorder)
        '188502002',  # Lymphosarcoma of intra-abdominal lymph nodes (disorder)
        '188503007',  # Lymphosarcoma of lymph nodes of axilla and upper limb (disorder)
        '188504001',  # Lymphosarcoma of lymph nodes of inguinal region and lower limb (disorder)
        '188505000',  # Lymphosarcoma of intrapelvic lymph nodes (disorder)
        '188506004',  # Lymphosarcoma of spleen (disorder)
        '188507008',  # Lymphosarcoma of lymph nodes of multiple sites (disorder)
        '188511002',  # Burkitt's lymphoma of intrathoracic lymph nodes (disorder)
        '188512009',  # Burkitt's lymphoma of intra-abdominal lymph nodes (disorder)
        '188513004',  # Burkitt's lymphoma of lymph nodes of axilla and upper limb (disorder)
        '188514005',  # Burkitt's lymphoma of lymph nodes of inguinal region and lower limb (disorder)
        '188515006',  # Burkitt's lymphoma of intrapelvic lymph nodes (disorder)
        '188516007',  # Burkitt's lymphoma of spleen (disorder)
        '188517003',  # Burkitt's lymphoma of lymph nodes of multiple sites (disorder)
        '188524002',  # Hodgkin's paragranuloma of intrathoracic lymph nodes (disorder)
        '188529007',  # Hodgkin's paragranuloma of intrapelvic lymph nodes (disorder)
        '188531003',  # Hodgkin's paragranuloma of lymph nodes of multiple sites (disorder)
        '188536008',  # Hodgkin's granuloma of intra-abdominal lymph nodes (disorder)
        '188537004',  # Hodgkin's granuloma of lymph nodes of axilla and upper limb (disorder)
        '188538009',  # Hodgkin's granuloma of lymph nodes of inguinal region and lower limb (disorder)
        '188541000',  # Hodgkin's granuloma of lymph nodes of multiple sites (disorder)
        '188547001',  # Hodgkin's sarcoma of lymph nodes of axilla and upper limb (disorder)
        '188548006',  # Hodgkin's sarcoma of lymph nodes of inguinal region and lower limb (disorder)
        '188551004',  # Hodgkin's sarcoma of lymph nodes of multiple sites (disorder)
        '188554007',  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of head, face and neck (disorder)
        '188558005',  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of axilla and upper limb (disorder)
        '188559002',  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of inguinal region and lower limb (disorder)
        '188562004',  # Hodgkin's disease, lymphocytic-histiocytic predominance of lymph nodes of multiple sites (disorder)
        '188565002',  # Hodgkin's disease, nodular sclerosis of lymph nodes of head, face and neck (disorder)
        '188566001',  # Hodgkin's disease, nodular sclerosis of intrathoracic lymph nodes (disorder)
        '188567005',  # Hodgkin's disease, nodular sclerosis of intra-abdominal lymph nodes (disorder)
        '188568000',  # Hodgkin's disease, nodular sclerosis of lymph nodes of axilla and upper limb (disorder)
        '188569008',  # Hodgkin's disease, nodular sclerosis of lymph nodes of inguinal region and lower limb (disorder)
        '188570009',  # Hodgkin's disease, nodular sclerosis of intrapelvic lymph nodes (disorder)
        '188572001',  # Hodgkin's disease, nodular sclerosis of lymph nodes of multiple sites (disorder)
        '188575004',  # Hodgkin's disease, mixed cellularity of lymph nodes of head, face and neck (disorder)
        '188576003',  # Hodgkin's disease, mixed cellularity of intrathoracic lymph nodes (disorder)
        '188577007',  # Hodgkin's disease, mixed cellularity of intra-abdominal lymph nodes (disorder)
        '188578002',  # Hodgkin's disease, mixed cellularity of lymph nodes of axilla and upper limb (disorder)
        '188579005',  # Hodgkin's disease, mixed cellularity of lymph nodes of inguinal region and lower limb (disorder)
        '188580008',  # Hodgkin's disease, mixed cellularity of intrapelvic lymph nodes (disorder)
        '188582000',  # Hodgkin's disease, mixed cellularity of lymph nodes of multiple sites (disorder)
        '188585003',  # Hodgkin's disease, lymphocytic depletion of lymph nodes of head, face and neck (disorder)
        '188586002',  # Hodgkin's disease, lymphocytic depletion of intrathoracic lymph nodes (disorder)
        '188587006',  # Hodgkin's disease, lymphocytic depletion of intra-abdominal lymph nodes (disorder)
        '188589009',  # Hodgkin's disease, lymphocytic depletion of lymph nodes of axilla and upper limb (disorder)
        '188590000',  # Hodgkin's disease, lymphocytic depletion of lymph nodes of inguinal region and lower limb (disorder)
        '188591001',  # Hodgkin's disease, lymphocytic depletion of intrapelvic lymph nodes (disorder)
        '188592008',  # Hodgkin's disease, lymphocytic depletion of spleen (disorder)
        '188593003',  # Hodgkin's disease, lymphocytic depletion of lymph nodes of multiple sites (disorder)
        '188609000',  # Nodular lymphoma of lymph nodes of head, face and neck (disorder)
        '188612002',  # Nodular lymphoma of lymph nodes of axilla and upper limb (disorder)
        '188613007',  # Nodular lymphoma of lymph nodes of inguinal region and lower limb (disorder)
        '188627002',  # Mycosis fungoides of lymph nodes of multiple sites (disorder)
        '188631008',  # Sézary's disease of intrathoracic lymph nodes (disorder)
        '188632001',  # Sézary's disease of intra-abdominal lymph nodes (disorder)
        '188633006',  # Sézary's disease of lymph nodes of axilla and upper limb (disorder)
        '188634000',  # Sézary's disease of lymph nodes of inguinal region and lower limb (disorder)
        '188635004',  # Sézary's disease of intrapelvic lymph nodes (disorder)
        '188637007',  # Sézary's disease of lymph nodes of multiple sites (disorder)
        '188641006',  # Malignant histiocytosis of lymph nodes of axilla and upper limb (disorder)
        '188642004',  # Malignant histiocytosis of lymph nodes of inguinal region and lower limb (disorder)
        '188645002',  # Leukemic reticuloendotheliosis of lymph nodes of head, face and neck (disorder)
        '188648000',  # Leukemic reticuloendotheliosis of lymph nodes of axilla and upper limb (disorder)
        '188649008',  # Leukemic reticuloendotheliosis of lymph nodes of inguinal region and lower limb (disorder)
        '188660004',  # Malignant mast cell tumors (disorder)
        '188662007',  # Mast cell malignancy of lymph nodes of head, face and neck (disorder)
        '188663002',  # Mast cell malignancy of intrathoracic lymph nodes (disorder)
        '188664008',  # Mast cell malignancy of intra-abdominal lymph nodes (disorder)
        '188665009',  # Mast cell malignancy of lymph nodes of axilla and upper limb (disorder)
        '188666005',  # Mast cell malignancy of lymph nodes of inguinal region and lower limb (disorder)
        '188667001',  # Mast cell malignancy of intrapelvic lymph nodes (disorder)
        '188668006',  # Mast cell malignancy of spleen (disorder)
        '188669003',  # Mast cell malignancy of lymph nodes of multiple sites (disorder)
        '188672005',  # Follicular non-Hodgkin's mixed small cleaved and large cell lymphoma (disorder)
        '188674006',  # Diffuse malignant lymphoma - small non-cleaved cell (disorder)
        '188675007',  # Malignant lymphoma - small cleaved cell (disorder)
        '188676008',  # Malignant lymphoma - mixed small and large cell (disorder)
        '188679001',  # Diffuse non-Hodgkin's lymphoma undifferentiated (diffuse) (disorder)
        '188718006',  # Malignant plasma cell neoplasm, extramedullary plasmacytoma (disorder)
        '188725004',  # Lymphoid leukemia (disorder)
        '188726003',  # Subacute lymphoid leukemia (disorder)
        '188728002',  # Aleukemic lymphoid leukemia (disorder)
        '188729005',  # Adult T-cell leukemia (disorder)
        '188732008',  # Myeloid leukemia (disorder)
        '188733003',  # Chronic eosinophilic leukemia (disorder)
        '188734009',  # Chronic neutrophilic leukemia (disorder)
        '188736006',  # Subacute myeloid leukemia (disorder)
        '188737002',  # Chloroma (disorder)
        '188738007',  # Granulocytic sarcoma (disorder)
        '188741003',  # Aleukemic myeloid leukemia (disorder)
        '188744006',  # Monocytic leukemia (disorder)
        '188745007',  # Chronic monocytic leukemia (disorder)
        '188746008',  # Subacute monocytic leukemia (disorder)
        '188748009',  # Aleukemic monocytic leukemia (disorder)
        '188754005',  # Megakaryocytic leukemia (disorder)
        '188768003',  # Myelomonocytic leukemia (disorder)
        '188770007',  # Subacute myelomonocytic leukemia (disorder)
        '189509003',  # Refractory anemia without sideroblasts, so stated (disorder)
        '190030009',  # Compound leukemias (disorder)
        '232075002',  # Lymphoma of retina (disorder)
        '236512004',  # Leukemic infiltrate of kidney (disorder)
        '236513009',  # Lymphoma of kidney (disorder)
        '240531002',  # African Burkitt's lymphoma (disorder)
        '254792006',  # Proliferating angioendotheliomatosis (disorder)
        '255101006',  # Sézary disease of skin (disorder)
        '255102004',  # Angioendotheliomatosis (disorder)
        '255191003',  # Localized malignant reticulohistiocytoma (disorder)
        '269475001',  # Malignant tumor of lymphoid, hemopoietic AND/OR related tissue (disorder)
        '269476000',  # Nodular lymphoma (disorder)
        '274905008',  # Malignant lymphoma - lymphocytic, intermediate differentiation (disorder)
        '276811008',  # Gastric lymphoma (disorder)
        '276815004',  # Lymphoma of intestine (disorder)
        '276836002',  # Primary cerebral lymphoma (disorder)
        '277473004',  # B-cell chronic lymphocytic leukemia (disorder)
        '277474005',  # B-cell chronic lymphocytic leukemia variant (disorder)
        '277545003',  # T-cell chronic lymphocytic leukemia (disorder)
        '277549009',  # Chronic lymphocytic prolymphocytic leukemia syndrome (disorder)
        '277550009',  # Richter's syndrome (disorder)
        '277551008',  # Splenic lymphoma with villous lymphocytes (disorder)
        '277567002',  # T-cell prolymphocytic leukemia (disorder)
        '277568007',  # Hairy cell leukemia variant (disorder)
        '277569004',  # Large granular lymphocytic leukemia (disorder)
        '277570003',  # Lymphoma with spill (disorder)
        '277571004',  # B-cell acute lymphoblastic leukemia (disorder)
        '277572006',  # Precursor B-cell acute lymphoblastic leukemia (disorder)
        '277573001',  # Common acute lymphoblastic leukemia (disorder)
        '277574007',  # Null cell acute lymphoblastic leukemia (disorder)
        '277575008',  # T-cell acute lymphoblastic leukemia (disorder)
        '277587001',  # Juvenile chronic myeloid leukemia (disorder)
        '277589003',  # Atypical chronic myeloid leukemia (disorder)
        '277597005',  # Myelodysplastic syndrome with isolated del(5q) (disorder)
        '277601005',  # Acute monoblastic leukemia (disorder)
        '277602003',  # Acute megakaryoblastic leukemia (disorder)
        '277604002',  # Acute eosinophilic leukemia (disorder)
        '277609007',  # Hodgkin's disease, lymphocytic predominance - diffuse (disorder)
        '277610002',  # Hodgkin's disease, nodular sclerosis - lymphocytic predominance (disorder)
        '277611003',  # Hodgkin's disease, nodular sclerosis - mixed cellularity (disorder)
        '277612005',  # Hodgkin's disease, nodular sclerosis - lymphocytic depletion (disorder)
        '277613000',  # Cutaneous/peripheral T-cell lymphoma (disorder)
        '277614006',  # Prethymic and thymic T-cell lymphoma/leukemia (disorder)
        '277615007',  # Low grade B-cell lymphoma (disorder)
        '277616008',  # Diffuse low grade B-cell lymphoma (disorder)
        '277617004',  # High grade B-cell lymphoma (disorder)
        '277618009',  # Follicular low grade B-cell lymphoma (disorder)
        '277619001',  # B-cell prolymphocytic leukemia (disorder)
        '277622004',  # Mucosa-associated lymphoma (disorder)
        '277623009',  # Monocytoid B-cell lymphoma (disorder)
        '277624003',  # Follicular malignant lymphoma - mixed cell type (disorder)
        '277625002',  # Follicular malignant lymphoma - small cleaved cell (disorder)
        '277626001',  # Diffuse high grade B-cell lymphoma (disorder)
        '277627005',  # Nodular high grade B-cell lymphoma (disorder)
        '277628000',  # Diffuse malignant lymphoma - large cleaved cell (disorder)
        '277629008',  # Diffuse malignant lymphoma - large non-cleaved cell (disorder)
        '277632006',  # Diffuse malignant lymphoma - centroblastic polymorphic (disorder)
        '277637000',  # Large cell anaplastic lymphoma (disorder)
        '277642008',  # Low grade T-cell lymphoma (disorder)
        '277643003',  # High grade T-cell lymphoma (disorder)
        '277651000',  # Peripheral T-cell lymphoma - pleomorphic small cell (disorder)
        '277653002',  # Peripheral T-cell lymphoma - pleomorphic medium and large cell (disorder)
        '277654008',  # Enteropathy-associated T-cell lymphoma (disorder)
        '277664004',  # Malignant lymphoma of testis (disorder)
        '278051002',  # Malignant lymphoma of thyroid gland (disorder)
        '278052009',  # Malignant lymphoma of breast (disorder)
        '278189009',  # Hypergranular promyelocytic leukemia (disorder)
        '278453007',  # Acute biphenotypic leukemia (disorder)
        '285769009',  # Acute promyelocytic leukemia - hypogranular variant (disorder)
        '285776004',  # Intermediate grade B-cell lymphoma (disorder)
        '285839005',  # Acute myelomonocytic leukemia - eosinophilic variant (disorder)
        '302841002',  # Malignant lymphoma - small lymphocytic (disorder)
        '302842009',  # Diffuse malignant lymphoma - centroblastic (disorder)
        '302845006',  # Nodular malignant lymphoma, lymphocytic - well differentiated (disorder)
        '302848008',  # Nodular malignant lymphoma, lymphocytic - intermediate differentiation (disorder)
        '302855005',  # Subacute leukemia (disorder)
        '302856006',  # Aleukemic leukemia (disorder)
        '303017006',  # Malignant lymphoma, convoluted cell type (disorder)
        '303055001',  # Malignant lymphoma, follicular center cell (disorder)
        '303056000',  # Malignant lymphoma, follicular center cell, cleaved (disorder)
        '303057009',  # Malignant lymphoma, follicular center cell, non-cleaved (disorder)
        '307340003',  # Monosomy 7 syndrome (disorder)
        '307341004',  # Atypical hairy cell leukemia (disorder)
        '307592006',  # Basophilic leukemia (disorder)
        '307617006',  # Neutrophilic leukemia (disorder)
        '307622006',  # Prolymphocytic lymphosarcoma (disorder)
        '307623001',  # Malignant lymphoma - lymphoplasmacytic (disorder)
        '307624007',  # Diffuse malignant lymphoma - centroblastic-centrocytic (disorder)
        '307625008',  # Malignant lymphoma - centrocytic (disorder)
        '307633009',  # Hodgkin's disease, lymphocytic depletion, diffuse fibrosis (disorder)
        '307634003',  # Hodgkin's disease, lymphocytic depletion, reticular type (disorder)
        '307635002',  # Hodgkin's disease, nodular sclerosis - cellular phase (disorder)
        '307636001',  # Malignant lymphoma, mixed lymphocytic-histiocytic, nodular (disorder)
        '307637005',  # Malignant lymphoma, centroblastic-centrocytic, follicular (disorder)
        '307646004',  # Malignant lymphoma, lymphocytic, poorly differentiated, nodular (disorder)
        '307647008',  # Malignant lymphoma, centroblastic type, follicular (disorder)
        '307649006',  # Microglioma (disorder)
        '307650006',  # Histiocytic medullary reticulosis (disorder)
        '308121000',  # Follicular non-Hodgkin's lymphoma (disorder)
        '359631009',  # Acute myeloid leukemia, minimal differentiation, FAB M0 (disorder)
        '359640008',  # Acute myeloid leukemia without maturation, FAB M1 (disorder)
        '359648001',  # Acute myeloid leukemia with maturation, FAB M2 (disorder)
        '371012000',  # Acute lymphoblastic leukemia, transitional pre-B-cell (disorder)
        '371134001',  # Malignant lymphoma, large cell, polymorphous, immunoblastic (disorder)
        '373168002',  # Reticulosarcoma (disorder)
        '397008008',  # Aggressive lymphadenopathic mastocytosis with eosinophilia (disorder)
        '397009000',  # Mast cell malignancy (disorder)
        '397011009',  # Mast cell malignancy of lymph nodes (disorder)
        '397015000',  # Systemic mastocytosis with associated clonal hematological non-mast cell lineage disease (disorder)
        '398623004',  # Refractory anemia with excess blasts (disorder)
        '400001003',  # Primary cutaneous lymphoma (disorder)
        '400122007',  # Primary cutaneous T-cell lymphoma (disorder)
        '402880009',  # Primary cutaneous large T-cell lymphoma (disorder)
        '402881008',  # Primary cutaneous B-cell lymphoma (disorder)
        '402882001',  # Hodgkin's disease affecting skin (disorder)
        '404103007',  # Lymphomatoid papulosis type A (CD-30 positive type) (disorder)
        '404104001',  # Lymphomatoid papulosis type B - mycosis fungoides-like (disorder)
        '404105000',  # Lymphomatoid papulosis type C (anaplastic large-cell lymphoma-like) (disorder)
        '404106004',  # Lymphomatoid papulosis with Hodgkin's disease (disorder)
        '404107008',  # Patch/plaque stage mycosis fungoides (disorder)
        '404108003',  # Poikilodermatous mycosis fungoides (disorder)
        '404109006',  # Follicular mucinosis type mycosis fungoides (disorder)
        '404110001',  # Hypomelanotic mycosis fungoides (disorder)
        '404111002',  # Lymphomatoid papulosis-associated mycosis fungoides (disorder)
        '404112009',  # Granulomatous mycosis fungoides (disorder)
        '404113004',  # Tumor stage mycosis fungoides (disorder)
        '404114005',  # Erythrodermic mycosis fungoides (disorder)
        '404115006',  # Bullous mycosis fungoides (disorder)
        '404116007',  # Mycosis fungoides with systemic infiltration (disorder)
        '404117003',  # Spongiotic mycosis fungoides (disorder)
        '404118008',  # Syringotropic mycosis fungoides (disorder)
        '404119000',  # Pagetoid reticulosis (disorder)
        '404120006',  # Localized pagetoid reticulosis (disorder)
        '404121005',  # Generalized pagetoid reticulosis (disorder)
        '404122003',  # Leukemic infiltration of skin (chronic T-cell lymphocytic leukemia) (disorder)
        '404123008',  # Leukemic infiltration of skin (T-cell prolymphocytic leukemia) (disorder)
        '404124002',  # Leukemic infiltration of skin (T-cell lymphoblastic leukemia) (disorder)
        '404126000',  # CD-30 positive pleomorphic large T-cell cutaneous lymphoma (disorder)
        '404127009',  # CD-30 positive T-immunoblastic cutaneous lymphoma (disorder)
        '404128004',  # CD-30 negative cutaneous T-cell lymphoma (disorder)
        '404129007',  # CD-30 negative anaplastic large T-cell cutaneous lymphoma (disorder)
        '404130002',  # CD-30 negative pleomorphic large T-cell cutaneous lymphoma (disorder)
        '404131003',  # CD-30 negative T-immunoblastic cutaneous lymphoma (disorder)
        '404133000',  # Subcutaneous panniculitic cutaneous T-cell lymphoma (disorder)
        '404134006',  # Anaplastic large T-cell systemic malignant lymphoma (disorder)
        '404135007',  # Angiocentric natural killer/T-cell malignant lymphoma involving skin (disorder)
        '404136008',  # Aggressive natural killer-cell leukemia involving skin (disorder)
        '404137004',  # Precursor B-cell lymphoblastic lymphoma involving skin (disorder)
        '404138009',  # Small lymphocytic B-cell lymphoma involving skin (disorder)
        '404139001',  # Leukemic infiltration of skin in hairy-cell leukemia (disorder)
        '404140004',  # Primary cutaneous marginal zone B-cell lymphoma (disorder)
        '404141000',  # Primary cutaneous immunocytoma (disorder)
        '404142007',  # Primary cutaneous plasmacytoma (disorder)
        '404143002',  # Primary cutaneous follicular center B-cell lymphoma (disorder)
        '404144008',  # Primary cutaneous diffuse large cell B-cell lymphoma (disorder)
        '404145009',  # Primary cutaneous anaplastic large cell B-cell lymphoma (disorder)
        '404147001',  # Follicular center B-cell lymphoma (nodal/systemic with skin involvement) (disorder)
        '404148006',  # Diffuse large B-cell lymphoma (nodal/systemic with skin involvement) (disorder)
        '404149003',  # Lymphoplasmacytic B-cell lymphoma, nodal/systemic with skin involvement (disorder)
        '404150003',  # Mantle cell B-cell lymphoma (nodal/systemic with skin involvement) (disorder)
        '404151004',  # Leukemic infiltration of skin in myeloid leukemia (disorder)
        '404152006',  # Leukemic infiltration of skin in acute myeloid leukemia (disorder)
        '404153001',  # Leukemic infiltration of skin in chronic myeloid leukemia (disorder)
        '404154007',  # Leukemic infiltration of skin in monocytic leukemia (disorder)
        '404155008',  # Granulocytic sarcoma affecting skin (disorder)
        '404157000',  # Specific skin infiltration in Hodgkin's disease (disorder)
        '404169008',  # Malignant histiocytosis involving skin (disorder)
        '404172001',  # Mast cell leukemia affecting skin (disorder)
        '413389003',  # Accelerated phase chronic myeloid leukemia (disorder)
        '413441006',  # Acute monocytic leukemia, FAB M5b (disorder)
        '413442004',  # Acute monocytic/monoblastic leukemia (disorder)
        '413537009',  # Angioimmunoblastic T-cell lymphoma (disorder)
        '413656006',  # Blastic phase chronic myeloid leukemia (disorder)
        '413842007',  # Chronic myeloid leukemia in lymphoid blast crisis (disorder)
        '413843002',  # Chronic myeloid leukemia in myeloid blast crisis (disorder)
        '413847001',  # Chronic phase chronic myeloid leukemia (disorder)
        '414166008',  # Extranodal natural killer/T-cell lymphoma, nasal type (disorder)
        '414780005',  # Mucosa-associated lymphoid tissue lymphoma of orbit (disorder)
        '414785000',  # Multiple solitary plasmacytomas (disorder)
        '415112005',  # Plasmacytoma (disorder)
        '415283002',  # Refractory anemia with excess blasts-1 (disorder)
        '415284008',  # Refractory anemia with excess blasts-2 (disorder)
        '415285009',  # Refractory cytopenia with multilineage dysplasia (disorder)
        '415286005',  # Refractory cytopenia with multilineage dysplasia and ringed sideroblasts (disorder)
        '415287001',  # Relapsing chronic myeloid leukemia (disorder)
        '420302007',  # Reticulosarcoma associated with acquired immunodeficiency syndrome (disorder)
        '420519005',  # Malignant lymphoma of the eye region (disorder)
        '420788006',  # Intraocular non-Hodgkin malignant lymphoma (disorder)
        '421246008',  # Precursor T-cell lymphoblastic lymphoma (disorder)
        '421283008',  # Primary lymphoma of brain associated with acquired immunodeficiency syndrome (disorder)
        '422853008',  # Lymphoma of retroperitoneal space (disorder)
        '425688002',  # Philadelphia chromosome-positive acute lymphoblastic leukemia (disorder)
        '425749006',  # Subacute myeloid leukemia in remission (disorder)
        '425869007',  # Acute promyelocytic leukemia, FAB M3, in remission (disorder)
        '425941003',  # Precursor B-cell acute lymphoblastic leukemia in remission (disorder)
        '426071002',  # Hodgkin's disease in remission (disorder)
        '426124006',  # Acute myeloid leukemia with maturation, FAB M2, in remission (disorder)
        '426217000',  # Aleukemic leukemia in remission (disorder)
        '426248008',  # Aleukemic lymphoid leukemia in remission (disorder)
        '426336007',  # Solitary osseous myeloma (disorder)
        '426370008',  # Subacute lymphoid leukemia in remission (disorder)
        '426642002',  # Erythroleukemia, FAB M6 in remission (disorder)
        '426885008',  # Hodgkin's disease, lymphocytic depletion of lymph nodes of head (disorder)
        '427056005',  # Subacute leukemia in remission (disorder)
        '427141003',  # Malignant lymphoma in remission (disorder)
        '427642009',  # T-cell acute lymphoblastic leukemia in remission (disorder)
        '427658007',  # Acute myelomonocytic leukemia, FAB M4, in remission (disorder)
        '430338009',  # Smoldering chronic lymphocytic leukemia (disorder)
        '441559006',  # Mantle cell lymphoma of spleen (disorder)
        '441962003',  # Large cell lymphoma of intrapelvic lymph nodes (disorder)
        '442537007',  # Non-Hodgkin lymphoma associated with Human immunodeficiency virus infection (disorder)
        '443487006',  # Mantle cell lymphoma (disorder)
        '444597005',  # Extranodal marginal zone lymphoma of mucosa-associated lymphoid tissue of stomach (disorder)
        '444910004',  # Primary mediastinal (thymic) large B-cell lymphoma (disorder)
        '444911000',  # Acute myeloid leukemia with t(9:11)(p22;q23); MLLT3-MLL (disorder)
        '445227008',  # Juvenile myelomonocytic leukemia (disorder)
        '445269007',  # Extranodal marginal zone B-cell lymphoma of mucosa-associated lymphoid tissue (disorder)
        '445406001',  # Hepatosplenic T-cell lymphoma (disorder)
        '445448008',  # Acute myeloid leukemia with myelodysplasia-related changes (disorder)
        '447100004',  # Marginal zone lymphoma (disorder)
        '447656001',  # Lymphoma of pylorus of stomach (disorder)
        '447658000',  # Lymphoma of fundus of stomach (disorder)
        '447766003',  # Lymphoma of pyloric antrum of stomach (disorder)
        '447805007',  # Lymphoma of greater curvature of stomach (disorder)
        '447806008',  # Lymphoma of cardia of stomach (disorder)
        '447989004',  # Non-Hodgkin's lymphoma of extranodal site (disorder)
        '448212009',  # Anaplastic lymphoma kinase negative anaplastic large cell lymphoma (disorder)
        '448213004',  # Diffuse non-Hodgkin's lymphoma of prostate (disorder)
        '448217003',  # Follicular non-Hodgkin's lymphoma of prostate (disorder)
        '448220006',  # Non-Hodgkin's lymphoma of bone (disorder)
        '448231003',  # Follicular non-Hodgkin's lymphoma of nose (disorder)
        '448254007',  # Non-Hodgkin's lymphoma of central nervous system (disorder)
        '448269008',  # Lymphoma of lesser curvature of stomach (disorder)
        '448317000',  # Follicular non-Hodgkin's lymphoma of soft tissue (disorder)
        '448319002',  # Diffuse non-Hodgkin's lymphoma of nasopharynx (disorder)
        '448354009',  # Non-Hodgkin's lymphoma of intestine (disorder)
        '448371005',  # Non-Hodgkin's lymphoma of nasopharynx (disorder)
        '448372003',  # Non-Hodgkin's lymphoma of lung (disorder)
        '448376000',  # Non-Hodgkin's lymphoma of ovary (disorder)
        '448384001',  # Non-Hodgkin's lymphoma of nose (disorder)
        '448386004',  # Non-Hodgkin's lymphoma of oral cavity (disorder)
        '448387008',  # Non-Hodgkin's lymphoma of testis (disorder)
        '448447004',  # Non-Hodgkin's lymphoma of skin (disorder)
        '448465000',  # Diffuse non-Hodgkin's lymphoma of testis (disorder)
        '448468003',  # Diffuse non-Hodgkin's lymphoma of oral cavity (disorder)
        '448553002',  # Lymphoma of pelvis (disorder)
        '448555009',  # Lymphoma of body of stomach (disorder)
        '448560008',  # Diffuse non-Hodgkin's lymphoma of extranodal site (disorder)
        '448561007',  # Follicular non-Hodgkin's lymphoma of extranodal site (disorder)
        '448607004',  # Diffuse non-Hodgkin's lymphoma of uterine cervix (disorder)
        '448609001',  # Diffuse non-Hodgkin's lymphoma of ovary (disorder)
        '448663003',  # Diffuse non-Hodgkin's lymphoma of stomach (disorder)
        '448666006',  # Follicular non-Hodgkin's lymphoma of bone (disorder)
        '448672006',  # Follicular non-Hodgkin's lymphoma of lung (disorder)
        '448709005',  # Non-Hodgkin's lymphoma of stomach (disorder)
        '448738008',  # Non-Hodgkin's lymphoma of soft tissue (disorder)
        '448774004',  # Non-Hodgkin's lymphoma of uterine cervix (disorder)
        '448865007',  # Follicular non-Hodgkin's lymphoma of skin (disorder)
        '448867004',  # Diffuse non-Hodgkin's lymphoma of lung (disorder)
        '448995000',  # Follicular non-Hodgkin's lymphoma of central nervous system (disorder)
        '449053004',  # Lymphoma of lower esophagus (disorder)
        '449058008',  # Follicular non-Hodgkin's lymphoma of tonsil (disorder)
        '449059000',  # Follicular non-Hodgkin's lymphoma of uterine cervix (disorder)
        '449063007',  # Follicular non-Hodgkin's lymphoma of oral cavity (disorder)
        '449065000',  # Diffuse non-Hodgkin's lymphoma of nose (disorder)
        '449072004',  # Lymphoma of gastrointestinal tract (disorder)
        '449074003',  # Lymphoma of small intestine (disorder)
        '449075002',  # Lymphoma of cardioesophageal junction (disorder)
        '449108003',  # Philadelphia chromosome positive chronic myelogenous leukemia (disorder)
        '449173006',  # Diffuse non-Hodgkin's lymphoma of tonsil (disorder)
        '449176003',  # Diffuse non-Hodgkin's lymphoma of intestine (disorder)
        '449177007',  # Diffuse non-Hodgkin's lymphoma of bone (disorder)
        '449216004',  # Diffuse non-Hodgkin's lymphoma of soft tissue (disorder)
        '449217008',  # Diffuse non-Hodgkin's lymphoma of skin (disorder)
        '449218003',  # Lymphoma of sigmoid colon (disorder)
        '449219006',  # Follicular non-Hodgkin's lymphoma of nasopharynx (disorder)
        '449220000',  # Diffuse follicle center lymphoma (disorder)
        '449221001',  # Diffuse non-Hodgkin's lymphoma of central nervous system (disorder)
        '449222008',  # Follicular non-Hodgkin's lymphoma of stomach (disorder)
        '449292003',  # Non-Hodgkin's lymphoma of tonsil (disorder)
        '449307001',  # Follicular non-Hodgkin's lymphoma of ovary (disorder)
        '449318001',  # Non-Hodgkin's lymphoma of prostate (disorder)
        '449386007',  # Philadelphia chromosome negative chronic myelogenous leukemia (disorder)
        '449418000',  # Follicular non-Hodgkin's lymphoma of testis (disorder)
        '449419008',  # Follicular non-Hodgkin's lymphoma of intestine (disorder)
        '698646006',  # Acute monoblastic leukemia in remission (disorder)
        '699657009',  # Hepatosplenic gamma-delta cell lymphoma (disorder)
        '699818003',  # T-cell large granular lymphocytic leukemia (disorder)
        '702446006',  # Core binding factor acute myeloid leukemia (disorder)
        '702476004',  # Therapy-related myelodysplastic syndrome (disorder)
        '702785000',  # Large cell anaplastic lymphoma T cell and Null cell type (disorder)
        '702786004',  # Follicular non-Hodgkin's lymphoma diffuse follicle center sub-type grade 1 (disorder)
        '702977001',  # Follicular non-Hodgkin's lymphoma diffuse follicle center cell sub-type grade 2 (disorder)
        '703387000',  # Acute myeloid leukemia with normal karyotype (disorder)
        '703626001',  # Anaplastic large cell lymphoma, T/Null cell, primary systemic type (disorder)
        '705061009',  # Childhood myelodysplastic syndrome (disorder)
        '709471005',  # Periodontitis co-occurrent with leukemia (disorder)
        '713325002',  # Primary cerebral lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        '713483007',  # Reticulosarcoma co-occurrent with human immunodeficiency virus infection (disorder)
        '713516007',  # Primary effusion lymphoma (disorder)
        '713718006',  # Diffuse non-Hodgkin immunoblastic lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        '713897006',  # Burkitt lymphoma co-occurrent with human immunodeficiency virus infection (disorder)
        '714251006',  # Philadelphia chromosome-negative precursor B-cell acute lymphoblastic leukemia (disorder)
        '714463003',  # Primary effusion lymphoma co-occurrent with infection caused by Human herpesvirus 8 (disorder)
        '715664005',  # Interdigitating dendritic cell sarcoma (disorder)
        '715950008',  # Anaplastic lymphoma kinase positive large B-cell lymphoma (disorder)
        '716788007',  # Epstein-Barr virus positive diffuse large B-cell lymphoma of elderly (disorder)
        '718200007',  # Primary pulmonary lymphoma (disorder)
        '721302006',  # Refractory anemia with ringed sideroblasts associated with marked thrombocytosis (disorder)
        '721303001',  # Refractory neutropenia (disorder)
        '721304007',  # Refractory thrombocytopenia (disorder)
        '721305008',  # Acute myeloid leukemia due to recurrent genetic abnormality (disorder)
        '721306009',  # Therapy related acute myeloid leukemia and myelodysplastic syndrome (disorder)
        '721308005',  # Acute leukemia of ambiguous lineage (disorder)
        '721310007',  # Aggressive natural killer-cell leukemia (disorder)
        '721313009',  # Indeterminate dendritic cell neoplasm (disorder)
        '721314003',  # Fibroblastic reticular cell neoplasm (disorder)
        '721555001',  # Follicular lymphoma of small intestine (disorder)
        '721762007',  # Adult T-cell leukemia/lymphoma of skin (disorder)
        '722795004',  # Meningeal leukemia (disorder)
        '722953004',  # B-cell lymphoma unclassifiable with features intermediate between Burkitt lymphoma and diffuse large B-cell lymphoma (disorder)
        '722954005',  # B-cell lymphoma unclassifiable with features intermediate between classical Hodgkin lymphoma and diffuse large B-cell lymphoma (disorder)
        '723889003',  # B lymphoblastic leukemia lymphoma with t(9:22) (q34;q11.2); BCR-ABL 1 (disorder)
        '724644005',  # Myeloid leukemia co-occurrent with Down syndrome (disorder)
        '724645006',  # T-cell histiocyte rich large B-cell lymphoma (disorder)
        '724647003',  # Diffuse large B-cell lymphoma co-occurrent with chronic inflammation caused by Epstein-Barr virus (disorder)
        '724648008',  # Plasmablastic lymphoma (disorder)
        '724649000',  # Langerhans cell sarcoma (disorder)
        '724650000',  # Primary follicular dendritic cell sarcoma (disorder)
        '725390002',  # Acute myeloid leukemia with t(8;16)(p11;p13) translocation (disorder)
        '725437002',  # Chronic lymphocytic leukemia genetic mutation variant (disorder)
        '726721002',  # Nodal marginal zone B-cell lymphoma (disorder)
        '733598001',  # Acute myeloid leukemia with t(6;9)(p23;q34) translocation (disorder)
        '733627006',  # Primary cutaneous gamma-delta-positive T-cell lymphoma (disorder)
        '734066005',  # Diffuse large B-cell lymphoma of central nervous system (disorder)
        '734522002',  # Acute myeloid leukemia with FMS-like tyrosine kinase-3 mutation (disorder)
        '735332000',  # Primary cutaneous diffuse large cell B-cell lymphoma of lower extremity (disorder)
        '736322001',  # Pediatric follicular lymphoma (disorder)
        '738527001',  # Myeloid and/or lymphoid neoplasm associated with platelet derived growth factor receptor alpha rearrangement (disorder)
        '762315004',  # Therapy related acute myeloid leukemia due to and following administration of antineoplastic agent (disorder)
        '762690000',  # Classical Hodgkin lymphoma (disorder)
        '763309005',  # Acute myeloid leukemia with nucleophosmin 1 somatic mutation (disorder)
        '763477007',  # Primary lymphoma of conjunctiva (disorder)
        '763666008',  # Splenic marginal zone B-cell lymphoma (disorder)
        '763719001',  # Hydroa vacciniforme-like lymphoma (disorder)
        '763796007',  # Megakaryoblastic acute myeloid leukemia with t(1;22)(p13;q13) (disorder)
        '763884007',  # Splenic diffuse red pulp small B-cell lymphoma (disorder)
        '764855007',  # Acute myeloid leukemia with CCAAT/enhancer binding protein alpha somatic mutation (disorder)
        '764940002',  # Inherited acute myeloid leukemia (disorder)
        '765136002',  # Primary cutaneous CD8 positive aggressive epidermotropic cytotoxic T-cell lymphoma (disorder)
        '765328000',  # Classic mycosis fungoides (disorder)
        '766045006',  # Acute myeloid leukemia and myelodysplastic syndrome related to alkylating agent (disorder)
        '766046007',  # Acute myeloid leukemia and myelodysplastic syndrome related to topoisomerase type 2 inhibitor (disorder)
        '766048008',  # Acute myeloid leukemia and myelodysplastic syndrome related to radiation (disorder)
        '766935007',  # Primary bone lymphoma (disorder)
        '770402000',  # Aleukemic mast cell leukemia (disorder)
        '773537001',  # Differentiation syndrome due to and following chemotherapy co-occurrent with acute promyelocytic leukemia (disorder)
        '773995001',  # Primary cutaneous anaplastic large cell lymphoma (disorder)
        '780841002',  # Classic mast cell leukemia (disorder)
        '780844005',  # Acute myeloid leukemia with inv(3)(q21q26.2) or t(3;3)(q21;q26.2); RPN1-EVI1 (disorder)
        '783541009',  # Breast implant-associated anaplastic large-cell lymphoma (disorder)
        '783565007',  # Indolent T-cell lymphoproliferative disorder of gastrointestinal tract (disorder)
        '783615009',  # Erythropoietic uroporphyria associated with myeloid malignancy (disorder)
        '788874003',  # B-cell prolymphocytic leukemia in remission (disorder)
        '788972003',  # Juvenile myelomonocytic leukemia in remission (disorder)
        '830057003',  # Relapsing classical Hodgkin lymphoma (disorder)
        '836486002',  # Lymphomatous infiltrate of kidney (disorder)
        '838340006',  # B lymphoblastic leukemia lymphoma with t(5;14)(q31;q32); IL3-IGH (disorder)
        '838341005',  # B lymphoblastic leukemia lymphoma with t(v;11q23); MLL rearranged (disorder)
        '838342003',  # B lymphoblastic leukemia lymphoma with t(12;21) (p13;q22); TEL/AML1 (ETV6-RUNX1) (disorder)
        '838343008',  # B lymphoblastic leukemia lymphoma with t(1;19)(Q23;P13.3); E2A-PBX1 (TCF3/PBX1) (disorder)
        '838344002',  # B lymphoblastic leukemia lymphoma with hypodiploidy (disorder)
        '838346000',  # B lymphoblastic leukemia lymphoma with hyperdiploidy (disorder)
        '838355002',  # Acute myeloid leukemia with inv(16)(p13.1q22) or t(16;16)(p13.1;q22) CBFB-MYH11 (disorder)
        '840423002',  # Diffuse large B-cell lymphoma of small intestine (disorder)
        '840424008',  # Diffuse large B-cell lymphoma of stomach (disorder)
        '866098005',  # Large B-cell lymphoma arising in HHV8-associated multicentric Castleman disease (disorder)
        '12281000132104',  # Relapsing acute myeloid leukemia (disorder)
        '12291000132102',  # Refractory acute myeloid leukemia (disorder)
        '12301000132103',  # Acute lymphoid leukemia relapse (disorder)
        '12311000132101',  # Refractory acute lymphoid leukemia (disorder)
        '61291000119103',  # Disorder of central nervous system co-occurrent and due to acute lymphoid leukemia in remission (disorder)
        '61301000119102',  # Disorder of central nervous system co-occurrent and due to acute lymphoid leukemia (disorder)
        '61311000119104',  # Acute myeloid leukemia, minimal differentiation, FAB M0 in remission (disorder)
        '61321000119106',  # Acute myeloid leukemia without maturation, FAB M1 in remission (disorder)
        '68261000119102',  # Diffuse non-Hodgkin's lymphoma Lugano stage I (disorder)
        '68271000119108',  # Diffuse non-Hodgkin's lymphoma Lugano stage II (disorder)
        '68291000119109',  # Diffuse non-Hodgkin's lymphoma Lugano stage IV (disorder)
        '84811000119107',  # Follicular dendritic sarcoma of intraabdominal lymph node (disorder)
        '84831000119102',  # Follicular dendritic sarcoma of lymph nodes of multiple sites (disorder)
        '116691000119101',  # Marginal zone lymphoma of spleen (disorder)
        '116711000119103',  # Marginal zone lymphoma of inguinal lymph node (disorder)
        '116741000119104',  # Marginal zone lymphoma of thoracic lymph node (disorder)
        '116811000119106',  # Non-Hodgkin lymphoma of central nervous system metastatic to lymph node of lower limb (disorder)
        '116821000119104',  # Non-Hodgkin lymphoma of central nervous system metastatic to lymph node of upper limb (disorder)
        '116841000119105',  # Marginal zone lymphoma of lymph nodes of multiple sites (disorder)
        '116871000119103',  # Mantle cell lymphoma of lymph nodes of multiple sites (disorder)
        '117061000119101',  # Peripheral T-cell lymphoma of spleen (disorder)
        '117091000119108',  # Peripheral T-cell lymphoma of axillary lymph node (disorder)
        '117111000119100',  # Peripheral T-cell lymphoma of thoracic lymph node (disorder)
        '117211000119105',  # Peripheral T-cell lymphoma of lymph nodes of multiple sites (disorder)
        '122881000119107',  # Chronic monocytic leukemia in relapse (disorder)
        '122901000119109',  # Myeloid leukemia in relapse (disorder)
        '122951000119108',  # Lymphoid leukemia in relapse (disorder)
        '122961000119105',  # Chronic lymphoid leukemia in relapse (disorder)
        '122981000119101',  # Plasma cell leukemia in relapse (disorder)
        '123781000119107',  # Chronic leukemia in relapse (disorder)
        '127961000119108',  # Aleukemic myeloid leukemia in relapse (disorder)
        '133751000119102',  # Lymphoma of colon (disorder)
        '350951000119101',  # B-cell lymphoma of intra-abdominal lymph nodes (disorder)
        '351211000119104',  # B-cell lymphoma of lymph nodes of multiple sites (disorder)
        '352251000119109',  # Small lymphocytic B-cell lymphoma of lymph nodes of multiple sites (disorder)
        '352411000119109',  # Small lymphocytic B-cell lymphoma of intra-abdominal lymph nodes (disorder)
        '352791000119108',  # Non-Hodgkin's lymphoma of lymph nodes of multiple sites (disorder)
        '354851000119101',  # Follicular non-Hodgkin's lymphoma of lymph nodes of multiple sites (disorder)
        '450521000124109',  # Marginal zone lymphoma of axillary lymph node (disorder)
        '456911000124108',  # Diffuse non-Hodgkin's lymphoma Lugano stage III (disorder)
        '461311000124107',  # Refractory Hodgkin lymphoma (disorder)
        '461351000124108',  # Breast implant-associated anaplastic large cell lymphoma (disorder)
    }


class Measles(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  measles infections.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of measles infections.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = 'Measles'
    OID = '2.16.840.1.113883.3.464.1003.110.12.1053'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'B05.0',  # Measles complicated by encephalitis
        'B05.1',  # Measles complicated by meningitis
        'B05.2',  # Measles complicated by pneumonia
        'B05.3',  # Measles complicated by otitis media
        'B05.4',  # Measles with intestinal complications
        'B05.81',  # Measles keratitis and keratoconjunctivitis
        'B05.89',  # Other measles complications
        'B05.9',  # Measles without complication
    }
    SNOMEDCT = {
        '14189004',  # Measles (disorder)
        '28463004',  # Measles keratitis (disorder)
        '38921001',  # Measles with complication (disorder)
        '60013002',  # Measles keratoconjunctivitis (disorder)
        '74918002',  # Measles myocarditis (disorder)
        '111873003',  # Measles without complication (disorder)
        '161419000',  # History of measles (situation)
        '186561002',  # Measles complicated by meningitis (disorder)
        '186562009',  # Measles with intestinal complications (disorder)
        '195900001',  # Pneumonia co-occurrent and due to measles (disorder)
        '240483006',  # Atypical measles (disorder)
        '240484000',  # Modified measles (disorder)
        '359686005',  # Van Bogaert's sclerosing leukoencephalitis (disorder)
        '371111005',  # Measles immune (finding)
        '406592004',  # Measles of the central nervous system (disorder)
        '105841000119101',  # Immunity to measles by positive serology (finding)
    }


class Mumps(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  mumps infections.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of mumps infections.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = 'Mumps'
    OID = '2.16.840.1.113883.3.464.1003.110.12.1032'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'B26.0',  # Mumps orchitis
        'B26.1',  # Mumps meningitis
        'B26.2',  # Mumps encephalitis
        'B26.3',  # Mumps pancreatitis
        'B26.81',  # Mumps hepatitis
        'B26.82',  # Mumps myocarditis
        'B26.83',  # Mumps nephritis
        'B26.84',  # Mumps polyneuropathy
        'B26.85',  # Mumps arthritis
        'B26.89',  # Other mumps complications
        'B26.9',  # Mumps without complication
    }
    SNOMEDCT = {
        '10665004',  # Mumps pancreatitis (disorder)
        '17121006',  # Mumps nephritis (disorder)
        '31524007',  # Mumps polyneuropathy (disorder)
        '31646008',  # Mumps encephalitis (disorder)
        '36989005',  # Mumps (disorder)
        '40099009',  # Meningoencephalitis caused by mumps (disorder)
        '44201003',  # Mumps meningitis (disorder)
        '63462008',  # Mumps myocarditis (disorder)
        '72071001',  # Mumps thyroiditis (disorder)
        '74717002',  # Mumps with complication (disorder)
        '75548002',  # Mumps oophoritis (disorder)
        '78580004',  # Mumps orchitis (disorder)
        '89231008',  # Mumps hepatitis (disorder)
        '89764009',  # Mumps arthritis (disorder)
        '111870000',  # Mumps without complication (disorder)
        '161420006',  # History of mumps (situation)
        '235123001',  # Mumps acute sialadenitis (disorder)
        '236771002',  # Mumps epididymo-orchitis (disorder)
        '237443002',  # Mastitis of mumps (disorder)
        '240526004',  # Mumps parotitis (disorder)
        '240527008',  # Mumps submandibular sialadenitis (disorder)
        '240529006',  # Mumps sublingual sialadenitis (disorder)
        '371112003',  # Mumps immune (finding)
        '105821000119107',  # Immunity to mumps by positive serology (finding)
    }


class Rubella(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  rubella infections.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of rubella infections.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = 'Rubella'
    OID = '2.16.840.1.113883.3.464.1003.110.12.1037'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'B06.00',  # Rubella with neurological complication, unspecified
        'B06.01',  # Rubella encephalitis
        'B06.02',  # Rubella meningitis
        'B06.09',  # Other neurological complications of rubella
        'B06.81',  # Rubella pneumonia
        'B06.82',  # Rubella arthritis
        'B06.89',  # Other rubella complications
        'B06.9',  # Rubella without complication
    }
    SNOMEDCT = {
        '10082001',  # Progressive rubella panencephalitis (disorder)
        '13225007',  # Rubella meningoencephalitis (disorder)
        '19431000',  # Arthritis co-occurrent and due to rubella (disorder)
        '36653000',  # Rubella (disorder)
        '51490003',  # Disease caused by Rubivirus (disorder)
        '64190005',  # Rubella myocarditis (disorder)
        '79303006',  # Expanded rubella syndrome (disorder)
        '128191000',  # Encephalomyelitis co-occurrent and due to rubella (disorder)
        '161421005',  # History of rubella (situation)
        '165792000',  # Rubella antibody present - immune (finding)
        '186567003',  # Rubella with neurological complication (disorder)
        '186570004',  # Rubella deafness (disorder)
        '192689006',  # Rubella encephalitis (disorder)
        '231985001',  # Rubella retinopathy (disorder)
        '232312000',  # Endocochlear rubella (disorder)
        '240485004',  # Hemorrhagic rubella (disorder)
        '253227001',  # Rubella cataract (disorder)
        '406112006',  # Rubella enzyme-linked immunosorbent assay test result, greater than 10iu/ml rubella specific immunoglobulin G detected (finding)
        '406113001',  # Rubella enzyme-linked immunosorbent assay test result, greater than 15iu/ml rubella specific immunoglobulin G detected (finding)
        '1092361000119109',  # Rubella pneumonia (disorder)
        '10759761000119100',  # Rubella in mother complicating childbirth (disorder)
    }


class SevereCombinedImmunodeficiency(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  severe combined immunodeficiency including both genetic and other causes.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a severe combined immunodeficiency including genetic conditions and other causes.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = 'Severe Combined Immunodeficiency'
    OID = '2.16.840.1.113883.3.464.1003.120.12.1007'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'D81.0',  # Severe combined immunodeficiency [SCID] with reticular dysgenesis
        'D81.1',  # Severe combined immunodeficiency [SCID] with low T- and B-cell numbers
        'D81.2',  # Severe combined immunodeficiency [SCID] with low or normal B-cell numbers
        'D81.9',  # Combined immunodeficiency, unspecified
    }
    SNOMEDCT = {
        '3439009',  # Severe combined immunodeficiency due to absent peripheral T cell maturation (disorder)
        '22406001',  # Severe combined immunodeficiency due to absent lymphoid stem cells (disorder)
        '31323000',  # Severe combined immunodeficiency disease (disorder)
        '36980009',  # Severe combined immunodeficiency due to absent adenosine deaminase (disorder)
        '44940001',  # Adenosine deaminase deficiency (disorder)
        '45390000',  # Severe combined immunodeficiency due to absent interleukin-2 production (disorder)
        '49555001',  # Severe combined immunodeficiency due to absent T cell receptor (disorder)
        '55602000',  # Nezelof's syndrome (disorder)
        '71904008',  # Severe combined immunodeficiency due to absent class II human leukocyte antigens (disorder)
        '111584000',  # Reticular dysgenesis (disorder)
        '111587007',  # Severe combined immunodeficiency due to absent interleukin-2 receptor (disorder)
        '190993005',  # Autosomal recessive severe combined immunodeficiency (disorder)
        '190996002',  # Severe combined immunodeficiency with reticular dysgenesis (disorder)
        '190997006',  # Severe combined immunodeficiency with low T- and B-cell numbers (disorder)
        '190998001',  # Severe combined immunodeficiency with low or normal B-cell numbers (disorder)
        '191001007',  # Major histocompatibility complex class I deficiency (disorder)
        '191002000',  # Major histocompatibility complex class II deficiency (disorder)
        '234570002',  # Severe combined immunodeficiency with maternofetal engraftment (disorder)
        '234571003',  # Warts, hypogammaglobulinemia, infections, and myelokathexis (disorder)
        '350353007',  # De Vaal's syndrome (disorder)
        '351287008',  # Reticular dysgenesis with congenital aleukocytosis (disorder)
        '362993009',  # Autosomal recessive severe combined immunodeficiency disease (disorder)
        '715982006',  # Severe combined immunodeficiency due to deoxyribonucleic acid cross-link repair protein 1c deficiency (disorder)
        '716378008',  # Combined immunodeficiency due to Zeta-chain associated protein kinase 70 deficiency (disorder)
        '716871006',  # Severe combined immunodeficiency due to deoxyribonucleic acid dependent protein kinase catalytic subunit deficiency (disorder)
        '718107000',  # Severe combined immunodeficiency T-cell negative B-cell positive due to janus kinase-3 deficiency (disorder)
        '720345008',  # Severe T-cell immunodeficiency, congenital alopecia, nail dystrophy syndrome (disorder)
        '720853005',  # Severe combined immunodeficiency, microcephaly, growth retardation, sensitivity to ionizing radiation syndrome (disorder)
        '720986005',  # Anhidrotic ectodermal dysplasia, immunodeficiency, osteopetrosis, lymphedema syndrome (disorder)
        '721977007',  # Lung fibrosis, immunodeficiency, 46,XX gonadal dysgenesis syndrome (disorder)
        '722067005',  # Severe combined immunodeficiency with hypereosinophilia (disorder)
        '724177005',  # Ligase 4 syndrome (disorder)
        '724361001',  # Hepatic veno-occlusive disease with immunodeficiency syndrome (disorder)
        '725135004',  # Combined immunodeficiency due to CD3gamma deficiency (disorder)
        '725136003',  # Immunodeficiency by defective expression of human leukocyte antigen class 1 (disorder)
        '725290000',  # Combined immunodeficiency due to partial recombination-activating gene 1 deficiency (disorder)
    }


class VaricellaZoster(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  varicella zoster infections.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of varicella zoster infections.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS117v10
    """

    VALUE_SET_NAME = 'Varicella Zoster'
    OID = '2.16.840.1.113883.3.464.1003.110.12.1039'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'B01.0',  # Varicella meningitis
        'B01.11',  # Varicella encephalitis and encephalomyelitis
        'B01.12',  # Varicella myelitis
        'B01.2',  # Varicella pneumonia
        'B01.81',  # Varicella keratitis
        'B01.89',  # Other varicella complications
        'B01.9',  # Varicella without complication
        'B02.0',  # Zoster encephalitis
        'B02.1',  # Zoster meningitis
        'B02.21',  # Postherpetic geniculate ganglionitis
        'B02.22',  # Postherpetic trigeminal neuralgia
        'B02.23',  # Postherpetic polyneuropathy
        'B02.24',  # Postherpetic myelitis
        'B02.29',  # Other postherpetic nervous system involvement
        'B02.30',  # Zoster ocular disease, unspecified
        'B02.31',  # Zoster conjunctivitis
        'B02.32',  # Zoster iridocyclitis
        'B02.33',  # Zoster keratitis
        'B02.34',  # Zoster scleritis
        'B02.39',  # Other herpes zoster eye disease
        'B02.7',  # Disseminated zoster
        'B02.8',  # Zoster with other complications
        'B02.9',  # Zoster without complications
    }
    SNOMEDCT = {
        '4740000',  # Herpes zoster (disorder)
        '10698009',  # Herpes zoster iridocyclitis (disorder)
        '21954000',  # Herpes zoster auricularis (disorder)
        '23737006',  # Chickenpox with complication (disorder)
        '24059009',  # Acute cerebellar ataxia caused by varicella (disorder)
        '36292003',  # Otitis externa caused by Herpes zoster (disorder)
        '38907003',  # Varicella (disorder)
        '42448002',  # Herpes zoster keratoconjunctivitis (disorder)
        '49183009',  # Herpes zoster with complication (disorder)
        '55560002',  # Disseminated herpes zoster (disorder)
        '87513003',  # Herpes zoster ophthalmicus (disorder)
        '111859007',  # Herpes zoster without complication (disorder)
        '111861003',  # Herpes zoster with nervous system complication (disorder)
        '161423008',  # History of chickenpox (situation)
        '186524006',  # Herpes zoster with ophthalmic complication (disorder)
        '195911009',  # Chickenpox pneumonia (disorder)
        '230176008',  # Herpes zoster encephalitis (disorder)
        '230198004',  # Varicella transverse myelitis (disorder)
        '230262004',  # Amyotrophy caused by herpes zoster (disorder)
        '230536009',  # Acute trigeminal herpes zoster (disorder)
        '232400003',  # Acute herpes zoster pharyngitis (disorder)
        '235059009',  # Herpes zoster infection of oral mucosa (disorder)
        '240468001',  # Neurological varicella (disorder)
        '240470005',  # Thoracic herpes zoster infection (disorder)
        '240471009',  # Lumbar herpes zoster infection (disorder)
        '240472002',  # Maxillary herpes zoster infection (disorder)
        '240473007',  # Mandibular herpes zoster infection (disorder)
        '240474001',  # Multidermatomal herpes zoster infection (disorder)
        '309465005',  # Varicella-zoster virus infection (disorder)
        '371113008',  # Varicella immune (finding)
        '397573005',  # Herpes zoster keratitis (disorder)
        '400020001',  # Herpes zoster dermatitis (disorder)
        '402897003',  # Herpes zoster involving cervical dermatome (disorder)
        '402898008',  # Herpes zoster involving sacral dermatome (disorder)
        '402899000',  # Anetoderma following varicella (disorder)
        '410500004',  # Hutchinson's sign (disorder)
        '410509003',  # Herpes zoster conjunctivitis (disorder)
        '421029004',  # Herpes zoster retinitis (disorder)
        '422127002',  # Herpes zoster associated with acquired immunodeficiency syndrome (disorder)
        '422446008',  # Herpes zoster subepithelial infiltrates (disorder)
        '422471006',  # Herpes zoster acute retinal necrosis (disorder)
        '422666006',  # Herpes zoster corneal epitheliopathy (disorder)
        '423333008',  # Exanthem caused by chicken pox (disorder)
        '423628002',  # Acute herpes zoster neuropathy (disorder)
        '424353002',  # Corneal haze caused by herpes zoster (disorder)
        '424435009',  # Herpes zoster pseudodendrites (disorder)
        '424801004',  # Herpes zoster corneal endotheliitis (disorder)
        '424941009',  # Herpes zoster radiculitis (disorder)
        '425356002',  # Herpes zoster retinal pigment epithelial detachment (disorder)
        '426570007',  # Herpes zoster interstitial keratitis (disorder)
        '428633000',  # Exanthem caused by herpes zoster (disorder)
        '713250002',  # Gingival disease caused by varicella-zoster virus (disorder)
        '713733003',  # Infection caused by herpes zoster virus co-occurrent with human immunodeficiency virus infection (disorder)
        '713964006',  # Multidermatomal infection caused by Herpes zoster co-occurrent with human immunodeficiency virus infection (disorder)
        '715223009',  # Fetal varicella syndrome (disorder)
        '723109003',  # Herpes zoster of skin and/or mucous membranes without complications (disorder)
        '838357005',  # Acute glossopharyngeal neuropathy caused by Herpes zoster (disorder)
        '12551000132107',  # Cranial neuropathy due to Herpes zoster (disorder)
        '12561000132105',  # Bacterial infection co-occurrent and due to herpes zoster (disorder)
        '12571000132104',  # Pneumonitis due to Herpes zoster (disorder)
        '98541000119101',  # Herpes zoster myelitis (disorder)
        '331071000119101',  # Herpes zoster iritis (disorder)
        '681221000119108',  # History of herpes zoster (situation)
        '15678761000119105',  # Iridocyclitis of right eye due to herpes zoster (disorder)
        '15678801000119102',  # Iridocyclitis of left eye due to herpes zoster (disorder)
        '15680201000119106',  # Keratoconjunctivitis of right eye due to herpes zoster (disorder)
        '15680241000119108',  # Keratoconjunctivitis of left eye due to herpes zoster (disorder)
        '15685081000119102',  # Corneal epitheliopathy of right eye due to herpes zoster (disorder)
        '15685121000119100',  # Corneal epitheliopathy of left eye due to herpes zoster (disorder)
        '15685201000119100',  # Interstitial keratitis of left eye caused by Herpes zoster virus (disorder)
        '15685281000119108',  # Interstitial keratitis of right eye caused by Herpes zoster virus (disorder)
        '15936581000119108',  # Otitis externa of right external auditory canal caused by human herpesvirus 3 (disorder)
        '15936621000119108',  # Otitis externa of left external auditory canal caused by human herpesvirus 3 (disorder)
        '15989271000119107',  # Conjunctivitis of right eye caused by herpes zoster virus (disorder)
        '15989311000119107',  # Conjunctivitis of bilateral eyes caused by herpes zoster virus (disorder)
        '15989351000119108',  # Conjunctivitis of left eye caused by herpes zoster virus (disorder)
        '15991711000119108',  # Iritis of right eye caused by herpes zoster virus (disorder)
        '15991791000119104',  # Iritis of left eye caused by herpes zoster virus (disorder)
    }


class Diabetes(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  diabetes.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of type I or type II diabetes.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of gestational diabetes or steroid-induced diabetes.

    ** Used in:** CMS347v5, CMS134v10, CMS131v10, CMS122v10
    """

    VALUE_SET_NAME = 'Diabetes'
    OID = '2.16.840.1.113883.3.464.1003.103.12.1001'
    DEFINITION_VERSION = '20190315'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'E10.10',  # Type 1 diabetes mellitus with ketoacidosis without coma
        'E10.11',  # Type 1 diabetes mellitus with ketoacidosis with coma
        'E10.21',  # Type 1 diabetes mellitus with diabetic nephropathy
        'E10.22',  # Type 1 diabetes mellitus with diabetic chronic kidney disease
        'E10.29',  # Type 1 diabetes mellitus with other diabetic kidney complication
        'E10.311',  # Type 1 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E10.319',  # Type 1 diabetes mellitus with unspecified diabetic retinopathy without macular edema
        'E10.321',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema
        'E10.3211',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E10.3212',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E10.3213',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E10.3219',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, unspecified eye
        'E10.329',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema
        'E10.3291',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        'E10.3292',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        'E10.3293',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        'E10.3299',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, unspecified eye
        'E10.331',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema
        'E10.3311',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E10.3312',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E10.3313',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E10.3319',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, unspecified eye
        'E10.339',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema
        'E10.3391',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        'E10.3392',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        'E10.3393',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        'E10.3399',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, unspecified eye
        'E10.341',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema
        'E10.3411',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E10.3412',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E10.3413',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E10.3419',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, unspecified eye
        'E10.349',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema
        'E10.3491',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        'E10.3492',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        'E10.3493',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        'E10.3499',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, unspecified eye
        'E10.351',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema
        'E10.3511',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E10.3512',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E10.3513',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E10.3519',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, unspecified eye
        'E10.3521',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E10.3522',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E10.3523',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E10.3529',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, unspecified eye
        'E10.3531',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E10.3532',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E10.3533',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E10.3539',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, unspecified eye
        'E10.3541',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E10.3542',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E10.3543',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E10.3549',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, unspecified eye
        'E10.3551',  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E10.3552',  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E10.3553',  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E10.3559',  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, unspecified eye
        'E10.359',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema
        'E10.3591',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        'E10.3592',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        'E10.3593',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        'E10.3599',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, unspecified eye
        'E10.36',  # Type 1 diabetes mellitus with diabetic cataract
        'E10.37X1',  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        'E10.37X2',  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        'E10.37X3',  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        'E10.37X9',  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, unspecified eye
        'E10.39',  # Type 1 diabetes mellitus with other diabetic ophthalmic complication
        'E10.40',  # Type 1 diabetes mellitus with diabetic neuropathy, unspecified
        'E10.41',  # Type 1 diabetes mellitus with diabetic mononeuropathy
        'E10.42',  # Type 1 diabetes mellitus with diabetic polyneuropathy
        'E10.43',  # Type 1 diabetes mellitus with diabetic autonomic (poly)neuropathy
        'E10.44',  # Type 1 diabetes mellitus with diabetic amyotrophy
        'E10.49',  # Type 1 diabetes mellitus with other diabetic neurological complication
        'E10.51',  # Type 1 diabetes mellitus with diabetic peripheral angiopathy without gangrene
        'E10.52',  # Type 1 diabetes mellitus with diabetic peripheral angiopathy with gangrene
        'E10.59',  # Type 1 diabetes mellitus with other circulatory complications
        'E10.610',  # Type 1 diabetes mellitus with diabetic neuropathic arthropathy
        'E10.618',  # Type 1 diabetes mellitus with other diabetic arthropathy
        'E10.620',  # Type 1 diabetes mellitus with diabetic dermatitis
        'E10.621',  # Type 1 diabetes mellitus with foot ulcer
        'E10.622',  # Type 1 diabetes mellitus with other skin ulcer
        'E10.628',  # Type 1 diabetes mellitus with other skin complications
        'E10.630',  # Type 1 diabetes mellitus with periodontal disease
        'E10.638',  # Type 1 diabetes mellitus with other oral complications
        'E10.641',  # Type 1 diabetes mellitus with hypoglycemia with coma
        'E10.649',  # Type 1 diabetes mellitus with hypoglycemia without coma
        'E10.65',  # Type 1 diabetes mellitus with hyperglycemia
        'E10.69',  # Type 1 diabetes mellitus with other specified complication
        'E10.8',  # Type 1 diabetes mellitus with unspecified complications
        'E10.9',  # Type 1 diabetes mellitus without complications
        'E11.00',  # Type 2 diabetes mellitus with hyperosmolarity without nonketotic hyperglycemic-hyperosmolar coma (NKHHC)
        'E11.01',  # Type 2 diabetes mellitus with hyperosmolarity with coma
        'E11.10',  # Type 2 diabetes mellitus with ketoacidosis without coma
        'E11.11',  # Type 2 diabetes mellitus with ketoacidosis with coma
        'E11.21',  # Type 2 diabetes mellitus with diabetic nephropathy
        'E11.22',  # Type 2 diabetes mellitus with diabetic chronic kidney disease
        'E11.29',  # Type 2 diabetes mellitus with other diabetic kidney complication
        'E11.311',  # Type 2 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E11.319',  # Type 2 diabetes mellitus with unspecified diabetic retinopathy without macular edema
        'E11.321',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema
        'E11.3211',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E11.3212',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E11.3213',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E11.3219',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, unspecified eye
        'E11.329',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema
        'E11.3291',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        'E11.3292',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        'E11.3293',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        'E11.3299',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, unspecified eye
        'E11.331',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema
        'E11.3311',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E11.3312',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E11.3313',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E11.3319',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, unspecified eye
        'E11.339',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema
        'E11.3391',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        'E11.3392',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        'E11.3393',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        'E11.3399',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, unspecified eye
        'E11.341',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema
        'E11.3411',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E11.3412',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E11.3413',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E11.3419',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, unspecified eye
        'E11.349',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema
        'E11.3491',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        'E11.3492',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        'E11.3493',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        'E11.3499',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, unspecified eye
        'E11.351',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema
        'E11.3511',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E11.3512',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E11.3513',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E11.3519',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, unspecified eye
        'E11.3521',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E11.3522',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E11.3523',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E11.3529',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, unspecified eye
        'E11.3531',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E11.3532',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E11.3533',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E11.3539',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, unspecified eye
        'E11.3541',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E11.3542',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E11.3543',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E11.3549',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, unspecified eye
        'E11.3551',  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E11.3552',  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E11.3553',  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E11.3559',  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, unspecified eye
        'E11.359',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema
        'E11.3591',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        'E11.3592',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        'E11.3593',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        'E11.3599',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, unspecified eye
        'E11.36',  # Type 2 diabetes mellitus with diabetic cataract
        'E11.37X1',  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        'E11.37X2',  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        'E11.37X3',  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        'E11.37X9',  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, unspecified eye
        'E11.39',  # Type 2 diabetes mellitus with other diabetic ophthalmic complication
        'E11.40',  # Type 2 diabetes mellitus with diabetic neuropathy, unspecified
        'E11.41',  # Type 2 diabetes mellitus with diabetic mononeuropathy
        'E11.42',  # Type 2 diabetes mellitus with diabetic polyneuropathy
        'E11.43',  # Type 2 diabetes mellitus with diabetic autonomic (poly)neuropathy
        'E11.44',  # Type 2 diabetes mellitus with diabetic amyotrophy
        'E11.49',  # Type 2 diabetes mellitus with other diabetic neurological complication
        'E11.51',  # Type 2 diabetes mellitus with diabetic peripheral angiopathy without gangrene
        'E11.52',  # Type 2 diabetes mellitus with diabetic peripheral angiopathy with gangrene
        'E11.59',  # Type 2 diabetes mellitus with other circulatory complications
        'E11.610',  # Type 2 diabetes mellitus with diabetic neuropathic arthropathy
        'E11.618',  # Type 2 diabetes mellitus with other diabetic arthropathy
        'E11.620',  # Type 2 diabetes mellitus with diabetic dermatitis
        'E11.621',  # Type 2 diabetes mellitus with foot ulcer
        'E11.622',  # Type 2 diabetes mellitus with other skin ulcer
        'E11.628',  # Type 2 diabetes mellitus with other skin complications
        'E11.630',  # Type 2 diabetes mellitus with periodontal disease
        'E11.638',  # Type 2 diabetes mellitus with other oral complications
        'E11.641',  # Type 2 diabetes mellitus with hypoglycemia with coma
        'E11.649',  # Type 2 diabetes mellitus with hypoglycemia without coma
        'E11.65',  # Type 2 diabetes mellitus with hyperglycemia
        'E11.69',  # Type 2 diabetes mellitus with other specified complication
        'E11.8',  # Type 2 diabetes mellitus with unspecified complications
        'E11.9',  # Type 2 diabetes mellitus without complications
        'E13.00',  # Other specified diabetes mellitus with hyperosmolarity without nonketotic hyperglycemic-hyperosmolar coma (NKHHC)
        'E13.01',  # Other specified diabetes mellitus with hyperosmolarity with coma
        'E13.10',  # Other specified diabetes mellitus with ketoacidosis without coma
        'E13.11',  # Other specified diabetes mellitus with ketoacidosis with coma
        'E13.21',  # Other specified diabetes mellitus with diabetic nephropathy
        'E13.22',  # Other specified diabetes mellitus with diabetic chronic kidney disease
        'E13.29',  # Other specified diabetes mellitus with other diabetic kidney complication
        'E13.311',  # Other specified diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E13.319',  # Other specified diabetes mellitus with unspecified diabetic retinopathy without macular edema
        'E13.321',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema
        'E13.3211',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E13.3212',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E13.3213',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E13.3219',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, unspecified eye
        'E13.329',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema
        'E13.3291',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        'E13.3292',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        'E13.3293',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        'E13.3299',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, unspecified eye
        'E13.331',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema
        'E13.3311',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E13.3312',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E13.3313',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E13.3319',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, unspecified eye
        'E13.339',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema
        'E13.3391',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        'E13.3392',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        'E13.3393',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        'E13.3399',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, unspecified eye
        'E13.341',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema
        'E13.3411',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E13.3412',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E13.3413',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E13.3419',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, unspecified eye
        'E13.349',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema
        'E13.3491',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        'E13.3492',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        'E13.3493',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        'E13.3499',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, unspecified eye
        'E13.351',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema
        'E13.3511',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E13.3512',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E13.3513',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E13.3519',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, unspecified eye
        'E13.3521',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E13.3522',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E13.3523',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E13.3529',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, unspecified eye
        'E13.3531',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E13.3532',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E13.3533',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E13.3539',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, unspecified eye
        'E13.3541',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E13.3542',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E13.3543',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E13.3549',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, unspecified eye
        'E13.3551',  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E13.3552',  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E13.3553',  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E13.3559',  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, unspecified eye
        'E13.359',  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema
        'E13.3591',  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        'E13.3592',  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        'E13.3593',  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        'E13.3599',  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, unspecified eye
        'E13.36',  # Other specified diabetes mellitus with diabetic cataract
        'E13.37X1',  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        'E13.37X2',  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        'E13.37X3',  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        'E13.37X9',  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, unspecified eye
        'E13.39',  # Other specified diabetes mellitus with other diabetic ophthalmic complication
        'E13.40',  # Other specified diabetes mellitus with diabetic neuropathy, unspecified
        'E13.41',  # Other specified diabetes mellitus with diabetic mononeuropathy
        'E13.42',  # Other specified diabetes mellitus with diabetic polyneuropathy
        'E13.43',  # Other specified diabetes mellitus with diabetic autonomic (poly)neuropathy
        'E13.44',  # Other specified diabetes mellitus with diabetic amyotrophy
        'E13.49',  # Other specified diabetes mellitus with other diabetic neurological complication
        'E13.51',  # Other specified diabetes mellitus with diabetic peripheral angiopathy without gangrene
        'E13.52',  # Other specified diabetes mellitus with diabetic peripheral angiopathy with gangrene
        'E13.59',  # Other specified diabetes mellitus with other circulatory complications
        'E13.610',  # Other specified diabetes mellitus with diabetic neuropathic arthropathy
        'E13.618',  # Other specified diabetes mellitus with other diabetic arthropathy
        'E13.620',  # Other specified diabetes mellitus with diabetic dermatitis
        'E13.621',  # Other specified diabetes mellitus with foot ulcer
        'E13.622',  # Other specified diabetes mellitus with other skin ulcer
        'E13.628',  # Other specified diabetes mellitus with other skin complications
        'E13.630',  # Other specified diabetes mellitus with periodontal disease
        'E13.638',  # Other specified diabetes mellitus with other oral complications
        'E13.641',  # Other specified diabetes mellitus with hypoglycemia with coma
        'E13.649',  # Other specified diabetes mellitus with hypoglycemia without coma
        'E13.65',  # Other specified diabetes mellitus with hyperglycemia
        'E13.69',  # Other specified diabetes mellitus with other specified complication
        'E13.8',  # Other specified diabetes mellitus with unspecified complications
        'E13.9',  # Other specified diabetes mellitus without complications
        'O24.011',  # Pre-existing type 1 diabetes mellitus, in pregnancy, first trimester
        'O24.012',  # Pre-existing type 1 diabetes mellitus, in pregnancy, second trimester
        'O24.013',  # Pre-existing type 1 diabetes mellitus, in pregnancy, third trimester
        'O24.019',  # Pre-existing type 1 diabetes mellitus, in pregnancy, unspecified trimester
        'O24.02',  # Pre-existing type 1 diabetes mellitus, in childbirth
        'O24.03',  # Pre-existing type 1 diabetes mellitus, in the puerperium
        'O24.111',  # Pre-existing type 2 diabetes mellitus, in pregnancy, first trimester
        'O24.112',  # Pre-existing type 2 diabetes mellitus, in pregnancy, second trimester
        'O24.113',  # Pre-existing type 2 diabetes mellitus, in pregnancy, third trimester
        'O24.119',  # Pre-existing type 2 diabetes mellitus, in pregnancy, unspecified trimester
        'O24.12',  # Pre-existing type 2 diabetes mellitus, in childbirth
        'O24.13',  # Pre-existing type 2 diabetes mellitus, in the puerperium
        'O24.311',  # Unspecified pre-existing diabetes mellitus in pregnancy, first trimester
        'O24.312',  # Unspecified pre-existing diabetes mellitus in pregnancy, second trimester
        'O24.313',  # Unspecified pre-existing diabetes mellitus in pregnancy, third trimester
        'O24.319',  # Unspecified pre-existing diabetes mellitus in pregnancy, unspecified trimester
        'O24.32',  # Unspecified pre-existing diabetes mellitus in childbirth
        'O24.33',  # Unspecified pre-existing diabetes mellitus in the puerperium
        'O24.811',  # Other pre-existing diabetes mellitus in pregnancy, first trimester
        'O24.812',  # Other pre-existing diabetes mellitus in pregnancy, second trimester
        'O24.813',  # Other pre-existing diabetes mellitus in pregnancy, third trimester
        'O24.819',  # Other pre-existing diabetes mellitus in pregnancy, unspecified trimester
        'O24.82',  # Other pre-existing diabetes mellitus in childbirth
        'O24.83',  # Other pre-existing diabetes mellitus in the puerperium
    }
    ICD9CM = {
        '250.00',  # Diabetes mellitus without mention of complication, type II or unspecified type, not stated as uncontrolled
        '250.01',  # Diabetes mellitus without mention of complication, type I [juvenile type], not stated as uncontrolled
        '250.02',  # Diabetes mellitus without mention of complication, type II or unspecified type, uncontrolled
        '250.03',  # Diabetes mellitus without mention of complication, type I [juvenile type], uncontrolled
        '250.10',  # Diabetes with ketoacidosis, type II or unspecified type, not stated as uncontrolled
        '250.11',  # Diabetes with ketoacidosis, type I [juvenile type], not stated as uncontrolled
        '250.12',  # Diabetes with ketoacidosis, type II or unspecified type, uncontrolled
        '250.13',  # Diabetes with ketoacidosis, type I [juvenile type], uncontrolled
        '250.20',  # Diabetes with hyperosmolarity, type II or unspecified type, not stated as uncontrolled
        '250.21',  # Diabetes with hyperosmolarity, type I [juvenile type], not stated as uncontrolled
        '250.22',  # Diabetes with hyperosmolarity, type II or unspecified type, uncontrolled
        '250.23',  # Diabetes with hyperosmolarity, type I [juvenile type], uncontrolled
        '250.30',  # Diabetes with other coma, type II or unspecified type, not stated as uncontrolled
        '250.31',  # Diabetes with other coma, type I [juvenile type], not stated as uncontrolled
        '250.32',  # Diabetes with other coma, type II or unspecified type, uncontrolled
        '250.33',  # Diabetes with other coma, type I [juvenile type], uncontrolled
        '250.40',  # Diabetes with renal manifestations, type II or unspecified type, not stated as uncontrolled
        '250.41',  # Diabetes with renal manifestations, type I [juvenile type], not stated as uncontrolled
        '250.42',  # Diabetes with renal manifestations, type II or unspecified type, uncontrolled
        '250.43',  # Diabetes with renal manifestations, type I [juvenile type], uncontrolled
        '250.50',  # Diabetes with ophthalmic manifestations, type II or unspecified type, not stated as uncontrolled
        '250.51',  # Diabetes with ophthalmic manifestations, type I [juvenile type], not stated as uncontrolled
        '250.52',  # Diabetes with ophthalmic manifestations, type II or unspecified type, uncontrolled
        '250.53',  # Diabetes with ophthalmic manifestations, type I [juvenile type], uncontrolled
        '250.60',  # Diabetes with neurological manifestations, type II or unspecified type, not stated as uncontrolled
        '250.61',  # Diabetes with neurological manifestations, type I [juvenile type], not stated as uncontrolled
        '250.62',  # Diabetes with neurological manifestations, type II or unspecified type, uncontrolled
        '250.63',  # Diabetes with neurological manifestations, type I [juvenile type], uncontrolled
        '250.70',  # Diabetes with peripheral circulatory disorders, type II or unspecified type, not stated as uncontrolled
        '250.71',  # Diabetes with peripheral circulatory disorders, type I [juvenile type], not stated as uncontrolled
        '250.72',  # Diabetes with peripheral circulatory disorders, type II or unspecified type, uncontrolled
        '250.73',  # Diabetes with peripheral circulatory disorders, type I [juvenile type], uncontrolled
        '250.80',  # Diabetes with other specified manifestations, type II or unspecified type, not stated as uncontrolled
        '250.81',  # Diabetes with other specified manifestations, type I [juvenile type], not stated as uncontrolled
        '250.82',  # Diabetes with other specified manifestations, type II or unspecified type, uncontrolled
        '250.83',  # Diabetes with other specified manifestations, type I [juvenile type], uncontrolled
        '250.90',  # Diabetes with unspecified complication, type II or unspecified type, not stated as uncontrolled
        '250.91',  # Diabetes with unspecified complication, type I [juvenile type], not stated as uncontrolled
        '250.92',  # Diabetes with unspecified complication, type II or unspecified type, uncontrolled
        '250.93',  # Diabetes with unspecified complication, type I [juvenile type], uncontrolled
        '357.2',  # Polyneuropathy in diabetes
        '362.01',  # Background diabetic retinopathy
        '362.02',  # Proliferative diabetic retinopathy
        '362.03',  # Nonproliferative diabetic retinopathy NOS
        '362.04',  # Mild nonproliferative diabetic retinopathy
        '362.05',  # Moderate nonproliferative diabetic retinopathy
        '362.06',  # Severe nonproliferative diabetic retinopathy
        '362.07',  # Diabetic macular edema
        '366.41',  # Diabetic cataract
        '648.00',  # Diabetes mellitus of mother, complicating pregnancy, childbirth, or the puerperium, unspecified as to episode of care or not applicable
        '648.01',  # Diabetes mellitus of mother, complicating pregnancy, childbirth, or the puerperium, delivered, with or without mention of antepartum condition
        '648.02',  # Diabetes mellitus of mother, complicating pregnancy, childbirth, or the puerperium, delivered, with mention of postpartum complication
        '648.03',  # Diabetes mellitus of mother, complicating pregnancy, childbirth, or the puerperium, antepartum condition or complication
        '648.04',  # Diabetes mellitus of mother, complicating pregnancy, childbirth, or the puerperium, postpartum condition or complication
    }
    SNOMEDCT = {
        '9859006',  # Acanthosis nigricans due to type 2 diabetes mellitus (disorder)
        '23045005',  # Insulin dependent diabetes mellitus type IA (disorder)
        '28032008',  # Insulin dependent diabetes mellitus type IB (disorder)
        '44054006',  # Diabetes mellitus type 2 (disorder)
        '46635009',  # Diabetes mellitus type 1 (disorder)
        '81531005',  # Diabetes mellitus type 2 in obese (disorder)
        '190330002',  # Hyperosmolar coma due to type 1 diabetes mellitus (disorder)
        '190331003',  # Hyperosmolar coma due to type 2 diabetes mellitus (disorder)
        '190368000',  # Type I diabetes mellitus with ulcer (disorder)
        '190369008',  # Type I diabetes mellitus with gangrene (disorder)
        '190372001',  # Type I diabetes mellitus maturity onset (disorder)
        '190389009',  # Type II diabetes mellitus with ulcer (disorder)
        '199229001',  # Pre-existing type 1 diabetes mellitus (disorder)
        '199230006',  # Pre-existing type 2 diabetes mellitus (disorder)
        '237599002',  # Insulin treated type 2 diabetes mellitus (disorder)
        '237604008',  # Maturity onset diabetes of the young, type 2 (disorder)
        '237618001',  # Insulin-dependent diabetes mellitus secretory diarrhea syndrome (disorder)
        '313435000',  # Type I diabetes mellitus without complication (disorder)
        '313436004',  # Type II diabetes mellitus without complication (disorder)
        '314771006',  # Type I diabetes mellitus with hypoglycemic coma (disorder)
        '314893005',  # Arthropathy due to type 1 diabetes mellitus (disorder)
        '314902007',  # Peripheral angiopathy due to type 2 diabetes mellitus (disorder)
        '314903002',  # Arthropathy due to type 2 diabetes mellitus (disorder)
        '314904008',  # Type II diabetes mellitus with neuropathic arthropathy (disorder)
        '359642000',  # Diabetes mellitus type 2 in nonobese (disorder)
        '420279001',  # Renal disorder due to type 2 diabetes mellitus (disorder)
        '420436000',  # Mononeuropathy due to type 2 diabetes mellitus (disorder)
        '420486006',  # Exudative maculopathy due to type 1 diabetes mellitus (disorder)
        '420789003',  # Retinopathy due to type 1 diabetes mellitus (disorder)
        '420918009',  # Mononeuropathy due to type 1 diabetes mellitus (disorder)
        '421075007',  # Ketoacidotic coma due to type 1 diabetes mellitus (disorder)
        '421326000',  # Disorder of nervous system due to type 2 diabetes mellitus (disorder)
        '421365002',  # Peripheral circulatory disorder due to type 1 diabetes mellitus (disorder)
        '421437000',  # Hypoglycemic coma due to type 1 diabetes mellitus (disorder)
        '421468001',  # Disorder of nervous system due to type 1 diabetes mellitus (disorder)
        '421779007',  # Exudative maculopathy due to type 2 diabetes mellitus (disorder)
        '421847006',  # Ketoacidotic coma due to type 2 diabetes mellitus (disorder)
        '421893009',  # Renal disorder due to type 1 diabetes mellitus (disorder)
        '422034002',  # Retinopathy due to type 2 diabetes mellitus (disorder)
        '422099009',  # Disorder of eye due to type 2 diabetes mellitus (disorder)
        '422166005',  # Peripheral circulatory disorder due to type 2 diabetes mellitus (disorder)
        '427027005',  # Lumbosacral radiculoplexus neuropathy due to type 2 diabetes mellitus (disorder)
        '427571000',  # Lumbosacral radiculoplexus neuropathy due to type 1 diabetes mellitus (disorder)
        '428007007',  # Erectile dysfunction due to type 2 diabetes mellitus (disorder)
        '609562003',  # Maturity onset diabetes of the young, type 1 (disorder)
        '609564002',  # Pre-existing type 1 diabetes mellitus in pregnancy (disorder)
        '609566000',  # Pregnancy and type 1 diabetes mellitus (disorder)
        '609567009',  # Pre-existing type 2 diabetes mellitus in pregnancy (disorder)
        '712882000',  # Autonomic neuropathy due to type 1 diabetes mellitus (disorder)
        '712883005',  # Autonomic neuropathy due to type 2 diabetes mellitus (disorder)
        '713702000',  # Gastroparesis due to type 1 diabetes mellitus (disorder)
        '713703005',  # Gastroparesis due to type 2 diabetes mellitus (disorder)
        '713705003',  # Polyneuropathy due to type 1 diabetes mellitus (disorder)
        '713706002',  # Polyneuropathy due to type 2 diabetes mellitus (disorder)
        '719216001',  # Hypoglycemic coma due to type 2 diabetes mellitus (disorder)
        '739681000',  # Disorder of eye due to type 1 diabetes mellitus (disorder)
        '770098001',  # Cranial nerve palsy due to type 1 diabetes mellitus (disorder)
        '691000119103',  # Erectile dysfunction due to type 1 diabetes mellitus (disorder)
        '1481000119100',  # Diabetes mellitus type 2 without retinopathy (disorder)
        '1501000119109',  # Proliferative retinopathy due to type 2 diabetes mellitus (disorder)
        '1511000119107',  # Peripheral neuropathy due to type 2 diabetes mellitus (disorder)
        '1551000119108',  # Nonproliferative retinopathy due to type 2 diabetes mellitus (disorder)
        '28331000119107',  # Retinal edema due to type 2 diabetes mellitus (disorder)
        '31211000119101',  # Peripheral angiopathy due to type 1 diabetes mellitus (disorder)
        '31321000119102',  # Diabetes mellitus type 1 without retinopathy (disorder)
        '41911000119107',  # Glaucoma due to type 2 diabetes mellitus (disorder)
        '60951000119105',  # Blindness due to type 2 diabetes mellitus (disorder)
        '60961000119107',  # Nonproliferative diabetic retinopathy due to type 1 diabetes mellitus (disorder)
        '60971000119101',  # Proliferative retinopathy due to type 1 diabetes mellitus (disorder)
        '60991000119100',  # Blindness due to type 1 diabetes mellitus (disorder)
        '71441000119104',  # Nephrotic syndrome due to type 2 diabetes mellitus (disorder)
        '71721000119101',  # Nephrotic syndrome due to type 1 diabetes mellitus (disorder)
        '71791000119104',  # Peripheral neuropathy due to type 1 diabetes mellitus (disorder)
        '82541000119100',  # Traction detachment of retina due to type 2 diabetes mellitus (disorder)
        '82551000119103',  # Rubeosis iridis due to type 2 diabetes mellitus (disorder)
        '82571000119107',  # Traction detachment of retina due to type 1 diabetes mellitus (disorder)
        '82581000119105',  # Rubeosis iridis due to type 1 diabetes mellitus (disorder)
        '87921000119104',  # Cranial nerve palsy due to type 2 diabetes mellitus (disorder)
        '97331000119101',  # Macular edema and retinopathy due to type 2 diabetes mellitus (disorder)
        '102781000119107',  # Sensory neuropathy due to type 1 diabetes mellitus (disorder)
        '104941000119109',  # Ischemia of retina due to type 1 diabetes mellitus (disorder)
        '104961000119108',  # Ischemia of retina due to type 2 diabetes mellitus (disorder)
        '109171000119104',  # Retinal edema due to type 1 diabetes mellitus (disorder)
        '110181000119105',  # Peripheral sensory neuropathy due to type 2 diabetes mellitus (disorder)
        '138881000119106',  # Mild nonproliferative retinopathy due to type 1 diabetes mellitus (disorder)
        '138891000119109',  # Moderate nonproliferative retinopathy due to type 1 diabetes mellitus (disorder)
        '138901000119108',  # Severe nonproliferative retinopathy due to diabetes mellitus type 1 (disorder)
        '138911000119106',  # Mild nonproliferative retinopathy due to type 2 diabetes mellitus (disorder)
        '138921000119104',  # Moderate nonproliferative retinopathy due to type 2 diabetes mellitus (disorder)
        '138941000119105',  # Severe nonproliferative retinopathy due to diabetes mellitus type 2 (disorder)
        '368101000119109',  # Periodontal disease co-occurrent and due to diabetes mellitus type 2 (disorder)
        '368521000119107',  # Disorder of nerve co-occurrent and due to type 1 diabetes mellitus (disorder)
        '368581000119106',  # Neuropathy due to type 2 diabetes mellitus (disorder)
    }


class FrailtyDiagnosis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  potential indicators of frailty.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent that a diagnosis of potential frailty when coupled with older age: dependence on durable medical equipment for long-term use, limitation of activities of daily living due to disability, need for assistance with care, pressure ulcers, and falls.

    **Exclusion Criteria:** Exclude concepts that indicate potential frailty when associated with pregnancy, fetal, juvenile, childhood or neonatal conditions; codes that indicate temporary need of durable medical equipment; codes that indicate accidental falls caused by another person, being pushed/shoved/collided into, falls on ice due to ice skates, falls due to impact against a baby buggy/shopping cart/wheelbarrow.

    ** Used in:** CMS134v10, CMS165v10, CMS131v10, CMS122v10, CMS125v10, CMS130v10
    """

    VALUE_SET_NAME = 'Frailty Diagnosis'
    OID = '2.16.840.1.113883.3.464.1003.113.12.1074'
    DEFINITION_VERSION = '20190315'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'L89.001',  # Pressure ulcer of unspecified elbow, stage 1
        'L89.002',  # Pressure ulcer of unspecified elbow, stage 2
        'L89.003',  # Pressure ulcer of unspecified elbow, stage 3
        'L89.004',  # Pressure ulcer of unspecified elbow, stage 4
        'L89.006',  # Pressure-induced deep tissue damage of unspecified elbow
        'L89.009',  # Pressure ulcer of unspecified elbow, unspecified stage
        'L89.010',  # Pressure ulcer of right elbow, unstageable
        'L89.011',  # Pressure ulcer of right elbow, stage 1
        'L89.012',  # Pressure ulcer of right elbow, stage 2
        'L89.013',  # Pressure ulcer of right elbow, stage 3
        'L89.014',  # Pressure ulcer of right elbow, stage 4
        'L89.016',  # Pressure-induced deep tissue damage of right elbow
        'L89.019',  # Pressure ulcer of right elbow, unspecified stage
        'L89.020',  # Pressure ulcer of left elbow, unstageable
        'L89.021',  # Pressure ulcer of left elbow, stage 1
        'L89.022',  # Pressure ulcer of left elbow, stage 2
        'L89.023',  # Pressure ulcer of left elbow, stage 3
        'L89.024',  # Pressure ulcer of left elbow, stage 4
        'L89.026',  # Pressure-induced deep tissue damage of left elbow
        'L89.029',  # Pressure ulcer of left elbow, unspecified stage
        'L89.100',  # Pressure ulcer of unspecified part of back, unstageable
        'L89.101',  # Pressure ulcer of unspecified part of back, stage 1
        'L89.102',  # Pressure ulcer of unspecified part of back, stage 2
        'L89.103',  # Pressure ulcer of unspecified part of back, stage 3
        'L89.104',  # Pressure ulcer of unspecified part of back, stage 4
        'L89.106',  # Pressure-induced deep tissue damage of unspecified part of back
        'L89.109',  # Pressure ulcer of unspecified part of back, unspecified stage
        'L89.110',  # Pressure ulcer of right upper back, unstageable
        'L89.111',  # Pressure ulcer of right upper back, stage 1
        'L89.112',  # Pressure ulcer of right upper back, stage 2
        'L89.113',  # Pressure ulcer of right upper back, stage 3
        'L89.114',  # Pressure ulcer of right upper back, stage 4
        'L89.116',  # Pressure-induced deep tissue damage of right upper back
        'L89.119',  # Pressure ulcer of right upper back, unspecified stage
        'L89.120',  # Pressure ulcer of left upper back, unstageable
        'L89.121',  # Pressure ulcer of left upper back, stage 1
        'L89.122',  # Pressure ulcer of left upper back, stage 2
        'L89.123',  # Pressure ulcer of left upper back, stage 3
        'L89.124',  # Pressure ulcer of left upper back, stage 4
        'L89.126',  # Pressure-induced deep tissue damage of left upper back
        'L89.129',  # Pressure ulcer of left upper back, unspecified stage
        'L89.130',  # Pressure ulcer of right lower back, unstageable
        'L89.131',  # Pressure ulcer of right lower back, stage 1
        'L89.132',  # Pressure ulcer of right lower back, stage 2
        'L89.133',  # Pressure ulcer of right lower back, stage 3
        'L89.134',  # Pressure ulcer of right lower back, stage 4
        'L89.136',  # Pressure-induced deep tissue damage of right lower back
        'L89.139',  # Pressure ulcer of right lower back, unspecified stage
        'L89.140',  # Pressure ulcer of left lower back, unstageable
        'L89.141',  # Pressure ulcer of left lower back, stage 1
        'L89.142',  # Pressure ulcer of left lower back, stage 2
        'L89.143',  # Pressure ulcer of left lower back, stage 3
        'L89.144',  # Pressure ulcer of left lower back, stage 4
        'L89.146',  # Pressure-induced deep tissue damage of left lower back
        'L89.149',  # Pressure ulcer of left lower back, unspecified stage
        'L89.150',  # Pressure ulcer of sacral region, unstageable
        'L89.151',  # Pressure ulcer of sacral region, stage 1
        'L89.152',  # Pressure ulcer of sacral region, stage 2
        'L89.153',  # Pressure ulcer of sacral region, stage 3
        'L89.154',  # Pressure ulcer of sacral region, stage 4
        'L89.156',  # Pressure-induced deep tissue damage of sacral region
        'L89.159',  # Pressure ulcer of sacral region, unspecified stage
        'L89.200',  # Pressure ulcer of unspecified hip, unstageable
        'L89.201',  # Pressure ulcer of unspecified hip, stage 1
        'L89.202',  # Pressure ulcer of unspecified hip, stage 2
        'L89.203',  # Pressure ulcer of unspecified hip, stage 3
        'L89.204',  # Pressure ulcer of unspecified hip, stage 4
        'L89.206',  # Pressure-induced deep tissue damage of unspecified hip
        'L89.209',  # Pressure ulcer of unspecified hip, unspecified stage
        'L89.210',  # Pressure ulcer of right hip, unstageable
        'L89.211',  # Pressure ulcer of right hip, stage 1
        'L89.212',  # Pressure ulcer of right hip, stage 2
        'L89.213',  # Pressure ulcer of right hip, stage 3
        'L89.214',  # Pressure ulcer of right hip, stage 4
        'L89.216',  # Pressure-induced deep tissue damage of right hip
        'L89.219',  # Pressure ulcer of right hip, unspecified stage
        'L89.220',  # Pressure ulcer of left hip, unstageable
        'L89.221',  # Pressure ulcer of left hip, stage 1
        'L89.222',  # Pressure ulcer of left hip, stage 2
        'L89.223',  # Pressure ulcer of left hip, stage 3
        'L89.224',  # Pressure ulcer of left hip, stage 4
        'L89.226',  # Pressure-induced deep tissue damage of left hip
        'L89.229',  # Pressure ulcer of left hip, unspecified stage
        'L89.300',  # Pressure ulcer of unspecified buttock, unstageable
        'L89.301',  # Pressure ulcer of unspecified buttock, stage 1
        'L89.302',  # Pressure ulcer of unspecified buttock, stage 2
        'L89.303',  # Pressure ulcer of unspecified buttock, stage 3
        'L89.304',  # Pressure ulcer of unspecified buttock, stage 4
        'L89.306',  # Pressure-induced deep tissue damage of unspecified buttock
        'L89.309',  # Pressure ulcer of unspecified buttock, unspecified stage
        'L89.310',  # Pressure ulcer of right buttock, unstageable
        'L89.311',  # Pressure ulcer of right buttock, stage 1
        'L89.312',  # Pressure ulcer of right buttock, stage 2
        'L89.313',  # Pressure ulcer of right buttock, stage 3
        'L89.314',  # Pressure ulcer of right buttock, stage 4
        'L89.316',  # Pressure-induced deep tissue damage of right buttock
        'L89.319',  # Pressure ulcer of right buttock, unspecified stage
        'L89.320',  # Pressure ulcer of left buttock, unstageable
        'L89.321',  # Pressure ulcer of left buttock, stage 1
        'L89.322',  # Pressure ulcer of left buttock, stage 2
        'L89.323',  # Pressure ulcer of left buttock, stage 3
        'L89.324',  # Pressure ulcer of left buttock, stage 4
        'L89.326',  # Pressure-induced deep tissue damage of left buttock
        'L89.329',  # Pressure ulcer of left buttock, unspecified stage
        'L89.40',  # Pressure ulcer of contiguous site of back, buttock and hip, unspecified stage
        'L89.41',  # Pressure ulcer of contiguous site of back, buttock and hip, stage 1
        'L89.42',  # Pressure ulcer of contiguous site of back, buttock and hip, stage 2
        'L89.43',  # Pressure ulcer of contiguous site of back, buttock and hip, stage 3
        'L89.44',  # Pressure ulcer of contiguous site of back, buttock and hip, stage 4
        'L89.45',  # Pressure ulcer of contiguous site of back, buttock and hip, unstageable
        'L89.46',  # Pressure-induced deep tissue damage of contiguous site of back, buttock and hip
        'L89.500',  # Pressure ulcer of unspecified ankle, unstageable
        'L89.501',  # Pressure ulcer of unspecified ankle, stage 1
        'L89.502',  # Pressure ulcer of unspecified ankle, stage 2
        'L89.503',  # Pressure ulcer of unspecified ankle, stage 3
        'L89.504',  # Pressure ulcer of unspecified ankle, stage 4
        'L89.506',  # Pressure-induced deep tissue damage of unspecified ankle
        'L89.509',  # Pressure ulcer of unspecified ankle, unspecified stage
        'L89.510',  # Pressure ulcer of right ankle, unstageable
        'L89.511',  # Pressure ulcer of right ankle, stage 1
        'L89.512',  # Pressure ulcer of right ankle, stage 2
        'L89.513',  # Pressure ulcer of right ankle, stage 3
        'L89.514',  # Pressure ulcer of right ankle, stage 4
        'L89.516',  # Pressure-induced deep tissue damage of right ankle
        'L89.519',  # Pressure ulcer of right ankle, unspecified stage
        'L89.520',  # Pressure ulcer of left ankle, unstageable
        'L89.521',  # Pressure ulcer of left ankle, stage 1
        'L89.522',  # Pressure ulcer of left ankle, stage 2
        'L89.523',  # Pressure ulcer of left ankle, stage 3
        'L89.524',  # Pressure ulcer of left ankle, stage 4
        'L89.526',  # Pressure-induced deep tissue damage of left ankle
        'L89.529',  # Pressure ulcer of left ankle, unspecified stage
        'L89.600',  # Pressure ulcer of unspecified heel, unstageable
        'L89.601',  # Pressure ulcer of unspecified heel, stage 1
        'L89.602',  # Pressure ulcer of unspecified heel, stage 2
        'L89.603',  # Pressure ulcer of unspecified heel, stage 3
        'L89.604',  # Pressure ulcer of unspecified heel, stage 4
        'L89.606',  # Pressure-induced deep tissue damage of unspecified heel
        'L89.609',  # Pressure ulcer of unspecified heel, unspecified stage
        'L89.610',  # Pressure ulcer of right heel, unstageable
        'L89.611',  # Pressure ulcer of right heel, stage 1
        'L89.612',  # Pressure ulcer of right heel, stage 2
        'L89.613',  # Pressure ulcer of right heel, stage 3
        'L89.614',  # Pressure ulcer of right heel, stage 4
        'L89.616',  # Pressure-induced deep tissue damage of right heel
        'L89.619',  # Pressure ulcer of right heel, unspecified stage
        'L89.620',  # Pressure ulcer of left heel, unstageable
        'L89.621',  # Pressure ulcer of left heel, stage 1
        'L89.622',  # Pressure ulcer of left heel, stage 2
        'L89.623',  # Pressure ulcer of left heel, stage 3
        'L89.624',  # Pressure ulcer of left heel, stage 4
        'L89.626',  # Pressure-induced deep tissue damage of left heel
        'L89.629',  # Pressure ulcer of left heel, unspecified stage
        'L89.810',  # Pressure ulcer of head, unstageable
        'L89.811',  # Pressure ulcer of head, stage 1
        'L89.812',  # Pressure ulcer of head, stage 2
        'L89.813',  # Pressure ulcer of head, stage 3
        'L89.814',  # Pressure ulcer of head, stage 4
        'L89.816',  # Pressure-induced deep tissue damage of head
        'L89.819',  # Pressure ulcer of head, unspecified stage
        'L89.890',  # Pressure ulcer of other site, unstageable
        'L89.891',  # Pressure ulcer of other site, stage 1
        'L89.892',  # Pressure ulcer of other site, stage 2
        'L89.893',  # Pressure ulcer of other site, stage 3
        'L89.894',  # Pressure ulcer of other site, stage 4
        'L89.896',  # Pressure-induced deep tissue damage of other site
        'L89.899',  # Pressure ulcer of other site, unspecified stage
        'L89.90',  # Pressure ulcer of unspecified site, unspecified stage
        'L89.91',  # Pressure ulcer of unspecified site, stage 1
        'L89.92',  # Pressure ulcer of unspecified site, stage 2
        'L89.93',  # Pressure ulcer of unspecified site, stage 3
        'L89.94',  # Pressure ulcer of unspecified site, stage 4
        'L89.95',  # Pressure ulcer of unspecified site, unstageable
        'L89.96',  # Pressure-induced deep tissue damage of unspecified site
        'M62.50',  # Muscle wasting and atrophy, not elsewhere classified, unspecified site
        'M62.81',  # Muscle weakness (generalized)
        'M62.84',  # Sarcopenia
        'W01.0XXA',  # Fall on same level from slipping, tripping and stumbling without subsequent striking against object, initial encounter
        'W01.0XXD',  # Fall on same level from slipping, tripping and stumbling without subsequent striking against object, subsequent encounter
        'W01.0XXS',  # Fall on same level from slipping, tripping and stumbling without subsequent striking against object, sequela
        'W01.10XA',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified object, initial encounter
        'W01.10XD',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified object, subsequent encounter
        'W01.10XS',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified object, sequela
        'W01.110A',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against sharp glass, initial encounter
        'W01.110D',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against sharp glass, subsequent encounter
        'W01.110S',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against sharp glass, sequela
        'W01.111A',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against power tool or machine, initial encounter
        'W01.111D',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against power tool or machine, subsequent encounter
        'W01.111S',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against power tool or machine, sequela
        'W01.118A',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other sharp object, initial encounter
        'W01.118D',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other sharp object, subsequent encounter
        'W01.118S',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other sharp object, sequela
        'W01.119A',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified sharp object, initial encounter
        'W01.119D',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified sharp object, subsequent encounter
        'W01.119S',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against unspecified sharp object, sequela
        'W01.190A',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against furniture, initial encounter
        'W01.190D',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against furniture, subsequent encounter
        'W01.190S',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against furniture, sequela
        'W01.198A',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other object, initial encounter
        'W01.198D',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other object, subsequent encounter
        'W01.198S',  # Fall on same level from slipping, tripping and stumbling with subsequent striking against other object, sequela
        'W06.XXXA',  # Fall from bed, initial encounter
        'W06.XXXD',  # Fall from bed, subsequent encounter
        'W06.XXXS',  # Fall from bed, sequela
        'W07.XXXA',  # Fall from chair, initial encounter
        'W07.XXXD',  # Fall from chair, subsequent encounter
        'W07.XXXS',  # Fall from chair, sequela
        'W08.XXXA',  # Fall from other furniture, initial encounter
        'W08.XXXD',  # Fall from other furniture, subsequent encounter
        'W08.XXXS',  # Fall from other furniture, sequela
        'W10.0XXA',  # Fall (on)(from) escalator, initial encounter
        'W10.0XXD',  # Fall (on)(from) escalator, subsequent encounter
        'W10.0XXS',  # Fall (on)(from) escalator, sequela
        'W10.1XXA',  # Fall (on)(from) sidewalk curb, initial encounter
        'W10.1XXD',  # Fall (on)(from) sidewalk curb, subsequent encounter
        'W10.1XXS',  # Fall (on)(from) sidewalk curb, sequela
        'W10.2XXA',  # Fall (on)(from) incline, initial encounter
        'W10.2XXD',  # Fall (on)(from) incline, subsequent encounter
        'W10.2XXS',  # Fall (on)(from) incline, sequela
        'W10.8XXA',  # Fall (on) (from) other stairs and steps, initial encounter
        'W10.8XXD',  # Fall (on) (from) other stairs and steps, subsequent encounter
        'W10.8XXS',  # Fall (on) (from) other stairs and steps, sequela
        'W10.9XXA',  # Fall (on) (from) unspecified stairs and steps, initial encounter
        'W10.9XXD',  # Fall (on) (from) unspecified stairs and steps, subsequent encounter
        'W10.9XXS',  # Fall (on) (from) unspecified stairs and steps, sequela
        'W18.00XA',  # Striking against unspecified object with subsequent fall, initial encounter
        'W18.00XD',  # Striking against unspecified object with subsequent fall, subsequent encounter
        'W18.00XS',  # Striking against unspecified object with subsequent fall, sequela
        'W18.02XA',  # Striking against glass with subsequent fall, initial encounter
        'W18.02XD',  # Striking against glass with subsequent fall, subsequent encounter
        'W18.02XS',  # Striking against glass with subsequent fall, sequela
        'W18.09XA',  # Striking against other object with subsequent fall, initial encounter
        'W18.09XD',  # Striking against other object with subsequent fall, subsequent encounter
        'W18.09XS',  # Striking against other object with subsequent fall, sequela
        'W18.11XA',  # Fall from or off toilet without subsequent striking against object, initial encounter
        'W18.11XD',  # Fall from or off toilet without subsequent striking against object, subsequent encounter
        'W18.11XS',  # Fall from or off toilet without subsequent striking against object, sequela
        'W18.12XA',  # Fall from or off toilet with subsequent striking against object, initial encounter
        'W18.12XD',  # Fall from or off toilet with subsequent striking against object, subsequent encounter
        'W18.12XS',  # Fall from or off toilet with subsequent striking against object, sequela
        'W18.2XXA',  # Fall in (into) shower or empty bathtub, initial encounter
        'W18.2XXD',  # Fall in (into) shower or empty bathtub, subsequent encounter
        'W18.2XXS',  # Fall in (into) shower or empty bathtub, sequela
        'W18.30XA',  # Fall on same level, unspecified, initial encounter
        'W18.30XD',  # Fall on same level, unspecified, subsequent encounter
        'W18.30XS',  # Fall on same level, unspecified, sequela
        'W18.31XA',  # Fall on same level due to stepping on an object, initial encounter
        'W18.31XD',  # Fall on same level due to stepping on an object, subsequent encounter
        'W18.31XS',  # Fall on same level due to stepping on an object, sequela
        'W18.39XA',  # Other fall on same level, initial encounter
        'W18.39XD',  # Other fall on same level, subsequent encounter
        'W18.39XS',  # Other fall on same level, sequela
        'W19.XXXA',  # Unspecified fall, initial encounter
        'W19.XXXD',  # Unspecified fall, subsequent encounter
        'W19.XXXS',  # Unspecified fall, sequela
        'Y92.199',  # Unspecified place in other specified residential institution as the place of occurrence of the external cause
        'Z59.3',  # Problems related to living in residential institution
        'Z73.6',  # Limitation of activities due to disability
        'Z74.01',  # Bed confinement status
        'Z74.09',  # Other reduced mobility
        'Z74.1',  # Need for assistance with personal care
        'Z74.2',  # Need for assistance at home and no other household member able to render care
        'Z74.3',  # Need for continuous supervision
        'Z74.8',  # Other problems related to care provider dependency
        'Z74.9',  # Problem related to care provider dependency, unspecified
        'Z91.81',  # History of falling
        'Z99.11',  # Dependence on respirator [ventilator] status
        'Z99.3',  # Dependence on wheelchair
        'Z99.81',  # Dependence on supplemental oxygen
        'Z99.89',  # Dependence on other enabling machines and devices
    }
    SNOMEDCT = {
        '16728003',  # Inherited spastic paresis (disorder)
        '17886000',  # Fall from wheelchair (event)
        '20902002',  # Fall from bed (event)
        '33036003',  # Fall on same level (event)
        '40104005',  # Fall in home (event)
        '44188002',  # Fall in shower (event)
        '52702003',  # Chronic fatigue syndrome (disorder)
        '56307009',  # Fall from table (event)
        '67223001',  # Fall on escalator (event)
        '74541001',  # Fall from bench (event)
        '83468000',  # Fall from chair (event)
        '90619006',  # Fall in bathtub (event)
        '129588001',  # Adult failure to thrive syndrome (disorder)
        '162845004',  # On examination - failure to thrive (disorder)
        '214436006',  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting (event)
        '214437002',  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, driver of motor vehicle injured (event)
        '214438007',  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, passenger of motor vehicle injured (event)
        '214439004',  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, motor cyclist injured (event)
        '214441003',  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, occupant of tram injured (event)
        '214442005',  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, rider of animal or occupant of animal-drawn vehicle injured (event)
        '214443000',  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, pedal cyclist injured (event)
        '214444006',  # Noncollision motor vehicle traffic accident involving fall down stairs of motor bus while boarding or alighting, pedestrian injured (event)
        '217082002',  # Accidental fall (event)
        '217083007',  # Fall on or from stairs or steps (event)
        '217084001',  # Fall on or from escalator (event)
        '217086004',  # Fall from escalator (event)
        '217088003',  # Fall on or from stairs (event)
        '217090002',  # Fall from stairs (event)
        '217092005',  # Fall on or from steps (event)
        '217093000',  # Fall on steps (event)
        '217094006',  # Fall from steps (event)
        '217142006',  # Fall from chair or bed (event)
        '217154006',  # Fall on same level from slipping, tripping or stumbling (event)
        '217155007',  # Fall on same level from slipping (event)
        '217156008',  # Fall on same level from tripping (event)
        '217157004',  # Fall on same level from stumbling (event)
        '217158009',  # Fall on moving sidewalk (event)
        '217173005',  # Fall from bump against object (event)
        '225558004',  # Pressure ulcer of back (disorder)
        '225562005',  # Pressure ulcer of sacral region (disorder)
        '225563000',  # Pressure ulcer of buttock (disorder)
        '242109009',  # Fall down stairs of motor bus while boarding or alighting (event)
        '242389003',  # Fall due to wet surface (event)
        '242390007',  # Fall due to polished surface (event)
        '242391006',  # Fall due to discarded object (event)
        '242392004',  # Fall in bath or shower (event)
        '242395002',  # Fall due to trip on loose carpet (event)
        '242396001',  # Fall due to uneven surface indoors (event)
        '242413007',  # Fall from furniture (event)
        '242414001',  # Fall from stool (event)
        '242419006',  # Fall from toilet seat (event)
        '269699007',  # Fall on same level from impact against object (event)
        '274918000',  # Fall on same level due to nature of surface (event)
        '414188008',  # Fall down stairs (event)
        '414189000',  # Fall down steps (event)
        '414190009',  # Fall on stairs (event)
        '427849003',  # Fall on hard surface (event)
        '428484005',  # Organic failure to thrive (disorder)
        '429621003',  # Fall on concrete (event)
        '699214007',  # Pressure ulcer of hip (disorder)
        '699215008',  # Pressure ulcer of lower back (disorder)
        '699216009',  # Pressure ulcer of coccygeal region (disorder)
        '699218005',  # Pressure ulcer of upper back (disorder)
        '715504003',  # Spastic paraparesis co-occurrent with deafness (disorder)
        '763829004',  # Oculopharyngodistal myopathy (disorder)
        '823018004',  # Muscle paresis due to and following neuromuscular blockade (disorder)
        '92341000119107',  # Weakness of extremities as sequela of stroke (disorder)
        '138371000119104',  # Paresis as late effect of poliomyelitis (disorder)
        '459821000124104',  # McArdle sign (finding)
        '8960001000004106',  # Pressure ulcer of skin of right ischial tuberosity region (disorder)
        '10637031000119106',  # Pressure ulcer of buttock stage 1 (disorder)
        '10637071000119109',  # Pressure ulcer of buttock stage 2 (disorder)
        '10637111000119102',  # Pressure ulcer of buttock stage 3 (disorder)
        '10637151000119101',  # Pressure ulcer of buttock stage 4 (disorder)
    }


class CongenitalOrAcquiredAbsenceOfCervix(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent diagnoses related to absence of a cervix.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of an absence of cervix with or without remaining uterus, congenital or acquired absence of cervix, radical or total hysterectomy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of a partial hysterectomy, absence of uterus with remaining cervical stump.

    ** Used in:** CMS124v10
    """

    VALUE_SET_NAME = 'Congenital or Acquired Absence of Cervix'
    OID = '2.16.840.1.113883.3.464.1003.111.12.1016'
    DEFINITION_VERSION = '20210224'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'Q51.5',  # Agenesis and aplasia of cervix
        'Z90.710',  # Acquired absence of both cervix and uterus
        'Z90.712',  # Acquired absence of cervix with remaining uterus
    }
    ICD9CM = {
        '618.5',  # Prolapse of vaginal vault after hysterectomy
        '752.43',  # Cervical agenesis
        'V88.01',  # Acquired absence of both cervix and uterus
        'V88.03',  # Acquired absence of cervix with remaining uterus
    }
    SNOMEDCT = {
        '37687000',  # Congenital absence of cervix (disorder)
        '248911005',  # Uterine cervix absent (finding)
        '428078001',  # History of total hysterectomy (situation)
        '429290001',  # History of radical hysterectomy (situation)
        '429763009',  # History of total hysterectomy with bilateral salpingo-oophorectomy (situation)
        '723171001',  # Acquired absence of cervix and uterus (disorder)
        '10738891000119107',  # History of total hysterectomy without abnormal cervical Papanicolaou smear (situation)
    }


class HistoryOfBilateralMastectomy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of a history of bilateral mastectomy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a history of a bilateral mastectomy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of a history of a unilateral or unspecified mastectomy.

    ** Used in:** CMS125v10
    """

    VALUE_SET_NAME = 'History of bilateral mastectomy'
    OID = '2.16.840.1.113883.3.464.1003.198.12.1068'
    DEFINITION_VERSION = '20171219'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'Z90.13',  # Acquired absence of bilateral breasts and nipples
    }
    SNOMEDCT = {
        '428529004',  # History of bilateral mastectomy (situation)
        '136071000119101',  # History of bilateral prophylactic mastectomy (situation)
    }


class StatusPostLeftMastectomy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of a history of a left mastectomy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a left mastectomy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of a right, bilateral, or unspecified mastectomy.

    ** Used in:** CMS125v10
    """

    VALUE_SET_NAME = 'Status Post Left Mastectomy'
    OID = '2.16.840.1.113883.3.464.1003.198.12.1069'
    DEFINITION_VERSION = '20171216'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'Z90.12',  # Acquired absence of left breast and nipple
    }
    SNOMEDCT = {
        '429009003',  # History of left mastectomy (situation)
        '137671000119105',  # History of prophylactic mastectomy of left breast (situation)
    }


class StatusPostRightMastectomy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of a history of a right mastectomy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a right mastectomy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of a left, bilateral, or unspecified mastectomy.

    ** Used in:** CMS125v10
    """

    VALUE_SET_NAME = 'Status Post Right Mastectomy'
    OID = '2.16.840.1.113883.3.464.1003.198.12.1070'
    DEFINITION_VERSION = '20171216'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'Z90.11',  # Acquired absence of right breast and nipple
    }
    SNOMEDCT = {
        '429242008',  # History of right mastectomy (situation)
        '137681000119108',  # History of prophylactic mastectomy of right breast (situation)
    }


class UnilateralMastectomy_UnspecifiedLaterality(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of a unilateral mastectomy where the side was not specified.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a unilateral mastectomy where the side was unspecified.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of a bilateral mastectomy or a unilateral mastectomy where the side is specified.

    ** Used in:** CMS125v10
    """

    VALUE_SET_NAME = 'Unilateral Mastectomy, Unspecified Laterality'
    OID = '2.16.840.1.113883.3.464.1003.198.12.1071'
    DEFINITION_VERSION = '20171216'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'Z90.10',  # Acquired absence of unspecified breast and nipple
    }
    ICD9CM = {
        'V45.71',  # Acquired absence of breast and nipple
    }
    SNOMEDCT = {
        '8115005',  # Unilateral radical mastectomy (situation)
        '22964006',  # Unilateral simple mastectomy (situation)
        '72269009',  # Unilateral modified radical mastectomy (situation)
        '73359007',  # Unilateral mastectomy (situation)
        '88764002',  # Unilateral extended simple mastectomy (situation)
        '248802009',  # Absence of breast (finding)
    }


class MajorDepression(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  major depression.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of major depression including mild, moderate, and severe depression, maternal depression, and depression with and without psychotic features.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of depression in remission.

    ** Used in:** CMS128v10
    """

    VALUE_SET_NAME = 'Major Depression'
    OID = '2.16.840.1.113883.3.464.1003.105.12.1007'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'F32.0',  # Major depressive disorder, single episode, mild
        'F32.1',  # Major depressive disorder, single episode, moderate
        'F32.2',  # Major depressive disorder, single episode, severe without psychotic features
        'F32.3',  # Major depressive disorder, single episode, severe with psychotic features
        'F32.4',  # Major depressive disorder, single episode, in partial remission
        'F32.9',  # Major depressive disorder, single episode, unspecified
        'F33.0',  # Major depressive disorder, recurrent, mild
        'F33.1',  # Major depressive disorder, recurrent, moderate
        'F33.2',  # Major depressive disorder, recurrent severe without psychotic features
        'F33.3',  # Major depressive disorder, recurrent, severe with psychotic symptoms
        'F33.41',  # Major depressive disorder, recurrent, in partial remission
        'F33.9',  # Major depressive disorder, recurrent, unspecified
    }
    SNOMEDCT = {
        '832007',  # Moderate major depression (disorder)
        '2618002',  # Chronic recurrent major depressive disorder (disorder)
        '14183003',  # Chronic major depressive disorder, single episode (disorder)
        '15193003',  # Severe recurrent major depression with psychotic features, mood-incongruent (disorder)
        '15639000',  # Moderate major depression, single episode (disorder)
        '18818009',  # Moderate recurrent major depression (disorder)
        '20250007',  # Severe major depression, single episode, with psychotic features, mood-incongruent (disorder)
        '25922000',  # Major depressive disorder, single episode with postpartum onset (disorder)
        '28475009',  # Severe recurrent major depression with psychotic features (disorder)
        '33078009',  # Severe recurrent major depression with psychotic features, mood-congruent (disorder)
        '33736005',  # Severe major depression with psychotic features, mood-congruent (disorder)
        '36474008',  # Severe recurrent major depression without psychotic features (disorder)
        '36923009',  # Major depression, single episode (disorder)
        '38694004',  # Recurrent major depressive disorder with atypical features (disorder)
        '39809009',  # Recurrent major depressive disorder with catatonic features (disorder)
        '40379007',  # Mild recurrent major depression (disorder)
        '42925002',  # Major depressive disorder, single episode with atypical features (disorder)
        '60099002',  # Severe major depression with psychotic features, mood-incongruent (disorder)
        '63778009',  # Major depressive disorder, single episode with melancholic features (disorder)
        '66344007',  # Recurrent major depression (disorder)
        '69392006',  # Major depressive disorder, single episode with catatonic features (disorder)
        '71336009',  # Recurrent major depressive disorder with postpartum onset (disorder)
        '73867007',  # Severe major depression with psychotic features (disorder)
        '75084000',  # Severe major depression without psychotic features (disorder)
        '76441001',  # Severe major depression, single episode, without psychotic features (disorder)
        '77911002',  # Severe major depression, single episode, with psychotic features, mood-congruent (disorder)
        '79298009',  # Mild major depression, single episode (disorder)
        '87512008',  # Mild major depression (disorder)
        '191604000',  # Single major depressive episode, severe, with psychosis (disorder)
        '191610000',  # Recurrent major depressive episodes, mild (disorder)
        '191611001',  # Recurrent major depressive episodes, moderate (disorder)
        '191613003',  # Recurrent major depressive episodes, severe, with psychosis (disorder)
        '268621008',  # Recurrent major depressive episodes (disorder)
        '319768000',  # Recurrent major depressive disorder with melancholic features (disorder)
        '320751009',  # Major depression, melancholic type (disorder)
        '370143000',  # Major depressive disorder (disorder)
        '430852001',  # Severe major depression, single episode, with psychotic features (disorder)
        '450714000',  # Severe major depression (disorder)
        '719592004',  # Moderately severe major depression (disorder)
        '720451004',  # Minimal recurrent major depression (disorder)
        '720452006',  # Moderately severe recurrent major depression (disorder)
        '720453001',  # Moderately severe major depression single episode (disorder)
        '720454007',  # Minimal major depression single episode (disorder)
        '720455008',  # Minimal major depression (disorder)
        '726772006',  # Major depression with psychotic features (disorder)
        '251000119105',  # Severe major depression, single episode (disorder)
        '281000119103',  # Severe recurrent major depression (disorder)
        '10811121000119102',  # Major depressive disorder in mother complicating childbirth (disorder)
        '10811161000119107',  # Major depressive disorder in mother complicating pregnancy (disorder)
        '16264621000119109',  # Recurrent mild major depressive disorder co-occurrent with anxiety (disorder)
        '16264901000119109',  # Recurrent moderate major depressive disorder co-occurrent with anxiety (disorder)
        '16265951000119109',  # Mild major depressive disorder co-occurrent with anxiety single episode (disorder)
        '16266831000119100',  # Moderate major depressive disorder co-occurrent with anxiety single episode (disorder)
        '16266991000119108',  # Severe major depressive disorder co-occurrent with anxiety single episode (disorder)
    }


class PainRelatedToProstateCancer(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent types of pain or conditions that could cause pain that could be associated with prostate cancer.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent  types of pain or conditions that could cause pain that could be associated with prostate cancer.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS129v11
    """

    VALUE_SET_NAME = 'Pain Related to Prostate Cancer'
    OID = '2.16.840.1.113883.3.526.3.451'
    DEFINITION_VERSION = '20200306'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'G89.3',  # Neoplasm related pain (acute) (chronic)
        'M43.27',  # Fusion of spine, lumbosacral region
        'M43.28',  # Fusion of spine, sacral and sacrococcygeal region
        'M53.2X7',  # Spinal instabilities, lumbosacral region
        'M53.2X8',  # Spinal instabilities, sacral and sacrococcygeal region
        'M53.3',  # Sacrococcygeal disorders, not elsewhere classified
        'M53.86',  # Other specified dorsopathies, lumbar region
        'M53.87',  # Other specified dorsopathies, lumbosacral region
        'M53.88',  # Other specified dorsopathies, sacral and sacrococcygeal region
        'M54.6',  # Pain in thoracic spine
        'M54.89',  # Other dorsalgia
        'M54.9',  # Dorsalgia, unspecified
        'M85.9',  # Disorder of bone density and structure, unspecified
        'M89.9',  # Disorder of bone, unspecified
        'M94.9',  # Disorder of cartilage, unspecified
        'R07.82',  # Intercostal pain
        'R07.89',  # Other chest pain
        'R07.9',  # Chest pain, unspecified
        'R10.0',  # Acute abdomen
        'R10.9',  # Unspecified abdominal pain
    }
    SNOMEDCT = {
        '3200003',  # Sacrocoxalgia (finding)
        '12584003',  # Bone pain (finding)
        '20793008',  # Scapulalgia (finding)
        '34789001',  # Pain in the coccyx (finding)
        '36729000',  # Prostatic pain (finding)
        '48926009',  # Pain in spine (finding)
        '89638008',  # Xiphodynia (finding)
        '161977000',  # Costal margin chest pain (finding)
        '203508001',  # Bony pelvic pain (finding)
        '203509009',  # Clavicle pain (finding)
        '267981009',  # Pain in thoracic spine (finding)
        '282743009',  # Malignant bone pain (finding)
        '297217002',  # Rib pain (finding)
        '298731003',  # Pain of sternum (finding)
        '298740004',  # Rib tender (finding)
        '301773003',  # Pain in femur (finding)
        '426135001',  # Chronic prostatitis - chronic pelvic pain syndrome (disorder)
    }


class ProstateCancer(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for prostate cancer.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of prostate cancer.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS129v11, CMS645v5
    """

    VALUE_SET_NAME = 'Prostate Cancer'
    OID = '2.16.840.1.113883.3.526.3.319'
    DEFINITION_VERSION = '20200306'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'C61',  # Malignant neoplasm of prostate
    }
    SNOMEDCT = {
        '93974005',  # Primary malignant neoplasm of prostate (disorder)
        '94503003',  # Secondary malignant neoplasm of prostate (disorder)
        '254900004',  # Carcinoma of prostate (disorder)
        '278060005',  # Endometrioid carcinoma of prostate (disorder)
        '314969001',  # Local recurrence of malignant tumor of prostate (disorder)
        '369485004',  # Malignant tumor involving prostate by direct extension from bladder (disorder)
        '369486003',  # Malignant tumor involving prostate by separate metastasis from bladder (disorder)
        '396198006',  # Small cell carcinoma of prostate (disorder)
        '399068003',  # Malignant tumor of prostate (disorder)
        '399490008',  # Adenocarcinoma of prostate (disorder)
        '399590005',  # Squamous cell carcinoma of prostate (disorder)
        '427492003',  # Hormone refractory prostate cancer (disorder)
    }


class MalignantNeoplasmOfColon(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  malignant neoplasms of the colon.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of malignant neoplasms of the colon, rectum and anus.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS130v10
    """

    VALUE_SET_NAME = 'Malignant Neoplasm of Colon'
    OID = '2.16.840.1.113883.3.464.1003.108.12.1001'
    DEFINITION_VERSION = '20171222'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'C18.0',  # Malignant neoplasm of cecum
        'C18.1',  # Malignant neoplasm of appendix
        'C18.2',  # Malignant neoplasm of ascending colon
        'C18.3',  # Malignant neoplasm of hepatic flexure
        'C18.4',  # Malignant neoplasm of transverse colon
        'C18.5',  # Malignant neoplasm of splenic flexure
        'C18.6',  # Malignant neoplasm of descending colon
        'C18.7',  # Malignant neoplasm of sigmoid colon
        'C18.8',  # Malignant neoplasm of overlapping sites of colon
        'C18.9',  # Malignant neoplasm of colon, unspecified
        'C19',  # Malignant neoplasm of rectosigmoid junction
        'C20',  # Malignant neoplasm of rectum
        'C21.2',  # Malignant neoplasm of cloacogenic zone
        'C21.8',  # Malignant neoplasm of overlapping sites of rectum, anus and anal canal
        'C78.5',  # Secondary malignant neoplasm of large intestine and rectum
    }
    ICD9CM = {
        '153.0',  # Malignant neoplasm of hepatic flexure
        '153.1',  # Malignant neoplasm of transverse colon
        '153.2',  # Malignant neoplasm of descending colon
        '153.3',  # Malignant neoplasm of sigmoid colon
        '153.4',  # Malignant neoplasm of cecum
        '153.5',  # Malignant neoplasm of appendix vermiformis
        '153.6',  # Malignant neoplasm of ascending colon
        '153.7',  # Malignant neoplasm of splenic flexure
        '153.8',  # Malignant neoplasm of other specified sites of large intestine
        '153.9',  # Malignant neoplasm of colon, unspecified site
        '154.0',  # Malignant neoplasm of rectosigmoid junction
        '154.1',  # Malignant neoplasm of rectum
        '197.5',  # Secondary malignant neoplasm of large intestine and rectum
    }
    SNOMEDCT = {
        '93761005',  # Primary malignant neoplasm of colon (disorder)
        '93771007',  # Primary malignant neoplasm of descending colon (disorder)
        '93826009',  # Primary malignant neoplasm of hepatic flexure of colon (disorder)
        '93980002',  # Primary malignant neoplasm of rectosigmoid junction (disorder)
        '94006002',  # Primary malignant neoplasm of sigmoid colon (disorder)
        '94072004',  # Primary malignant neoplasm of splenic flexure of colon (disorder)
        '94105000',  # Primary malignant neoplasm of transverse colon (disorder)
        '94179005',  # Secondary malignant neoplasm of ascending colon (disorder)
        '94260004',  # Secondary malignant neoplasm of colon (disorder)
        '94271003',  # Secondary malignant neoplasm of descending colon (disorder)
        '94328005',  # Secondary malignant neoplasm of hepatic flexure of colon (disorder)
        '94509004',  # Secondary malignant neoplasm of rectosigmoid junction (disorder)
        '94538001',  # Secondary malignant neoplasm of sigmoid colon (disorder)
        '94604000',  # Secondary malignant neoplasm of splenic flexure of colon (disorder)
        '94643001',  # Secondary malignant neoplasm of transverse colon (disorder)
        '109838007',  # Overlapping malignant neoplasm of colon (disorder)
        '187757001',  # Malignant neoplasm, overlapping lesion of colon (disorder)
        '269533000',  # Carcinoma of colon (disorder)
        '269544008',  # Carcinoma of the rectosigmoid junction (disorder)
        '285312008',  # Carcinoma of sigmoid colon (disorder)
        '285611007',  # Metastasis to colon of unknown primary (disorder)
        '301756000',  # Adenocarcinoma of sigmoid colon (disorder)
        '312111009',  # Carcinoma of ascending colon (disorder)
        '312112002',  # Carcinoma of transverse colon (disorder)
        '312113007',  # Carcinoma of descending colon (disorder)
        '312114001',  # Carcinoma of hepatic flexure (disorder)
        '312115000',  # Carcinoma of splenic flexure (disorder)
        '314965007',  # Local recurrence of malignant tumor of colon (disorder)
        '315058005',  # Hereditary nonpolyposis colon cancer (disorder)
        '363406005',  # Malignant neoplasm of colon (disorder)
        '363407001',  # Malignant tumor of hepatic flexure (disorder)
        '363408006',  # Malignant tumor of transverse colon (disorder)
        '363409003',  # Malignant tumor of descending colon (disorder)
        '363410008',  # Malignant tumor of sigmoid colon (disorder)
        '363412000',  # Malignant tumor of ascending colon (disorder)
        '363413005',  # Malignant tumor of splenic flexure (disorder)
        '363414004',  # Malignant tumor of rectosigmoid junction (disorder)
        '363510005',  # Malignant tumor of large intestine (disorder)
        '425178004',  # Adenocarcinoma of rectosigmoid junction (disorder)
        '449218003',  # Lymphoma of sigmoid colon (disorder)
        '716654007',  # Non-polyposis Turcot syndrome (disorder)
        '721695008',  # Primary adenocarcinoma of ascending colon and right flexure (disorder)
        '721696009',  # Primary adenocarcinoma of transverse colon (disorder)
        '721699002',  # Primary adenocarcinoma of descending colon and splenic flexure (disorder)
        '737058005',  # Microsatellite instability-high colorectal cancer (disorder)
        '766981007',  # Squamous cell carcinoma of colon (disorder)
        '1701000119104',  # Primary adenocarcinoma of colon (disorder)
        '96281000119107',  # Overlapping malignant neoplasm of colon and rectum (disorder)
        '96981000119102',  # Malignant neoplasm of rectosigmoid junction metastatic to brain (disorder)
        '130381000119103',  # Primary malignant neuroendocrine neoplasm of ascending colon (disorder)
        '133751000119102',  # Lymphoma of colon (disorder)
        '184881000119106',  # Primary adenocarcinoma of rectosigmoid junction (disorder)
        '681601000119101',  # Primary adenocarcinoma of ascending colon (disorder)
    }


class DiabeticRetinopathy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of diabetic retinopathy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of diabetic retinopathy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS142v10, CMS133v10, CMS131v10
    """

    VALUE_SET_NAME = 'Diabetic Retinopathy'
    OID = '2.16.840.1.113883.3.526.3.327'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'E08.311',  # Diabetes mellitus due to underlying condition with unspecified diabetic retinopathy with macular edema
        'E08.319',  # Diabetes mellitus due to underlying condition with unspecified diabetic retinopathy without macular edema
        'E08.3211',  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E08.3212',  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E08.3213',  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E08.3291',  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy without macular edema, right eye
        'E08.3292',  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy without macular edema, left eye
        'E08.3293',  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        'E08.3311',  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E08.3312',  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E08.3313',  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E08.3391',  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        'E08.3392',  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        'E08.3393',  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        'E08.3411',  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E08.3412',  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E08.3413',  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E08.3491',  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy without macular edema, right eye
        'E08.3492',  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy without macular edema, left eye
        'E08.3493',  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        'E08.3511',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, right eye
        'E08.3512',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, left eye
        'E08.3513',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, bilateral
        'E08.3521',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E08.3522',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E08.3523',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E08.3531',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E08.3532',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E08.3533',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E08.3541',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E08.3542',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E08.3543',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E08.3551',  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, right eye
        'E08.3552',  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, left eye
        'E08.3553',  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, bilateral
        'E08.3591',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy without macular edema, right eye
        'E08.3592',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy without macular edema, left eye
        'E08.3593',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy without macular edema, bilateral
        'E09.311',  # Drug or chemical induced diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E09.319',  # Drug or chemical induced diabetes mellitus with unspecified diabetic retinopathy without macular edema
        'E09.3211',  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E09.3212',  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E09.3213',  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E09.3291',  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        'E09.3292',  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        'E09.3293',  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        'E09.3311',  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E09.3312',  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E09.3313',  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E09.3391',  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        'E09.3392',  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        'E09.3393',  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        'E09.3411',  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E09.3412',  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E09.3413',  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E09.3491',  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        'E09.3492',  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        'E09.3493',  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        'E09.3511',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E09.3512',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E09.3513',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E09.3521',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E09.3522',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E09.3523',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E09.3531',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E09.3532',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E09.3533',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E09.3541',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E09.3542',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E09.3543',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E09.3551',  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E09.3552',  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E09.3553',  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E09.3591',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        'E09.3592',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        'E09.3593',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        'E10.311',  # Type 1 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E10.319',  # Type 1 diabetes mellitus with unspecified diabetic retinopathy without macular edema
        'E10.3211',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E10.3212',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E10.3213',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E10.3291',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        'E10.3292',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        'E10.3293',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        'E10.3311',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E10.3312',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E10.3313',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E10.3391',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        'E10.3392',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        'E10.3393',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        'E10.3411',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E10.3412',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E10.3413',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E10.3491',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        'E10.3492',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        'E10.3493',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        'E10.3511',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E10.3512',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E10.3513',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E10.3521',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E10.3522',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E10.3523',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E10.3531',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E10.3532',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E10.3533',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E10.3541',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E10.3542',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E10.3543',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E10.3551',  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E10.3552',  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E10.3553',  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E10.3591',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        'E10.3592',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        'E10.3593',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        'E11.311',  # Type 2 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E11.319',  # Type 2 diabetes mellitus with unspecified diabetic retinopathy without macular edema
        'E11.3211',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E11.3212',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E11.3213',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E11.3291',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        'E11.3292',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        'E11.3293',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        'E11.3311',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E11.3312',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E11.3313',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E11.3391',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        'E11.3392',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        'E11.3393',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        'E11.3411',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E11.3412',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E11.3413',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E11.3491',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        'E11.3492',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        'E11.3493',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        'E11.3511',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E11.3512',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E11.3513',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E11.3521',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E11.3522',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E11.3523',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E11.3531',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E11.3532',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E11.3533',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E11.3541',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E11.3542',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E11.3543',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E11.3551',  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E11.3552',  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E11.3553',  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E11.3591',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        'E11.3592',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        'E11.3593',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
        'E13.311',  # Other specified diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E13.319',  # Other specified diabetes mellitus with unspecified diabetic retinopathy without macular edema
        'E13.3211',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E13.3212',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E13.3213',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E13.3291',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, right eye
        'E13.3292',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, left eye
        'E13.3293',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy without macular edema, bilateral
        'E13.3311',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E13.3312',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E13.3313',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E13.3391',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, right eye
        'E13.3392',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, left eye
        'E13.3393',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy without macular edema, bilateral
        'E13.3411',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E13.3412',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E13.3413',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E13.3491',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, right eye
        'E13.3492',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, left eye
        'E13.3493',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy without macular edema, bilateral
        'E13.3511',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E13.3512',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E13.3513',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E13.3521',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E13.3522',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E13.3523',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E13.3531',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E13.3532',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E13.3533',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E13.3541',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E13.3542',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E13.3543',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E13.3551',  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E13.3552',  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E13.3553',  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E13.3591',  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, right eye
        'E13.3592',  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, left eye
        'E13.3593',  # Other specified diabetes mellitus with proliferative diabetic retinopathy without macular edema, bilateral
    }
    SNOMEDCT = {
        '4855003',  # Retinopathy due to diabetes mellitus (disorder)
        '25412000',  # Microaneurysm of retinal artery due to diabetes mellitus (disorder)
        '59276001',  # Proliferative retinopathy due to diabetes mellitus (disorder)
        '193349004',  # Preproliferative retinopathy due to diabetes mellitus (disorder)
        '193350004',  # Advanced maculopathy due to diabetes mellitus (disorder)
        '232020009',  # Disorder of macula due to diabetes mellitus (disorder)
        '232021008',  # Proliferative retinopathy with optic disc neovascularization due to diabetes mellitus (disorder)
        '232022001',  # Proliferative retinopathy with neovascularization elsewhere than the optic disc due to diabetes mellitus (disorder)
        '232023006',  # Traction detachment of retina due to diabetes mellitus (disorder)
        '311782002',  # Advanced retinal disease due to diabetes mellitus (disorder)
        '312903003',  # Mild nonproliferative retinopathy due to diabetes mellitus (disorder)
        '312904009',  # Moderate nonproliferative retinopathy due to diabetes mellitus (disorder)
        '312905005',  # Severe nonproliferative retinopathy due to diabetes mellitus (disorder)
        '312906006',  # Non-high-risk proliferative retinopathy due to diabetes mellitus (disorder)
        '312907002',  # High risk proliferative retinopathy due to diabetes mellitus (disorder)
        '312908007',  # Quiescent proliferative retinopathy due to diabetes mellitus (disorder)
        '312909004',  # Proliferative retinopathy with iris neovascularization due to diabetes mellitus (disorder)
        '312912001',  # Macular edema due to diabetes mellitus (disorder)
        '314010006',  # Diffuse exudative maculopathy due to diabetes mellitus (disorder)
        '314011005',  # Focal exudative maculopathy due to diabetes mellitus (disorder)
        '314014002',  # Ischemic maculopathy due to diabetes mellitus (disorder)
        '314015001',  # Mixed maculopathy due to diabetes mellitus (disorder)
        '390834004',  # Nonproliferative retinopathy due to diabetes mellitus (disorder)
        '399862001',  # High risk proliferative retinopathy without macular edema due to diabetes mellitus (disorder)
        '399863006',  # Very severe nonproliferative retinopathy without macular edema due to diabetes mellitus (disorder)
        '399864000',  # Macular edema not clinically significant due to diabetes mellitus (disorder)
        '399865004',  # Very severe proliferative retinopathy due to diabetes mellitus (disorder)
        '399866003',  # Venous beading of retina due to diabetes mellitus (disorder)
        '399868002',  # Intraretinal microvascular anomaly due to diabetes mellitus (disorder)
        '399869005',  # High risk proliferative retinopathy not amenable to photocoagulation due to diabetes mellitus (disorder)
        '399870006',  # Non-high-risk proliferative retinopathy with no macular edema due to diabetes mellitus (disorder)
        '399871005',  # Visually threatening retinopathy due to diabetes mellitus (disorder)
        '399872003',  # Severe nonproliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        '399873008',  # Severe nonproliferative retinopathy without macular edema due to diabetes mellitus (disorder)
        '399874002',  # High risk proliferative retinopathy with clinically significant macula edema due to diabetes mellitus (disorder)
        '399875001',  # Non-high-risk proliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        '399876000',  # Very severe nonproliferative retinopathy due to diabetes mellitus (disorder)
        '399877009',  # Very severe nonproliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        '420486006',  # Exudative maculopathy due to type 1 diabetes mellitus (disorder)
        '420789003',  # Retinopathy due to type 1 diabetes mellitus (disorder)
        '421779007',  # Exudative maculopathy due to type 2 diabetes mellitus (disorder)
        '422034002',  # Retinopathy due to type 2 diabetes mellitus (disorder)
        '870420005',  # Severe nonproliferative retinopathy with venous beading of retina due to diabetes mellitus (disorder)
    }


class AcuteAndSubacuteIridocyclitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of acute and subacute iridocyclitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of acute and subacute iridocyclitis, including primary and secondary iridocyclitis, and hypopyon.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Acute and Subacute Iridocyclitis'
    OID = '2.16.840.1.113883.3.526.3.1241'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H20.00',  # Unspecified acute and subacute iridocyclitis
        'H20.011',  # Primary iridocyclitis, right eye
        'H20.012',  # Primary iridocyclitis, left eye
        'H20.013',  # Primary iridocyclitis, bilateral
        'H20.021',  # Recurrent acute iridocyclitis, right eye
        'H20.022',  # Recurrent acute iridocyclitis, left eye
        'H20.023',  # Recurrent acute iridocyclitis, bilateral
        'H20.031',  # Secondary infectious iridocyclitis, right eye
        'H20.032',  # Secondary infectious iridocyclitis, left eye
        'H20.033',  # Secondary infectious iridocyclitis, bilateral
        'H20.041',  # Secondary noninfectious iridocyclitis, right eye
        'H20.042',  # Secondary noninfectious iridocyclitis, left eye
        'H20.043',  # Secondary noninfectious iridocyclitis, bilateral
        'H20.051',  # Hypopyon, right eye
        'H20.052',  # Hypopyon, left eye
        'H20.053',  # Hypopyon, bilateral
    }
    SNOMEDCT = {
        '4927003',  # Acute anterior uveitis (disorder)
        '6869001',  # Recurrent iridocyclitis (disorder)
        '9091006',  # Gonococcal iridocyclitis (disorder)
        '10698009',  # Herpes zoster iridocyclitis (disorder)
        '12630008',  # Primary iridocyclitis (disorder)
        '13608004',  # Herpes simplex iridocyclitis (disorder)
        '14427009',  # Subacute iritis (disorder)
        '20555001',  # Herpes simplex iritis (disorder)
        '29050005',  # Acute iritis (disorder)
        '29538005',  # Glaucomatocyclitic crisis (disorder)
        '30080002',  # Secondary syphilitic iridocyclitis (disorder)
        '43193009',  # Gouty iritis (disorder)
        '46916006',  # Noninfectious secondary iridocyclitis (disorder)
        '64015001',  # Subacute anterior uveitis (disorder)
        '65074000',  # Iritis (disorder)
        '70461003',  # Lens-induced iridocyclitis (disorder)
        '77971008',  # Iridocyclitis (disorder)
        '89114005',  # Infectious secondary iridocyclitis (disorder)
        '193487008',  # Secondary infected iridocyclitis (disorder)
        '193488003',  # Secondary non-infected iridocyclitis (disorder)
        '193489006',  # Iritis due to diabetes mellitus (disorder)
        '231947004',  # Anterior uveitis idiopathic (disorder)
        '267618008',  # Acute and subacute iridocyclitis (disorder)
        '267619000',  # Non-infectious anterior uveitis (disorder)
        '410480004',  # Iritis in Behcet's syndrome (disorder)
        '410482007',  # Iritis in psoriatic arthritis (disorder)
        '410692006',  # Anterior uveitis (disorder)
        '413851004',  # Cidofovir-induced anterior uveitis (disorder)
        '414600000',  # Lepromatous anterior uveitis (disorder)
        '416666007',  # Anterior uveitis in juvenile idiopathic arthritis (disorder)
        '417020006',  # Traumatic iritis (disorder)
        '420485005',  # Herpetic iridocyclitis (disorder)
        '733317009',  # Primary anterior uveitis (disorder)
        '35881000119106',  # Noninfectious secondary iritis (disorder)
        '220101000119108',  # Infectious secondary iritis (disorder)
        '331071000119101',  # Herpes zoster iritis (disorder)
        '332921000119107',  # Primary iridocyclitis of right eye (disorder)
        '336531000119103',  # Acute iritis of right eye (disorder)
        '336541000119107',  # Acute anterior uveitis of right eye (disorder)
        '336921000119106',  # Lens induced iritis (disorder)
        '338531000119102',  # Primary iridocyclitis of left eye (disorder)
        '342141000119102',  # Acute iritis of left eye (disorder)
        '342151000119100',  # Acute anterior uveitis of left eye (disorder)
        '342381000119109',  # Gonococcal iritis (disorder)
        '344301000119109',  # Primary iridocyclitis of bilateral eyes (disorder)
        '347301000119103',  # Acute iritis of bilateral eyes (disorder)
        '347311000119100',  # Acute anterior uveitis of bilateral eyes (disorder)
        '352941000119102',  # Sarcoid iridocyclitis (disorder)
        '675361000119103',  # Subacute iridocyclitis of bilateral eyes (disorder)
        '675371000119109',  # Subacute iridocyclitis of left eye (disorder)
        '675381000119107',  # Subacute iridocyclitis of right eye (disorder)
        '15638451000119104',  # Iritis of right eye (disorder)
        '15638491000119109',  # Iritis of left eye (disorder)
        '15638531000119109',  # Iritis of bilateral eyes (disorder)
        '15679361000119103',  # Iridocyclitis of left eye (disorder)
        '15679401000119107',  # Iridocyclitis of right eye (disorder)
        '15679441000119109',  # Iridocyclitis of bilateral eyes (disorder)
    }


class Amblyopia(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of amblyopia.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of amblyopia, including suspected amblyopia.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Amblyopia'
    OID = '2.16.840.1.113883.3.526.3.1448'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H53.001',  # Unspecified amblyopia, right eye
        'H53.002',  # Unspecified amblyopia, left eye
        'H53.003',  # Unspecified amblyopia, bilateral
        'H53.011',  # Deprivation amblyopia, right eye
        'H53.012',  # Deprivation amblyopia, left eye
        'H53.013',  # Deprivation amblyopia, bilateral
        'H53.021',  # Refractive amblyopia, right eye
        'H53.022',  # Refractive amblyopia, left eye
        'H53.023',  # Refractive amblyopia, bilateral
        'H53.031',  # Strabismic amblyopia, right eye
        'H53.032',  # Strabismic amblyopia, left eye
        'H53.033',  # Strabismic amblyopia, bilateral
        'H53.041',  # Amblyopia suspect, right eye
        'H53.042',  # Amblyopia suspect, left eye
        'H53.043',  # Amblyopia suspect, bilateral
    }
    SNOMEDCT = {
        '965003',  # Toxic amblyopia (disorder)
        '35600002',  # Strabismic amblyopia (disorder)
        '70701004',  # Alcoholic amblyopia (disorder)
        '75834006',  # Strachan's syndrome (disorder)
        '90927000',  # Refractive amblyopia (disorder)
        '193638002',  # Stimulus deprivation amblyopia (disorder)
        '232147001',  # Anisometropic amblyopia (disorder)
        '361120000',  # Nutritional amblyopia (disorder)
        '387742006',  # Amblyopia (disorder)
        '387743001',  # Functional amblyopia (disorder)
        '396184003',  # Isometropic amblyopia (disorder)
        '410486005',  # Occlusion amblyopia (disorder)
        '331961000119108',  # Strabismic amblyopia of right eye (disorder)
        '332701000119101',  # Refractive amblyopia of right eye (disorder)
        '335461000119102',  # Deprivation amblyopia of right eye (disorder)
        '336451000119108',  # Amblyopia of right eye (disorder)
        '337581000119100',  # Strabismic amblyopia of left eye (disorder)
        '338311000119102',  # Refractive amblyopia of left eye (disorder)
        '341071000119100',  # Deprivation amblyopia of left eye (disorder)
        '342061000119102',  # Amblyopia of left eye (disorder)
        '343371000119101',  # Strabismic amblyopia of bilateral eyes (disorder)
        '344081000119104',  # Refractive amblyopia of bilateral eyes (disorder)
        '346331000119107',  # Deprivation amblyopia of bilateral eyes (disorder)
        '347241000119107',  # Amblyopia of bilateral eyes (disorder)
    }


class BurnConfinedToEyeAndAdnexa(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of a burn of the eye and/or adnexa.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of burns of various parts of the eye, or eye corrosion.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Burn Confined to Eye and Adnexa'
    OID = '2.16.840.1.113883.3.526.3.1409'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'T26.01XA',  # Burn of right eyelid and periocular area, initial encounter
        'T26.02XA',  # Burn of left eyelid and periocular area, initial encounter
        'T26.11XA',  # Burn of cornea and conjunctival sac, right eye, initial encounter
        'T26.12XA',  # Burn of cornea and conjunctival sac, left eye, initial encounter
        'T26.21XA',  # Burn with resulting rupture and destruction of right eyeball, initial encounter
        'T26.22XA',  # Burn with resulting rupture and destruction of left eyeball, initial encounter
        'T26.31XA',  # Burns of other specified parts of right eye and adnexa, initial encounter
        'T26.32XA',  # Burns of other specified parts of left eye and adnexa, initial encounter
        'T26.41XA',  # Burn of right eye and adnexa, part unspecified, initial encounter
        'T26.42XA',  # Burn of left eye and adnexa, part unspecified, initial encounter
        'T26.51XA',  # Corrosion of right eyelid and periocular area, initial encounter
        'T26.52XA',  # Corrosion of left eyelid and periocular area, initial encounter
        'T26.61XA',  # Corrosion of cornea and conjunctival sac, right eye, initial encounter
        'T26.62XA',  # Corrosion of cornea and conjunctival sac, left eye, initial encounter
        'T26.71XA',  # Corrosion with resulting rupture and destruction of right eyeball, initial encounter
        'T26.72XA',  # Corrosion with resulting rupture and destruction of left eyeball, initial encounter
        'T26.81XA',  # Corrosions of other specified parts of right eye and adnexa, initial encounter
        'T26.82XA',  # Corrosions of other specified parts of left eye and adnexa, initial encounter
        'T26.91XA',  # Corrosion of right eye and adnexa, part unspecified, initial encounter
        'T26.92XA',  # Corrosion of left eye and adnexa, part unspecified, initial encounter
    }
    SNOMEDCT = {
        '38916004',  # Alkaline chemical burn of cornea AND/OR conjunctival sac (disorder)
        '63542002',  # Acid chemical burn of cornea AND/OR conjunctival sac (disorder)
        '110170006',  # Burn of periorbital region (disorder)
        '110171005',  # Epidermal burn of periorbital region (disorder)
        '110172003',  # Partial thickness burn of periorbital region (disorder)
        '110173008',  # Full thickness burn of periorbital region (disorder)
        '211659009',  # Chemical burn of eyelid region (disorder)
        '211661000',  # Alkaline chemical burn of cornea and conjunctival sac (disorder)
        '211662007',  # Acid chemical burn of cornea and conjunctival sac (disorder)
        '231819004',  # Thermal burn of eyelid (disorder)
        '231844007',  # Lacrimal punctum burn (disorder)
        '231867009',  # Thermal injury to conjunctiva (disorder)
        '231869007',  # Chemical burn injury to conjunctiva (disorder)
        '231884003',  # Burn of sclera (disorder)
        '231943000',  # Thermal and radiation injury to the cornea (disorder)
        '231945007',  # Chemical injury to cornea (disorder)
        '232028002',  # Laser coagulation burn to retina (disorder)
        '269232009',  # Burn confined to eye and adnexa (disorder)
        '274204004',  # Burn of cornea (disorder)
        '274205003',  # Burn of eye region (disorder)
        '284537006',  # Eyelid burn (disorder)
        '284541005',  # Burn of conjunctiva (disorder)
        '287132002',  # Epidermal burn of eyelid (disorder)
        '287133007',  # Burn to eyelid - blister (disorder)
        '287134001',  # Full thickness burn of eyelid (disorder)
        '287137008',  # Burn to cornea - blister (disorder)
        '287138003',  # Burn to cornea - full thickness (disorder)
        '312646002',  # Burn of skin of eye region (disorder)
        '314531003',  # Burn of anterior eyeball segment (disorder)
        '418591009',  # Corneal wound burn (disorder)
        '423838005',  # Scleral surgical wound burn (disorder)
        '445832003',  # Chemical burn of eyelid (disorder)
        '445850005',  # Deep third degree burn of eye region (disorder)
        '446234004',  # Alkaline chemical burn of conjunctival sac (disorder)
        '446556000',  # Acid chemical burn of cornea (disorder)
        '446856005',  # Burn of conjuctival sac (disorder)
        '447094002',  # Alkaline chemical burn of cornea (disorder)
        '447095001',  # Acid chemical burn of conjunctival sac (disorder)
        '447333000',  # Burn of ocular adnexa (disorder)
    }


class CataractSecondaryToOcularDisorders(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of cataract that is a secondary diagnosis to other ocular conditions.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of cataract secondary to degenerative or inflammatory ocular disorders, cataract with neovascularization, or subcapsular glaucomatous flecks.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Cataract Secondary to Ocular Disorders'
    OID = '2.16.840.1.113883.3.526.3.1410'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H26.211',  # Cataract with neovascularization, right eye
        'H26.212',  # Cataract with neovascularization, left eye
        'H26.213',  # Cataract with neovascularization, bilateral
        'H26.221',  # Cataract secondary to ocular disorders (degenerative) (inflammatory), right eye
        'H26.222',  # Cataract secondary to ocular disorders (degenerative) (inflammatory), left eye
        'H26.223',  # Cataract secondary to ocular disorders (degenerative) (inflammatory), bilateral
    }
    SNOMEDCT = {
        '67733005',  # Subcapsular glaucomatous flecks (disorder)
        '76309006',  # Cataract with neovascularization (disorder)
        '193600001',  # Cataract secondary to ocular disease (disorder)
        '193602009',  # Cataract due to inflammatory disorder (disorder)
        '15738161000119104',  # Cataract of right eye due to ocular disease (disorder)
        '15738201000119109',  # Cataract of left eye due ocular disease (disorder)
        '15738241000119106',  # Cataract of left eye due to inflammatory disorder (disorder)
        '15738281000119101',  # Cataract of right eye due to inflammatory disorder (disorder)
        '15738321000119106',  # Cataract of bilateral eyes due to inflammatory disorder (disorder)
    }


class Cataract_Congenital(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of congenital cataract.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of congenital cataract.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Cataract, Congenital'
    OID = '2.16.840.1.113883.3.526.3.1412'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'Q12.0',  # Congenital cataract
    }
    SNOMEDCT = {
        '21590003',  # Congenital zonular cataract (disorder)
        '28550007',  # Congenital capsular cataract (disorder)
        '29590001',  # Congenital total cataract (disorder)
        '66499004',  # Congenital cortical cataract (disorder)
        '76562003',  # Congenital subcapsular cataract (disorder)
        '79410001',  # Congenital cataract (disorder)
        '204128001',  # Congenital lamellar cataract (disorder)
        '204138006',  # Congenital blue dot cataract (disorder)
        '204139003',  # Congenital membranous cataract (disorder)
        '253223002',  # Congenital polar cataract (disorder)
        '253224008',  # Congenital anterior polar cataract (disorder)
        '253225009',  # Congenital posterior polar cataract (disorder)
        '253226005',  # Congenital sutural cataract (disorder)
        '253227001',  # Rubella cataract (disorder)
        '335841000119103',  # Congenital nuclear cataract of right eye (disorder)
        '341451000119100',  # Congenital nuclear cataract of left eye (disorder)
        '346711000119101',  # Congenital combined form cataract of bilateral eyes (disorder)
        '346731000119106',  # Congenital anterior subcapsular polar cataract of bilateral eyes (disorder)
        '349281000119109',  # Congenital zonular cataract of bilateral eyes (disorder)
        '679911000119104',  # Congenital cortical cataract of left eye (disorder)
        '679931000119109',  # Congenital cortical cataract of right eye (disorder)
        '681041000119106',  # Congenital cortical cataract of bilateral eyes (disorder)
    }


class Cataract_MatureOrHypermature(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of mature or hypermature cataract.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of mature or hypermature cataract, including senile or intumescent cataract.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Cataract, Mature or Hypermature'
    OID = '2.16.840.1.113883.3.526.3.1413'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H26.9',  # Unspecified cataract
    }
    SNOMEDCT = {
        '849000',  # Mature cataract (disorder)
        '8656007',  # Total traumatic cataract (disorder)
        '95724003',  # Intumescent cataract (disorder)
        '193590000',  # Total, mature senile cataract (disorder)
        '264443002',  # Morgagnian cataract (disorder)
        '267626000',  # Hypermature cataract (disorder)
        '217791000119107',  # Hypermature senile cataract (disorder)
        '347461000119107',  # Hypermature senile cataract of right eye (disorder)
        '347521000119103',  # Hypermature senile cataract of left eye (disorder)
        '347581000119104',  # Hypermature senile cataract of bilateral eyes (disorder)
    }


class Cataract_PosteriorPolar(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of posterior polar cataract.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of posterior polar cataract, including capsular or subcapsular cataract.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Cataract, Posterior Polar'
    OID = '2.16.840.1.113883.3.526.3.1414'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H25.041',  # Posterior subcapsular polar age-related cataract, right eye
        'H25.042',  # Posterior subcapsular polar age-related cataract, left eye
        'H25.043',  # Posterior subcapsular polar age-related cataract, bilateral
    }
    SNOMEDCT = {
        '5318001',  # Posterior subcapsular polar senile cataract (disorder)
        '34533008',  # Posterior subcapsular polar cataract (disorder)
        '253225009',  # Congenital posterior polar cataract (disorder)
        '315353005',  # Posterior subcapsular cataract (disorder)
        '335831000119107',  # Congenital posterior subcapsular polar cataract of right eye (disorder)
        '341441000119102',  # Congenital posterior subcapsular polar cataract of left eye (disorder)
        '342821000119103',  # Congenital posterior subcapsular polar cataract (disorder)
        '346691000119104',  # Congenital posterior subcapsular polar cataract of bilateral eyes (disorder)
        '347561000119108',  # Posterior subcapsular polar senile cataract of bilateral eyes (disorder)
        '1078791000119109',  # Posterior subcapsular polar senile cataract of left eye (disorder)
        '1078801000119105',  # Posterior subcapsular polar senile cataract of right eye (disorder)
        '15639401000119105',  # Cataract of posterior subcapsule of right eye (disorder)
        '15639441000119107',  # Cataract of posterior subcapsule of left eye (disorder)
        '15639481000119102',  # Cataract of posterior subcapsule of bilateral eyes (disorder)
        '15737681000119103',  # Anterior subcapsular polar cataract of bilateral eyes (disorder)
        '15737881000119104',  # Posterior subcapsular polar cataract of bilateral eyes (disorder)
        '15737921000119106',  # Posterior subcapsular polar cataract of left eye (disorder)
        '15737961000119101',  # Posterior subcapsular polar cataract of right eye (disorder)
        '16477021000119103',  # Anterior subcapsular polar senile cataract of bilateral eyes (disorder)
    }


class CentralCornealUlcer(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of central corneal ulcer.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of central corneal ulcer or keratitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Central Corneal Ulcer'
    OID = '2.16.840.1.113883.3.526.3.1428'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H16.011',  # Central corneal ulcer, right eye
        'H16.012',  # Central corneal ulcer, left eye
        'H16.013',  # Central corneal ulcer, bilateral
    }
    SNOMEDCT = {
        '7426009',  # Central corneal ulcer (disorder)
        '42513006',  # Punctate keratitis (disorder)
        '95732006',  # Necrotizing keratitis (disorder)
        '95733001',  # Eosinophilic keratitis (disorder)
        '231901007',  # Neuropathic corneal ulcer (disorder)
        '397567009',  # Keratoconjunctivitis caused by molluscum contagiosum (disorder)
        '397570008',  # Herpes simplex stromal keratitis (disorder)
        '332801000119108',  # Punctate keratitis of right eye (disorder)
        '338411000119106',  # Punctate keratitis of left eye (disorder)
        '344181000119103',  # Punctate keratitis of bilateral eyes (disorder)
    }


class CertainTypesOfIridocyclitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of various types of iridocyclitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of lens-induced iridocyclitis, Fuchs' heterochromic cyclitis, Vogt-Koyanagi syndrome, and other types of iridocyclitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Certain Types of Iridocyclitis'
    OID = '2.16.840.1.113883.3.526.3.1415'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H20.21',  # Lens-induced iridocyclitis, right eye
        'H20.22',  # Lens-induced iridocyclitis, left eye
        'H20.23',  # Lens-induced iridocyclitis, bilateral
        'H20.811',  # Fuchs' heterochromic cyclitis, right eye
        'H20.812',  # Fuchs' heterochromic cyclitis, left eye
        'H20.813',  # Fuchs' heterochromic cyclitis, bilateral
        'H20.821',  # Vogt-Koyanagi syndrome, right eye
        'H20.822',  # Vogt-Koyanagi syndrome, left eye
        'H20.823',  # Vogt-Koyanagi syndrome, bilateral
        'H20.9',  # Unspecified iridocyclitis
    }
    SNOMEDCT = {
        '6869001',  # Recurrent iridocyclitis (disorder)
        '9091006',  # Gonococcal iridocyclitis (disorder)
        '10698009',  # Herpes zoster iridocyclitis (disorder)
        '11226001',  # Fuchs' heterochromic cyclitis (disorder)
        '12630008',  # Primary iridocyclitis (disorder)
        '13608004',  # Herpes simplex iridocyclitis (disorder)
        '23798003',  # Translucency of iris (disorder)
        '29538005',  # Glaucomatocyclitic crisis (disorder)
        '30080002',  # Secondary syphilitic iridocyclitis (disorder)
        '46916006',  # Noninfectious secondary iridocyclitis (disorder)
        '70461003',  # Lens-induced iridocyclitis (disorder)
        '77971008',  # Iridocyclitis (disorder)
        '89114005',  # Infectious secondary iridocyclitis (disorder)
        '193487008',  # Secondary infected iridocyclitis (disorder)
        '193488003',  # Secondary non-infected iridocyclitis (disorder)
        '267619000',  # Non-infectious anterior uveitis (disorder)
        '420485005',  # Herpetic iridocyclitis (disorder)
        '15678641000119101',  # Iridocyclitis of right eye caused by human herpes simplex virus (disorder)
        '15678681000119106',  # Iridocyclitis of left eye caused by human herpes simplex virus (disorder)
        '15678761000119105',  # Iridocyclitis of right eye due to herpes zoster (disorder)
        '15678801000119102',  # Iridocyclitis of left eye due to herpes zoster (disorder)
    }


class ChoroidalDegenerations(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of choroidal degenerations.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of degenerations of the choroid, including choroid atrophy and dystrophy, angioid streaks, and choroideremia.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Choroidal Degenerations'
    OID = '2.16.840.1.113883.3.526.3.1450'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H35.33',  # Angioid streaks of macula
    }
    SNOMEDCT = {
        '19871000',  # Total generalized choroidal dystrophy (disorder)
        '24950005',  # Partial generalized choroidal dystrophy (disorder)
        '38513001',  # Senile atrophy of choroid (disorder)
        '41250005',  # Diffuse choroidal sclerosis (disorder)
        '59753003',  # Total circumpapillary dystrophy of choroid (disorder)
        '74469006',  # Hereditary choroidal dystrophy (disorder)
        '75241009',  # Choroideremia (disorder)
        '86103006',  # Angioid streaks of choroid (disorder)
        '92820007',  # Circumpapillary dystrophy of choroid (disorder)
        '193463006',  # Diffuse secondary choroid atrophy (disorder)
        '193466003',  # Partial circumpapillary choroid dystrophy (disorder)
        '231996009',  # Central areolar choroidal sclerosis (disorder)
        '231997000',  # Peripapillary choroidal sclerosis (disorder)
        '312927001',  # Bietti's crystalline retinopathy (disorder)
        '406446000',  # Choroidal degeneration (disorder)
    }


class ChoroidalDetachment(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of choroidal detachment.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis choroidal detachment, or effusion.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Choroidal Detachment'
    OID = '2.16.840.1.113883.3.526.3.1451'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H31.401',  # Unspecified choroidal detachment, right eye
        'H31.402',  # Unspecified choroidal detachment, left eye
        'H31.403',  # Unspecified choroidal detachment, bilateral
        'H31.411',  # Hemorrhagic choroidal detachment, right eye
        'H31.412',  # Hemorrhagic choroidal detachment, left eye
        'H31.413',  # Hemorrhagic choroidal detachment, bilateral
        'H31.421',  # Serous choroidal detachment, right eye
        'H31.422',  # Serous choroidal detachment, left eye
        'H31.423',  # Serous choroidal detachment, bilateral
    }
    SNOMEDCT = {
        '19031009',  # Hemorrhagic choroidal detachment (disorder)
        '27871007',  # Choroidal detachment (disorder)
        '78584008',  # Serous choroidal detachment (disorder)
        '232000008',  # Choroidal effusion (disorder)
        '247198004',  # Annular choroidal detachment (disorder)
        '278805002',  # Uveal effusion syndrome (disorder)
        '778049009',  # Idiopathic uveal effusion syndrome (disorder)
        '332111000119102',  # Serous choroidal detachment of right eye (disorder)
        '333881000119109',  # Minor opacity of right cornea (disorder)
        '334901000119109',  # Hemorrhagic detachment of right choroid (disorder)
        '336051000119104',  # Rupture of choroid of right eye (disorder)
        '336061000119102',  # Hemorrhage of choroid of right eye (disorder)
        '336071000119108',  # Choroidal detachment of right eye (disorder)
        '337721000119107',  # Serous choroidal detachment of left eye (disorder)
        '339481000119109',  # Minor opacity of left cornea (disorder)
        '340511000119109',  # Hemorrhagic detachment of left choroid (disorder)
        '341661000119101',  # Rupture of choroid of left eye (disorder)
        '341671000119107',  # Hemorrhage of choroid of left eye (disorder)
        '341681000119105',  # Choroidal detachment of left eye (disorder)
        '343491000119109',  # Serous choroidal detachment of bilateral eyes (disorder)
        '344611000119101',  # Peripheral opacity of bilateral corneas (disorder)
    }


class ChoroidalHemorrhageAndRupture(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of choroidal hemorrhage and/or rupture.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of choroidal hemorrhage and rupture.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Choroidal Hemorrhage and Rupture'
    OID = '2.16.840.1.113883.3.526.3.1452'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H31.301',  # Unspecified choroidal hemorrhage, right eye
        'H31.302',  # Unspecified choroidal hemorrhage, left eye
        'H31.303',  # Unspecified choroidal hemorrhage, bilateral
        'H31.311',  # Expulsive choroidal hemorrhage, right eye
        'H31.312',  # Expulsive choroidal hemorrhage, left eye
        'H31.313',  # Expulsive choroidal hemorrhage, bilateral
        'H31.321',  # Choroidal rupture, right eye
        'H31.322',  # Choroidal rupture, left eye
        'H31.323',  # Choroidal rupture, bilateral
    }
    SNOMEDCT = {
        '122003',  # Choroidal hemorrhage (disorder)
        '2532009',  # Choroidal rupture (disorder)
        '11623000',  # Expulsive choroidal hemorrhage (disorder)
        '193474002',  # Choroidal hemorrhage and rupture (disorder)
        '419596007',  # Suprachoroidal hemorrhage (disorder)
        '336051000119104',  # Rupture of choroid of right eye (disorder)
        '336061000119102',  # Hemorrhage of choroid of right eye (disorder)
        '341661000119101',  # Rupture of choroid of left eye (disorder)
        '341671000119107',  # Hemorrhage of choroid of left eye (disorder)
    }


class ChronicIridocyclitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of chronic iridocyclitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of chronic or recurrent iridocyclitis or tuberculous iridocyclitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Chronic Iridocyclitis'
    OID = '2.16.840.1.113883.3.526.3.1416'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'A18.54',  # Tuberculous iridocyclitis
        'H20.11',  # Chronic iridocyclitis, right eye
        'H20.12',  # Chronic iridocyclitis, left eye
        'H20.13',  # Chronic iridocyclitis, bilateral
        'H20.9',  # Unspecified iridocyclitis
    }
    SNOMEDCT = {
        '6869001',  # Recurrent iridocyclitis (disorder)
        '186266008',  # Tuberculous chronic iridocyclitis (disorder)
        '193493000',  # Chronic iridocyclitis due to another disorder (disorder)
        '416449002',  # Chronic iridocyclitis in young girls (disorder)
    }


class CloudyCornea(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of cloudy cornea.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of opacity of the cornea, adherent or corneal leukoma, and haziness of the cornea.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Cloudy Cornea'
    OID = '2.16.840.1.113883.3.526.3.1417'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H17.01',  # Adherent leukoma, right eye
        'H17.02',  # Adherent leukoma, left eye
        'H17.03',  # Adherent leukoma, bilateral
        'H17.11',  # Central corneal opacity, right eye
        'H17.12',  # Central corneal opacity, left eye
        'H17.13',  # Central corneal opacity, bilateral
        'H17.811',  # Minor opacity of cornea, right eye
        'H17.812',  # Minor opacity of cornea, left eye
        'H17.813',  # Minor opacity of cornea, bilateral
        'H17.821',  # Peripheral opacity of cornea, right eye
        'H17.822',  # Peripheral opacity of cornea, left eye
        'H17.823',  # Peripheral opacity of cornea, bilateral
    }
    SNOMEDCT = {
        '22135003',  # Corneal leukoma interfering with central vision (disorder)
        '30337003',  # Adherent leukoma (disorder)
        '54359000',  # Congenital corneal opacity not interfering with vision (disorder)
        '55713007',  # Peripheral opacity of cornea (disorder)
        '56530002',  # Central opacity of cornea (disorder)
        '64634000',  # Corneal opacity (disorder)
        '66104007',  # Minor opacity of cornea (disorder)
        '72494008',  # Corneal nebula (disorder)
        '204149000',  # Congenital corneal opacity without visual deficit (disorder)
        '413921009',  # Corneal subepithelial haze (disorder)
        '424353002',  # Corneal haze caused by herpes zoster (disorder)
        '425961009',  # Corneal haze caused by herpes simplex (disorder)
        '333881000119109',  # Minor opacity of right cornea (disorder)
        '336501000119105',  # Adherent leukoma of right eye (disorder)
        '339481000119109',  # Minor opacity of left cornea (disorder)
        '342111000119101',  # Adherent leukoma of left eye (disorder)
        '344611000119101',  # Peripheral opacity of bilateral corneas (disorder)
        '347271000119100',  # Adherent leukoma of bilateral eyes (disorder)
        '1099031000119105',  # Opacity of cornea of bilateral eyes (disorder)
        '1099041000119101',  # Opacity of cornea of left eye (disorder)
        '1099051000119104',  # Opacity of cornea of right eye (disorder)
    }


class CornealEdema(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of corneal edema.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of corneal edema, including keratopathy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Corneal Edema'
    OID = '2.16.840.1.113883.3.526.3.1418'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H18.11',  # Bullous keratopathy, right eye
        'H18.12',  # Bullous keratopathy, left eye
        'H18.13',  # Bullous keratopathy, bilateral
        'H18.20',  # Unspecified corneal edema
        'H18.221',  # Idiopathic corneal edema, right eye
        'H18.222',  # Idiopathic corneal edema, left eye
        'H18.223',  # Idiopathic corneal edema, bilateral
        'H18.231',  # Secondary corneal edema, right eye
        'H18.232',  # Secondary corneal edema, left eye
        'H18.233',  # Secondary corneal edema, bilateral
        'H18.421',  # Band keratopathy, right eye
        'H18.422',  # Band keratopathy, left eye
        'H18.423',  # Band keratopathy, bilateral
        'H18.43',  # Other calcerous corneal degeneration
    }
    SNOMEDCT = {
        '1794009',  # Idiopathic corneal edema (disorder)
        '27194006',  # Corneal edema (disorder)
        '27536004',  # Secondary corneal edema (disorder)
        '49362009',  # Corneal edema caused by wearing of contact lenses (disorder)
        '57207003',  # Bullous keratopathy (disorder)
        '129627002',  # Keratitis bullosa (disorder)
        '231921008',  # Phakic corneal edema (disorder)
        '231922001',  # Pseudophakic corneal edema (disorder)
        '231923006',  # Aphakic corneal edema (disorder)
        '248503005',  # Fine corneal edema (disorder)
        '373428006',  # Corneal epithelial edema (finding)
        '373429003',  # Bullous corneal epithelial edema (finding)
        '373430008',  # Edema of corneal stroma (disorder)
        '336241000119105',  # Bullous keratopathy of right eye (disorder)
        '341851000119102',  # Bullous keratopathy of left eye (disorder)
        '347061000119104',  # Bullous keratopathy of bilateral eyes (disorder)
        '678991000119103',  # Pseudophakic bullous keratopathy of left eye (disorder)
        '679011000119100',  # Pseudophakic bullous keratopathy of right eye (disorder)
        '15736081000119108',  # Microcystic edema of cornea of left eye (disorder)
        '15736161000119100',  # Microcystic edema of cornea of right eye (disorder)
        '15736201000119105',  # Edema of cornea of left eye (disorder)
        '15736241000119107',  # Edema of cornea of bilateral eyes (disorder)
        '15736281000119102',  # Edema of cornea of right eye (disorder)
    }


class CornealOpacityAndOtherDisordersOfCornea(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of corneal opacity and other corneal disorders.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of opacity of the cornea, adherent or corneal leukoma, scarring of the cornea, and corneal haze.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Corneal Opacity and Other Disorders of Cornea'
    OID = '2.16.840.1.113883.3.526.3.1419'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H17.01',  # Adherent leukoma, right eye
        'H17.02',  # Adherent leukoma, left eye
        'H17.03',  # Adherent leukoma, bilateral
        'H17.11',  # Central corneal opacity, right eye
        'H17.12',  # Central corneal opacity, left eye
        'H17.13',  # Central corneal opacity, bilateral
        'H17.89',  # Other corneal scars and opacities
        'H17.9',  # Unspecified corneal scar and opacity
    }
    SNOMEDCT = {
        '13282008',  # Congenital corneal opacity interfering with vision (disorder)
        '16350000',  # Corneal macula interfering with central vision (disorder)
        '22135003',  # Corneal leukoma interfering with central vision (disorder)
        '30337003',  # Adherent leukoma (disorder)
        '54359000',  # Congenital corneal opacity not interfering with vision (disorder)
        '55713007',  # Peripheral opacity of cornea (disorder)
        '56530002',  # Central opacity of cornea (disorder)
        '64634000',  # Corneal opacity (disorder)
        '66104007',  # Minor opacity of cornea (disorder)
        '72494008',  # Corneal nebula (disorder)
        '95479005',  # Congenital sclerocornea (disorder)
        '95726001',  # Corneal scar (disorder)
        '193795008',  # Corneal scars and opacities (disorder)
        '204148008',  # Congenital corneal opacity with visual deficit (disorder)
        '204149000',  # Congenital corneal opacity without visual deficit (disorder)
        '268160006',  # Congenital corneal opacity (disorder)
        '314005007',  # Visually significant corneal scar (disorder)
        '370505008',  # Corneal facet (disorder)
        '413921009',  # Corneal subepithelial haze (disorder)
        '419496009',  # Congenital corneal keloid (disorder)
        '422947009',  # Corneal subepithelial scar (finding)
        '424353002',  # Corneal haze caused by herpes zoster (disorder)
        '425961009',  # Corneal haze caused by herpes simplex (disorder)
        '336501000119105',  # Adherent leukoma of right eye (disorder)
        '342111000119101',  # Adherent leukoma of left eye (disorder)
        '347271000119100',  # Adherent leukoma of bilateral eyes (disorder)
        '1099031000119105',  # Opacity of cornea of bilateral eyes (disorder)
        '1099041000119101',  # Opacity of cornea of left eye (disorder)
        '1099051000119104',  # Opacity of cornea of right eye (disorder)
        '15999271000119108',  # Scar of cornea of right eye (disorder)
        '15999311000119108',  # Scar of cornea of bilateral eyes (disorder)
        '15999351000119109',  # Scar of cornea of left eye (disorder)
    }


class DegenerationOfMaculaAndPosteriorPole(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of degeneration of macula and posterior pole.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of macula and/or posterior pole degenerations, including age-related degenerations, macular cyst, maculopathy, macular puckering, macular hole, atrophy or dystrophy of macula.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Degeneration of Macula and Posterior Pole'
    OID = '2.16.840.1.113883.3.526.3.1453'
    DEFINITION_VERSION = '20210210'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H35.30',  # Unspecified macular degeneration
        'H35.3110',  # Nonexudative age-related macular degeneration, right eye, stage unspecified
        'H35.3111',  # Nonexudative age-related macular degeneration, right eye, early dry stage
        'H35.3112',  # Nonexudative age-related macular degeneration, right eye, intermediate dry stage
        'H35.3113',  # Nonexudative age-related macular degeneration, right eye, advanced atrophic without subfoveal involvement
        'H35.3114',  # Nonexudative age-related macular degeneration, right eye, advanced atrophic with subfoveal involvement
        'H35.3120',  # Nonexudative age-related macular degeneration, left eye, stage unspecified
        'H35.3121',  # Nonexudative age-related macular degeneration, left eye, early dry stage
        'H35.3122',  # Nonexudative age-related macular degeneration, left eye, intermediate dry stage
        'H35.3123',  # Nonexudative age-related macular degeneration, left eye, advanced atrophic without subfoveal involvement
        'H35.3124',  # Nonexudative age-related macular degeneration, left eye, advanced atrophic with subfoveal involvement
        'H35.3130',  # Nonexudative age-related macular degeneration, bilateral, stage unspecified
        'H35.3131',  # Nonexudative age-related macular degeneration, bilateral, early dry stage
        'H35.3132',  # Nonexudative age-related macular degeneration, bilateral, intermediate dry stage
        'H35.3133',  # Nonexudative age-related macular degeneration, bilateral, advanced atrophic without subfoveal involvement
        'H35.3134',  # Nonexudative age-related macular degeneration, bilateral, advanced atrophic with subfoveal involvement
        'H35.3210',  # Exudative age-related macular degeneration, right eye, stage unspecified
        'H35.3211',  # Exudative age-related macular degeneration, right eye, with active choroidal neovascularization
        'H35.3212',  # Exudative age-related macular degeneration, right eye, with inactive choroidal neovascularization
        'H35.3213',  # Exudative age-related macular degeneration, right eye, with inactive scar
        'H35.3220',  # Exudative age-related macular degeneration, left eye, stage unspecified
        'H35.3221',  # Exudative age-related macular degeneration, left eye, with active choroidal neovascularization
        'H35.3222',  # Exudative age-related macular degeneration, left eye, with inactive choroidal neovascularization
        'H35.3223',  # Exudative age-related macular degeneration, left eye, with inactive scar
        'H35.3230',  # Exudative age-related macular degeneration, bilateral, stage unspecified
        'H35.3231',  # Exudative age-related macular degeneration, bilateral, with active choroidal neovascularization
        'H35.3232',  # Exudative age-related macular degeneration, bilateral, with inactive choroidal neovascularization
        'H35.3233',  # Exudative age-related macular degeneration, bilateral, with inactive scar
        'H35.341',  # Macular cyst, hole, or pseudohole, right eye
        'H35.342',  # Macular cyst, hole, or pseudohole, left eye
        'H35.343',  # Macular cyst, hole, or pseudohole, bilateral
        'H35.351',  # Cystoid macular degeneration, right eye
        'H35.352',  # Cystoid macular degeneration, left eye
        'H35.353',  # Cystoid macular degeneration, bilateral
        'H35.361',  # Drusen (degenerative) of macula, right eye
        'H35.362',  # Drusen (degenerative) of macula, left eye
        'H35.363',  # Drusen (degenerative) of macula, bilateral
        'H35.371',  # Puckering of macula, right eye
        'H35.372',  # Puckering of macula, left eye
        'H35.373',  # Puckering of macula, bilateral
        'H35.381',  # Toxic maculopathy, right eye
        'H35.382',  # Toxic maculopathy, left eye
        'H35.383',  # Toxic maculopathy, bilateral
    }
    SNOMEDCT = {
        '14046000',  # Cystoid macular retinal degeneration (disorder)
        '18410006',  # Scarred macula (disorder)
        '37231002',  # Macular retinal edema (disorder)
        '44115007',  # Toxic maculopathy (disorder)
        '47673003',  # Fundus flavimaculatus (disorder)
        '70099003',  # Stargardt's disease (disorder)
        '71047006',  # Familial pseudoinflammatory macular degeneration (disorder)
        '90036004',  # Vitelliform dystrophy (disorder)
        '193350004',  # Advanced maculopathy due to diabetes mellitus (disorder)
        '193387007',  # Cystoid macular edema (disorder)
        '204179007',  # Congenital macular changes (disorder)
        '232006002',  # Macular hole (disorder)
        '232013002',  # Macular retinoschisis (disorder)
        '232020009',  # Disorder of macula due to diabetes mellitus (disorder)
        '232049001',  # Adult vitelliform macular dystrophy (disorder)
        '232050001',  # Bull's eye macular dystrophy (disorder)
        '232051002',  # Pattern dystrophy of macula (disorder)
        '232071006',  # Occult neovascularization of macula (disorder)
        '232072004',  # Forster-Fuchs' spot (disorder)
        '247146001',  # Macular pigment deposit (disorder)
        '247147005',  # Macular diffuse atrophy (disorder)
        '247149008',  # Macular choroidal sclerosis (disorder)
        '247158001',  # Macular pigment epithelial detachment (disorder)
        '247159009',  # Macular pigment epithelial rip (disorder)
        '247172005',  # Retinal macular break (disorder)
        '247200005',  # Choroidal atrophy (finding)
        '264633009',  # Drusen stage macular degeneration (disorder)
        '267611002',  # Degeneration of macula and posterior pole (disorder)
        '267612009',  # Degeneration of macula due to cyst, hole or pseudohole (disorder)
        '267718000',  # Age-related macular degeneration (disorder)
        '276436007',  # Hereditary macular dystrophy (disorder)
        '312701008',  # Impending macular hole (disorder)
        '312702001',  # Full thickness macular hole stage II (disorder)
        '312703006',  # Full thickness macular hole stage III (disorder)
        '312704000',  # Full thickness macular hole stage IV (disorder)
        '312898002',  # Myopic macular degeneration (disorder)
        '312899005',  # Fibrovascular macular scar (disorder)
        '312901001',  # Vitreomacular traction syndrome (disorder)
        '312902008',  # Macular pseudohole (disorder)
        '312911008',  # Clinically significant macular edema (disorder)
        '312912001',  # Macular edema due to diabetes mellitus (disorder)
        '312913006',  # Ischemic maculopathy (disorder)
        '312920004',  # Postoperative cystoid macular edema (disorder)
        '312921000',  # Autosomal dominant cystoid macular edema (disorder)
        '312922007',  # Uveitis related cystoid macular edema (disorder)
        '312925009',  # North Carolina macular dystrophy (disorder)
        '312999006',  # Disorder of macula of retina (disorder)
        '313002004',  # Acute macular neuroretinitis (disorder)
        '314010006',  # Diffuse exudative maculopathy due to diabetes mellitus (disorder)
        '314011005',  # Focal exudative maculopathy due to diabetes mellitus (disorder)
        '314014002',  # Ischemic maculopathy due to diabetes mellitus (disorder)
        '314015001',  # Mixed maculopathy due to diabetes mellitus (disorder)
        '314427006',  # Drusen plus pigment change stage macular degeneration (disorder)
        '371098002',  # Post-radiation maculopathy (disorder)
        '399864000',  # Macular edema not clinically significant due to diabetes mellitus (disorder)
        '399872003',  # Severe nonproliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        '399874002',  # High risk proliferative retinopathy with clinically significant macula edema due to diabetes mellitus (disorder)
        '399875001',  # Non-high-risk proliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        '399877009',  # Very severe nonproliferative retinopathy with clinically significant macular edema due to diabetes mellitus (disorder)
        '414173003',  # Exudative age-related macular degeneration (disorder)
        '414875008',  # Nonexudative age-related macular degeneration (disorder)
        '418918009',  # Macular infarction (disorder)
        '420030000',  # Serous macular detachment (disorder)
        '420486006',  # Exudative maculopathy due to type 1 diabetes mellitus (disorder)
        '421779007',  # Exudative maculopathy due to type 2 diabetes mellitus (disorder)
        '422338006',  # Degenerative disorder of macula (disorder)
        '423084001',  # Macular exudate (disorder)
        '425622004',  # Scar of macula following trauma (disorder)
        '429449002',  # Congenital hypoplasia of fovea centralis (disorder)
        '432789001',  # Noncystoid edema of macula of retina (disorder)
        '443437004',  # Full thickness hole of macula lutea (disorder)
        '789220007',  # Atrophy of macula lutea (disorder)
        '335531000119109',  # Cystoid macular degeneration of right retina (disorder)
        '341141000119106',  # Cystoid macular degeneration of left retina (disorder)
        '346401000119108',  # Cystoid macular degeneration of bilateral retinas (disorder)
    }


class DegenerativeDisordersOfGlobe(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of degenerative disorders of globe.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of degenerative disorders of globe, including degenerative myopia, chalcosis, siderosis, and other degenerative diagnoses.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Degenerative Disorders of Globe'
    OID = '2.16.840.1.113883.3.526.3.1454'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H44.21',  # Degenerative myopia, right eye
        'H44.22',  # Degenerative myopia, left eye
        'H44.23',  # Degenerative myopia, bilateral
        'H44.2A1',  # Degenerative myopia with choroidal neovascularization, right eye
        'H44.2A2',  # Degenerative myopia with choroidal neovascularization, left eye
        'H44.2A3',  # Degenerative myopia with choroidal neovascularization, bilateral eye
        'H44.2B1',  # Degenerative myopia with macular hole, right eye
        'H44.2B2',  # Degenerative myopia with macular hole, left eye
        'H44.2B3',  # Degenerative myopia with macular hole, bilateral eye
        'H44.2C1',  # Degenerative myopia with retinal detachment, right eye
        'H44.2C2',  # Degenerative myopia with retinal detachment, left eye
        'H44.2C3',  # Degenerative myopia with retinal detachment, bilateral eye
        'H44.2D1',  # Degenerative myopia with foveoschisis, right eye
        'H44.2D2',  # Degenerative myopia with foveoschisis, left eye
        'H44.2D3',  # Degenerative myopia with foveoschisis, bilateral eye
        'H44.2E1',  # Degenerative myopia with other maculopathy, right eye
        'H44.2E2',  # Degenerative myopia with other maculopathy, left eye
        'H44.2E3',  # Degenerative myopia with other maculopathy, bilateral eye
        'H44.311',  # Chalcosis, right eye
        'H44.312',  # Chalcosis, left eye
        'H44.313',  # Chalcosis, bilateral
        'H44.321',  # Siderosis of eye, right eye
        'H44.322',  # Siderosis of eye, left eye
        'H44.323',  # Siderosis of eye, bilateral
        'H44.391',  # Other degenerative disorders of globe, right eye
        'H44.392',  # Other degenerative disorders of globe, left eye
        'H44.393',  # Other degenerative disorders of globe, bilateral
    }
    SNOMEDCT = {
        '264008',  # Blind hypertensive eye (disorder)
        '3577000',  # Retinal lattice degeneration (disorder)
        '14046000',  # Cystoid macular retinal degeneration (disorder)
        '25277000',  # Siderosis of eye (disorder)
        '35545005',  # Secondary pigmentary retinal degeneration (disorder)
        '37075008',  # Microcystoid retinal degeneration (disorder)
        '40787005',  # Conjunctival degeneration (disorder)
        '54184008',  # Senile reticular retinal degeneration (disorder)
        '60189009',  # Vitreous degeneration (disorder)
        '61536007',  # Peripheral retinal degeneration (disorder)
        '62585004',  # Degenerative disorder of eye (disorder)
        '69134001',  # Paving stone retinal degeneration (disorder)
        '71047006',  # Familial pseudoinflammatory macular degeneration (disorder)
        '95695004',  # Degeneration of retina (disorder)
        '95696003',  # Glaucomatous retinal degeneration (disorder)
        '95697007',  # Generalized retinal degeneration (disorder)
        '95699005',  # Multifocal retinal degeneration (disorder)
        '95700006',  # Sudden acquired retinal degeneration (disorder)
        '111521006',  # Corneal degeneration (disorder)
        '231918006',  # Chalcosis of cornea (disorder)
        '231919003',  # Siderosis of cornea (disorder)
        '232001007',  # Snail-track retinal degeneration (disorder)
        '232002000',  # Snowflake retinal degeneration (disorder)
        '246877005',  # Bulbar conjunctival degeneration (disorder)
        '247177004',  # Chorioretinal degeneration (disorder)
        '247182006',  # Vitreoretinal degeneration (disorder)
        '247183001',  # Peripheral snowflake retinal degeneration (disorder)
        '264633009',  # Drusen stage macular degeneration (disorder)
        '267611002',  # Degeneration of macula and posterior pole (disorder)
        '267612009',  # Degeneration of macula due to cyst, hole or pseudohole (disorder)
        '267718000',  # Age-related macular degeneration (disorder)
        '314427006',  # Drusen plus pigment change stage macular degeneration (disorder)
        '414173003',  # Exudative age-related macular degeneration (disorder)
        '414875008',  # Nonexudative age-related macular degeneration (disorder)
        '417939004',  # Amyloid corneal degeneration (disorder)
        '418871005',  # Spheroidal degeneration of conjunctiva (disorder)
        '428212003',  # Photoreceptor degeneration (disorder)
    }


class DiabeticMacularEdema(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of diabetic macular edema.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of diabetic macular edema or diabetic retinopathy with macular edema.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Diabetic Macular Edema'
    OID = '2.16.840.1.113883.3.526.3.1455'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'E08.311',  # Diabetes mellitus due to underlying condition with unspecified diabetic retinopathy with macular edema
        'E08.3211',  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E08.3212',  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E08.3213',  # Diabetes mellitus due to underlying condition with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E08.3311',  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E08.3312',  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E08.3313',  # Diabetes mellitus due to underlying condition with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E08.3411',  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E08.3412',  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E08.3413',  # Diabetes mellitus due to underlying condition with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E08.3511',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, right eye
        'E08.3512',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, left eye
        'E08.3513',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with macular edema, bilateral
        'E08.3521',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E08.3522',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E08.3523',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E08.3531',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E08.3532',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E08.3533',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E08.3541',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E08.3542',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E08.3543',  # Diabetes mellitus due to underlying condition with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E08.3551',  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, right eye
        'E08.3552',  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, left eye
        'E08.3553',  # Diabetes mellitus due to underlying condition with stable proliferative diabetic retinopathy, bilateral
        'E08.37X1',  # Diabetes mellitus due to underlying condition with diabetic macular edema, resolved following treatment, right eye
        'E08.37X2',  # Diabetes mellitus due to underlying condition with diabetic macular edema, resolved following treatment, left eye
        'E08.37X3',  # Diabetes mellitus due to underlying condition with diabetic macular edema, resolved following treatment, bilateral
        'E09.311',  # Drug or chemical induced diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E09.3211',  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E09.3212',  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E09.3213',  # Drug or chemical induced diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E09.3311',  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E09.3312',  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E09.3313',  # Drug or chemical induced diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E09.3411',  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E09.3412',  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E09.3413',  # Drug or chemical induced diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E09.3511',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E09.3512',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E09.3513',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E09.3521',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E09.3522',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E09.3523',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E09.3531',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E09.3532',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E09.3533',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E09.3541',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E09.3542',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E09.3543',  # Drug or chemical induced diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E09.3551',  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E09.3552',  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E09.3553',  # Drug or chemical induced diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E09.37X1',  # Drug or chemical induced diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        'E09.37X2',  # Drug or chemical induced diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        'E09.37X3',  # Drug or chemical induced diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        'E10.311',  # Type 1 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E10.3211',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E10.3212',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E10.3213',  # Type 1 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E10.3311',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E10.3312',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E10.3313',  # Type 1 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E10.3411',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E10.3412',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E10.3413',  # Type 1 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E10.3511',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E10.3512',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E10.3513',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E10.3521',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E10.3522',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E10.3523',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E10.3531',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E10.3532',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E10.3533',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E10.3541',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E10.3542',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E10.3543',  # Type 1 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E10.3551',  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E10.3552',  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E10.3553',  # Type 1 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E10.37X1',  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        'E10.37X2',  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        'E10.37X3',  # Type 1 diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        'E11.311',  # Type 2 diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E11.3211',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E11.3212',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E11.3213',  # Type 2 diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E11.3311',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E11.3312',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E11.3313',  # Type 2 diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E11.3411',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E11.3412',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E11.3413',  # Type 2 diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E11.3511',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E11.3512',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E11.3513',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E11.3521',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E11.3522',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E11.3523',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E11.3531',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E11.3532',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E11.3533',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E11.3541',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E11.3542',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E11.3543',  # Type 2 diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E11.3551',  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E11.3552',  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E11.3553',  # Type 2 diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E11.37X1',  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        'E11.37X2',  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        'E11.37X3',  # Type 2 diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
        'E13.311',  # Other specified diabetes mellitus with unspecified diabetic retinopathy with macular edema
        'E13.3211',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, right eye
        'E13.3212',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, left eye
        'E13.3213',  # Other specified diabetes mellitus with mild nonproliferative diabetic retinopathy with macular edema, bilateral
        'E13.3311',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, right eye
        'E13.3312',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, left eye
        'E13.3313',  # Other specified diabetes mellitus with moderate nonproliferative diabetic retinopathy with macular edema, bilateral
        'E13.3411',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, right eye
        'E13.3412',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, left eye
        'E13.3413',  # Other specified diabetes mellitus with severe nonproliferative diabetic retinopathy with macular edema, bilateral
        'E13.3511',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, right eye
        'E13.3512',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, left eye
        'E13.3513',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with macular edema, bilateral
        'E13.3521',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, right eye
        'E13.3522',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, left eye
        'E13.3523',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment involving the macula, bilateral
        'E13.3531',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, right eye
        'E13.3532',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, left eye
        'E13.3533',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with traction retinal detachment not involving the macula, bilateral
        'E13.3541',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, right eye
        'E13.3542',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, left eye
        'E13.3543',  # Other specified diabetes mellitus with proliferative diabetic retinopathy with combined traction retinal detachment and rhegmatogenous retinal detachment, bilateral
        'E13.3551',  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, right eye
        'E13.3552',  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, left eye
        'E13.3553',  # Other specified diabetes mellitus with stable proliferative diabetic retinopathy, bilateral
        'E13.37X1',  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, right eye
        'E13.37X2',  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, left eye
        'E13.37X3',  # Other specified diabetes mellitus with diabetic macular edema, resolved following treatment, bilateral
    }
    SNOMEDCT = {
        '312912001',  # Macular edema due to diabetes mellitus (disorder)
        '399864000',  # Macular edema not clinically significant due to diabetes mellitus (disorder)
        '769219006',  # Macular edema due to type 1 diabetes mellitus (disorder)
        '769220000',  # Macular edema due to type 2 diabetes mellitus (disorder)
    }


class DisordersOfOpticChiasm(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of disorders of optic chiasm.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of optic chiasm disorders or injuries or optic glioma.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Disorders of Optic Chiasm'
    OID = '2.16.840.1.113883.3.526.3.1457'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H47.41',  # Disorders of optic chiasm in (due to) inflammatory disorders
        'H47.42',  # Disorders of optic chiasm in (due to) neoplasm
        'H47.43',  # Disorders of optic chiasm in (due to) vascular disorders
        'H47.49',  # Disorders of optic chiasm in (due to) other disorders
    }
    SNOMEDCT = {
        '64246009',  # Disorder of optic chiasm associated with non-pituitary neoplasm (disorder)
        '69820004',  # Injury of optic chiasm (disorder)
        '70476006',  # Optic chiasm disorder (disorder)
        '89654006',  # Disorder of optic chiasm associated with vascular disorder (disorder)
        '194057002',  # Disorder of optic chiasm due to pituitary disorder (disorder)
        '194058007',  # Disorder of optic chiasm due to non-pituitary neoplasm (disorder)
        '194059004',  # Disorder of optic chiasm due to vascular disorder (disorder)
        '194060009',  # Inflammatory disorder of optic chiasm (disorder)
        '230517004',  # Compression of optic chiasm (disorder)
        '302902003',  # Infarction of optic chiasm (disorder)
        '404653000',  # Chiasmal glioma (disorder)
        '404664002',  # Malignant optic glioma (disorder)
        '431769004',  # Disorder of optic chiasm due to neoplasm (disorder)
        '703429003',  # Malignant optic glioma of adulthood (disorder)
    }


class DisordersOfVisualCortex(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of disorders of visual cortex.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a disorder of visual cortex, including inflammatory disorders, neoplasm, and cortical blindness.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Disorders of Visual Cortex'
    OID = '2.16.840.1.113883.3.526.3.1458'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H47.611',  # Cortical blindness, right side of brain
        'H47.612',  # Cortical blindness, left side of brain
        'H47.621',  # Disorders of visual cortex in (due to) inflammatory disorders, right side of brain
        'H47.622',  # Disorders of visual cortex in (due to) inflammatory disorders, left side of brain
        'H47.631',  # Disorders of visual cortex in (due to) neoplasm, right side of brain
        'H47.632',  # Disorders of visual cortex in (due to) neoplasm, left side of brain
        'H47.641',  # Disorders of visual cortex in (due to) vascular disorders, right side of brain
        'H47.642',  # Disorders of visual cortex in (due to) vascular disorders, left side of brain
    }
    SNOMEDCT = {
        '68574006',  # Cortical blindness (disorder)
        '128329001',  # Disorder of visual cortex (disorder)
        '194068002',  # Visual cortex disorder due to neoplasm (disorder)
        '194069005',  # Visual cortex disorder due to vascular disorder (disorder)
        '194070006',  # Inflammatory disorder of visual cortex (disorder)
        '230526001',  # Compression of visual cortex (disorder)
        '302904002',  # Infarction of visual cortex (disorder)
        '732251003',  # Cortical blindness, intellectual disability, polydactyly syndrome (disorder)
        '342741000119103',  # Cortical blindness of right side of brain (disorder)
        '342751000119101',  # Cortical blindness of left side of brain (disorder)
    }


class DisseminatedChorioretinitisAndDisseminatedRetinochoroiditis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of disseminated chorioretinitis and/or retinochoroiditis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of disseminated chorioretinal inflammation, disseminated retinochoroiditis, and placoid pigment epitheliopathy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Disseminated Chorioretinitis and Disseminated Retinochoroiditis'
    OID = '2.16.840.1.113883.3.526.3.1459'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'A18.53',  # Tuberculous chorioretinitis
        'H30.101',  # Unspecified disseminated chorioretinal inflammation, right eye
        'H30.102',  # Unspecified disseminated chorioretinal inflammation, left eye
        'H30.103',  # Unspecified disseminated chorioretinal inflammation, bilateral
        'H30.111',  # Disseminated chorioretinal inflammation of posterior pole, right eye
        'H30.112',  # Disseminated chorioretinal inflammation of posterior pole, left eye
        'H30.113',  # Disseminated chorioretinal inflammation of posterior pole, bilateral
        'H30.121',  # Disseminated chorioretinal inflammation, peripheral right eye
        'H30.122',  # Disseminated chorioretinal inflammation, peripheral, left eye
        'H30.123',  # Disseminated chorioretinal inflammation, peripheral, bilateral
        'H30.131',  # Disseminated chorioretinal inflammation, generalized, right eye
        'H30.132',  # Disseminated chorioretinal inflammation, generalized, left eye
        'H30.133',  # Disseminated chorioretinal inflammation, generalized, bilateral
        'H30.141',  # Acute posterior multifocal placoid pigment epitheliopathy, right eye
        'H30.142',  # Acute posterior multifocal placoid pigment epitheliopathy, left eye
        'H30.143',  # Acute posterior multifocal placoid pigment epitheliopathy, bilateral
    }
    SNOMEDCT = {
        '14894002',  # Generalized disseminated choroiditis AND chorioretinitis (disorder)
        '48225000',  # Tuberculous disseminated chorioretinitis (disorder)
        '51579006',  # Disseminated choroiditis AND chorioretinitis, posterior pole (disorder)
        '61517002',  # Metastatic disseminated retinitis AND retinochoroiditis (disorder)
        '69811008',  # Peripheral disseminated choroiditis AND chorioretinitis (disorder)
        '77939001',  # Syphilitic disseminated retinochoroiditis (disorder)
        '78769001',  # Disseminated chorioretinitis (disorder)
        '193442009',  # Disseminated posterior pole chorioretinitis (disorder)
        '193443004',  # Disseminated peripheral chorioretinitis (disorder)
        '193444005',  # General disseminated chorioretinitis (disorder)
        '371099005',  # Generalized disseminated choroiditis AND chorioretinitis with acute necrosis (disorder)
    }


class FocalChorioretinitisAndFocalRetinochoroiditis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of focal chorioretinitis and/or retinochoroiditis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of focal chorioretinitis, choroiditis, retinochoroiditis and retinitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Focal Chorioretinitis and Focal Retinochoroiditis'
    OID = '2.16.840.1.113883.3.526.3.1460'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H30.001',  # Unspecified focal chorioretinal inflammation, right eye
        'H30.002',  # Unspecified focal chorioretinal inflammation, left eye
        'H30.003',  # Unspecified focal chorioretinal inflammation, bilateral
        'H30.011',  # Focal chorioretinal inflammation, juxtapapillary, right eye
        'H30.012',  # Focal chorioretinal inflammation, juxtapapillary, left eye
        'H30.013',  # Focal chorioretinal inflammation, juxtapapillary, bilateral
        'H30.021',  # Focal chorioretinal inflammation of posterior pole, right eye
        'H30.022',  # Focal chorioretinal inflammation of posterior pole, left eye
        'H30.023',  # Focal chorioretinal inflammation of posterior pole, bilateral
        'H30.031',  # Focal chorioretinal inflammation, peripheral, right eye
        'H30.032',  # Focal chorioretinal inflammation, peripheral, left eye
        'H30.033',  # Focal chorioretinal inflammation, peripheral, bilateral
        'H30.041',  # Focal chorioretinal inflammation, macular or paramacular, right eye
        'H30.042',  # Focal chorioretinal inflammation, macular or paramacular, left eye
        'H30.043',  # Focal chorioretinal inflammation, macular or paramacular, bilateral
    }
    SNOMEDCT = {
        '6313001',  # Juxtapapillary focal choroiditis AND chorioretinitis (disorder)
        '15847003',  # Focal chorioretinitis (disorder)
        '36535003',  # Peripheral focal retinitis AND retinochoroiditis (disorder)
        '42031001',  # Macular focal retinitis AND retinochoroiditis (disorder)
        '44674000',  # Focal choroiditis AND chorioretinitis of other posterior pole (disorder)
        '46497009',  # Juxtapapillary focal retinitis AND retinochoroiditis (disorder)
        '56787009',  # Peripheral focal choroiditis AND chorioretinitis (disorder)
        '88290000',  # Focal chorioretinitis caused by acquired toxoplasmosis (disorder)
        '193434008',  # Focal juxtapapillary choroiditis (disorder)
        '193436005',  # Juxtapapillary focal chorioretinitis (disorder)
        '193438006',  # Peripheral focal retinochoroiditis (disorder)
    }


class Glaucoma(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for glaucoma.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** This value set may use a model element related to Diagnosis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Glaucoma'
    OID = '2.16.840.1.113883.3.526.3.1423'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H40.10X0',  # Unspecified open-angle glaucoma, stage unspecified
        'H40.10X1',  # Unspecified open-angle glaucoma, mild stage
        'H40.10X2',  # Unspecified open-angle glaucoma, moderate stage
        'H40.10X3',  # Unspecified open-angle glaucoma, severe stage
        'H40.10X4',  # Unspecified open-angle glaucoma, indeterminate stage
        'H40.1110',  # Primary open-angle glaucoma, right eye, stage unspecified
        'H40.1111',  # Primary open-angle glaucoma, right eye, mild stage
        'H40.1112',  # Primary open-angle glaucoma, right eye, moderate stage
        'H40.1113',  # Primary open-angle glaucoma, right eye, severe stage
        'H40.1114',  # Primary open-angle glaucoma, right eye, indeterminate stage
        'H40.1120',  # Primary open-angle glaucoma, left eye, stage unspecified
        'H40.1121',  # Primary open-angle glaucoma, left eye, mild stage
        'H40.1122',  # Primary open-angle glaucoma, left eye, moderate stage
        'H40.1123',  # Primary open-angle glaucoma, left eye, severe stage
        'H40.1124',  # Primary open-angle glaucoma, left eye, indeterminate stage
        'H40.1130',  # Primary open-angle glaucoma, bilateral, stage unspecified
        'H40.1131',  # Primary open-angle glaucoma, bilateral, mild stage
        'H40.1132',  # Primary open-angle glaucoma, bilateral, moderate stage
        'H40.1133',  # Primary open-angle glaucoma, bilateral, severe stage
        'H40.1134',  # Primary open-angle glaucoma, bilateral, indeterminate stage
        'H40.1210',  # Low-tension glaucoma, right eye, stage unspecified
        'H40.1211',  # Low-tension glaucoma, right eye, mild stage
        'H40.1212',  # Low-tension glaucoma, right eye, moderate stage
        'H40.1213',  # Low-tension glaucoma, right eye, severe stage
        'H40.1214',  # Low-tension glaucoma, right eye, indeterminate stage
        'H40.1220',  # Low-tension glaucoma, left eye, stage unspecified
        'H40.1221',  # Low-tension glaucoma, left eye, mild stage
        'H40.1222',  # Low-tension glaucoma, left eye, moderate stage
        'H40.1223',  # Low-tension glaucoma, left eye, severe stage
        'H40.1224',  # Low-tension glaucoma, left eye, indeterminate stage
        'H40.1230',  # Low-tension glaucoma, bilateral, stage unspecified
        'H40.1231',  # Low-tension glaucoma, bilateral, mild stage
        'H40.1232',  # Low-tension glaucoma, bilateral, moderate stage
        'H40.1233',  # Low-tension glaucoma, bilateral, severe stage
        'H40.1234',  # Low-tension glaucoma, bilateral, indeterminate stage
        'H40.1310',  # Pigmentary glaucoma, right eye, stage unspecified
        'H40.1311',  # Pigmentary glaucoma, right eye, mild stage
        'H40.1312',  # Pigmentary glaucoma, right eye, moderate stage
        'H40.1313',  # Pigmentary glaucoma, right eye, severe stage
        'H40.1314',  # Pigmentary glaucoma, right eye, indeterminate stage
        'H40.1320',  # Pigmentary glaucoma, left eye, stage unspecified
        'H40.1321',  # Pigmentary glaucoma, left eye, mild stage
        'H40.1322',  # Pigmentary glaucoma, left eye, moderate stage
        'H40.1323',  # Pigmentary glaucoma, left eye, severe stage
        'H40.1324',  # Pigmentary glaucoma, left eye, indeterminate stage
        'H40.1330',  # Pigmentary glaucoma, bilateral, stage unspecified
        'H40.1331',  # Pigmentary glaucoma, bilateral, mild stage
        'H40.1332',  # Pigmentary glaucoma, bilateral, moderate stage
        'H40.1333',  # Pigmentary glaucoma, bilateral, severe stage
        'H40.1334',  # Pigmentary glaucoma, bilateral, indeterminate stage
        'H40.1410',  # Capsular glaucoma with pseudoexfoliation of lens, right eye, stage unspecified
        'H40.1411',  # Capsular glaucoma with pseudoexfoliation of lens, right eye, mild stage
        'H40.1412',  # Capsular glaucoma with pseudoexfoliation of lens, right eye, moderate stage
        'H40.1413',  # Capsular glaucoma with pseudoexfoliation of lens, right eye, severe stage
        'H40.1414',  # Capsular glaucoma with pseudoexfoliation of lens, right eye, indeterminate stage
        'H40.1420',  # Capsular glaucoma with pseudoexfoliation of lens, left eye, stage unspecified
        'H40.1421',  # Capsular glaucoma with pseudoexfoliation of lens, left eye, mild stage
        'H40.1422',  # Capsular glaucoma with pseudoexfoliation of lens, left eye, moderate stage
        'H40.1423',  # Capsular glaucoma with pseudoexfoliation of lens, left eye, severe stage
        'H40.1424',  # Capsular glaucoma with pseudoexfoliation of lens, left eye, indeterminate stage
        'H40.1430',  # Capsular glaucoma with pseudoexfoliation of lens, bilateral, stage unspecified
        'H40.1431',  # Capsular glaucoma with pseudoexfoliation of lens, bilateral, mild stage
        'H40.1432',  # Capsular glaucoma with pseudoexfoliation of lens, bilateral, moderate stage
        'H40.1433',  # Capsular glaucoma with pseudoexfoliation of lens, bilateral, severe stage
        'H40.1434',  # Capsular glaucoma with pseudoexfoliation of lens, bilateral, indeterminate stage
        'H40.151',  # Residual stage of open-angle glaucoma, right eye
        'H40.152',  # Residual stage of open-angle glaucoma, left eye
        'H40.153',  # Residual stage of open-angle glaucoma, bilateral
        'H40.20X0',  # Unspecified primary angle-closure glaucoma, stage unspecified
        'H40.20X1',  # Unspecified primary angle-closure glaucoma, mild stage
        'H40.20X2',  # Unspecified primary angle-closure glaucoma, moderate stage
        'H40.20X3',  # Unspecified primary angle-closure glaucoma, severe stage
        'H40.20X4',  # Unspecified primary angle-closure glaucoma, indeterminate stage
        'H40.211',  # Acute angle-closure glaucoma, right eye
        'H40.212',  # Acute angle-closure glaucoma, left eye
        'H40.213',  # Acute angle-closure glaucoma, bilateral
        'H40.2210',  # Chronic angle-closure glaucoma, right eye, stage unspecified
        'H40.2211',  # Chronic angle-closure glaucoma, right eye, mild stage
        'H40.2212',  # Chronic angle-closure glaucoma, right eye, moderate stage
        'H40.2213',  # Chronic angle-closure glaucoma, right eye, severe stage
        'H40.2214',  # Chronic angle-closure glaucoma, right eye, indeterminate stage
        'H40.2220',  # Chronic angle-closure glaucoma, left eye, stage unspecified
        'H40.2221',  # Chronic angle-closure glaucoma, left eye, mild stage
        'H40.2222',  # Chronic angle-closure glaucoma, left eye, moderate stage
        'H40.2223',  # Chronic angle-closure glaucoma, left eye, severe stage
        'H40.2224',  # Chronic angle-closure glaucoma, left eye, indeterminate stage
        'H40.2230',  # Chronic angle-closure glaucoma, bilateral, stage unspecified
        'H40.2231',  # Chronic angle-closure glaucoma, bilateral, mild stage
        'H40.2232',  # Chronic angle-closure glaucoma, bilateral, moderate stage
        'H40.2233',  # Chronic angle-closure glaucoma, bilateral, severe stage
        'H40.2234',  # Chronic angle-closure glaucoma, bilateral, indeterminate stage
        'H40.231',  # Intermittent angle-closure glaucoma, right eye
        'H40.232',  # Intermittent angle-closure glaucoma, left eye
        'H40.233',  # Intermittent angle-closure glaucoma, bilateral
        'H40.241',  # Residual stage of angle-closure glaucoma, right eye
        'H40.242',  # Residual stage of angle-closure glaucoma, left eye
        'H40.243',  # Residual stage of angle-closure glaucoma, bilateral
        'H40.31X0',  # Glaucoma secondary to eye trauma, right eye, stage unspecified
        'H40.31X1',  # Glaucoma secondary to eye trauma, right eye, mild stage
        'H40.31X2',  # Glaucoma secondary to eye trauma, right eye, moderate stage
        'H40.31X3',  # Glaucoma secondary to eye trauma, right eye, severe stage
        'H40.31X4',  # Glaucoma secondary to eye trauma, right eye, indeterminate stage
        'H40.32X0',  # Glaucoma secondary to eye trauma, left eye, stage unspecified
        'H40.32X1',  # Glaucoma secondary to eye trauma, left eye, mild stage
        'H40.32X2',  # Glaucoma secondary to eye trauma, left eye, moderate stage
        'H40.32X3',  # Glaucoma secondary to eye trauma, left eye, severe stage
        'H40.32X4',  # Glaucoma secondary to eye trauma, left eye, indeterminate stage
        'H40.33X0',  # Glaucoma secondary to eye trauma, bilateral, stage unspecified
        'H40.33X1',  # Glaucoma secondary to eye trauma, bilateral, mild stage
        'H40.33X2',  # Glaucoma secondary to eye trauma, bilateral, moderate stage
        'H40.33X3',  # Glaucoma secondary to eye trauma, bilateral, severe stage
        'H40.33X4',  # Glaucoma secondary to eye trauma, bilateral, indeterminate stage
        'H40.41X0',  # Glaucoma secondary to eye inflammation, right eye, stage unspecified
        'H40.41X1',  # Glaucoma secondary to eye inflammation, right eye, mild stage
        'H40.41X2',  # Glaucoma secondary to eye inflammation, right eye, moderate stage
        'H40.41X3',  # Glaucoma secondary to eye inflammation, right eye, severe stage
        'H40.41X4',  # Glaucoma secondary to eye inflammation, right eye, indeterminate stage
        'H40.42X0',  # Glaucoma secondary to eye inflammation, left eye, stage unspecified
        'H40.42X1',  # Glaucoma secondary to eye inflammation, left eye, mild stage
        'H40.42X2',  # Glaucoma secondary to eye inflammation, left eye, moderate stage
        'H40.42X3',  # Glaucoma secondary to eye inflammation, left eye, severe stage
        'H40.42X4',  # Glaucoma secondary to eye inflammation, left eye, indeterminate stage
        'H40.43X0',  # Glaucoma secondary to eye inflammation, bilateral, stage unspecified
        'H40.43X1',  # Glaucoma secondary to eye inflammation, bilateral, mild stage
        'H40.43X2',  # Glaucoma secondary to eye inflammation, bilateral, moderate stage
        'H40.43X3',  # Glaucoma secondary to eye inflammation, bilateral, severe stage
        'H40.43X4',  # Glaucoma secondary to eye inflammation, bilateral, indeterminate stage
        'H40.51X0',  # Glaucoma secondary to other eye disorders, right eye, stage unspecified
        'H40.51X1',  # Glaucoma secondary to other eye disorders, right eye, mild stage
        'H40.51X2',  # Glaucoma secondary to other eye disorders, right eye, moderate stage
        'H40.51X3',  # Glaucoma secondary to other eye disorders, right eye, severe stage
        'H40.51X4',  # Glaucoma secondary to other eye disorders, right eye, indeterminate stage
        'H40.52X0',  # Glaucoma secondary to other eye disorders, left eye, stage unspecified
        'H40.52X1',  # Glaucoma secondary to other eye disorders, left eye, mild stage
        'H40.52X2',  # Glaucoma secondary to other eye disorders, left eye, moderate stage
        'H40.52X3',  # Glaucoma secondary to other eye disorders, left eye, severe stage
        'H40.52X4',  # Glaucoma secondary to other eye disorders, left eye, indeterminate stage
        'H40.53X0',  # Glaucoma secondary to other eye disorders, bilateral, stage unspecified
        'H40.53X1',  # Glaucoma secondary to other eye disorders, bilateral, mild stage
        'H40.53X2',  # Glaucoma secondary to other eye disorders, bilateral, moderate stage
        'H40.53X3',  # Glaucoma secondary to other eye disorders, bilateral, severe stage
        'H40.53X4',  # Glaucoma secondary to other eye disorders, bilateral, indeterminate stage
        'H40.61X0',  # Glaucoma secondary to drugs, right eye, stage unspecified
        'H40.61X1',  # Glaucoma secondary to drugs, right eye, mild stage
        'H40.61X2',  # Glaucoma secondary to drugs, right eye, moderate stage
        'H40.61X3',  # Glaucoma secondary to drugs, right eye, severe stage
        'H40.61X4',  # Glaucoma secondary to drugs, right eye, indeterminate stage
        'H40.62X0',  # Glaucoma secondary to drugs, left eye, stage unspecified
        'H40.62X1',  # Glaucoma secondary to drugs, left eye, mild stage
        'H40.62X2',  # Glaucoma secondary to drugs, left eye, moderate stage
        'H40.62X3',  # Glaucoma secondary to drugs, left eye, severe stage
        'H40.62X4',  # Glaucoma secondary to drugs, left eye, indeterminate stage
        'H40.63X0',  # Glaucoma secondary to drugs, bilateral, stage unspecified
        'H40.63X1',  # Glaucoma secondary to drugs, bilateral, mild stage
        'H40.63X2',  # Glaucoma secondary to drugs, bilateral, moderate stage
        'H40.63X3',  # Glaucoma secondary to drugs, bilateral, severe stage
        'H40.63X4',  # Glaucoma secondary to drugs, bilateral, indeterminate stage
        'H40.811',  # Glaucoma with increased episcleral venous pressure, right eye
        'H40.812',  # Glaucoma with increased episcleral venous pressure, left eye
        'H40.813',  # Glaucoma with increased episcleral venous pressure, bilateral
        'H40.821',  # Hypersecretion glaucoma, right eye
        'H40.822',  # Hypersecretion glaucoma, left eye
        'H40.823',  # Hypersecretion glaucoma, bilateral
        'H40.831',  # Aqueous misdirection, right eye
        'H40.832',  # Aqueous misdirection, left eye
        'H40.833',  # Aqueous misdirection, bilateral
        'H40.89',  # Other specified glaucoma
        'Q15.0',  # Congenital glaucoma
    }
    SNOMEDCT = {
        '1654001',  # Corticosteroid-induced glaucoma (disorder)
        '15374009',  # Secondary glaucoma due to aphakia (disorder)
        '19144002',  # Absolute glaucoma (disorder)
        '19309007',  # Glaucoma associated with vascular disorder (disorder)
        '21571006',  # Secondary angle-closure glaucoma (disorder)
        '21928008',  # Secondary open-angle glaucoma (disorder)
        '23986001',  # Glaucoma (disorder)
        '27735002',  # Glaucoma associated with tumors AND/OR cysts (disorder)
        '29369005',  # Hypersecretion glaucoma (disorder)
        '30041005',  # Acute angle-closure glaucoma (disorder)
        '32893002',  # Phacolytic glaucoma (disorder)
        '33647009',  # Anatomical narrow angle glaucoma (disorder)
        '34623005',  # Glaucoma with increased episcleral venous pressure (disorder)
        '37155002',  # Glaucoma associated with ocular inflammation (disorder)
        '45623002',  # Glaucoma associated with anterior segment anomaly (disorder)
        '46168003',  # Pigmentary glaucoma (disorder)
        '50485007',  # Low tension glaucoma (disorder)
        '53667005',  # Glaucoma associated with systemic syndromes (disorder)
        '65460003',  # Intermittent angle-closure glaucoma (disorder)
        '66725002',  # Glaucoma due to perforating injury (disorder)
        '66747002',  # Glaucoma associated with ocular disorder (disorder)
        '68241007',  # Glaucoma associated with ocular trauma (disorder)
        '77075001',  # Primary open angle glaucoma (disorder)
        '84333006',  # Glaucoma associated with lens disorder (disorder)
        '84494001',  # Open-angle glaucoma (disorder)
        '89215000',  # Postoperative angle-closure glaucoma (disorder)
        '92829008',  # Glaucoma due to combination of mechanisms (disorder)
        '95213001',  # Primary glaucoma due to combination of mechanisms (disorder)
        '95250000',  # Secondary glaucoma due to combination mechanisms (disorder)
        '95717004',  # Secondary glaucoma (disorder)
        '111513000',  # Advanced open-angle glaucoma (disorder)
        '111514006',  # Pseudoexfoliation glaucoma (disorder)
        '193548006',  # Steroid-induced glaucoma glaucomatous stage (disorder)
        '193552006',  # Glaucoma due to chamber angle anomaly (disorder)
        '193553001',  # Glaucoma due to iris anomaly (disorder)
        '193555008',  # Glaucoma due to systemic syndrome (disorder)
        '193561006',  # Secondary angle-closure glaucoma with pupil block (disorder)
        '193562004',  # Glaucoma due to ocular vascular disorder (disorder)
        '204113001',  # Congenital glaucoma (disorder)
        '206248004',  # Traumatic glaucoma due to birth trauma (disorder)
        '232081005',  # Iatrogenic glaucoma (disorder)
        '232082003',  # Iatrogenic angle-closure glaucoma (disorder)
        '232083008',  # Glaucoma and corneal anomaly (disorder)
        '232086000',  # Neovascular glaucoma (disorder)
        '232087009',  # Glaucoma with intraocular hemorrhage (disorder)
        '232088004',  # Ghost cell glaucoma (disorder)
        '232090003',  # Glaucoma following surgery (disorder)
        '267625001',  # Glaucoma due to ocular tumor or cyst (disorder)
        '275477002',  # Glaucoma due to ocular cyst (disorder)
        '314017009',  # Acute-on-chronic glaucoma (disorder)
        '314033007',  # Uveitic glaucoma (disorder)
        '314784002',  # Secondary angle-closure glaucoma - synechial (disorder)
        '370504007',  # Aqueous humor misdirect (disorder)
        '392030001',  # Hemolytic glaucoma (disorder)
        '392288006',  # Primary angle-closure glaucoma (disorder)
        '392291006',  # Angle-closure glaucoma (disorder)
        '392300000',  # Phacomorphic glaucoma (disorder)
        '392352004',  # Angle recession glaucoma (disorder)
        '404634005',  # Glaucoma associated with iridocorneal endothelial syndrome (disorder)
        '404648005',  # Lens particle glaucoma (disorder)
        '415176004',  # Primary congenital glaucoma (disorder)
        '444863008',  # Anatomical narrow angle glaucoma with borderline intraocular pressure (disorder)
        '698840003',  # Neovascular glaucoma due to hyphema (disorder)
        '713457002',  # Neovascular glaucoma due to diabetes mellitus (disorder)
        '715144004',  # Glaucoma caused by silicone oil (disorder)
        '716166002',  # Microcornea with glaucoma and absent frontal sinus syndrome (disorder)
        '721898008',  # Corticosteroid-induced glaucoma of bilateral eyes (disorder)
        '722321001',  # Corticosteroid-induced glaucoma of left eye (disorder)
        '722329004',  # Corticosteroid-induced glaucoma of right eye (disorder)
        '787051000',  # Open-angle glaucoma of left eye (disorder)
        '787052007',  # Open-angle glaucoma of right eye (disorder)
        '24151000119106',  # Steroid-induced open angle glaucoma (disorder)
        '41911000119107',  # Glaucoma due to type 2 diabetes mellitus (disorder)
        '60981000119103',  # Glaucoma due to diabetes mellitus type 1 (disorder)
        '336611000119109',  # Acute angle-closure glaucoma of right eye (disorder)
        '336631000119104',  # Absolute glaucoma right eye (disorder)
        '342221000119104',  # Acute angle-closure glaucoma of left eye (disorder)
        '342241000119105',  # Absolute glaucoma left eye (disorder)
        '347381000119106',  # Bilateral acute angle-closure glaucoma (disorder)
        '347401000119106',  # Bilateral absolute glaucoma (disorder)
        '12239301000119102',  # Bilateral open-angle glaucoma (disorder)
        '12239421000119101',  # Bilateral glaucoma (disorder)
        '12239461000119106',  # Glaucoma of left eye (disorder)
        '12239501000119106',  # Glaucoma of right eye (disorder)
        '15633281000119103',  # Bilateral primary open angle glaucoma (disorder)
        '15633321000119108',  # Primary open angle glaucoma of left eye (disorder)
        '15640441000119104',  # Primary open angle glaucoma of right eye (disorder)
        '15673001000119103',  # Congenital glaucoma of bilateral eyes (disorder)
        '15673081000119106',  # Congenital glaucoma of left eye (disorder)
        '15679801000119105',  # Uveitis-glaucoma-hyphema syndrome of bilateral eyes (disorder)
        '15736441000119108',  # Primary angle-closure glaucoma of bilateral eyes (disorder)
        '15736481000119103',  # Primary angle-closure glaucoma of right eye (disorder)
        '15736521000119103',  # Primary angle-closure glaucoma of left eye (disorder)
        '15736561000119108',  # Neovascular glaucoma of bilateral eyes (disorder)
        '15736601000119108',  # Neovascular glaucoma of right eye (disorder)
        '15736641000119105',  # Neovascular glaucoma of left eye (disorder)
        '15736681000119100',  # Narrow-angle glaucoma of right eye (disorder)
        '15736721000119106',  # Angle-closure glaucoma of bilateral eyes (disorder)
        '15736761000119101',  # Narrow angle glaucoma of left eye (disorder)
        '15738841000119105',  # Glaucoma of bilateral eyes due to combination of mechanisms (disorder)
        '15738881000119100',  # Glaucoma of left eye due to combination of mechanisms (disorder)
        '15738921000119107',  # Glaucoma of right eye due to combination of mechanisms (disorder)
        '15739041000119106',  # Glaucoma of bilateral eyes due to iris anomaly (disorder)
        '15739081000119101',  # Glaucoma of bilateral eyes due to anterior segment anomaly (disorder)
        '15739121000119104',  # Glaucoma of right eye due to anterior segment anomaly (disorder)
        '15739161000119109',  # Glaucoma of left eye due to anterior segment anomaly (disorder)
        '15739201000119104',  # Glaucoma of left eye due to ocular vascular disorder (disorder)
        '15739241000119102',  # Glaucoma of right eye due to ocular vascular disorder (disorder)
        '15739281000119107',  # Glaucoma of bilateral eyes due to ocular vascular disorder (disorder)
        '15739321000119102',  # Glaucoma of right eye due to chamber angle anomaly (disorder)
        '15739361000119107',  # Glaucoma of bilateral eyes due to chamber angle anomaly (disorder)
        '15739401000119103',  # Glaucoma of left eye due to chamber angle anomaly (disorder)
        '15739441000119101',  # Glaucoma of right eye due to trauma (disorder)
        '15739481000119106',  # Glaucoma of left eye due to trauma (disorder)
        '15739561000119101',  # Glaucoma of right eye due to lens disorder (disorder)
        '15739641000119104',  # Glaucoma of left eye due to lens disorder (disorder)
        '15739681000119109',  # Glaucoma of right eye due to systemic disorder (disorder)
        '15739721000119103',  # Glaucoma of left eye due to systemic disorder (disorder)
        '15739761000119108',  # Glaucoma of bilateral eyes due to systemic disorder (disorder)
        '15993671000119108',  # Open-angle glaucoma of right eye caused by steroid (disorder)
        '15993711000119107',  # Open-angle glaucoma of left eye caused by steroid (disorder)
        '15993751000119108',  # Open-angle glaucoma of bilateral eyes caused by steroid (disorder)
        '15996831000119101',  # Glaucoma caused by Onchocerca volvulus (disorder)
    }


class GlaucomaAssociatedWithCongenitalAnomalies_Dystrophies_AndSystemicSyndromes(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of glaucoma associated with congenital anomalies, dystrophies, and systemic syndromes.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of glaucoma that may be due to congenital anomalies, dystrophies, or other systemic syndromes, including trauma, or a secondary diagnosis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Glaucoma Associated with Congenital Anomalies, Dystrophies, and Systemic Syndromes'
    OID = '2.16.840.1.113883.3.526.3.1461'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H40.31X0',  # Glaucoma secondary to eye trauma, right eye, stage unspecified
        'H40.31X1',  # Glaucoma secondary to eye trauma, right eye, mild stage
        'H40.31X2',  # Glaucoma secondary to eye trauma, right eye, moderate stage
        'H40.31X3',  # Glaucoma secondary to eye trauma, right eye, severe stage
        'H40.31X4',  # Glaucoma secondary to eye trauma, right eye, indeterminate stage
        'H40.32X0',  # Glaucoma secondary to eye trauma, left eye, stage unspecified
        'H40.32X1',  # Glaucoma secondary to eye trauma, left eye, mild stage
        'H40.32X2',  # Glaucoma secondary to eye trauma, left eye, moderate stage
        'H40.32X3',  # Glaucoma secondary to eye trauma, left eye, severe stage
        'H40.32X4',  # Glaucoma secondary to eye trauma, left eye, indeterminate stage
        'H40.33X0',  # Glaucoma secondary to eye trauma, bilateral, stage unspecified
        'H40.33X1',  # Glaucoma secondary to eye trauma, bilateral, mild stage
        'H40.33X2',  # Glaucoma secondary to eye trauma, bilateral, moderate stage
        'H40.33X3',  # Glaucoma secondary to eye trauma, bilateral, severe stage
        'H40.33X4',  # Glaucoma secondary to eye trauma, bilateral, indeterminate stage
        'H40.41X0',  # Glaucoma secondary to eye inflammation, right eye, stage unspecified
        'H40.41X1',  # Glaucoma secondary to eye inflammation, right eye, mild stage
        'H40.41X2',  # Glaucoma secondary to eye inflammation, right eye, moderate stage
        'H40.41X3',  # Glaucoma secondary to eye inflammation, right eye, severe stage
        'H40.41X4',  # Glaucoma secondary to eye inflammation, right eye, indeterminate stage
        'H40.42X0',  # Glaucoma secondary to eye inflammation, left eye, stage unspecified
        'H40.42X1',  # Glaucoma secondary to eye inflammation, left eye, mild stage
        'H40.42X2',  # Glaucoma secondary to eye inflammation, left eye, moderate stage
        'H40.42X3',  # Glaucoma secondary to eye inflammation, left eye, severe stage
        'H40.42X4',  # Glaucoma secondary to eye inflammation, left eye, indeterminate stage
        'H40.43X0',  # Glaucoma secondary to eye inflammation, bilateral, stage unspecified
        'H40.43X1',  # Glaucoma secondary to eye inflammation, bilateral, mild stage
        'H40.43X2',  # Glaucoma secondary to eye inflammation, bilateral, moderate stage
        'H40.43X3',  # Glaucoma secondary to eye inflammation, bilateral, severe stage
        'H40.43X4',  # Glaucoma secondary to eye inflammation, bilateral, indeterminate stage
        'H40.51X0',  # Glaucoma secondary to other eye disorders, right eye, stage unspecified
        'H40.51X1',  # Glaucoma secondary to other eye disorders, right eye, mild stage
        'H40.51X2',  # Glaucoma secondary to other eye disorders, right eye, moderate stage
        'H40.51X3',  # Glaucoma secondary to other eye disorders, right eye, severe stage
        'H40.51X4',  # Glaucoma secondary to other eye disorders, right eye, indeterminate stage
        'H40.52X0',  # Glaucoma secondary to other eye disorders, left eye, stage unspecified
        'H40.52X1',  # Glaucoma secondary to other eye disorders, left eye, mild stage
        'H40.52X2',  # Glaucoma secondary to other eye disorders, left eye, moderate stage
        'H40.52X3',  # Glaucoma secondary to other eye disorders, left eye, severe stage
        'H40.52X4',  # Glaucoma secondary to other eye disorders, left eye, indeterminate stage
        'H40.53X0',  # Glaucoma secondary to other eye disorders, bilateral, stage unspecified
        'H40.53X1',  # Glaucoma secondary to other eye disorders, bilateral, mild stage
        'H40.53X2',  # Glaucoma secondary to other eye disorders, bilateral, moderate stage
        'H40.53X3',  # Glaucoma secondary to other eye disorders, bilateral, severe stage
        'H40.53X4',  # Glaucoma secondary to other eye disorders, bilateral, indeterminate stage
        'H40.811',  # Glaucoma with increased episcleral venous pressure, right eye
        'H40.812',  # Glaucoma with increased episcleral venous pressure, left eye
        'H40.813',  # Glaucoma with increased episcleral venous pressure, bilateral
        'H40.821',  # Hypersecretion glaucoma, right eye
        'H40.822',  # Hypersecretion glaucoma, left eye
        'H40.823',  # Hypersecretion glaucoma, bilateral
        'H40.831',  # Aqueous misdirection, right eye
        'H40.832',  # Aqueous misdirection, left eye
        'H40.833',  # Aqueous misdirection, bilateral
        'H40.89',  # Other specified glaucoma
        'H40.9',  # Unspecified glaucoma
        'H42',  # Glaucoma in diseases classified elsewhere
    }
    SNOMEDCT = {
        '19309007',  # Glaucoma associated with vascular disorder (disorder)
        '29538005',  # Glaucomatocyclitic crisis (disorder)
        '32893002',  # Phacolytic glaucoma (disorder)
        '37155002',  # Glaucoma associated with ocular inflammation (disorder)
        '45623002',  # Glaucoma associated with anterior segment anomaly (disorder)
        '46168003',  # Pigmentary glaucoma (disorder)
        '66725002',  # Glaucoma due to perforating injury (disorder)
        '66747002',  # Glaucoma associated with ocular disorder (disorder)
        '68241007',  # Glaucoma associated with ocular trauma (disorder)
        '84333006',  # Glaucoma associated with lens disorder (disorder)
        '95717004',  # Secondary glaucoma (disorder)
        '193552006',  # Glaucoma due to chamber angle anomaly (disorder)
        '193553001',  # Glaucoma due to iris anomaly (disorder)
        '193555008',  # Glaucoma due to systemic syndrome (disorder)
        '193556009',  # Glaucoma in endocrine, nutritional and metabolic diseases (disorder)
        '193561006',  # Secondary angle-closure glaucoma with pupil block (disorder)
        '206248004',  # Traumatic glaucoma due to birth trauma (disorder)
        '232083008',  # Glaucoma and corneal anomaly (disorder)
        '232086000',  # Neovascular glaucoma (disorder)
        '314033007',  # Uveitic glaucoma (disorder)
        '370504007',  # Aqueous humor misdirect (disorder)
        '392300000',  # Phacomorphic glaucoma (disorder)
        '392352004',  # Angle recession glaucoma (disorder)
        '404634005',  # Glaucoma associated with iridocorneal endothelial syndrome (disorder)
        '404648005',  # Lens particle glaucoma (disorder)
        '418435001',  # Macular corneal dystrophy Type II (disorder)
        '333151000119108',  # Phacolytic glaucoma of right eye (disorder)
        '338761000119103',  # Phacolytic glaucoma of left eye (disorder)
        '15673041000119101',  # Congenital glaucoma of right eye (disorder)
        '15679801000119105',  # Uveitis-glaucoma-hyphema syndrome of bilateral eyes (disorder)
        '15736561000119108',  # Neovascular glaucoma of bilateral eyes (disorder)
        '15736601000119108',  # Neovascular glaucoma of right eye (disorder)
        '15736641000119105',  # Neovascular glaucoma of left eye (disorder)
        '15739041000119106',  # Glaucoma of bilateral eyes due to iris anomaly (disorder)
        '15739081000119101',  # Glaucoma of bilateral eyes due to anterior segment anomaly (disorder)
        '15739121000119104',  # Glaucoma of right eye due to anterior segment anomaly (disorder)
        '15739161000119109',  # Glaucoma of left eye due to anterior segment anomaly (disorder)
        '15739321000119102',  # Glaucoma of right eye due to chamber angle anomaly (disorder)
        '15739361000119107',  # Glaucoma of bilateral eyes due to chamber angle anomaly (disorder)
        '15739401000119103',  # Glaucoma of left eye due to chamber angle anomaly (disorder)
        '15739441000119101',  # Glaucoma of right eye due to trauma (disorder)
        '15739481000119106',  # Glaucoma of left eye due to trauma (disorder)
        '15739561000119101',  # Glaucoma of right eye due to lens disorder (disorder)
        '15739681000119109',  # Glaucoma of right eye due to systemic disorder (disorder)
        '15739721000119103',  # Glaucoma of left eye due to systemic disorder (disorder)
        '15739761000119108',  # Glaucoma of bilateral eyes due to systemic disorder (disorder)
    }


class HereditaryChoroidalDystrophies(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of hereditary choroidal dystrophies.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of hereditary choroidal dystrophies, including choroideremia and atrophy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Hereditary Choroidal Dystrophies'
    OID = '2.16.840.1.113883.3.526.3.1462'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H31.20',  # Hereditary choroidal dystrophy, unspecified
        'H31.21',  # Choroideremia
        'H31.22',  # Choroidal dystrophy (central areolar) (generalized) (peripapillary)
        'H31.23',  # Gyrate atrophy, choroid
        'H31.29',  # Other hereditary choroidal dystrophy
    }
    SNOMEDCT = {
        '19871000',  # Total generalized choroidal dystrophy (disorder)
        '24950005',  # Partial generalized choroidal dystrophy (disorder)
        '41250005',  # Diffuse choroidal sclerosis (disorder)
        '59753003',  # Total circumpapillary dystrophy of choroid (disorder)
        '74469006',  # Hereditary choroidal dystrophy (disorder)
        '75241009',  # Choroideremia (disorder)
        '92820007',  # Circumpapillary dystrophy of choroid (disorder)
        '193466003',  # Partial circumpapillary choroid dystrophy (disorder)
        '231996009',  # Central areolar choroidal sclerosis (disorder)
        '231997000',  # Peripapillary choroidal sclerosis (disorder)
        '247200005',  # Choroidal atrophy (finding)
        '312918002',  # Choroidal dystrophy (disorder)
        '314467007',  # Gyrate atrophy (disorder)
        '715984007',  # Cerebellar ataxia with hypogonadism and choroidal dystrophy syndrome (disorder)
        '717761005',  # Choroideremia with deafness and obesity syndrome (disorder)
        '720850008',  # Choroidal atrophy and alopecia syndrome (disorder)
        '733110004',  # Van den Bosch syndrome (disorder)
    }


class HereditaryCornealDystrophies(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of hereditary corneal dystrophies.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of hereditary corneal dystrophies, including congenital, epithelial/juvenile, granular, lattice or macular corneal dystrophies.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Hereditary Corneal Dystrophies'
    OID = '2.16.840.1.113883.3.526.3.1424'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H18.511',  # Endothelial corneal dystrophy, right eye
        'H18.512',  # Endothelial corneal dystrophy, left eye
        'H18.513',  # Endothelial corneal dystrophy, bilateral
        'H18.521',  # Epithelial (juvenile) corneal dystrophy, right eye
        'H18.522',  # Epithelial (juvenile) corneal dystrophy, left eye
        'H18.523',  # Epithelial (juvenile) corneal dystrophy, bilateral
        'H18.531',  # Granular corneal dystrophy, right eye
        'H18.532',  # Granular corneal dystrophy, left eye
        'H18.533',  # Granular corneal dystrophy, bilateral
        'H18.541',  # Lattice corneal dystrophy, right eye
        'H18.542',  # Lattice corneal dystrophy, left eye
        'H18.543',  # Lattice corneal dystrophy, bilateral
        'H18.551',  # Macular corneal dystrophy, right eye
        'H18.552',  # Macular corneal dystrophy, left eye
        'H18.553',  # Macular corneal dystrophy, bilateral
        'H18.591',  # Other hereditary corneal dystrophies, right eye
        'H18.592',  # Other hereditary corneal dystrophies, left eye
        'H18.593',  # Other hereditary corneal dystrophies, bilateral
    }
    SNOMEDCT = {
        '1674008',  # Juvenile epithelial corneal dystrophy (disorder)
        '5587004',  # Corneal dystrophy (disorder)
        '29504002',  # Posterior polymorphous corneal dystrophy (disorder)
        '39662004',  # Crystalline corneal dystrophy (disorder)
        '45283008',  # Granular corneal dystrophy (disorder)
        '60258001',  # Macular corneal dystrophy (disorder)
        '75663006',  # Ring-like corneal dystrophy (disorder)
        '77797009',  # Hereditary corneal dystrophy (disorder)
        '95488001',  # Congenital macular corneal dystrophy (disorder)
        '123670005',  # Adiposa cornea dystrophy (disorder)
        '129623003',  # Iridocorneal endothelial syndrome (disorder)
        '193837009',  # Combined corneal dystrophy (disorder)
        '193839007',  # Fuchs' corneal dystrophy (disorder)
        '231930000',  # Reis-Bucklers' corneal dystrophy (disorder)
        '231931001',  # Stromal corneal dystrophy (disorder)
        '231933003',  # Lattice corneal dystrophy, isolated form (disorder)
        '231934009',  # Pre-descemet's corneal dystrophy (disorder)
        '361199007',  # Lattice corneal dystrophy (disorder)
        '373426005',  # Epithelial basement membrane dystrophy (disorder)
        '397568004',  # Granular corneal dystrophy type II (disorder)
        '404633004',  # Cogan-Reese syndrome (disorder)
        '416633008',  # Congenital hereditary endothelial dystrophy type 1 (disorder)
        '416960004',  # Corneal endothelial dystrophy (disorder)
        '417065002',  # Thiel-Behnke corneal dystrophy (disorder)
        '417183007',  # Fleck corneal dystrophy (disorder)
        '417395001',  # Congenital hereditary endothelial dystrophy type 2 (disorder)
        '417651000',  # Congenital hereditary endothelial dystrophy (disorder)
        '418054005',  # Macular corneal dystrophy Type I (disorder)
        '418435001',  # Macular corneal dystrophy Type II (disorder)
        '419039007',  # Granular corneal dystrophy type I (disorder)
        '419197009',  # Lattice corneal dystrophy Type I (disorder)
        '419395007',  # Schnyder crystalline cornea dystrophy (disorder)
        '419900000',  # Gelatinous droplike corneal dystrophy (disorder)
        '430888006',  # Dystrophy of anterior cornea (disorder)
    }


class HereditaryRetinalDystrophies(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of hereditary retinal dystrophies.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of hereditary retinal or vitreoretinal dystrophies, including Usher syndrome.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Hereditary Retinal Dystrophies'
    OID = '2.16.840.1.113883.3.526.3.1463'
    DEFINITION_VERSION = '20210219'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H35.50',  # Unspecified hereditary retinal dystrophy
        'H35.51',  # Vitreoretinal dystrophy
        'H35.52',  # Pigmentary retinal dystrophy
        'H35.53',  # Other dystrophies primarily involving the sensory retina
        'H35.54',  # Dystrophies primarily involving the retinal pigment epithelium
        'H36',  # Retinal disorders in diseases classified elsewhere
    }
    SNOMEDCT = {
        '15059000',  # Retinal dystrophy in systemic lipidosis (disorder)
        '28835009',  # Retinitis pigmentosa (disorder)
        '41799005',  # Hereditary retinal dystrophy (disorder)
        '57838006',  # Retinitis pigmentosa-deafness syndrome (disorder)
        '79556007',  # Vitreoretinal dystrophy (disorder)
        '80328002',  # Progressive cone-rod dystrophy (disorder)
        '192788009',  # Retinal dystrophy in cerebroretinal lipidosis (disorder)
        '193400007',  # Hereditary retinal dystrophies in lipidoses (disorder)
        '193409008',  # Hyaline retinal dystrophy (disorder)
        '193415008',  # Tapetoretinal dystrophy (disorder)
        '232052009',  # Autosomal dominant retinitis pigmentosa (disorder)
        '232053004',  # Autosomal recessive retinitis pigmentosa (disorder)
        '232056007',  # Progressive rod dystrophy (disorder)
        '232057003',  # Usher syndrome type 1 (disorder)
        '232058008',  # Usher syndrome type 2 (disorder)
        '267613004',  # Progressive cone dystrophy (without rod involvement) (disorder)
        '312917007',  # Cone dystrophy (disorder)
        '313003009',  # Rod dystrophy (disorder)
        '764939004',  # Fundus albipunctatus (disorder)
    }


class HypotonyOfEye(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of hypotony of eye.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of hypotony of eye.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Hypotony of Eye'
    OID = '2.16.840.1.113883.3.526.3.1426'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H44.40',  # Unspecified hypotony of eye
        'H44.411',  # Flat anterior chamber hypotony of right eye
        'H44.412',  # Flat anterior chamber hypotony of left eye
        'H44.413',  # Flat anterior chamber hypotony of eye, bilateral
        'H44.421',  # Hypotony of right eye due to ocular fistula
        'H44.422',  # Hypotony of left eye due to ocular fistula
        'H44.423',  # Hypotony of eye due to ocular fistula, bilateral
        'H44.431',  # Hypotony of eye due to other ocular disorders, right eye
        'H44.432',  # Hypotony of eye due to other ocular disorders, left eye
        'H44.433',  # Hypotony of eye due to other ocular disorders, bilateral
        'H44.441',  # Primary hypotony of right eye
        'H44.442',  # Primary hypotony of left eye
        'H44.443',  # Primary hypotony of eye, bilateral
    }
    SNOMEDCT = {
        '2251002',  # Primary hypotony of eye (disorder)
        '19721008',  # Hypotony of eye (disorder)
        '53885001',  # Hypotony of eye co-occurrent and due to ocular fistula (disorder)
        '95255005',  # Secondary hypotony of eye (disorder)
        '424497004',  # Optic disc edema due to hypotony (disorder)
        '700280001',  # Flat anterior chamber of eye with hypotony (disorder)
    }


class InjuryToOpticNerveAndPathways(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of injury to optic nerve and optic pathways.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of injuries to optic nerve, optic chiasm, optic tract, optic pathways and optic visual cortex.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Injury to Optic Nerve and Pathways'
    OID = '2.16.840.1.113883.3.526.3.1427'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'S04.011A',  # Injury of optic nerve, right eye, initial encounter
        'S04.012A',  # Injury of optic nerve, left eye, initial encounter
        'S04.02XA',  # Injury of optic chiasm, initial encounter
        'S04.031A',  # Injury of optic tract and pathways, right side, initial encounter
        'S04.032A',  # Injury of optic tract and pathways, left side, initial encounter
        'S04.041A',  # Injury of visual cortex, right side, initial encounter
        'S04.042A',  # Injury of visual cortex, left side, initial encounter
    }
    SNOMEDCT = {
        '48233004',  # Traumatic optic nerve injury (disorder)
        '69820004',  # Injury of optic chiasm (disorder)
        '80499003',  # Traumatic injury of visual pathways (disorder)
        '212141007',  # Optic nerve and pathway injury (disorder)
        '212142000',  # Visual cortex injury (disorder)
        '230513000',  # Avulsion of optic nerve (disorder)
        '230514006',  # Radiation damage to optic nerve (disorder)
        '230515007',  # Traumatic optic neuropathy (disorder)
        '230519001',  # Injury of optic tract (disorder)
        '735649003',  # Injury of bilateral optic tracts (disorder)
        '736321008',  # Injury of both visual cortices (disorder)
        '762362006',  # Traumatic injury of left optic tract (disorder)
        '762363001',  # Traumatic injury of right optic tract (disorder)
        '762436001',  # Injury of left visual cortex (disorder)
        '762617003',  # Injury of right visual cortex (disorder)
    }


class MacularScarOfPosteriorPolar(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of macular scar of posterior polar.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of macular or posterior pole scarring.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Macular Scar of Posterior Polar'
    OID = '2.16.840.1.113883.3.526.3.1559'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H31.011',  # Macula scars of posterior pole (postinflammatory) (post-traumatic), right eye
        'H31.012',  # Macula scars of posterior pole (postinflammatory) (post-traumatic), left eye
        'H31.013',  # Macula scars of posterior pole (postinflammatory) (post-traumatic), bilateral
    }
    SNOMEDCT = {
        '18410006',  # Scarred macula (disorder)
        '21939006',  # Scar of posterior pole of eye (disorder)
        '267611002',  # Degeneration of macula and posterior pole (disorder)
        '425622004',  # Scar of macula following trauma (disorder)
        '737273002',  # Degeneration of posterior pole of eye (disorder)
        '774010000',  # Scarred macula of left eye (disorder)
        '344341000119106',  # Postinflammatory chorioretinal scar of bilateral eyes (disorder)
        '15631531000119100',  # Scarred macula of bilateral eyes (disorder)
        '15631611000119106',  # Scarred macula of right eye (disorder)
        '15710081000119102',  # Scar of macula of right eye following trauma (disorder)
        '15710121000119100',  # Scar of macula of bilateral eyes following trauma (disorder)
        '15710161000119105',  # Scar of macula of left eye following trauma (disorder)
    }


class MorgagnianCataract(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of morgagnian cataract.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of a morgagnian cataract.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Morgagnian Cataract'
    OID = '2.16.840.1.113883.3.526.3.1558'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H25.21',  # Age-related cataract, morgagnian type, right eye
        'H25.22',  # Age-related cataract, morgagnian type, left eye
        'H25.23',  # Age-related cataract, morgagnian type, bilateral
    }
    SNOMEDCT = {
        '264443002',  # Morgagnian cataract (disorder)
    }


class NystagmusAndOtherIrregularEyeMovements(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for nystagmus and other irregular eye movements.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of nystagmus.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Nystagmus and Other Irregular Eye Movements'
    OID = '2.16.840.1.113883.3.526.3.1465'
    DEFINITION_VERSION = '20210308'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H55.00',  # Unspecified nystagmus
        'H55.01',  # Congenital nystagmus
        'H55.02',  # Latent nystagmus
        'H55.03',  # Visual deprivation nystagmus
        'H55.04',  # Dissociated nystagmus
        'H55.09',  # Other forms of nystagmus
        'H55.81',  # Deficient saccadic eye movements
        'H55.82',  # Deficient smooth pursuit eye movements
        'H55.89',  # Other irregular eye movements
    }
    SNOMEDCT = {
        '563001',  # Nystagmus (disorder)
        '3327002',  # Jerk nystagmus (disorder)
        '8134005',  # Periodic alternating nystagmus (disorder)
        '9074007',  # Retraction nystagmus (disorder)
        '9520006',  # Dissociated nystagmus (disorder)
        '29356006',  # End-position nystagmus (disorder)
        '35743001',  # Pendular nystagmus (disorder)
        '39737009',  # Symptomatic nystagmus (disorder)
        '40782004',  # Hennebert's syndrome (disorder)
        '44526006',  # Rotary nystagmus (disorder)
        '45339001',  # Visual deprivation nystagmus (disorder)
        '46888001',  # Nystagmus associated with disorder of the vestibular system (disorder)
        '64635004',  # Congenital nystagmus (disorder)
        '68610001',  # Latent nystagmus (disorder)
        '81756001',  # Horizontal nystagmus (disorder)
        '83655001',  # Rebound nystagmus (disorder)
        '95777008',  # Temporary nystagmus (disorder)
        '95778003',  # Permanent nystagmus (disorder)
        '95779006',  # Unidirectional nystagmus (disorder)
        '95780009',  # Multidirectional nystagmus (disorder)
        '95781008',  # Conjugate nystagmus (disorder)
        '95782001',  # See-saw nystagmus (disorder)
        '95783006',  # Rotational nystagmus (disorder)
        '95784000',  # Convergence nystagmus (disorder)
        '95785004',  # Positional nystagmus (disorder)
        '95786003',  # Fatigable positional nystagmus (disorder)
        '95787007',  # Non-fatigable positional nystagmus (disorder)
        '95788002',  # Positional nystagmus, variable eye direction (disorder)
        '95790001',  # Fixation nystagmus (disorder)
        '111533001',  # Vertical nystagmus (disorder)
        '232124002',  # Voluntary nystagmus (disorder)
        '246761002',  # Uniocular nystagmus (disorder)
        '246762009',  # Gaze paretic nystagmus (disorder)
        '278468008',  # Pressure-induced nystagmus (disorder)
        '278469000',  # Sound intensity-induced nystagmus (disorder)
        '285765003',  # Manifest-latent nystagmus (disorder)
        '307670000',  # Congenital nystagmus with sensory abnormality (disorder)
        '307671001',  # Congenital nystagmus without sensory abnormality (disorder)
        '307672008',  # Oculopalatal myoclonus (disorder)
        '307673003',  # Peripheral vestibular nystagmus (disorder)
        '307674009',  # Benign paroxysmal positional vertigo nystagmus (disorder)
        '307675005',  # Central vestibular nystagmus (disorder)
        '307676006',  # Downbeat nystagmus (disorder)
        '307677002',  # Upbeat nystagmus (disorder)
        '307678007',  # Torsional central vestibular nystagmus (disorder)
        '307679004',  # Hemi-seesaw nystagmus (disorder)
        '307680001',  # Abducting nystagmus (disorder)
        '307700006',  # Ataxic nystagmus (disorder)
        '314935002',  # Acquired nystagmus (disorder)
        '359687001',  # Acquired pendular nystagmus (disorder)
        '373590007',  # Convergence retraction nystagmus (disorder)
        '400948003',  # Spasmus nutans (disorder)
        '400949006',  # Divergence retraction nystagmus (disorder)
        '404676002',  # Bruns nystagmus (disorder)
        '404677006',  # Elliptical nystagmus (disorder)
        '414489005',  # Infantile nystagmus syndrome (disorder)
        '418472007',  # Heartbeat nystagmus (disorder)
        '733334001',  # Perverted nystagmus (disorder)
        '733335000',  # Eyelid nystagmus (disorder)
        '762438000',  # Physiological nystagmus (disorder)
        '762439008',  # Torsional nystagmus (disorder)
        '6881000119109',  # Sensory nystagmus (disorder)
        '431951000124101',  # Idiopathic nystagmus (disorder)
    }


class OpenWoundOfEyeball(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of open wound of the eyeball.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of choroidal rupture, traumatic enucleation of eye, and avulsion, contusion, laceration, penetration and/or rupture of eye.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Open Wound of Eyeball'
    OID = '2.16.840.1.113883.3.526.3.1430'
    DEFINITION_VERSION = '20210406'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'S05.11XA',  # Contusion of eyeball and orbital tissues, right eye, initial encounter
        'S05.12XA',  # Contusion of eyeball and orbital tissues, left eye, initial encounter
        'S05.21XA',  # Ocular laceration and rupture with prolapse or loss of intraocular tissue, right eye, initial encounter
        'S05.22XA',  # Ocular laceration and rupture with prolapse or loss of intraocular tissue, left eye, initial encounter
        'S05.31XA',  # Ocular laceration without prolapse or loss of intraocular tissue, right eye, initial encounter
        'S05.32XA',  # Ocular laceration without prolapse or loss of intraocular tissue, left eye, initial encounter
        'S05.51XA',  # Penetrating wound with foreign body of right eyeball, initial encounter
        'S05.52XA',  # Penetrating wound with foreign body of left eyeball, initial encounter
        'S05.61XA',  # Penetrating wound without foreign body of right eyeball, initial encounter
        'S05.62XA',  # Penetrating wound without foreign body of left eyeball, initial encounter
        'S05.71XA',  # Avulsion of right eye, initial encounter
        'S05.72XA',  # Avulsion of left eye, initial encounter
        'S05.8X1A',  # Other injuries of right eye and orbit, initial encounter
        'S05.8X2A',  # Other injuries of left eye and orbit, initial encounter
    }
    SNOMEDCT = {
        '1869002',  # Rupture of sphincter of pupil (disorder)
        '2532009',  # Choroidal rupture (disorder)
        '3018008',  # Penetration of eyeball with magnetic foreign body (disorder)
        '12193006',  # Avulsion of eye (disorder)
        '21549001',  # Superficial laceration of cornea (disorder)
        '23293006',  # Rupture of eye with partial loss of intraocular tissue (disorder)
        '37911007',  # Ocular laceration with prolapse AND/OR exposure of intraocular tissue (disorder)
        '41658006',  # Traumatic enucleation of eye (disorder)
        '42312000',  # Rupture in Descemet's membrane (disorder)
        '42697001',  # Contusion of orbital tissues (disorder)
        '44199000',  # Penetration of eyeball with nonmagnetic foreign body (disorder)
        '85100000',  # Open wound of eyeball (disorder)
        '95718009',  # Rupture of lens (disorder)
        '95725002',  # Corneal laceration (disorder)
        '110262009',  # Contusion of periorbital region (disorder)
        '125687005',  # Contusion of ocular adnexa (disorder)
        '210285009',  # Ocular laceration with intraocular prolapse (disorder)
        '211483007',  # Contusion of globe of eye (disorder)
        '231791008',  # Rupture of globe (disorder)
        '231795004',  # Laceration of globe of eye (disorder)
        '231952009',  # Tears of iris stroma (disorder)
        '315296002',  # Penetrating wound of eye (disorder)
        '367423000',  # Contusion of eye (disorder)
        '416179002',  # Contusion of eye AND ocular adnexa (disorder)
        '416786002',  # Penetrating injury of lens (disorder)
        '416931008',  # Corneoscleral laceration (disorder)
        '417654008',  # Contusion of ocular adnexa and periocular tissues (disorder)
        '419743001',  # Ruptured Descemet membrane due to birth trauma (disorder)
        '722613000',  # Penetrating wound of eyeball without foreign body (disorder)
        '735657000',  # Penetrating injury of both eyeballs (disorder)
        '870312007',  # Penetrating wound of eyeball with foreign body (disorder)
        '10942751000119100',  # Commotio retinae of right retina (disorder)
        '10942791000119105',  # Commotio retinae of left retina (disorder)
        '12403101000119108',  # Penetrating wound of right eye caused by foreign body (disorder)
    }


class OpticAtrophy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for optic atrophy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of optic atrophy, which may include atrophy of the optic disc.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Optic Atrophy'
    OID = '2.16.840.1.113883.3.526.3.1466'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H47.20',  # Unspecified optic atrophy
        'H47.211',  # Primary optic atrophy, right eye
        'H47.212',  # Primary optic atrophy, left eye
        'H47.213',  # Primary optic atrophy, bilateral
        'H47.22',  # Hereditary optic atrophy
        'H47.231',  # Glaucomatous optic atrophy, right eye
        'H47.232',  # Glaucomatous optic atrophy, left eye
        'H47.233',  # Glaucomatous optic atrophy, bilateral
        'H47.291',  # Other optic atrophy, right eye
        'H47.292',  # Other optic atrophy, left eye
        'H47.293',  # Other optic atrophy, bilateral
    }
    SNOMEDCT = {
        '1207009',  # Glaucomatous atrophy of optic disc (disorder)
        '2065009',  # Dominant hereditary optic atrophy (disorder)
        '19448008',  # Optic atrophy associated with retinal dystrophy (disorder)
        '21098003',  # Primary optic atrophy (disorder)
        '22386003',  # Syphilitic optic atrophy (disorder)
        '24003004',  # Postinflammatory optic atrophy (disorder)
        '26360005',  # Hereditary optic atrophy (disorder)
        '58610003',  # Leber's optic atrophy (disorder)
        '70694009',  # Diabetes mellitus AND insipidus with optic atrophy AND deafness (disorder)
        '76976005',  # Optic atrophy (disorder)
        '95774001',  # Atrophy of optic disc (disorder)
        '95775000',  # Retrobulbar optic nerve atrophy (disorder)
        '111527005',  # Partial optic atrophy (disorder)
        '123615007',  # Secondary optic atrophy (disorder)
        '128203003',  # Hereditary motor and sensory neuropathy with optic atrophy (disorder)
        '194043004',  # Optic atrophy secondary to retinal disease (disorder)
        '230511003',  # Optic atrophy secondary to papilledema (disorder)
        '239028001',  # Odontotrichomelic syndrome (disorder)
        '247219000',  # Atrophy of sector of optic disc (disorder)
        '312944002',  # Compressive optic atrophy (disorder)
        '359683002',  # Complete optic atrophy (disorder)
        '715374003',  # Autosomal dominant optic atrophy plus syndrome (disorder)
        '717336005',  # Autosomal dominant optic atrophy classic form (disorder)
        '717975006',  # Autosomal dominant optic atrophy and peripheral neuropathy syndrome (disorder)
        '719517009',  # Autosomal dominant optic atrophy and cataract (disorder)
        '838307002',  # Childhood-onset autosomal dominant optic atrophy (disorder)
        '838345001',  # Autosomal recessive optic atrophy type 6 (disorder)
        '870319003',  # Optic atrophy due to late syphilis (disorder)
        '348051000119102',  # Atrophy of right optic nerve following inflammation (disorder)
        '348501000119108',  # Atrophy of left optic nerve following inflammation (disorder)
        '680991000119108',  # Atrophy of bilateral optic nerves following inflammation (disorder)
    }


class OpticNeuritis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of optic neuritis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of optic neuritis, neuropathy, neuroretinitis, and papillitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Optic Neuritis'
    OID = '2.16.840.1.113883.3.526.3.1467'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H46.01',  # Optic papillitis, right eye
        'H46.02',  # Optic papillitis, left eye
        'H46.03',  # Optic papillitis, bilateral
        'H46.11',  # Retrobulbar neuritis, right eye
        'H46.12',  # Retrobulbar neuritis, left eye
        'H46.13',  # Retrobulbar neuritis, bilateral
        'H46.2',  # Nutritional optic neuropathy
        'H46.3',  # Toxic optic neuropathy
        'H46.8',  # Other optic neuritis
        'H46.9',  # Unspecified optic neuritis
    }
    SNOMEDCT = {
        '25044007',  # Neuromyelitis optica (disorder)
        '26125006',  # Toxic optic neuropathy (disorder)
        '36276008',  # Syphilitic retrobulbar neuritis (disorder)
        '51604006',  # Acute retrobulbar neuritis (disorder)
        '66760008',  # Optic neuritis (disorder)
        '73221001',  # Optic papillitis (disorder)
        '73431005',  # Meningococcal optic neuritis (disorder)
        '82108004',  # Nutritional optic neuropathy (disorder)
        '230507009',  # Retrobulbar neuritis (disorder)
        '314537004',  # Optic papillopathy due to diabetes mellitus (disorder)
        '314538009',  # Tobacco related optic neuropathy (disorder)
        '314539001',  # Alcohol related optic neuropathy (disorder)
        '314540004',  # Drug induced optic neuropathy (disorder)
        '314787009',  # Optic neuroretinitis (disorder)
        '410471004',  # Neuroretinitis (disorder)
        '410472006',  # Bartonella henselae neuroretinitis (disorder)
        '416913007',  # Toxoplasma neuroretinitis (disorder)
        '420920007',  # Optic neuropathy due to vitamin B12 deficiency (disorder)
        '2691000124105',  # Recurrent optic neuritis (disorder)
        '432521000124106',  # Bilateral optic neuritis (disorder)
        '15631011000119102',  # Neuritis of right optic nerve (disorder)
        '15631051000119101',  # Neuritis of left optic nerve (disorder)
    }


class OtherBackgroundRetinopathyAndRetinalVascularChanges(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of other background retinopathy and retinal vascular changes.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of various retinal changes and retinopathies, including exudative retinopathy, retinal neovascularization, arteriolitis, arteritis, endarteritis, phlebitis, and vasculitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Other Background Retinopathy and Retinal Vascular Changes'
    OID = '2.16.840.1.113883.3.526.3.1469'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H35.021',  # Exudative retinopathy, right eye
        'H35.022',  # Exudative retinopathy, left eye
        'H35.023',  # Exudative retinopathy, bilateral
        'H35.051',  # Retinal neovascularization, unspecified, right eye
        'H35.052',  # Retinal neovascularization, unspecified, left eye
        'H35.053',  # Retinal neovascularization, unspecified, bilateral
        'H35.061',  # Retinal vasculitis, right eye
        'H35.062',  # Retinal vasculitis, left eye
        'H35.063',  # Retinal vasculitis, bilateral
    }
    SNOMEDCT = {
        '7089008',  # Retinal phlebitis (disorder)
        '11255009',  # Retinal arteritis (disorder)
        '25506007',  # Exudative retinopathy (disorder)
        '54122009',  # Eales' disease (disorder)
        '61267008',  # Retinal neovascularization (disorder)
        '71719003',  # Thrombophlebitis of retinal vein (disorder)
        '75971007',  # Choroidal retinal neovascularization (disorder)
        '77628002',  # Retinal vasculitis (disorder)
        '86191000',  # Retinal endarteritis (disorder)
        '247099009',  # Optic disc neovascularization (disorder)
        '247100001',  # Peripheral retinal neovascularization (disorder)
        '314265001',  # Classic choroidal neovascular membrane (disorder)
        '414297000',  # Frosted branch angiitis (disorder)
        '416703007',  # Retinal vasculitis due to polyarteritis nodosa (disorder)
        '417019000',  # Retinal arteriolitis (disorder)
        '417303004',  # Retinal vasculitis due to systemic lupus erythematosus (disorder)
        '417544004',  # Segmental retinal periarteritis (disorder)
        '677651000119102',  # Neovascularization of choroid of left eye (disorder)
        '677681000119109',  # Neovascularization of choroid of right eye (disorder)
    }


class OtherDisordersOfOpticNerve(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis for other disorders of optic nerve.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of ischemic optic neuropathy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Other Disorders of Optic Nerve'
    OID = '2.16.840.1.113883.3.526.3.1471'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H47.011',  # Ischemic optic neuropathy, right eye
        'H47.012',  # Ischemic optic neuropathy, left eye
        'H47.013',  # Ischemic optic neuropathy, bilateral
    }
    SNOMEDCT = {
        '14357004',  # Ischemic optic neuropathy (disorder)
        '230508004',  # Arteritic ischemic optic neuropathy (disorder)
        '230509007',  # Non-arteritic ischemic optic neuropathy (disorder)
        '404659001',  # Anterior ischemic optic neuropathy (disorder)
        '415147002',  # Posterior ischemic optic neuropathy (disorder)
        '733506009',  # Arteritic anterior ischemic optic neuropathy (disorder)
        '838324008',  # Autoimmune optic neuropathy (disorder)
        '838363001',  # Arteritic anterior ischemic optic neuropathy due to giant cell arteritis (disorder)
        '860808008',  # Idiopathic non-arteritic ischemic optic neuropathy (disorder)
        '860884007',  # Idiopathic non-arteritic posterior ischemic optic neuropathy (disorder)
        '334651000119101',  # Ischemic optic neuropathy of right eye (disorder)
        '340251000119108',  # Ischemic optic neuropathy of left eye (disorder)
        '345611000119102',  # Ischemic optic neuropathy of bilateral eyes (disorder)
        '15731601000119100',  # Anterior ischemic optic neuropathy of left eye (disorder)
        '15731641000119103',  # Anterior ischemic optic neuropathy of right eye (disorder)
    }


class OtherEndophthalmitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of other endophthalmitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of ophthalmia nodosa, parasitic cyst or endophthalmitis, panuveitis, sympathetic uveitis, or other endophthalmitis diagnoses.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Other Endophthalmitis'
    OID = '2.16.840.1.113883.3.526.3.1473'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H16.241',  # Ophthalmia nodosa, right eye
        'H16.242',  # Ophthalmia nodosa, left eye
        'H16.243',  # Ophthalmia nodosa, bilateral
        'H21.331',  # Parasitic cyst of iris, ciliary body or anterior chamber, right eye
        'H21.332',  # Parasitic cyst of iris, ciliary body or anterior chamber, left eye
        'H21.333',  # Parasitic cyst of iris, ciliary body or anterior chamber, bilateral
        'H33.121',  # Parasitic cyst of retina, right eye
        'H33.122',  # Parasitic cyst of retina, left eye
        'H33.123',  # Parasitic cyst of retina, bilateral
        'H44.111',  # Panuveitis, right eye
        'H44.112',  # Panuveitis, left eye
        'H44.113',  # Panuveitis, bilateral
        'H44.121',  # Parasitic endophthalmitis, unspecified, right eye
        'H44.122',  # Parasitic endophthalmitis, unspecified, left eye
        'H44.123',  # Parasitic endophthalmitis, unspecified, bilateral
        'H44.131',  # Sympathetic uveitis, right eye
        'H44.132',  # Sympathetic uveitis, left eye
        'H44.133',  # Sympathetic uveitis, bilateral
        'H44.19',  # Other endophthalmitis
    }
    SNOMEDCT = {
        '12371008',  # Ophthalmia nodosa (disorder)
        '57100005',  # Parasitic endophthalmitis (disorder)
        '75315001',  # Sympathetic uveitis (disorder)
        '75614007',  # Panuveitis (disorder)
        '231992006',  # Onchocerca chorioretinitis (disorder)
        '240842000',  # Ocular onchocerciasis (disorder)
        '415737009',  # Toxocara endophthalmitis (disorder)
        '766933000',  # Idiopathic panuveitis (disorder)
        '333391000119101',  # Panuveitis of right eye (disorder)
        '339001000119107',  # Panuveitis of left eye (disorder)
        '344751000119109',  # Panuveitis of bilateral eyes (disorder)
    }


class OtherProliferativeRetinopathy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of other proliferative retinopathy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of premature retinopathy.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Other Proliferative Retinopathy'
    OID = '2.16.840.1.113883.3.526.3.1480'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H35.101',  # Retinopathy of prematurity, unspecified, right eye
        'H35.102',  # Retinopathy of prematurity, unspecified, left eye
        'H35.103',  # Retinopathy of prematurity, unspecified, bilateral
        'H35.111',  # Retinopathy of prematurity, stage 0, right eye
        'H35.112',  # Retinopathy of prematurity, stage 0, left eye
        'H35.113',  # Retinopathy of prematurity, stage 0, bilateral
        'H35.121',  # Retinopathy of prematurity, stage 1, right eye
        'H35.122',  # Retinopathy of prematurity, stage 1, left eye
        'H35.123',  # Retinopathy of prematurity, stage 1, bilateral
        'H35.131',  # Retinopathy of prematurity, stage 2, right eye
        'H35.132',  # Retinopathy of prematurity, stage 2, left eye
        'H35.133',  # Retinopathy of prematurity, stage 2, bilateral
        'H35.141',  # Retinopathy of prematurity, stage 3, right eye
        'H35.142',  # Retinopathy of prematurity, stage 3, left eye
        'H35.143',  # Retinopathy of prematurity, stage 3, bilateral
        'H35.151',  # Retinopathy of prematurity, stage 4, right eye
        'H35.152',  # Retinopathy of prematurity, stage 4, left eye
        'H35.153',  # Retinopathy of prematurity, stage 4, bilateral
        'H35.161',  # Retinopathy of prematurity, stage 5, right eye
        'H35.162',  # Retinopathy of prematurity, stage 5, left eye
        'H35.163',  # Retinopathy of prematurity, stage 5, bilateral
        'H35.171',  # Retrolental fibroplasia, right eye
        'H35.172',  # Retrolental fibroplasia, left eye
        'H35.173',  # Retrolental fibroplasia, bilateral
    }
    SNOMEDCT = {
        '408847006',  # Retinopathy of prematurity stage 1 - demarcation line (disorder)
        '408848001',  # Retinopathy of prematurity stage 2 - intraretinal ridge (disorder)
        '408849009',  # Retinopathy of prematurity stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        '408850009',  # Retinopathy of prematurity stage 4 - subtotal retinal detachment (disorder)
        '408851008',  # Retinopathy of prematurity stage 5 - total retinal detachment (disorder)
        '415297005',  # Retinopathy of prematurity (disorder)
        '332371000119107',  # Retinopathy of prematurity of right eye stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        '332381000119105',  # Retinopathy of prematurity of right eye stage 2 - intraretinal ridge (disorder)
        '332391000119108',  # Retinopathy of prematurity of right eye stage 1 - demarcation line (disorder)
        '332411000119108',  # Retinopathy of prematurity of right eye (disorder)
        '337981000119108',  # Retinopathy of prematurity of left eye stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        '337991000119106',  # Retinopathy of prematurity of left eye stage 2 - intraretinal ridge (disorder)
        '338001000119102',  # Retinopathy of prematurity of left eye stage 1 - demarcation line (disorder)
        '338021000119106',  # Retinopathy of prematurity of left eye (disorder)
        '343751000119106',  # Retinopathy of prematurity of bilateral eyes stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        '343761000119108',  # Retinopathy of prematurity of bilateral eyes stage 2 - intraretinal ridge (disorder)
        '343771000119102',  # Retinopathy of prematurity of bilateral eyes stage 1 - demarcation line (disorder)
        '343791000119101',  # Retinopathy of prematurity of bilateral eyes (disorder)
    }


class OtherAndUnspecifiedFormsOfChorioretinitisAndRetinochoroiditis(ValueSet):
    """
    **Clinical Focus:** Includes concepts that represent a diagnosis of posterior cyclitis, chorioretinal inflammation, other chorioretinitis or retinitis or retinochoroiditis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes only relevant concepts associated with a diagnosis of posterior cyclitis, chorioretinal inflammation, other chorioretinitis or retinitis or retinochoroiditis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Other and Unspecified Forms of Chorioretinitis and Retinochoroiditis'
    OID = '2.16.840.1.113883.3.526.3.1468'
    DEFINITION_VERSION = '20210219'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H30.21',  # Posterior cyclitis, right eye
        'H30.22',  # Posterior cyclitis, left eye
        'H30.23',  # Posterior cyclitis, bilateral
        'H30.811',  # Harada's disease, right eye
        'H30.812',  # Harada's disease, left eye
        'H30.813',  # Harada's disease, bilateral
        'H30.891',  # Other chorioretinal inflammations, right eye
        'H30.892',  # Other chorioretinal inflammations, left eye
        'H30.893',  # Other chorioretinal inflammations, bilateral
        'H30.91',  # Unspecified chorioretinal inflammation, right eye
        'H30.92',  # Unspecified chorioretinal inflammation, left eye
        'H30.93',  # Unspecified chorioretinal inflammation, bilateral
    }
    SNOMEDCT = {
        '6313001',  # Juxtapapillary focal choroiditis AND chorioretinitis (disorder)
        '9941009',  # Congenital syphilitic choroiditis (disorder)
        '14894002',  # Generalized disseminated choroiditis AND chorioretinitis (disorder)
        '15847003',  # Focal chorioretinitis (disorder)
        '22455005',  # Cytomegaloviral retinitis (disorder)
        '36535003',  # Peripheral focal retinitis AND retinochoroiditis (disorder)
        '42031001',  # Macular focal retinitis AND retinochoroiditis (disorder)
        '44674000',  # Focal choroiditis AND chorioretinitis of other posterior pole (disorder)
        '46497009',  # Juxtapapillary focal retinitis AND retinochoroiditis (disorder)
        '46627006',  # Chorioretinitis (disorder)
        '48225000',  # Tuberculous disseminated chorioretinitis (disorder)
        '51579006',  # Disseminated choroiditis AND chorioretinitis, posterior pole (disorder)
        '51930008',  # Focal retinitis AND retinochoroiditis of other posterior pole (disorder)
        '56787009',  # Peripheral focal choroiditis AND chorioretinitis (disorder)
        '61517002',  # Metastatic disseminated retinitis AND retinochoroiditis (disorder)
        '61922005',  # Disseminated retinitis (disorder)
        '66281009',  # Secondary syphilitic chorioretinitis (disorder)
        '69811008',  # Peripheral disseminated choroiditis AND chorioretinitis (disorder)
        '77628002',  # Retinal vasculitis (disorder)
        '77939001',  # Syphilitic disseminated retinochoroiditis (disorder)
        '78769001',  # Disseminated chorioretinitis (disorder)
        '88290000',  # Focal chorioretinitis caused by acquired toxoplasmosis (disorder)
        '95682009',  # Granulomatous chorioretinitis (disorder)
        '95685006',  # Granulomatous retinitis (disorder)
        '186263000',  # Tuberculous chorioretinitis (disorder)
        '187039009',  # Histoplasma capsulatum with retinitis (disorder)
        '187194004',  # Toxoplasmosis chorioretinitis (disorder)
        '193434008',  # Focal juxtapapillary choroiditis (disorder)
        '193436005',  # Juxtapapillary focal chorioretinitis (disorder)
        '193438006',  # Peripheral focal retinochoroiditis (disorder)
        '193442009',  # Disseminated posterior pole chorioretinitis (disorder)
        '193443004',  # Disseminated peripheral chorioretinitis (disorder)
        '193444005',  # General disseminated chorioretinitis (disorder)
        '193497004',  # Vogt-Koyanagi-Harada disease (disorder)
        '231981005',  # Birdshot chorioretinitis (disorder)
        '231984002',  # Viral retinitis (disorder)
        '231988004',  # Bacterial chorioretinitis (disorder)
        '231990003',  # Candida retinitis (disorder)
        '231992006',  # Onchocerca chorioretinitis (disorder)
        '231994007',  # Toxocara chorioretinitis (disorder)
        '312491004',  # Serpiginous choroiditis (disorder)
        '312930008',  # Sarcoid chorioretinitis (disorder)
        '312934004',  # Syphilitic chorioretinitis (disorder)
        '312955002',  # Tertiary syphilitic chorioretinitis (disorder)
        '314031009',  # Acute toxoplasmosis chorioretinitis (disorder)
        '314032002',  # Inactive toxoplasmosis chorioretinitis (disorder)
        '314428001',  # Posterior cyclitis (disorder)
        '359668007',  # Focal retinitis (disorder)
        '359698000',  # Parasitic chorioretinitis (disorder)
        '359707007',  # Fungal chorioretinitis (disorder)
        '371099005',  # Generalized disseminated choroiditis AND chorioretinitis with acute necrosis (disorder)
        '373667006',  # Posterior pole focal chorioretinitis (disorder)
        '399463004',  # Retinitis (disorder)
        '410470003',  # Syphilitic retinitis (disorder)
        '414783007',  # Multifocal choroiditis (disorder)
        '415218004',  # Punctate outer retinal toxoplasmosis (disorder)
        '416589006',  # Toxoplasma retinitis (disorder)
        '416877003',  # Cryptococcal chorioretinitis (disorder)
        '417405005',  # Chorioretinitis with coccidioidmycosis (disorder)
        '417547006',  # Cryptococcal retinitis (disorder)
        '421029004',  # Herpes zoster retinitis (disorder)
        '441854007',  # Reactivation of toxoplasmosis chorioretinitis (disorder)
        '866099002',  # Leber idiopathic stellate neuroretinitis (disorder)
        '866133000',  # Infectious neuroretinitis (disorder)
        '346251000119106',  # Disseminated chorioretinitis of bilateral eyes (disorder)
        '15677801000119103',  # Chorioretinitis of right eye caused by secondary syphilis (disorder)
        '15677881000119106',  # Chorioretinitis of left eye caused by secondary syphilis (disorder)
        '15685401000119104',  # Retinitis of bilateral eyes caused by Toxoplasma gondii (disorder)
        '15685441000119102',  # Retinitis of bilateral eyes caused by Treponema pallidum (disorder)
        '15685801000119101',  # Retinitis of right eye (disorder)
        '15685841000119104',  # Retinitis of left eye (disorder)
        '15685881000119109',  # Retinitis of bilateral eyes (disorder)
        '15992631000119101',  # Neuroretinitis of right eye (disorder)
        '15992671000119103',  # Neuroretinitis of left eye (disorder)
        '15992711000119104',  # Neuroretinitis of bilateral eyes (disorder)
        '15993391000119102',  # Retinitis of bilateral eyes caused by secondary syphilis (disorder)
    }


class PathologicMyopia(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis for pathologic myopia.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of degenerative disorders of eye, Forster-Fuchs' spot, malignant or severe myopia, and myopic macular degeneration.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Pathologic Myopia'
    OID = '2.16.840.1.113883.3.526.3.1432'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H44.21',  # Degenerative myopia, right eye
        'H44.22',  # Degenerative myopia, left eye
        'H44.23',  # Degenerative myopia, bilateral
        'H44.2A1',  # Degenerative myopia with choroidal neovascularization, right eye
        'H44.2A2',  # Degenerative myopia with choroidal neovascularization, left eye
        'H44.2A3',  # Degenerative myopia with choroidal neovascularization, bilateral eye
        'H44.2B1',  # Degenerative myopia with macular hole, right eye
        'H44.2B2',  # Degenerative myopia with macular hole, left eye
        'H44.2B3',  # Degenerative myopia with macular hole, bilateral eye
        'H44.2C1',  # Degenerative myopia with retinal detachment, right eye
        'H44.2C2',  # Degenerative myopia with retinal detachment, left eye
        'H44.2C3',  # Degenerative myopia with retinal detachment, bilateral eye
        'H44.2D1',  # Degenerative myopia with foveoschisis, right eye
        'H44.2D2',  # Degenerative myopia with foveoschisis, left eye
        'H44.2D3',  # Degenerative myopia with foveoschisis, bilateral eye
        'H44.2E1',  # Degenerative myopia with other maculopathy, right eye
        'H44.2E2',  # Degenerative myopia with other maculopathy, left eye
        'H44.2E3',  # Degenerative myopia with other maculopathy, bilateral eye
        'H44.30',  # Unspecified degenerative disorder of globe
    }
    SNOMEDCT = {
        '3712000',  # Degenerated eye (disorder)
        '32022003',  # Degenerative progressive high myopia (disorder)
        '34187009',  # Severe myopia (disorder)
        '62585004',  # Degenerative disorder of eye (disorder)
        '193276002',  # Malignant myopia (disorder)
        '232072004',  # Forster-Fuchs' spot (disorder)
        '232135007',  # Index myopia (disorder)
        '312898002',  # Myopic macular degeneration (disorder)
        '846621001',  # Degenerative myopia of left eye (disorder)
        '335471000119108',  # Degenerative progressive high myopia of right eye (disorder)
        '346481000119100',  # Neovascularization of bilateral corneas (disorder)
        '16320791000119106',  # Myopic macular degeneration of bilateral eyes (disorder)
        '16837461000119106',  # Neovascularization of choroid of right eye due to degenerative myopia (disorder)
        '16837521000119102',  # Neovascularization of choroid of left eye due to degenerative myopia (disorder)
    }


class PosteriorLenticonus(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis for posterior lenticonus.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of posterior lenticonus, including spherophakia.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Posterior Lenticonus'
    OID = '2.16.840.1.113883.3.526.3.1433'
    DEFINITION_VERSION = '20210314'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'Q12.2',  # Coloboma of lens
        'Q12.4',  # Spherophakia
        'Q12.8',  # Other congenital lens malformations
    }
    SNOMEDCT = {
        '33521009',  # Congenital anomaly of lens shape (disorder)
        '95482000',  # Posterior lenticonus (disorder)
        '95483005',  # Posterior lentiglobus (disorder)
        '204134008',  # Coloboma of lens (disorder)
        '416671000',  # Microspherophakia (disorder)
        '418653000',  # Secondary lens coloboma (disorder)
        '419544009',  # Primary lens coloboma (disorder)
        '773690008',  # Microcornea, posterior megalolenticonus, persistent fetal vasculature, coloboma syndrome (disorder)
        '840458000',  # Coloboma of bilateral lenses (disorder)
        '15987871000119100',  # Congenital coloboma of right lens (disorder)
        '15987911000119102',  # Congenital coloboma of left lens (disorder)
    }


class PriorPenetratingKeratoplasty(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of penetrating keratoplasty.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of keratoconus.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Prior Penetrating Keratoplasty'
    OID = '2.16.840.1.113883.3.526.3.1475'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H18.601',  # Keratoconus, unspecified, right eye
        'H18.602',  # Keratoconus, unspecified, left eye
        'H18.603',  # Keratoconus, unspecified, bilateral
        'H18.611',  # Keratoconus, stable, right eye
        'H18.612',  # Keratoconus, stable, left eye
        'H18.613',  # Keratoconus, stable, bilateral
        'H18.621',  # Keratoconus, unstable, right eye
        'H18.622',  # Keratoconus, unstable, left eye
        'H18.623',  # Keratoconus, unstable, bilateral
    }
    SNOMEDCT = {
        '63216003',  # Congenital keratoconus (disorder)
        '65636009',  # Keratoconus (disorder)
        '75340008',  # Congenital keratoconus posticus circumscriptus (disorder)
        '111523009',  # Acute hydrops keratoconus (disorder)
        '193844000',  # Keratoconus, stable condition (disorder)
        '314758008',  # Posterior keratoconus (disorder)
        '700281002',  # Unstable keratoconus (disorder)
        '334551000119102',  # Unstable keratoconus of right eye (disorder)
        '334561000119100',  # Stable keratoconus of right eye (disorder)
        '334571000119106',  # Keratoconus of right cornea (disorder)
        '340151000119101',  # Unstable keratoconus of left eye (disorder)
        '340161000119104',  # Stable keratoconus of left eye (disorder)
        '340171000119105',  # Keratoconus of left cornea (disorder)
        '345511000119103',  # Unstable keratoconus of bilateral eyes (disorder)
        '345521000119105',  # Stable keratoconus of bilateral eyes (disorder)
        '345531000119108',  # Keratoconus of bilateral corneas (disorder)
        '348371000119109',  # Acute hydrops keratoconus of right eye (disorder)
        '348821000119101',  # Acute hydrops keratoconus of left eye (disorder)
        '349321000119104',  # Acute hydrops keratoconus of bilateral eyes (disorder)
    }


class PurulentEndophthalmitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for purulent endophthalmitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of purulent endophthalmitis, panophthalmitis, vitreous abscess, and other types of endophthalmitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Purulent Endophthalmitis'
    OID = '2.16.840.1.113883.3.526.3.1477'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H44.001',  # Unspecified purulent endophthalmitis, right eye
        'H44.002',  # Unspecified purulent endophthalmitis, left eye
        'H44.003',  # Unspecified purulent endophthalmitis, bilateral
        'H44.011',  # Panophthalmitis (acute), right eye
        'H44.012',  # Panophthalmitis (acute), left eye
        'H44.013',  # Panophthalmitis (acute), bilateral
        'H44.021',  # Vitreous abscess (chronic), right eye
        'H44.022',  # Vitreous abscess (chronic), left eye
        'H44.023',  # Vitreous abscess (chronic), bilateral
    }
    SNOMEDCT = {
        '1493002',  # Acute endophthalmitis (disorder)
        '1833005',  # Phacoanaphylactic endophthalmitis (disorder)
        '1847009',  # Endophthalmitis (disorder)
        '13978000',  # Chronic endophthalmitis (disorder)
        '41720003',  # Purulent endophthalmitis (disorder)
        '314544008',  # Postoperative endophthalmitis (disorder)
        '314545009',  # Acute postoperative endophthalmitis (disorder)
        '314546005',  # Chronic postoperative endophthalmitis (disorder)
        '314547001',  # Post-traumatic endophthalmitis (disorder)
        '314548006',  # Endogenous endophthalmitis (disorder)
        '410495002',  # Postoperative endophthalmitis caused by Propionibacterium acnes (disorder)
        '410496001',  # Postoperative sterile endophthalmitis (disorder)
        '410497005',  # Postoperative infectious endophthalmitis (disorder)
        '413583003',  # Aspergillus endophthalmitis (disorder)
        '413626002',  # Bacterial endophthalmitis (disorder)
        '413665004',  # Bleb associated endophthalmitis (disorder)
        '413746000',  # Candida endophthalmitis (disorder)
        '414303007',  # Fungal endophthalmitis (disorder)
        '414491002',  # Infectious endophthalmitis (disorder)
        '415182001',  # Propionibacterium acnes endophthalmitis (disorder)
        '415584007',  # Sterile endophthalmitis (disorder)
        '415737009',  # Toxocara endophthalmitis (disorder)
        '416296007',  # Cryptococcal endophthalmitis (disorder)
        '348381000119107',  # Acute endophthalmitis of right eye (disorder)
        '348831000119103',  # Acute endophthalmitis of left eye (disorder)
        '15678441000119103',  # Chronic endophthalmitis of left eye (disorder)
        '15678481000119108',  # Chronic endophthalmitis of right eye (disorder)
    }


class RetinalDetachmentWithRetinalDefect(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis for retinal detachment with a retinal defect.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of various types of retinal detachments.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Retinal Detachment with Retinal Defect'
    OID = '2.16.840.1.113883.3.526.3.1478'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H33.001',  # Unspecified retinal detachment with retinal break, right eye
        'H33.002',  # Unspecified retinal detachment with retinal break, left eye
        'H33.003',  # Unspecified retinal detachment with retinal break, bilateral
        'H33.011',  # Retinal detachment with single break, right eye
        'H33.012',  # Retinal detachment with single break, left eye
        'H33.013',  # Retinal detachment with single break, bilateral
        'H33.021',  # Retinal detachment with multiple breaks, right eye
        'H33.022',  # Retinal detachment with multiple breaks, left eye
        'H33.023',  # Retinal detachment with multiple breaks, bilateral
        'H33.031',  # Retinal detachment with giant retinal tear, right eye
        'H33.032',  # Retinal detachment with giant retinal tear, left eye
        'H33.033',  # Retinal detachment with giant retinal tear, bilateral
        'H33.041',  # Retinal detachment with retinal dialysis, right eye
        'H33.042',  # Retinal detachment with retinal dialysis, left eye
        'H33.043',  # Retinal detachment with retinal dialysis, bilateral
        'H33.051',  # Total retinal detachment, right eye
        'H33.052',  # Total retinal detachment, left eye
        'H33.053',  # Total retinal detachment, bilateral
        'H33.8',  # Other retinal detachments
    }
    SNOMEDCT = {
        '3598000',  # Partial recent retinal detachment with single defect (disorder)
        '4178006',  # Partial recent retinal detachment with multiple defects (disorder)
        '7219007',  # Traction detachment with vitreoretinal organization (disorder)
        '19620000',  # Rhegmatogenous retinal detachment (disorder)
        '34711008',  # Traction detachment of retina (disorder)
        '38579007',  # Partial recent retinal detachment with retinal dialysis (disorder)
        '38599001',  # Serous retinal detachment (disorder)
        '42059000',  # Retinal detachment (disorder)
        '56202001',  # Retinal detachment with retinal defect (disorder)
        '193319001',  # New partial retinal detachment with single defect (disorder)
        '193320007',  # New partial retinal detachment with multiple defects (disorder)
        '193322004',  # New partial retinal detachment with retinal dialysis (disorder)
        '193323009',  # Recent total retinal detachment (disorder)
        '193324003',  # Recent subtotal retinal detachment (disorder)
        '193325002',  # Old partial retinal detachment (disorder)
        '193326001',  # Old total retinal detachment (disorder)
        '193327005',  # Old subtotal retinal detachment (disorder)
        '232008001',  # Rhegmatogenous retinal detachment - macula on (disorder)
        '232009009',  # Rhegmatogenous retinal detachment - macula off (disorder)
        '232010004',  # Traction retinal detachment involving macula (disorder)
        '232011000',  # Traction retinal detachment sparing macula (disorder)
        '232023006',  # Traction detachment of retina due to diabetes mellitus (disorder)
        '247165009',  # Retinal detachment - subretinal fluid (disorder)
        '314494006',  # Pseudophakic retinal detachment (disorder)
        '331831000119109',  # Total detachment of right retina (disorder)
        '337451000119106',  # Total detachment of left retina (disorder)
    }


class RetinalVascularOcclusion(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of retinal vascular occlusion.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of retinal vein or artery occlusion.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Retinal Vascular Occlusion'
    OID = '2.16.840.1.113883.3.526.3.1479'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H34.11',  # Central retinal artery occlusion, right eye
        'H34.12',  # Central retinal artery occlusion, left eye
        'H34.13',  # Central retinal artery occlusion, bilateral
        'H34.231',  # Retinal artery branch occlusion, right eye
        'H34.232',  # Retinal artery branch occlusion, left eye
        'H34.233',  # Retinal artery branch occlusion, bilateral
        'H34.8110',  # Central retinal vein occlusion, right eye, with macular edema
        'H34.8111',  # Central retinal vein occlusion, right eye, with retinal neovascularization
        'H34.8112',  # Central retinal vein occlusion, right eye, stable
        'H34.8120',  # Central retinal vein occlusion, left eye, with macular edema
        'H34.8121',  # Central retinal vein occlusion, left eye, with retinal neovascularization
        'H34.8122',  # Central retinal vein occlusion, left eye, stable
        'H34.8130',  # Central retinal vein occlusion, bilateral, with macular edema
        'H34.8131',  # Central retinal vein occlusion, bilateral, with retinal neovascularization
        'H34.8132',  # Central retinal vein occlusion, bilateral, stable
        'H34.8310',  # Tributary (branch) retinal vein occlusion, right eye, with macular edema
        'H34.8311',  # Tributary (branch) retinal vein occlusion, right eye, with retinal neovascularization
        'H34.8312',  # Tributary (branch) retinal vein occlusion, right eye, stable
        'H34.8320',  # Tributary (branch) retinal vein occlusion, left eye, with macular edema
        'H34.8321',  # Tributary (branch) retinal vein occlusion, left eye, with retinal neovascularization
        'H34.8322',  # Tributary (branch) retinal vein occlusion, left eye, stable
        'H34.8330',  # Tributary (branch) retinal vein occlusion, bilateral, with macular edema
        'H34.8331',  # Tributary (branch) retinal vein occlusion, bilateral, with retinal neovascularization
        'H34.8332',  # Tributary (branch) retinal vein occlusion, bilateral, stable
    }
    SNOMEDCT = {
        '776009',  # Partial occlusion of retinal artery (disorder)
        '24596005',  # Venous retinal branch occlusion (disorder)
        '38742007',  # Central retinal artery occlusion (disorder)
        '50821009',  # Arterial retinal branch occlusion (disorder)
        '65593009',  # Partial occlusion of retinal vein (disorder)
        '68478007',  # Central retinal vein occlusion (disorder)
        '73757007',  # Retinal vascular occlusion (disorder)
        '87224000',  # Transient arterial retinal occlusion (disorder)
        '232035005',  # Retinal artery occlusion (disorder)
        '232037002',  # Branch macular artery occlusion (disorder)
        '232038007',  # Central retinal vein occlusion with neovascularization (disorder)
        '232039004',  # Central retinal vein occlusion with macular edema (disorder)
        '232040002',  # Central retinal vein occlusion - juvenile (disorder)
        '232041003',  # Central retinal vein occlusion - juvenile with neovascularization (disorder)
        '232042005',  # Central retinal vein occlusion - juvenile with macular edema (disorder)
        '232043000',  # Hemispheric retinal vein occlusion (disorder)
        '232044006',  # Hemispheric retinal vein occlusion with neovascularization (disorder)
        '232045007',  # Hemispheric retinal vein occlusion with macular edema (disorder)
        '232046008',  # Branch retinal vein occlusion with neovascularization (disorder)
        '232048009',  # Branch retinal vein occlusion with macular edema (disorder)
        '247121001',  # Macular branch retinal vein occlusion (disorder)
        '312997008',  # Central retinal vein occlusion - ischemic (disorder)
        '312998003',  # Central retinal vein occlusion - non-ischemic (disorder)
        '314000002',  # Branch retinal vein occlusion with no neovascularization (disorder)
        '333281000119108',  # Partial occlusion of right retinal artery (disorder)
        '336191000119105',  # Occlusion of right central retinal artery (disorder)
        '337421000119103',  # Transient occlusion of left retinal artery (disorder)
        '338891000119100',  # Partial occlusion of left retinal artery (disorder)
        '341801000119101',  # Occlusion of left central retinal artery (disorder)
        '677771000119102',  # Occlusion of bilateral branch retinal arteries (disorder)
        '677781000119104',  # Occlusion of left branch retinal artery (disorder)
        '677791000119101',  # Occlusion of right branch retinal artery (disorder)
        '677821000119109',  # Partial occlusion of left retinal vein (disorder)
        '677831000119107',  # Partial occlusion of right retinal vein (disorder)
        '680941000119100',  # Partial occlusion of bilateral retinal veins (disorder)
        '15997911000119103',  # Vascular occlusion of retina of left eye (disorder)
        '15997951000119102',  # Vascular occlusion of retina of right eye (disorder)
        '16279401000119108',  # Occlusion of right cilioretinal artery (disorder)
        '16279441000119105',  # Occlusion of left cilioretinal artery (disorder)
        '16711111000119100',  # Retinal neovascularization due to occlusion of bilateral central retinal veins (disorder)
        '16711631000119103',  # Retinal neovascularization due to occlusion of branch of retinal vein of bilateral eyes (disorder)
    }


class RetrolentalFibroplasias(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis for retrolental fibroplasias.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of retrolental fibroplasias, including retinopathy of prematurity.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Retrolental Fibroplasias'
    OID = '2.16.840.1.113883.3.526.3.1438'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H35.171',  # Retrolental fibroplasia, right eye
        'H35.172',  # Retrolental fibroplasia, left eye
        'H35.173',  # Retrolental fibroplasia, bilateral
    }
    SNOMEDCT = {
        '408847006',  # Retinopathy of prematurity stage 1 - demarcation line (disorder)
        '408848001',  # Retinopathy of prematurity stage 2 - intraretinal ridge (disorder)
        '408849009',  # Retinopathy of prematurity stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        '408850009',  # Retinopathy of prematurity stage 4 - subtotal retinal detachment (disorder)
        '408851008',  # Retinopathy of prematurity stage 5 - total retinal detachment (disorder)
        '415297005',  # Retinopathy of prematurity (disorder)
        '332371000119107',  # Retinopathy of prematurity of right eye stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        '332381000119105',  # Retinopathy of prematurity of right eye stage 2 - intraretinal ridge (disorder)
        '332391000119108',  # Retinopathy of prematurity of right eye stage 1 - demarcation line (disorder)
        '332411000119108',  # Retinopathy of prematurity of right eye (disorder)
        '337981000119108',  # Retinopathy of prematurity of left eye stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        '337991000119106',  # Retinopathy of prematurity of left eye stage 2 - intraretinal ridge (disorder)
        '338001000119102',  # Retinopathy of prematurity of left eye stage 1 - demarcation line (disorder)
        '338021000119106',  # Retinopathy of prematurity of left eye (disorder)
        '343751000119106',  # Retinopathy of prematurity of bilateral eyes stage 3 - ridge with extraretinal fibrovascular proliferation (disorder)
        '343761000119108',  # Retinopathy of prematurity of bilateral eyes stage 2 - intraretinal ridge (disorder)
        '343771000119102',  # Retinopathy of prematurity of bilateral eyes stage 1 - demarcation line (disorder)
        '343791000119101',  # Retinopathy of prematurity of bilateral eyes (disorder)
    }


class Scleritis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for scleritis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of specific types of scleritis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Scleritis'
    OID = '2.16.840.1.113762.1.4.1226.1'
    DEFINITION_VERSION = '20210308'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H15.021',  # Brawny scleritis, right eye
        'H15.022',  # Brawny scleritis, left eye
        'H15.023',  # Brawny scleritis, bilateral
        'H15.031',  # Posterior scleritis, right eye
        'H15.032',  # Posterior scleritis, left eye
        'H15.033',  # Posterior scleritis, bilateral
        'H15.041',  # Scleritis with corneal involvement, right eye
        'H15.042',  # Scleritis with corneal involvement, left eye
        'H15.043',  # Scleritis with corneal involvement, bilateral
        'H15.051',  # Scleromalacia perforans, right eye
        'H15.052',  # Scleromalacia perforans, left eye
        'H15.053',  # Scleromalacia perforans, bilateral
        'H15.091',  # Other scleritis, right eye
        'H15.092',  # Other scleritis, left eye
        'H15.093',  # Other scleritis, bilateral
    }
    SNOMEDCT = {
        '26664005',  # Scleromalacia perforans (disorder)
        '27886001',  # Sclerosing keratitis (disorder)
        '42574005',  # Scleritis with corneal involvement (disorder)
        '50675000',  # Scleral abscess (disorder)
        '59165007',  # Rheumatoid scleritis (disorder)
        '78370002',  # Scleritis (disorder)
        '91612009',  # Brawny scleritis (disorder)
        '95195003',  # Nodular scleritis (disorder)
        '95680001',  # Sclerouveitis (disorder)
        '95795006',  # Superficial scleritis (disorder)
        '95796007',  # Deep scleritis (disorder)
        '231876002',  # Non-pyogenic granulomatous scleritis (disorder)
        '267660007',  # Posterior scleritis (disorder)
        '410578007',  # Infective scleritis (disorder)
        '416879000',  # Scleritis due to granulomatosis with polyangiitis (disorder)
        '417290008',  # Sclerokeratitis (disorder)
        '332981000119106',  # Posterior scleritis of right eye (disorder)
        '338591000119103',  # Posterior scleritis of left eye (disorder)
        '344361000119105',  # Posterior scleritis of bilateral eyes (disorder)
    }


class SeparationOfRetinalLayers(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for separation of the layers of the retina.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of serous chorioretinopathy and types of retinal detachment.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Separation of Retinal Layers'
    OID = '2.16.840.1.113883.3.526.3.1482'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H35.711',  # Central serous chorioretinopathy, right eye
        'H35.712',  # Central serous chorioretinopathy, left eye
        'H35.713',  # Central serous chorioretinopathy, bilateral
        'H35.721',  # Serous detachment of retinal pigment epithelium, right eye
        'H35.722',  # Serous detachment of retinal pigment epithelium, left eye
        'H35.723',  # Serous detachment of retinal pigment epithelium, bilateral
        'H35.731',  # Hemorrhagic detachment of retinal pigment epithelium, right eye
        'H35.732',  # Hemorrhagic detachment of retinal pigment epithelium, left eye
        'H35.733',  # Hemorrhagic detachment of retinal pigment epithelium, bilateral
    }
    SNOMEDCT = {
        '3598000',  # Partial recent retinal detachment with single defect (disorder)
        '4178006',  # Partial recent retinal detachment with multiple defects (disorder)
        '7219007',  # Traction detachment with vitreoretinal organization (disorder)
        '19620000',  # Rhegmatogenous retinal detachment (disorder)
        '34711008',  # Traction detachment of retina (disorder)
        '38579007',  # Partial recent retinal detachment with retinal dialysis (disorder)
        '38599001',  # Serous retinal detachment (disorder)
        '42059000',  # Retinal detachment (disorder)
        '51987004',  # Retinal detachment without retinal defect (disorder)
        '56202001',  # Retinal detachment with retinal defect (disorder)
        '193319001',  # New partial retinal detachment with single defect (disorder)
        '193320007',  # New partial retinal detachment with multiple defects (disorder)
        '193322004',  # New partial retinal detachment with retinal dialysis (disorder)
        '193323009',  # Recent total retinal detachment (disorder)
        '193324003',  # Recent subtotal retinal detachment (disorder)
        '193325002',  # Old partial retinal detachment (disorder)
        '193326001',  # Old total retinal detachment (disorder)
        '193327005',  # Old subtotal retinal detachment (disorder)
        '232004004',  # Giant retinal tear (disorder)
        '232008001',  # Rhegmatogenous retinal detachment - macula on (disorder)
        '232009009',  # Rhegmatogenous retinal detachment - macula off (disorder)
        '232010004',  # Traction retinal detachment involving macula (disorder)
        '232011000',  # Traction retinal detachment sparing macula (disorder)
        '232012007',  # Combined traction and rhegmatogenous retinal detachment (disorder)
        '232015009',  # Solid retinal detachment (disorder)
        '232023006',  # Traction detachment of retina due to diabetes mellitus (disorder)
        '232034009',  # Central serous retinopathy with pit of optic disc (disorder)
        '247165009',  # Retinal detachment - subretinal fluid (disorder)
        '312923002',  # Chronic central serous chorioretinopathy (disorder)
        '312924008',  # Inactive central serous chorioretinopathy (disorder)
        '312947009',  # Acute central serous chorioretinopathy (disorder)
        '312956001',  # Central serous chorioretinopathy (disorder)
        '312957005',  # Variant central serous chorioretinopathy (disorder)
        '314006008',  # Acute central serous retinopathy with subretinal fluid (disorder)
        '314007004',  # Chronic central serous retinopathy with diffuse retinal pigment epithelial detachment (disorder)
        '314008009',  # Inactive central serous retinopathy with focal retinal pigment epithelial detachment (disorder)
        '314009001',  # Central serous retinopathy with small retinal pigment epithelial detachment (disorder)
        '334891000119105',  # Hemorrhagic detachment of right retinal pigment epithelium (disorder)
        '340501000119106',  # Hemorrhagic detachment of left retinal pigment epithelium (disorder)
        '345811000119101',  # Hemorrhagic detachment of bilateral retinal pigment epithelium (disorder)
    }


class TraumaticCataract(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for traumatic cataract.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of traumatic cataract, including concussion cataract.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Traumatic Cataract'
    OID = '2.16.840.1.113883.3.526.3.1443'
    DEFINITION_VERSION = '20210219'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H26.101',  # Unspecified traumatic cataract, right eye
        'H26.102',  # Unspecified traumatic cataract, left eye
        'H26.103',  # Unspecified traumatic cataract, bilateral
        'H26.111',  # Localized traumatic opacities, right eye
        'H26.112',  # Localized traumatic opacities, left eye
        'H26.113',  # Localized traumatic opacities, bilateral
        'H26.121',  # Partially resolved traumatic cataract, right eye
        'H26.122',  # Partially resolved traumatic cataract, left eye
        'H26.123',  # Partially resolved traumatic cataract, bilateral
        'H26.131',  # Total traumatic cataract, right eye
        'H26.132',  # Total traumatic cataract, left eye
        'H26.133',  # Total traumatic cataract, bilateral
    }
    SNOMEDCT = {
        '8656007',  # Total traumatic cataract (disorder)
        '34361001',  # Traumatic cataract (disorder)
        '193598007',  # Partial resolved traumatic cataract (disorder)
        '231967008',  # Concussion cataract (disorder)
        '231968003',  # Cataract following rupture of capsule (disorder)
        '866114005',  # Localized cataract opacities due to and following traumatic injury (disorder)
    }


class Uveitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts  for diagnoses of uveitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of uveitis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Uveitis'
    OID = '2.16.840.1.113883.3.526.3.1444'
    DEFINITION_VERSION = '20210219'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H44.111',  # Panuveitis, right eye
        'H44.112',  # Panuveitis, left eye
        'H44.113',  # Panuveitis, bilateral
        'H44.131',  # Sympathetic uveitis, right eye
        'H44.132',  # Sympathetic uveitis, left eye
        'H44.133',  # Sympathetic uveitis, bilateral
    }
    SNOMEDCT = {
        '4927003',  # Acute anterior uveitis (disorder)
        '16553002',  # Choroiditis (disorder)
        '33382000',  # Panophthalmitis (disorder)
        '43363007',  # Posterior uveitis (disorder)
        '64015001',  # Subacute anterior uveitis (disorder)
        '75315001',  # Sympathetic uveitis (disorder)
        '75614007',  # Panuveitis (disorder)
        '95706000',  # Phacoantigenic uveitis (disorder)
        '95707009',  # Phacotoxic uveitis (disorder)
        '128473001',  # Uveitis (disorder)
        '186854007',  # Uveitis due to secondary syphilis (disorder)
        '231947004',  # Anterior uveitis idiopathic (disorder)
        '314029000',  # Tuberculous uveitis (disorder)
        '314429009',  # Intermediate uveitis (disorder)
        '398155003',  # Chronic anterior uveitis (disorder)
        '410473001',  # Lyme uveitis (disorder)
        '410476009',  # Uveitis due to leptospirosis (disorder)
        '410477000',  # Infective uveitis (disorder)
        '410494003',  # Lens-induced uveitis (disorder)
        '410501000',  # Drug-induced uveitis (disorder)
        '410692006',  # Anterior uveitis (disorder)
        '413716003',  # Brucellosis uveitis (disorder)
        '413851004',  # Cidofovir-induced anterior uveitis (disorder)
        '414600000',  # Lepromatous anterior uveitis (disorder)
        '414689001',  # Metipranolol-induced anterior uveitis (disorder)
        '414996002',  # Pamidronic acid-induced uveitis (disorder)
        '415344002',  # Rifabutin-induced anterior uveitis (disorder)
        '415359008',  # Sarcoid uveitis (disorder)
        '416666007',  # Anterior uveitis in juvenile idiopathic arthritis (disorder)
        '417321001',  # Human T-cell Lymphoma Virus Type -1 associated uveitis (disorder)
        '418839003',  # Tubulointerstitial nephritis with uveitis syndrome (disorder)
        '428676003',  # Uveitis caused by ranibizumab (disorder)
        '444248002',  # Chronic inflammation of uveal tract (disorder)
        '733317009',  # Primary anterior uveitis (disorder)
        '766933000',  # Idiopathic panuveitis (disorder)
        '870199008',  # Non-infectious uveitis of posterior segment of eye (disorder)
        '870273007',  # Posterior uveitis due to late syphilis (disorder)
        '870329005',  # Panuveitis caused by virus (disorder)
        '870331001',  # Panuveitis caused by Treponema pallidum (disorder)
        '870332008',  # Panuveitis caused by Herpesvirus (disorder)
        '870333003',  # Panuveitis caused by Mycobacterium tuberculosis (disorder)
        '870335005',  # Panuveitis caused by Borrelia burgdorferi (disorder)
        '336541000119107',  # Acute anterior uveitis of right eye (disorder)
        '342151000119100',  # Acute anterior uveitis of left eye (disorder)
        '347311000119100',  # Acute anterior uveitis of bilateral eyes (disorder)
    }


class VascularDisordersOfIrisAndCiliaryBody(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of vascular disorders of iris and ciliary body.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis for a vascular disorder of the iris and/or ciliary body, including rubeosis iridis.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Vascular Disorders of Iris and Ciliary Body'
    OID = '2.16.840.1.113883.3.526.3.1445'
    DEFINITION_VERSION = '20210214'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H21.1X1',  # Other vascular disorders of iris and ciliary body, right eye
        'H21.1X2',  # Other vascular disorders of iris and ciliary body, left eye
        'H21.1X3',  # Other vascular disorders of iris and ciliary body, bilateral
    }
    SNOMEDCT = {
        '51995000',  # Rubeosis iridis (disorder)
        '193501009',  # Iris and ciliary body vascular disorder (disorder)
        '678191000119101',  # Rubeosis iridis of bilateral eyes (disorder)
        '678201000119103',  # Rubeosis iridis of left eye (disorder)
        '678211000119100',  # Rubeosis iridis of right eye (disorder)
    }


class VisualFieldDefects(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for visual field defects.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of visual field defects, including scotoma and decreased field of vision.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS133v10
    """

    VALUE_SET_NAME = 'Visual Field Defects'
    OID = '2.16.840.1.113883.3.526.3.1446'
    DEFINITION_VERSION = '20210220'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H53.411',  # Scotoma involving central area, right eye
        'H53.412',  # Scotoma involving central area, left eye
        'H53.413',  # Scotoma involving central area, bilateral
        'H53.421',  # Scotoma of blind spot area, right eye
        'H53.422',  # Scotoma of blind spot area, left eye
        'H53.423',  # Scotoma of blind spot area, bilateral
        'H53.431',  # Sector or arcuate defects, right eye
        'H53.432',  # Sector or arcuate defects, left eye
        'H53.433',  # Sector or arcuate defects, bilateral
        'H53.451',  # Other localized visual field defect, right eye
        'H53.452',  # Other localized visual field defect, left eye
        'H53.453',  # Other localized visual field defect, bilateral
        'H53.461',  # Homonymous bilateral field defects, right side
        'H53.462',  # Homonymous bilateral field defects, left side
        'H53.481',  # Generalized contraction of visual field, right eye
        'H53.482',  # Generalized contraction of visual field, left eye
        'H53.483',  # Generalized contraction of visual field, bilateral
    }
    SNOMEDCT = {
        '1151008',  # Visual field constriction (finding)
        '2228002',  # Scintillating scotoma (finding)
        '4806007',  # Bjerrum scotoma (finding)
        '12184005',  # Visual field defect (finding)
        '15462009',  # Arcuate scotoma (finding)
        '19900004',  # Decreased peripheral vision (finding)
        '21273008',  # Relative scotoma (finding)
        '23388006',  # Visual field scotoma (finding)
        '24370009',  # Seidel scotoma (finding)
        '33014001',  # Centrocecal scotoma (finding)
        '33048000',  # Peripheral visual field defect (finding)
        '33970004',  # Enlarged blind spot (finding)
        '38950008',  # Central scotoma (finding)
        '49073008',  # Absolute scotoma (finding)
        '59695000',  # Bilateral visual field constriction (finding)
        '64418005',  # Paracentral scotoma (finding)
        '65133002',  # Physiologic scotoma (finding)
        '129625005',  # Ring scotoma (finding)
        '129626006',  # Quadrantic scotoma (finding)
        '193670002',  # Blind spot scotoma (finding)
        '193671003',  # Paracecal scotoma (finding)
        '193672005',  # Angioscotoma (finding)
        '246656009',  # Loss of part of visual field (finding)
        '246658005',  # Seeing blob in vision (finding)
        '246659002',  # Peripheral scotoma (finding)
        '246660007',  # Peripheral ring scotoma (finding)
        '246661006',  # Local peripheral scotoma (finding)
        '246662004',  # Altitudinal scotoma (finding)
        '267628004',  # Generalized visual field constriction (finding)
        '345121001',  # Heteronymous bilateral visual field defects (finding)
        '370961005',  # Glaucomatous visual field defect (finding)
        '404650002',  # Pericentral scotoma (finding)
        '416626008',  # Functional visual field loss (finding)
        '420990001',  # Tunnel visual field constriction (finding)
        '421561005',  # Concentric visual field constriction (finding)
        '445923001',  # Sector visual field defect (finding)
        '446020008',  # Arcuate visual field defect (finding)
        '336171000119109',  # Central scotoma of right eye (finding)
        '341781000119100',  # Central scotoma of left eye (finding)
        '346991000119107',  # Central scotoma of bilateral eyes (finding)
        '16059151000119108',  # Scotoma of left eye (finding)
        '16059191000119103',  # Scotoma of bilateral eyes (finding)
    }


class DiabeticNephropathy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  diabetic nephropathy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis for diabetic nephropathy.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS134v10
    """

    VALUE_SET_NAME = 'Diabetic Nephropathy'
    OID = '2.16.840.1.113883.3.464.1003.109.12.1004'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'E08.21',  # Diabetes mellitus due to underlying condition with diabetic nephropathy
        'E08.22',  # Diabetes mellitus due to underlying condition with diabetic chronic kidney disease
        'E08.29',  # Diabetes mellitus due to underlying condition with other diabetic kidney complication
        'E09.21',  # Drug or chemical induced diabetes mellitus with diabetic nephropathy
        'E09.22',  # Drug or chemical induced diabetes mellitus with diabetic chronic kidney disease
        'E09.29',  # Drug or chemical induced diabetes mellitus with other diabetic kidney complication
        'E10.21',  # Type 1 diabetes mellitus with diabetic nephropathy
        'E10.22',  # Type 1 diabetes mellitus with diabetic chronic kidney disease
        'E10.29',  # Type 1 diabetes mellitus with other diabetic kidney complication
        'E11.21',  # Type 2 diabetes mellitus with diabetic nephropathy
        'E11.22',  # Type 2 diabetes mellitus with diabetic chronic kidney disease
        'E11.29',  # Type 2 diabetes mellitus with other diabetic kidney complication
        'E13.21',  # Other specified diabetes mellitus with diabetic nephropathy
        'E13.22',  # Other specified diabetes mellitus with diabetic chronic kidney disease
        'E13.29',  # Other specified diabetes mellitus with other diabetic kidney complication
    }
    ICD9CM = {
        '250.41',  # Diabetes with renal manifestations, type I [juvenile type], not stated as uncontrolled
        '250.42',  # Diabetes with renal manifestations, type II or unspecified type, uncontrolled
        '250.43',  # Diabetes with renal manifestations, type I [juvenile type], uncontrolled
    }
    SNOMEDCT = {
        '38046004',  # Diffuse glomerulosclerosis of kidney due to diabetes mellitus (disorder)
        '54181000',  # Diabetes-nephrosis syndrome (disorder)
        '127013003',  # Disorder of kidney due to diabetes mellitus (disorder)
        '401110002',  # Type 1 diabetes mellitus with persistent microalbuminuria (disorder)
        '420715001',  # Persistent microalbuminuria due to type 2 diabetes mellitus (disorder)
        '421986006',  # Persistent proteinuria due to type 2 diabetes mellitus (disorder)
        '445170001',  # Macroalbuminuric nephropathy due to diabetes mellitus (disorder)
        '792926007',  # Armanni-Ebstein kidney due to diabetes mellitus (disorder)
        '71701000119105',  # Hypertension in chronic kidney disease due to type 1 diabetes mellitus (disorder)
        '90721000119101',  # Chronic kidney disease stage 1 due to type 1 diabetes mellitus (disorder)
        '90731000119103',  # Chronic kidney disease stage 2 due to type 1 diabetes mellitus (disorder)
        '90741000119107',  # Chronic kidney disease stage 3 due to type 1 diabetes mellitus (disorder)
        '90751000119109',  # Chronic kidney disease stage 4 due to type 1 diabetes mellitus (disorder)
        '90761000119106',  # Chronic kidney disease stage 5 due to type 1 diabetes mellitus (disorder)
        '90771000119100',  # End stage renal disease on dialysis due to type 1 diabetes mellitus (disorder)
        '96441000119101',  # Chronic kidney disease due to type 1 diabetes mellitus (disorder)
    }


class GlomerulonephritisAndNephroticSyndrome(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  glomerulonephritis and nephrotic syndrome.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of glomerulonephritis and nephrotic syndrome.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS134v10
    """

    VALUE_SET_NAME = 'Glomerulonephritis and Nephrotic Syndrome'
    OID = '2.16.840.1.113883.3.464.1003.109.12.1018'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'N00.0',  # Acute nephritic syndrome with minor glomerular abnormality
        'N00.1',  # Acute nephritic syndrome with focal and segmental glomerular lesions
        'N00.2',  # Acute nephritic syndrome with diffuse membranous glomerulonephritis
        'N00.3',  # Acute nephritic syndrome with diffuse mesangial proliferative glomerulonephritis
        'N00.4',  # Acute nephritic syndrome with diffuse endocapillary proliferative glomerulonephritis
        'N00.5',  # Acute nephritic syndrome with diffuse mesangiocapillary glomerulonephritis
        'N00.6',  # Acute nephritic syndrome with dense deposit disease
        'N00.7',  # Acute nephritic syndrome with diffuse crescentic glomerulonephritis
        'N00.8',  # Acute nephritic syndrome with other morphologic changes
        'N00.9',  # Acute nephritic syndrome with unspecified morphologic changes
        'N00.A',  # Acute nephritic syndrome with C3 glomerulonephritis
        'N01.0',  # Rapidly progressive nephritic syndrome with minor glomerular abnormality
        'N01.1',  # Rapidly progressive nephritic syndrome with focal and segmental glomerular lesions
        'N01.2',  # Rapidly progressive nephritic syndrome with diffuse membranous glomerulonephritis
        'N01.3',  # Rapidly progressive nephritic syndrome with diffuse mesangial proliferative glomerulonephritis
        'N01.4',  # Rapidly progressive nephritic syndrome with diffuse endocapillary proliferative glomerulonephritis
        'N01.5',  # Rapidly progressive nephritic syndrome with diffuse mesangiocapillary glomerulonephritis
        'N01.6',  # Rapidly progressive nephritic syndrome with dense deposit disease
        'N01.7',  # Rapidly progressive nephritic syndrome with diffuse crescentic glomerulonephritis
        'N01.8',  # Rapidly progressive nephritic syndrome with other morphologic changes
        'N01.9',  # Rapidly progressive nephritic syndrome with unspecified morphologic changes
        'N01.A',  # Rapidly progressive nephritic syndrome with C3 glomerulonephritis
        'N02.0',  # Recurrent and persistent hematuria with minor glomerular abnormality
        'N02.1',  # Recurrent and persistent hematuria with focal and segmental glomerular lesions
        'N02.2',  # Recurrent and persistent hematuria with diffuse membranous glomerulonephritis
        'N02.3',  # Recurrent and persistent hematuria with diffuse mesangial proliferative glomerulonephritis
        'N02.4',  # Recurrent and persistent hematuria with diffuse endocapillary proliferative glomerulonephritis
        'N02.5',  # Recurrent and persistent hematuria with diffuse mesangiocapillary glomerulonephritis
        'N02.6',  # Recurrent and persistent hematuria with dense deposit disease
        'N02.7',  # Recurrent and persistent hematuria with diffuse crescentic glomerulonephritis
        'N02.8',  # Recurrent and persistent hematuria with other morphologic changes
        'N02.9',  # Recurrent and persistent hematuria with unspecified morphologic changes
        'N02.A',  # Recurrent and persistent hematuria with C3 glomerulonephritis
        'N03.0',  # Chronic nephritic syndrome with minor glomerular abnormality
        'N03.1',  # Chronic nephritic syndrome with focal and segmental glomerular lesions
        'N03.2',  # Chronic nephritic syndrome with diffuse membranous glomerulonephritis
        'N03.3',  # Chronic nephritic syndrome with diffuse mesangial proliferative glomerulonephritis
        'N03.4',  # Chronic nephritic syndrome with diffuse endocapillary proliferative glomerulonephritis
        'N03.5',  # Chronic nephritic syndrome with diffuse mesangiocapillary glomerulonephritis
        'N03.6',  # Chronic nephritic syndrome with dense deposit disease
        'N03.7',  # Chronic nephritic syndrome with diffuse crescentic glomerulonephritis
        'N03.8',  # Chronic nephritic syndrome with other morphologic changes
        'N03.9',  # Chronic nephritic syndrome with unspecified morphologic changes
        'N03.A',  # Chronic nephritic syndrome with C3 glomerulonephritis
        'N04.0',  # Nephrotic syndrome with minor glomerular abnormality
        'N04.1',  # Nephrotic syndrome with focal and segmental glomerular lesions
        'N04.2',  # Nephrotic syndrome with diffuse membranous glomerulonephritis
        'N04.3',  # Nephrotic syndrome with diffuse mesangial proliferative glomerulonephritis
        'N04.4',  # Nephrotic syndrome with diffuse endocapillary proliferative glomerulonephritis
        'N04.5',  # Nephrotic syndrome with diffuse mesangiocapillary glomerulonephritis
        'N04.6',  # Nephrotic syndrome with dense deposit disease
        'N04.7',  # Nephrotic syndrome with diffuse crescentic glomerulonephritis
        'N04.8',  # Nephrotic syndrome with other morphologic changes
        'N04.9',  # Nephrotic syndrome with unspecified morphologic changes
        'N04.A',  # Nephrotic syndrome with C3 glomerulonephritis
        'N05.0',  # Unspecified nephritic syndrome with minor glomerular abnormality
        'N05.1',  # Unspecified nephritic syndrome with focal and segmental glomerular lesions
        'N05.2',  # Unspecified nephritic syndrome with diffuse membranous glomerulonephritis
        'N05.3',  # Unspecified nephritic syndrome with diffuse mesangial proliferative glomerulonephritis
        'N05.4',  # Unspecified nephritic syndrome with diffuse endocapillary proliferative glomerulonephritis
        'N05.5',  # Unspecified nephritic syndrome with diffuse mesangiocapillary glomerulonephritis
        'N05.6',  # Unspecified nephritic syndrome with dense deposit disease
        'N05.7',  # Unspecified nephritic syndrome with diffuse crescentic glomerulonephritis
        'N05.8',  # Unspecified nephritic syndrome with other morphologic changes
        'N05.9',  # Unspecified nephritic syndrome with unspecified morphologic changes
        'N05.A',  # Unspecified nephritic syndrome with C3 glomerulonephritis
        'N06.0',  # Isolated proteinuria with minor glomerular abnormality
        'N06.1',  # Isolated proteinuria with focal and segmental glomerular lesions
        'N06.2',  # Isolated proteinuria with diffuse membranous glomerulonephritis
        'N06.3',  # Isolated proteinuria with diffuse mesangial proliferative glomerulonephritis
        'N06.4',  # Isolated proteinuria with diffuse endocapillary proliferative glomerulonephritis
        'N06.5',  # Isolated proteinuria with diffuse mesangiocapillary glomerulonephritis
        'N06.6',  # Isolated proteinuria with dense deposit disease
        'N06.7',  # Isolated proteinuria with diffuse crescentic glomerulonephritis
        'N06.8',  # Isolated proteinuria with other morphologic lesion
        'N06.9',  # Isolated proteinuria with unspecified morphologic lesion
        'N06.A',  # Isolated proteinuria with C3 glomerulonephritis
        'N07.0',  # Hereditary nephropathy, not elsewhere classified with minor glomerular abnormality
        'N07.1',  # Hereditary nephropathy, not elsewhere classified with focal and segmental glomerular lesions
        'N07.2',  # Hereditary nephropathy, not elsewhere classified with diffuse membranous glomerulonephritis
        'N07.3',  # Hereditary nephropathy, not elsewhere classified with diffuse mesangial proliferative glomerulonephritis
        'N07.4',  # Hereditary nephropathy, not elsewhere classified with diffuse endocapillary proliferative glomerulonephritis
        'N07.5',  # Hereditary nephropathy, not elsewhere classified with diffuse mesangiocapillary glomerulonephritis
        'N07.6',  # Hereditary nephropathy, not elsewhere classified with dense deposit disease
        'N07.7',  # Hereditary nephropathy, not elsewhere classified with diffuse crescentic glomerulonephritis
        'N07.8',  # Hereditary nephropathy, not elsewhere classified with other morphologic lesions
        'N07.9',  # Hereditary nephropathy, not elsewhere classified with unspecified morphologic lesions
        'N07.A',  # Hereditary nephropathy, not elsewhere classified with C3 glomerulonephritis
        'N08',  # Glomerular disorders in diseases classified elsewhere
        'N14.0',  # Analgesic nephropathy
        'N14.1',  # Nephropathy induced by other drugs, medicaments and biological substances
        'N14.2',  # Nephropathy induced by unspecified drug, medicament or biological substance
        'N14.3',  # Nephropathy induced by heavy metals
        'N14.4',  # Toxic nephropathy, not elsewhere classified
        'N25.0',  # Renal osteodystrophy
        'N25.1',  # Nephrogenic diabetes insipidus
        'N25.81',  # Secondary hyperparathyroidism of renal origin
        'N25.89',  # Other disorders resulting from impaired renal tubular function
        'N25.9',  # Disorder resulting from impaired renal tubular function, unspecified
        'N26.1',  # Atrophy of kidney (terminal)
        'N26.2',  # Page kidney
        'N26.9',  # Renal sclerosis, unspecified
        'Q60.0',  # Renal agenesis, unilateral
        'Q60.1',  # Renal agenesis, bilateral
        'Q60.2',  # Renal agenesis, unspecified
        'Q60.3',  # Renal hypoplasia, unilateral
        'Q60.4',  # Renal hypoplasia, bilateral
        'Q60.5',  # Renal hypoplasia, unspecified
        'Q60.6',  # Potter's syndrome
        'Q61.00',  # Congenital renal cyst, unspecified
        'Q61.01',  # Congenital single renal cyst
        'Q61.02',  # Congenital multiple renal cysts
        'Q61.11',  # Cystic dilatation of collecting ducts
        'Q61.19',  # Other polycystic kidney, infantile type
        'Q61.2',  # Polycystic kidney, adult type
        'Q61.3',  # Polycystic kidney, unspecified
        'Q61.4',  # Renal dysplasia
        'Q61.5',  # Medullary cystic kidney
        'Q61.8',  # Other cystic kidney diseases
        'Q61.9',  # Cystic kidney disease, unspecified
    }
    ICD9CM = {
        '580.4',  # Acute glomerulonephritis with lesion of rapidly progressive glomerulonephritis
        '580.81',  # Acute glomerulonephritis in diseases classified elsewhere
        '580.89',  # Acute glomerulonephritis with other specified pathological lesion in kidney
        '580.9',  # Acute glomerulonephritis with unspecified pathological lesion in kidney
        '581.0',  # Nephrotic syndrome with lesion of proliferative glomerulonephritis
        '581.1',  # Nephrotic syndrome with lesion of membranous glomerulonephritis
        '581.2',  # Nephrotic syndrome with lesion of membranoproliferative glomerulonephritis
        '581.3',  # Nephrotic syndrome with lesion of minimal change glomerulonephritis
        '581.81',  # Nephrotic syndrome in diseases classified elsewhere
        '581.89',  # Nephrotic syndrome with other specified pathological lesion in kidney
        '581.9',  # Nephrotic syndrome with unspecified pathological lesion in kidney
        '582.0',  # Chronic glomerulonephritis with lesion of proliferative glomerulonephritis
        '582.1',  # Chronic glomerulonephritis with lesion of membranous glomerulonephritis
        '582.2',  # Chronic glomerulonephritis with lesion of membranoproliferative glomerulonephritis
        '582.4',  # Chronic glomerulonephritis with lesion of rapidly progressive glomerulonephritis
        '582.81',  # Chronic glomerulonephritis in diseases classified elsewhere
        '582.89',  # Chronic glomerulonephritis with other specified pathological lesion in kidney
        '582.9',  # Chronic glomerulonephritis with unspecified pathological lesion in kidney
        '583.0',  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of proliferative glomerulonephritis
        '583.1',  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of membranous glomerulonephritis
        '583.2',  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of membranoproliferative glomerulonephritis
        '583.4',  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of rapidly progressive glomerulonephritis
        '583.6',  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of renal cortical necrosis
        '583.7',  # Nephritis and nephropathy, not specified as acute or chronic, with lesion of renal medullary necrosis
        '583.81',  # Nephritis and nephropathy, not specified as acute or chronic, in diseases classified elsewhere
        '583.89',  # Nephritis and nephropathy, not specified as acute or chronic, with other specified pathological lesion in kidney
        '583.9',  # Nephritis and nephropathy, not specified as acute or chronic, with unspecified pathological lesion in kidney
        '588.0',  # Renal osteodystrophy
        '588.1',  # Nephrogenic diabetes insipidus
        '588.81',  # Secondary hyperparathyroidism (of renal origin)
        '588.89',  # Other specified disorders resulting from impaired renal function
        '588.9',  # Unspecified disorder resulting from impaired renal function
    }
    SNOMEDCT = {
        '1426004',  # Necrotizing glomerulonephritis (disorder)
        '1776003',  # Renal tubular acidosis (disorder)
        '3704008',  # Diffuse endocapillary proliferative glomerulonephritis (disorder)
        '7703001',  # Renal tubular defect (disorder)
        '7724006',  # Nephritic syndrome (disorder)
        '8199003',  # Focal embolic nephritis syndrome (disorder)
        '11480007',  # Idiopathic granulomatous interstitial nephropathy (disorder)
        '13335004',  # Sclerosing glomerulonephritis (disorder)
        '19351000',  # Acute glomerulonephritis (disorder)
        '20917003',  # Chronic glomerulonephritis (disorder)
        '22702000',  # Glomerulitis (disorder)
        '24790002',  # Proximal renal tubular acidosis (disorder)
        '25821008',  # Focal glomerular sclerosis (disorder)
        '26121002',  # Balkan nephropathy (disorder)
        '33763006',  # Hypercalcemic nephropathy (disorder)
        '35546006',  # Mesangial proliferative glomerulonephritis (disorder)
        '36171008',  # Glomerulonephritis (disorder)
        '37061001',  # Granulomatous sarcoid nephropathy (disorder)
        '37085009',  # Shunt nephritis (disorder)
        '40233000',  # Nephrotic-nephritic syndrome (disorder)
        '44785005',  # Minimal change disease (disorder)
        '45927004',  # Immunologic mediated tubulointerstitial nephritis (disorder)
        '48796009',  # Congenital nephrotic syndrome (disorder)
        '52254009',  # Nephrotic syndrome (disorder)
        '52845002',  # Nephritis (disorder)
        '53556002',  # Cis-platinum nephropathy (disorder)
        '54879000',  # Hypokalemic nephropathy (disorder)
        '55652009',  # Idiopathic crescentic glomerulonephritis, type III (disorder)
        '58574008',  # Acute nephropathy (disorder)
        '59400006',  # Analgesic nephropathy (disorder)
        '59479006',  # Mesangiocapillary glomerulonephritis, type II (disorder)
        '60926001',  # Chronic tubulointerstitial nephritis (disorder)
        '61165007',  # Hereditary nephrogenic diabetes insipidus (disorder)
        '61503006',  # Acute nephritis (disorder)
        '62240004',  # Benign arteriolar nephrosclerosis (disorder)
        '62332007',  # Infantile nephropathic cystinosis (disorder)
        '62853008',  # Anti-glomerular basement membrane tubulointerstitial nephritis (disorder)
        '64168005',  # Idiopathic crescentic glomerulonephritis, type I (disorder)
        '64212008',  # Diffuse crescentic glomerulonephritis (disorder)
        '68544003',  # Acute post-streptococcal glomerulonephritis (disorder)
        '68779003',  # Primary immunoglobulin A nephropathy (disorder)
        '73305009',  # Fibrillary glomerulonephritis (disorder)
        '75888001',  # Mesangiocapillary glomerulonephritis, type I (disorder)
        '77182004',  # Membranous glomerulonephritis (disorder)
        '78815005',  # Hereditary tubulointerstitial disorder (disorder)
        '79385002',  # Lowe syndrome (disorder)
        '80321008',  # Mesangiocapillary glomerulonephritis (disorder)
        '81363003',  # Malignant arteriolar nephrosclerosis (disorder)
        '81475007',  # Acquired nephrogenic diabetes insipidus (disorder)
        '82112005',  # Idiopathic chronic tubulointerstitial nephritis (disorder)
        '83866005',  # Focal AND segmental proliferative glomerulonephritis (disorder)
        '83923004',  # Familial interstitial nephritis (disorder)
        '90493000',  # Arteriolar nephrosclerosis (disorder)
        '110996009',  # Armanni-Ebstein kidney (disorder)
        '111395007',  # Nephrogenic diabetes insipidus (disorder)
        '118951003',  # Drug-induced nephrogenic diabetes insipidus (disorder)
        '123609007',  # Subacute glomerulonephritis (disorder)
        '123610002',  # Healed glomerulonephritis (disorder)
        '123752003',  # Immune-complex glomerulonephritis (disorder)
        '194909006',  # Acute pericarditis co-occurrent and due to uremia (disorder)
        '197579006',  # Acute proliferative glomerulonephritis (disorder)
        '197580009',  # Acute nephritis with lesions of necrotizing glomerulitis (disorder)
        '197582001',  # Acute glomerulonephritis associated with another disorder (disorder)
        '197584000',  # Acute focal nephritis (disorder)
        '197585004',  # Acute diffuse nephritis (disorder)
        '197589005',  # Nephrotic syndrome with proliferative glomerulonephritis (disorder)
        '197590001',  # Nephrotic syndrome with membranous glomerulonephritis (disorder)
        '197591002',  # Nephrotic syndrome with membranoproliferative glomerulonephritis (disorder)
        '197593004',  # Nephrotic syndrome, minor glomerular abnormality (disorder)
        '197594005',  # Nephrotic syndrome, focal and segmental glomerular lesions (disorder)
        '197595006',  # Nephrotic syndrome, diffuse membranous glomerulonephritis (disorder)
        '197596007',  # Nephrotic syndrome, diffuse mesangial proliferative glomerulonephritis (disorder)
        '197597003',  # Nephrotic syndrome, diffuse endocapillary proliferative glomerulonephritis (disorder)
        '197598008',  # Nephrotic syndrome, diffuse mesangiocapillary glomerulonephritis (disorder)
        '197599000',  # Nephrotic syndrome, dense deposit disease (disorder)
        '197600002',  # Nephrotic syndrome, diffuse crescentic glomerulonephritis (disorder)
        '197601003',  # Finnish congenital nephrotic syndrome (disorder)
        '197603000',  # Nephrotic syndrome associated with another disorder (disorder)
        '197605007',  # Nephrotic syndrome due to diabetes mellitus (disorder)
        '197607004',  # Nephrotic syndrome in polyarteritis nodosa (disorder)
        '197613008',  # Chronic mesangial proliferative glomerulonephritis (disorder)
        '197616000',  # Chronic glomerulonephritis associated with another disorder (disorder)
        '197617009',  # Chronic exudative glomerulonephritis (disorder)
        '197618004',  # Chronic focal glomerulonephritis (disorder)
        '197619007',  # Chronic diffuse glomerulonephritis (disorder)
        '197626007',  # Focal membranoproliferative glomerulonephritis (disorder)
        '197629000',  # Anaphylactoid glomerulonephritis (disorder)
        '197661001',  # Glomerulosclerosis (disorder)
        '197663003',  # Impaired renal function disorder (disorder)
        '197664009',  # Phosphate-losing tubular disorders (disorder)
        '197671004',  # Renal function impairment with growth failure (disorder)
        '197679002',  # Glomerular disease (disorder)
        '197681000',  # Acute nephritic syndrome, minor glomerular abnormality (disorder)
        '197682007',  # Acute nephritic syndrome, focal and segmental glomerular lesions (disorder)
        '197683002',  # Acute nephritic syndrome, diffuse membranous glomerulonephritis (disorder)
        '197684008',  # Acute nephritic syndrome, diffuse mesangial proliferative glomerulonephritis (disorder)
        '197685009',  # Acute nephritic syndrome, diffuse endocapillary proliferative glomerulonephritis (disorder)
        '197686005',  # Acute nephritic syndrome, diffuse mesangiocapillary glomerulonephritis (disorder)
        '197687001',  # Acute nephritic syndrome, dense deposit disease (disorder)
        '197688006',  # Acute nephritic syndrome, diffuse crescentic glomerulonephritis (disorder)
        '197689003',  # Rapidly progressive nephritic syndrome (disorder)
        '197690007',  # Rapidly progressive nephritic syndrome, minor glomerular abnormality (disorder)
        '197691006',  # Rapidly progressive nephritic syndrome, focal and segmental glomerular lesions (disorder)
        '197692004',  # Rapidly progressive nephritic syndrome, diffuse membranous glomerulonephritis (disorder)
        '197693009',  # Rapidly progressive nephritic syndrome, diffuse mesangial proliferative glomerulonephritis (disorder)
        '197694003',  # Rapidly progressive nephritic syndrome, diffuse endocapillary proliferative glomerulonephritis (disorder)
        '197695002',  # Rapidly progressive nephritic syndrome, diffuse mesangiocapillary glomerulonephritis (disorder)
        '197696001',  # Rapidly progressive nephritic syndrome, dense deposit disease (disorder)
        '197697005',  # Rapidly progressive nephritic syndrome, diffuse crescentic glomerulonephritis (disorder)
        '197707007',  # Chronic nephritic syndrome (disorder)
        '197708002',  # Chronic nephritic syndrome, minor glomerular abnormality (disorder)
        '197709005',  # Chronic nephritic syndrome, focal and segmental glomerular lesions (disorder)
        '197712008',  # Chronic nephritic syndrome, diffuse endocapillary proliferative glomerulonephritis (disorder)
        '197713003',  # Chronic nephritic syndrome, diffuse mesangiocapillary glomerulonephritis (disorder)
        '197714009',  # Chronic nephritic syndrome, dense deposit disease (disorder)
        '197715005',  # Chronic nephritic syndrome, diffuse crescentic glomerulonephritis (disorder)
        '197716006',  # Isolated proteinuria with specified morphological lesion (finding)
        '197717002',  # Isolated proteinuria with specified morphological lesion, minor glomerular abnormality (finding)
        '197719004',  # Isolated proteinuria with specified morphological lesion, focal and segmental glomerular lesions (finding)
        '197720005',  # Isolated proteinuria with specified morphological lesion, diffuse membranous glomerulonephritis (finding)
        '197721009',  # Isolated proteinuria with specified morphological lesion, diffuse mesangial proliferative glomerulonephritis (finding)
        '197722002',  # Isolated proteinuria with specified morphological lesion, diffuse endocapillary proliferative glomerulonephritis (finding)
        '197723007',  # Isolated proteinuria with specified morphological lesion, diffuse mesangiocapillary glomerulonephritis (finding)
        '197724001',  # Isolated proteinuria with specified morphological lesion, dense deposit disease (finding)
        '197725000',  # Isolated proteinuria with specified morphological lesion, diffuse concentric glomerulonephritis (finding)
        '197738008',  # Glomerular disorders in neoplastic diseases (disorder)
        '197739000',  # Glomerular disorders in blood diseases and disorders involving the immune mechanism (disorder)
        '197742006',  # Renal tubulo-interstitial disorders in neoplastic diseases (disorder)
        '197743001',  # Renal tubulo-interstitial disorders in blood diseases and disorders involving the immune mechanism (disorder)
        '197744007',  # Renal tubulo-interstitial disorders in metabolic diseases (disorder)
        '197745008',  # Renal tubulo-interstitial disorders in systemic connective tissue disorders (disorder)
        '197747000',  # Renal tubulo-interstitial disorders in transplant rejection (disorder)
        '197769007',  # Acute pyelonephritis with medullary necrosis (disorder)
        '234485006',  # Epstein syndrome (disorder)
        '236374007',  # Chronic infective interstitial nephritis (disorder)
        '236380004',  # Steroid-sensitive nephrotic syndrome (disorder)
        '236381000',  # Steroid-resistant nephrotic syndrome (disorder)
        '236382007',  # Steroid-dependent nephrotic syndrome (disorder)
        '236383002',  # Familial mesangial sclerosis (disorder)
        '236384008',  # Congenital nephrotic syndrome with focal glomerulosclerosis (disorder)
        '236385009',  # Drash syndrome (disorder)
        '236392004',  # Rapidly progressive glomerulonephritis (disorder)
        '236393009',  # Endocapillary glomerulonephritis (disorder)
        '236394003',  # Idiopathic endocapillary glomerulonephritis (disorder)
        '236395002',  # Post-infectious glomerulonephritis (disorder)
        '236397005',  # Post-infectious glomerulonephritis - Garland variety (disorder)
        '236398000',  # Crescentic glomerulonephritis (disorder)
        '236399008',  # Steroid-sensitive minimal change glomerulonephritis (disorder)
        '236400001',  # Steroid-resistant minimal change glomerulonephritis (disorder)
        '236401002',  # Steroid-dependent minimal change glomerulonephritis (disorder)
        '236402009',  # Glomerulopathy due to ischemia (disorder)
        '236403004',  # Focal segmental glomerulosclerosis (disorder)
        '236404005',  # Classical focal segmental glomerulosclerosis (disorder)
        '236405006',  # Hyperfiltration focal segmental glomerulosclerosis (disorder)
        '236407003',  # Immunoglobulin A nephropathy (disorder)
        '236409000',  # Mesangiocapillary glomerulonephritis type III (disorder)
        '236410005',  # Mesangiocapillary glomerulonephritis type IV (disorder)
        '236411009',  # Immunoglobulin M nephropathy (disorder)
        '236412002',  # C1q nephropathy (disorder)
        '236413007',  # Membranous glomerulonephritis - stage I (disorder)
        '236414001',  # Membranous glomerulonephritis - stage II (disorder)
        '236415000',  # Membranous glomerulonephritis - stage III (disorder)
        '236416004',  # Membranous glomerulonephritis - stage IV (disorder)
        '236417008',  # Membranous glomerulonephritis stage V (disorder)
        '236418003',  # Thin basement membrane disease (disorder)
        '236419006',  # Progressive hereditary glomerulonephritis without deafness (disorder)
        '236422008',  # Fechtner syndrome (disorder)
        '236452000',  # Chronic drug-induced tubulointerstitial nephritis (disorder)
        '236453005',  # Chronic tubulointerstitial nephritis caused by heavy metal (disorder)
        '236454004',  # Chronic lead nephropathy (disorder)
        '236455003',  # Saturnine nephropathy (disorder)
        '236456002',  # Chronic cadmium nephropathy (disorder)
        '236457006',  # Chronic mercury nephropathy (disorder)
        '236458001',  # Granulomatous tubulointerstitial nephritis (disorder)
        '236460004',  # Familial proximal renal tubular acidosis (disorder)
        '236461000',  # Distal renal tubular acidosis (disorder)
        '236463002',  # Hyperkalemic renal tubular acidosis (disorder)
        '236466005',  # Congenital Fanconi syndrome (disorder)
        '236479001',  # Magnesium-losing nephropathy (disorder)
        '236480003',  # Inherited magnesium-losing nephropathy (disorder)
        '236499007',  # Microalbuminuric nephropathy due to diabetes mellitus (disorder)
        '236500003',  # Proteinuric nephropathy due to diabetes mellitus (disorder)
        '236503001',  # Acute scleroderma renal crisis (disorder)
        '236506009',  # Goodpasture's disease (disorder)
        '236508005',  # Malignancy-associated glomerulonephritis (disorder)
        '236524006',  # Acute radiation nephritis (disorder)
        '236530006',  # Pulmonic stenosis and congenital nephrosis (disorder)
        '236532003',  # Renal tubular acidosis with progressive nerve deafness (disorder)
        '236535001',  # Glomerulopathy with giant fibrillar deposits (disorder)
        '236587002',  # Transplant glomerulopathy (disorder)
        '236588007',  # Transplant glomerulopathy - early form (disorder)
        '236589004',  # Transplant glomerulopathy - late form (disorder)
        '236590008',  # Cytomegalovirus-induced glomerulonephritis (disorder)
        '239932005',  # Primary pauci-immune necrotizing and crescentic glomerulonephritis (disorder)
        '253864004',  # Familial hypoplastic, glomerulocystic kidney (disorder)
        '266549004',  # Nephrotic syndrome with minimal change glomerulonephritis (disorder)
        '270494003',  # Acute papillary necrosis (disorder)
        '276585000',  # Congenital nephritis (disorder)
        '282364005',  # Immunoglobulin A nephropathy associated with liver disease (disorder)
        '307755009',  # Renal tubulo-interstitial disorder in systemic lupus erythematosus (disorder)
        '309426007',  # Glomerulopathy due to diabetes mellitus (disorder)
        '311366001',  # Kimmelstiel-Wilson syndrome (disorder)
        '359694003',  # Idiopathic crescentic glomerulonephritis, type II (disorder)
        '363234001',  # Nephrotic syndrome secondary to systemic disease (disorder)
        '370494002',  # Protein-losing nephropathy (disorder)
        '399094007',  # Familial hematuria (disorder)
        '399190000',  # Non-progressive hereditary glomerulonephritis (disorder)
        '399340005',  # Hereditary nephritis (disorder)
        '418839003',  # Tubulointerstitial nephritis with uveitis syndrome (disorder)
        '420514000',  # Persistent proteinuria due to type 1 diabetes mellitus (disorder)
        '420715001',  # Persistent microalbuminuria due to type 2 diabetes mellitus (disorder)
        '421305000',  # Persistent microalbuminuria due to type 1 diabetes mellitus (disorder)
        '421986006',  # Persistent proteinuria due to type 2 diabetes mellitus (disorder)
        '424188009',  # Idiopathic interstitial tubulonephritis (disorder)
        '425384007',  # Sarcoidosis with glomerulonephritis (disorder)
        '428255004',  # Tubulointerstitial nephritis (disorder)
        '439990003',  # Drug-induced tubulointerstitial nephritis (disorder)
        '440018001',  # Heavy-metal-induced tubulointerstitial nephritis (disorder)
        '441815006',  # Proliferative glomerulonephritis (disorder)
        '444977005',  # Autosomal dominant focal segmental glomerulosclerosis (disorder)
        '445170001',  # Macroalbuminuric nephropathy due to diabetes mellitus (disorder)
        '445258009',  # Idiopathic rapidly progressive glomerulonephritis (disorder)
        '445388002',  # Autosomal recessive focal segmental glomerulosclerosis (disorder)
        '445404003',  # Familial immunoglobulin A nephropathy (disorder)
        '446923008',  # Lipoprotein glomerulopathy (disorder)
        '700107006',  # Bartter syndrome antenatal type 1 (disorder)
        '700111000',  # Bartter syndrome type 3 (disorder)
        '700112007',  # Bartter syndrome type 4 (disorder)
        '702718005',  # Acute on chronic tubulointerstitial nephritis (disorder)
        '707742001',  # Bartter syndrome (disorder)
        '707755000',  # Familial hypokalemic and hypomagnesemic tubulopathy (disorder)
        '707756004',  # Gitelman syndrome (disorder)
        '717760006',  # Multi-drug resistant nephrotic syndrome (disorder)
        '717766000',  # Alport syndrome autosomal dominant (disorder)
        '717767009',  # Alport syndrome autosomal recessive (disorder)
        '717768004',  # Alport syndrome X-linked (disorder)
        '717791000',  # Bartter syndrome type 4a (disorder)
        '719839000',  # Tubular renal disease with cardiomyopathy syndrome (disorder)
        '720982007',  # Alport syndrome, intellectual disability, midface hypoplasia, elliptocytosis syndrome (disorder)
        '721175003',  # Idiopathic glomerular disease (disorder)
        '721207002',  # Seizure, sensorineural deafness, ataxia, intellectual disability, electrolyte imbalance syndrome (disorder)
        '721297008',  # Galloway Mowat syndrome (disorder)
        '721840000',  # Hyperuricemia, anemia, renal failure syndrome (disorder)
        '722086002',  # Membranous glomerulonephritis due to malignant neoplastic disease (disorder)
        '722118005',  # Congenital nephrotic syndrome due to congenital infection (disorder)
        '722119002',  # Idiopathic membranous glomerulonephritis (disorder)
        '722120008',  # Membranous glomerulonephritis caused by drug (disorder)
        '722139003',  # Focal segmental glomerulosclerosis caused by lithium (disorder)
        '722147003',  # Focal segmental glomerulosclerosis due to sickle cell disease (disorder)
        '722168002',  # Membranous glomerulonephritis co-occurrent with infectious disease (disorder)
        '722294004',  # Autosomal dominant intermediate Charcot-Marie-Tooth disease type E (disorder)
        '722369003',  # Congenital nephrotic syndrome due to diffuse mesangial sclerosis (disorder)
        '722381004',  # Congenital cataract, nephropathy, encephalopathy syndrome (disorder)
        '722433005',  # Dyschondrosteosis and nephritis syndrome (disorder)
        '722468005',  # Distal renal tubular acidosis co-occurrent with sensorineural deafness (disorder)
        '722758004',  # Complement component 3 glomerulopathy (disorder)
        '722759007',  # Glomerulopathy with fibronectin deposits 2 (disorder)
        '722760002',  # Dense deposit disease (disorder)
        '722761003',  # Complement component 3 glomerulonephritis (disorder)
        '722948009',  # Glomerular disorder due to non-neuropathic heredofamilial amyloidosis (disorder)
        '723074006',  # Renal papillary necrosis due to diabetes mellitus (disorder)
        '724099000',  # Acquired monoclonal immunoglobulin light chain-associated Fanconi syndrome (disorder)
        '725592009',  # Congenital membranous nephropathy due to maternal anti-neutral endopeptidase alloimmunization (disorder)
        '726082003',  # Immunotactoid glomerulonephritis (disorder)
        '726541005',  # Acute kidney injury due to trauma (disorder)
        '733453005',  # Congenital nephrotic syndrome, interstitial lung disease, epidermolysis bullosa syndrome (disorder)
        '766249007',  # Deafness, nephritis, anorectal malformation syndrome (disorder)
        '770414008',  # Alport syndrome (disorder)
        '771447009',  # Laminin subunit beta 2 related infantile-onset nephrotic syndrome (disorder)
        '773647007',  # Nephrotic syndrome, deafness, pretibial epidermolysis bullosa syndrome (disorder)
        '776416004',  # Hyperuricemia, pulmonary hypertension, renal failure, alkalosis syndrome (disorder)
        '782738008',  # Karyomegalic interstitial nephritis (disorder)
        '782771007',  # Mitochondrial deoxyribonucleic acid depletion syndrome hepatocerebrorenal form (disorder)
        '783157004',  # Leigh syndrome with nephrotic syndrome (disorder)
        '783614008',  # Familial steroid-resistant nephrotic syndrome with sensorineural deafness (disorder)
        '783620009',  # Dominant hypophosphatemia with nephrolithiasis and/or osteoporosis (disorder)
        '783787000',  # Retinal vasculopathy with cerebral leukoencephalopathy and systemic manifestations (disorder)
        '788613004',  # Anti-glomerular basement membrane disease (disorder)
        '792926007',  # Armanni-Ebstein kidney due to diabetes mellitus (disorder)
        '818952002',  # Fibronectin glomerulopathy (disorder)
        '829971006',  # Non-amyloid monoclonal immunoglobulin deposition disease (disorder)
        '838353009',  # Acute pyelonephritis caused by bacterium (disorder)
        '860892003',  # Glomerular disease due to infectious disease (disorder)
        '860893008',  # Glomerular disease due to parasitic disease (disorder)
        '12491000132101',  # Isolated proteinuria (finding)
        '12511000132108',  # Persistent proteinuria (finding)
        '15911000119103',  # Primary tubulointerstitial nephritis (disorder)
        '18521000119106',  # Microalbuminuria due to type 1 diabetes mellitus (disorder)
        '28191000119109',  # Chronic nephritic syndrome with membranous glomerulonephritis (disorder)
        '42231000009109',  # Bacterial nephritis (disorder)
        '50411000119108',  # Secondary tubulointerstitial nephritis (disorder)
        '71441000119104',  # Nephrotic syndrome due to type 2 diabetes mellitus (disorder)
        '71721000119101',  # Nephrotic syndrome due to type 1 diabetes mellitus (disorder)
        '90781000119102',  # Microalbuminuria due to type 2 diabetes mellitus (disorder)
        '90971000119103',  # Glomerulonephritis co-occurrent and due to vasculitis (disorder)
        '101711000119105',  # Glomerulonephritis co-occurrent and due to antineutrophil cytoplasmic antibody positive vasculitis (disorder)
        '157141000119108',  # Proteinuria due to type 2 diabetes mellitus (disorder)
        '243421000119104',  # Proteinuria due to type 1 diabetes mellitus (disorder)
        '320681000119102',  # Tubulointerstitial nephropathy due to Sjogren syndrome (disorder)
        '368901000119106',  # Chronic nephritic syndrome co-occurrent and due to membranoproliferative glomerulonephritis type III (disorder)
        '368921000119102',  # Nephritic syndrome co-occurrent and due to membranoproliferative glomerulonephritis type III (disorder)
    }


class HypertensiveChronicKidneyDisease(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of hypertensive chronic kidney disease.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis for hypertensive chronic kidney disease.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS134v10
    """

    VALUE_SET_NAME = 'Hypertensive Chronic Kidney Disease'
    OID = '2.16.840.1.113883.3.464.1003.109.12.1017'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'I12.0',  # Hypertensive chronic kidney disease with stage 5 chronic kidney disease or end stage renal disease
        'I12.9',  # Hypertensive chronic kidney disease with stage 1 through stage 4 chronic kidney disease, or unspecified chronic kidney disease
        'I13.0',  # Hypertensive heart and chronic kidney disease with heart failure and stage 1 through stage 4 chronic kidney disease, or unspecified chronic kidney disease
        'I13.10',  # Hypertensive heart and chronic kidney disease without heart failure, with stage 1 through stage 4 chronic kidney disease, or unspecified chronic kidney disease
        'I13.11',  # Hypertensive heart and chronic kidney disease without heart failure, with stage 5 chronic kidney disease, or end stage renal disease
        'I13.2',  # Hypertensive heart and chronic kidney disease with heart failure and with stage 5 chronic kidney disease, or end stage renal disease
        'I15.0',  # Renovascular hypertension
        'I15.1',  # Hypertension secondary to other renal disorders
    }
    ICD9CM = {
        '403.01',  # Hypertensive chronic kidney disease, malignant, with chronic kidney disease stage V or end stage renal disease
        '403.10',  # Hypertensive chronic kidney disease, benign, with chronic kidney disease stage I through stage IV, or unspecified
        '403.11',  # Hypertensive chronic kidney disease, benign, with chronic kidney disease stage V or end stage renal disease
        '403.90',  # Hypertensive chronic kidney disease, unspecified, with chronic kidney disease stage I through stage IV, or unspecified
        '403.91',  # Hypertensive chronic kidney disease, unspecified, with chronic kidney disease stage V or end stage renal disease
        '404.00',  # Hypertensive heart and chronic kidney disease, malignant, without heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        '404.01',  # Hypertensive heart and chronic kidney disease, malignant, with heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        '404.02',  # Hypertensive heart and chronic kidney disease, malignant, without heart failure and with chronic kidney disease stage V or end stage renal disease
        '404.03',  # Hypertensive heart and chronic kidney disease, malignant, with heart failure and with chronic kidney disease stage V or end stage renal disease
        '404.10',  # Hypertensive heart and chronic kidney disease, benign, without heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        '404.11',  # Hypertensive heart and chronic kidney disease, benign, with heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        '404.12',  # Hypertensive heart and chronic kidney disease, benign, without heart failure and with chronic kidney disease stage V or end stage renal disease
        '404.13',  # Hypertensive heart and chronic kidney disease, benign, with heart failure and chronic kidney disease stage V or end stage renal disease
        '404.90',  # Hypertensive heart and chronic kidney disease, unspecified, without heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        '404.91',  # Hypertensive heart and chronic kidney disease, unspecified, with heart failure and with chronic kidney disease stage I through stage IV, or unspecified
        '404.92',  # Hypertensive heart and chronic kidney disease, unspecified, without heart failure and with chronic kidney disease stage V or end stage renal disease
        '404.93',  # Hypertensive heart and chronic kidney disease, unspecified, with heart failure and chronic kidney disease stage V or end stage renal disease
    }
    SNOMEDCT = {
        '193003',  # Benign hypertensive renal disease (disorder)
        '14973001',  # Renal sclerosis with hypertension (disorder)
        '19769006',  # High-renin essential hypertension (disorder)
        '23130000',  # Paroxysmal hypertension (disorder)
        '28119000',  # Renal hypertension (disorder)
        '32916005',  # Nephrosclerosis (disorder)
        '38481006',  # Hypertensive renal disease (disorder)
        '39018007',  # Renal arterial hypertension (disorder)
        '45281005',  # Atherosclerosis of renal artery (disorder)
        '49220004',  # Hypertensive renal failure (disorder)
        '57684003',  # Parenchymal renal hypertension (disorder)
        '62240004',  # Benign arteriolar nephrosclerosis (disorder)
        '65443008',  # Malignant hypertensive renal disease (disorder)
        '66052004',  # Benign hypertensive heart AND renal disease (disorder)
        '66610008',  # Malignant hypertensive heart AND renal disease (disorder)
        '73410007',  # Benign secondary renovascular hypertension (disorder)
        '78544004',  # Chronic hypertensive uremia (disorder)
        '81363003',  # Malignant arteriolar nephrosclerosis (disorder)
        '86234004',  # Hypertensive heart AND renal disease (disorder)
        '90493000',  # Arteriolar nephrosclerosis (disorder)
        '111438007',  # Hypertension secondary to renal disease in obstetric context (disorder)
        '123799005',  # Renovascular hypertension (disorder)
        '123800009',  # Goldblatt hypertension (disorder)
        '194774006',  # Hypertensive renal disease with renal failure (disorder)
        '194783001',  # Secondary malignant renovascular hypertension (disorder)
        '302910002',  # Atherosclerotic renal artery stenosis (disorder)
        '397748008',  # Hypertension with albuminuria (disorder)
        '427889009',  # Hypertension associated with transplantation (disorder)
        '428575007',  # Hypertension secondary to kidney transplant (disorder)
        '473392002',  # Hypertensive nephrosclerosis (disorder)
        '285831000119108',  # Malignant hypertensive chronic kidney disease (disorder)
        '285841000119104',  # Malignant hypertensive end stage renal disease (disorder)
        '286371000119107',  # Malignant hypertensive end stage renal disease on dialysis (disorder)
        '431431000124100',  # Arteriosclerosis of renal artery (disorder)
    }


class KidneyFailure(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of kidney failure.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of kidney failure and kidney disease.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS134v10
    """

    VALUE_SET_NAME = 'Kidney Failure'
    OID = '2.16.840.1.113883.3.464.1003.109.12.1028'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'N17.0',  # Acute kidney failure with tubular necrosis
        'N17.1',  # Acute kidney failure with acute cortical necrosis
        'N17.2',  # Acute kidney failure with medullary necrosis
        'N17.8',  # Other acute kidney failure
        'N17.9',  # Acute kidney failure, unspecified
        'N18.1',  # Chronic kidney disease, stage 1
        'N18.2',  # Chronic kidney disease, stage 2 (mild)
        'N18.3',  # Chronic kidney disease, stage 3 (moderate)
        'N18.30',  # Chronic kidney disease, stage 3 unspecified
        'N18.31',  # Chronic kidney disease, stage 3a
        'N18.32',  # Chronic kidney disease, stage 3b
        'N18.4',  # Chronic kidney disease, stage 4 (severe)
        'N18.5',  # Chronic kidney disease, stage 5
        'N18.6',  # End stage renal disease
        'N18.9',  # Chronic kidney disease, unspecified
        'N19',  # Unspecified kidney failure
    }
    ICD9CM = {
        '584.5',  # Acute kidney failure with lesion of tubular necrosis
        '584.6',  # Acute kidney failure with lesion of renal cortical necrosis
        '584.7',  # Acute kidney failure with lesion of renal medullary [papillary] necrosis
        '584.8',  # Acute kidney failure with other specified pathological lesion in kidney
        '584.9',  # Acute kidney failure, unspecified
        '585.1',  # Chronic kidney disease, Stage I
        '585.2',  # Chronic kidney disease, Stage II (mild)
        '585.3',  # Chronic kidney disease, Stage III (moderate)
        '585.4',  # Chronic kidney disease, Stage IV (severe)
        '585.5',  # Chronic kidney disease, Stage V
        '585.6',  # End stage renal disease
        '585.9',  # Chronic kidney disease, unspecified
        '586',  # Renal failure, unspecified
        '587',  # Renal sclerosis, unspecified
    }
    SNOMEDCT = {
        '14669001',  # Acute renal failure syndrome (disorder)
        '46177005',  # End-stage renal disease (disorder)
        '236434000',  # End stage renal failure untreated by renal replacement therapy (disorder)
        '236435004',  # End stage renal failure on dialysis (disorder)
        '236436003',  # End stage renal failure with renal transplant (disorder)
        '429224003',  # Acute renal failure due to acute cortical necrosis (disorder)
        '431855005',  # Chronic kidney disease stage 1 (disorder)
        '431857002',  # Chronic kidney disease stage 4 (disorder)
        '433144002',  # Chronic kidney disease stage 3 (disorder)
        '433146000',  # Chronic kidney disease stage 5 (disorder)
        '700378005',  # Chronic kidney disease stage 3A (disorder)
        '700379002',  # Chronic kidney disease stage 3B (disorder)
        '722098007',  # Chronic kidney disease following donor nephrectomy (disorder)
        '722149000',  # Chronic kidney disease following excision of renal neoplasm (disorder)
        '722150000',  # Chronic kidney disease due to systemic infection (disorder)
        '722467000',  # Chronic kidney disease due to traumatic loss of kidney (disorder)
        '723190009',  # Chronic renal insufficiency (disorder)
        '721000119107',  # Chronic kidney disease stage 4 due to type 2 diabetes mellitus (disorder)
        '731000119105',  # Chronic kidney disease stage 3 due to type 2 diabetes mellitus (disorder)
        '90741000119107',  # Chronic kidney disease stage 3 due to type 1 diabetes mellitus (disorder)
        '90751000119109',  # Chronic kidney disease stage 4 due to type 1 diabetes mellitus (disorder)
        '96721000119103',  # Hypertensive heart AND chronic kidney disease stage 4 (disorder)
        '96741000119109',  # Hypertensive heart AND chronic kidney disease stage 2 (disorder)
        '129151000119102',  # Chronic kidney disease stage 4 due to hypertension (disorder)
        '129171000119106',  # Chronic kidney disease stage 3 due to hypertension (disorder)
        '129181000119109',  # Chronic kidney disease stage 2 due to hypertension (disorder)
        '140111000119107',  # Hypertension in chronic kidney disease stage 4 due to type 2 diabetes mellitus (disorder)
        '140131000119102',  # Hypertension in chronic kidney disease stage 2 due to type 2 diabetes mellitus (disorder)
        '145681000119101',  # Acute renal failure due to tubular necrosis (disorder)
        '284981000119102',  # Chronic kidney disease stage 2 due to benign hypertension (disorder)
        '285001000119105',  # Chronic kidney disease stage 4 due to benign hypertension (disorder)
        '285861000119100',  # Malignant hypertensive chronic kidney disease stage 2 (disorder)
        '285881000119109',  # Malignant hypertensive chronic kidney disease stage 4 (disorder)
    }


class Proteinuria(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  proteinuria

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of proteinuria.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS134v10
    """

    VALUE_SET_NAME = 'Proteinuria'
    OID = '2.16.840.1.113883.3.526.3.1003'
    DEFINITION_VERSION = '20200307'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'R80.1',  # Persistent proteinuria, unspecified
        'R80.8',  # Other proteinuria
        'R80.9',  # Proteinuria, unspecified
    }
    SNOMEDCT = {
        '12178007',  # Nephrogenous proteinuria (finding)
        '29738008',  # Proteinuria (finding)
        '34993002',  # Mixed proteinuria (finding)
        '167582007',  # Urine microalbumin positive (finding)
        '263808002',  # Microproteinuria (finding)
        '264867001',  # Nephrotic range proteinuria (finding)
        '274769005',  # Albuminuria (finding)
        '312975006',  # Microalbuminuria (finding)
    }


class AllergyToAceInhibitorOrArb(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnosis of allergy to an angiotensin-converting enzyme (ACE) inhibitor and angiotensin-receptor blocker (ARB).

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of allergy to an angiotensin-converting enzyme (ACE) inhibitor and angiotensin-receptor blocker (ARB).

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS135v10
    """

    VALUE_SET_NAME = 'Allergy to ACE Inhibitor or ARB'
    OID = '2.16.840.1.113883.3.526.3.1211'
    DEFINITION_VERSION = '20170908'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    SNOMEDCT = {
        '609534005',  # Non-allergic hypersensitivity to losartan (finding)
        '609537003',  # Non-allergic hypersensitivity to angiotensin-converting enzyme inhibitor (finding)
        '609538008',  # Non-allergic hypersensitivity to captopril (finding)
        '609539000',  # Non-allergic hypersensitivity to cilazapril (finding)
        '609540003',  # Non-allergic hypersensitivity to enalapril (finding)
        '609541004',  # Non-allergic hypersensitivity to fosinopril (finding)
        '609542006',  # Non-allergic hypersensitivity to lisinopril (finding)
        '609543001',  # Non-allergic hypersensitivity to perindopril (finding)
        '609544007',  # Non-allergic hypersensitivity to quinapril (finding)
        '609545008',  # Non-allergic hypersensitivity to ramipril (finding)
        '609546009',  # Non-allergic hypersensitivity to trandolapril (finding)
        '609549002',  # Non-allergic hypersensitivity to angiotensin II receptor antagonist (finding)
        '609550002',  # History of angiotensin converting enzyme inhibitor pseudoallergy (situation)
    }


class HeartFailure(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of heart failure.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of heart failure.

    **Exclusion Criteria:** Excludes concepts that represent chronic and acute right-sided heart failure, right heart failure, right ventricular failure, cardiac insufficiency due to prosthesis and following cardiac surgery, or pulmonale.

    ** Used in:** CMS135v10, CMS144v10, CMS90v11
    """

    VALUE_SET_NAME = 'Heart Failure'
    OID = '2.16.840.1.113883.3.526.3.376'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'I11.0',  # Hypertensive heart disease with heart failure
        'I13.0',  # Hypertensive heart and chronic kidney disease with heart failure and stage 1 through stage 4 chronic kidney disease, or unspecified chronic kidney disease
        'I13.2',  # Hypertensive heart and chronic kidney disease with heart failure and with stage 5 chronic kidney disease, or end stage renal disease
        'I50.1',  # Left ventricular failure, unspecified
        'I50.20',  # Unspecified systolic (congestive) heart failure
        'I50.21',  # Acute systolic (congestive) heart failure
        'I50.22',  # Chronic systolic (congestive) heart failure
        'I50.23',  # Acute on chronic systolic (congestive) heart failure
        'I50.30',  # Unspecified diastolic (congestive) heart failure
        'I50.31',  # Acute diastolic (congestive) heart failure
        'I50.32',  # Chronic diastolic (congestive) heart failure
        'I50.33',  # Acute on chronic diastolic (congestive) heart failure
        'I50.40',  # Unspecified combined systolic (congestive) and diastolic (congestive) heart failure
        'I50.41',  # Acute combined systolic (congestive) and diastolic (congestive) heart failure
        'I50.42',  # Chronic combined systolic (congestive) and diastolic (congestive) heart failure
        'I50.43',  # Acute on chronic combined systolic (congestive) and diastolic (congestive) heart failure
        'I50.814',  # Right heart failure due to left heart failure
        'I50.82',  # Biventricular heart failure
        'I50.83',  # High output heart failure
        'I50.84',  # End stage heart failure
        'I50.89',  # Other heart failure
        'I50.9',  # Heart failure, unspecified
    }
    SNOMEDCT = {
        '364006',  # Acute left-sided heart failure (disorder)
        '5148006',  # Hypertensive heart disease with congestive heart failure (disorder)
        '5375005',  # Chronic left-sided congestive heart failure (disorder)
        '10091002',  # High output heart failure (disorder)
        '10633002',  # Acute congestive heart failure (disorder)
        '25544003',  # Low output heart failure (disorder)
        '42343007',  # Congestive heart failure (disorder)
        '43736008',  # Rheumatic left ventricular failure (disorder)
        '44088000',  # Low cardiac output syndrome (disorder)
        '44313006',  # Right heart failure secondary to left heart failure (disorder)
        '46113002',  # Hypertensive heart failure (disorder)
        '48447003',  # Chronic heart failure (disorder)
        '56675007',  # Acute heart failure (disorder)
        '74960003',  # Acute left-sided congestive heart failure (disorder)
        '82523003',  # Congestive rheumatic heart failure (disorder)
        '83105008',  # Malignant hypertensive heart disease with congestive heart failure (disorder)
        '84114007',  # Heart failure (disorder)
        '85232009',  # Left heart failure (disorder)
        '88805009',  # Chronic congestive heart failure (disorder)
        '90727007',  # Pleural effusion due to congestive heart failure (disorder)
        '92506005',  # Biventricular congestive heart failure (disorder)
        '111283005',  # Chronic left-sided heart failure (disorder)
        '194767001',  # Benign hypertensive heart disease with congestive cardiac failure (disorder)
        '194779001',  # Hypertensive heart and renal disease with (congestive) heart failure (disorder)
        '194781004',  # Hypertensive heart and renal disease with both (congestive) heart failure and renal failure (disorder)
        '195111005',  # Decompensated cardiac failure (disorder)
        '195112003',  # Compensated cardiac failure (disorder)
        '195114002',  # Acute left ventricular failure (disorder)
        '206586007',  # Congenital cardiac failure (disorder)
        '233924009',  # Heart failure as a complication of care (disorder)
        '314206003',  # Refractory heart failure (disorder)
        '410431009',  # Cardiorespiratory failure (disorder)
        '417996009',  # Systolic heart failure (disorder)
        '418304008',  # Diastolic heart failure (disorder)
        '424404003',  # Decompensated chronic heart failure (disorder)
        '426263006',  # Congestive heart failure due to left ventricular systolic dysfunction (disorder)
        '426611007',  # Congestive heart failure due to valvular disease (disorder)
        '441481004',  # Chronic systolic heart failure (disorder)
        '441530006',  # Chronic diastolic heart failure (disorder)
        '443253003',  # Acute on chronic systolic heart failure (disorder)
        '443254009',  # Acute systolic heart failure (disorder)
        '443343001',  # Acute diastolic heart failure (disorder)
        '443344007',  # Acute on chronic diastolic heart failure (disorder)
        '471880001',  # Heart failure due to end stage congenital heart disease (disorder)
        '698594003',  # Symptomatic congestive heart failure (disorder)
        '703272007',  # Heart failure with reduced ejection fraction (disorder)
        '703273002',  # Heart failure with reduced ejection fraction due to coronary artery disease (disorder)
        '703274008',  # Heart failure with reduced ejection fraction due to myocarditis (disorder)
        '703275009',  # Heart failure with reduced ejection fraction due to cardiomyopathy (disorder)
        '703276005',  # Heart failure with reduced ejection fraction due to heart valve disease (disorder)
        '717840005',  # Congestive heart failure stage B (disorder)
        '871617000',  # Low output heart failure due to and following Fontan operation (disorder)
        '15781000119107',  # Hypertensive heart AND chronic kidney disease with congestive heart failure (disorder)
        '23341000119109',  # Congestive heart failure with right heart failure (disorder)
        '67431000119105',  # Congestive heart failure stage D (disorder)
        '67441000119101',  # Congestive heart failure stage C (disorder)
        '72481000119103',  # Congestive heart failure as early postoperative complication (disorder)
        '101281000119107',  # Congestive heart failure due to cardiomyopathy (disorder)
        '120851000119104',  # Systolic heart failure stage D (disorder)
        '120861000119102',  # Systolic heart failure stage C (disorder)
        '120871000119108',  # Systolic heart failure stage B (disorder)
        '120881000119106',  # Diastolic heart failure stage D (disorder)
        '120891000119109',  # Diastolic heart failure stage C (disorder)
        '120901000119108',  # Diastolic heart failure stage B (disorder)
        '153931000119109',  # Acute combined systolic and diastolic heart failure (disorder)
        '153941000119100',  # Chronic combined systolic and diastolic heart failure (disorder)
        '153951000119103',  # Acute on chronic combined systolic and diastolic heart failure (disorder)
        '15629541000119106',  # Congestive heart failure stage C due to ischemic cardiomyopathy (disorder)
        '15629591000119103',  # Congestive heart failure stage B due to ischemic cardiomyopathy (disorder)
        '15629641000119107',  # Systolic heart failure stage B due to ischemic cardiomyopathy (disorder)
        '15629741000119102',  # Systolic heart failure stage C due to ischemic cardiomyopathy (disorder)
        '15964701000119109',  # Acute cor pulmonale co-occurrent and due to saddle embolus of pulmonary artery (disorder)
    }


class IntoleranceToAceInhibitorOrArb(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses of intolerance to angiotensin-converting enzyme (ACE) inhibitor or angiotensin II receptor blockers (ARB).

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of intolerance to ACE inhibitor or ARB.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS135v10
    """

    VALUE_SET_NAME = 'Intolerance to ACE Inhibitor or ARB'
    OID = '2.16.840.1.113883.3.526.3.1212'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    SNOMEDCT = {
        '293500009',  # Angiotensin-converting-enzyme inhibitor adverse reaction (disorder)
        '293501008',  # Captopril adverse reaction (disorder)
        '293502001',  # Lisinopril adverse reaction (disorder)
        '293503006',  # Quinapril adverse reaction (disorder)
        '293504000',  # Ramipril adverse reaction (disorder)
        '293505004',  # Cilazapril adverse reaction (disorder)
        '293506003',  # Trandolapril adverse reaction (disorder)
        '293507007',  # Enalapril adverse reaction (disorder)
        '293508002',  # Fosinopril adverse reaction (disorder)
        '293509005',  # Perindopril adverse reaction (disorder)
        '293513003',  # Adverse reaction caused by losartan (disorder)
        '371627004',  # Angiotensin converting enzyme inhibitor-aggravated angioedema (disorder)
        '407564000',  # Angiotensin converting enzyme inhibitor not tolerated (situation)
        '407565004',  # Angiotensin II receptor antagonist not tolerated (situation)
        '407590002',  # Angiotensin II receptor antagonist adverse reaction (disorder)
        '422593004',  # Acute renal failure caused by angiotensin-converting-enzyme inhibitor (disorder)
    }


class ModerateOrSevereLvsd(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of moderate or severe left ventricular systolic dysfunction (LVSD).

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of moderate or severe LVSD.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS135v10, CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = 'Moderate or Severe LVSD'
    OID = '2.16.840.1.113883.3.526.3.1090'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    SNOMEDCT = {
        '981000124106',  # Moderate left ventricular systolic dysfunction (disorder)
        '991000124109',  # Severe left ventricular systolic dysfunction (disorder)
    }


class Pregnancy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of pregnancy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of pregnancy.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS135v10, CMS165v10, CMS155v10
    """

    VALUE_SET_NAME = 'Pregnancy'
    OID = '2.16.840.1.113883.3.526.3.378'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'O09.00',  # Supervision of pregnancy with history of infertility, unspecified trimester
        'O09.01',  # Supervision of pregnancy with history of infertility, first trimester
        'O09.02',  # Supervision of pregnancy with history of infertility, second trimester
        'O09.03',  # Supervision of pregnancy with history of infertility, third trimester
        'O09.10',  # Supervision of pregnancy with history of ectopic pregnancy, unspecified trimester
        'O09.11',  # Supervision of pregnancy with history of ectopic pregnancy, first trimester
        'O09.12',  # Supervision of pregnancy with history of ectopic pregnancy, second trimester
        'O09.13',  # Supervision of pregnancy with history of ectopic pregnancy, third trimester
        'O09.211',  # Supervision of pregnancy with history of pre-term labor, first trimester
        'O09.212',  # Supervision of pregnancy with history of pre-term labor, second trimester
        'O09.213',  # Supervision of pregnancy with history of pre-term labor, third trimester
        'O09.219',  # Supervision of pregnancy with history of pre-term labor, unspecified trimester
        'O09.291',  # Supervision of pregnancy with other poor reproductive or obstetric history, first trimester
        'O09.292',  # Supervision of pregnancy with other poor reproductive or obstetric history, second trimester
        'O09.293',  # Supervision of pregnancy with other poor reproductive or obstetric history, third trimester
        'O09.299',  # Supervision of pregnancy with other poor reproductive or obstetric history, unspecified trimester
        'O09.30',  # Supervision of pregnancy with insufficient antenatal care, unspecified trimester
        'O09.31',  # Supervision of pregnancy with insufficient antenatal care, first trimester
        'O09.32',  # Supervision of pregnancy with insufficient antenatal care, second trimester
        'O09.33',  # Supervision of pregnancy with insufficient antenatal care, third trimester
        'O09.40',  # Supervision of pregnancy with grand multiparity, unspecified trimester
        'O09.41',  # Supervision of pregnancy with grand multiparity, first trimester
        'O09.42',  # Supervision of pregnancy with grand multiparity, second trimester
        'O09.43',  # Supervision of pregnancy with grand multiparity, third trimester
        'O09.511',  # Supervision of elderly primigravida, first trimester
        'O09.512',  # Supervision of elderly primigravida, second trimester
        'O09.513',  # Supervision of elderly primigravida, third trimester
        'O09.519',  # Supervision of elderly primigravida, unspecified trimester
        'O09.521',  # Supervision of elderly multigravida, first trimester
        'O09.522',  # Supervision of elderly multigravida, second trimester
        'O09.523',  # Supervision of elderly multigravida, third trimester
        'O09.529',  # Supervision of elderly multigravida, unspecified trimester
        'O09.611',  # Supervision of young primigravida, first trimester
        'O09.612',  # Supervision of young primigravida, second trimester
        'O09.613',  # Supervision of young primigravida, third trimester
        'O09.619',  # Supervision of young primigravida, unspecified trimester
        'O09.621',  # Supervision of young multigravida, first trimester
        'O09.622',  # Supervision of young multigravida, second trimester
        'O09.623',  # Supervision of young multigravida, third trimester
        'O09.629',  # Supervision of young multigravida, unspecified trimester
        'O09.70',  # Supervision of high risk pregnancy due to social problems, unspecified trimester
        'O09.71',  # Supervision of high risk pregnancy due to social problems, first trimester
        'O09.72',  # Supervision of high risk pregnancy due to social problems, second trimester
        'O09.73',  # Supervision of high risk pregnancy due to social problems, third trimester
        'O09.811',  # Supervision of pregnancy resulting from assisted reproductive technology, first trimester
        'O09.812',  # Supervision of pregnancy resulting from assisted reproductive technology, second trimester
        'O09.813',  # Supervision of pregnancy resulting from assisted reproductive technology, third trimester
        'O09.819',  # Supervision of pregnancy resulting from assisted reproductive technology, unspecified trimester
        'O09.821',  # Supervision of pregnancy with history of in utero procedure during previous pregnancy, first trimester
        'O09.822',  # Supervision of pregnancy with history of in utero procedure during previous pregnancy, second trimester
        'O09.823',  # Supervision of pregnancy with history of in utero procedure during previous pregnancy, third trimester
        'O09.829',  # Supervision of pregnancy with history of in utero procedure during previous pregnancy, unspecified trimester
        'O09.891',  # Supervision of other high risk pregnancies, first trimester
        'O09.892',  # Supervision of other high risk pregnancies, second trimester
        'O09.893',  # Supervision of other high risk pregnancies, third trimester
        'O09.899',  # Supervision of other high risk pregnancies, unspecified trimester
        'O09.90',  # Supervision of high risk pregnancy, unspecified, unspecified trimester
        'O09.91',  # Supervision of high risk pregnancy, unspecified, first trimester
        'O09.92',  # Supervision of high risk pregnancy, unspecified, second trimester
        'O09.93',  # Supervision of high risk pregnancy, unspecified, third trimester
        'O09.A0',  # Supervision of pregnancy with history of molar pregnancy, unspecified trimester
        'O09.A1',  # Supervision of pregnancy with history of molar pregnancy, first trimester
        'O09.A2',  # Supervision of pregnancy with history of molar pregnancy, second trimester
        'O09.A3',  # Supervision of pregnancy with history of molar pregnancy, third trimester
        'O10.011',  # Pre-existing essential hypertension complicating pregnancy, first trimester
        'O10.012',  # Pre-existing essential hypertension complicating pregnancy, second trimester
        'O10.013',  # Pre-existing essential hypertension complicating pregnancy, third trimester
        'O10.019',  # Pre-existing essential hypertension complicating pregnancy, unspecified trimester
        'O10.111',  # Pre-existing hypertensive heart disease complicating pregnancy, first trimester
        'O10.112',  # Pre-existing hypertensive heart disease complicating pregnancy, second trimester
        'O10.113',  # Pre-existing hypertensive heart disease complicating pregnancy, third trimester
        'O10.119',  # Pre-existing hypertensive heart disease complicating pregnancy, unspecified trimester
        'O10.211',  # Pre-existing hypertensive chronic kidney disease complicating pregnancy, first trimester
        'O10.212',  # Pre-existing hypertensive chronic kidney disease complicating pregnancy, second trimester
        'O10.213',  # Pre-existing hypertensive chronic kidney disease complicating pregnancy, third trimester
        'O10.219',  # Pre-existing hypertensive chronic kidney disease complicating pregnancy, unspecified trimester
        'O10.311',  # Pre-existing hypertensive heart and chronic kidney disease complicating pregnancy, first trimester
        'O10.312',  # Pre-existing hypertensive heart and chronic kidney disease complicating pregnancy, second trimester
        'O10.313',  # Pre-existing hypertensive heart and chronic kidney disease complicating pregnancy, third trimester
        'O10.319',  # Pre-existing hypertensive heart and chronic kidney disease complicating pregnancy, unspecified trimester
        'O10.411',  # Pre-existing secondary hypertension complicating pregnancy, first trimester
        'O10.412',  # Pre-existing secondary hypertension complicating pregnancy, second trimester
        'O10.413',  # Pre-existing secondary hypertension complicating pregnancy, third trimester
        'O10.419',  # Pre-existing secondary hypertension complicating pregnancy, unspecified trimester
        'O10.911',  # Unspecified pre-existing hypertension complicating pregnancy, first trimester
        'O10.912',  # Unspecified pre-existing hypertension complicating pregnancy, second trimester
        'O10.913',  # Unspecified pre-existing hypertension complicating pregnancy, third trimester
        'O10.919',  # Unspecified pre-existing hypertension complicating pregnancy, unspecified trimester
        'O11.1',  # Pre-existing hypertension with pre-eclampsia, first trimester
        'O11.2',  # Pre-existing hypertension with pre-eclampsia, second trimester
        'O11.3',  # Pre-existing hypertension with pre-eclampsia, third trimester
        'O11.9',  # Pre-existing hypertension with pre-eclampsia, unspecified trimester
        'O12.00',  # Gestational edema, unspecified trimester
        'O12.01',  # Gestational edema, first trimester
        'O12.02',  # Gestational edema, second trimester
        'O12.03',  # Gestational edema, third trimester
        'O12.10',  # Gestational proteinuria, unspecified trimester
        'O12.11',  # Gestational proteinuria, first trimester
        'O12.12',  # Gestational proteinuria, second trimester
        'O12.13',  # Gestational proteinuria, third trimester
        'O12.20',  # Gestational edema with proteinuria, unspecified trimester
        'O12.21',  # Gestational edema with proteinuria, first trimester
        'O12.22',  # Gestational edema with proteinuria, second trimester
        'O12.23',  # Gestational edema with proteinuria, third trimester
        'O13.1',  # Gestational [pregnancy-induced] hypertension without significant proteinuria, first trimester
        'O13.2',  # Gestational [pregnancy-induced] hypertension without significant proteinuria, second trimester
        'O13.3',  # Gestational [pregnancy-induced] hypertension without significant proteinuria, third trimester
        'O13.9',  # Gestational [pregnancy-induced] hypertension without significant proteinuria, unspecified trimester
        'O14.00',  # Mild to moderate pre-eclampsia, unspecified trimester
        'O14.02',  # Mild to moderate pre-eclampsia, second trimester
        'O14.03',  # Mild to moderate pre-eclampsia, third trimester
        'O14.10',  # Severe pre-eclampsia, unspecified trimester
        'O14.12',  # Severe pre-eclampsia, second trimester
        'O14.13',  # Severe pre-eclampsia, third trimester
        'O14.20',  # HELLP syndrome (HELLP), unspecified trimester
        'O14.22',  # HELLP syndrome (HELLP), second trimester
        'O14.23',  # HELLP syndrome (HELLP), third trimester
        'O14.90',  # Unspecified pre-eclampsia, unspecified trimester
        'O14.92',  # Unspecified pre-eclampsia, second trimester
        'O14.93',  # Unspecified pre-eclampsia, third trimester
        'O15.00',  # Eclampsia complicating pregnancy, unspecified trimester
        'O15.02',  # Eclampsia complicating pregnancy, second trimester
        'O15.03',  # Eclampsia complicating pregnancy, third trimester
        'O16.1',  # Unspecified maternal hypertension, first trimester
        'O16.2',  # Unspecified maternal hypertension, second trimester
        'O16.3',  # Unspecified maternal hypertension, third trimester
        'O16.9',  # Unspecified maternal hypertension, unspecified trimester
        'O20.0',  # Threatened abortion
        'O20.8',  # Other hemorrhage in early pregnancy
        'O20.9',  # Hemorrhage in early pregnancy, unspecified
        'O21.0',  # Mild hyperemesis gravidarum
        'O21.1',  # Hyperemesis gravidarum with metabolic disturbance
        'O21.2',  # Late vomiting of pregnancy
        'O21.8',  # Other vomiting complicating pregnancy
        'O21.9',  # Vomiting of pregnancy, unspecified
        'O22.00',  # Varicose veins of lower extremity in pregnancy, unspecified trimester
        'O22.01',  # Varicose veins of lower extremity in pregnancy, first trimester
        'O22.02',  # Varicose veins of lower extremity in pregnancy, second trimester
        'O22.03',  # Varicose veins of lower extremity in pregnancy, third trimester
        'O22.10',  # Genital varices in pregnancy, unspecified trimester
        'O22.11',  # Genital varices in pregnancy, first trimester
        'O22.12',  # Genital varices in pregnancy, second trimester
        'O22.13',  # Genital varices in pregnancy, third trimester
        'O22.20',  # Superficial thrombophlebitis in pregnancy, unspecified trimester
        'O22.21',  # Superficial thrombophlebitis in pregnancy, first trimester
        'O22.22',  # Superficial thrombophlebitis in pregnancy, second trimester
        'O22.23',  # Superficial thrombophlebitis in pregnancy, third trimester
        'O22.30',  # Deep phlebothrombosis in pregnancy, unspecified trimester
        'O22.31',  # Deep phlebothrombosis in pregnancy, first trimester
        'O22.32',  # Deep phlebothrombosis in pregnancy, second trimester
        'O22.33',  # Deep phlebothrombosis in pregnancy, third trimester
        'O22.40',  # Hemorrhoids in pregnancy, unspecified trimester
        'O22.41',  # Hemorrhoids in pregnancy, first trimester
        'O22.42',  # Hemorrhoids in pregnancy, second trimester
        'O22.43',  # Hemorrhoids in pregnancy, third trimester
        'O22.50',  # Cerebral venous thrombosis in pregnancy, unspecified trimester
        'O22.51',  # Cerebral venous thrombosis in pregnancy, first trimester
        'O22.52',  # Cerebral venous thrombosis in pregnancy, second trimester
        'O22.53',  # Cerebral venous thrombosis in pregnancy, third trimester
        'O22.8X1',  # Other venous complications in pregnancy, first trimester
        'O22.8X2',  # Other venous complications in pregnancy, second trimester
        'O22.8X3',  # Other venous complications in pregnancy, third trimester
        'O22.8X9',  # Other venous complications in pregnancy, unspecified trimester
        'O22.90',  # Venous complication in pregnancy, unspecified, unspecified trimester
        'O22.91',  # Venous complication in pregnancy, unspecified, first trimester
        'O22.92',  # Venous complication in pregnancy, unspecified, second trimester
        'O22.93',  # Venous complication in pregnancy, unspecified, third trimester
        'O23.00',  # Infections of kidney in pregnancy, unspecified trimester
        'O23.01',  # Infections of kidney in pregnancy, first trimester
        'O23.02',  # Infections of kidney in pregnancy, second trimester
        'O23.03',  # Infections of kidney in pregnancy, third trimester
        'O23.10',  # Infections of bladder in pregnancy, unspecified trimester
        'O23.11',  # Infections of bladder in pregnancy, first trimester
        'O23.12',  # Infections of bladder in pregnancy, second trimester
        'O23.13',  # Infections of bladder in pregnancy, third trimester
        'O23.20',  # Infections of urethra in pregnancy, unspecified trimester
        'O23.21',  # Infections of urethra in pregnancy, first trimester
        'O23.22',  # Infections of urethra in pregnancy, second trimester
        'O23.23',  # Infections of urethra in pregnancy, third trimester
        'O23.30',  # Infections of other parts of urinary tract in pregnancy, unspecified trimester
        'O23.31',  # Infections of other parts of urinary tract in pregnancy, first trimester
        'O23.32',  # Infections of other parts of urinary tract in pregnancy, second trimester
        'O23.33',  # Infections of other parts of urinary tract in pregnancy, third trimester
        'O23.40',  # Unspecified infection of urinary tract in pregnancy, unspecified trimester
        'O23.41',  # Unspecified infection of urinary tract in pregnancy, first trimester
        'O23.42',  # Unspecified infection of urinary tract in pregnancy, second trimester
        'O23.43',  # Unspecified infection of urinary tract in pregnancy, third trimester
        'O23.511',  # Infections of cervix in pregnancy, first trimester
        'O23.512',  # Infections of cervix in pregnancy, second trimester
        'O23.513',  # Infections of cervix in pregnancy, third trimester
        'O23.519',  # Infections of cervix in pregnancy, unspecified trimester
        'O23.521',  # Salpingo-oophoritis in pregnancy, first trimester
        'O23.522',  # Salpingo-oophoritis in pregnancy, second trimester
        'O23.523',  # Salpingo-oophoritis in pregnancy, third trimester
        'O23.529',  # Salpingo-oophoritis in pregnancy, unspecified trimester
        'O23.591',  # Infection of other part of genital tract in pregnancy, first trimester
        'O23.592',  # Infection of other part of genital tract in pregnancy, second trimester
        'O23.593',  # Infection of other part of genital tract in pregnancy, third trimester
        'O23.599',  # Infection of other part of genital tract in pregnancy, unspecified trimester
        'O23.90',  # Unspecified genitourinary tract infection in pregnancy, unspecified trimester
        'O23.91',  # Unspecified genitourinary tract infection in pregnancy, first trimester
        'O23.92',  # Unspecified genitourinary tract infection in pregnancy, second trimester
        'O23.93',  # Unspecified genitourinary tract infection in pregnancy, third trimester
        'O24.011',  # Pre-existing type 1 diabetes mellitus, in pregnancy, first trimester
        'O24.012',  # Pre-existing type 1 diabetes mellitus, in pregnancy, second trimester
        'O24.013',  # Pre-existing type 1 diabetes mellitus, in pregnancy, third trimester
        'O24.019',  # Pre-existing type 1 diabetes mellitus, in pregnancy, unspecified trimester
        'O24.111',  # Pre-existing type 2 diabetes mellitus, in pregnancy, first trimester
        'O24.112',  # Pre-existing type 2 diabetes mellitus, in pregnancy, second trimester
        'O24.113',  # Pre-existing type 2 diabetes mellitus, in pregnancy, third trimester
        'O24.119',  # Pre-existing type 2 diabetes mellitus, in pregnancy, unspecified trimester
        'O24.311',  # Unspecified pre-existing diabetes mellitus in pregnancy, first trimester
        'O24.312',  # Unspecified pre-existing diabetes mellitus in pregnancy, second trimester
        'O24.313',  # Unspecified pre-existing diabetes mellitus in pregnancy, third trimester
        'O24.319',  # Unspecified pre-existing diabetes mellitus in pregnancy, unspecified trimester
        'O24.410',  # Gestational diabetes mellitus in pregnancy, diet controlled
        'O24.414',  # Gestational diabetes mellitus in pregnancy, insulin controlled
        'O24.415',  # Gestational diabetes mellitus in pregnancy, controlled by oral hypoglycemic drugs
        'O24.419',  # Gestational diabetes mellitus in pregnancy, unspecified control
        'O24.811',  # Other pre-existing diabetes mellitus in pregnancy, first trimester
        'O24.812',  # Other pre-existing diabetes mellitus in pregnancy, second trimester
        'O24.813',  # Other pre-existing diabetes mellitus in pregnancy, third trimester
        'O24.819',  # Other pre-existing diabetes mellitus in pregnancy, unspecified trimester
        'O24.911',  # Unspecified diabetes mellitus in pregnancy, first trimester
        'O24.912',  # Unspecified diabetes mellitus in pregnancy, second trimester
        'O24.913',  # Unspecified diabetes mellitus in pregnancy, third trimester
        'O24.919',  # Unspecified diabetes mellitus in pregnancy, unspecified trimester
        'O25.10',  # Malnutrition in pregnancy, unspecified trimester
        'O25.11',  # Malnutrition in pregnancy, first trimester
        'O25.12',  # Malnutrition in pregnancy, second trimester
        'O25.13',  # Malnutrition in pregnancy, third trimester
        'O26.00',  # Excessive weight gain in pregnancy, unspecified trimester
        'O26.01',  # Excessive weight gain in pregnancy, first trimester
        'O26.02',  # Excessive weight gain in pregnancy, second trimester
        'O26.03',  # Excessive weight gain in pregnancy, third trimester
        'O26.10',  # Low weight gain in pregnancy, unspecified trimester
        'O26.11',  # Low weight gain in pregnancy, first trimester
        'O26.12',  # Low weight gain in pregnancy, second trimester
        'O26.13',  # Low weight gain in pregnancy, third trimester
        'O26.20',  # Pregnancy care for patient with recurrent pregnancy loss, unspecified trimester
        'O26.21',  # Pregnancy care for patient with recurrent pregnancy loss, first trimester
        'O26.22',  # Pregnancy care for patient with recurrent pregnancy loss, second trimester
        'O26.23',  # Pregnancy care for patient with recurrent pregnancy loss, third trimester
        'O26.30',  # Retained intrauterine contraceptive device in pregnancy, unspecified trimester
        'O26.31',  # Retained intrauterine contraceptive device in pregnancy, first trimester
        'O26.32',  # Retained intrauterine contraceptive device in pregnancy, second trimester
        'O26.33',  # Retained intrauterine contraceptive device in pregnancy, third trimester
        'O26.40',  # Herpes gestationis, unspecified trimester
        'O26.41',  # Herpes gestationis, first trimester
        'O26.42',  # Herpes gestationis, second trimester
        'O26.43',  # Herpes gestationis, third trimester
        'O26.50',  # Maternal hypotension syndrome, unspecified trimester
        'O26.51',  # Maternal hypotension syndrome, first trimester
        'O26.52',  # Maternal hypotension syndrome, second trimester
        'O26.53',  # Maternal hypotension syndrome, third trimester
        'O26.611',  # Liver and biliary tract disorders in pregnancy, first trimester
        'O26.612',  # Liver and biliary tract disorders in pregnancy, second trimester
        'O26.613',  # Liver and biliary tract disorders in pregnancy, third trimester
        'O26.619',  # Liver and biliary tract disorders in pregnancy, unspecified trimester
        'O26.711',  # Subluxation of symphysis (pubis) in pregnancy, first trimester
        'O26.712',  # Subluxation of symphysis (pubis) in pregnancy, second trimester
        'O26.713',  # Subluxation of symphysis (pubis) in pregnancy, third trimester
        'O26.719',  # Subluxation of symphysis (pubis) in pregnancy, unspecified trimester
        'O26.811',  # Pregnancy related exhaustion and fatigue, first trimester
        'O26.812',  # Pregnancy related exhaustion and fatigue, second trimester
        'O26.813',  # Pregnancy related exhaustion and fatigue, third trimester
        'O26.819',  # Pregnancy related exhaustion and fatigue, unspecified trimester
        'O26.821',  # Pregnancy related peripheral neuritis, first trimester
        'O26.822',  # Pregnancy related peripheral neuritis, second trimester
        'O26.823',  # Pregnancy related peripheral neuritis, third trimester
        'O26.829',  # Pregnancy related peripheral neuritis, unspecified trimester
        'O26.831',  # Pregnancy related renal disease, first trimester
        'O26.832',  # Pregnancy related renal disease, second trimester
        'O26.833',  # Pregnancy related renal disease, third trimester
        'O26.839',  # Pregnancy related renal disease, unspecified trimester
        'O26.841',  # Uterine size-date discrepancy, first trimester
        'O26.842',  # Uterine size-date discrepancy, second trimester
        'O26.843',  # Uterine size-date discrepancy, third trimester
        'O26.849',  # Uterine size-date discrepancy, unspecified trimester
        'O26.851',  # Spotting complicating pregnancy, first trimester
        'O26.852',  # Spotting complicating pregnancy, second trimester
        'O26.853',  # Spotting complicating pregnancy, third trimester
        'O26.859',  # Spotting complicating pregnancy, unspecified trimester
        'O26.86',  # Pruritic urticarial papules and plaques of pregnancy (PUPPP)
        'O26.872',  # Cervical shortening, second trimester
        'O26.873',  # Cervical shortening, third trimester
        'O26.879',  # Cervical shortening, unspecified trimester
        'O26.891',  # Other specified pregnancy related conditions, first trimester
        'O26.892',  # Other specified pregnancy related conditions, second trimester
        'O26.893',  # Other specified pregnancy related conditions, third trimester
        'O26.899',  # Other specified pregnancy related conditions, unspecified trimester
        'O26.90',  # Pregnancy related conditions, unspecified, unspecified trimester
        'O26.91',  # Pregnancy related conditions, unspecified, first trimester
        'O26.92',  # Pregnancy related conditions, unspecified, second trimester
        'O26.93',  # Pregnancy related conditions, unspecified, third trimester
        'O28.0',  # Abnormal hematological finding on antenatal screening of mother
        'O28.1',  # Abnormal biochemical finding on antenatal screening of mother
        'O28.2',  # Abnormal cytological finding on antenatal screening of mother
        'O28.3',  # Abnormal ultrasonic finding on antenatal screening of mother
        'O28.4',  # Abnormal radiological finding on antenatal screening of mother
        'O28.5',  # Abnormal chromosomal and genetic finding on antenatal screening of mother
        'O28.8',  # Other abnormal findings on antenatal screening of mother
        'O28.9',  # Unspecified abnormal findings on antenatal screening of mother
        'O29.011',  # Aspiration pneumonitis due to anesthesia during pregnancy, first trimester
        'O29.012',  # Aspiration pneumonitis due to anesthesia during pregnancy, second trimester
        'O29.013',  # Aspiration pneumonitis due to anesthesia during pregnancy, third trimester
        'O29.019',  # Aspiration pneumonitis due to anesthesia during pregnancy, unspecified trimester
        'O29.021',  # Pressure collapse of lung due to anesthesia during pregnancy, first trimester
        'O29.022',  # Pressure collapse of lung due to anesthesia during pregnancy, second trimester
        'O29.023',  # Pressure collapse of lung due to anesthesia during pregnancy, third trimester
        'O29.029',  # Pressure collapse of lung due to anesthesia during pregnancy, unspecified trimester
        'O29.091',  # Other pulmonary complications of anesthesia during pregnancy, first trimester
        'O29.092',  # Other pulmonary complications of anesthesia during pregnancy, second trimester
        'O29.093',  # Other pulmonary complications of anesthesia during pregnancy, third trimester
        'O29.099',  # Other pulmonary complications of anesthesia during pregnancy, unspecified trimester
        'O29.111',  # Cardiac arrest due to anesthesia during pregnancy, first trimester
        'O29.112',  # Cardiac arrest due to anesthesia during pregnancy, second trimester
        'O29.113',  # Cardiac arrest due to anesthesia during pregnancy, third trimester
        'O29.119',  # Cardiac arrest due to anesthesia during pregnancy, unspecified trimester
        'O29.121',  # Cardiac failure due to anesthesia during pregnancy, first trimester
        'O29.122',  # Cardiac failure due to anesthesia during pregnancy, second trimester
        'O29.123',  # Cardiac failure due to anesthesia during pregnancy, third trimester
        'O29.129',  # Cardiac failure due to anesthesia during pregnancy, unspecified trimester
        'O29.191',  # Other cardiac complications of anesthesia during pregnancy, first trimester
        'O29.192',  # Other cardiac complications of anesthesia during pregnancy, second trimester
        'O29.193',  # Other cardiac complications of anesthesia during pregnancy, third trimester
        'O29.199',  # Other cardiac complications of anesthesia during pregnancy, unspecified trimester
        'O29.211',  # Cerebral anoxia due to anesthesia during pregnancy, first trimester
        'O29.212',  # Cerebral anoxia due to anesthesia during pregnancy, second trimester
        'O29.213',  # Cerebral anoxia due to anesthesia during pregnancy, third trimester
        'O29.219',  # Cerebral anoxia due to anesthesia during pregnancy, unspecified trimester
        'O29.291',  # Other central nervous system complications of anesthesia during pregnancy, first trimester
        'O29.292',  # Other central nervous system complications of anesthesia during pregnancy, second trimester
        'O29.293',  # Other central nervous system complications of anesthesia during pregnancy, third trimester
        'O29.299',  # Other central nervous system complications of anesthesia during pregnancy, unspecified trimester
        'O29.3X1',  # Toxic reaction to local anesthesia during pregnancy, first trimester
        'O29.3X2',  # Toxic reaction to local anesthesia during pregnancy, second trimester
        'O29.3X3',  # Toxic reaction to local anesthesia during pregnancy, third trimester
        'O29.3X9',  # Toxic reaction to local anesthesia during pregnancy, unspecified trimester
        'O29.40',  # Spinal and epidural anesthesia induced headache during pregnancy, unspecified trimester
        'O29.41',  # Spinal and epidural anesthesia induced headache during pregnancy, first trimester
        'O29.42',  # Spinal and epidural anesthesia induced headache during pregnancy, second trimester
        'O29.43',  # Spinal and epidural anesthesia induced headache during pregnancy, third trimester
        'O29.5X1',  # Other complications of spinal and epidural anesthesia during pregnancy, first trimester
        'O29.5X2',  # Other complications of spinal and epidural anesthesia during pregnancy, second trimester
        'O29.5X3',  # Other complications of spinal and epidural anesthesia during pregnancy, third trimester
        'O29.5X9',  # Other complications of spinal and epidural anesthesia during pregnancy, unspecified trimester
        'O29.60',  # Failed or difficult intubation for anesthesia during pregnancy, unspecified trimester
        'O29.61',  # Failed or difficult intubation for anesthesia during pregnancy, first trimester
        'O29.62',  # Failed or difficult intubation for anesthesia during pregnancy, second trimester
        'O29.63',  # Failed or difficult intubation for anesthesia during pregnancy, third trimester
        'O29.8X1',  # Other complications of anesthesia during pregnancy, first trimester
        'O29.8X2',  # Other complications of anesthesia during pregnancy, second trimester
        'O29.8X3',  # Other complications of anesthesia during pregnancy, third trimester
        'O29.8X9',  # Other complications of anesthesia during pregnancy, unspecified trimester
        'O29.90',  # Unspecified complication of anesthesia during pregnancy, unspecified trimester
        'O29.91',  # Unspecified complication of anesthesia during pregnancy, first trimester
        'O29.92',  # Unspecified complication of anesthesia during pregnancy, second trimester
        'O29.93',  # Unspecified complication of anesthesia during pregnancy, third trimester
        'O30.001',  # Twin pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, first trimester
        'O30.002',  # Twin pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, second trimester
        'O30.003',  # Twin pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, third trimester
        'O30.009',  # Twin pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, unspecified trimester
        'O30.011',  # Twin pregnancy, monochorionic/monoamniotic, first trimester
        'O30.012',  # Twin pregnancy, monochorionic/monoamniotic, second trimester
        'O30.013',  # Twin pregnancy, monochorionic/monoamniotic, third trimester
        'O30.019',  # Twin pregnancy, monochorionic/monoamniotic, unspecified trimester
        'O30.021',  # Conjoined twin pregnancy, first trimester
        'O30.022',  # Conjoined twin pregnancy, second trimester
        'O30.023',  # Conjoined twin pregnancy, third trimester
        'O30.029',  # Conjoined twin pregnancy, unspecified trimester
        'O30.031',  # Twin pregnancy, monochorionic/diamniotic, first trimester
        'O30.032',  # Twin pregnancy, monochorionic/diamniotic, second trimester
        'O30.033',  # Twin pregnancy, monochorionic/diamniotic, third trimester
        'O30.039',  # Twin pregnancy, monochorionic/diamniotic, unspecified trimester
        'O30.041',  # Twin pregnancy, dichorionic/diamniotic, first trimester
        'O30.042',  # Twin pregnancy, dichorionic/diamniotic, second trimester
        'O30.043',  # Twin pregnancy, dichorionic/diamniotic, third trimester
        'O30.049',  # Twin pregnancy, dichorionic/diamniotic, unspecified trimester
        'O30.091',  # Twin pregnancy, unable to determine number of placenta and number of amniotic sacs, first trimester
        'O30.092',  # Twin pregnancy, unable to determine number of placenta and number of amniotic sacs, second trimester
        'O30.093',  # Twin pregnancy, unable to determine number of placenta and number of amniotic sacs, third trimester
        'O30.099',  # Twin pregnancy, unable to determine number of placenta and number of amniotic sacs, unspecified trimester
        'O30.101',  # Triplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, first trimester
        'O30.102',  # Triplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, second trimester
        'O30.103',  # Triplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, third trimester
        'O30.109',  # Triplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, unspecified trimester
        'O30.111',  # Triplet pregnancy with two or more monochorionic fetuses, first trimester
        'O30.112',  # Triplet pregnancy with two or more monochorionic fetuses, second trimester
        'O30.113',  # Triplet pregnancy with two or more monochorionic fetuses, third trimester
        'O30.119',  # Triplet pregnancy with two or more monochorionic fetuses, unspecified trimester
        'O30.121',  # Triplet pregnancy with two or more monoamniotic fetuses, first trimester
        'O30.122',  # Triplet pregnancy with two or more monoamniotic fetuses, second trimester
        'O30.123',  # Triplet pregnancy with two or more monoamniotic fetuses, third trimester
        'O30.129',  # Triplet pregnancy with two or more monoamniotic fetuses, unspecified trimester
        'O30.131',  # Triplet pregnancy, trichorionic/triamniotic, first trimester
        'O30.132',  # Triplet pregnancy, trichorionic/triamniotic, second trimester
        'O30.133',  # Triplet pregnancy, trichorionic/triamniotic, third trimester
        'O30.139',  # Triplet pregnancy, trichorionic/triamniotic, unspecified trimester
        'O30.191',  # Triplet pregnancy, unable to determine number of placenta and number of amniotic sacs, first trimester
        'O30.192',  # Triplet pregnancy, unable to determine number of placenta and number of amniotic sacs, second trimester
        'O30.193',  # Triplet pregnancy, unable to determine number of placenta and number of amniotic sacs, third trimester
        'O30.199',  # Triplet pregnancy, unable to determine number of placenta and number of amniotic sacs, unspecified trimester
        'O30.201',  # Quadruplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, first trimester
        'O30.202',  # Quadruplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, second trimester
        'O30.203',  # Quadruplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, third trimester
        'O30.209',  # Quadruplet pregnancy, unspecified number of placenta and unspecified number of amniotic sacs, unspecified trimester
        'O30.211',  # Quadruplet pregnancy with two or more monochorionic fetuses, first trimester
        'O30.212',  # Quadruplet pregnancy with two or more monochorionic fetuses, second trimester
        'O30.213',  # Quadruplet pregnancy with two or more monochorionic fetuses, third trimester
        'O30.219',  # Quadruplet pregnancy with two or more monochorionic fetuses, unspecified trimester
        'O30.221',  # Quadruplet pregnancy with two or more monoamniotic fetuses, first trimester
        'O30.222',  # Quadruplet pregnancy with two or more monoamniotic fetuses, second trimester
        'O30.223',  # Quadruplet pregnancy with two or more monoamniotic fetuses, third trimester
        'O30.229',  # Quadruplet pregnancy with two or more monoamniotic fetuses, unspecified trimester
        'O30.231',  # Quadruplet pregnancy, quadrachorionic/quadra-amniotic, first trimester
        'O30.232',  # Quadruplet pregnancy, quadrachorionic/quadra-amniotic, second trimester
        'O30.233',  # Quadruplet pregnancy, quadrachorionic/quadra-amniotic, third trimester
        'O30.239',  # Quadruplet pregnancy, quadrachorionic/quadra-amniotic, unspecified trimester
        'O30.291',  # Quadruplet pregnancy, unable to determine number of placenta and number of amniotic sacs, first trimester
        'O30.292',  # Quadruplet pregnancy, unable to determine number of placenta and number of amniotic sacs, second trimester
        'O30.293',  # Quadruplet pregnancy, unable to determine number of placenta and number of amniotic sacs, third trimester
        'O30.299',  # Quadruplet pregnancy, unable to determine number of placenta and number of amniotic sacs, unspecified trimester
        'O30.801',  # Other specified multiple gestation, unspecified number of placenta and unspecified number of amniotic sacs, first trimester
        'O30.802',  # Other specified multiple gestation, unspecified number of placenta and unspecified number of amniotic sacs, second trimester
        'O30.803',  # Other specified multiple gestation, unspecified number of placenta and unspecified number of amniotic sacs, third trimester
        'O30.809',  # Other specified multiple gestation, unspecified number of placenta and unspecified number of amniotic sacs, unspecified trimester
        'O30.811',  # Other specified multiple gestation with two or more monochorionic fetuses, first trimester
        'O30.812',  # Other specified multiple gestation with two or more monochorionic fetuses, second trimester
        'O30.813',  # Other specified multiple gestation with two or more monochorionic fetuses, third trimester
        'O30.819',  # Other specified multiple gestation with two or more monochorionic fetuses, unspecified trimester
        'O30.821',  # Other specified multiple gestation with two or more monoamniotic fetuses, first trimester
        'O30.822',  # Other specified multiple gestation with two or more monoamniotic fetuses, second trimester
        'O30.823',  # Other specified multiple gestation with two or more monoamniotic fetuses, third trimester
        'O30.829',  # Other specified multiple gestation with two or more monoamniotic fetuses, unspecified trimester
        'O30.831',  # Other specified multiple gestation, number of chorions and amnions are both equal to the number of fetuses, first trimester
        'O30.832',  # Other specified multiple gestation, number of chorions and amnions are both equal to the number of fetuses, second trimester
        'O30.833',  # Other specified multiple gestation, number of chorions and amnions are both equal to the number of fetuses, third trimester
        'O30.839',  # Other specified multiple gestation, number of chorions and amnions are both equal to the number of fetuses, unspecified trimester
        'O30.891',  # Other specified multiple gestation, unable to determine number of placenta and number of amniotic sacs, first trimester
        'O30.892',  # Other specified multiple gestation, unable to determine number of placenta and number of amniotic sacs, second trimester
        'O30.893',  # Other specified multiple gestation, unable to determine number of placenta and number of amniotic sacs, third trimester
        'O30.899',  # Other specified multiple gestation, unable to determine number of placenta and number of amniotic sacs, unspecified trimester
        'O30.90',  # Multiple gestation, unspecified, unspecified trimester
        'O30.91',  # Multiple gestation, unspecified, first trimester
        'O30.92',  # Multiple gestation, unspecified, second trimester
        'O30.93',  # Multiple gestation, unspecified, third trimester
        'O31.00X0',  # Papyraceous fetus, unspecified trimester, not applicable or unspecified
        'O31.00X1',  # Papyraceous fetus, unspecified trimester, fetus 1
        'O31.00X2',  # Papyraceous fetus, unspecified trimester, fetus 2
        'O31.00X3',  # Papyraceous fetus, unspecified trimester, fetus 3
        'O31.00X4',  # Papyraceous fetus, unspecified trimester, fetus 4
        'O31.00X5',  # Papyraceous fetus, unspecified trimester, fetus 5
        'O31.00X9',  # Papyraceous fetus, unspecified trimester, other fetus
        'O31.01X0',  # Papyraceous fetus, first trimester, not applicable or unspecified
        'O31.01X1',  # Papyraceous fetus, first trimester, fetus 1
        'O31.01X2',  # Papyraceous fetus, first trimester, fetus 2
        'O31.01X3',  # Papyraceous fetus, first trimester, fetus 3
        'O31.01X4',  # Papyraceous fetus, first trimester, fetus 4
        'O31.01X5',  # Papyraceous fetus, first trimester, fetus 5
        'O31.01X9',  # Papyraceous fetus, first trimester, other fetus
        'O31.02X0',  # Papyraceous fetus, second trimester, not applicable or unspecified
        'O31.02X1',  # Papyraceous fetus, second trimester, fetus 1
        'O31.02X2',  # Papyraceous fetus, second trimester, fetus 2
        'O31.02X3',  # Papyraceous fetus, second trimester, fetus 3
        'O31.02X4',  # Papyraceous fetus, second trimester, fetus 4
        'O31.02X5',  # Papyraceous fetus, second trimester, fetus 5
        'O31.02X9',  # Papyraceous fetus, second trimester, other fetus
        'O31.03X0',  # Papyraceous fetus, third trimester, not applicable or unspecified
        'O31.03X1',  # Papyraceous fetus, third trimester, fetus 1
        'O31.03X2',  # Papyraceous fetus, third trimester, fetus 2
        'O31.03X3',  # Papyraceous fetus, third trimester, fetus 3
        'O31.03X4',  # Papyraceous fetus, third trimester, fetus 4
        'O31.03X5',  # Papyraceous fetus, third trimester, fetus 5
        'O31.03X9',  # Papyraceous fetus, third trimester, other fetus
        'O31.10X0',  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, not applicable or unspecified
        'O31.10X1',  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, fetus 1
        'O31.10X2',  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, fetus 2
        'O31.10X3',  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, fetus 3
        'O31.10X4',  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, fetus 4
        'O31.10X5',  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, fetus 5
        'O31.10X9',  # Continuing pregnancy after spontaneous abortion of one fetus or more, unspecified trimester, other fetus
        'O31.11X0',  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, not applicable or unspecified
        'O31.11X1',  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, fetus 1
        'O31.11X2',  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, fetus 2
        'O31.11X3',  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, fetus 3
        'O31.11X4',  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, fetus 4
        'O31.11X5',  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, fetus 5
        'O31.11X9',  # Continuing pregnancy after spontaneous abortion of one fetus or more, first trimester, other fetus
        'O31.12X0',  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, not applicable or unspecified
        'O31.12X1',  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, fetus 1
        'O31.12X2',  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, fetus 2
        'O31.12X3',  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, fetus 3
        'O31.12X4',  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, fetus 4
        'O31.12X5',  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, fetus 5
        'O31.12X9',  # Continuing pregnancy after spontaneous abortion of one fetus or more, second trimester, other fetus
        'O31.13X0',  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, not applicable or unspecified
        'O31.13X1',  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, fetus 1
        'O31.13X2',  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, fetus 2
        'O31.13X3',  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, fetus 3
        'O31.13X4',  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, fetus 4
        'O31.13X5',  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, fetus 5
        'O31.13X9',  # Continuing pregnancy after spontaneous abortion of one fetus or more, third trimester, other fetus
        'O31.20X0',  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, not applicable or unspecified
        'O31.20X1',  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, fetus 1
        'O31.20X2',  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, fetus 2
        'O31.20X3',  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, fetus 3
        'O31.20X4',  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, fetus 4
        'O31.20X5',  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, fetus 5
        'O31.20X9',  # Continuing pregnancy after intrauterine death of one fetus or more, unspecified trimester, other fetus
        'O31.21X0',  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, not applicable or unspecified
        'O31.21X1',  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, fetus 1
        'O31.21X2',  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, fetus 2
        'O31.21X3',  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, fetus 3
        'O31.21X4',  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, fetus 4
        'O31.21X5',  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, fetus 5
        'O31.21X9',  # Continuing pregnancy after intrauterine death of one fetus or more, first trimester, other fetus
        'O31.22X0',  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, not applicable or unspecified
        'O31.22X1',  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, fetus 1
        'O31.22X2',  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, fetus 2
        'O31.22X3',  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, fetus 3
        'O31.22X4',  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, fetus 4
        'O31.22X5',  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, fetus 5
        'O31.22X9',  # Continuing pregnancy after intrauterine death of one fetus or more, second trimester, other fetus
        'O31.23X0',  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, not applicable or unspecified
        'O31.23X1',  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, fetus 1
        'O31.23X2',  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, fetus 2
        'O31.23X3',  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, fetus 3
        'O31.23X4',  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, fetus 4
        'O31.23X5',  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, fetus 5
        'O31.23X9',  # Continuing pregnancy after intrauterine death of one fetus or more, third trimester, other fetus
        'O31.30X0',  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, not applicable or unspecified
        'O31.30X1',  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, fetus 1
        'O31.30X2',  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, fetus 2
        'O31.30X3',  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, fetus 3
        'O31.30X4',  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, fetus 4
        'O31.30X5',  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, fetus 5
        'O31.30X9',  # Continuing pregnancy after elective fetal reduction of one fetus or more, unspecified trimester, other fetus
        'O31.31X0',  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, not applicable or unspecified
        'O31.31X1',  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, fetus 1
        'O31.31X2',  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, fetus 2
        'O31.31X3',  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, fetus 3
        'O31.31X4',  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, fetus 4
        'O31.31X5',  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, fetus 5
        'O31.31X9',  # Continuing pregnancy after elective fetal reduction of one fetus or more, first trimester, other fetus
        'O31.32X0',  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, not applicable or unspecified
        'O31.32X1',  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, fetus 1
        'O31.32X2',  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, fetus 2
        'O31.32X3',  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, fetus 3
        'O31.32X4',  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, fetus 4
        'O31.32X5',  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, fetus 5
        'O31.32X9',  # Continuing pregnancy after elective fetal reduction of one fetus or more, second trimester, other fetus
        'O31.33X0',  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, not applicable or unspecified
        'O31.33X1',  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, fetus 1
        'O31.33X2',  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, fetus 2
        'O31.33X3',  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, fetus 3
        'O31.33X4',  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, fetus 4
        'O31.33X5',  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, fetus 5
        'O31.33X9',  # Continuing pregnancy after elective fetal reduction of one fetus or more, third trimester, other fetus
        'O31.8X10',  # Other complications specific to multiple gestation, first trimester, not applicable or unspecified
        'O31.8X11',  # Other complications specific to multiple gestation, first trimester, fetus 1
        'O31.8X12',  # Other complications specific to multiple gestation, first trimester, fetus 2
        'O31.8X13',  # Other complications specific to multiple gestation, first trimester, fetus 3
        'O31.8X14',  # Other complications specific to multiple gestation, first trimester, fetus 4
        'O31.8X15',  # Other complications specific to multiple gestation, first trimester, fetus 5
        'O31.8X19',  # Other complications specific to multiple gestation, first trimester, other fetus
        'O31.8X20',  # Other complications specific to multiple gestation, second trimester, not applicable or unspecified
        'O31.8X21',  # Other complications specific to multiple gestation, second trimester, fetus 1
        'O31.8X22',  # Other complications specific to multiple gestation, second trimester, fetus 2
        'O31.8X23',  # Other complications specific to multiple gestation, second trimester, fetus 3
        'O31.8X24',  # Other complications specific to multiple gestation, second trimester, fetus 4
        'O31.8X25',  # Other complications specific to multiple gestation, second trimester, fetus 5
        'O31.8X29',  # Other complications specific to multiple gestation, second trimester, other fetus
        'O31.8X30',  # Other complications specific to multiple gestation, third trimester, not applicable or unspecified
        'O31.8X31',  # Other complications specific to multiple gestation, third trimester, fetus 1
        'O31.8X32',  # Other complications specific to multiple gestation, third trimester, fetus 2
        'O31.8X33',  # Other complications specific to multiple gestation, third trimester, fetus 3
        'O31.8X34',  # Other complications specific to multiple gestation, third trimester, fetus 4
        'O31.8X35',  # Other complications specific to multiple gestation, third trimester, fetus 5
        'O31.8X39',  # Other complications specific to multiple gestation, third trimester, other fetus
        'O31.8X90',  # Other complications specific to multiple gestation, unspecified trimester, not applicable or unspecified
        'O31.8X91',  # Other complications specific to multiple gestation, unspecified trimester, fetus 1
        'O31.8X92',  # Other complications specific to multiple gestation, unspecified trimester, fetus 2
        'O31.8X93',  # Other complications specific to multiple gestation, unspecified trimester, fetus 3
        'O31.8X94',  # Other complications specific to multiple gestation, unspecified trimester, fetus 4
        'O31.8X95',  # Other complications specific to multiple gestation, unspecified trimester, fetus 5
        'O31.8X99',  # Other complications specific to multiple gestation, unspecified trimester, other fetus
        'O32.0XX0',  # Maternal care for unstable lie, not applicable or unspecified
        'O32.0XX1',  # Maternal care for unstable lie, fetus 1
        'O32.0XX2',  # Maternal care for unstable lie, fetus 2
        'O32.0XX3',  # Maternal care for unstable lie, fetus 3
        'O32.0XX4',  # Maternal care for unstable lie, fetus 4
        'O32.0XX5',  # Maternal care for unstable lie, fetus 5
        'O32.0XX9',  # Maternal care for unstable lie, other fetus
        'O32.1XX0',  # Maternal care for breech presentation, not applicable or unspecified
        'O32.1XX1',  # Maternal care for breech presentation, fetus 1
        'O32.1XX2',  # Maternal care for breech presentation, fetus 2
        'O32.1XX3',  # Maternal care for breech presentation, fetus 3
        'O32.1XX4',  # Maternal care for breech presentation, fetus 4
        'O32.1XX5',  # Maternal care for breech presentation, fetus 5
        'O32.1XX9',  # Maternal care for breech presentation, other fetus
        'O32.2XX0',  # Maternal care for transverse and oblique lie, not applicable or unspecified
        'O32.2XX1',  # Maternal care for transverse and oblique lie, fetus 1
        'O32.2XX2',  # Maternal care for transverse and oblique lie, fetus 2
        'O32.2XX3',  # Maternal care for transverse and oblique lie, fetus 3
        'O32.2XX4',  # Maternal care for transverse and oblique lie, fetus 4
        'O32.2XX5',  # Maternal care for transverse and oblique lie, fetus 5
        'O32.2XX9',  # Maternal care for transverse and oblique lie, other fetus
        'O32.3XX0',  # Maternal care for face, brow and chin presentation, not applicable or unspecified
        'O32.3XX1',  # Maternal care for face, brow and chin presentation, fetus 1
        'O32.3XX2',  # Maternal care for face, brow and chin presentation, fetus 2
        'O32.3XX3',  # Maternal care for face, brow and chin presentation, fetus 3
        'O32.3XX4',  # Maternal care for face, brow and chin presentation, fetus 4
        'O32.3XX5',  # Maternal care for face, brow and chin presentation, fetus 5
        'O32.3XX9',  # Maternal care for face, brow and chin presentation, other fetus
        'O32.4XX0',  # Maternal care for high head at term, not applicable or unspecified
        'O32.4XX1',  # Maternal care for high head at term, fetus 1
        'O32.4XX2',  # Maternal care for high head at term, fetus 2
        'O32.4XX3',  # Maternal care for high head at term, fetus 3
        'O32.4XX4',  # Maternal care for high head at term, fetus 4
        'O32.4XX5',  # Maternal care for high head at term, fetus 5
        'O32.4XX9',  # Maternal care for high head at term, other fetus
        'O32.6XX0',  # Maternal care for compound presentation, not applicable or unspecified
        'O32.6XX1',  # Maternal care for compound presentation, fetus 1
        'O32.6XX2',  # Maternal care for compound presentation, fetus 2
        'O32.6XX3',  # Maternal care for compound presentation, fetus 3
        'O32.6XX4',  # Maternal care for compound presentation, fetus 4
        'O32.6XX5',  # Maternal care for compound presentation, fetus 5
        'O32.6XX9',  # Maternal care for compound presentation, other fetus
        'O32.8XX0',  # Maternal care for other malpresentation of fetus, not applicable or unspecified
        'O32.8XX1',  # Maternal care for other malpresentation of fetus, fetus 1
        'O32.8XX2',  # Maternal care for other malpresentation of fetus, fetus 2
        'O32.8XX3',  # Maternal care for other malpresentation of fetus, fetus 3
        'O32.8XX4',  # Maternal care for other malpresentation of fetus, fetus 4
        'O32.8XX5',  # Maternal care for other malpresentation of fetus, fetus 5
        'O32.8XX9',  # Maternal care for other malpresentation of fetus, other fetus
        'O32.9XX0',  # Maternal care for malpresentation of fetus, unspecified, not applicable or unspecified
        'O32.9XX1',  # Maternal care for malpresentation of fetus, unspecified, fetus 1
        'O32.9XX2',  # Maternal care for malpresentation of fetus, unspecified, fetus 2
        'O32.9XX3',  # Maternal care for malpresentation of fetus, unspecified, fetus 3
        'O32.9XX4',  # Maternal care for malpresentation of fetus, unspecified, fetus 4
        'O32.9XX5',  # Maternal care for malpresentation of fetus, unspecified, fetus 5
        'O32.9XX9',  # Maternal care for malpresentation of fetus, unspecified, other fetus
        'O33.0',  # Maternal care for disproportion due to deformity of maternal pelvic bones
        'O33.1',  # Maternal care for disproportion due to generally contracted pelvis
        'O33.2',  # Maternal care for disproportion due to inlet contraction of pelvis
        'O33.3XX0',  # Maternal care for disproportion due to outlet contraction of pelvis, not applicable or unspecified
        'O33.3XX1',  # Maternal care for disproportion due to outlet contraction of pelvis, fetus 1
        'O33.3XX2',  # Maternal care for disproportion due to outlet contraction of pelvis, fetus 2
        'O33.3XX3',  # Maternal care for disproportion due to outlet contraction of pelvis, fetus 3
        'O33.3XX4',  # Maternal care for disproportion due to outlet contraction of pelvis, fetus 4
        'O33.3XX5',  # Maternal care for disproportion due to outlet contraction of pelvis, fetus 5
        'O33.3XX9',  # Maternal care for disproportion due to outlet contraction of pelvis, other fetus
        'O33.4XX0',  # Maternal care for disproportion of mixed maternal and fetal origin, not applicable or unspecified
        'O33.4XX1',  # Maternal care for disproportion of mixed maternal and fetal origin, fetus 1
        'O33.4XX2',  # Maternal care for disproportion of mixed maternal and fetal origin, fetus 2
        'O33.4XX3',  # Maternal care for disproportion of mixed maternal and fetal origin, fetus 3
        'O33.4XX4',  # Maternal care for disproportion of mixed maternal and fetal origin, fetus 4
        'O33.4XX5',  # Maternal care for disproportion of mixed maternal and fetal origin, fetus 5
        'O33.4XX9',  # Maternal care for disproportion of mixed maternal and fetal origin, other fetus
        'O33.5XX0',  # Maternal care for disproportion due to unusually large fetus, not applicable or unspecified
        'O33.5XX1',  # Maternal care for disproportion due to unusually large fetus, fetus 1
        'O33.5XX2',  # Maternal care for disproportion due to unusually large fetus, fetus 2
        'O33.5XX3',  # Maternal care for disproportion due to unusually large fetus, fetus 3
        'O33.5XX4',  # Maternal care for disproportion due to unusually large fetus, fetus 4
        'O33.5XX5',  # Maternal care for disproportion due to unusually large fetus, fetus 5
        'O33.5XX9',  # Maternal care for disproportion due to unusually large fetus, other fetus
        'O33.6XX0',  # Maternal care for disproportion due to hydrocephalic fetus, not applicable or unspecified
        'O33.6XX1',  # Maternal care for disproportion due to hydrocephalic fetus, fetus 1
        'O33.6XX2',  # Maternal care for disproportion due to hydrocephalic fetus, fetus 2
        'O33.6XX3',  # Maternal care for disproportion due to hydrocephalic fetus, fetus 3
        'O33.6XX4',  # Maternal care for disproportion due to hydrocephalic fetus, fetus 4
        'O33.6XX5',  # Maternal care for disproportion due to hydrocephalic fetus, fetus 5
        'O33.6XX9',  # Maternal care for disproportion due to hydrocephalic fetus, other fetus
        'O33.7XX0',  # Maternal care for disproportion due to other fetal deformities, not applicable or unspecified
        'O33.7XX1',  # Maternal care for disproportion due to other fetal deformities, fetus 1
        'O33.7XX2',  # Maternal care for disproportion due to other fetal deformities, fetus 2
        'O33.7XX3',  # Maternal care for disproportion due to other fetal deformities, fetus 3
        'O33.7XX4',  # Maternal care for disproportion due to other fetal deformities, fetus 4
        'O33.7XX5',  # Maternal care for disproportion due to other fetal deformities, fetus 5
        'O33.7XX9',  # Maternal care for disproportion due to other fetal deformities, other fetus
        'O33.8',  # Maternal care for disproportion of other origin
        'O33.9',  # Maternal care for disproportion, unspecified
        'O34.00',  # Maternal care for unspecified congenital malformation of uterus, unspecified trimester
        'O34.01',  # Maternal care for unspecified congenital malformation of uterus, first trimester
        'O34.02',  # Maternal care for unspecified congenital malformation of uterus, second trimester
        'O34.03',  # Maternal care for unspecified congenital malformation of uterus, third trimester
        'O34.10',  # Maternal care for benign tumor of corpus uteri, unspecified trimester
        'O34.11',  # Maternal care for benign tumor of corpus uteri, first trimester
        'O34.12',  # Maternal care for benign tumor of corpus uteri, second trimester
        'O34.13',  # Maternal care for benign tumor of corpus uteri, third trimester
        'O34.211',  # Maternal care for low transverse scar from previous cesarean delivery
        'O34.212',  # Maternal care for vertical scar from previous cesarean delivery
        'O34.219',  # Maternal care for unspecified type scar from previous cesarean delivery
        'O34.29',  # Maternal care due to uterine scar from other previous surgery
        'O34.30',  # Maternal care for cervical incompetence, unspecified trimester
        'O34.31',  # Maternal care for cervical incompetence, first trimester
        'O34.32',  # Maternal care for cervical incompetence, second trimester
        'O34.33',  # Maternal care for cervical incompetence, third trimester
        'O34.40',  # Maternal care for other abnormalities of cervix, unspecified trimester
        'O34.41',  # Maternal care for other abnormalities of cervix, first trimester
        'O34.42',  # Maternal care for other abnormalities of cervix, second trimester
        'O34.43',  # Maternal care for other abnormalities of cervix, third trimester
        'O34.511',  # Maternal care for incarceration of gravid uterus, first trimester
        'O34.512',  # Maternal care for incarceration of gravid uterus, second trimester
        'O34.513',  # Maternal care for incarceration of gravid uterus, third trimester
        'O34.519',  # Maternal care for incarceration of gravid uterus, unspecified trimester
        'O34.521',  # Maternal care for prolapse of gravid uterus, first trimester
        'O34.522',  # Maternal care for prolapse of gravid uterus, second trimester
        'O34.523',  # Maternal care for prolapse of gravid uterus, third trimester
        'O34.529',  # Maternal care for prolapse of gravid uterus, unspecified trimester
        'O34.531',  # Maternal care for retroversion of gravid uterus, first trimester
        'O34.532',  # Maternal care for retroversion of gravid uterus, second trimester
        'O34.533',  # Maternal care for retroversion of gravid uterus, third trimester
        'O34.539',  # Maternal care for retroversion of gravid uterus, unspecified trimester
        'O34.591',  # Maternal care for other abnormalities of gravid uterus, first trimester
        'O34.592',  # Maternal care for other abnormalities of gravid uterus, second trimester
        'O34.593',  # Maternal care for other abnormalities of gravid uterus, third trimester
        'O34.599',  # Maternal care for other abnormalities of gravid uterus, unspecified trimester
        'O34.60',  # Maternal care for abnormality of vagina, unspecified trimester
        'O34.61',  # Maternal care for abnormality of vagina, first trimester
        'O34.62',  # Maternal care for abnormality of vagina, second trimester
        'O34.63',  # Maternal care for abnormality of vagina, third trimester
        'O34.70',  # Maternal care for abnormality of vulva and perineum, unspecified trimester
        'O34.71',  # Maternal care for abnormality of vulva and perineum, first trimester
        'O34.72',  # Maternal care for abnormality of vulva and perineum, second trimester
        'O34.73',  # Maternal care for abnormality of vulva and perineum, third trimester
        'O34.80',  # Maternal care for other abnormalities of pelvic organs, unspecified trimester
        'O34.81',  # Maternal care for other abnormalities of pelvic organs, first trimester
        'O34.82',  # Maternal care for other abnormalities of pelvic organs, second trimester
        'O34.83',  # Maternal care for other abnormalities of pelvic organs, third trimester
        'O34.90',  # Maternal care for abnormality of pelvic organ, unspecified, unspecified trimester
        'O34.91',  # Maternal care for abnormality of pelvic organ, unspecified, first trimester
        'O34.92',  # Maternal care for abnormality of pelvic organ, unspecified, second trimester
        'O34.93',  # Maternal care for abnormality of pelvic organ, unspecified, third trimester
        'O35.0XX0',  # Maternal care for (suspected) central nervous system malformation in fetus, not applicable or unspecified
        'O35.0XX1',  # Maternal care for (suspected) central nervous system malformation in fetus, fetus 1
        'O35.0XX2',  # Maternal care for (suspected) central nervous system malformation in fetus, fetus 2
        'O35.0XX3',  # Maternal care for (suspected) central nervous system malformation in fetus, fetus 3
        'O35.0XX4',  # Maternal care for (suspected) central nervous system malformation in fetus, fetus 4
        'O35.0XX5',  # Maternal care for (suspected) central nervous system malformation in fetus, fetus 5
        'O35.0XX9',  # Maternal care for (suspected) central nervous system malformation in fetus, other fetus
        'O35.1XX0',  # Maternal care for (suspected) chromosomal abnormality in fetus, not applicable or unspecified
        'O35.1XX1',  # Maternal care for (suspected) chromosomal abnormality in fetus, fetus 1
        'O35.1XX2',  # Maternal care for (suspected) chromosomal abnormality in fetus, fetus 2
        'O35.1XX3',  # Maternal care for (suspected) chromosomal abnormality in fetus, fetus 3
        'O35.1XX4',  # Maternal care for (suspected) chromosomal abnormality in fetus, fetus 4
        'O35.1XX5',  # Maternal care for (suspected) chromosomal abnormality in fetus, fetus 5
        'O35.1XX9',  # Maternal care for (suspected) chromosomal abnormality in fetus, other fetus
        'O35.2XX0',  # Maternal care for (suspected) hereditary disease in fetus, not applicable or unspecified
        'O35.2XX1',  # Maternal care for (suspected) hereditary disease in fetus, fetus 1
        'O35.2XX2',  # Maternal care for (suspected) hereditary disease in fetus, fetus 2
        'O35.2XX3',  # Maternal care for (suspected) hereditary disease in fetus, fetus 3
        'O35.2XX4',  # Maternal care for (suspected) hereditary disease in fetus, fetus 4
        'O35.2XX5',  # Maternal care for (suspected) hereditary disease in fetus, fetus 5
        'O35.2XX9',  # Maternal care for (suspected) hereditary disease in fetus, other fetus
        'O35.3XX0',  # Maternal care for (suspected) damage to fetus from viral disease in mother, not applicable or unspecified
        'O35.3XX1',  # Maternal care for (suspected) damage to fetus from viral disease in mother, fetus 1
        'O35.3XX2',  # Maternal care for (suspected) damage to fetus from viral disease in mother, fetus 2
        'O35.3XX3',  # Maternal care for (suspected) damage to fetus from viral disease in mother, fetus 3
        'O35.3XX4',  # Maternal care for (suspected) damage to fetus from viral disease in mother, fetus 4
        'O35.3XX5',  # Maternal care for (suspected) damage to fetus from viral disease in mother, fetus 5
        'O35.3XX9',  # Maternal care for (suspected) damage to fetus from viral disease in mother, other fetus
        'O35.4XX0',  # Maternal care for (suspected) damage to fetus from alcohol, not applicable or unspecified
        'O35.4XX1',  # Maternal care for (suspected) damage to fetus from alcohol, fetus 1
        'O35.4XX2',  # Maternal care for (suspected) damage to fetus from alcohol, fetus 2
        'O35.4XX3',  # Maternal care for (suspected) damage to fetus from alcohol, fetus 3
        'O35.4XX4',  # Maternal care for (suspected) damage to fetus from alcohol, fetus 4
        'O35.4XX5',  # Maternal care for (suspected) damage to fetus from alcohol, fetus 5
        'O35.4XX9',  # Maternal care for (suspected) damage to fetus from alcohol, other fetus
        'O35.5XX0',  # Maternal care for (suspected) damage to fetus by drugs, not applicable or unspecified
        'O35.5XX1',  # Maternal care for (suspected) damage to fetus by drugs, fetus 1
        'O35.5XX2',  # Maternal care for (suspected) damage to fetus by drugs, fetus 2
        'O35.5XX3',  # Maternal care for (suspected) damage to fetus by drugs, fetus 3
        'O35.5XX4',  # Maternal care for (suspected) damage to fetus by drugs, fetus 4
        'O35.5XX5',  # Maternal care for (suspected) damage to fetus by drugs, fetus 5
        'O35.5XX9',  # Maternal care for (suspected) damage to fetus by drugs, other fetus
        'O35.6XX0',  # Maternal care for (suspected) damage to fetus by radiation, not applicable or unspecified
        'O35.6XX1',  # Maternal care for (suspected) damage to fetus by radiation, fetus 1
        'O35.6XX2',  # Maternal care for (suspected) damage to fetus by radiation, fetus 2
        'O35.6XX3',  # Maternal care for (suspected) damage to fetus by radiation, fetus 3
        'O35.6XX4',  # Maternal care for (suspected) damage to fetus by radiation, fetus 4
        'O35.6XX5',  # Maternal care for (suspected) damage to fetus by radiation, fetus 5
        'O35.6XX9',  # Maternal care for (suspected) damage to fetus by radiation, other fetus
        'O35.7XX0',  # Maternal care for (suspected) damage to fetus by other medical procedures, not applicable or unspecified
        'O35.7XX1',  # Maternal care for (suspected) damage to fetus by other medical procedures, fetus 1
        'O35.7XX2',  # Maternal care for (suspected) damage to fetus by other medical procedures, fetus 2
        'O35.7XX3',  # Maternal care for (suspected) damage to fetus by other medical procedures, fetus 3
        'O35.7XX4',  # Maternal care for (suspected) damage to fetus by other medical procedures, fetus 4
        'O35.7XX5',  # Maternal care for (suspected) damage to fetus by other medical procedures, fetus 5
        'O35.7XX9',  # Maternal care for (suspected) damage to fetus by other medical procedures, other fetus
        'O35.8XX0',  # Maternal care for other (suspected) fetal abnormality and damage, not applicable or unspecified
        'O35.8XX1',  # Maternal care for other (suspected) fetal abnormality and damage, fetus 1
        'O35.8XX2',  # Maternal care for other (suspected) fetal abnormality and damage, fetus 2
        'O35.8XX3',  # Maternal care for other (suspected) fetal abnormality and damage, fetus 3
        'O35.8XX4',  # Maternal care for other (suspected) fetal abnormality and damage, fetus 4
        'O35.8XX5',  # Maternal care for other (suspected) fetal abnormality and damage, fetus 5
        'O35.8XX9',  # Maternal care for other (suspected) fetal abnormality and damage, other fetus
        'O35.9XX0',  # Maternal care for (suspected) fetal abnormality and damage, unspecified, not applicable or unspecified
        'O35.9XX1',  # Maternal care for (suspected) fetal abnormality and damage, unspecified, fetus 1
        'O35.9XX2',  # Maternal care for (suspected) fetal abnormality and damage, unspecified, fetus 2
        'O35.9XX3',  # Maternal care for (suspected) fetal abnormality and damage, unspecified, fetus 3
        'O35.9XX4',  # Maternal care for (suspected) fetal abnormality and damage, unspecified, fetus 4
        'O35.9XX5',  # Maternal care for (suspected) fetal abnormality and damage, unspecified, fetus 5
        'O35.9XX9',  # Maternal care for (suspected) fetal abnormality and damage, unspecified, other fetus
        'O36.0110',  # Maternal care for anti-D [Rh] antibodies, first trimester, not applicable or unspecified
        'O36.0111',  # Maternal care for anti-D [Rh] antibodies, first trimester, fetus 1
        'O36.0112',  # Maternal care for anti-D [Rh] antibodies, first trimester, fetus 2
        'O36.0113',  # Maternal care for anti-D [Rh] antibodies, first trimester, fetus 3
        'O36.0114',  # Maternal care for anti-D [Rh] antibodies, first trimester, fetus 4
        'O36.0115',  # Maternal care for anti-D [Rh] antibodies, first trimester, fetus 5
        'O36.0119',  # Maternal care for anti-D [Rh] antibodies, first trimester, other fetus
        'O36.0120',  # Maternal care for anti-D [Rh] antibodies, second trimester, not applicable or unspecified
        'O36.0121',  # Maternal care for anti-D [Rh] antibodies, second trimester, fetus 1
        'O36.0122',  # Maternal care for anti-D [Rh] antibodies, second trimester, fetus 2
        'O36.0123',  # Maternal care for anti-D [Rh] antibodies, second trimester, fetus 3
        'O36.0124',  # Maternal care for anti-D [Rh] antibodies, second trimester, fetus 4
        'O36.0125',  # Maternal care for anti-D [Rh] antibodies, second trimester, fetus 5
        'O36.0129',  # Maternal care for anti-D [Rh] antibodies, second trimester, other fetus
        'O36.0130',  # Maternal care for anti-D [Rh] antibodies, third trimester, not applicable or unspecified
        'O36.0131',  # Maternal care for anti-D [Rh] antibodies, third trimester, fetus 1
        'O36.0132',  # Maternal care for anti-D [Rh] antibodies, third trimester, fetus 2
        'O36.0133',  # Maternal care for anti-D [Rh] antibodies, third trimester, fetus 3
        'O36.0134',  # Maternal care for anti-D [Rh] antibodies, third trimester, fetus 4
        'O36.0135',  # Maternal care for anti-D [Rh] antibodies, third trimester, fetus 5
        'O36.0139',  # Maternal care for anti-D [Rh] antibodies, third trimester, other fetus
        'O36.0190',  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, not applicable or unspecified
        'O36.0191',  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, fetus 1
        'O36.0192',  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, fetus 2
        'O36.0193',  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, fetus 3
        'O36.0194',  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, fetus 4
        'O36.0195',  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, fetus 5
        'O36.0199',  # Maternal care for anti-D [Rh] antibodies, unspecified trimester, other fetus
        'O36.0910',  # Maternal care for other rhesus isoimmunization, first trimester, not applicable or unspecified
        'O36.0911',  # Maternal care for other rhesus isoimmunization, first trimester, fetus 1
        'O36.0912',  # Maternal care for other rhesus isoimmunization, first trimester, fetus 2
        'O36.0913',  # Maternal care for other rhesus isoimmunization, first trimester, fetus 3
        'O36.0914',  # Maternal care for other rhesus isoimmunization, first trimester, fetus 4
        'O36.0915',  # Maternal care for other rhesus isoimmunization, first trimester, fetus 5
        'O36.0919',  # Maternal care for other rhesus isoimmunization, first trimester, other fetus
        'O36.0920',  # Maternal care for other rhesus isoimmunization, second trimester, not applicable or unspecified
        'O36.0921',  # Maternal care for other rhesus isoimmunization, second trimester, fetus 1
        'O36.0922',  # Maternal care for other rhesus isoimmunization, second trimester, fetus 2
        'O36.0923',  # Maternal care for other rhesus isoimmunization, second trimester, fetus 3
        'O36.0924',  # Maternal care for other rhesus isoimmunization, second trimester, fetus 4
        'O36.0925',  # Maternal care for other rhesus isoimmunization, second trimester, fetus 5
        'O36.0929',  # Maternal care for other rhesus isoimmunization, second trimester, other fetus
        'O36.0930',  # Maternal care for other rhesus isoimmunization, third trimester, not applicable or unspecified
        'O36.0931',  # Maternal care for other rhesus isoimmunization, third trimester, fetus 1
        'O36.0932',  # Maternal care for other rhesus isoimmunization, third trimester, fetus 2
        'O36.0933',  # Maternal care for other rhesus isoimmunization, third trimester, fetus 3
        'O36.0934',  # Maternal care for other rhesus isoimmunization, third trimester, fetus 4
        'O36.0935',  # Maternal care for other rhesus isoimmunization, third trimester, fetus 5
        'O36.0939',  # Maternal care for other rhesus isoimmunization, third trimester, other fetus
        'O36.0990',  # Maternal care for other rhesus isoimmunization, unspecified trimester, not applicable or unspecified
        'O36.0991',  # Maternal care for other rhesus isoimmunization, unspecified trimester, fetus 1
        'O36.0992',  # Maternal care for other rhesus isoimmunization, unspecified trimester, fetus 2
        'O36.0993',  # Maternal care for other rhesus isoimmunization, unspecified trimester, fetus 3
        'O36.0994',  # Maternal care for other rhesus isoimmunization, unspecified trimester, fetus 4
        'O36.0995',  # Maternal care for other rhesus isoimmunization, unspecified trimester, fetus 5
        'O36.0999',  # Maternal care for other rhesus isoimmunization, unspecified trimester, other fetus
        'O36.1110',  # Maternal care for Anti-A sensitization, first trimester, not applicable or unspecified
        'O36.1111',  # Maternal care for Anti-A sensitization, first trimester, fetus 1
        'O36.1112',  # Maternal care for Anti-A sensitization, first trimester, fetus 2
        'O36.1113',  # Maternal care for Anti-A sensitization, first trimester, fetus 3
        'O36.1114',  # Maternal care for Anti-A sensitization, first trimester, fetus 4
        'O36.1115',  # Maternal care for Anti-A sensitization, first trimester, fetus 5
        'O36.1119',  # Maternal care for Anti-A sensitization, first trimester, other fetus
        'O36.1120',  # Maternal care for Anti-A sensitization, second trimester, not applicable or unspecified
        'O36.1121',  # Maternal care for Anti-A sensitization, second trimester, fetus 1
        'O36.1122',  # Maternal care for Anti-A sensitization, second trimester, fetus 2
        'O36.1123',  # Maternal care for Anti-A sensitization, second trimester, fetus 3
        'O36.1124',  # Maternal care for Anti-A sensitization, second trimester, fetus 4
        'O36.1125',  # Maternal care for Anti-A sensitization, second trimester, fetus 5
        'O36.1129',  # Maternal care for Anti-A sensitization, second trimester, other fetus
        'O36.1130',  # Maternal care for Anti-A sensitization, third trimester, not applicable or unspecified
        'O36.1131',  # Maternal care for Anti-A sensitization, third trimester, fetus 1
        'O36.1132',  # Maternal care for Anti-A sensitization, third trimester, fetus 2
        'O36.1133',  # Maternal care for Anti-A sensitization, third trimester, fetus 3
        'O36.1134',  # Maternal care for Anti-A sensitization, third trimester, fetus 4
        'O36.1135',  # Maternal care for Anti-A sensitization, third trimester, fetus 5
        'O36.1139',  # Maternal care for Anti-A sensitization, third trimester, other fetus
        'O36.1190',  # Maternal care for Anti-A sensitization, unspecified trimester, not applicable or unspecified
        'O36.1191',  # Maternal care for Anti-A sensitization, unspecified trimester, fetus 1
        'O36.1192',  # Maternal care for Anti-A sensitization, unspecified trimester, fetus 2
        'O36.1193',  # Maternal care for Anti-A sensitization, unspecified trimester, fetus 3
        'O36.1194',  # Maternal care for Anti-A sensitization, unspecified trimester, fetus 4
        'O36.1195',  # Maternal care for Anti-A sensitization, unspecified trimester, fetus 5
        'O36.1199',  # Maternal care for Anti-A sensitization, unspecified trimester, other fetus
        'O36.1910',  # Maternal care for other isoimmunization, first trimester, not applicable or unspecified
        'O36.1911',  # Maternal care for other isoimmunization, first trimester, fetus 1
        'O36.1912',  # Maternal care for other isoimmunization, first trimester, fetus 2
        'O36.1913',  # Maternal care for other isoimmunization, first trimester, fetus 3
        'O36.1914',  # Maternal care for other isoimmunization, first trimester, fetus 4
        'O36.1915',  # Maternal care for other isoimmunization, first trimester, fetus 5
        'O36.1919',  # Maternal care for other isoimmunization, first trimester, other fetus
        'O36.1920',  # Maternal care for other isoimmunization, second trimester, not applicable or unspecified
        'O36.1921',  # Maternal care for other isoimmunization, second trimester, fetus 1
        'O36.1922',  # Maternal care for other isoimmunization, second trimester, fetus 2
        'O36.1923',  # Maternal care for other isoimmunization, second trimester, fetus 3
        'O36.1924',  # Maternal care for other isoimmunization, second trimester, fetus 4
        'O36.1925',  # Maternal care for other isoimmunization, second trimester, fetus 5
        'O36.1929',  # Maternal care for other isoimmunization, second trimester, other fetus
        'O36.1930',  # Maternal care for other isoimmunization, third trimester, not applicable or unspecified
        'O36.1931',  # Maternal care for other isoimmunization, third trimester, fetus 1
        'O36.1932',  # Maternal care for other isoimmunization, third trimester, fetus 2
        'O36.1933',  # Maternal care for other isoimmunization, third trimester, fetus 3
        'O36.1934',  # Maternal care for other isoimmunization, third trimester, fetus 4
        'O36.1935',  # Maternal care for other isoimmunization, third trimester, fetus 5
        'O36.1939',  # Maternal care for other isoimmunization, third trimester, other fetus
        'O36.1990',  # Maternal care for other isoimmunization, unspecified trimester, not applicable or unspecified
        'O36.1991',  # Maternal care for other isoimmunization, unspecified trimester, fetus 1
        'O36.1992',  # Maternal care for other isoimmunization, unspecified trimester, fetus 2
        'O36.1993',  # Maternal care for other isoimmunization, unspecified trimester, fetus 3
        'O36.1994',  # Maternal care for other isoimmunization, unspecified trimester, fetus 4
        'O36.1995',  # Maternal care for other isoimmunization, unspecified trimester, fetus 5
        'O36.1999',  # Maternal care for other isoimmunization, unspecified trimester, other fetus
        'O36.20X0',  # Maternal care for hydrops fetalis, unspecified trimester, not applicable or unspecified
        'O36.20X1',  # Maternal care for hydrops fetalis, unspecified trimester, fetus 1
        'O36.20X2',  # Maternal care for hydrops fetalis, unspecified trimester, fetus 2
        'O36.20X3',  # Maternal care for hydrops fetalis, unspecified trimester, fetus 3
        'O36.20X4',  # Maternal care for hydrops fetalis, unspecified trimester, fetus 4
        'O36.20X5',  # Maternal care for hydrops fetalis, unspecified trimester, fetus 5
        'O36.20X9',  # Maternal care for hydrops fetalis, unspecified trimester, other fetus
        'O36.21X0',  # Maternal care for hydrops fetalis, first trimester, not applicable or unspecified
        'O36.21X1',  # Maternal care for hydrops fetalis, first trimester, fetus 1
        'O36.21X2',  # Maternal care for hydrops fetalis, first trimester, fetus 2
        'O36.21X3',  # Maternal care for hydrops fetalis, first trimester, fetus 3
        'O36.21X4',  # Maternal care for hydrops fetalis, first trimester, fetus 4
        'O36.21X5',  # Maternal care for hydrops fetalis, first trimester, fetus 5
        'O36.21X9',  # Maternal care for hydrops fetalis, first trimester, other fetus
        'O36.22X0',  # Maternal care for hydrops fetalis, second trimester, not applicable or unspecified
        'O36.22X1',  # Maternal care for hydrops fetalis, second trimester, fetus 1
        'O36.22X2',  # Maternal care for hydrops fetalis, second trimester, fetus 2
        'O36.22X3',  # Maternal care for hydrops fetalis, second trimester, fetus 3
        'O36.22X4',  # Maternal care for hydrops fetalis, second trimester, fetus 4
        'O36.22X5',  # Maternal care for hydrops fetalis, second trimester, fetus 5
        'O36.22X9',  # Maternal care for hydrops fetalis, second trimester, other fetus
        'O36.23X0',  # Maternal care for hydrops fetalis, third trimester, not applicable or unspecified
        'O36.23X1',  # Maternal care for hydrops fetalis, third trimester, fetus 1
        'O36.23X2',  # Maternal care for hydrops fetalis, third trimester, fetus 2
        'O36.23X3',  # Maternal care for hydrops fetalis, third trimester, fetus 3
        'O36.23X4',  # Maternal care for hydrops fetalis, third trimester, fetus 4
        'O36.23X5',  # Maternal care for hydrops fetalis, third trimester, fetus 5
        'O36.23X9',  # Maternal care for hydrops fetalis, third trimester, other fetus
        'O36.4XX0',  # Maternal care for intrauterine death, not applicable or unspecified
        'O36.4XX1',  # Maternal care for intrauterine death, fetus 1
        'O36.4XX2',  # Maternal care for intrauterine death, fetus 2
        'O36.4XX3',  # Maternal care for intrauterine death, fetus 3
        'O36.4XX4',  # Maternal care for intrauterine death, fetus 4
        'O36.4XX5',  # Maternal care for intrauterine death, fetus 5
        'O36.4XX9',  # Maternal care for intrauterine death, other fetus
        'O36.5110',  # Maternal care for known or suspected placental insufficiency, first trimester, not applicable or unspecified
        'O36.5111',  # Maternal care for known or suspected placental insufficiency, first trimester, fetus 1
        'O36.5112',  # Maternal care for known or suspected placental insufficiency, first trimester, fetus 2
        'O36.5113',  # Maternal care for known or suspected placental insufficiency, first trimester, fetus 3
        'O36.5114',  # Maternal care for known or suspected placental insufficiency, first trimester, fetus 4
        'O36.5115',  # Maternal care for known or suspected placental insufficiency, first trimester, fetus 5
        'O36.5119',  # Maternal care for known or suspected placental insufficiency, first trimester, other fetus
        'O36.5120',  # Maternal care for known or suspected placental insufficiency, second trimester, not applicable or unspecified
        'O36.5121',  # Maternal care for known or suspected placental insufficiency, second trimester, fetus 1
        'O36.5122',  # Maternal care for known or suspected placental insufficiency, second trimester, fetus 2
        'O36.5123',  # Maternal care for known or suspected placental insufficiency, second trimester, fetus 3
        'O36.5124',  # Maternal care for known or suspected placental insufficiency, second trimester, fetus 4
        'O36.5125',  # Maternal care for known or suspected placental insufficiency, second trimester, fetus 5
        'O36.5129',  # Maternal care for known or suspected placental insufficiency, second trimester, other fetus
        'O36.5130',  # Maternal care for known or suspected placental insufficiency, third trimester, not applicable or unspecified
        'O36.5131',  # Maternal care for known or suspected placental insufficiency, third trimester, fetus 1
        'O36.5132',  # Maternal care for known or suspected placental insufficiency, third trimester, fetus 2
        'O36.5133',  # Maternal care for known or suspected placental insufficiency, third trimester, fetus 3
        'O36.5134',  # Maternal care for known or suspected placental insufficiency, third trimester, fetus 4
        'O36.5135',  # Maternal care for known or suspected placental insufficiency, third trimester, fetus 5
        'O36.5139',  # Maternal care for known or suspected placental insufficiency, third trimester, other fetus
        'O36.5190',  # Maternal care for known or suspected placental insufficiency, unspecified trimester, not applicable or unspecified
        'O36.5191',  # Maternal care for known or suspected placental insufficiency, unspecified trimester, fetus 1
        'O36.5192',  # Maternal care for known or suspected placental insufficiency, unspecified trimester, fetus 2
        'O36.5193',  # Maternal care for known or suspected placental insufficiency, unspecified trimester, fetus 3
        'O36.5194',  # Maternal care for known or suspected placental insufficiency, unspecified trimester, fetus 4
        'O36.5195',  # Maternal care for known or suspected placental insufficiency, unspecified trimester, fetus 5
        'O36.5199',  # Maternal care for known or suspected placental insufficiency, unspecified trimester, other fetus
        'O36.5910',  # Maternal care for other known or suspected poor fetal growth, first trimester, not applicable or unspecified
        'O36.5911',  # Maternal care for other known or suspected poor fetal growth, first trimester, fetus 1
        'O36.5912',  # Maternal care for other known or suspected poor fetal growth, first trimester, fetus 2
        'O36.5913',  # Maternal care for other known or suspected poor fetal growth, first trimester, fetus 3
        'O36.5914',  # Maternal care for other known or suspected poor fetal growth, first trimester, fetus 4
        'O36.5915',  # Maternal care for other known or suspected poor fetal growth, first trimester, fetus 5
        'O36.5919',  # Maternal care for other known or suspected poor fetal growth, first trimester, other fetus
        'O36.5920',  # Maternal care for other known or suspected poor fetal growth, second trimester, not applicable or unspecified
        'O36.5921',  # Maternal care for other known or suspected poor fetal growth, second trimester, fetus 1
        'O36.5922',  # Maternal care for other known or suspected poor fetal growth, second trimester, fetus 2
        'O36.5923',  # Maternal care for other known or suspected poor fetal growth, second trimester, fetus 3
        'O36.5924',  # Maternal care for other known or suspected poor fetal growth, second trimester, fetus 4
        'O36.5925',  # Maternal care for other known or suspected poor fetal growth, second trimester, fetus 5
        'O36.5929',  # Maternal care for other known or suspected poor fetal growth, second trimester, other fetus
        'O36.5930',  # Maternal care for other known or suspected poor fetal growth, third trimester, not applicable or unspecified
        'O36.5931',  # Maternal care for other known or suspected poor fetal growth, third trimester, fetus 1
        'O36.5932',  # Maternal care for other known or suspected poor fetal growth, third trimester, fetus 2
        'O36.5933',  # Maternal care for other known or suspected poor fetal growth, third trimester, fetus 3
        'O36.5934',  # Maternal care for other known or suspected poor fetal growth, third trimester, fetus 4
        'O36.5935',  # Maternal care for other known or suspected poor fetal growth, third trimester, fetus 5
        'O36.5939',  # Maternal care for other known or suspected poor fetal growth, third trimester, other fetus
        'O36.5990',  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, not applicable or unspecified
        'O36.5991',  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, fetus 1
        'O36.5992',  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, fetus 2
        'O36.5993',  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, fetus 3
        'O36.5994',  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, fetus 4
        'O36.5995',  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, fetus 5
        'O36.5999',  # Maternal care for other known or suspected poor fetal growth, unspecified trimester, other fetus
        'O36.60X0',  # Maternal care for excessive fetal growth, unspecified trimester, not applicable or unspecified
        'O36.60X1',  # Maternal care for excessive fetal growth, unspecified trimester, fetus 1
        'O36.60X2',  # Maternal care for excessive fetal growth, unspecified trimester, fetus 2
        'O36.60X3',  # Maternal care for excessive fetal growth, unspecified trimester, fetus 3
        'O36.60X4',  # Maternal care for excessive fetal growth, unspecified trimester, fetus 4
        'O36.60X5',  # Maternal care for excessive fetal growth, unspecified trimester, fetus 5
        'O36.60X9',  # Maternal care for excessive fetal growth, unspecified trimester, other fetus
        'O36.61X0',  # Maternal care for excessive fetal growth, first trimester, not applicable or unspecified
        'O36.61X1',  # Maternal care for excessive fetal growth, first trimester, fetus 1
        'O36.61X2',  # Maternal care for excessive fetal growth, first trimester, fetus 2
        'O36.61X3',  # Maternal care for excessive fetal growth, first trimester, fetus 3
        'O36.61X4',  # Maternal care for excessive fetal growth, first trimester, fetus 4
        'O36.61X5',  # Maternal care for excessive fetal growth, first trimester, fetus 5
        'O36.61X9',  # Maternal care for excessive fetal growth, first trimester, other fetus
        'O36.62X0',  # Maternal care for excessive fetal growth, second trimester, not applicable or unspecified
        'O36.62X1',  # Maternal care for excessive fetal growth, second trimester, fetus 1
        'O36.62X2',  # Maternal care for excessive fetal growth, second trimester, fetus 2
        'O36.62X3',  # Maternal care for excessive fetal growth, second trimester, fetus 3
        'O36.62X4',  # Maternal care for excessive fetal growth, second trimester, fetus 4
        'O36.62X5',  # Maternal care for excessive fetal growth, second trimester, fetus 5
        'O36.62X9',  # Maternal care for excessive fetal growth, second trimester, other fetus
        'O36.63X0',  # Maternal care for excessive fetal growth, third trimester, not applicable or unspecified
        'O36.63X1',  # Maternal care for excessive fetal growth, third trimester, fetus 1
        'O36.63X2',  # Maternal care for excessive fetal growth, third trimester, fetus 2
        'O36.63X3',  # Maternal care for excessive fetal growth, third trimester, fetus 3
        'O36.63X4',  # Maternal care for excessive fetal growth, third trimester, fetus 4
        'O36.63X5',  # Maternal care for excessive fetal growth, third trimester, fetus 5
        'O36.63X9',  # Maternal care for excessive fetal growth, third trimester, other fetus
        'O36.70X0',  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, not applicable or unspecified
        'O36.70X1',  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, fetus 1
        'O36.70X2',  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, fetus 2
        'O36.70X3',  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, fetus 3
        'O36.70X4',  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, fetus 4
        'O36.70X5',  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, fetus 5
        'O36.70X9',  # Maternal care for viable fetus in abdominal pregnancy, unspecified trimester, other fetus
        'O36.71X0',  # Maternal care for viable fetus in abdominal pregnancy, first trimester, not applicable or unspecified
        'O36.71X1',  # Maternal care for viable fetus in abdominal pregnancy, first trimester, fetus 1
        'O36.71X2',  # Maternal care for viable fetus in abdominal pregnancy, first trimester, fetus 2
        'O36.71X3',  # Maternal care for viable fetus in abdominal pregnancy, first trimester, fetus 3
        'O36.71X4',  # Maternal care for viable fetus in abdominal pregnancy, first trimester, fetus 4
        'O36.71X5',  # Maternal care for viable fetus in abdominal pregnancy, first trimester, fetus 5
        'O36.71X9',  # Maternal care for viable fetus in abdominal pregnancy, first trimester, other fetus
        'O36.72X0',  # Maternal care for viable fetus in abdominal pregnancy, second trimester, not applicable or unspecified
        'O36.72X1',  # Maternal care for viable fetus in abdominal pregnancy, second trimester, fetus 1
        'O36.72X2',  # Maternal care for viable fetus in abdominal pregnancy, second trimester, fetus 2
        'O36.72X3',  # Maternal care for viable fetus in abdominal pregnancy, second trimester, fetus 3
        'O36.72X4',  # Maternal care for viable fetus in abdominal pregnancy, second trimester, fetus 4
        'O36.72X5',  # Maternal care for viable fetus in abdominal pregnancy, second trimester, fetus 5
        'O36.72X9',  # Maternal care for viable fetus in abdominal pregnancy, second trimester, other fetus
        'O36.73X0',  # Maternal care for viable fetus in abdominal pregnancy, third trimester, not applicable or unspecified
        'O36.73X1',  # Maternal care for viable fetus in abdominal pregnancy, third trimester, fetus 1
        'O36.73X2',  # Maternal care for viable fetus in abdominal pregnancy, third trimester, fetus 2
        'O36.73X3',  # Maternal care for viable fetus in abdominal pregnancy, third trimester, fetus 3
        'O36.73X4',  # Maternal care for viable fetus in abdominal pregnancy, third trimester, fetus 4
        'O36.73X5',  # Maternal care for viable fetus in abdominal pregnancy, third trimester, fetus 5
        'O36.73X9',  # Maternal care for viable fetus in abdominal pregnancy, third trimester, other fetus
        'O36.8120',  # Decreased fetal movements, second trimester, not applicable or unspecified
        'O36.8121',  # Decreased fetal movements, second trimester, fetus 1
        'O36.8122',  # Decreased fetal movements, second trimester, fetus 2
        'O36.8123',  # Decreased fetal movements, second trimester, fetus 3
        'O36.8124',  # Decreased fetal movements, second trimester, fetus 4
        'O36.8125',  # Decreased fetal movements, second trimester, fetus 5
        'O36.8129',  # Decreased fetal movements, second trimester, other fetus
        'O36.8130',  # Decreased fetal movements, third trimester, not applicable or unspecified
        'O36.8131',  # Decreased fetal movements, third trimester, fetus 1
        'O36.8132',  # Decreased fetal movements, third trimester, fetus 2
        'O36.8133',  # Decreased fetal movements, third trimester, fetus 3
        'O36.8134',  # Decreased fetal movements, third trimester, fetus 4
        'O36.8135',  # Decreased fetal movements, third trimester, fetus 5
        'O36.8139',  # Decreased fetal movements, third trimester, other fetus
        'O36.8190',  # Decreased fetal movements, unspecified trimester, not applicable or unspecified
        'O36.8191',  # Decreased fetal movements, unspecified trimester, fetus 1
        'O36.8192',  # Decreased fetal movements, unspecified trimester, fetus 2
        'O36.8193',  # Decreased fetal movements, unspecified trimester, fetus 3
        'O36.8194',  # Decreased fetal movements, unspecified trimester, fetus 4
        'O36.8195',  # Decreased fetal movements, unspecified trimester, fetus 5
        'O36.8199',  # Decreased fetal movements, unspecified trimester, other fetus
        'O36.8210',  # Fetal anemia and thrombocytopenia, first trimester, not applicable or unspecified
        'O36.8211',  # Fetal anemia and thrombocytopenia, first trimester, fetus 1
        'O36.8212',  # Fetal anemia and thrombocytopenia, first trimester, fetus 2
        'O36.8213',  # Fetal anemia and thrombocytopenia, first trimester, fetus 3
        'O36.8214',  # Fetal anemia and thrombocytopenia, first trimester, fetus 4
        'O36.8215',  # Fetal anemia and thrombocytopenia, first trimester, fetus 5
        'O36.8219',  # Fetal anemia and thrombocytopenia, first trimester, other fetus
        'O36.8220',  # Fetal anemia and thrombocytopenia, second trimester, not applicable or unspecified
        'O36.8221',  # Fetal anemia and thrombocytopenia, second trimester, fetus 1
        'O36.8222',  # Fetal anemia and thrombocytopenia, second trimester, fetus 2
        'O36.8223',  # Fetal anemia and thrombocytopenia, second trimester, fetus 3
        'O36.8224',  # Fetal anemia and thrombocytopenia, second trimester, fetus 4
        'O36.8225',  # Fetal anemia and thrombocytopenia, second trimester, fetus 5
        'O36.8229',  # Fetal anemia and thrombocytopenia, second trimester, other fetus
        'O36.8230',  # Fetal anemia and thrombocytopenia, third trimester, not applicable or unspecified
        'O36.8231',  # Fetal anemia and thrombocytopenia, third trimester, fetus 1
        'O36.8232',  # Fetal anemia and thrombocytopenia, third trimester, fetus 2
        'O36.8233',  # Fetal anemia and thrombocytopenia, third trimester, fetus 3
        'O36.8234',  # Fetal anemia and thrombocytopenia, third trimester, fetus 4
        'O36.8235',  # Fetal anemia and thrombocytopenia, third trimester, fetus 5
        'O36.8239',  # Fetal anemia and thrombocytopenia, third trimester, other fetus
        'O36.8290',  # Fetal anemia and thrombocytopenia, unspecified trimester, not applicable or unspecified
        'O36.8291',  # Fetal anemia and thrombocytopenia, unspecified trimester, fetus 1
        'O36.8292',  # Fetal anemia and thrombocytopenia, unspecified trimester, fetus 2
        'O36.8293',  # Fetal anemia and thrombocytopenia, unspecified trimester, fetus 3
        'O36.8294',  # Fetal anemia and thrombocytopenia, unspecified trimester, fetus 4
        'O36.8295',  # Fetal anemia and thrombocytopenia, unspecified trimester, fetus 5
        'O36.8299',  # Fetal anemia and thrombocytopenia, unspecified trimester, other fetus
        'O36.8310',  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, not applicable or unspecified
        'O36.8311',  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, fetus 1
        'O36.8312',  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, fetus 2
        'O36.8313',  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, fetus 3
        'O36.8314',  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, fetus 4
        'O36.8315',  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, fetus 5
        'O36.8319',  # Maternal care for abnormalities of the fetal heart rate or rhythm, first trimester, other fetus
        'O36.8320',  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, not applicable or unspecified
        'O36.8321',  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, fetus 1
        'O36.8322',  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, fetus 2
        'O36.8323',  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, fetus 3
        'O36.8324',  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, fetus 4
        'O36.8325',  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, fetus 5
        'O36.8329',  # Maternal care for abnormalities of the fetal heart rate or rhythm, second trimester, other fetus
        'O36.8330',  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, not applicable or unspecified
        'O36.8331',  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, fetus 1
        'O36.8332',  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, fetus 2
        'O36.8333',  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, fetus 3
        'O36.8334',  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, fetus 4
        'O36.8335',  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, fetus 5
        'O36.8339',  # Maternal care for abnormalities of the fetal heart rate or rhythm, third trimester, other fetus
        'O36.8390',  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, not applicable or unspecified
        'O36.8391',  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, fetus 1
        'O36.8392',  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, fetus 2
        'O36.8393',  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, fetus 3
        'O36.8394',  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, fetus 4
        'O36.8395',  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, fetus 5
        'O36.8399',  # Maternal care for abnormalities of the fetal heart rate or rhythm, unspecified trimester, other fetus
        'O36.8910',  # Maternal care for other specified fetal problems, first trimester, not applicable or unspecified
        'O36.8911',  # Maternal care for other specified fetal problems, first trimester, fetus 1
        'O36.8912',  # Maternal care for other specified fetal problems, first trimester, fetus 2
        'O36.8913',  # Maternal care for other specified fetal problems, first trimester, fetus 3
        'O36.8914',  # Maternal care for other specified fetal problems, first trimester, fetus 4
        'O36.8915',  # Maternal care for other specified fetal problems, first trimester, fetus 5
        'O36.8919',  # Maternal care for other specified fetal problems, first trimester, other fetus
        'O36.8920',  # Maternal care for other specified fetal problems, second trimester, not applicable or unspecified
        'O36.8921',  # Maternal care for other specified fetal problems, second trimester, fetus 1
        'O36.8922',  # Maternal care for other specified fetal problems, second trimester, fetus 2
        'O36.8923',  # Maternal care for other specified fetal problems, second trimester, fetus 3
        'O36.8924',  # Maternal care for other specified fetal problems, second trimester, fetus 4
        'O36.8925',  # Maternal care for other specified fetal problems, second trimester, fetus 5
        'O36.8929',  # Maternal care for other specified fetal problems, second trimester, other fetus
        'O36.8930',  # Maternal care for other specified fetal problems, third trimester, not applicable or unspecified
        'O36.8931',  # Maternal care for other specified fetal problems, third trimester, fetus 1
        'O36.8932',  # Maternal care for other specified fetal problems, third trimester, fetus 2
        'O36.8933',  # Maternal care for other specified fetal problems, third trimester, fetus 3
        'O36.8934',  # Maternal care for other specified fetal problems, third trimester, fetus 4
        'O36.8935',  # Maternal care for other specified fetal problems, third trimester, fetus 5
        'O36.8939',  # Maternal care for other specified fetal problems, third trimester, other fetus
        'O36.8990',  # Maternal care for other specified fetal problems, unspecified trimester, not applicable or unspecified
        'O36.8991',  # Maternal care for other specified fetal problems, unspecified trimester, fetus 1
        'O36.8992',  # Maternal care for other specified fetal problems, unspecified trimester, fetus 2
        'O36.8993',  # Maternal care for other specified fetal problems, unspecified trimester, fetus 3
        'O36.8994',  # Maternal care for other specified fetal problems, unspecified trimester, fetus 4
        'O36.8995',  # Maternal care for other specified fetal problems, unspecified trimester, fetus 5
        'O36.8999',  # Maternal care for other specified fetal problems, unspecified trimester, other fetus
        'O36.90X0',  # Maternal care for fetal problem, unspecified, unspecified trimester, not applicable or unspecified
        'O36.90X1',  # Maternal care for fetal problem, unspecified, unspecified trimester, fetus 1
        'O36.90X2',  # Maternal care for fetal problem, unspecified, unspecified trimester, fetus 2
        'O36.90X3',  # Maternal care for fetal problem, unspecified, unspecified trimester, fetus 3
        'O36.90X4',  # Maternal care for fetal problem, unspecified, unspecified trimester, fetus 4
        'O36.90X5',  # Maternal care for fetal problem, unspecified, unspecified trimester, fetus 5
        'O36.90X9',  # Maternal care for fetal problem, unspecified, unspecified trimester, other fetus
        'O36.91X0',  # Maternal care for fetal problem, unspecified, first trimester, not applicable or unspecified
        'O36.91X1',  # Maternal care for fetal problem, unspecified, first trimester, fetus 1
        'O36.91X2',  # Maternal care for fetal problem, unspecified, first trimester, fetus 2
        'O36.91X3',  # Maternal care for fetal problem, unspecified, first trimester, fetus 3
        'O36.91X4',  # Maternal care for fetal problem, unspecified, first trimester, fetus 4
        'O36.91X5',  # Maternal care for fetal problem, unspecified, first trimester, fetus 5
        'O36.91X9',  # Maternal care for fetal problem, unspecified, first trimester, other fetus
        'O36.92X0',  # Maternal care for fetal problem, unspecified, second trimester, not applicable or unspecified
        'O36.92X1',  # Maternal care for fetal problem, unspecified, second trimester, fetus 1
        'O36.92X2',  # Maternal care for fetal problem, unspecified, second trimester, fetus 2
        'O36.92X3',  # Maternal care for fetal problem, unspecified, second trimester, fetus 3
        'O36.92X4',  # Maternal care for fetal problem, unspecified, second trimester, fetus 4
        'O36.92X5',  # Maternal care for fetal problem, unspecified, second trimester, fetus 5
        'O36.92X9',  # Maternal care for fetal problem, unspecified, second trimester, other fetus
        'O36.93X0',  # Maternal care for fetal problem, unspecified, third trimester, not applicable or unspecified
        'O36.93X1',  # Maternal care for fetal problem, unspecified, third trimester, fetus 1
        'O36.93X2',  # Maternal care for fetal problem, unspecified, third trimester, fetus 2
        'O36.93X3',  # Maternal care for fetal problem, unspecified, third trimester, fetus 3
        'O36.93X4',  # Maternal care for fetal problem, unspecified, third trimester, fetus 4
        'O36.93X5',  # Maternal care for fetal problem, unspecified, third trimester, fetus 5
        'O36.93X9',  # Maternal care for fetal problem, unspecified, third trimester, other fetus
        'O40.1XX0',  # Polyhydramnios, first trimester, not applicable or unspecified
        'O40.1XX1',  # Polyhydramnios, first trimester, fetus 1
        'O40.1XX2',  # Polyhydramnios, first trimester, fetus 2
        'O40.1XX3',  # Polyhydramnios, first trimester, fetus 3
        'O40.1XX4',  # Polyhydramnios, first trimester, fetus 4
        'O40.1XX5',  # Polyhydramnios, first trimester, fetus 5
        'O40.1XX9',  # Polyhydramnios, first trimester, other fetus
        'O40.2XX0',  # Polyhydramnios, second trimester, not applicable or unspecified
        'O40.2XX1',  # Polyhydramnios, second trimester, fetus 1
        'O40.2XX2',  # Polyhydramnios, second trimester, fetus 2
        'O40.2XX3',  # Polyhydramnios, second trimester, fetus 3
        'O40.2XX4',  # Polyhydramnios, second trimester, fetus 4
        'O40.2XX5',  # Polyhydramnios, second trimester, fetus 5
        'O40.2XX9',  # Polyhydramnios, second trimester, other fetus
        'O40.3XX0',  # Polyhydramnios, third trimester, not applicable or unspecified
        'O40.3XX1',  # Polyhydramnios, third trimester, fetus 1
        'O40.3XX2',  # Polyhydramnios, third trimester, fetus 2
        'O40.3XX3',  # Polyhydramnios, third trimester, fetus 3
        'O40.3XX4',  # Polyhydramnios, third trimester, fetus 4
        'O40.3XX5',  # Polyhydramnios, third trimester, fetus 5
        'O40.3XX9',  # Polyhydramnios, third trimester, other fetus
        'O40.9XX0',  # Polyhydramnios, unspecified trimester, not applicable or unspecified
        'O40.9XX1',  # Polyhydramnios, unspecified trimester, fetus 1
        'O40.9XX2',  # Polyhydramnios, unspecified trimester, fetus 2
        'O40.9XX3',  # Polyhydramnios, unspecified trimester, fetus 3
        'O40.9XX4',  # Polyhydramnios, unspecified trimester, fetus 4
        'O40.9XX5',  # Polyhydramnios, unspecified trimester, fetus 5
        'O40.9XX9',  # Polyhydramnios, unspecified trimester, other fetus
        'O41.00X0',  # Oligohydramnios, unspecified trimester, not applicable or unspecified
        'O41.00X1',  # Oligohydramnios, unspecified trimester, fetus 1
        'O41.00X2',  # Oligohydramnios, unspecified trimester, fetus 2
        'O41.00X3',  # Oligohydramnios, unspecified trimester, fetus 3
        'O41.00X4',  # Oligohydramnios, unspecified trimester, fetus 4
        'O41.00X5',  # Oligohydramnios, unspecified trimester, fetus 5
        'O41.00X9',  # Oligohydramnios, unspecified trimester, other fetus
        'O41.01X0',  # Oligohydramnios, first trimester, not applicable or unspecified
        'O41.01X1',  # Oligohydramnios, first trimester, fetus 1
        'O41.01X2',  # Oligohydramnios, first trimester, fetus 2
        'O41.01X3',  # Oligohydramnios, first trimester, fetus 3
        'O41.01X4',  # Oligohydramnios, first trimester, fetus 4
        'O41.01X5',  # Oligohydramnios, first trimester, fetus 5
        'O41.01X9',  # Oligohydramnios, first trimester, other fetus
        'O41.02X0',  # Oligohydramnios, second trimester, not applicable or unspecified
        'O41.02X1',  # Oligohydramnios, second trimester, fetus 1
        'O41.02X2',  # Oligohydramnios, second trimester, fetus 2
        'O41.02X3',  # Oligohydramnios, second trimester, fetus 3
        'O41.02X4',  # Oligohydramnios, second trimester, fetus 4
        'O41.02X5',  # Oligohydramnios, second trimester, fetus 5
        'O41.02X9',  # Oligohydramnios, second trimester, other fetus
        'O41.03X0',  # Oligohydramnios, third trimester, not applicable or unspecified
        'O41.03X1',  # Oligohydramnios, third trimester, fetus 1
        'O41.03X2',  # Oligohydramnios, third trimester, fetus 2
        'O41.03X3',  # Oligohydramnios, third trimester, fetus 3
        'O41.03X4',  # Oligohydramnios, third trimester, fetus 4
        'O41.03X5',  # Oligohydramnios, third trimester, fetus 5
        'O41.03X9',  # Oligohydramnios, third trimester, other fetus
        'O41.1010',  # Infection of amniotic sac and membranes, unspecified, first trimester, not applicable or unspecified
        'O41.1011',  # Infection of amniotic sac and membranes, unspecified, first trimester, fetus 1
        'O41.1012',  # Infection of amniotic sac and membranes, unspecified, first trimester, fetus 2
        'O41.1013',  # Infection of amniotic sac and membranes, unspecified, first trimester, fetus 3
        'O41.1014',  # Infection of amniotic sac and membranes, unspecified, first trimester, fetus 4
        'O41.1015',  # Infection of amniotic sac and membranes, unspecified, first trimester, fetus 5
        'O41.1019',  # Infection of amniotic sac and membranes, unspecified, first trimester, other fetus
        'O41.1020',  # Infection of amniotic sac and membranes, unspecified, second trimester, not applicable or unspecified
        'O41.1021',  # Infection of amniotic sac and membranes, unspecified, second trimester, fetus 1
        'O41.1022',  # Infection of amniotic sac and membranes, unspecified, second trimester, fetus 2
        'O41.1023',  # Infection of amniotic sac and membranes, unspecified, second trimester, fetus 3
        'O41.1024',  # Infection of amniotic sac and membranes, unspecified, second trimester, fetus 4
        'O41.1025',  # Infection of amniotic sac and membranes, unspecified, second trimester, fetus 5
        'O41.1029',  # Infection of amniotic sac and membranes, unspecified, second trimester, other fetus
        'O41.1030',  # Infection of amniotic sac and membranes, unspecified, third trimester, not applicable or unspecified
        'O41.1031',  # Infection of amniotic sac and membranes, unspecified, third trimester, fetus 1
        'O41.1032',  # Infection of amniotic sac and membranes, unspecified, third trimester, fetus 2
        'O41.1033',  # Infection of amniotic sac and membranes, unspecified, third trimester, fetus 3
        'O41.1034',  # Infection of amniotic sac and membranes, unspecified, third trimester, fetus 4
        'O41.1035',  # Infection of amniotic sac and membranes, unspecified, third trimester, fetus 5
        'O41.1039',  # Infection of amniotic sac and membranes, unspecified, third trimester, other fetus
        'O41.1090',  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, not applicable or unspecified
        'O41.1091',  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, fetus 1
        'O41.1092',  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, fetus 2
        'O41.1093',  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, fetus 3
        'O41.1094',  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, fetus 4
        'O41.1095',  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, fetus 5
        'O41.1099',  # Infection of amniotic sac and membranes, unspecified, unspecified trimester, other fetus
        'O41.1210',  # Chorioamnionitis, first trimester, not applicable or unspecified
        'O41.1211',  # Chorioamnionitis, first trimester, fetus 1
        'O41.1212',  # Chorioamnionitis, first trimester, fetus 2
        'O41.1213',  # Chorioamnionitis, first trimester, fetus 3
        'O41.1214',  # Chorioamnionitis, first trimester, fetus 4
        'O41.1215',  # Chorioamnionitis, first trimester, fetus 5
        'O41.1219',  # Chorioamnionitis, first trimester, other fetus
        'O41.1220',  # Chorioamnionitis, second trimester, not applicable or unspecified
        'O41.1221',  # Chorioamnionitis, second trimester, fetus 1
        'O41.1222',  # Chorioamnionitis, second trimester, fetus 2
        'O41.1223',  # Chorioamnionitis, second trimester, fetus 3
        'O41.1224',  # Chorioamnionitis, second trimester, fetus 4
        'O41.1225',  # Chorioamnionitis, second trimester, fetus 5
        'O41.1229',  # Chorioamnionitis, second trimester, other fetus
        'O41.1230',  # Chorioamnionitis, third trimester, not applicable or unspecified
        'O41.1231',  # Chorioamnionitis, third trimester, fetus 1
        'O41.1232',  # Chorioamnionitis, third trimester, fetus 2
        'O41.1233',  # Chorioamnionitis, third trimester, fetus 3
        'O41.1234',  # Chorioamnionitis, third trimester, fetus 4
        'O41.1235',  # Chorioamnionitis, third trimester, fetus 5
        'O41.1239',  # Chorioamnionitis, third trimester, other fetus
        'O41.1290',  # Chorioamnionitis, unspecified trimester, not applicable or unspecified
        'O41.1291',  # Chorioamnionitis, unspecified trimester, fetus 1
        'O41.1292',  # Chorioamnionitis, unspecified trimester, fetus 2
        'O41.1293',  # Chorioamnionitis, unspecified trimester, fetus 3
        'O41.1294',  # Chorioamnionitis, unspecified trimester, fetus 4
        'O41.1295',  # Chorioamnionitis, unspecified trimester, fetus 5
        'O41.1299',  # Chorioamnionitis, unspecified trimester, other fetus
        'O41.1410',  # Placentitis, first trimester, not applicable or unspecified
        'O41.1411',  # Placentitis, first trimester, fetus 1
        'O41.1412',  # Placentitis, first trimester, fetus 2
        'O41.1413',  # Placentitis, first trimester, fetus 3
        'O41.1414',  # Placentitis, first trimester, fetus 4
        'O41.1415',  # Placentitis, first trimester, fetus 5
        'O41.1419',  # Placentitis, first trimester, other fetus
        'O41.1420',  # Placentitis, second trimester, not applicable or unspecified
        'O41.1421',  # Placentitis, second trimester, fetus 1
        'O41.1422',  # Placentitis, second trimester, fetus 2
        'O41.1423',  # Placentitis, second trimester, fetus 3
        'O41.1424',  # Placentitis, second trimester, fetus 4
        'O41.1425',  # Placentitis, second trimester, fetus 5
        'O41.1429',  # Placentitis, second trimester, other fetus
        'O41.1430',  # Placentitis, third trimester, not applicable or unspecified
        'O41.1431',  # Placentitis, third trimester, fetus 1
        'O41.1432',  # Placentitis, third trimester, fetus 2
        'O41.1433',  # Placentitis, third trimester, fetus 3
        'O41.1434',  # Placentitis, third trimester, fetus 4
        'O41.1435',  # Placentitis, third trimester, fetus 5
        'O41.1439',  # Placentitis, third trimester, other fetus
        'O41.1490',  # Placentitis, unspecified trimester, not applicable or unspecified
        'O41.1491',  # Placentitis, unspecified trimester, fetus 1
        'O41.1492',  # Placentitis, unspecified trimester, fetus 2
        'O41.1493',  # Placentitis, unspecified trimester, fetus 3
        'O41.1494',  # Placentitis, unspecified trimester, fetus 4
        'O41.1495',  # Placentitis, unspecified trimester, fetus 5
        'O41.1499',  # Placentitis, unspecified trimester, other fetus
        'O41.8X10',  # Other specified disorders of amniotic fluid and membranes, first trimester, not applicable or unspecified
        'O41.8X11',  # Other specified disorders of amniotic fluid and membranes, first trimester, fetus 1
        'O41.8X12',  # Other specified disorders of amniotic fluid and membranes, first trimester, fetus 2
        'O41.8X13',  # Other specified disorders of amniotic fluid and membranes, first trimester, fetus 3
        'O41.8X14',  # Other specified disorders of amniotic fluid and membranes, first trimester, fetus 4
        'O41.8X15',  # Other specified disorders of amniotic fluid and membranes, first trimester, fetus 5
        'O41.8X19',  # Other specified disorders of amniotic fluid and membranes, first trimester, other fetus
        'O41.8X20',  # Other specified disorders of amniotic fluid and membranes, second trimester, not applicable or unspecified
        'O41.8X21',  # Other specified disorders of amniotic fluid and membranes, second trimester, fetus 1
        'O41.8X22',  # Other specified disorders of amniotic fluid and membranes, second trimester, fetus 2
        'O41.8X23',  # Other specified disorders of amniotic fluid and membranes, second trimester, fetus 3
        'O41.8X24',  # Other specified disorders of amniotic fluid and membranes, second trimester, fetus 4
        'O41.8X25',  # Other specified disorders of amniotic fluid and membranes, second trimester, fetus 5
        'O41.8X29',  # Other specified disorders of amniotic fluid and membranes, second trimester, other fetus
        'O41.8X30',  # Other specified disorders of amniotic fluid and membranes, third trimester, not applicable or unspecified
        'O41.8X31',  # Other specified disorders of amniotic fluid and membranes, third trimester, fetus 1
        'O41.8X32',  # Other specified disorders of amniotic fluid and membranes, third trimester, fetus 2
        'O41.8X33',  # Other specified disorders of amniotic fluid and membranes, third trimester, fetus 3
        'O41.8X34',  # Other specified disorders of amniotic fluid and membranes, third trimester, fetus 4
        'O41.8X35',  # Other specified disorders of amniotic fluid and membranes, third trimester, fetus 5
        'O41.8X39',  # Other specified disorders of amniotic fluid and membranes, third trimester, other fetus
        'O41.8X90',  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, not applicable or unspecified
        'O41.8X91',  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, fetus 1
        'O41.8X92',  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, fetus 2
        'O41.8X93',  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, fetus 3
        'O41.8X94',  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, fetus 4
        'O41.8X95',  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, fetus 5
        'O41.8X99',  # Other specified disorders of amniotic fluid and membranes, unspecified trimester, other fetus
        'O41.90X0',  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, not applicable or unspecified
        'O41.90X1',  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, fetus 1
        'O41.90X2',  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, fetus 2
        'O41.90X3',  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, fetus 3
        'O41.90X4',  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, fetus 4
        'O41.90X5',  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, fetus 5
        'O41.90X9',  # Disorder of amniotic fluid and membranes, unspecified, unspecified trimester, other fetus
        'O41.91X0',  # Disorder of amniotic fluid and membranes, unspecified, first trimester, not applicable or unspecified
        'O41.91X1',  # Disorder of amniotic fluid and membranes, unspecified, first trimester, fetus 1
        'O41.91X2',  # Disorder of amniotic fluid and membranes, unspecified, first trimester, fetus 2
        'O41.91X3',  # Disorder of amniotic fluid and membranes, unspecified, first trimester, fetus 3
        'O41.91X4',  # Disorder of amniotic fluid and membranes, unspecified, first trimester, fetus 4
        'O41.91X5',  # Disorder of amniotic fluid and membranes, unspecified, first trimester, fetus 5
        'O41.91X9',  # Disorder of amniotic fluid and membranes, unspecified, first trimester, other fetus
        'O41.92X0',  # Disorder of amniotic fluid and membranes, unspecified, second trimester, not applicable or unspecified
        'O41.92X1',  # Disorder of amniotic fluid and membranes, unspecified, second trimester, fetus 1
        'O41.92X2',  # Disorder of amniotic fluid and membranes, unspecified, second trimester, fetus 2
        'O41.92X3',  # Disorder of amniotic fluid and membranes, unspecified, second trimester, fetus 3
        'O41.92X4',  # Disorder of amniotic fluid and membranes, unspecified, second trimester, fetus 4
        'O41.92X5',  # Disorder of amniotic fluid and membranes, unspecified, second trimester, fetus 5
        'O41.92X9',  # Disorder of amniotic fluid and membranes, unspecified, second trimester, other fetus
        'O41.93X0',  # Disorder of amniotic fluid and membranes, unspecified, third trimester, not applicable or unspecified
        'O41.93X1',  # Disorder of amniotic fluid and membranes, unspecified, third trimester, fetus 1
        'O41.93X2',  # Disorder of amniotic fluid and membranes, unspecified, third trimester, fetus 2
        'O41.93X3',  # Disorder of amniotic fluid and membranes, unspecified, third trimester, fetus 3
        'O41.93X4',  # Disorder of amniotic fluid and membranes, unspecified, third trimester, fetus 4
        'O41.93X5',  # Disorder of amniotic fluid and membranes, unspecified, third trimester, fetus 5
        'O41.93X9',  # Disorder of amniotic fluid and membranes, unspecified, third trimester, other fetus
        'O42.00',  # Premature rupture of membranes, onset of labor within 24 hours of rupture, unspecified weeks of gestation
        'O42.011',  # Preterm premature rupture of membranes, onset of labor within 24 hours of rupture, first trimester
        'O42.012',  # Preterm premature rupture of membranes, onset of labor within 24 hours of rupture, second trimester
        'O42.013',  # Preterm premature rupture of membranes, onset of labor within 24 hours of rupture, third trimester
        'O42.019',  # Preterm premature rupture of membranes, onset of labor within 24 hours of rupture, unspecified trimester
        'O42.02',  # Full-term premature rupture of membranes, onset of labor within 24 hours of rupture
        'O42.10',  # Premature rupture of membranes, onset of labor more than 24 hours following rupture, unspecified weeks of gestation
        'O42.111',  # Preterm premature rupture of membranes, onset of labor more than 24 hours following rupture, first trimester
        'O42.112',  # Preterm premature rupture of membranes, onset of labor more than 24 hours following rupture, second trimester
        'O42.113',  # Preterm premature rupture of membranes, onset of labor more than 24 hours following rupture, third trimester
        'O42.119',  # Preterm premature rupture of membranes, onset of labor more than 24 hours following rupture, unspecified trimester
        'O42.12',  # Full-term premature rupture of membranes, onset of labor more than 24 hours following rupture
        'O42.90',  # Premature rupture of membranes, unspecified as to length of time between rupture and onset of labor, unspecified weeks of gestation
        'O42.911',  # Preterm premature rupture of membranes, unspecified as to length of time between rupture and onset of labor, first trimester
        'O42.912',  # Preterm premature rupture of membranes, unspecified as to length of time between rupture and onset of labor, second trimester
        'O42.913',  # Preterm premature rupture of membranes, unspecified as to length of time between rupture and onset of labor, third trimester
        'O42.919',  # Preterm premature rupture of membranes, unspecified as to length of time between rupture and onset of labor, unspecified trimester
        'O42.92',  # Full-term premature rupture of membranes, unspecified as to length of time between rupture and onset of labor
        'O43.011',  # Fetomaternal placental transfusion syndrome, first trimester
        'O43.012',  # Fetomaternal placental transfusion syndrome, second trimester
        'O43.013',  # Fetomaternal placental transfusion syndrome, third trimester
        'O43.019',  # Fetomaternal placental transfusion syndrome, unspecified trimester
        'O43.021',  # Fetus-to-fetus placental transfusion syndrome, first trimester
        'O43.022',  # Fetus-to-fetus placental transfusion syndrome, second trimester
        'O43.023',  # Fetus-to-fetus placental transfusion syndrome, third trimester
        'O43.029',  # Fetus-to-fetus placental transfusion syndrome, unspecified trimester
        'O43.101',  # Malformation of placenta, unspecified, first trimester
        'O43.102',  # Malformation of placenta, unspecified, second trimester
        'O43.103',  # Malformation of placenta, unspecified, third trimester
        'O43.109',  # Malformation of placenta, unspecified, unspecified trimester
        'O43.111',  # Circumvallate placenta, first trimester
        'O43.112',  # Circumvallate placenta, second trimester
        'O43.113',  # Circumvallate placenta, third trimester
        'O43.119',  # Circumvallate placenta, unspecified trimester
        'O43.121',  # Velamentous insertion of umbilical cord, first trimester
        'O43.122',  # Velamentous insertion of umbilical cord, second trimester
        'O43.123',  # Velamentous insertion of umbilical cord, third trimester
        'O43.129',  # Velamentous insertion of umbilical cord, unspecified trimester
        'O43.191',  # Other malformation of placenta, first trimester
        'O43.192',  # Other malformation of placenta, second trimester
        'O43.193',  # Other malformation of placenta, third trimester
        'O43.199',  # Other malformation of placenta, unspecified trimester
        'O43.211',  # Placenta accreta, first trimester
        'O43.212',  # Placenta accreta, second trimester
        'O43.213',  # Placenta accreta, third trimester
        'O43.219',  # Placenta accreta, unspecified trimester
        'O43.221',  # Placenta increta, first trimester
        'O43.222',  # Placenta increta, second trimester
        'O43.223',  # Placenta increta, third trimester
        'O43.229',  # Placenta increta, unspecified trimester
        'O43.231',  # Placenta percreta, first trimester
        'O43.232',  # Placenta percreta, second trimester
        'O43.233',  # Placenta percreta, third trimester
        'O43.239',  # Placenta percreta, unspecified trimester
        'O43.811',  # Placental infarction, first trimester
        'O43.812',  # Placental infarction, second trimester
        'O43.813',  # Placental infarction, third trimester
        'O43.819',  # Placental infarction, unspecified trimester
        'O43.891',  # Other placental disorders, first trimester
        'O43.892',  # Other placental disorders, second trimester
        'O43.893',  # Other placental disorders, third trimester
        'O43.899',  # Other placental disorders, unspecified trimester
        'O43.90',  # Unspecified placental disorder, unspecified trimester
        'O43.91',  # Unspecified placental disorder, first trimester
        'O43.92',  # Unspecified placental disorder, second trimester
        'O43.93',  # Unspecified placental disorder, third trimester
        'O44.00',  # Complete placenta previa NOS or without hemorrhage, unspecified trimester
        'O44.01',  # Complete placenta previa NOS or without hemorrhage, first trimester
        'O44.02',  # Complete placenta previa NOS or without hemorrhage, second trimester
        'O44.03',  # Complete placenta previa NOS or without hemorrhage, third trimester
        'O44.10',  # Complete placenta previa with hemorrhage, unspecified trimester
        'O44.11',  # Complete placenta previa with hemorrhage, first trimester
        'O44.12',  # Complete placenta previa with hemorrhage, second trimester
        'O44.13',  # Complete placenta previa with hemorrhage, third trimester
        'O44.20',  # Partial placenta previa NOS or without hemorrhage, unspecified trimester
        'O44.21',  # Partial placenta previa NOS or without hemorrhage, first trimester
        'O44.22',  # Partial placenta previa NOS or without hemorrhage, second trimester
        'O44.23',  # Partial placenta previa NOS or without hemorrhage, third trimester
        'O44.30',  # Partial placenta previa with hemorrhage, unspecified trimester
        'O44.31',  # Partial placenta previa with hemorrhage, first trimester
        'O44.32',  # Partial placenta previa with hemorrhage, second trimester
        'O44.33',  # Partial placenta previa with hemorrhage, third trimester
        'O44.40',  # Low lying placenta NOS or without hemorrhage, unspecified trimester
        'O44.41',  # Low lying placenta NOS or without hemorrhage, first trimester
        'O44.42',  # Low lying placenta NOS or without hemorrhage, second trimester
        'O44.43',  # Low lying placenta NOS or without hemorrhage, third trimester
        'O44.50',  # Low lying placenta with hemorrhage, unspecified trimester
        'O44.51',  # Low lying placenta with hemorrhage, first trimester
        'O44.52',  # Low lying placenta with hemorrhage, second trimester
        'O44.53',  # Low lying placenta with hemorrhage, third trimester
        'O45.001',  # Premature separation of placenta with coagulation defect, unspecified, first trimester
        'O45.002',  # Premature separation of placenta with coagulation defect, unspecified, second trimester
        'O45.003',  # Premature separation of placenta with coagulation defect, unspecified, third trimester
        'O45.009',  # Premature separation of placenta with coagulation defect, unspecified, unspecified trimester
        'O45.011',  # Premature separation of placenta with afibrinogenemia, first trimester
        'O45.012',  # Premature separation of placenta with afibrinogenemia, second trimester
        'O45.013',  # Premature separation of placenta with afibrinogenemia, third trimester
        'O45.019',  # Premature separation of placenta with afibrinogenemia, unspecified trimester
        'O45.021',  # Premature separation of placenta with disseminated intravascular coagulation, first trimester
        'O45.022',  # Premature separation of placenta with disseminated intravascular coagulation, second trimester
        'O45.023',  # Premature separation of placenta with disseminated intravascular coagulation, third trimester
        'O45.029',  # Premature separation of placenta with disseminated intravascular coagulation, unspecified trimester
        'O45.091',  # Premature separation of placenta with other coagulation defect, first trimester
        'O45.092',  # Premature separation of placenta with other coagulation defect, second trimester
        'O45.093',  # Premature separation of placenta with other coagulation defect, third trimester
        'O45.099',  # Premature separation of placenta with other coagulation defect, unspecified trimester
        'O45.8X1',  # Other premature separation of placenta, first trimester
        'O45.8X2',  # Other premature separation of placenta, second trimester
        'O45.8X3',  # Other premature separation of placenta, third trimester
        'O45.8X9',  # Other premature separation of placenta, unspecified trimester
        'O45.90',  # Premature separation of placenta, unspecified, unspecified trimester
        'O45.91',  # Premature separation of placenta, unspecified, first trimester
        'O45.92',  # Premature separation of placenta, unspecified, second trimester
        'O45.93',  # Premature separation of placenta, unspecified, third trimester
        'O46.001',  # Antepartum hemorrhage with coagulation defect, unspecified, first trimester
        'O46.002',  # Antepartum hemorrhage with coagulation defect, unspecified, second trimester
        'O46.003',  # Antepartum hemorrhage with coagulation defect, unspecified, third trimester
        'O46.009',  # Antepartum hemorrhage with coagulation defect, unspecified, unspecified trimester
        'O46.011',  # Antepartum hemorrhage with afibrinogenemia, first trimester
        'O46.012',  # Antepartum hemorrhage with afibrinogenemia, second trimester
        'O46.013',  # Antepartum hemorrhage with afibrinogenemia, third trimester
        'O46.019',  # Antepartum hemorrhage with afibrinogenemia, unspecified trimester
        'O46.021',  # Antepartum hemorrhage with disseminated intravascular coagulation, first trimester
        'O46.022',  # Antepartum hemorrhage with disseminated intravascular coagulation, second trimester
        'O46.023',  # Antepartum hemorrhage with disseminated intravascular coagulation, third trimester
        'O46.029',  # Antepartum hemorrhage with disseminated intravascular coagulation, unspecified trimester
        'O46.091',  # Antepartum hemorrhage with other coagulation defect, first trimester
        'O46.092',  # Antepartum hemorrhage with other coagulation defect, second trimester
        'O46.093',  # Antepartum hemorrhage with other coagulation defect, third trimester
        'O46.099',  # Antepartum hemorrhage with other coagulation defect, unspecified trimester
        'O46.8X1',  # Other antepartum hemorrhage, first trimester
        'O46.8X2',  # Other antepartum hemorrhage, second trimester
        'O46.8X3',  # Other antepartum hemorrhage, third trimester
        'O46.8X9',  # Other antepartum hemorrhage, unspecified trimester
        'O46.90',  # Antepartum hemorrhage, unspecified, unspecified trimester
        'O46.91',  # Antepartum hemorrhage, unspecified, first trimester
        'O46.92',  # Antepartum hemorrhage, unspecified, second trimester
        'O46.93',  # Antepartum hemorrhage, unspecified, third trimester
        'O47.00',  # False labor before 37 completed weeks of gestation, unspecified trimester
        'O47.02',  # False labor before 37 completed weeks of gestation, second trimester
        'O47.03',  # False labor before 37 completed weeks of gestation, third trimester
        'O47.1',  # False labor at or after 37 completed weeks of gestation
        'O47.9',  # False labor, unspecified
        'O48.0',  # Post-term pregnancy
        'O48.1',  # Prolonged pregnancy
        'O60.00',  # Preterm labor without delivery, unspecified trimester
        'O60.02',  # Preterm labor without delivery, second trimester
        'O60.03',  # Preterm labor without delivery, third trimester
        'O71.00',  # Rupture of uterus before onset of labor, unspecified trimester
        'O71.02',  # Rupture of uterus before onset of labor, second trimester
        'O71.03',  # Rupture of uterus before onset of labor, third trimester
        'O88.011',  # Air embolism in pregnancy, first trimester
        'O88.012',  # Air embolism in pregnancy, second trimester
        'O88.013',  # Air embolism in pregnancy, third trimester
        'O88.019',  # Air embolism in pregnancy, unspecified trimester
        'O88.111',  # Amniotic fluid embolism in pregnancy, first trimester
        'O88.112',  # Amniotic fluid embolism in pregnancy, second trimester
        'O88.113',  # Amniotic fluid embolism in pregnancy, third trimester
        'O88.119',  # Amniotic fluid embolism in pregnancy, unspecified trimester
        'O88.211',  # Thromboembolism in pregnancy, first trimester
        'O88.212',  # Thromboembolism in pregnancy, second trimester
        'O88.213',  # Thromboembolism in pregnancy, third trimester
        'O88.219',  # Thromboembolism in pregnancy, unspecified trimester
        'O88.311',  # Pyemic and septic embolism in pregnancy, first trimester
        'O88.312',  # Pyemic and septic embolism in pregnancy, second trimester
        'O88.313',  # Pyemic and septic embolism in pregnancy, third trimester
        'O88.319',  # Pyemic and septic embolism in pregnancy, unspecified trimester
        'O88.811',  # Other embolism in pregnancy, first trimester
        'O88.812',  # Other embolism in pregnancy, second trimester
        'O88.813',  # Other embolism in pregnancy, third trimester
        'O88.819',  # Other embolism in pregnancy, unspecified trimester
        'O90.3',  # Peripartum cardiomyopathy
        'O91.011',  # Infection of nipple associated with pregnancy, first trimester
        'O91.012',  # Infection of nipple associated with pregnancy, second trimester
        'O91.013',  # Infection of nipple associated with pregnancy, third trimester
        'O91.019',  # Infection of nipple associated with pregnancy, unspecified trimester
        'O91.111',  # Abscess of breast associated with pregnancy, first trimester
        'O91.112',  # Abscess of breast associated with pregnancy, second trimester
        'O91.113',  # Abscess of breast associated with pregnancy, third trimester
        'O91.119',  # Abscess of breast associated with pregnancy, unspecified trimester
        'O91.211',  # Nonpurulent mastitis associated with pregnancy, first trimester
        'O91.212',  # Nonpurulent mastitis associated with pregnancy, second trimester
        'O91.213',  # Nonpurulent mastitis associated with pregnancy, third trimester
        'O91.219',  # Nonpurulent mastitis associated with pregnancy, unspecified trimester
        'O92.011',  # Retracted nipple associated with pregnancy, first trimester
        'O92.012',  # Retracted nipple associated with pregnancy, second trimester
        'O92.013',  # Retracted nipple associated with pregnancy, third trimester
        'O92.019',  # Retracted nipple associated with pregnancy, unspecified trimester
        'O92.111',  # Cracked nipple associated with pregnancy, first trimester
        'O92.112',  # Cracked nipple associated with pregnancy, second trimester
        'O92.113',  # Cracked nipple associated with pregnancy, third trimester
        'O92.119',  # Cracked nipple associated with pregnancy, unspecified trimester
        'O92.20',  # Unspecified disorder of breast associated with pregnancy and the puerperium
        'O92.29',  # Other disorders of breast associated with pregnancy and the puerperium
        'O98.011',  # Tuberculosis complicating pregnancy, first trimester
        'O98.012',  # Tuberculosis complicating pregnancy, second trimester
        'O98.013',  # Tuberculosis complicating pregnancy, third trimester
        'O98.019',  # Tuberculosis complicating pregnancy, unspecified trimester
        'O98.111',  # Syphilis complicating pregnancy, first trimester
        'O98.112',  # Syphilis complicating pregnancy, second trimester
        'O98.113',  # Syphilis complicating pregnancy, third trimester
        'O98.119',  # Syphilis complicating pregnancy, unspecified trimester
        'O98.211',  # Gonorrhea complicating pregnancy, first trimester
        'O98.212',  # Gonorrhea complicating pregnancy, second trimester
        'O98.213',  # Gonorrhea complicating pregnancy, third trimester
        'O98.219',  # Gonorrhea complicating pregnancy, unspecified trimester
        'O98.311',  # Other infections with a predominantly sexual mode of transmission complicating pregnancy, first trimester
        'O98.312',  # Other infections with a predominantly sexual mode of transmission complicating pregnancy, second trimester
        'O98.313',  # Other infections with a predominantly sexual mode of transmission complicating pregnancy, third trimester
        'O98.319',  # Other infections with a predominantly sexual mode of transmission complicating pregnancy, unspecified trimester
        'O98.411',  # Viral hepatitis complicating pregnancy, first trimester
        'O98.412',  # Viral hepatitis complicating pregnancy, second trimester
        'O98.413',  # Viral hepatitis complicating pregnancy, third trimester
        'O98.419',  # Viral hepatitis complicating pregnancy, unspecified trimester
        'O98.511',  # Other viral diseases complicating pregnancy, first trimester
        'O98.512',  # Other viral diseases complicating pregnancy, second trimester
        'O98.513',  # Other viral diseases complicating pregnancy, third trimester
        'O98.519',  # Other viral diseases complicating pregnancy, unspecified trimester
        'O98.611',  # Protozoal diseases complicating pregnancy, first trimester
        'O98.612',  # Protozoal diseases complicating pregnancy, second trimester
        'O98.613',  # Protozoal diseases complicating pregnancy, third trimester
        'O98.619',  # Protozoal diseases complicating pregnancy, unspecified trimester
        'O98.711',  # Human immunodeficiency virus [HIV] disease complicating pregnancy, first trimester
        'O98.712',  # Human immunodeficiency virus [HIV] disease complicating pregnancy, second trimester
        'O98.713',  # Human immunodeficiency virus [HIV] disease complicating pregnancy, third trimester
        'O98.719',  # Human immunodeficiency virus [HIV] disease complicating pregnancy, unspecified trimester
        'O98.811',  # Other maternal infectious and parasitic diseases complicating pregnancy, first trimester
        'O98.812',  # Other maternal infectious and parasitic diseases complicating pregnancy, second trimester
        'O98.813',  # Other maternal infectious and parasitic diseases complicating pregnancy, third trimester
        'O98.819',  # Other maternal infectious and parasitic diseases complicating pregnancy, unspecified trimester
        'O98.911',  # Unspecified maternal infectious and parasitic disease complicating pregnancy, first trimester
        'O98.912',  # Unspecified maternal infectious and parasitic disease complicating pregnancy, second trimester
        'O98.913',  # Unspecified maternal infectious and parasitic disease complicating pregnancy, third trimester
        'O98.919',  # Unspecified maternal infectious and parasitic disease complicating pregnancy, unspecified trimester
        'O99.011',  # Anemia complicating pregnancy, first trimester
        'O99.012',  # Anemia complicating pregnancy, second trimester
        'O99.013',  # Anemia complicating pregnancy, third trimester
        'O99.019',  # Anemia complicating pregnancy, unspecified trimester
        'O99.111',  # Other diseases of the blood and blood-forming organs and certain disorders involving the immune mechanism complicating pregnancy, first trimester
        'O99.112',  # Other diseases of the blood and blood-forming organs and certain disorders involving the immune mechanism complicating pregnancy, second trimester
        'O99.113',  # Other diseases of the blood and blood-forming organs and certain disorders involving the immune mechanism complicating pregnancy, third trimester
        'O99.119',  # Other diseases of the blood and blood-forming organs and certain disorders involving the immune mechanism complicating pregnancy, unspecified trimester
        'O99.210',  # Obesity complicating pregnancy, unspecified trimester
        'O99.211',  # Obesity complicating pregnancy, first trimester
        'O99.212',  # Obesity complicating pregnancy, second trimester
        'O99.213',  # Obesity complicating pregnancy, third trimester
        'O99.280',  # Endocrine, nutritional and metabolic diseases complicating pregnancy, unspecified trimester
        'O99.281',  # Endocrine, nutritional and metabolic diseases complicating pregnancy, first trimester
        'O99.282',  # Endocrine, nutritional and metabolic diseases complicating pregnancy, second trimester
        'O99.283',  # Endocrine, nutritional and metabolic diseases complicating pregnancy, third trimester
        'O99.310',  # Alcohol use complicating pregnancy, unspecified trimester
        'O99.311',  # Alcohol use complicating pregnancy, first trimester
        'O99.312',  # Alcohol use complicating pregnancy, second trimester
        'O99.313',  # Alcohol use complicating pregnancy, third trimester
        'O99.320',  # Drug use complicating pregnancy, unspecified trimester
        'O99.321',  # Drug use complicating pregnancy, first trimester
        'O99.322',  # Drug use complicating pregnancy, second trimester
        'O99.323',  # Drug use complicating pregnancy, third trimester
        'O99.330',  # Smoking (tobacco) complicating pregnancy, unspecified trimester
        'O99.331',  # Smoking (tobacco) complicating pregnancy, first trimester
        'O99.332',  # Smoking (tobacco) complicating pregnancy, second trimester
        'O99.333',  # Smoking (tobacco) complicating pregnancy, third trimester
        'O99.340',  # Other mental disorders complicating pregnancy, unspecified trimester
        'O99.341',  # Other mental disorders complicating pregnancy, first trimester
        'O99.342',  # Other mental disorders complicating pregnancy, second trimester
        'O99.343',  # Other mental disorders complicating pregnancy, third trimester
        'O99.350',  # Diseases of the nervous system complicating pregnancy, unspecified trimester
        'O99.351',  # Diseases of the nervous system complicating pregnancy, first trimester
        'O99.352',  # Diseases of the nervous system complicating pregnancy, second trimester
        'O99.353',  # Diseases of the nervous system complicating pregnancy, third trimester
        'O99.411',  # Diseases of the circulatory system complicating pregnancy, first trimester
        'O99.412',  # Diseases of the circulatory system complicating pregnancy, second trimester
        'O99.413',  # Diseases of the circulatory system complicating pregnancy, third trimester
        'O99.419',  # Diseases of the circulatory system complicating pregnancy, unspecified trimester
        'O99.511',  # Diseases of the respiratory system complicating pregnancy, first trimester
        'O99.512',  # Diseases of the respiratory system complicating pregnancy, second trimester
        'O99.513',  # Diseases of the respiratory system complicating pregnancy, third trimester
        'O99.519',  # Diseases of the respiratory system complicating pregnancy, unspecified trimester
        'O99.611',  # Diseases of the digestive system complicating pregnancy, first trimester
        'O99.612',  # Diseases of the digestive system complicating pregnancy, second trimester
        'O99.613',  # Diseases of the digestive system complicating pregnancy, third trimester
        'O99.619',  # Diseases of the digestive system complicating pregnancy, unspecified trimester
        'O99.711',  # Diseases of the skin and subcutaneous tissue complicating pregnancy, first trimester
        'O99.712',  # Diseases of the skin and subcutaneous tissue complicating pregnancy, second trimester
        'O99.713',  # Diseases of the skin and subcutaneous tissue complicating pregnancy, third trimester
        'O99.719',  # Diseases of the skin and subcutaneous tissue complicating pregnancy, unspecified trimester
        'O99.810',  # Abnormal glucose complicating pregnancy
        'O99.820',  # Streptococcus B carrier state complicating pregnancy
        'O99.830',  # Other infection carrier state complicating pregnancy
        'O99.840',  # Bariatric surgery status complicating pregnancy, unspecified trimester
        'O99.841',  # Bariatric surgery status complicating pregnancy, first trimester
        'O99.842',  # Bariatric surgery status complicating pregnancy, second trimester
        'O99.843',  # Bariatric surgery status complicating pregnancy, third trimester
        'O9A.111',  # Malignant neoplasm complicating pregnancy, first trimester
        'O9A.112',  # Malignant neoplasm complicating pregnancy, second trimester
        'O9A.113',  # Malignant neoplasm complicating pregnancy, third trimester
        'O9A.119',  # Malignant neoplasm complicating pregnancy, unspecified trimester
        'O9A.211',  # Injury, poisoning and certain other consequences of external causes complicating pregnancy, first trimester
        'O9A.212',  # Injury, poisoning and certain other consequences of external causes complicating pregnancy, second trimester
        'O9A.213',  # Injury, poisoning and certain other consequences of external causes complicating pregnancy, third trimester
        'O9A.219',  # Injury, poisoning and certain other consequences of external causes complicating pregnancy, unspecified trimester
        'O9A.311',  # Physical abuse complicating pregnancy, first trimester
        'O9A.312',  # Physical abuse complicating pregnancy, second trimester
        'O9A.313',  # Physical abuse complicating pregnancy, third trimester
        'O9A.319',  # Physical abuse complicating pregnancy, unspecified trimester
        'O9A.411',  # Sexual abuse complicating pregnancy, first trimester
        'O9A.412',  # Sexual abuse complicating pregnancy, second trimester
        'O9A.413',  # Sexual abuse complicating pregnancy, third trimester
        'O9A.419',  # Sexual abuse complicating pregnancy, unspecified trimester
        'O9A.511',  # Psychological abuse complicating pregnancy, first trimester
        'O9A.512',  # Psychological abuse complicating pregnancy, second trimester
        'O9A.513',  # Psychological abuse complicating pregnancy, third trimester
        'O9A.519',  # Psychological abuse complicating pregnancy, unspecified trimester
        'Z33.1',  # Pregnant state, incidental
        'Z33.3',  # Pregnant state, gestational carrier
        'Z34.00',  # Encounter for supervision of normal first pregnancy, unspecified trimester
        'Z34.01',  # Encounter for supervision of normal first pregnancy, first trimester
        'Z34.02',  # Encounter for supervision of normal first pregnancy, second trimester
        'Z34.03',  # Encounter for supervision of normal first pregnancy, third trimester
        'Z34.80',  # Encounter for supervision of other normal pregnancy, unspecified trimester
        'Z34.81',  # Encounter for supervision of other normal pregnancy, first trimester
        'Z34.82',  # Encounter for supervision of other normal pregnancy, second trimester
        'Z34.83',  # Encounter for supervision of other normal pregnancy, third trimester
        'Z34.90',  # Encounter for supervision of normal pregnancy, unspecified, unspecified trimester
        'Z34.91',  # Encounter for supervision of normal pregnancy, unspecified, first trimester
        'Z34.92',  # Encounter for supervision of normal pregnancy, unspecified, second trimester
        'Z34.93',  # Encounter for supervision of normal pregnancy, unspecified, third trimester
        'Z3A.01',  # Less than 8 weeks gestation of pregnancy
        'Z3A.08',  # 8 weeks gestation of pregnancy
        'Z3A.09',  # 9 weeks gestation of pregnancy
        'Z3A.10',  # 10 weeks gestation of pregnancy
        'Z3A.11',  # 11 weeks gestation of pregnancy
        'Z3A.12',  # 12 weeks gestation of pregnancy
        'Z3A.13',  # 13 weeks gestation of pregnancy
        'Z3A.14',  # 14 weeks gestation of pregnancy
        'Z3A.15',  # 15 weeks gestation of pregnancy
        'Z3A.16',  # 16 weeks gestation of pregnancy
        'Z3A.17',  # 17 weeks gestation of pregnancy
        'Z3A.18',  # 18 weeks gestation of pregnancy
        'Z3A.19',  # 19 weeks gestation of pregnancy
        'Z3A.20',  # 20 weeks gestation of pregnancy
        'Z3A.21',  # 21 weeks gestation of pregnancy
        'Z3A.22',  # 22 weeks gestation of pregnancy
        'Z3A.23',  # 23 weeks gestation of pregnancy
        'Z3A.24',  # 24 weeks gestation of pregnancy
        'Z3A.25',  # 25 weeks gestation of pregnancy
        'Z3A.26',  # 26 weeks gestation of pregnancy
        'Z3A.27',  # 27 weeks gestation of pregnancy
        'Z3A.28',  # 28 weeks gestation of pregnancy
        'Z3A.29',  # 29 weeks gestation of pregnancy
        'Z3A.30',  # 30 weeks gestation of pregnancy
        'Z3A.31',  # 31 weeks gestation of pregnancy
        'Z3A.32',  # 32 weeks gestation of pregnancy
        'Z3A.33',  # 33 weeks gestation of pregnancy
        'Z3A.34',  # 34 weeks gestation of pregnancy
        'Z3A.35',  # 35 weeks gestation of pregnancy
        'Z3A.36',  # 36 weeks gestation of pregnancy
        'Z3A.37',  # 37 weeks gestation of pregnancy
        'Z3A.38',  # 38 weeks gestation of pregnancy
        'Z3A.39',  # 39 weeks gestation of pregnancy
        'Z3A.40',  # 40 weeks gestation of pregnancy
        'Z3A.41',  # 41 weeks gestation of pregnancy
        'Z3A.42',  # 42 weeks gestation of pregnancy
        'Z3A.49',  # Greater than 42 weeks gestation of pregnancy
    }
    SNOMEDCT = {
        '9279009',  # Extra-amniotic pregnancy (finding)
        '11082009',  # Abnormal pregnancy (finding)
        '14080002',  # Illegitimate pregnancy, life event (finding)
        '14418008',  # Precocious pregnancy (finding)
        '16356006',  # Multiple pregnancy (disorder)
        '38720006',  # Septuplet pregnancy (disorder)
        '41587001',  # Third trimester pregnancy (finding)
        '41991004',  # Angiectasis pregnancy (disorder)
        '43990006',  # Sextuplet pregnancy (disorder)
        '45307008',  # Extrachorial pregnancy (finding)
        '47200007',  # High risk pregnancy (finding)
        '57630001',  # First trimester pregnancy (finding)
        '58532003',  # Unwanted pregnancy (finding)
        '59466002',  # Second trimester pregnancy (finding)
        '60810003',  # Quadruplet pregnancy (disorder)
        '64254006',  # Triplet pregnancy (disorder)
        '65147003',  # Twin pregnancy (disorder)
        '65727000',  # Intrauterine pregnancy (finding)
        '72892002',  # Normal pregnancy (finding)
        '77386006',  # Pregnant (finding)
        '80997009',  # Quintuplet pregnancy (disorder)
        '83074005',  # Unplanned pregnancy (finding)
        '87527008',  # Term pregnancy (finding)
        '90968009',  # Prolonged pregnancy (disorder)
        '102872000',  # Pregnancy on oral contraceptive (finding)
        '102875003',  # Surrogate pregnancy (finding)
        '127364007',  # Primigravida (finding)
        '169561007',  # Pregnant - blood test confirms (finding)
        '169562000',  # Pregnant - vaginal examination confirms (finding)
        '169563005',  # Pregnant - on history (finding)
        '169564004',  # Pregnant - on abdominal palpation (finding)
        '169565003',  # Pregnant - planned (finding)
        '169566002',  # Pregnancy unplanned but wanted (finding)
        '169567006',  # Pregnancy unplanned and unwanted (finding)
        '169568001',  # Unplanned pregnancy unknown if child is wanted (finding)
        '199064003',  # Post-term pregnancy - not delivered (disorder)
        '199306007',  # Continuing pregnancy after abortion of one fetus or more (disorder)
        '237233002',  # Concealed pregnancy (finding)
        '237238006',  # Pregnancy with uncertain dates (finding)
        '237239003',  # Low risk pregnancy (finding)
        '237240001',  # Teenage pregnancy (finding)
        '237241002',  # Viable pregnancy (finding)
        '237244005',  # Single pregnancy (finding)
        '248985009',  # Presentation of pregnancy (finding)
        '276367008',  # Wanted pregnancy (finding)
        '281307002',  # Uncertain viability of pregnancy (finding)
        '314204000',  # Early stage of pregnancy (finding)
        '429187001',  # Continuing pregnancy after intrauterine death of twin fetus (disorder)
        '442478007',  # Multiple pregnancy involving intrauterine pregnancy and tubal pregnancy (disorder)
        '444661007',  # High risk pregnancy due to history of preterm labor (finding)
        '459167000',  # Monochorionic twin pregnancy (disorder)
        '459169002',  # Monochorionic diamniotic twin pregnancy with similar amniotic fluid volumes (disorder)
        '459170001',  # Monochorionic diamniotic twin pregnancy with dissimilar amniotic fluid volumes (disorder)
        '472321009',  # Continuing pregnancy after intrauterine death of one twin with intrauterine retention of dead twin (disorder)
        '713575004',  # Dizygotic twin pregnancy (disorder)
        '713576003',  # Monozygotic twin pregnancy (disorder)
        '10231000132102',  # In-vitro fertilization pregnancy (finding)
        '22281000119101',  # Post-term pregnancy of 40 to 42 weeks (disorder)
        '35381000119101',  # Quadruplet pregnancy with loss of one or more fetuses (disorder)
        '36801000119105',  # Continuing triplet pregnancy after spontaneous abortion of one or more fetuses (disorder)
        '134781000119106',  # High risk pregnancy due to recurrent miscarriage (finding)
        '457811000124103',  # Normal pregnancy in primigravida (finding)
        '457821000124106',  # Normal pregnancy in multigravida (finding)
    }


class RenalFailureDueToAceInhibitor(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for renal failure due to angiotensin converting enzyme (ACE) inhibitors.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of renal failure due to ACE inhibitor.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS135v10
    """

    VALUE_SET_NAME = 'Renal Failure Due to ACE Inhibitor'
    OID = '2.16.840.1.113883.3.526.3.1151'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    SNOMEDCT = {
        '236429004',  # Acute drug-induced renal failure (disorder)
        '422593004',  # Acute renal failure caused by angiotensin-converting-enzyme inhibitor (disorder)
    }


class Narcolepsy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  narcolepsy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of narcolepsy.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS136v11
    """

    VALUE_SET_NAME = 'Narcolepsy'
    OID = '2.16.840.1.113883.3.464.1003.114.12.1011'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'G47.411',  # Narcolepsy with cataplexy
        'G47.419',  # Narcolepsy without cataplexy
        'G47.421',  # Narcolepsy in conditions classified elsewhere with cataplexy
        'G47.429',  # Narcolepsy in conditions classified elsewhere without cataplexy
    }
    ICD9CM = {
        '347.00',  # Narcolepsy, without cataplexy
        '347.01',  # Narcolepsy, with cataplexy
        '347.10',  # Narcolepsy in conditions classified elsewhere, without cataplexy
        '347.11',  # Narcolepsy in conditions classified elsewhere, with cataplexy
    }
    SNOMEDCT = {
        '60380001',  # Narcolepsy (disorder)
        '193042000',  # Cataplexy and narcolepsy (disorder)
        '427426006',  # Sleep attack (disorder)
        '735676003',  # Narcolepsy type 1 (disorder)
        '91521000119104',  # Narcolepsy without cataplexy (disorder)
        '434241000124107',  # Secondary narcolepsy (disorder)
        '434251000124109',  # Secondary narcolepsy without cataplexy (disorder)
        '434261000124106',  # Secondary narcolepsy with cataplexy (disorder)
    }


class AlcoholAndDrugDependence(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  alcohol and drug dependence.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of alcohol and drug abuse and dependence.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis of alcohol and drug use.

    ** Used in:** CMS137v10
    """

    VALUE_SET_NAME = 'Alcohol and Drug Dependence'
    OID = '2.16.840.1.113883.3.464.1003.106.12.1001'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'F10.10',  # Alcohol abuse, uncomplicated
        'F10.120',  # Alcohol abuse with intoxication, uncomplicated
        'F10.121',  # Alcohol abuse with intoxication delirium
        'F10.129',  # Alcohol abuse with intoxication, unspecified
        'F10.130',  # Alcohol abuse with withdrawal, uncomplicated
        'F10.131',  # Alcohol abuse with withdrawal delirium
        'F10.132',  # Alcohol abuse with withdrawal with perceptual disturbance
        'F10.139',  # Alcohol abuse with withdrawal, unspecified
        'F10.14',  # Alcohol abuse with alcohol-induced mood disorder
        'F10.150',  # Alcohol abuse with alcohol-induced psychotic disorder with delusions
        'F10.151',  # Alcohol abuse with alcohol-induced psychotic disorder with hallucinations
        'F10.159',  # Alcohol abuse with alcohol-induced psychotic disorder, unspecified
        'F10.180',  # Alcohol abuse with alcohol-induced anxiety disorder
        'F10.181',  # Alcohol abuse with alcohol-induced sexual dysfunction
        'F10.182',  # Alcohol abuse with alcohol-induced sleep disorder
        'F10.188',  # Alcohol abuse with other alcohol-induced disorder
        'F10.19',  # Alcohol abuse with unspecified alcohol-induced disorder
        'F10.20',  # Alcohol dependence, uncomplicated
        'F10.220',  # Alcohol dependence with intoxication, uncomplicated
        'F10.221',  # Alcohol dependence with intoxication delirium
        'F10.229',  # Alcohol dependence with intoxication, unspecified
        'F10.230',  # Alcohol dependence with withdrawal, uncomplicated
        'F10.231',  # Alcohol dependence with withdrawal delirium
        'F10.232',  # Alcohol dependence with withdrawal with perceptual disturbance
        'F10.239',  # Alcohol dependence with withdrawal, unspecified
        'F10.24',  # Alcohol dependence with alcohol-induced mood disorder
        'F10.250',  # Alcohol dependence with alcohol-induced psychotic disorder with delusions
        'F10.251',  # Alcohol dependence with alcohol-induced psychotic disorder with hallucinations
        'F10.259',  # Alcohol dependence with alcohol-induced psychotic disorder, unspecified
        'F10.26',  # Alcohol dependence with alcohol-induced persisting amnestic disorder
        'F10.27',  # Alcohol dependence with alcohol-induced persisting dementia
        'F10.280',  # Alcohol dependence with alcohol-induced anxiety disorder
        'F10.281',  # Alcohol dependence with alcohol-induced sexual dysfunction
        'F10.282',  # Alcohol dependence with alcohol-induced sleep disorder
        'F10.288',  # Alcohol dependence with other alcohol-induced disorder
        'F10.29',  # Alcohol dependence with unspecified alcohol-induced disorder
        'F11.10',  # Opioid abuse, uncomplicated
        'F11.120',  # Opioid abuse with intoxication, uncomplicated
        'F11.121',  # Opioid abuse with intoxication delirium
        'F11.122',  # Opioid abuse with intoxication with perceptual disturbance
        'F11.129',  # Opioid abuse with intoxication, unspecified
        'F11.13',  # Opioid abuse with withdrawal
        'F11.14',  # Opioid abuse with opioid-induced mood disorder
        'F11.150',  # Opioid abuse with opioid-induced psychotic disorder with delusions
        'F11.151',  # Opioid abuse with opioid-induced psychotic disorder with hallucinations
        'F11.159',  # Opioid abuse with opioid-induced psychotic disorder, unspecified
        'F11.181',  # Opioid abuse with opioid-induced sexual dysfunction
        'F11.182',  # Opioid abuse with opioid-induced sleep disorder
        'F11.188',  # Opioid abuse with other opioid-induced disorder
        'F11.19',  # Opioid abuse with unspecified opioid-induced disorder
        'F11.20',  # Opioid dependence, uncomplicated
        'F11.220',  # Opioid dependence with intoxication, uncomplicated
        'F11.221',  # Opioid dependence with intoxication delirium
        'F11.222',  # Opioid dependence with intoxication with perceptual disturbance
        'F11.229',  # Opioid dependence with intoxication, unspecified
        'F11.23',  # Opioid dependence with withdrawal
        'F11.24',  # Opioid dependence with opioid-induced mood disorder
        'F11.250',  # Opioid dependence with opioid-induced psychotic disorder with delusions
        'F11.251',  # Opioid dependence with opioid-induced psychotic disorder with hallucinations
        'F11.259',  # Opioid dependence with opioid-induced psychotic disorder, unspecified
        'F11.281',  # Opioid dependence with opioid-induced sexual dysfunction
        'F11.282',  # Opioid dependence with opioid-induced sleep disorder
        'F11.288',  # Opioid dependence with other opioid-induced disorder
        'F11.29',  # Opioid dependence with unspecified opioid-induced disorder
        'F12.10',  # Cannabis abuse, uncomplicated
        'F12.120',  # Cannabis abuse with intoxication, uncomplicated
        'F12.121',  # Cannabis abuse with intoxication delirium
        'F12.122',  # Cannabis abuse with intoxication with perceptual disturbance
        'F12.129',  # Cannabis abuse with intoxication, unspecified
        'F12.13',  # Cannabis abuse with withdrawal
        'F12.150',  # Cannabis abuse with psychotic disorder with delusions
        'F12.151',  # Cannabis abuse with psychotic disorder with hallucinations
        'F12.159',  # Cannabis abuse with psychotic disorder, unspecified
        'F12.180',  # Cannabis abuse with cannabis-induced anxiety disorder
        'F12.188',  # Cannabis abuse with other cannabis-induced disorder
        'F12.19',  # Cannabis abuse with unspecified cannabis-induced disorder
        'F12.20',  # Cannabis dependence, uncomplicated
        'F12.220',  # Cannabis dependence with intoxication, uncomplicated
        'F12.221',  # Cannabis dependence with intoxication delirium
        'F12.222',  # Cannabis dependence with intoxication with perceptual disturbance
        'F12.229',  # Cannabis dependence with intoxication, unspecified
        'F12.23',  # Cannabis dependence with withdrawal
        'F12.250',  # Cannabis dependence with psychotic disorder with delusions
        'F12.251',  # Cannabis dependence with psychotic disorder with hallucinations
        'F12.259',  # Cannabis dependence with psychotic disorder, unspecified
        'F12.280',  # Cannabis dependence with cannabis-induced anxiety disorder
        'F12.288',  # Cannabis dependence with other cannabis-induced disorder
        'F12.29',  # Cannabis dependence with unspecified cannabis-induced disorder
        'F13.10',  # Sedative, hypnotic or anxiolytic abuse, uncomplicated
        'F13.120',  # Sedative, hypnotic or anxiolytic abuse with intoxication, uncomplicated
        'F13.121',  # Sedative, hypnotic or anxiolytic abuse with intoxication delirium
        'F13.129',  # Sedative, hypnotic or anxiolytic abuse with intoxication, unspecified
        'F13.130',  # Sedative, hypnotic or anxiolytic abuse with withdrawal, uncomplicated
        'F13.131',  # Sedative, hypnotic or anxiolytic abuse with withdrawal delirium
        'F13.132',  # Sedative, hypnotic or anxiolytic abuse with withdrawal with perceptual disturbance
        'F13.139',  # Sedative, hypnotic or anxiolytic abuse with withdrawal, unspecified
        'F13.14',  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced mood disorder
        'F13.150',  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced psychotic disorder with delusions
        'F13.151',  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced psychotic disorder with hallucinations
        'F13.159',  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced psychotic disorder, unspecified
        'F13.180',  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced anxiety disorder
        'F13.181',  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced sexual dysfunction
        'F13.182',  # Sedative, hypnotic or anxiolytic abuse with sedative, hypnotic or anxiolytic-induced sleep disorder
        'F13.188',  # Sedative, hypnotic or anxiolytic abuse with other sedative, hypnotic or anxiolytic-induced disorder
        'F13.19',  # Sedative, hypnotic or anxiolytic abuse with unspecified sedative, hypnotic or anxiolytic-induced disorder
        'F13.20',  # Sedative, hypnotic or anxiolytic dependence, uncomplicated
        'F13.220',  # Sedative, hypnotic or anxiolytic dependence with intoxication, uncomplicated
        'F13.221',  # Sedative, hypnotic or anxiolytic dependence with intoxication delirium
        'F13.229',  # Sedative, hypnotic or anxiolytic dependence with intoxication, unspecified
        'F13.230',  # Sedative, hypnotic or anxiolytic dependence with withdrawal, uncomplicated
        'F13.231',  # Sedative, hypnotic or anxiolytic dependence with withdrawal delirium
        'F13.232',  # Sedative, hypnotic or anxiolytic dependence with withdrawal with perceptual disturbance
        'F13.239',  # Sedative, hypnotic or anxiolytic dependence with withdrawal, unspecified
        'F13.24',  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced mood disorder
        'F13.250',  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced psychotic disorder with delusions
        'F13.251',  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced psychotic disorder with hallucinations
        'F13.259',  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced psychotic disorder, unspecified
        'F13.26',  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced persisting amnestic disorder
        'F13.27',  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced persisting dementia
        'F13.280',  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced anxiety disorder
        'F13.281',  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced sexual dysfunction
        'F13.282',  # Sedative, hypnotic or anxiolytic dependence with sedative, hypnotic or anxiolytic-induced sleep disorder
        'F13.288',  # Sedative, hypnotic or anxiolytic dependence with other sedative, hypnotic or anxiolytic-induced disorder
        'F13.29',  # Sedative, hypnotic or anxiolytic dependence with unspecified sedative, hypnotic or anxiolytic-induced disorder
        'F14.10',  # Cocaine abuse, uncomplicated
        'F14.120',  # Cocaine abuse with intoxication, uncomplicated
        'F14.121',  # Cocaine abuse with intoxication with delirium
        'F14.122',  # Cocaine abuse with intoxication with perceptual disturbance
        'F14.129',  # Cocaine abuse with intoxication, unspecified
        'F14.13',  # Cocaine abuse, unspecified with withdrawal
        'F14.14',  # Cocaine abuse with cocaine-induced mood disorder
        'F14.150',  # Cocaine abuse with cocaine-induced psychotic disorder with delusions
        'F14.151',  # Cocaine abuse with cocaine-induced psychotic disorder with hallucinations
        'F14.159',  # Cocaine abuse with cocaine-induced psychotic disorder, unspecified
        'F14.180',  # Cocaine abuse with cocaine-induced anxiety disorder
        'F14.181',  # Cocaine abuse with cocaine-induced sexual dysfunction
        'F14.182',  # Cocaine abuse with cocaine-induced sleep disorder
        'F14.188',  # Cocaine abuse with other cocaine-induced disorder
        'F14.19',  # Cocaine abuse with unspecified cocaine-induced disorder
        'F14.20',  # Cocaine dependence, uncomplicated
        'F14.220',  # Cocaine dependence with intoxication, uncomplicated
        'F14.221',  # Cocaine dependence with intoxication delirium
        'F14.222',  # Cocaine dependence with intoxication with perceptual disturbance
        'F14.229',  # Cocaine dependence with intoxication, unspecified
        'F14.23',  # Cocaine dependence with withdrawal
        'F14.24',  # Cocaine dependence with cocaine-induced mood disorder
        'F14.250',  # Cocaine dependence with cocaine-induced psychotic disorder with delusions
        'F14.251',  # Cocaine dependence with cocaine-induced psychotic disorder with hallucinations
        'F14.259',  # Cocaine dependence with cocaine-induced psychotic disorder, unspecified
        'F14.280',  # Cocaine dependence with cocaine-induced anxiety disorder
        'F14.281',  # Cocaine dependence with cocaine-induced sexual dysfunction
        'F14.282',  # Cocaine dependence with cocaine-induced sleep disorder
        'F14.288',  # Cocaine dependence with other cocaine-induced disorder
        'F14.29',  # Cocaine dependence with unspecified cocaine-induced disorder
        'F15.10',  # Other stimulant abuse, uncomplicated
        'F15.120',  # Other stimulant abuse with intoxication, uncomplicated
        'F15.121',  # Other stimulant abuse with intoxication delirium
        'F15.122',  # Other stimulant abuse with intoxication with perceptual disturbance
        'F15.129',  # Other stimulant abuse with intoxication, unspecified
        'F15.13',  # Other stimulant abuse with withdrawal
        'F15.14',  # Other stimulant abuse with stimulant-induced mood disorder
        'F15.150',  # Other stimulant abuse with stimulant-induced psychotic disorder with delusions
        'F15.151',  # Other stimulant abuse with stimulant-induced psychotic disorder with hallucinations
        'F15.159',  # Other stimulant abuse with stimulant-induced psychotic disorder, unspecified
        'F15.180',  # Other stimulant abuse with stimulant-induced anxiety disorder
        'F15.181',  # Other stimulant abuse with stimulant-induced sexual dysfunction
        'F15.182',  # Other stimulant abuse with stimulant-induced sleep disorder
        'F15.188',  # Other stimulant abuse with other stimulant-induced disorder
        'F15.19',  # Other stimulant abuse with unspecified stimulant-induced disorder
        'F15.20',  # Other stimulant dependence, uncomplicated
        'F15.220',  # Other stimulant dependence with intoxication, uncomplicated
        'F15.221',  # Other stimulant dependence with intoxication delirium
        'F15.222',  # Other stimulant dependence with intoxication with perceptual disturbance
        'F15.229',  # Other stimulant dependence with intoxication, unspecified
        'F15.23',  # Other stimulant dependence with withdrawal
        'F15.24',  # Other stimulant dependence with stimulant-induced mood disorder
        'F15.250',  # Other stimulant dependence with stimulant-induced psychotic disorder with delusions
        'F15.251',  # Other stimulant dependence with stimulant-induced psychotic disorder with hallucinations
        'F15.259',  # Other stimulant dependence with stimulant-induced psychotic disorder, unspecified
        'F15.280',  # Other stimulant dependence with stimulant-induced anxiety disorder
        'F15.281',  # Other stimulant dependence with stimulant-induced sexual dysfunction
        'F15.282',  # Other stimulant dependence with stimulant-induced sleep disorder
        'F15.288',  # Other stimulant dependence with other stimulant-induced disorder
        'F15.29',  # Other stimulant dependence with unspecified stimulant-induced disorder
        'F16.10',  # Hallucinogen abuse, uncomplicated
        'F16.120',  # Hallucinogen abuse with intoxication, uncomplicated
        'F16.121',  # Hallucinogen abuse with intoxication with delirium
        'F16.122',  # Hallucinogen abuse with intoxication with perceptual disturbance
        'F16.129',  # Hallucinogen abuse with intoxication, unspecified
        'F16.14',  # Hallucinogen abuse with hallucinogen-induced mood disorder
        'F16.150',  # Hallucinogen abuse with hallucinogen-induced psychotic disorder with delusions
        'F16.151',  # Hallucinogen abuse with hallucinogen-induced psychotic disorder with hallucinations
        'F16.159',  # Hallucinogen abuse with hallucinogen-induced psychotic disorder, unspecified
        'F16.180',  # Hallucinogen abuse with hallucinogen-induced anxiety disorder
        'F16.183',  # Hallucinogen abuse with hallucinogen persisting perception disorder (flashbacks)
        'F16.188',  # Hallucinogen abuse with other hallucinogen-induced disorder
        'F16.19',  # Hallucinogen abuse with unspecified hallucinogen-induced disorder
        'F16.20',  # Hallucinogen dependence, uncomplicated
        'F16.220',  # Hallucinogen dependence with intoxication, uncomplicated
        'F16.221',  # Hallucinogen dependence with intoxication with delirium
        'F16.229',  # Hallucinogen dependence with intoxication, unspecified
        'F16.24',  # Hallucinogen dependence with hallucinogen-induced mood disorder
        'F16.250',  # Hallucinogen dependence with hallucinogen-induced psychotic disorder with delusions
        'F16.251',  # Hallucinogen dependence with hallucinogen-induced psychotic disorder with hallucinations
        'F16.259',  # Hallucinogen dependence with hallucinogen-induced psychotic disorder, unspecified
        'F16.280',  # Hallucinogen dependence with hallucinogen-induced anxiety disorder
        'F16.283',  # Hallucinogen dependence with hallucinogen persisting perception disorder (flashbacks)
        'F16.288',  # Hallucinogen dependence with other hallucinogen-induced disorder
        'F16.29',  # Hallucinogen dependence with unspecified hallucinogen-induced disorder
        'F18.10',  # Inhalant abuse, uncomplicated
        'F18.120',  # Inhalant abuse with intoxication, uncomplicated
        'F18.121',  # Inhalant abuse with intoxication delirium
        'F18.129',  # Inhalant abuse with intoxication, unspecified
        'F18.14',  # Inhalant abuse with inhalant-induced mood disorder
        'F18.150',  # Inhalant abuse with inhalant-induced psychotic disorder with delusions
        'F18.151',  # Inhalant abuse with inhalant-induced psychotic disorder with hallucinations
        'F18.159',  # Inhalant abuse with inhalant-induced psychotic disorder, unspecified
        'F18.17',  # Inhalant abuse with inhalant-induced dementia
        'F18.180',  # Inhalant abuse with inhalant-induced anxiety disorder
        'F18.188',  # Inhalant abuse with other inhalant-induced disorder
        'F18.19',  # Inhalant abuse with unspecified inhalant-induced disorder
        'F18.20',  # Inhalant dependence, uncomplicated
        'F18.220',  # Inhalant dependence with intoxication, uncomplicated
        'F18.221',  # Inhalant dependence with intoxication delirium
        'F18.229',  # Inhalant dependence with intoxication, unspecified
        'F18.24',  # Inhalant dependence with inhalant-induced mood disorder
        'F18.250',  # Inhalant dependence with inhalant-induced psychotic disorder with delusions
        'F18.251',  # Inhalant dependence with inhalant-induced psychotic disorder with hallucinations
        'F18.259',  # Inhalant dependence with inhalant-induced psychotic disorder, unspecified
        'F18.27',  # Inhalant dependence with inhalant-induced dementia
        'F18.280',  # Inhalant dependence with inhalant-induced anxiety disorder
        'F18.288',  # Inhalant dependence with other inhalant-induced disorder
        'F18.29',  # Inhalant dependence with unspecified inhalant-induced disorder
        'F19.10',  # Other psychoactive substance abuse, uncomplicated
        'F19.120',  # Other psychoactive substance abuse with intoxication, uncomplicated
        'F19.121',  # Other psychoactive substance abuse with intoxication delirium
        'F19.122',  # Other psychoactive substance abuse with intoxication with perceptual disturbances
        'F19.129',  # Other psychoactive substance abuse with intoxication, unspecified
        'F19.130',  # Other psychoactive substance abuse with withdrawal, uncomplicated
        'F19.131',  # Other psychoactive substance abuse with withdrawal delirium
        'F19.132',  # Other psychoactive substance abuse with withdrawal with perceptual disturbance
        'F19.139',  # Other psychoactive substance abuse with withdrawal, unspecified
        'F19.14',  # Other psychoactive substance abuse with psychoactive substance-induced mood disorder
        'F19.150',  # Other psychoactive substance abuse with psychoactive substance-induced psychotic disorder with delusions
        'F19.151',  # Other psychoactive substance abuse with psychoactive substance-induced psychotic disorder with hallucinations
        'F19.159',  # Other psychoactive substance abuse with psychoactive substance-induced psychotic disorder, unspecified
        'F19.16',  # Other psychoactive substance abuse with psychoactive substance-induced persisting amnestic disorder
        'F19.17',  # Other psychoactive substance abuse with psychoactive substance-induced persisting dementia
        'F19.180',  # Other psychoactive substance abuse with psychoactive substance-induced anxiety disorder
        'F19.181',  # Other psychoactive substance abuse with psychoactive substance-induced sexual dysfunction
        'F19.182',  # Other psychoactive substance abuse with psychoactive substance-induced sleep disorder
        'F19.188',  # Other psychoactive substance abuse with other psychoactive substance-induced disorder
        'F19.19',  # Other psychoactive substance abuse with unspecified psychoactive substance-induced disorder
        'F19.20',  # Other psychoactive substance dependence, uncomplicated
        'F19.220',  # Other psychoactive substance dependence with intoxication, uncomplicated
        'F19.221',  # Other psychoactive substance dependence with intoxication delirium
        'F19.222',  # Other psychoactive substance dependence with intoxication with perceptual disturbance
        'F19.229',  # Other psychoactive substance dependence with intoxication, unspecified
        'F19.230',  # Other psychoactive substance dependence with withdrawal, uncomplicated
        'F19.231',  # Other psychoactive substance dependence with withdrawal delirium
        'F19.232',  # Other psychoactive substance dependence with withdrawal with perceptual disturbance
        'F19.239',  # Other psychoactive substance dependence with withdrawal, unspecified
        'F19.24',  # Other psychoactive substance dependence with psychoactive substance-induced mood disorder
        'F19.250',  # Other psychoactive substance dependence with psychoactive substance-induced psychotic disorder with delusions
        'F19.251',  # Other psychoactive substance dependence with psychoactive substance-induced psychotic disorder with hallucinations
        'F19.259',  # Other psychoactive substance dependence with psychoactive substance-induced psychotic disorder, unspecified
        'F19.26',  # Other psychoactive substance dependence with psychoactive substance-induced persisting amnestic disorder
        'F19.27',  # Other psychoactive substance dependence with psychoactive substance-induced persisting dementia
        'F19.280',  # Other psychoactive substance dependence with psychoactive substance-induced anxiety disorder
        'F19.281',  # Other psychoactive substance dependence with psychoactive substance-induced sexual dysfunction
        'F19.282',  # Other psychoactive substance dependence with psychoactive substance-induced sleep disorder
        'F19.288',  # Other psychoactive substance dependence with other psychoactive substance-induced disorder
        'F19.29',  # Other psychoactive substance dependence with unspecified psychoactive substance-induced disorder
    }
    SNOMEDCT = {
        '281004',  # Dementia associated with alcoholism (disorder)
        '1383008',  # Hallucinogen mood disorder (disorder)
        '1686006',  # Sedative, hypnotic AND/OR anxiolytic-induced anxiety disorder (disorder)
        '1973000',  # Sedative, hypnotic AND/OR anxiolytic-induced psychotic disorder with delusions (disorder)
        '2403008',  # Psychoactive substance dependence (disorder)
        '5002000',  # Inhalant dependence (disorder)
        '5444000',  # Sedative, hypnotic AND/OR anxiolytic intoxication delirium (disorder)
        '5602001',  # Opioid abuse (disorder)
        '6348008',  # Cocaine-induced psychotic disorder with hallucinations (disorder)
        '7052005',  # Alcohol hallucinosis (disorder)
        '7071007',  # Phencyclidine abuse (disorder)
        '7200002',  # Alcoholism (disorder)
        '8635005',  # Alcohol withdrawal delirium (disorder)
        '8837000',  # Amphetamine delirium (disorder)
        '10028000',  # Uncomplicated sedative, hypnotic AND/OR anxiolytic withdrawal (disorder)
        '10327003',  # Cocaine-induced mood disorder (disorder)
        '11387009',  # Psychoactive substance-induced organic mental disorder (disorder)
        '14784000',  # Opioid-induced organic mental disorder (disorder)
        '15167005',  # Alcohol abuse (disorder)
        '15277004',  # Hallucinogen-induced anxiety disorder (disorder)
        '18653004',  # Alcohol intoxication delirium (disorder)
        '18689007',  # Inhalant intoxication delirium (disorder)
        '19445006',  # Opioid-induced psychotic disorder with hallucinations (disorder)
        '20385005',  # Opioid-induced psychotic disorder with delusions (disorder)
        '20876004',  # Inhalant-induced anxiety disorder (disorder)
        '21647008',  # Amphetamine dependence (disorder)
        '22574000',  # Cocaine-induced sleep disorder (disorder)
        '25753007',  # Amphetamine-induced sleep disorder (disorder)
        '26714005',  # Cannabis-induced psychotic disorder with hallucinations (disorder)
        '27956007',  # Cocaine intoxication (disorder)
        '28864000',  # Sedative, hypnotic AND/OR anxiolytic-induced mood disorder (disorder)
        '29212009',  # Alcohol-induced organic mental disorder (disorder)
        '29733004',  # Opioid-induced mood disorder (disorder)
        '30491001',  # Cocaine delusional disorder (disorder)
        '31956009',  # Cocaine dependence (disorder)
        '32009006',  # Hallucinogen hallucinosis (disorder)
        '32358001',  # Amphetamine delusional disorder (disorder)
        '32875003',  # Inhalant-induced persisting dementia (disorder)
        '34938008',  # Alcohol-induced anxiety disorder (disorder)
        '37344009',  # Cannabis abuse (disorder)
        '38247002',  # Hallucinogen dependence (disorder)
        '39807006',  # Cannabis intoxication delirium (disorder)
        '39951001',  # Cannabis-induced anxiety disorder (disorder)
        '40571009',  # Hallucinogen intoxication delirium (disorder)
        '41083005',  # Alcohol-induced sleep disorder (disorder)
        '42344001',  # Alcohol-induced psychosis (disorder)
        '43497001',  # Amphetamine-induced mood disorder (disorder)
        '46975003',  # Cocaine-induced organic mental disorder (disorder)
        '47664006',  # Sedative, hypnotic AND/OR anxiolytic-induced psychotic disorder with hallucinations (disorder)
        '50320000',  # Hallucinogen intoxication (disorder)
        '50933003',  # Hallucinogen delusional disorder (disorder)
        '51443000',  # Amphetamine-induced psychotic disorder with hallucinations (disorder)
        '51493001',  # Cocaine-induced anxiety disorder (disorder)
        '52866005',  # Opioid intoxication delirium (disorder)
        '53050002',  # Hallucinogen-induced organic mental disorder (disorder)
        '53936005',  # Alcohol-induced mood disorder (disorder)
        '56194001',  # Caffeine-induced sleep disorder (disorder)
        '57588009',  # Sedative, hypnotic AND/OR anxiolytic-induced sleep disorder (disorder)
        '59651006',  # Sedative, hypnotic AND/OR anxiolytic-induced persisting dementia (disorder)
        '61104008',  # Inhalant-induced organic mental disorder (disorder)
        '61144001',  # Alcohol-induced psychotic disorder with delusions (disorder)
        '63649001',  # Cannabis delusional disorder (disorder)
        '63983005',  # Inhalant-induced psychotic disorder with hallucinations (disorder)
        '64386003',  # Sedative abuse (disorder)
        '66590003',  # Alcohol dependence (disorder)
        '70328006',  # Cocaine delirium (disorder)
        '70340006',  # Inhalant abuse (disorder)
        '70701004',  # Alcoholic amblyopia (disorder)
        '70932007',  # Amphetamine-induced sexual dysfunction (disorder)
        '71328000',  # Opioid-induced sexual dysfunction (disorder)
        '73097000',  # Alcohol amnestic disorder (disorder)
        '74851005',  # Hallucinogen abuse (disorder)
        '74934004',  # Psychoactive substance-induced withdrawal syndrome (disorder)
        '75122001',  # Inhalant-induced psychotic disorder with delusions (disorder)
        '75544000',  # Opioid dependence (disorder)
        '77355000',  # Cannabis-induced organic mental disorder (disorder)
        '78267003',  # Cocaine abuse (disorder)
        '78358001',  # Amphetamine withdrawal (disorder)
        '80868005',  # Cocaine withdrawal (disorder)
        '82339009',  # Amphetamine-induced anxiety disorder (disorder)
        '83367009',  # Amphetamine-induced organic mental disorder (disorder)
        '84758004',  # Amphetamine abuse (disorder)
        '85005007',  # Cannabis dependence (disorder)
        '85561006',  # Uncomplicated alcohol withdrawal (disorder)
        '86325007',  # Non megaloblastic anemia due to alcoholism (disorder)
        '87132004',  # Opioid withdrawal (disorder)
        '87810006',  # Megaloblastic anemia caused by alcoholism (disorder)
        '88926005',  # Opioid-induced sleep disorder (disorder)
        '89451009',  # Inhalant-induced mood disorder (disorder)
        '91388009',  # Psychoactive substance abuse (disorder)
        '95635002',  # Caffeine withdrawal (disorder)
        '95661003',  # Ergotamine withdrawal (disorder)
        '191475009',  # Chronic alcoholic brain syndrome (disorder)
        '191476005',  # Alcohol withdrawal hallucinosis (disorder)
        '191478006',  # Alcoholic paranoia (disorder)
        '191480000',  # Alcohol withdrawal syndrome (disorder)
        '191802004',  # Acute alcoholic intoxication in alcoholism (disorder)
        '191804003',  # Continuous acute alcoholic intoxication in alcoholism (disorder)
        '191805002',  # Episodic acute alcoholic intoxication in alcoholism (disorder)
        '191811004',  # Continuous chronic alcoholism (disorder)
        '191812006',  # Episodic chronic alcoholism (disorder)
        '191819002',  # Continuous opioid dependence (disorder)
        '191820008',  # Episodic opioid dependence (disorder)
        '191825003',  # Hypnotic or anxiolytic dependence, continuous (disorder)
        '191826002',  # Hypnotic or anxiolytic dependence, episodic (disorder)
        '191831000',  # Cocaine dependence, continuous (disorder)
        '191832007',  # Cocaine dependence, episodic (disorder)
        '191837001',  # Cannabis dependence, continuous (disorder)
        '191838006',  # Cannabis dependence, episodic (disorder)
        '191843004',  # Amphetamine or psychostimulant dependence, continuous (disorder)
        '191844005',  # Amphetamine or psychostimulant dependence, episodic (disorder)
        '191849000',  # Hallucinogen dependence, continuous (disorder)
        '191850000',  # Hallucinogen dependence, episodic (disorder)
        '191853003',  # Glue sniffing dependence (disorder)
        '191855005',  # Glue sniffing dependence, continuous (disorder)
        '191856006',  # Glue sniffing dependence, episodic (disorder)
        '191865004',  # Combined opioid with non-opioid drug dependence (disorder)
        '191867007',  # Combined opioid with non-opioid drug dependence, continuous (disorder)
        '191868002',  # Combined opioid with non-opioid drug dependence, episodic (disorder)
        '191882002',  # Nondependent alcohol abuse, continuous (disorder)
        '191883007',  # Nondependent alcohol abuse, episodic (disorder)
        '191891003',  # Nondependent cannabis abuse (disorder)
        '191893000',  # Nondependent cannabis abuse, continuous (disorder)
        '191894006',  # Nondependent cannabis abuse, episodic (disorder)
        '191899001',  # Nondependent hallucinogen abuse, continuous (disorder)
        '191900006',  # Nondependent hallucinogen abuse, episodic (disorder)
        '191905001',  # Nondependent hypnotic or anxiolytic abuse, continuous (disorder)
        '191906000',  # Nondependent hypnotic or anxiolytic abuse, episodic (disorder)
        '191909007',  # Nondependent opioid abuse (disorder)
        '191912005',  # Nondependent opioid abuse, continuous (disorder)
        '191913000',  # Nondependent opioid abuse, episodic (disorder)
        '191916008',  # Nondependent cocaine abuse (disorder)
        '191918009',  # Nondependent cocaine abuse, continuous (disorder)
        '191919001',  # Nondependent cocaine abuse, episodic (disorder)
        '191924003',  # Nondependent amphetamine or psychostimulant abuse, continuous (disorder)
        '191925002',  # Nondependent amphetamine or psychostimulant abuse, episodic (disorder)
        '230443000',  # Narcotic withdrawal epilepsy (disorder)
        '231461004',  # Hypnotic or anxiolytic abuse (disorder)
        '231462006',  # Barbiturate abuse (disorder)
        '231468005',  # Lysergic acid diethylamide dependence (disorder)
        '231469002',  # Mescaline dependence (disorder)
        '231470001',  # Anxiolytic dependence (disorder)
        '231472009',  # Barbiturate dependence (disorder)
        '231473004',  # Benzodiazepine dependence (disorder)
        '231474005',  # Diazepam dependence (disorder)
        '231477003',  # Heroin dependence (disorder)
        '231478008',  # Methadone dependence (disorder)
        '231479000',  # Morphine dependence (disorder)
        '231480002',  # Opium dependence (disorder)
        '268640002',  # Hypnotic or anxiolytic dependence (disorder)
        '268645007',  # Nondependent alcohol abuse (disorder)
        '268646008',  # Nondependent hallucinogen abuse (disorder)
        '268647004',  # Nondependent hypnotic or anxiolytic abuse (disorder)
        '268648009',  # Nondependent amphetamine or other psychostimulant abuse (disorder)
        '275471001',  # Psychostimulant dependence (disorder)
        '284591009',  # Persistent alcohol abuse (disorder)
        '301643003',  # Sedative, hypnotic AND/OR anxiolytic-induced persisting amnestic disorder (disorder)
        '308742005',  # Alcohol withdrawal-induced convulsion (disorder)
        '312098001',  # Sedative, hypnotic AND/OR anxiolytic-related disorder (disorder)
        '312936002',  # Anxiolytic-induced organic mental disorder (disorder)
        '361151007',  # Sedative withdrawal delirium (disorder)
        '416714005',  # Episodic flashbacks (disorder)
        '417143004',  # Short duration flashbacks (disorder)
        '425339005',  # Barbiturate withdrawal (disorder)
        '425885002',  # Continuous phencyclidine abuse (disorder)
        '426001001',  # Fentanyl dependence (disorder)
        '426095000',  # Continuous inhalant abuse (disorder)
        '426873000',  # Methamphetamine dependence (disorder)
        '427205009',  # Amphetamine abuse, continuous (disorder)
        '427229002',  # Episodic inhalant abuse (disorder)
        '427327003',  # Sedative dependence (disorder)
        '428219007',  # Caffeine dependence (disorder)
        '428370001',  # Psychostimulant withdrawal (disorder)
        '429692000',  # Amphetamine abuse, episodic (disorder)
        '441527004',  # Stimulant abuse (disorder)
        '442406005',  # Stimulant dependence (disorder)
        '699449003',  # Methamphetamine abuse (disorder)
        '703845008',  # Buprenorphine dependence (disorder)
        '703848005',  # Cannabis withdrawal (disorder)
        '703849002',  # Benzodiazepine withdrawal (disorder)
        '703850002',  # Benzodiazepine withdrawal delirium (disorder)
        '713583005',  # Mild alcohol dependence (disorder)
        '713862009',  # Severe alcohol dependence (disorder)
        '714829008',  # Moderate alcohol dependence (disorder)
        '723926008',  # Perceptual disturbances and seizures co-occurrent and due to alcohol withdrawal (disorder)
        '723927004',  # Psychotic disorder caused by alcohol with schizophreniform symptoms (disorder)
        '723928009',  # Mood disorder with depressive symptoms caused by alcohol (disorder)
        '723929001',  # Mood disorder with manic symptoms caused by alcohol (disorder)
        '723930006',  # Mood disorder with mixed manic and depressive symptoms caused by alcohol (disorder)
        '723933008',  # Cannabis dependence with current use (disorder)
        '723936000',  # Psychotic disorder caused by cannabis (disorder)
        '723937009',  # Sleep disorder caused by cannabis (disorder)
        '724653003',  # Opioid dependence with current use (disorder)
        '724655005',  # Psychotic disorder caused by opioid (disorder)
        '724656006',  # Hypnotic dependence (disorder)
        '724657002',  # Sedative dependence with current use (disorder)
        '724658007',  # Hypnotic dependence with current use (disorder)
        '724659004',  # Anxiolytic dependence with current use (disorder)
        '724660009',  # Hypnotic withdrawal (disorder)
        '724661008',  # Anxiolytic withdrawal (disorder)
        '724662001',  # Sedative withdrawal without complication (disorder)
        '724663006',  # Hypnotic withdrawal without complication (disorder)
        '724664000',  # Anxiolytic withdrawal without complication (disorder)
        '724665004',  # Perceptual disturbances co-occurrent and due to sedative withdrawal (disorder)
        '724666003',  # Perceptual disturbances co-occurrent and due to hypnotic withdrawal (disorder)
        '724667007',  # Perceptual disturbances co-occurrent and due to anxiolytic withdrawal (disorder)
        '724668002',  # Seizure co-occurrent and due to sedative withdrawal (disorder)
        '724669005',  # Seizure co-occurrent and due to hypnotic withdrawal (disorder)
        '724670006',  # Seizure co-occurrent and due to anxiolytic withdrawal (disorder)
        '724671005',  # Perceptual disturbances and seizures co-occurrent and due to sedative withdrawal (disorder)
        '724672003',  # Perceptual disturbances and seizures co-occurrent and due to hypnotic withdrawal (disorder)
        '724675001',  # Psychotic disorder caused by anxiolytic (disorder)
        '724688003',  # Cocaine dependence with current use (disorder)
        '724689006',  # Psychotic disorder caused by cocaine (disorder)
        '724690002',  # Mood disorder with depressive symptoms caused by cocaine (disorder)
        '724691003',  # Mood disorder with manic symptoms caused by cocaine (disorder)
        '724692005',  # Mood disorder with mixed depressive and manic symptoms caused by cocaine (disorder)
        '724693000',  # Obsessive compulsive disorder caused by cocaine (disorder)
        '724695007',  # Hallucinogen dependence with current use (disorder)
        '724696008',  # Psychotic disorder caused by hallucinogen (disorder)
        '724700000',  # Volatile inhalant dependence with current use (disorder)
        '724701001',  # Volatile inhalant withdrawal (disorder)
        '724702008',  # Psychotic disorder caused by volatile inhalant (disorder)
        '724704009',  # Methylenedioxymethamphetamine dependence with current use (disorder)
        '724706006',  # Psychotic disorder caused by methylenedioxymethamphetamine (disorder)
        '724712001',  # Harmful use of dissociative drug (disorder)
        '724714000',  # Dependence caused by dissociative substance (disorder)
        '724715004',  # Dependence caused by ketamine (disorder)
        '724720004',  # Mood disorder caused by dissociative drug (disorder)
        '724724008',  # Psychoactive substance dependence with current use (disorder)
        '724725009',  # Psychoactive substance withdrawal without complication (disorder)
        '724726005',  # Perceptual disturbances co-occurrent and due to psychoactive substance withdrawal (disorder)
        '724727001',  # Seizure co-occurrent and due to psychoactive substance withdrawal (disorder)
        '724728006',  # Perceptual disturbances and seizures co-occurrent and due to psychoactive substance withdrawal (disorder)
        '724732000',  # Sleep disorder caused by psychoactive substance (disorder)
        '735750005',  # Psychotic disorder with schizophreniform symptoms caused by cocaine (disorder)
        '737338002',  # Synthetic cannabinoid withdrawal (disorder)
        '762320004',  # Mood disorder with manic symptoms caused by opioid (disorder)
        '762321000',  # Mood disorder with depressive symptoms caused by opioid (disorder)
        '762322007',  # Mood disorder with mixed depressive and manic symptoms caused by opioid (disorder)
        '762324008',  # Delirium caused by stimulant (disorder)
        '762325009',  # Psychotic disorder caused by stimulant (disorder)
        '762326005',  # Psychotic disorder with hallucinations caused by stimulant (disorder)
        '762327001',  # Psychotic disorder with delusions caused by stimulant (disorder)
        '762333005',  # Sexual dysfunction caused by stimulant (disorder)
        '762334004',  # Sleep disorder caused by stimulant (disorder)
        '762335003',  # Mood disorder with manic symptoms caused by hallucinogen (disorder)
        '762336002',  # Mood disorder with depressive symptoms caused by hallucinogen (disorder)
        '762337006',  # Mood disorder with mixed depressive and manic symptoms caused by hallucinogen (disorder)
        '762338001',  # Mood disorder with manic symptoms caused by volatile inhalant (disorder)
        '762339009',  # Mood disorder with depressive symptoms caused by volatile inhalant (disorder)
        '762340006',  # Mood disorder with mixed depressive and manic symptoms caused by volatile inhalant (disorder)
        '762344002',  # Mood disorder with manic symptoms caused by dissociative drug (disorder)
        '762345001',  # Mood disorder with depressive symptoms caused by dissociative drug (disorder)
        '762346000',  # Mood disorder with mixed depressive and manic symptoms caused by dissociative drug (disorder)
        '762517008',  # Sexual dysfunction caused by synthetic cathinone (disorder)
        '762672001',  # Synthetic cathinone withdrawal (disorder)
        '785277001',  # Harmful pattern of use of amfetamine and amfetamine derivative (disorder)
        '788863007',  # Amnestic disorder caused by psychoactive substance (disorder)
        '788904003',  # Disorder caused by dissociative drug (disorder)
        '788905002',  # Disorder caused by stimulant (disorder)
        '788955005',  # Impulse control disorder caused by cocaine (disorder)
        '788956006',  # Impulse control disorder caused by psychoactive substance (disorder)
        '788957002',  # Impulse control disorder caused by stimulant (disorder)
        '788983007',  # Mood disorder caused by cannabis (disorder)
        '789119007',  # Mood disorder caused by synthetic cannabinoid (disorder)
        '817962007',  # Chlordiazepoxide dependence (disorder)
        '838527002',  # Amfetamine and amfetamine derivative drug dependence (disorder)
        '838528007',  # Mood disorder with manic symptoms caused by amfetamine and amfetamine derivative (disorder)
        '838529004',  # Mood disorder with mixed depressive and manic symptoms caused by amfetamine and amfetamine derivative (disorder)
        '838530009',  # Mood disorder with depressive symptoms caused by amfetamine and amfetamine derivative (disorder)
        '1081000119105',  # Opioid dependence, on agonist therapy (disorder)
        '23601000119102',  # Sedative withdrawal (disorder)
        '34111000119108',  # Nondependent amphetamine abuse (disorder)
        '86391000119101',  # Inhalant dependence, continuous (disorder)
        '86401000119104',  # Inhalant dependence, episodic (disorder)
        '97571000119109',  # Thrombocytopenia co-occurrent and due to alcoholism (disorder)
        '125851000119106',  # Continuous sedative abuse (disorder)
        '135301000119103',  # Parasomnia caused by alcohol (disorder)
        '135311000119100',  # Insomnia caused by alcohol (disorder)
        '135321000119107',  # Hypersomnia caused by alcohol (disorder)
        '144981000119109',  # Nondependent intraveous amphetamine abuse (disorder)
        '145101000119102',  # Intravenous cocaine abuse (disorder)
        '145121000119106',  # Intravenous nondependent opioid abuse (disorder)
        '145841000119107',  # Episodic phencyclidine abuse (disorder)
        '154211000119108',  # Chronic pancreatitis due to chronic alcoholism (disorder)
        '288021000119107',  # Disorder due to alcohol abuse (disorder)
        '288031000119105',  # Alcohol induced disorder co-occurrent and due to alcohol dependence (disorder)
        '288461000119105',  # Psychosis caused by inhalant (disorder)
        '288851000119106',  # Opioid-induced mood disorder due to opioid abuse (disorder)
        '288861000119108',  # Opioid-induced mood disorder due to opioid dependence (disorder)
        '10741871000119101',  # Alcohol dependence in pregnancy (disorder)
        '10755041000119100',  # Alcohol dependence in childbirth (disorder)
        '11047881000119101',  # Cannabis hyperemesis syndrome co-occurrent and due to cannabis abuse (disorder)
        '11048011000119103',  # Cannabis hyperemesis syndrome co-occurrent and due to cannabis dependence (disorder)
        '12398651000119100',  # Sleep disorder caused by methamphetamine (disorder)
    }


class LimitedLifeExpectancy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  limited life expectancy, including poor prognosis or terminal illness.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis that indicates a poor prognosis of a patient’s health status such as terminal or pre-terminal illness.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS138v10
    """

    VALUE_SET_NAME = 'Limited Life Expectancy'
    OID = '2.16.840.1.113883.3.526.3.1259'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    SNOMEDCT = {
        '27143004',  # Patient status determination, pre-terminal (finding)
        '111947009',  # Functional disorder present, condition terminal (finding)
        '162607003',  # Terminal illness - early stage (finding)
        '162608008',  # Terminal illness - late stage (finding)
        '170969009',  # Prognosis bad (finding)
        '300936002',  # Terminal illness (finding)
    }


class PrimaryOpenAngleGlaucoma(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of primary open-angle glaucoma (POAG).

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of primary open-angle glaucoma (POAG) and open-angle glaucoma.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis for 'unspecified eye.'

    ** Used in:** CMS143v10
    """

    VALUE_SET_NAME = 'Primary Open-Angle Glaucoma'
    OID = '2.16.840.1.113883.3.526.3.326'
    DEFINITION_VERSION = '20210209'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'H40.10X0',  # Unspecified open-angle glaucoma, stage unspecified
        'H40.10X1',  # Unspecified open-angle glaucoma, mild stage
        'H40.10X2',  # Unspecified open-angle glaucoma, moderate stage
        'H40.10X3',  # Unspecified open-angle glaucoma, severe stage
        'H40.10X4',  # Unspecified open-angle glaucoma, indeterminate stage
        'H40.1110',  # Primary open-angle glaucoma, right eye, stage unspecified
        'H40.1111',  # Primary open-angle glaucoma, right eye, mild stage
        'H40.1112',  # Primary open-angle glaucoma, right eye, moderate stage
        'H40.1113',  # Primary open-angle glaucoma, right eye, severe stage
        'H40.1114',  # Primary open-angle glaucoma, right eye, indeterminate stage
        'H40.1120',  # Primary open-angle glaucoma, left eye, stage unspecified
        'H40.1121',  # Primary open-angle glaucoma, left eye, mild stage
        'H40.1122',  # Primary open-angle glaucoma, left eye, moderate stage
        'H40.1123',  # Primary open-angle glaucoma, left eye, severe stage
        'H40.1124',  # Primary open-angle glaucoma, left eye, indeterminate stage
        'H40.1130',  # Primary open-angle glaucoma, bilateral, stage unspecified
        'H40.1131',  # Primary open-angle glaucoma, bilateral, mild stage
        'H40.1132',  # Primary open-angle glaucoma, bilateral, moderate stage
        'H40.1133',  # Primary open-angle glaucoma, bilateral, severe stage
        'H40.1134',  # Primary open-angle glaucoma, bilateral, indeterminate stage
        'H40.1210',  # Low-tension glaucoma, right eye, stage unspecified
        'H40.1211',  # Low-tension glaucoma, right eye, mild stage
        'H40.1212',  # Low-tension glaucoma, right eye, moderate stage
        'H40.1213',  # Low-tension glaucoma, right eye, severe stage
        'H40.1214',  # Low-tension glaucoma, right eye, indeterminate stage
        'H40.1220',  # Low-tension glaucoma, left eye, stage unspecified
        'H40.1221',  # Low-tension glaucoma, left eye, mild stage
        'H40.1222',  # Low-tension glaucoma, left eye, moderate stage
        'H40.1223',  # Low-tension glaucoma, left eye, severe stage
        'H40.1224',  # Low-tension glaucoma, left eye, indeterminate stage
        'H40.1230',  # Low-tension glaucoma, bilateral, stage unspecified
        'H40.1231',  # Low-tension glaucoma, bilateral, mild stage
        'H40.1232',  # Low-tension glaucoma, bilateral, moderate stage
        'H40.1233',  # Low-tension glaucoma, bilateral, severe stage
        'H40.1234',  # Low-tension glaucoma, bilateral, indeterminate stage
        'H40.151',  # Residual stage of open-angle glaucoma, right eye
        'H40.152',  # Residual stage of open-angle glaucoma, left eye
        'H40.153',  # Residual stage of open-angle glaucoma, bilateral
    }
    SNOMEDCT = {
        '50485007',  # Low tension glaucoma (disorder)
        '77075001',  # Primary open angle glaucoma (disorder)
        '84494001',  # Open-angle glaucoma (disorder)
        '111513000',  # Advanced open-angle glaucoma (disorder)
        '193533000',  # Open-angle glaucoma - borderline (disorder)
        '15633281000119103',  # Bilateral primary open angle glaucoma (disorder)
        '15633321000119108',  # Primary open angle glaucoma of left eye (disorder)
        '15640441000119104',  # Primary open angle glaucoma of right eye (disorder)
    }


class AllergyToBetaBlockerTherapy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of an allergy to beta blocker therapy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of an allergy to beta blocker therapy.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = 'Allergy to Beta Blocker Therapy'
    OID = '2.16.840.1.113883.3.526.3.1177'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    SNOMEDCT = {
        '293962009',  # Allergy to beta adrenergic receptor antagonist (finding)
        '293963004',  # Allergy to beta-1 adrenergic receptor antagonist (finding)
        '293964005',  # Allergy to acebutolol (finding)
        '293965006',  # Allergy to atenolol (finding)
        '293966007',  # Allergy to betaxolol (finding)
        '293967003',  # Allergy to bisoprolol (finding)
        '293968008',  # Allergy to celiprolol (finding)
        '293969000',  # Allergy to esmolol (finding)
        '293970004',  # Allergy to metoprolol (finding)
        '293972007',  # Allergy to nadolol (finding)
        '293973002',  # Allergy to pindolol (finding)
        '293974008',  # Allergy to carvedilol (finding)
        '293975009',  # Allergy to metipranolol (finding)
        '293976005',  # Allergy to carteolol (finding)
        '293977001',  # Allergy to labetalol (finding)
        '293978006',  # Allergy to levobunolol (finding)
        '293979003',  # Allergy to oxprenolol (finding)
        '293980000',  # Allergy to penbutolol (finding)
        '293981001',  # Allergy to practolol (finding)
        '293982008',  # Allergy to propranolol (finding)
        '293983003',  # Allergy to sotalol (finding)
        '293984009',  # Allergy to timolol (finding)
    }


class Arrhythmia(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of arrhythmia.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of arrhythmia.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = 'Arrhythmia'
    OID = '2.16.840.1.113883.3.526.3.366'
    DEFINITION_VERSION = '20200306'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'I49.8',  # Other specified cardiac arrhythmias
        'I49.9',  # Cardiac arrhythmia, unspecified
    }
    SNOMEDCT = {
        '184004',  # Withdrawal arrhythmia (disorder)
        '10164001',  # Parasystole (disorder)
        '10626002',  # Multifocal premature ventricular complexes (disorder)
        '11157007',  # Ventricular bigeminy (disorder)
        '11849007',  # Atrioventricular junctional rhythm (disorder)
        '13640000',  # Fusion beats (disorder)
        '17338001',  # Ventricular premature beats (disorder)
        '17366009',  # Atrial arrhythmia (disorder)
        '17869006',  # Anomalous atrioventricular excitation (disorder)
        '26950008',  # Chronic ectopic atrial tachycardia (disorder)
        '27337007',  # Unifocal premature ventricular complexes (disorder)
        '29320008',  # Ectopic rhythm (disorder)
        '33413000',  # Ectopic beats (disorder)
        '36083008',  # Sick sinus syndrome (disorder)
        '38274001',  # Interpolated ventricular premature complexes (disorder)
        '39260000',  # Nonparoxysmal atrioventricular nodal tachycardia (disorder)
        '39357005',  # Paroxysmal atrial tachycardia with block (disorder)
        '40593004',  # Fibrillation (disorder)
        '44808001',  # Conduction disorder of the heart (disorder)
        '47830009',  # Junctional escape beats (disorder)
        '49982000',  # Multifocal atrial tachycardia (disorder)
        '55475008',  # Lown-Ganong-Levine syndrome (disorder)
        '59272004',  # Ventricular parasystole (disorder)
        '60423000',  # Sinus node dysfunction (disorder)
        '61277005',  # Accelerated idioventricular rhythm (disorder)
        '63232000',  # Multifocal premature beats (disorder)
        '63593006',  # Supraventricular premature beats (disorder)
        '69730002',  # Idiojunctional tachycardia (disorder)
        '71908006',  # Ventricular fibrillation (disorder)
        '72654001',  # Supraventricular arrhythmia (disorder)
        '74390002',  # Wolff-Parkinson-White pattern (disorder)
        '74615001',  # Tachycardia-bradycardia (disorder)
        '75532003',  # Ventricular escape beat (disorder)
        '81681009',  # Junctional premature beats (disorder)
        '81898007',  # Ventricular escape rhythm (disorder)
        '88412007',  # Atrio-ventricular node arrhythmia (disorder)
        '195060002',  # Ventricular pre-excitation (disorder)
        '195069001',  # Paroxysmal atrial tachycardia (disorder)
        '195071001',  # Paroxysmal junctional tachycardia (disorder)
        '195072008',  # Paroxysmal nodal tachycardia (disorder)
        '195083004',  # Ventricular fibrillation and flutter (disorder)
        '233891009',  # Sinoatrial node tachycardia (disorder)
        '233892002',  # Ectopic atrial tachycardia (disorder)
        '233893007',  # Re-entrant atrial tachycardia (disorder)
        '233894001',  # Incessant atrial tachycardia (disorder)
        '233895000',  # Ectopic atrioventricular node tachycardia (disorder)
        '233904005',  # Permanent junctional reciprocating tachycardia (disorder)
        '233915000',  # Paroxysmal familial ventricular fibrillation (disorder)
        '233922008',  # Concealed accessory pathway (disorder)
        '233923003',  # Unidirectional retrograde accessory pathway (disorder)
        '234172002',  # Electromechanical dissociation (disorder)
        '251161003',  # Slow ventricular response (disorder)
        '251162005',  # Atrio-ventricular-junctional (nodal) bradycardia (disorder)
        '251163000',  # Atrio-ventricular junctional (nodal) arrest (disorder)
        '251164006',  # Junctional premature complex (disorder)
        '251165007',  # Atrioventricular junctional (nodal) tachycardia (disorder)
        '251166008',  # Atrioventricular nodal re-entry tachycardia (disorder)
        '251167004',  # Aberrant premature complexes (disorder)
        '251168009',  # Supraventricular bigeminy (disorder)
        '251170000',  # Blocked premature atrial contraction (disorder)
        '251172008',  # Run of atrial premature complexes (disorder)
        '251173003',  # Atrial bigeminy (disorder)
        '251174009',  # Atrial trigeminy (disorder)
        '251175005',  # Ventricular premature complex (disorder)
        '251176006',  # Multiple premature ventricular complexes (disorder)
        '251177002',  # Run of ventricular premature complexes (disorder)
        '251178007',  # Ventricular interpolated complexes (disorder)
        '251179004',  # Multiple ventricular interpolated complexes (disorder)
        '251180001',  # Ventricular trigeminy (disorder)
        '251181002',  # Ventricular quadrigeminy (disorder)
        '251182009',  # Paired ventricular premature complexes (disorder)
        '251186007',  # Ventricular escape complex (disorder)
        '251187003',  # Atrial escape complex (disorder)
        '251188008',  # Atrial parasystole (disorder)
        '276796006',  # Atrial tachycardia (disorder)
        '284470004',  # Premature atrial contraction (disorder)
        '287057009',  # Atrial premature complex (disorder)
        '309809007',  # Electromechanical dissociation with successful resuscitation (disorder)
        '406461004',  # Ectopic atrial beats (disorder)
        '418341009',  # Atrioventricular conduction disorder (disorder)
        '418818005',  # Brugada syndrome (disorder)
        '419752005',  # Sinoatrial nodal reentrant tachycardia (disorder)
        '421869004',  # Bradyarrhythmia (disorder)
        '422348008',  # Andersen Tawil syndrome (disorder)
        '429243003',  # Sustained ventricular fibrillation (disorder)
    }


class Asthma(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of diagnoses for asthma.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of asthma.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = 'Asthma'
    OID = '2.16.840.1.113883.3.526.3.362'
    DEFINITION_VERSION = '20200306'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'J45.20',  # Mild intermittent asthma, uncomplicated
        'J45.21',  # Mild intermittent asthma with (acute) exacerbation
        'J45.22',  # Mild intermittent asthma with status asthmaticus
        'J45.30',  # Mild persistent asthma, uncomplicated
        'J45.31',  # Mild persistent asthma with (acute) exacerbation
        'J45.32',  # Mild persistent asthma with status asthmaticus
        'J45.40',  # Moderate persistent asthma, uncomplicated
        'J45.41',  # Moderate persistent asthma with (acute) exacerbation
        'J45.42',  # Moderate persistent asthma with status asthmaticus
        'J45.50',  # Severe persistent asthma, uncomplicated
        'J45.51',  # Severe persistent asthma with (acute) exacerbation
        'J45.52',  # Severe persistent asthma with status asthmaticus
        'J45.901',  # Unspecified asthma with (acute) exacerbation
        'J45.902',  # Unspecified asthma with status asthmaticus
        'J45.909',  # Unspecified asthma, uncomplicated
        'J45.990',  # Exercise induced bronchospasm
        'J45.991',  # Cough variant asthma
        'J45.998',  # Other asthma
    }
    SNOMEDCT = {
        '11641008',  # Millers' asthma (disorder)
        '12428000',  # Intrinsic asthma without status asthmaticus (disorder)
        '13151001',  # Flax-dressers' disease (disorder)
        '31387002',  # Exercise-induced asthma (disorder)
        '55570000',  # Asthma without status asthmaticus (disorder)
        '56968009',  # Asthma caused by wood dust (disorder)
        '59786004',  # Weavers' cough (disorder)
        '63088003',  # Allergic asthma without status asthmaticus (disorder)
        '85761009',  # Byssinosis (disorder)
        '92807009',  # Chemical-induced asthma (disorder)
        '93432008',  # Drug-induced asthma (disorder)
        '195949008',  # Chronic asthmatic bronchitis (disorder)
        '195967001',  # Asthma (disorder)
        '195977004',  # Mixed asthma (disorder)
        '225057002',  # Brittle asthma (disorder)
        '233672007',  # Byssinosis grade 3 (disorder)
        '233678006',  # Childhood asthma (disorder)
        '233679003',  # Late onset asthma (disorder)
        '233683003',  # Hay fever with asthma (disorder)
        '233688007',  # Sulfite-induced asthma (disorder)
        '266361008',  # Non-allergic asthma (disorder)
        '281239006',  # Exacerbation of asthma (disorder)
        '304527002',  # Acute asthma (disorder)
        '370218001',  # Mild asthma (disorder)
        '370219009',  # Moderate asthma (disorder)
        '370220003',  # Occasional asthma (disorder)
        '370221004',  # Severe asthma (disorder)
        '389145006',  # Allergic asthma (disorder)
        '405944004',  # Asthmatic bronchitis (disorder)
        '407674008',  # Aspirin-induced asthma (disorder)
        '409663006',  # Cough variant asthma (disorder)
        '423889005',  # Non-immunoglobulin E mediated allergic asthma (disorder)
        '424199006',  # Substance induced asthma (disorder)
        '424643009',  # Immunoglobulin E-mediated allergic asthma (disorder)
        '425969006',  # Exacerbation of intermittent asthma (disorder)
        '426656000',  # Severe persistent asthma (disorder)
        '426979002',  # Mild persistent asthma (disorder)
        '427295004',  # Moderate persistent asthma (disorder)
        '427603009',  # Intermittent asthma (disorder)
        '427679007',  # Mild intermittent asthma (disorder)
        '442025000',  # Acute exacerbation of chronic asthmatic bronchitis (disorder)
        '707445000',  # Exacerbation of mild persistent asthma (disorder)
        '707446004',  # Exacerbation of moderate persistent asthma (disorder)
        '707447008',  # Exacerbation of severe persistent asthma (disorder)
        '708038006',  # Acute exacerbation of asthma (disorder)
        '708090002',  # Acute severe exacerbation of asthma (disorder)
        '708093000',  # Acute exacerbation of immunoglobulin E-mediated allergic asthma (disorder)
        '708094006',  # Acute exacerbation of intrinsic asthma (disorder)
        '708095007',  # Acute severe exacerbation of immunoglobin E-mediated allergic asthma (disorder)
        '708096008',  # Acute severe exacerbation of intrinsic asthma (disorder)
        '782520007',  # Exacerbation of allergic asthma due to infection (disorder)
    }


class AtrioventricularBlock(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of atrioventricular block.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of atrioventricular block.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = 'Atrioventricular Block'
    OID = '2.16.840.1.113883.3.526.3.367'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'I44.0',  # Atrioventricular block, first degree
        'I44.1',  # Atrioventricular block, second degree
        'I44.2',  # Atrioventricular block, complete
        'I44.30',  # Unspecified atrioventricular block
        'I44.39',  # Other atrioventricular block
    }
    SNOMEDCT = {
        '27885002',  # Complete atrioventricular block (disorder)
        '28189009',  # Mobitz type II atrioventricular block (disorder)
        '54016002',  # Mobitz type I incomplete atrioventricular block (disorder)
        '195042002',  # Second degree atrioventricular block (disorder)
        '204383001',  # Congenital complete atrioventricular heart block (disorder)
        '233917008',  # Atrioventricular block (disorder)
        '233918003',  # Postoperative complete heart block (disorder)
        '251114004',  # Intermittent second degree atrioventricular block (disorder)
        '283645003',  # Lev's syndrome (disorder)
        '733125004',  # Acquired complete atrioventricular block (disorder)
        '102451000119107',  # Complete atrioventricular block as complication of atrioventricular nodal ablation (disorder)
        '284941000119107',  # High degree second degree atrioventricular block (disorder)
    }


class Bradycardia(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of bradycardia.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of bradycardia.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = 'Bradycardia'
    OID = '2.16.840.1.113883.3.526.3.412'
    DEFINITION_VERSION = '20200306'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'I49.5',  # Sick sinus syndrome
        'I49.8',  # Other specified cardiac arrhythmias
        'R00.1',  # Bradycardia, unspecified
    }
    SNOMEDCT = {
        '29894000',  # Vagal autonomic bradycardia (disorder)
        '44602002',  # Persistent sinus bradycardia (disorder)
        '49044005',  # Severe sinus bradycardia (disorder)
        '49710005',  # Sinus bradycardia (disorder)
        '251162005',  # Atrio-ventricular-junctional (nodal) bradycardia (disorder)
        '397841007',  # Drug-induced bradycardia (disorder)
    }


class CardiacPacerInSitu(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of cardiac pacer in situ.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of cardiac pacer in situ.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = 'Cardiac Pacer in Situ'
    OID = '2.16.840.1.113883.3.526.3.368'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'Z95.0',  # Presence of cardiac pacemaker
        'Z95.810',  # Presence of automatic (implantable) cardiac defibrillator
    }
    SNOMEDCT = {
        '441509002',  # Cardiac pacemaker in situ (finding)
    }


class Hypotension(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of hypotension.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of hypotension.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = 'Hypotension'
    OID = '2.16.840.1.113883.3.526.3.370'
    DEFINITION_VERSION = '20200306'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'I95.0',  # Idiopathic hypotension
        'I95.1',  # Orthostatic hypotension
        'I95.2',  # Hypotension due to drugs
        'I95.3',  # Hypotension of hemodialysis
        'I95.81',  # Postprocedural hypotension
        'I95.89',  # Other hypotension
        'I95.9',  # Hypotension, unspecified
    }
    SNOMEDCT = {
        '45007003',  # Low blood pressure (disorder)
        '61933008',  # Hyperadrenergic postural hypotension (disorder)
        '70247006',  # Hypoadrenergic postural hypotension (disorder)
        '75181005',  # Chronic orthostatic hypotension (disorder)
        '77545000',  # Chronic hypotension (disorder)
        '88887003',  # Maternal hypotension syndrome (disorder)
        '195506001',  # Idiopathic hypotension (disorder)
        '200113008',  # Maternal hypotension syndrome with antenatal problem (disorder)
        '200114002',  # Maternal hypotension syndrome with postnatal problem (disorder)
        '230664009',  # Sympathotonic orthostatic hypotension (disorder)
        '234171009',  # Drug-induced hypotension (disorder)
        '271870002',  # Low blood pressure reading (disorder)
        '286963007',  # Chronic hypotension - idiopathic (disorder)
        '371073003',  # Postural orthostatic tachycardia syndrome (disorder)
        '408667000',  # Hemodialysis-associated hypotension (disorder)
        '408668005',  # Iatrogenic hypotension (disorder)
        '429561008',  # Exertional hypotension (disorder)
    }


class IntoleranceToBetaBlockerTherapy(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of an intolerance to beta blocker therapy.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of an intolerance to beta blocker therapy.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS144v10, CMS145v10
    """

    VALUE_SET_NAME = 'Intolerance to Beta Blocker Therapy'
    OID = '2.16.840.1.113883.3.526.3.1178'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    SNOMEDCT = {
        '292419005',  # Beta-adrenoceptor blocking drug adverse reaction (disorder)
        '292420004',  # Cardioselective beta-blocker adverse reaction (disorder)
        '292421000',  # Acebutolol adverse reaction (disorder)
        '292424008',  # Betaxolol adverse reaction (disorder)
        '292425009',  # Bisoprolol adverse reaction (disorder)
        '292426005',  # Celiprolol adverse reaction (disorder)
        '292427001',  # Esmolol adverse reaction (disorder)
        '292428006',  # Metoprolol adverse reaction (disorder)
        '292430008',  # Nadolol adverse reaction (disorder)
        '292431007',  # Pindolol adverse reaction (disorder)
        '292432000',  # Carvedilol adverse reaction (disorder)
        '292433005',  # Metipranolol adverse reaction (disorder)
        '292434004',  # Carteolol adverse reaction (disorder)
        '292435003',  # Labetalol adverse reaction (disorder)
        '292436002',  # Levobunolol adverse reaction (disorder)
        '292437006',  # Oxprenolol adverse reaction (disorder)
        '292438001',  # Penbutolol adverse reaction (disorder)
        '292439009',  # Practolol adverse reaction (disorder)
        '292440006',  # Propranolol adverse reaction (disorder)
        '292441005',  # Sotalol adverse reaction (disorder)
        '292442003',  # Timolol adverse reaction (disorder)
        '418370000',  # Atenolol adverse reaction (disorder)
    }


class CoronaryArteryDiseaseNoMi(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts of a diagnosis of coronary artery disease without myocardial infarction.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of coronary artery disease without myocardial infarction.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS145v10
    """

    VALUE_SET_NAME = 'Coronary Artery Disease No MI'
    OID = '2.16.840.1.113883.3.526.3.369'
    DEFINITION_VERSION = '20200306'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'I20.0',  # Unstable angina
        'I20.1',  # Angina pectoris with documented spasm
        'I20.8',  # Other forms of angina pectoris
        'I20.9',  # Angina pectoris, unspecified
        'I24.0',  # Acute coronary thrombosis not resulting in myocardial infarction
        'I24.8',  # Other forms of acute ischemic heart disease
        'I24.9',  # Acute ischemic heart disease, unspecified
        'I25.10',  # Atherosclerotic heart disease of native coronary artery without angina pectoris
        'I25.110',  # Atherosclerotic heart disease of native coronary artery with unstable angina pectoris
        'I25.111',  # Atherosclerotic heart disease of native coronary artery with angina pectoris with documented spasm
        'I25.118',  # Atherosclerotic heart disease of native coronary artery with other forms of angina pectoris
        'I25.119',  # Atherosclerotic heart disease of native coronary artery with unspecified angina pectoris
        'I25.5',  # Ischemic cardiomyopathy
        'I25.6',  # Silent myocardial ischemia
        'I25.700',  # Atherosclerosis of coronary artery bypass graft(s), unspecified, with unstable angina pectoris
        'I25.701',  # Atherosclerosis of coronary artery bypass graft(s), unspecified, with angina pectoris with documented spasm
        'I25.708',  # Atherosclerosis of coronary artery bypass graft(s), unspecified, with other forms of angina pectoris
        'I25.709',  # Atherosclerosis of coronary artery bypass graft(s), unspecified, with unspecified angina pectoris
        'I25.710',  # Atherosclerosis of autologous vein coronary artery bypass graft(s) with unstable angina pectoris
        'I25.711',  # Atherosclerosis of autologous vein coronary artery bypass graft(s) with angina pectoris with documented spasm
        'I25.718',  # Atherosclerosis of autologous vein coronary artery bypass graft(s) with other forms of angina pectoris
        'I25.719',  # Atherosclerosis of autologous vein coronary artery bypass graft(s) with unspecified angina pectoris
        'I25.720',  # Atherosclerosis of autologous artery coronary artery bypass graft(s) with unstable angina pectoris
        'I25.721',  # Atherosclerosis of autologous artery coronary artery bypass graft(s) with angina pectoris with documented spasm
        'I25.728',  # Atherosclerosis of autologous artery coronary artery bypass graft(s) with other forms of angina pectoris
        'I25.729',  # Atherosclerosis of autologous artery coronary artery bypass graft(s) with unspecified angina pectoris
        'I25.730',  # Atherosclerosis of nonautologous biological coronary artery bypass graft(s) with unstable angina pectoris
        'I25.731',  # Atherosclerosis of nonautologous biological coronary artery bypass graft(s) with angina pectoris with documented spasm
        'I25.738',  # Atherosclerosis of nonautologous biological coronary artery bypass graft(s) with other forms of angina pectoris
        'I25.739',  # Atherosclerosis of nonautologous biological coronary artery bypass graft(s) with unspecified angina pectoris
        'I25.750',  # Atherosclerosis of native coronary artery of transplanted heart with unstable angina
        'I25.751',  # Atherosclerosis of native coronary artery of transplanted heart with angina pectoris with documented spasm
        'I25.758',  # Atherosclerosis of native coronary artery of transplanted heart with other forms of angina pectoris
        'I25.759',  # Atherosclerosis of native coronary artery of transplanted heart with unspecified angina pectoris
        'I25.760',  # Atherosclerosis of bypass graft of coronary artery of transplanted heart with unstable angina
        'I25.761',  # Atherosclerosis of bypass graft of coronary artery of transplanted heart with angina pectoris with documented spasm
        'I25.768',  # Atherosclerosis of bypass graft of coronary artery of transplanted heart with other forms of angina pectoris
        'I25.769',  # Atherosclerosis of bypass graft of coronary artery of transplanted heart with unspecified angina pectoris
        'I25.790',  # Atherosclerosis of other coronary artery bypass graft(s) with unstable angina pectoris
        'I25.791',  # Atherosclerosis of other coronary artery bypass graft(s) with angina pectoris with documented spasm
        'I25.798',  # Atherosclerosis of other coronary artery bypass graft(s) with other forms of angina pectoris
        'I25.799',  # Atherosclerosis of other coronary artery bypass graft(s) with unspecified angina pectoris
        'I25.810',  # Atherosclerosis of coronary artery bypass graft(s) without angina pectoris
        'I25.811',  # Atherosclerosis of native coronary artery of transplanted heart without angina pectoris
        'I25.812',  # Atherosclerosis of bypass graft of coronary artery of transplanted heart without angina pectoris
        'I25.82',  # Chronic total occlusion of coronary artery
        'I25.83',  # Coronary atherosclerosis due to lipid rich plaque
        'I25.89',  # Other forms of chronic ischemic heart disease
        'I25.9',  # Chronic ischemic heart disease, unspecified
        'Z95.1',  # Presence of aortocoronary bypass graft
        'Z95.5',  # Presence of coronary angioplasty implant and graft
        'Z98.61',  # Coronary angioplasty status
    }
    SNOMEDCT = {
        '4557003',  # Preinfarction syndrome (disorder)
        '10365005',  # Right main coronary artery thrombosis (disorder)
        '19057007',  # Status anginosus (disorder)
        '21470009',  # Syncope anginosa (disorder)
        '28248000',  # Left anterior descending coronary artery thrombosis (disorder)
        '29899005',  # Coronary artery embolism (disorder)
        '35928006',  # Nocturnal angina (disorder)
        '41334000',  # Angina, class II (disorder)
        '50570003',  # Aneurysm of coronary vessels (disorder)
        '53741008',  # Coronary arteriosclerosis (disorder)
        '59021001',  # Angina decubitus (disorder)
        '61490001',  # Angina, class I (disorder)
        '62207008',  # Syphilitic ostial coronary disease (disorder)
        '63739005',  # Coronary occlusion (disorder)
        '67682002',  # Coronary artery atheroma (disorder)
        '74218008',  # Coronary artery arising from main pulmonary artery (disorder)
        '75398000',  # Anomalous origin of coronary artery (disorder)
        '85284003',  # Angina, class III (disorder)
        '87343002',  # Prinzmetal angina (disorder)
        '89323001',  # Angina, class IV (disorder)
        '92517006',  # Calcific coronary arteriosclerosis (disorder)
        '123641001',  # Left coronary artery occlusion (disorder)
        '123642008',  # Right coronary artery occlusion (disorder)
        '194823009',  # Acute coronary insufficiency (disorder)
        '194828000',  # Angina (disorder)
        '194842008',  # Single coronary vessel disease (disorder)
        '194843003',  # Double coronary vessel disease (disorder)
        '233817007',  # Triple vessel disease of the heart (disorder)
        '233819005',  # Stable angina (disorder)
        '233821000',  # New onset angina (disorder)
        '233970002',  # Coronary artery stenosis (disorder)
        '300995000',  # Exercise-induced angina (disorder)
        '314116003',  # Post infarct angina (disorder)
        '315025001',  # Refractory angina (disorder)
        '315348000',  # Asymptomatic coronary heart disease (disorder)
        '371803003',  # Multi vessel coronary artery disease (disorder)
        '371804009',  # Left main coronary artery disease (disorder)
        '371805005',  # Significant coronary bypass graft disease (disorder)
        '371806006',  # Progressive angina (disorder)
        '371807002',  # Atypical angina (disorder)
        '371808007',  # Recurrent angina status post percutaneous transluminal coronary angioplasty (disorder)
        '371809004',  # Recurrent angina status post coronary stent placement (disorder)
        '371810009',  # Recurrent angina status post coronary artery bypass graft (disorder)
        '371811008',  # Recurrent angina status post rotational atherectomy (disorder)
        '371812001',  # Recurrent angina status post directional coronary atherectomy (disorder)
        '398274000',  # Coronary artery thrombosis (disorder)
        '408546009',  # Coronary artery bypass graft occlusion (disorder)
        '420006002',  # Obliterative coronary artery disease (disorder)
        '421327009',  # Coronary artery stent thrombosis (disorder)
        '427919004',  # Coronary arteriosclerosis caused by radiation (disorder)
        '429245005',  # Recurrent coronary arteriosclerosis after percutaneous transluminal coronary angioplasty (disorder)
        '429559004',  # Typical angina (disorder)
        '443502000',  # Atherosclerosis of coronary artery (disorder)
    }


class MyocardialInfarction(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for diagnoses of myocardial infarction.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of myocardial infarction.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS347v5, CMS145v10
    """

    VALUE_SET_NAME = 'Myocardial Infarction'
    OID = '2.16.840.1.113883.3.526.3.403'
    DEFINITION_VERSION = '20210218'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'I21.01',  # ST elevation (STEMI) myocardial infarction involving left main coronary artery
        'I21.02',  # ST elevation (STEMI) myocardial infarction involving left anterior descending coronary artery
        'I21.09',  # ST elevation (STEMI) myocardial infarction involving other coronary artery of anterior wall
        'I21.11',  # ST elevation (STEMI) myocardial infarction involving right coronary artery
        'I21.19',  # ST elevation (STEMI) myocardial infarction involving other coronary artery of inferior wall
        'I21.21',  # ST elevation (STEMI) myocardial infarction involving left circumflex coronary artery
        'I21.29',  # ST elevation (STEMI) myocardial infarction involving other sites
        'I21.3',  # ST elevation (STEMI) myocardial infarction of unspecified site
        'I21.4',  # Non-ST elevation (NSTEMI) myocardial infarction
        'I21.9',  # Acute myocardial infarction, unspecified
        'I21.A1',  # Myocardial infarction type 2
        'I21.A9',  # Other myocardial infarction type
        'I22.0',  # Subsequent ST elevation (STEMI) myocardial infarction of anterior wall
        'I22.1',  # Subsequent ST elevation (STEMI) myocardial infarction of inferior wall
        'I22.2',  # Subsequent non-ST elevation (NSTEMI) myocardial infarction
        'I22.8',  # Subsequent ST elevation (STEMI) myocardial infarction of other sites
        'I22.9',  # Subsequent ST elevation (STEMI) myocardial infarction of unspecified site
    }
    ICD9CM = {
        '410.00',  # Acute myocardial infarction of anterolateral wall, episode of care unspecified
        '410.01',  # Acute myocardial infarction of anterolateral wall, initial episode of care
        '410.02',  # Acute myocardial infarction of anterolateral wall, subsequent episode of care
        '410.10',  # Acute myocardial infarction of other anterior wall, episode of care unspecified
        '410.11',  # Acute myocardial infarction of other anterior wall, initial episode of care
        '410.12',  # Acute myocardial infarction of other anterior wall, subsequent episode of care
        '410.20',  # Acute myocardial infarction of inferolateral wall, episode of care unspecified
        '410.21',  # Acute myocardial infarction of inferolateral wall, initial episode of care
        '410.22',  # Acute myocardial infarction of inferolateral wall, subsequent episode of care
        '410.30',  # Acute myocardial infarction of inferoposterior wall, episode of care unspecified
        '410.31',  # Acute myocardial infarction of inferoposterior wall, initial episode of care
        '410.32',  # Acute myocardial infarction of inferoposterior wall, subsequent episode of care
        '410.40',  # Acute myocardial infarction of other inferior wall, episode of care unspecified
        '410.41',  # Acute myocardial infarction of other inferior wall, initial episode of care
        '410.42',  # Acute myocardial infarction of other inferior wall, subsequent episode of care
        '410.50',  # Acute myocardial infarction of other lateral wall, episode of care unspecified
        '410.51',  # Acute myocardial infarction of other lateral wall, initial episode of care
        '410.52',  # Acute myocardial infarction of other lateral wall, subsequent episode of care
        '410.60',  # True posterior wall infarction, episode of care unspecified
        '410.61',  # True posterior wall infarction, initial episode of care
        '410.62',  # True posterior wall infarction, subsequent episode of care
        '410.70',  # Subendocardial infarction, episode of care unspecified
        '410.71',  # Subendocardial infarction, initial episode of care
        '410.72',  # Subendocardial infarction, subsequent episode of care
        '410.80',  # Acute myocardial infarction of other specified sites, episode of care unspecified
        '410.81',  # Acute myocardial infarction of other specified sites, initial episode of care
        '410.82',  # Acute myocardial infarction of other specified sites, subsequent episode of care
        '410.90',  # Acute myocardial infarction of unspecified site, episode of care unspecified
        '410.91',  # Acute myocardial infarction of unspecified site, initial episode of care
        '410.92',  # Acute myocardial infarction of unspecified site, subsequent episode of care
    }
    SNOMEDCT = {
        '10273003',  # Acute infarction of papillary muscle (disorder)
        '15990001',  # Acute myocardial infarction of posterolateral wall (disorder)
        '22298006',  # Myocardial infarction (disorder)
        '30277009',  # Rupture of ventricle due to acute myocardial infarction (disorder)
        '32574007',  # Past myocardial infarction diagnosed on electrocardiogram AND/OR other special investigation, but currently presenting no symptoms (disorder)
        '42531007',  # Microinfarct of heart (disorder)
        '52035003',  # Acute anteroapical myocardial infarction (disorder)
        '54329005',  # Acute myocardial infarction of anterior wall (disorder)
        '57054005',  # Acute myocardial infarction (disorder)
        '58612006',  # Acute myocardial infarction of lateral wall (disorder)
        '62695002',  # Acute anteroseptal myocardial infarction (disorder)
        '65547006',  # Acute myocardial infarction of inferolateral wall (disorder)
        '70211005',  # Acute myocardial infarction of anterolateral wall (disorder)
        '70422006',  # Acute subendocardial infarction (disorder)
        '73795002',  # Acute myocardial infarction of inferior wall (disorder)
        '79009004',  # Acute myocardial infarction of septum (disorder)
        '129574000',  # Postoperative myocardial infarction (disorder)
        '194802003',  # True posterior myocardial infarction (disorder)
        '194809007',  # Acute atrial infarction (disorder)
        '194856005',  # Subsequent myocardial infarction (disorder)
        '233835003',  # Acute widespread myocardial infarction (disorder)
        '233838001',  # Acute posterior myocardial infarction (disorder)
        '233843008',  # Silent myocardial infarction (disorder)
        '304914007',  # Acute Q wave myocardial infarction (disorder)
        '307140009',  # Acute non-Q wave infarction (disorder)
        '314207007',  # Non-Q wave myocardial infarction (disorder)
        '371068009',  # Myocardial infarction with complication (disorder)
        '394710008',  # First myocardial infarction (disorder)
        '401303003',  # Acute ST segment elevation myocardial infarction (disorder)
        '401314000',  # Acute non-ST segment elevation myocardial infarction (disorder)
        '418044006',  # Myocardial infarction in recovery phase (disorder)
        '428196007',  # Mixed myocardial ischemia and infarction (disorder)
        '428752002',  # Recent myocardial infarction (situation)
    }


class AcutePharyngitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  bacterial or viral acute pharyngitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of either bacterial or viral acute pharyngitis.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS146v10
    """

    VALUE_SET_NAME = 'Acute Pharyngitis'
    OID = '2.16.840.1.113883.3.464.1003.102.12.1011'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'J02.0',  # Streptococcal pharyngitis
        'J02.8',  # Acute pharyngitis due to other specified organisms
        'J02.9',  # Acute pharyngitis, unspecified
    }
    SNOMEDCT = {
        '1532007',  # Viral pharyngitis (disorder)
        '40766000',  # Enteroviral lymphonodular pharyngitis (disorder)
        '43878008',  # Streptococcal sore throat (disorder)
        '58031004',  # Suppurative pharyngitis (disorder)
        '195655000',  # Acute gangrenous pharyngitis (disorder)
        '195656004',  # Acute phlegmonous pharyngitis (disorder)
        '195657008',  # Acute ulcerative pharyngitis (disorder)
        '195658003',  # Acute bacterial pharyngitis (disorder)
        '195659006',  # Acute pneumococcal pharyngitis (disorder)
        '195660001',  # Acute staphylococcal pharyngitis (disorder)
        '195662009',  # Acute viral pharyngitis (disorder)
        '232399005',  # Acute herpes simplex pharyngitis (disorder)
        '232400003',  # Acute herpes zoster pharyngitis (disorder)
        '363746003',  # Acute pharyngitis (disorder)
    }


class AcuteTonsillitis(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  acute tonsillitis.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of either bacterial or viral acute tonsillitis.

    **Exclusion Criteria:** No exclusions.

    ** Used in:** CMS146v10
    """

    VALUE_SET_NAME = 'Acute Tonsillitis'
    OID = '2.16.840.1.113883.3.464.1003.102.12.1012'
    DEFINITION_VERSION = '20170504'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'J03.00',  # Acute streptococcal tonsillitis, unspecified
        'J03.01',  # Acute recurrent streptococcal tonsillitis
        'J03.80',  # Acute tonsillitis due to other specified organisms
        'J03.81',  # Acute recurrent tonsillitis due to other specified organisms
        'J03.90',  # Acute tonsillitis, unspecified
        'J03.91',  # Acute recurrent tonsillitis, unspecified
    }
    SNOMEDCT = {
        '17741008',  # Acute tonsillitis (disorder)
        '195666007',  # Acute erythematous tonsillitis (disorder)
        '195668008',  # Acute ulcerative tonsillitis (disorder)
        '195669000',  # Acute catarrhal tonsillitis (disorder)
        '195670004',  # Acute gangrenous tonsillitis (disorder)
        '195671000',  # Acute bacterial tonsillitis (disorder)
        '195672007',  # Acute pneumococcal tonsillitis (disorder)
        '195673002',  # Acute staphylococcal tonsillitis (disorder)
        '195676005',  # Acute viral tonsillitis (disorder)
        '195677001',  # Recurrent acute tonsillitis (disorder)
        '302911003',  # Acute lingual tonsillitis (disorder)
        '10629231000119109',  # Recurrent acute streptococcal tonsillitis (disorder)
        '10629271000119107',  # Acute tonsillitis caused by Streptococcus (disorder)
    }


class ComorbidConditionsForRespiratoryConditions(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of immunocompromising and chronic respiratory conditions, which increase susceptibility of contracting an acute bacterial infection.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of immunocompromising and chronic respiratory conditions, which increase susceptibility of contracting an acute bacterial infection.

    **Exclusion Criteria:** Excludes concepts that represent a diagnosis that is not associated with increased susceptibility of contracting an acute bacterial infection.

    ** Used in:** CMS154v10, CMS146v10
    """

    VALUE_SET_NAME = 'Comorbid Conditions for Respiratory Conditions'
    OID = '2.16.840.1.113883.3.464.1003.102.12.1025'
    DEFINITION_VERSION = '20200310'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'A15.0',  # Tuberculosis of lung
        'A15.4',  # Tuberculosis of intrathoracic lymph nodes
        'A15.5',  # Tuberculosis of larynx, trachea and bronchus
        'A15.6',  # Tuberculous pleurisy
        'A15.7',  # Primary respiratory tuberculosis
        'A15.8',  # Other respiratory tuberculosis
        'A15.9',  # Respiratory tuberculosis unspecified
        'A17.0',  # Tuberculous meningitis
        'A17.1',  # Meningeal tuberculoma
        'A17.81',  # Tuberculoma of brain and spinal cord
        'A17.82',  # Tuberculous meningoencephalitis
        'A17.83',  # Tuberculous neuritis
        'A17.89',  # Other tuberculosis of nervous system
        'A17.9',  # Tuberculosis of nervous system, unspecified
        'A18.01',  # Tuberculosis of spine
        'A18.02',  # Tuberculous arthritis of other joints
        'A18.03',  # Tuberculosis of other bones
        'A18.09',  # Other musculoskeletal tuberculosis
        'A18.10',  # Tuberculosis of genitourinary system, unspecified
        'A18.11',  # Tuberculosis of kidney and ureter
        'A18.12',  # Tuberculosis of bladder
        'A18.13',  # Tuberculosis of other urinary organs
        'A18.14',  # Tuberculosis of prostate
        'A18.15',  # Tuberculosis of other male genital organs
        'A18.16',  # Tuberculosis of cervix
        'A18.17',  # Tuberculous female pelvic inflammatory disease
        'A18.18',  # Tuberculosis of other female genital organs
        'A18.2',  # Tuberculous peripheral lymphadenopathy
        'A18.31',  # Tuberculous peritonitis
        'A18.32',  # Tuberculous enteritis
        'A18.39',  # Retroperitoneal tuberculosis
        'A18.4',  # Tuberculosis of skin and subcutaneous tissue
        'A18.50',  # Tuberculosis of eye, unspecified
        'A18.51',  # Tuberculous episcleritis
        'A18.52',  # Tuberculous keratitis
        'A18.53',  # Tuberculous chorioretinitis
        'A18.54',  # Tuberculous iridocyclitis
        'A18.59',  # Other tuberculosis of eye
        'A18.6',  # Tuberculosis of (inner) (middle) ear
        'A18.7',  # Tuberculosis of adrenal glands
        'A18.81',  # Tuberculosis of thyroid gland
        'A18.82',  # Tuberculosis of other endocrine glands
        'A18.83',  # Tuberculosis of digestive tract organs, not elsewhere classified
        'A18.84',  # Tuberculosis of heart
        'A18.85',  # Tuberculosis of spleen
        'A18.89',  # Tuberculosis of other sites
        'A19.0',  # Acute miliary tuberculosis of a single specified site
        'A19.1',  # Acute miliary tuberculosis of multiple sites
        'A19.2',  # Acute miliary tuberculosis, unspecified
        'A19.8',  # Other miliary tuberculosis
        'A19.9',  # Miliary tuberculosis, unspecified
        'B44.81',  # Allergic bronchopulmonary aspergillosis
        'D57.01',  # Hb-SS disease with acute chest syndrome
        'D57.03',  # Hb-SS disease with cerebral vascular involvement
        'D57.09',  # Hb-SS disease with crisis with other specified complication
        'D57.211',  # Sickle-cell/Hb-C disease with acute chest syndrome
        'D57.213',  # Sickle-cell/Hb-C disease with cerebral vascular involvement
        'D57.218',  # Sickle-cell/Hb-C disease with crisis with other specified complication
        'D57.411',  # Sickle-cell thalassemia, unspecified, with acute chest syndrome
        'D57.413',  # Sickle-cell thalassemia, unspecified, with cerebral vascular involvement
        'D57.418',  # Sickle-cell thalassemia, unspecified, with crisis with other specified complication
        'D57.811',  # Other sickle-cell disorders with acute chest syndrome
        'D57.813',  # Other sickle-cell disorders with cerebral vascular involvement
        'D57.818',  # Other sickle-cell disorders with crisis with other specified complication
        'D61.810',  # Antineoplastic chemotherapy induced pancytopenia
        'D61.811',  # Other drug-induced pancytopenia
        'D61.818',  # Other pancytopenia
        'D70.0',  # Congenital agranulocytosis
        'D70.1',  # Agranulocytosis secondary to cancer chemotherapy
        'D70.2',  # Other drug-induced agranulocytosis
        'D70.3',  # Neutropenia due to infection
        'D70.4',  # Cyclic neutropenia
        'D70.8',  # Other neutropenia
        'D70.9',  # Neutropenia, unspecified
        'D71',  # Functional disorders of polymorphonuclear neutrophils
        'D72.0',  # Genetic anomalies of leukocytes
        'D75.81',  # Myelofibrosis
        'D76.1',  # Hemophagocytic lymphohistiocytosis
        'D76.2',  # Hemophagocytic syndrome, infection-associated
        'D76.3',  # Other histiocytosis syndromes
        'D86.0',  # Sarcoidosis of lung
        'D86.1',  # Sarcoidosis of lymph nodes
        'D86.2',  # Sarcoidosis of lung with sarcoidosis of lymph nodes
        'E84.0',  # Cystic fibrosis with pulmonary manifestations
        'E84.11',  # Meconium ileus in cystic fibrosis
        'E84.19',  # Cystic fibrosis with other intestinal manifestations
        'E84.8',  # Cystic fibrosis with other manifestations
        'E84.9',  # Cystic fibrosis, unspecified
        'J22',  # Unspecified acute lower respiratory infection
        'J41.0',  # Simple chronic bronchitis
        'J41.1',  # Mucopurulent chronic bronchitis
        'J41.8',  # Mixed simple and mucopurulent chronic bronchitis
        'J42',  # Unspecified chronic bronchitis
        'J47.0',  # Bronchiectasis with acute lower respiratory infection
        'J47.1',  # Bronchiectasis with (acute) exacerbation
        'J47.9',  # Bronchiectasis, uncomplicated
        'J60',  # Coalworker's pneumoconiosis
        'J61',  # Pneumoconiosis due to asbestos and other mineral fibers
        'J62.0',  # Pneumoconiosis due to talc dust
        'J62.8',  # Pneumoconiosis due to other dust containing silica
        'J63.0',  # Aluminosis (of lung)
        'J63.1',  # Bauxite fibrosis (of lung)
        'J63.2',  # Berylliosis
        'J63.3',  # Graphite fibrosis (of lung)
        'J63.4',  # Siderosis
        'J63.5',  # Stannosis
        'J63.6',  # Pneumoconiosis due to other specified inorganic dusts
        'J64',  # Unspecified pneumoconiosis
        'J65',  # Pneumoconiosis associated with tuberculosis
        'J66.0',  # Byssinosis
        'J66.1',  # Flax-dressers' disease
        'J66.2',  # Cannabinosis
        'J66.8',  # Airway disease due to other specific organic dusts
        'J67.0',  # Farmer's lung
        'J67.1',  # Bagassosis
        'J67.2',  # Bird fancier's lung
        'J67.3',  # Suberosis
        'J67.4',  # Maltworker's lung
        'J67.5',  # Mushroom-worker's lung
        'J67.6',  # Maple-bark-stripper's lung
        'J67.7',  # Air conditioner and humidifier lung
        'J67.8',  # Hypersensitivity pneumonitis due to other organic dusts
        'J67.9',  # Hypersensitivity pneumonitis due to unspecified organic dust
        'J68.0',  # Bronchitis and pneumonitis due to chemicals, gases, fumes and vapors
        'J68.1',  # Pulmonary edema due to chemicals, gases, fumes and vapors
        'J68.2',  # Upper respiratory inflammation due to chemicals, gases, fumes and vapors, not elsewhere classified
        'J68.3',  # Other acute and subacute respiratory conditions due to chemicals, gases, fumes and vapors
        'J68.9',  # Unspecified respiratory condition due to chemicals, gases, fumes and vapors
        'J69.0',  # Pneumonitis due to inhalation of food and vomit
        'J69.1',  # Pneumonitis due to inhalation of oils and essences
        'J69.8',  # Pneumonitis due to inhalation of other solids and liquids
        'J70.0',  # Acute pulmonary manifestations due to radiation
        'J70.1',  # Chronic and other pulmonary manifestations due to radiation
        'J70.2',  # Acute drug-induced interstitial lung disorders
        'J70.3',  # Chronic drug-induced interstitial lung disorders
        'J70.4',  # Drug-induced interstitial lung disorders, unspecified
        'J70.5',  # Respiratory conditions due to smoke inhalation
        'J70.8',  # Respiratory conditions due to other specified external agents
        'J70.9',  # Respiratory conditions due to unspecified external agent
        'J80',  # Acute respiratory distress syndrome
        'J81.0',  # Acute pulmonary edema
        'J81.1',  # Chronic pulmonary edema
        'J82',  # Pulmonary eosinophilia, not elsewhere classified
        'J82.81',  # Chronic eosinophilic pneumonia
        'J82.82',  # Acute eosinophilic pneumonia
        'J82.83',  # Eosinophilic asthma
        'J82.89',  # Other pulmonary eosinophilia, not elsewhere classified
        'J84.01',  # Alveolar proteinosis
        'J84.02',  # Pulmonary alveolar microlithiasis
        'J84.03',  # Idiopathic pulmonary hemosiderosis
        'J84.09',  # Other alveolar and parieto-alveolar conditions
        'J84.10',  # Pulmonary fibrosis, unspecified
        'J84.111',  # Idiopathic interstitial pneumonia, not otherwise specified
        'J84.112',  # Idiopathic pulmonary fibrosis
        'J84.113',  # Idiopathic non-specific interstitial pneumonitis
        'J84.114',  # Acute interstitial pneumonitis
        'J84.115',  # Respiratory bronchiolitis interstitial lung disease
        'J84.116',  # Cryptogenic organizing pneumonia
        'J84.117',  # Desquamative interstitial pneumonia
        'J84.17',  # Other interstitial pulmonary diseases with fibrosis in diseases classified elsewhere
        'J84.170',  # Interstitial lung disease with progressive fibrotic phenotype in diseases classified elsewhere
        'J84.178',  # Other interstitial pulmonary diseases with fibrosis in diseases classified elsewhere
        'J84.2',  # Lymphoid interstitial pneumonia
        'J84.81',  # Lymphangioleiomyomatosis
        'J84.82',  # Adult pulmonary Langerhans cell histiocytosis
        'J84.83',  # Surfactant mutations of the lung
        'J84.841',  # Neuroendocrine cell hyperplasia of infancy
        'J84.842',  # Pulmonary interstitial glycogenosis
        'J84.843',  # Alveolar capillary dysplasia with vein misalignment
        'J84.848',  # Other interstitial lung diseases of childhood
        'J84.89',  # Other specified interstitial pulmonary diseases
        'J84.9',  # Interstitial pulmonary disease, unspecified
        'J85.0',  # Gangrene and necrosis of lung
        'J85.1',  # Abscess of lung with pneumonia
        'J85.2',  # Abscess of lung without pneumonia
        'J85.3',  # Abscess of mediastinum
        'J86.0',  # Pyothorax with fistula
        'J86.9',  # Pyothorax without fistula
        'J90',  # Pleural effusion, not elsewhere classified
        'J91.0',  # Malignant pleural effusion
        'J91.8',  # Pleural effusion in other conditions classified elsewhere
        'J92.0',  # Pleural plaque with presence of asbestos
        'J92.9',  # Pleural plaque without asbestos
        'J93.0',  # Spontaneous tension pneumothorax
        'J93.11',  # Primary spontaneous pneumothorax
        'J93.12',  # Secondary spontaneous pneumothorax
        'J93.81',  # Chronic pneumothorax
        'J93.82',  # Other air leak
        'J93.83',  # Other pneumothorax
        'J93.9',  # Pneumothorax, unspecified
        'J94.0',  # Chylous effusion
        'J94.1',  # Fibrothorax
        'J94.2',  # Hemothorax
        'J94.8',  # Other specified pleural conditions
        'J94.9',  # Pleural condition, unspecified
        'J95.00',  # Unspecified tracheostomy complication
        'J95.01',  # Hemorrhage from tracheostomy stoma
        'J95.02',  # Infection of tracheostomy stoma
        'J95.03',  # Malfunction of tracheostomy stoma
        'J95.04',  # Tracheo-esophageal fistula following tracheostomy
        'J95.09',  # Other tracheostomy complication
        'J95.1',  # Acute pulmonary insufficiency following thoracic surgery
        'J95.2',  # Acute pulmonary insufficiency following nonthoracic surgery
        'J95.3',  # Chronic pulmonary insufficiency following surgery
        'J95.4',  # Chemical pneumonitis due to anesthesia
        'J95.5',  # Postprocedural subglottic stenosis
        'J95.61',  # Intraoperative hemorrhage and hematoma of a respiratory system organ or structure complicating a respiratory system procedure
        'J95.62',  # Intraoperative hemorrhage and hematoma of a respiratory system organ or structure complicating other procedure
        'J95.71',  # Accidental puncture and laceration of a respiratory system organ or structure during a respiratory system procedure
        'J95.72',  # Accidental puncture and laceration of a respiratory system organ or structure during other procedure
        'J95.811',  # Postprocedural pneumothorax
        'J95.812',  # Postprocedural air leak
        'J95.821',  # Acute postprocedural respiratory failure
        'J95.822',  # Acute and chronic postprocedural respiratory failure
        'J95.830',  # Postprocedural hemorrhage of a respiratory system organ or structure following a respiratory system procedure
        'J95.831',  # Postprocedural hemorrhage of a respiratory system organ or structure following other procedure
        'J95.84',  # Transfusion-related acute lung injury (TRALI)
        'J95.850',  # Mechanical complication of respirator
        'J95.851',  # Ventilator associated pneumonia
        'J95.859',  # Other complication of respirator [ventilator]
        'J95.860',  # Postprocedural hematoma of a respiratory system organ or structure following a respiratory system procedure
        'J95.861',  # Postprocedural hematoma of a respiratory system organ or structure following other procedure
        'J95.862',  # Postprocedural seroma of a respiratory system organ or structure following a respiratory system procedure
        'J95.863',  # Postprocedural seroma of a respiratory system organ or structure following other procedure
        'J95.88',  # Other intraoperative complications of respiratory system, not elsewhere classified
        'J95.89',  # Other postprocedural complications and disorders of respiratory system, not elsewhere classified
        'J96.00',  # Acute respiratory failure, unspecified whether with hypoxia or hypercapnia
        'J96.01',  # Acute respiratory failure with hypoxia
        'J96.02',  # Acute respiratory failure with hypercapnia
        'J96.10',  # Chronic respiratory failure, unspecified whether with hypoxia or hypercapnia
        'J96.11',  # Chronic respiratory failure with hypoxia
        'J96.12',  # Chronic respiratory failure with hypercapnia
        'J96.20',  # Acute and chronic respiratory failure, unspecified whether with hypoxia or hypercapnia
        'J96.21',  # Acute and chronic respiratory failure with hypoxia
        'J96.22',  # Acute and chronic respiratory failure with hypercapnia
        'J96.90',  # Respiratory failure, unspecified, unspecified whether with hypoxia or hypercapnia
        'J96.91',  # Respiratory failure, unspecified with hypoxia
        'J96.92',  # Respiratory failure, unspecified with hypercapnia
        'J98.51',  # Mediastinitis
        'J98.6',  # Disorders of diaphragm
        'J99',  # Respiratory disorders in diseases classified elsewhere
        'M05.10',  # Rheumatoid lung disease with rheumatoid arthritis of unspecified site
        'M05.111',  # Rheumatoid lung disease with rheumatoid arthritis of right shoulder
        'M05.112',  # Rheumatoid lung disease with rheumatoid arthritis of left shoulder
        'M05.119',  # Rheumatoid lung disease with rheumatoid arthritis of unspecified shoulder
        'M05.121',  # Rheumatoid lung disease with rheumatoid arthritis of right elbow
        'M05.122',  # Rheumatoid lung disease with rheumatoid arthritis of left elbow
        'M05.129',  # Rheumatoid lung disease with rheumatoid arthritis of unspecified elbow
        'M05.131',  # Rheumatoid lung disease with rheumatoid arthritis of right wrist
        'M05.132',  # Rheumatoid lung disease with rheumatoid arthritis of left wrist
        'M05.139',  # Rheumatoid lung disease with rheumatoid arthritis of unspecified wrist
        'M05.141',  # Rheumatoid lung disease with rheumatoid arthritis of right hand
        'M05.142',  # Rheumatoid lung disease with rheumatoid arthritis of left hand
        'M05.149',  # Rheumatoid lung disease with rheumatoid arthritis of unspecified hand
        'M05.151',  # Rheumatoid lung disease with rheumatoid arthritis of right hip
        'M05.152',  # Rheumatoid lung disease with rheumatoid arthritis of left hip
        'M05.159',  # Rheumatoid lung disease with rheumatoid arthritis of unspecified hip
        'M05.161',  # Rheumatoid lung disease with rheumatoid arthritis of right knee
        'M05.162',  # Rheumatoid lung disease with rheumatoid arthritis of left knee
        'M05.169',  # Rheumatoid lung disease with rheumatoid arthritis of unspecified knee
        'M05.171',  # Rheumatoid lung disease with rheumatoid arthritis of right ankle and foot
        'M05.172',  # Rheumatoid lung disease with rheumatoid arthritis of left ankle and foot
        'M05.179',  # Rheumatoid lung disease with rheumatoid arthritis of unspecified ankle and foot
        'M05.19',  # Rheumatoid lung disease with rheumatoid arthritis of multiple sites
        'M30.1',  # Polyarteritis with lung involvement [Churg-Strauss]
        'M32.13',  # Lung involvement in systemic lupus erythematosus
        'M33.01',  # Juvenile dermatomyositis with respiratory involvement
        'M33.11',  # Other dermatomyositis with respiratory involvement
        'M33.21',  # Polymyositis with respiratory involvement
        'M33.91',  # Dermatopolymyositis, unspecified with respiratory involvement
        'M34.81',  # Systemic sclerosis with lung involvement
        'M35.02',  # Sicca syndrome with lung involvement
        'O98.011',  # Tuberculosis complicating pregnancy, first trimester
        'O98.012',  # Tuberculosis complicating pregnancy, second trimester
        'O98.013',  # Tuberculosis complicating pregnancy, third trimester
        'O98.019',  # Tuberculosis complicating pregnancy, unspecified trimester
        'O98.02',  # Tuberculosis complicating childbirth
        'O98.03',  # Tuberculosis complicating the puerperium
        'P27.0',  # Wilson-Mikity syndrome
        'P27.1',  # Bronchopulmonary dysplasia originating in the perinatal period
        'P27.8',  # Other chronic respiratory diseases originating in the perinatal period
        'P27.9',  # Unspecified chronic respiratory disease originating in the perinatal period
        'Q25.45',  # Double aortic arch
        'Q25.47',  # Right aortic arch
        'Q25.48',  # Anomalous origin of subclavian artery
        'Q30.0',  # Choanal atresia
        'Q30.1',  # Agenesis and underdevelopment of nose
        'Q30.2',  # Fissured, notched and cleft nose
        'Q30.3',  # Congenital perforated nasal septum
        'Q30.8',  # Other congenital malformations of nose
        'Q30.9',  # Congenital malformation of nose, unspecified
        'Q31.0',  # Web of larynx
        'Q31.1',  # Congenital subglottic stenosis
        'Q31.2',  # Laryngeal hypoplasia
        'Q31.3',  # Laryngocele
        'Q31.5',  # Congenital laryngomalacia
        'Q31.8',  # Other congenital malformations of larynx
        'Q31.9',  # Congenital malformation of larynx, unspecified
        'Q32.0',  # Congenital tracheomalacia
        'Q32.1',  # Other congenital malformations of trachea
        'Q32.2',  # Congenital bronchomalacia
        'Q32.3',  # Congenital stenosis of bronchus
        'Q32.4',  # Other congenital malformations of bronchus
        'Q33.0',  # Congenital cystic lung
        'Q33.1',  # Accessory lobe of lung
        'Q33.2',  # Sequestration of lung
        'Q33.3',  # Agenesis of lung
        'Q33.4',  # Congenital bronchiectasis
        'Q33.5',  # Ectopic tissue in lung
        'Q33.6',  # Congenital hypoplasia and dysplasia of lung
        'Q33.8',  # Other congenital malformations of lung
        'Q33.9',  # Congenital malformation of lung, unspecified
        'Q34.0',  # Anomaly of pleura
        'Q34.1',  # Congenital cyst of mediastinum
        'Q34.8',  # Other specified congenital malformations of respiratory system
        'Q34.9',  # Congenital malformation of respiratory system, unspecified
        'Q39.0',  # Atresia of esophagus without fistula
        'Q39.1',  # Atresia of esophagus with tracheo-esophageal fistula
        'Q39.2',  # Congenital tracheo-esophageal fistula without atresia
        'Q39.3',  # Congenital stenosis and stricture of esophagus
        'Q39.4',  # Esophageal web
        'Q89.01',  # Asplenia (congenital)
        'Q89.3',  # Situs inversus
    }
    SNOMEDCT = {
        '599006',  # Chronic pneumothorax (disorder)
        '1259003',  # Schistosis (disorder)
        '4225003',  # Tuberculosis of nose (disorder)
        '4445009',  # Tuberculosis of genitourinary system (disorder)
        '4501007',  # Tuberculous ascites (disorder)
        '7343008',  # Liparitosis (disorder)
        '7444001',  # Tuberculosis of rectum (disorder)
        '8250007',  # Tuberculosis of subcutaneous cellular tissue (disorder)
        '8832006',  # Tuberculosis orificialis of anus (disorder)
        '10041001',  # Tracheal stenosis following tracheostomy (disorder)
        '10528009',  # Lupus vulgaris (disorder)
        '10706006',  # Tuberculosis of central nervous system (disorder)
        '10893003',  # Tuberculous adenitis (disorder)
        '11676005',  # Tuberculous leptomeningitis (disorder)
        '11689004',  # Linear atelectasis (disorder)
        '12137005',  # Tuberculous enteritis (disorder)
        '12677003',  # Tuberculosis of brain (disorder)
        '12985005',  # Tuberculosis of seminal vesicle (disorder)
        '13570003',  # Tuberculous tenosynovitis (disorder)
        '14188007',  # Tuberculosis of hip (disorder)
        '14527007',  # Tuberculous empyema (disorder)
        '15202009',  # Tuberculoma (disorder)
        '15284007',  # Tuberculosis of esophagus (disorder)
        '16409009',  # Tuberculosis of retroperitoneal lymph nodes (disorder)
        '16537000',  # Compression atelectasis (disorder)
        '16632002',  # Hemopneumothorax (disorder)
        '17136009',  # Tuberculosis of knee (disorder)
        '17653001',  # Tuberculosis of bones and/or joints (disorder)
        '20372007',  # Tuberculosis of prostate (disorder)
        '21504004',  # Pressure collapse of lung after anesthesia AND/OR sedation in labor AND/OR delivery (disorder)
        '22607003',  # Asbestosis (disorder)
        '22990009',  # Chronic tuberculosis (disorder)
        '23022004',  # Tuberculous bronchiectasis (disorder)
        '24235005',  # Tuberculous hydrothorax (disorder)
        '24810006',  # Tuberculosis orificialis of mouth (disorder)
        '25042006',  # Tracheostomy sepsis (disorder)
        '25629007',  # Acute tuberculosis (disorder)
        '25738004',  # Tuberculous synovitis (disorder)
        '25831001',  # Tuberculosis of nasal septum (disorder)
        '26935004',  # Tuberculosis of thyroid gland (disorder)
        '28085001',  # Empyema with bronchopleural fistula (disorder)
        '28295001',  # Middle lobe syndrome (disorder)
        '28399005',  # Tuberculosis of spleen (disorder)
        '29422001',  # Coal workers' pneumoconiosis (disorder)
        '29731002',  # Tuberculous pneumothorax (disorder)
        '30437004',  # Empyema with bronchocutaneous fistula (disorder)
        '31112008',  # Tuberculous meningoencephalitis (disorder)
        '31892009',  # Hemothorax (disorder)
        '32139003',  # Mixed dust pneumoconiosis (disorder)
        '32268008',  # Tuberculosis of bladder (disorder)
        '33072005',  # Traumatic hemothorax with open wound into thorax (disorder)
        '33325001',  # Compensatory emphysema (disorder)
        '33548005',  # Anthracosilicosis (disorder)
        '33631007',  # Pyopneumothorax (disorder)
        '33893009',  # Tuberculosis verrucosa cutis (disorder)
        '34004002',  # Siderosilicosis (disorder)
        '34286007',  # Empyema with hepatopleural fistula (disorder)
        '34320001',  # Tuberculous myelitis (disorder)
        '35585004',  # Tuberculous endometritis (disorder)
        '35786001',  # Tuberculoma of meninges (disorder)
        '35984006',  # Tuberculosis of vertebral column (disorder)
        '36696005',  # Kaolinosis (disorder)
        '37133005',  # Tuberculous pyelitis (disorder)
        '37711000',  # Cadmium pneumonitis (disorder)
        '37981002',  # Allergic bronchopulmonary aspergillosis (disorder)
        '38115001',  # Tuberculosis of spinal meninges (disorder)
        '38279006',  # Tuberculosis of bone (disorder)
        '38420005',  # Tuberculosis of large intestine (disorder)
        '39306006',  # Tuberculosis of anus (disorder)
        '40218008',  # Carbon electrode makers' pneumoconiosis (disorder)
        '40258005',  # Tuberculosis of testis (disorder)
        '40640008',  # Massive fibrosis of lung co-occurrent and due to silicosis (disorder)
        '40779009',  # Discoid atelectasis (disorder)
        '42434002',  # Pneumohemothorax due to trauma (disorder)
        '42458003',  # Hemothorax due to trauma (disorder)
        '42522007',  # Abscess of mediastinum (disorder)
        '42908004',  # Primary atelectasis, in perinatal period (disorder)
        '44182001',  # Tuberculous necrosis of bone (disorder)
        '44323002',  # Tuberculosis of kidney (disorder)
        '44547005',  # Chalicosis (disorder)
        '44572005',  # Tuberculous peritonitis (disorder)
        '44599007',  # Traumatic hemothorax without open wound into thorax (disorder)
        '46621007',  # Atelectasis (disorder)
        '47292005',  # Tuberculosis of myocardium (disorder)
        '47515009',  # Simple silicosis (disorder)
        '47604008',  # Miliary tuberculosis (disorder)
        '48225000',  # Tuberculous disseminated chorioretinitis (disorder)
        '48475001',  # Disorder of diaphragm (disorder)
        '49107007',  # Tuberculosis of eye (disorder)
        '49558004',  # Tuberculous salpingitis (disorder)
        '49807009',  # Tuberculous dactylitis (disorder)
        '49840000',  # Complicated silicosis (disorder)
        '50076003',  # Baritosis (disorder)
        '50589003',  # Silver polishers' lung disease (disorder)
        '50694005',  # Empyema with mediastinal fistula (disorder)
        '51014003',  # Reinfection tuberculosis (disorder)
        '51068008',  # Adult bronchiectasis (disorder)
        '51277007',  # Stannosis (disorder)
        '52721006',  # Tuberculosis of nasal sinus (disorder)
        '52987001',  # Tuberculosis of endocardium (disorder)
        '54084005',  # Cervical tuberculous lymphadenitis (disorder)
        '54959009',  # Perinatal atelectasis (disorder)
        '55419007',  # Tuberculosis of glottis (disorder)
        '55448001',  # Tracheoesophageal fistula following tracheostomy (disorder)
        '56498009',  # Tuberculosis of nasopharynx (disorder)
        '56717001',  # Tuberculosis (disorder)
        '58374007',  # Acute diffuse tuberculosis (disorder)
        '58437007',  # Tuberculosis of meninges (disorder)
        '58525007',  # Obstructive atelectasis (disorder)
        '58554001',  # Empyema of pleura (disorder)
        '58691003',  # Antimony pneumoconiosis (disorder)
        '58764007',  # Tuberculous osteomyelitis (disorder)
        '59113005',  # Perinatal secondary atelectasis (disorder)
        '59190009',  # Traumatic pneumohemothorax without open wound into thorax (disorder)
        '60136008',  # Tuberculosis of intestines (disorder)
        '61448007',  # Focal atelectasis (disorder)
        '62371005',  # Pulmonary siderosis (disorder)
        '62494007',  # Chronic miliary tuberculosis (disorder)
        '62819009',  # Tuberculous cervicitis (disorder)
        '63309002',  # Primary tuberculosis (disorder)
        '63808004',  # Tuberculous chancre (disorder)
        '64631008',  # Fullers' earth disease (disorder)
        '66151009',  # Perinatal complete pulmonary collapse (disorder)
        '66291003',  # Tuberculous phlyctenular keratoconjunctivitis (disorder)
        '66986005',  # Tuberculosis of skin (disorder)
        '67256000',  # Tuberculosis of pericardium (disorder)
        '68033004',  # Tracheostomy complication (disorder)
        '70341005',  # Tuberculous laryngitis (disorder)
        '71375000',  # Tuberculosis of mastoid process (disorder)
        '71943007',  # Chronic granulomatous tuberculosis (disorder)
        '72270005',  # Collagenous pneumoconiosis (disorder)
        '72807002',  # Tuberculous polyserositis (disorder)
        '73452002',  # Abscess of lung (disorder)
        '73477001',  # Traumatic pneumohemothorax with open wound into thorax (disorder)
        '74181004',  # Tuberculosis of female genital organs (disorder)
        '74387008',  # Tuberculosis of hilar lymph nodes (disorder)
        '74610006',  # Tuberculous erythema nodosum (disorder)
        '76157009',  # Bituminosis (disorder)
        '76630008',  # Fibrinopurulent pleurisy (disorder)
        '77038006',  # Tuberculosis of peripheral lymph nodes (disorder)
        '77690003',  # Interstitial emphysema of lung (disorder)
        '78436002',  # Tuberculosis of intrathoracic lymph nodes (disorder)
        '78887005',  # Seropurulent pleurisy (disorder)
        '79158003',  # Tuberculous interstitial keratitis (disorder)
        '80003002',  # Tuberculous pneumonia (disorder)
        '80010008',  # Isolated bronchial tuberculosis (disorder)
        '80602006',  # Nodular tuberculosis of lung (disorder)
        '81133003',  # Tuberculosis of bones of lower extremity (disorder)
        '81359005',  # Tuberculosis of ureter (disorder)
        '81554001',  # Tuberculosis of lung with involvement of bronchus (disorder)
        '82495004',  # Tuberculosis of mediastinal lymph nodes (disorder)
        '82796002',  # Scrofuloderma (disorder)
        '83652003',  # Tuberculosis of epididymis (disorder)
        '83784000',  # Tuberculosis of mediastinum (disorder)
        '84194006',  # Tuberculous oophoritis (disorder)
        '84288007',  # Tuberculous episcleritis (disorder)
        '85692003',  # Tuberculosis of tracheobronchial lymph nodes (disorder)
        '85940005',  # Tuberculous otitis media (disorder)
        '86263001',  # Cobaltosis (disorder)
        '87909002',  # Hard metal pneumoconiosis (disorder)
        '88356006',  # Primary tuberculous complex (disorder)
        '88687001',  # Manganese pneumonitis (disorder)
        '90117007',  # Tuberculous fibrosis of lung (disorder)
        '90302003',  # Tuberculosis of cerebral meninges (disorder)
        '90623003',  # Aluminosis of lung (disorder)
        '91036007',  # Complete atelectasis (disorder)
        '91483004',  # Tuberculous encephalitis (disorder)
        '111466004',  # Perinatal partial atelectasis (disorder)
        '111825008',  # Tuberculosis of conjunctiva (disorder)
        '111830007',  # Tuberculosis of urinary organs (disorder)
        '111832004',  # Primary progressive tuberculosis (disorder)
        '123593009',  # Complete atelectasis of left lung (disorder)
        '123594003',  # Complete atelectasis of right lung (disorder)
        '123712000',  # Acute atelectasis (disorder)
        '123713005',  # Chronic atelectasis (disorder)
        '154283005',  # Pulmonary tuberculosis (disorder)
        '182159002',  # Disseminated tuberculosis (disorder)
        '185113005',  # Tuberculosis of adrenal glands (disorder)
        '186172004',  # Tuberculous pleurisy in primary progressive tuberculosis (disorder)
        '186175002',  # Infiltrative lung tuberculosis (disorder)
        '186177005',  # Tuberculosis of lung with cavitation (disorder)
        '186178000',  # Tuberculosis of bronchus (disorder)
        '186182003',  # Tuberculosis of pleura (disorder)
        '186192006',  # Respiratory tuberculosis, bacteriologically and histologically confirmed (disorder)
        '186193001',  # Tuberculosis of lung, confirmed by sputum microscopy with or without culture (disorder)
        '186194007',  # Tuberculosis of lung, confirmed by culture only (disorder)
        '186195008',  # Tuberculosis of lung, confirmed histologically (disorder)
        '186199002',  # Tuberculosis of larynx, trachea and bronchus, confirmed bacteriologically and histologically (disorder)
        '186200004',  # Tuberculous pleurisy, confirmed bacteriologically and histologically (disorder)
        '186201000',  # Primary respiratory tuberculosis, confirmed bacteriologically and histologically (disorder)
        '186202007',  # Respiratory tuberculosis, not confirmed bacteriologically or histologically (disorder)
        '186203002',  # Tuberculosis of lung, bacteriologically and histologically negative (disorder)
        '186204008',  # Tuberculosis of lung, bacteriological and histological examination not done (disorder)
        '186217006',  # Tuberculous abscess of brain (disorder)
        '186225008',  # Tuberculosis of intestines, peritoneum and mesenteric glands (disorder)
        '186231006',  # Tuberculosis limb bones - Tuberculous dactylitis (disorder)
        '186259007',  # Scrofulous tuberculous abscess (disorder)
        '186263000',  # Tuberculous chorioretinitis (disorder)
        '186266008',  # Tuberculous chronic iridocyclitis (disorder)
        '186267004',  # Tuberculous keratoconjunctivitis (disorder)
        '186269001',  # Tuberculosis of ear (disorder)
        '186270000',  # Tuberculous Addison's disease (disorder)
        '186272008',  # Tuberculosis of stomach (disorder)
        '186273003',  # Tuberculosis of liver (disorder)
        '186276006',  # Acute miliary tuberculosis (disorder)
        '186278007',  # Acute miliary tuberculosis of multiple sites (disorder)
        '190710003',  # Tuberculous arthritis (disorder)
        '192745001',  # Tuberculous intracranial abscess (disorder)
        '194908003',  # Acute tuberculous pericarditis (disorder)
        '194949003',  # Acute myocarditis - tuberculous (disorder)
        '195953005',  # Mixed simple and mucopurulent chronic bronchitis (disorder)
        '195985008',  # Post-infective bronchiectasis (disorder)
        '196017002',  # Pneumoconiosis associated with tuberculosis (disorder)
        '196067009',  # Loculated empyema (disorder)
        '196102003',  # Spontaneous tension pneumothorax (disorder)
        '196107009',  # Abscess of lung and mediastinum (disorder)
        '196112005',  # Abscess of lung with pneumonia (disorder)
        '196133001',  # Lung disease with systemic sclerosis (disorder)
        '196136009',  # Lung disease co-occurrent with polymyositis (disorder)
        '196138005',  # Lung disease with systemic lupus erythematosus (disorder)
        '196143003',  # Postoperative atelectasis (disorder)
        '196188002',  # Acute pulmonary insufficiency following thoracic surgery (disorder)
        '196189005',  # Acute pulmonary insufficiency following nonthoracic surgery (disorder)
        '196190001',  # Chronic pulmonary insufficiency following surgery (disorder)
        '196191002',  # Postprocedural subglottic stenosis (disorder)
        '198241002',  # Female tuberculous pelvic inflammatory disease (disorder)
        '199179007',  # Maternal tuberculosis in the puerperium - baby delivered during current episode of care (disorder)
        '199180005',  # Maternal tuberculosis during pregnancy - baby not yet delivered (disorder)
        '199181009',  # Maternal tuberculosis in the puerperium - baby delivered during previous episode of care (disorder)
        '202994006',  # Tuberculous infection of tendon sheath (disorder)
        '203263006',  # Tuberculosis of cervical spine (disorder)
        '203264000',  # Tuberculosis of thoracic spine (disorder)
        '203265004',  # Tuberculosis of lumbar spine (disorder)
        '203267007',  # Tuberculosis of limb bones (disorder)
        '203269005',  # Tuberculosis of the upper arm bone (disorder)
        '203270006',  # Tuberculosis of the forearm bone (disorder)
        '203271005',  # Tuberculosis of the pelvic and/or thigh bones (disorder)
        '203272003',  # Tuberculosis of the lower leg bone (disorder)
        '203274002',  # Tuberculosis of multiple limb bones (disorder)
        '203278004',  # Tuberculosis of the bones of the shoulder region (disorder)
        '203279007',  # Tuberculosis of the bones of the hand (disorder)
        '203280005',  # Tuberculosis of the bones of the ankle and/or foot (disorder)
        '203282002',  # Tuberculosis of the bones of multiple sites (disorder)
        '213580002',  # Caseating tuberculoid granuloma (disorder)
        '233628009',  # Acquired bronchiectasis (disorder)
        '233629001',  # Idiopathic bronchiectasis (disorder)
        '233631005',  # Toxin-induced bronchiectasis (disorder)
        '233634002',  # Post-lung transplantation bronchiectasis (disorder)
        '233649005',  # Tuberculous chylothorax (disorder)
        '233658003',  # Pleural plaque (disorder)
        '233659006',  # Asbestos-induced pleural plaque (disorder)
        '233749003',  # Complicated pneumoconiosis (disorder)
        '233751004',  # Metal pneumoconiosis (disorder)
        '233754007',  # Cerium pneumoconiosis (disorder)
        '233755008',  # Nickel pneumoconiosis (disorder)
        '233756009',  # Thorium pneumoconiosis (disorder)
        '233757000',  # Zirconium pneumoconiosis (disorder)
        '233758005',  # Mica pneumoconiosis (disorder)
        '233759002',  # Mixed mineral dust pneumoconiosis (disorder)
        '233760007',  # Acute silicosis (disorder)
        '233761006',  # Subacute silicosis (disorder)
        '233762004',  # Chronic silicosis (disorder)
        '233763009',  # Silicotuberculosis (disorder)
        '233794009',  # Tuberculous bronchopleural fistula (disorder)
        '234090004',  # Tuberculous mesenteric adenitis (disorder)
        '235067001',  # Oral tuberculosis (disorder)
        '235694001',  # Tuberculous duodenitis (disorder)
        '236602004',  # Tuberculous stricture of ureter (disorder)
        '236684001',  # Tuberculous urethritis (disorder)
        '236770001',  # Tuberculous epididymo-orchitis (disorder)
        '237445009',  # Tuberculosis of breast (disorder)
        '238430004',  # Cutaneous complication of bacillus Calmette-Guerin immunization (disorder)
        '238431000',  # Bacillus Calmette-Guerin granuloma (disorder)
        '238432007',  # Bacillus Calmette-Guerin ulcer (disorder)
        '240376003',  # Tuberculosis of gastrointestinal tract (disorder)
        '240379005',  # Tuberculosis of male genital organs (disorder)
        '240380008',  # Acute tuberculous ulcer (disorder)
        '240381007',  # Acute miliary cutaneous tuberculosis (disorder)
        '240382000',  # Cutaneous tuberculous gumma (disorder)
        '266355005',  # Bullous emphysema with collapse (disorder)
        '266366003',  # Lung disease due to external agents (disorder)
        '267317007',  # Tuberculosis in pregnancy (disorder)
        '271423008',  # Tuberculosis of skin and subcutaneous tissue (disorder)
        '271503005',  # Pleural empyema with fistula (disorder)
        '271504004',  # Pleural empyema with no fistula (disorder)
        '281623008',  # Genital tuberculosis (disorder)
        '297188009',  # Clotted hemothorax (disorder)
        '301836007',  # Tuberculosis of colon (disorder)
        '302131003',  # Tuberculosis of heart (disorder)
        '307202004',  # Hemorrhagic pleural effusion (disorder)
        '314029000',  # Tuberculous uveitis (disorder)
        '326542006',  # Traumatic atelectasis (disorder)
        '398640008',  # Rheumatoid pneumoconiosis (disorder)
        '402973006',  # Bacillus Calmette-Guerin-induced lupus vulgaris (disorder)
        '402974000',  # Bacillus Calmette-Guerin-induced scrofuloderma (disorder)
        '402975004',  # Bacillus Calmette-Guerin-induced tuberculide (disorder)
        '405576001',  # Nontraumatic hemothorax (disorder)
        '413556004',  # Antibiotic resistant tuberculosis (disorder)
        '413852006',  # Ciprofloxacin resistant tuberculosis (disorder)
        '414146004',  # Ethambutol resistant tuberculosis (disorder)
        '414149006',  # Ethionamide resistant tuberculosis (disorder)
        '414546009',  # Isoniazid resistant tuberculosis (disorder)
        '415222009',  # Pyrazinamide resistant tuberculosis (disorder)
        '415345001',  # Rifampicin resistant tuberculosis (disorder)
        '415622003',  # Streptomycin resistant tuberculosis (disorder)
        '416265003',  # Tuberculoma of brain (disorder)
        '416484003',  # Sickle cell-hemoglobin E disease with crisis (disorder)
        '416826005',  # Sickle cell-thalassemia disease with crisis (disorder)
        '416903004',  # Tuberculoma of spinal cord (disorder)
        '417279003',  # Hemoglobin S sickling disorder with crisis (disorder)
        '417425009',  # Hemoglobin SS disease with crisis (disorder)
        '417484006',  # Tuberculous abscess of spinal cord (disorder)
        '417517009',  # Sickle cell-hemoglobin C disease with crisis (disorder)
        '417748003',  # Sickle cell-hemoglobin D disease with crisis (disorder)
        '418750009',  # On examination - atelectasis (disorder)
        '421571007',  # Tuberculosis associated with acquired immunodeficiency syndrome (disorder)
        '422481005',  # Extreme drug resistant tuberculosis (disorder)
        '423092005',  # Multidrug resistant tuberculosis (disorder)
        '423709000',  # Drug resistant tuberculosis (disorder)
        '423997002',  # Tuberculosis, extrapulmonary (disorder)
        '427099000',  # Active tuberculosis (disorder)
        '428782008',  # Postoperative hemothorax (disorder)
        '440206000',  # Hemoglobin SS disease with vasoocclusive crisis (disorder)
        '441770001',  # Tuberculosis of external body orifice (disorder)
        '443378001',  # Lady Windermere syndrome (disorder)
        '444108000',  # Acute sickle cell splenic sequestration crisis (disorder)
        '446123007',  # Tuberculous abscess (disorder)
        '446543007',  # Tuberculous abscess of lung (disorder)
        '446753005',  # Tuberculosis of oropharynx (disorder)
        '446926000',  # Tuberculous abscess of sacrum (disorder)
        '446946005',  # Reinfection pulmonary tuberculosis (disorder)
        '446986002',  # Tuberculous pleural effusion (disorder)
        '447006007',  # Relapse pulmonary tuberculosis (disorder)
        '447012002',  # Tuberculosis of spinal cord (disorder)
        '447253004',  # Tuberculous arachnoiditis (disorder)
        '447330002',  # Tuberculosis of abdomen (disorder)
        '447332005',  # Tuberculous abscess of epidural space (disorder)
        '447371008',  # Tuberculosis of ileum (disorder)
        '447399004',  # Tuberculous abscess of neck (disorder)
        '448370006',  # Round atelectasis (disorder)
        '449925002',  # Tuberculosis of bone of upper limb (disorder)
        '473426009',  # Primary tuberculous complex confirmed by culture (disorder)
        '473427000',  # Primary tuberculous complex confirmed histologically (disorder)
        '473428005',  # Primary tuberculous complex confirmed by microscopic examination (disorder)
        '473429002',  # Tuberculoma of spinal cord confirmed (disorder)
        '473430007',  # Primary tuberculous complex confirmed (disorder)
        '473431006',  # Primary tuberculous complex with bacteriological or histological examination results unknown (disorder)
        '699326004',  # Accidental puncture of pleura (disorder)
        '700272008',  # Tuberculosis of respiratory system (disorder)
        '700273003',  # Isolated tracheobronchial tuberculosis (disorder)
        '707149004',  # Mycobacterial lymphadenitis (disorder)
        '708025003',  # Pyopneumothorax following infection by Coccidioides (disorder)
        '710106005',  # Extensively drug resistant tuberculosis (disorder)
        '714394004',  # Lupus vulgaris of nose (disorder)
        '717697005',  # Tuberculous ulceration of vulva (disorder)
        '724502006',  # Respiratory disorder co-occurrent and due to microscopic polyangiitis (disorder)
        '724541008',  # Acute miliary tuberculosis of single site (disorder)
        '733497009',  # Chronic suppuration of bronchus (disorder)
        '733835007',  # Extrapulmonary tuberculosis co-occurrent with human immunodeficiency virus infection (disorder)
        '735521001',  # Human immunodeficiency virus World Health Organization 2007 stage 1 co-occurrent with tuberculosis (disorder)
        '735523003',  # Human immunodeficiency virus World Health Organization 2007 stage 2 co-occurrent with tuberculosis (disorder)
        '735525005',  # Human immunodeficiency virus World Health Organization 2007 stage 3 co-occurrent with tuberculosis (disorder)
        '735527002',  # Human immunodeficiency virus World Health Organization 2007 stage 4 co-occurrent with tuberculosis (disorder)
        '770674007',  # Tubercular lesion of lung (disorder)
        '782761005',  # Subacute invasive pulmonary aspergillosis (disorder)
        '789056000',  # Tuberculosis of inner ear (disorder)
        '789057009',  # Tuberculosis of middle ear (disorder)
        '860803004',  # Hydrocephalus due to tuberculosis of brain (disorder)
        '129211000119108',  # Isoniazid resistant tuberculosis of lung (disorder)
        '311671000119100',  # Juvenile dermatomyositis co-occurrent with respiratory involvement (disorder)
        '328561000119107',  # Primary spontaneous pneumothorax (disorder)
        '328571000119101',  # Secondary spontaneous pneumothorax (disorder)
    }


class CompetingConditionsForRespiratoryConditions(ValueSet):
    """
    **Clinical Focus:** The purpose of this value set is to represent concepts for a diagnosis of  competing conditions that may warrant treatment with an antibiotic medication for respiratory conditions.

    **Data Element Scope:** This value set may use a model element related to Diagnosis.

    **Inclusion Criteria:** Includes concepts that represent a diagnosis of acute bacterial infections that may warrant treatment with an antibiotic medication.

    **Exclusion Criteria:** Excludes concepts that represent an acute diagnosis that does not warrant treatment with an antibiotic medication.

    ** Used in:** CMS154v10, CMS146v10
    """

    VALUE_SET_NAME = 'Competing Conditions for Respiratory Conditions'
    OID = '2.16.840.1.113883.3.464.1003.102.12.1017'
    DEFINITION_VERSION = '20210220'
    EXPANSION_VERSION = 'eCQM Update 2021-05-06'

    ICD10CM = {
        'A00.0',  # Cholera due to Vibrio cholerae 01, biovar cholerae
        'A00.1',  # Cholera due to Vibrio cholerae 01, biovar eltor
        'A00.9',  # Cholera, unspecified
        'A01.00',  # Typhoid fever, unspecified
        'A01.01',  # Typhoid meningitis
        'A01.02',  # Typhoid fever with heart involvement
        'A01.03',  # Typhoid pneumonia
        'A01.04',  # Typhoid arthritis
        'A01.05',  # Typhoid osteomyelitis
        'A01.09',  # Typhoid fever with other complications
        'A01.1',  # Paratyphoid fever A
        'A01.2',  # Paratyphoid fever B
        'A01.3',  # Paratyphoid fever C
        'A01.4',  # Paratyphoid fever, unspecified
        'A02.0',  # Salmonella enteritis
        'A02.1',  # Salmonella sepsis
        'A02.20',  # Localized salmonella infection, unspecified
        'A02.21',  # Salmonella meningitis
        'A02.22',  # Salmonella pneumonia
        'A02.23',  # Salmonella arthritis
        'A02.24',  # Salmonella osteomyelitis
        'A02.25',  # Salmonella pyelonephritis
        'A02.29',  # Salmonella with other localized infection
        'A02.8',  # Other specified salmonella infections
        'A02.9',  # Salmonella infection, unspecified
        'A03.0',  # Shigellosis due to Shigella dysenteriae
        'A03.1',  # Shigellosis due to Shigella flexneri
        'A03.2',  # Shigellosis due to Shigella boydii
        'A03.3',  # Shigellosis due to Shigella sonnei
        'A03.8',  # Other shigellosis
        'A03.9',  # Shigellosis, unspecified
        'A04.0',  # Enteropathogenic Escherichia coli infection
        'A04.1',  # Enterotoxigenic Escherichia coli infection
        'A04.2',  # Enteroinvasive Escherichia coli infection
        'A04.3',  # Enterohemorrhagic Escherichia coli infection
        'A04.4',  # Other intestinal Escherichia coli infections
        'A04.5',  # Campylobacter enteritis
        'A04.6',  # Enteritis due to Yersinia enterocolitica
        'A04.7',  # Enterocolitis due to Clostridium difficile
        'A04.71',  # Enterocolitis due to Clostridium difficile, recurrent
        'A04.72',  # Enterocolitis due to Clostridium difficile, not specified as recurrent
        'A04.8',  # Other specified bacterial intestinal infections
        'A04.9',  # Bacterial intestinal infection, unspecified
        'A05.0',  # Foodborne staphylococcal intoxication
        'A05.1',  # Botulism food poisoning
        'A05.2',  # Foodborne Clostridium perfringens [Clostridium welchii] intoxication
        'A05.3',  # Foodborne Vibrio parahaemolyticus intoxication
        'A05.4',  # Foodborne Bacillus cereus intoxication
        'A05.5',  # Foodborne Vibrio vulnificus intoxication
        'A05.8',  # Other specified bacterial foodborne intoxications
        'A05.9',  # Bacterial foodborne intoxication, unspecified
        'A06.0',  # Acute amebic dysentery
        'A06.1',  # Chronic intestinal amebiasis
        'A06.2',  # Amebic nondysenteric colitis
        'A06.3',  # Ameboma of intestine
        'A06.4',  # Amebic liver abscess
        'A06.5',  # Amebic lung abscess
        'A06.6',  # Amebic brain abscess
        'A06.7',  # Cutaneous amebiasis
        'A06.81',  # Amebic cystitis
        'A06.82',  # Other amebic genitourinary infections
        'A06.89',  # Other amebic infections
        'A06.9',  # Amebiasis, unspecified
        'A07.0',  # Balantidiasis
        'A07.1',  # Giardiasis [lambliasis]
        'A07.2',  # Cryptosporidiosis
        'A07.3',  # Isosporiasis
        'A07.4',  # Cyclosporiasis
        'A07.8',  # Other specified protozoal intestinal diseases
        'A07.9',  # Protozoal intestinal disease, unspecified
        'A08.0',  # Rotaviral enteritis
        'A08.11',  # Acute gastroenteropathy due to Norwalk agent
        'A08.19',  # Acute gastroenteropathy due to other small round viruses
        'A08.2',  # Adenoviral enteritis
        'A08.31',  # Calicivirus enteritis
        'A08.32',  # Astrovirus enteritis
        'A08.39',  # Other viral enteritis
        'A08.4',  # Viral intestinal infection, unspecified
        'A08.8',  # Other specified intestinal infections
        'A09',  # Infectious gastroenteritis and colitis, unspecified
        'A37.00',  # Whooping cough due to Bordetella pertussis without pneumonia
        'A37.01',  # Whooping cough due to Bordetella pertussis with pneumonia
        'A37.10',  # Whooping cough due to Bordetella parapertussis without pneumonia
        'A37.11',  # Whooping cough due to Bordetella parapertussis with pneumonia
        'A37.80',  # Whooping cough due to other Bordetella species without pneumonia
        'A37.81',  # Whooping cough due to other Bordetella species with pneumonia
        'A37.90',  # Whooping cough, unspecified species without pneumonia
        'A37.91',  # Whooping cough, unspecified species with pneumonia
        'A44.0',  # Systemic bartonellosis
        'A44.1',  # Cutaneous and mucocutaneous bartonellosis
        'A44.8',  # Other forms of bartonellosis
        'A44.9',  # Bartonellosis, unspecified
        'A49.9',  # Bacterial infection, unspecified
        'A50.01',  # Early congenital syphilitic oculopathy
        'A50.02',  # Early congenital syphilitic osteochondropathy
        'A50.03',  # Early congenital syphilitic pharyngitis
        'A50.04',  # Early congenital syphilitic pneumonia
        'A50.05',  # Early congenital syphilitic rhinitis
        'A50.06',  # Early cutaneous congenital syphilis
        'A50.07',  # Early mucocutaneous congenital syphilis
        'A50.08',  # Early visceral congenital syphilis
        'A50.09',  # Other early congenital syphilis, symptomatic
        'A50.1',  # Early congenital syphilis, latent
        'A50.2',  # Early congenital syphilis, unspecified
        'A50.30',  # Late congenital syphilitic oculopathy, unspecified
        'A50.31',  # Late congenital syphilitic interstitial keratitis
        'A50.32',  # Late congenital syphilitic chorioretinitis
        'A50.39',  # Other late congenital syphilitic oculopathy
        'A50.40',  # Late congenital neurosyphilis, unspecified
        'A50.41',  # Late congenital syphilitic meningitis
        'A50.42',  # Late congenital syphilitic encephalitis
        'A50.43',  # Late congenital syphilitic polyneuropathy
        'A50.44',  # Late congenital syphilitic optic nerve atrophy
        'A50.45',  # Juvenile general paresis
        'A50.49',  # Other late congenital neurosyphilis
        'A50.51',  # Clutton's joints
        'A50.52',  # Hutchinson's teeth
        'A50.53',  # Hutchinson's triad
        'A50.54',  # Late congenital cardiovascular syphilis
        'A50.55',  # Late congenital syphilitic arthropathy
        'A50.56',  # Late congenital syphilitic osteochondropathy
        'A50.57',  # Syphilitic saddle nose
        'A50.59',  # Other late congenital syphilis, symptomatic
        'A50.6',  # Late congenital syphilis, latent
        'A50.7',  # Late congenital syphilis, unspecified
        'A50.9',  # Congenital syphilis, unspecified
        'A51.0',  # Primary genital syphilis
        'A51.1',  # Primary anal syphilis
        'A51.2',  # Primary syphilis of other sites
        'A51.31',  # Condyloma latum
        'A51.32',  # Syphilitic alopecia
        'A51.39',  # Other secondary syphilis of skin
        'A51.41',  # Secondary syphilitic meningitis
        'A51.42',  # Secondary syphilitic female pelvic disease
        'A51.43',  # Secondary syphilitic oculopathy
        'A51.44',  # Secondary syphilitic nephritis
        'A51.45',  # Secondary syphilitic hepatitis
        'A51.46',  # Secondary syphilitic osteopathy
        'A51.49',  # Other secondary syphilitic conditions
        'A51.5',  # Early syphilis, latent
        'A51.9',  # Early syphilis, unspecified
        'A52.00',  # Cardiovascular syphilis, unspecified
        'A52.01',  # Syphilitic aneurysm of aorta
        'A52.02',  # Syphilitic aortitis
        'A52.03',  # Syphilitic endocarditis
        'A52.04',  # Syphilitic cerebral arteritis
        'A52.05',  # Other cerebrovascular syphilis
        'A52.06',  # Other syphilitic heart involvement
        'A52.09',  # Other cardiovascular syphilis
        'A52.10',  # Symptomatic neurosyphilis, unspecified
        'A52.11',  # Tabes dorsalis
        'A52.12',  # Other cerebrospinal syphilis
        'A52.13',  # Late syphilitic meningitis
        'A52.14',  # Late syphilitic encephalitis
        'A52.15',  # Late syphilitic neuropathy
        'A52.16',  # Charcôt's arthropathy (tabetic)
        'A52.17',  # General paresis
        'A52.19',  # Other symptomatic neurosyphilis
        'A52.2',  # Asymptomatic neurosyphilis
        'A52.3',  # Neurosyphilis, unspecified
        'A52.71',  # Late syphilitic oculopathy
        'A52.72',  # Syphilis of lung and bronchus
        'A52.73',  # Symptomatic late syphilis of other respiratory organs
        'A52.74',  # Syphilis of liver and other viscera
        'A52.75',  # Syphilis of kidney and ureter
        'A52.76',  # Other genitourinary symptomatic late syphilis
        'A52.77',  # Syphilis of bone and joint
        'A52.78',  # Syphilis of other musculoskeletal tissue
        'A52.79',  # Other symptomatic late syphilis
        'A52.8',  # Late syphilis, latent
        'A52.9',  # Late syphilis, unspecified
        'A53.0',  # Latent syphilis, unspecified as early or late
        'A53.9',  # Syphilis, unspecified
        'A54.00',  # Gonococcal infection of lower genitourinary tract, unspecified
        'A54.01',  # Gonococcal cystitis and urethritis, unspecified
        'A54.02',  # Gonococcal vulvovaginitis, unspecified
        'A54.03',  # Gonococcal cervicitis, unspecified
        'A54.09',  # Other gonococcal infection of lower genitourinary tract
        'A54.1',  # Gonococcal infection of lower genitourinary tract with periurethral and accessory gland abscess
        'A54.21',  # Gonococcal infection of kidney and ureter
        'A54.22',  # Gonococcal prostatitis
        'A54.23',  # Gonococcal infection of other male genital organs
        'A54.24',  # Gonococcal female pelvic inflammatory disease
        'A54.29',  # Other gonococcal genitourinary infections
        'A54.30',  # Gonococcal infection of eye, unspecified
        'A54.31',  # Gonococcal conjunctivitis
        'A54.32',  # Gonococcal iridocyclitis
        'A54.33',  # Gonococcal keratitis
        'A54.39',  # Other gonococcal eye infection
        'A54.40',  # Gonococcal infection of musculoskeletal system, unspecified
        'A54.41',  # Gonococcal spondylopathy
        'A54.42',  # Gonococcal arthritis
        'A54.43',  # Gonococcal osteomyelitis
        'A54.49',  # Gonococcal infection of other musculoskeletal tissue
        'A54.5',  # Gonococcal pharyngitis
        'A54.6',  # Gonococcal infection of anus and rectum
        'A54.81',  # Gonococcal meningitis
        'A54.82',  # Gonococcal brain abscess
        'A54.83',  # Gonococcal heart infection
        'A54.84',  # Gonococcal pneumonia
        'A54.85',  # Gonococcal peritonitis
        'A54.86',  # Gonococcal sepsis
        'A54.89',  # Other gonococcal infections
        'A54.9',  # Gonococcal infection, unspecified
        'A55',  # Chlamydial lymphogranuloma (venereum)
        'A56.00',  # Chlamydial infection of lower genitourinary tract, unspecified
        'A56.01',  # Chlamydial cystitis and urethritis
        'A56.02',  # Chlamydial vulvovaginitis
        'A56.09',  # Other chlamydial infection of lower genitourinary tract
        'A56.11',  # Chlamydial female pelvic inflammatory disease
        'A56.19',  # Other chlamydial genitourinary infection
        'A56.2',  # Chlamydial infection of genitourinary tract, unspecified
        'A56.3',  # Chlamydial infection of anus and rectum
        'A56.4',  # Chlamydial infection of pharynx
        'A56.8',  # Sexually transmitted chlamydial infection of other sites
        'A57',  # Chancroid
        'A58',  # Granuloma inguinale
        'A59.00',  # Urogenital trichomoniasis, unspecified
        'A59.01',  # Trichomonal vulvovaginitis
        'A59.02',  # Trichomonal prostatitis
        'A59.03',  # Trichomonal cystitis and urethritis
        'A59.09',  # Other urogenital trichomoniasis
        'A59.8',  # Trichomoniasis of other sites
        'A59.9',  # Trichomoniasis, unspecified
        'A63.0',  # Anogenital (venereal) warts
        'A63.8',  # Other specified predominantly sexually transmitted diseases
        'A64',  # Unspecified sexually transmitted disease
        'A69.0',  # Necrotizing ulcerative stomatitis
        'A69.1',  # Other Vincent's infections
        'A69.20',  # Lyme disease, unspecified
        'A69.21',  # Meningitis due to Lyme disease
        'A69.22',  # Other neurologic disorders in Lyme disease
        'A69.23',  # Arthritis due to Lyme disease
        'A69.29',  # Other conditions associated with Lyme disease
        'A69.8',  # Other specified spirochetal infections
        'A69.9',  # Spirochetal infection, unspecified
        'B60.0',  # Babesiosis
        'B60.00',  # Babesiosis, unspecified
        'B60.01',  # Babesiosis due to Babesia microti
        'B60.02',  # Babesiosis due to Babesia duncani
        'B60.03',  # Babesiosis due to Babesia divergens
        'B60.09',  # Other babesiosis
        'B60.10',  # Acanthamebiasis, unspecified
        'B60.11',  # Meningoencephalitis due to Acanthamoeba (culbertsoni)
        'B60.12',  # Conjunctivitis due to Acanthamoeba
        'B60.13',  # Keratoconjunctivitis due to Acanthamoeba
        'B60.19',  # Other acanthamebic disease
        'B60.2',  # Naegleriasis
        'B60.8',  # Other specified protozoal diseases
        'B64',  # Unspecified protozoal disease
        'B78.1',  # Cutaneous strongyloidiasis
        'B96.89',  # Other specified bacterial agents as the cause of diseases classified elsewhere
        'E83.2',  # Disorders of zinc metabolism
        'H66.001',  # Acute suppurative otitis media without spontaneous rupture of ear drum, right ear
        'H66.002',  # Acute suppurative otitis media without spontaneous rupture of ear drum, left ear
        'H66.003',  # Acute suppurative otitis media without spontaneous rupture of ear drum, bilateral
        'H66.004',  # Acute suppurative otitis media without spontaneous rupture of ear drum, recurrent, right ear
        'H66.005',  # Acute suppurative otitis media without spontaneous rupture of ear drum, recurrent, left ear
        'H66.006',  # Acute suppurative otitis media without spontaneous rupture of ear drum, recurrent, bilateral
        'H66.007',  # Acute suppurative otitis media without spontaneous rupture of ear drum, recurrent, unspecified ear
        'H66.009',  # Acute suppurative otitis media without spontaneous rupture of ear drum, unspecified ear
        'H66.011',  # Acute suppurative otitis media with spontaneous rupture of ear drum, right ear
        'H66.012',  # Acute suppurative otitis media with spontaneous rupture of ear drum, left ear
        'H66.013',  # Acute suppurative otitis media with spontaneous rupture of ear drum, bilateral
        'H66.014',  # Acute suppurative otitis media with spontaneous rupture of ear drum, recurrent, right ear
        'H66.015',  # Acute suppurative otitis media with spontaneous rupture of ear drum, recurrent, left ear
        'H66.016',  # Acute suppurative otitis media with spontaneous rupture of ear drum, recurrent, bilateral
        'H66.017',  # Acute suppurative otitis media with spontaneous rupture of ear drum, recurrent, unspecified ear
        'H66.019',  # Acute suppurative otitis media with spontaneous rupture of ear drum, unspecified ear
        'H66.10',  # Chronic tubotympanic suppurative otitis media, unspecified
        'H66.11',  # Chronic tubotympanic suppurative otitis media, right ear
        'H66.12',  # Chronic tubotympanic suppurative otitis media, left ear
        'H66.13',  # Chronic tubotympanic suppurative otitis media, bilateral
        'H66.20',  # Chronic atticoantral suppurative otitis media, unspecified ear
        'H66.21',  # Chronic atticoantral suppurative otitis media, right ear
        'H66.22',  # Chronic atticoantral suppurative otitis media, left ear
        'H66.23',  # Chronic atticoantral suppurative otitis media, bilateral
        'H66.3X1',  # Other chronic suppurative otitis media, right ear
        'H66.3X2',  # Other chronic suppurative otitis media, left ear
        'H66.3X3',  # Other chronic suppurative otitis media, bilateral
        'H66.3X9',  # Other chronic suppurative otitis media, unspecified ear
        'H66.40',  # Suppurative otitis media, unspecified, unspecified ear
        'H66.41',  # Suppurative otitis media, unspecified, right ear
        'H66.42',  # Suppurative otitis media, unspecified, left ear
        'H66.43',  # Suppurative otitis media, unspecified, bilateral
        'H66.90',  # Otitis media, unspecified, unspecified ear
        'H66.91',  # Otitis media, unspecified, right ear
        'H66.92',  # Otitis media, unspecified, left ear
        'H66.93',  # Otitis media, unspecified, bilateral
        'H67.1',  # Otitis media in diseases classified elsewhere, right ear
        'H67.2',  # Otitis media in diseases classified elsewhere, left ear
        'H67.3',  # Otitis media in diseases classified elsewhere, bilateral
        'H67.9',  # Otitis media in diseases classified elsewhere, unspecified ear
        'H70.001',  # Acute mastoiditis without complications, right ear
        'H70.002',  # Acute mastoiditis without complications, left ear
        'H70.003',  # Acute mastoiditis without complications, bilateral
        'H70.009',  # Acute mastoiditis without complications, unspecified ear
        'H70.011',  # Subperiosteal abscess of mastoid, right ear
        'H70.012',  # Subperiosteal abscess of mastoid, left ear
        'H70.013',  # Subperiosteal abscess of mastoid, bilateral
        'H70.019',  # Subperiosteal abscess of mastoid, unspecified ear
        'H70.091',  # Acute mastoiditis with other complications, right ear
        'H70.092',  # Acute mastoiditis with other complications, left ear
        'H70.093',  # Acute mastoiditis with other complications, bilateral
        'H70.099',  # Acute mastoiditis with other complications, unspecified ear
        'H70.10',  # Chronic mastoiditis, unspecified ear
        'H70.11',  # Chronic mastoiditis, right ear
        'H70.12',  # Chronic mastoiditis, left ear
        'H70.13',  # Chronic mastoiditis, bilateral
        'H70.201',  # Unspecified petrositis, right ear
        'H70.202',  # Unspecified petrositis, left ear
        'H70.203',  # Unspecified petrositis, bilateral
        'H70.209',  # Unspecified petrositis, unspecified ear
        'H70.211',  # Acute petrositis, right ear
        'H70.212',  # Acute petrositis, left ear
        'H70.213',  # Acute petrositis, bilateral
        'H70.219',  # Acute petrositis, unspecified ear
        'H70.221',  # Chronic petrositis, right ear
        'H70.222',  # Chronic petrositis, left ear
        'H70.223',  # Chronic petrositis, bilateral
        'H70.229',  # Chronic petrositis, unspecified ear
        'H70.811',  # Postauricular fistula, right ear
        'H70.812',  # Postauricular fistula, left ear
        'H70.813',  # Postauricular fistula, bilateral
        'H70.819',  # Postauricular fistula, unspecified ear
        'H70.891',  # Other mastoiditis and related conditions, right ear
        'H70.892',  # Other mastoiditis and related conditions, left ear
        'H70.893',  # Other mastoiditis and related conditions, bilateral
        'H70.899',  # Other mastoiditis and related conditions, unspecified ear
        'H70.90',  # Unspecified mastoiditis, unspecified ear
        'H70.91',  # Unspecified mastoiditis, right ear
        'H70.92',  # Unspecified mastoiditis, left ear
        'H70.93',  # Unspecified mastoiditis, bilateral
        'H95.00',  # Recurrent cholesteatoma of postmastoidectomy cavity, unspecified ear
        'H95.01',  # Recurrent cholesteatoma of postmastoidectomy cavity, right ear
        'H95.02',  # Recurrent cholesteatoma of postmastoidectomy cavity, left ear
        'H95.03',  # Recurrent cholesteatoma of postmastoidectomy cavity, bilateral ears
        'H95.111',  # Chronic inflammation of postmastoidectomy cavity, right ear
        'H95.112',  # Chronic inflammation of postmastoidectomy cavity, left ear
        'H95.113',  # Chronic inflammation of postmastoidectomy cavity, bilateral ears
        'H95.119',  # Chronic inflammation of postmastoidectomy cavity, unspecified ear
        'H95.121',  # Granulation of postmastoidectomy cavity, right ear
        'H95.122',  # Granulation of postmastoidectomy cavity, left ear
        'H95.123',  # Granulation of postmastoidectomy cavity, bilateral ears
        'H95.129',  # Granulation of postmastoidectomy cavity, unspecified ear
        'H95.131',  # Mucosal cyst of postmastoidectomy cavity, right ear
        'H95.132',  # Mucosal cyst of postmastoidectomy cavity, left ear
        'H95.133',  # Mucosal cyst of postmastoidectomy cavity, bilateral ears
        'H95.139',  # Mucosal cyst of postmastoidectomy cavity, unspecified ear
        'H95.191',  # Other disorders following mastoidectomy, right ear
        'H95.192',  # Other disorders following mastoidectomy, left ear
        'H95.193',  # Other disorders following mastoidectomy, bilateral ears
        'H95.199',  # Other disorders following mastoidectomy, unspecified ear
        'H95.21',  # Intraoperative hemorrhage and hematoma of ear and mastoid process complicating a procedure on the ear and mastoid process
        'H95.22',  # Intraoperative hemorrhage and hematoma of ear and mastoid process complicating other procedure
        'H95.31',  # Accidental puncture and laceration of the ear and mastoid process during a procedure on the ear and mastoid process
        'H95.32',  # Accidental puncture and laceration of the ear and mastoid process during other procedure
        'H95.41',  # Postprocedural hemorrhage of ear and mastoid process following a procedure on the ear and mastoid process
        'H95.42',  # Postprocedural hemorrhage of ear and mastoid process following other procedure
        'H95.51',  # Postprocedural hematoma of ear and mastoid process following a procedure on the ear and mastoid process
        'H95.52',  # Postprocedural hematoma of ear and mastoid process following other procedure
        'H95.53',  # Postprocedural seroma of ear and mastoid process following a procedure on the ear and mastoid process
        'H95.54',  # Postprocedural seroma of ear and mastoid process following other procedure
        'H95.811',  # Postprocedural stenosis of right external ear canal
        'H95.812',  # Postprocedural stenosis of left external ear canal
        'H95.813',  # Postprocedural stenosis of external ear canal, bilateral
        'H95.819',  # Postprocedural stenosis of unspecified external ear canal
        'H95.88',  # Other intraoperative complications and disorders of the ear and mastoid process, not elsewhere classified
        'H95.89',  # Other postprocedural complications and disorders of the ear and mastoid process, not elsewhere classified
        'J01.00',  # Acute maxillary sinusitis, unspecified
        'J01.01',  # Acute recurrent maxillary sinusitis
        'J01.10',  # Acute frontal sinusitis, unspecified
        'J01.11',  # Acute recurrent frontal sinusitis
        'J01.20',  # Acute ethmoidal sinusitis, unspecified
        'J01.21',  # Acute recurrent ethmoidal sinusitis
        'J01.30',  # Acute sphenoidal sinusitis, unspecified
        'J01.31',  # Acute recurrent sphenoidal sinusitis
        'J01.40',  # Acute pansinusitis, unspecified
        'J01.41',  # Acute recurrent pansinusitis
        'J01.80',  # Other acute sinusitis
        'J01.81',  # Other acute recurrent sinusitis
        'J01.90',  # Acute sinusitis, unspecified
        'J01.91',  # Acute recurrent sinusitis, unspecified
        'J04.10',  # Acute tracheitis without obstruction
        'J04.11',  # Acute tracheitis with obstruction
        'J04.2',  # Acute laryngotracheitis
        'J05.0',  # Acute obstructive laryngitis [croup]
        'J05.10',  # Acute epiglottitis without obstruction
        'J05.11',  # Acute epiglottitis with obstruction
        'J13',  # Pneumonia due to Streptococcus pneumoniae
        'J14',  # Pneumonia due to Hemophilus influenzae
        'J15.0',  # Pneumonia due to Klebsiella pneumoniae
        'J15.1',  # Pneumonia due to Pseudomonas
        'J15.20',  # Pneumonia due to staphylococcus, unspecified
        'J15.211',  # Pneumonia due to Methicillin susceptible Staphylococcus aureus
        'J15.212',  # Pneumonia due to Methicillin resistant Staphylococcus aureus
        'J15.29',  # Pneumonia due to other staphylococcus
        'J15.3',  # Pneumonia due to streptococcus, group B
        'J15.4',  # Pneumonia due to other streptococci
        'J15.5',  # Pneumonia due to Escherichia coli
        'J15.6',  # Pneumonia due to other Gram-negative bacteria
        'J15.7',  # Pneumonia due to Mycoplasma pneumoniae
        'J15.8',  # Pneumonia due to other specified bacteria
        'J15.9',  # Unspecified bacterial pneumonia
        'J16.0',  # Chlamydial pneumonia
        'J16.8',  # Pneumonia due to other specified infectious organisms
        'J17',  # Pneumonia in diseases classified elsewhere
        'J18.0',  # Bronchopneumonia, unspecified organism
        'J18.1',  # Lobar pneumonia, unspecified organism
        'J18.8',  # Other pneumonia, unspecified organism
        'J18.9',  # Pneumonia, unspecified organism
        'J20.0',  # Acute bronchitis due to Mycoplasma pneumoniae
        'J20.1',  # Acute bronchitis due to Hemophilus influenzae
        'J20.2',  # Acute bronchitis due to streptococcus
        'J32.0',  # Chronic maxillary sinusitis
        'J32.1',  # Chronic frontal sinusitis
        'J32.2',  # Chronic ethmoidal sinusitis
        'J32.3',  # Chronic sphenoidal sinusitis
        'J32.4',  # Chronic pansinusitis
        'J32.8',  # Other chronic sinusitis
        'J32.9',  # Chronic sinusitis, unspecified
        'J35.01',  # Chronic tonsillitis
        'J35.02',  # Chronic adenoiditis
        'J35.03',  # Chronic tonsillitis and adenoiditis
        'J35.1',  # Hypertrophy of tonsils
        'J35.2',  # Hypertrophy of adenoids
        'J35.3',  # Hypertrophy of tonsils with hypertrophy of adenoids
        'J35.8',  # Other chronic diseases of tonsils and adenoids
        'J35.9',  # Chronic disease of tonsils and adenoids, unspecified
        'J38.7',  # Other diseases of larynx
        'J39.0',  # Retropharyngeal and parapharyngeal abscess
        'J39.1',  # Other abscess of pharynx
        'J39.2',  # Other diseases of pharynx
        'J39.8',  # Other specified diseases of upper respiratory tract
        'J39.9',  # Disease of upper respiratory tract, unspecified
        'K12.2',  # Cellulitis and abscess of mouth
        'L01.00',  # Impetigo, unspecified
        'L01.01',  # Non-bullous impetigo
        'L01.02',  # Bockhart's impetigo
        'L01.03',  # Bullous impetigo
        'L01.09',  # Other impetigo
        'L01.1',  # Impetiginization of other dermatoses
        'L03.011',  # Cellulitis of right finger
        'L03.012',  # Cellulitis of left finger
        'L03.019',  # Cellulitis of unspecified finger
        'L03.021',  # Acute lymphangitis of right finger
        'L03.022',  # Acute lymphangitis of left finger
        'L03.029',  # Acute lymphangitis of unspecified finger
        'L03.031',  # Cellulitis of right toe
        'L03.032',  # Cellulitis of left toe
        'L03.039',  # Cellulitis of unspecified toe
        'L03.041',  # Acute lymphangitis of right toe
        'L03.042',  # Acute lymphangitis of left toe
        'L03.049',  # Acute lymphangitis of unspecified toe
        'L03.111',  # Cellulitis of right axilla
        'L03.112',  # Cellulitis of left axilla
        'L03.113',  # Cellulitis of right upper limb
        'L03.114',  # Cellulitis of left upper limb
        'L03.115',  # Cellulitis of right lower limb
        'L03.116',  # Cellulitis of left lower limb
        'L03.119',  # Cellulitis of unspecified part of limb
        'L03.121',  # Acute lymphangitis of right axilla
        'L03.122',  # Acute lymphangitis of left axilla
        'L03.123',  # Acute lymphangitis of right upper limb
        'L03.124',  # Acute lymphangitis of left upper limb
        'L03.125',  # Acute lymphangitis of right lower limb
        'L03.126',  # Acute lymphangitis of left lower limb
        'L03.129',  # Acute lymphangitis of unspecified part of limb
        'L03.211',  # Cellulitis of face
        'L03.212',  # Acute lymphangitis of face
        'L03.213',  # Periorbital cellulitis
        'L03.221',  # Cellulitis of neck
        'L03.222',  # Acute lymphangitis of neck
        'L03.311',  # Cellulitis of abdominal wall
        'L03.312',  # Cellulitis of back [any part except buttock]
        'L03.313',  # Cellulitis of chest wall
        'L03.314',  # Cellulitis of groin
        'L03.315',  # Cellulitis of perineum
        'L03.316',  # Cellulitis of umbilicus
        'L03.317',  # Cellulitis of buttock
        'L03.319',  # Cellulitis of trunk, unspecified
        'L03.321',  # Acute lymphangitis of abdominal wall
        'L03.322',  # Acute lymphangitis of back [any part except buttock]
        'L03.323',  # Acute lymphangitis of chest wall
        'L03.324',  # Acute lymphangitis of groin
        'L03.325',  # Acute lymphangitis of perineum
        'L03.326',  # Acute lymphangitis of umbilicus
        'L03.327',  # Acute lymphangitis of buttock
        'L03.329',  # Acute lymphangitis of trunk, unspecified
        'L03.811',  # Cellulitis of head [any part, except face]
        'L03.818',  # Cellulitis of other sites
        'L03.891',  # Acute lymphangitis of head [any part, except face]
        'L03.898',  # Acute lymphangitis of other sites
        'L03.90',  # Cellulitis, unspecified
        'L03.91',  # Acute lymphangitis, unspecified
        'L04.0',  # Acute lymphadenitis of face, head and neck
        'L04.1',  # Acute lymphadenitis of trunk
        'L04.2',  # Acute lymphadenitis of upper limb
        'L04.3',  # Acute lymphadenitis of lower limb
        'L04.8',  # Acute lymphadenitis of other sites
        'L04.9',  # Acute lymphadenitis, unspecified
        'L08.1',  # Erythrasma
        'L08.81',  # Pyoderma vegetans
        'L08.82',  # Omphalitis not of newborn
        'L08.89',  # Other specified local infections of the skin and subcutaneous tissue
        'L08.9',  # Local infection of the skin and subcutaneous tissue, unspecified
        'L88',  # Pyoderma gangrenosum
        'L92.8',  # Other granulomatous disorders of the skin and subcutaneous tissue
        'L98.0',  # Pyogenic granuloma
        'L98.3',  # Eosinophilic cellulitis [Wells]
        'M46.20',  # Osteomyelitis of vertebra, site unspecified
        'M46.21',  # Osteomyelitis of vertebra, occipito-atlanto-axial region
        'M46.22',  # Osteomyelitis of vertebra, cervical region
        'M46.23',  # Osteomyelitis of vertebra, cervicothoracic region
        'M46.24',  # Osteomyelitis of vertebra, thoracic region
        'M46.25',  # Osteomyelitis of vertebra, thoracolumbar region
        'M46.26',  # Osteomyelitis of vertebra, lumbar region
        'M46.27',  # Osteomyelitis of vertebra, lumbosacral region
        'M46.28',  # Osteomyelitis of vertebra, sacral and sacrococcygeal region
        'M46.30',  # Infection of intervertebral disc (pyogenic), site unspecified
        'M46.31',  # Infection of intervertebral disc (pyogenic), occipito-atlanto-axial region
        'M46.32',  # Infection of intervertebral disc (pyogenic), cervical region
        'M46.33',  # Infection of intervertebral disc (pyogenic), cervicothoracic region
        'M46.34',  # Infection of intervertebral disc (pyogenic), thoracic region
        'M46.35',  # Infection of intervertebral disc (pyogenic), thoracolumbar region
        'M46.36',  # Infection of intervertebral disc (pyogenic), lumbar region
        'M46.37',  # Infection of intervertebral disc (pyogenic), lumbosacral region
        'M46.38',  # Infection of intervertebral disc (pyogenic), sacral and sacrococcygeal region
        'M46.39',  # Infection of intervertebral disc (pyogenic), multiple sites in spine
        'M89.00',  # Algoneurodystrophy, unspecified site
        'M89.011',  # Algoneurodystrophy, right shoulder
        'M89.012',  # Algoneurodystrophy, left shoulder
        'M89.019',  # Algoneurodystrophy, unspecified shoulder
        'M89.021',  # Algoneurodystrophy, right upper arm
        'M89.022',  # Algoneurodystrophy, left upper arm
        'M89.029',  # Algoneurodystrophy, unspecified upper arm
        'M89.031',  # Algoneurodystrophy, right forearm
        'M89.032',  # Algoneurodystrophy, left forearm
        'M89.039',  # Algoneurodystrophy, unspecified forearm
        'M89.041',  # Algoneurodystrophy, right hand
        'M89.042',  # Algoneurodystrophy, left hand
        'M89.049',  # Algoneurodystrophy, unspecified hand
        'M89.051',  # Algoneurodystrophy, right thigh
        'M89.052',  # Algoneurodystrophy, left thigh
        'M89.059',  # Algoneurodystrophy, unspecified thigh
        'M89.061',  # Algoneurodystrophy, right lower leg
        'M89.062',  # Algoneurodystrophy, left lower leg
        'M89.069',  # Algoneurodystrophy, unspecified lower leg
        'M89.071',  # Algoneurodystrophy, right ankle and foot
        'M89.072',  # Algoneurodystrophy, left ankle and foot
        'M89.079',  # Algoneurodystrophy, unspecified ankle and foot
        'M89.08',  # Algoneurodystrophy, other site
        'M89.09',  # Algoneurodystrophy, multiple sites
        'M89.121',  # Complete physeal arrest, right proximal humerus
        'M89.122',  # Complete physeal arrest, left proximal humerus
        'M89.123',  # Partial physeal arrest, right proximal humerus
        'M89.124',  # Partial physeal arrest, left proximal humerus
        'M89.125',  # Complete physeal arrest, right distal humerus
        'M89.126',  # Complete physeal arrest, left distal humerus
        'M89.127',  # Partial physeal arrest, right distal humerus
        'M89.128',  # Partial physeal arrest, left distal humerus
        'M89.129',  # Physeal arrest, humerus, unspecified
        'M89.131',  # Complete physeal arrest, right distal radius
        'M89.132',  # Complete physeal arrest, left distal radius
        'M89.133',  # Partial physeal arrest, right distal radius
        'M89.134',  # Partial physeal arrest, left distal radius
        'M89.138',  # Other physeal arrest of forearm
        'M89.139',  # Physeal arrest, forearm, unspecified
        'M89.151',  # Complete physeal arrest, right proximal femur
        'M89.152',  # Complete physeal arrest, left proximal femur
        'M89.153',  # Partial physeal arrest, right proximal femur
        'M89.154',  # Partial physeal arrest, left proximal femur
        'M89.155',  # Complete physeal arrest, right distal femur
        'M89.156',  # Complete physeal arrest, left distal femur
        'M89.157',  # Partial physeal arrest, right distal femur
        'M89.158',  # Partial physeal arrest, left distal femur
        'M89.159',  # Physeal arrest, femur, unspecified
        'M89.160',  # Complete physeal arrest, right proximal tibia
        'M89.161',  # Complete physeal arrest, left proximal tibia
        'M89.162',  # Partial physeal arrest, right proximal tibia
        'M89.163',  # Partial physeal arrest, left proximal tibia
        'M89.164',  # Complete physeal arrest, right distal tibia
        'M89.165',  # Complete physeal arrest, left distal tibia
        'M89.166',  # Partial physeal arrest, right distal tibia
        'M89.167',  # Partial physeal arrest, left distal tibia
        'M89.168',  # Other physeal arrest of lower leg
        'M89.169',  # Physeal arrest, lower leg, unspecified
        'M89.18',  # Physeal arrest, other site
        'M89.20',  # Other disorders of bone development and growth, unspecified site
        'M89.211',  # Other disorders of bone development and growth, right shoulder
        'M89.212',  # Other disorders of bone development and growth, left shoulder
        'M89.219',  # Other disorders of bone development and growth, unspecified shoulder
        'M89.221',  # Other disorders of bone development and growth, right humerus
        'M89.222',  # Other disorders of bone development and growth, left humerus
        'M89.229',  # Other disorders of bone development and growth, unspecified humerus
        'M89.231',  # Other disorders of bone development and growth, right ulna
        'M89.232',  # Other disorders of bone development and growth, left ulna
        'M89.233',  # Other disorders of bone development and growth, right radius
        'M89.234',  # Other disorders of bone development and growth, left radius
        'M89.239',  # Other disorders of bone development and growth, unspecified ulna and radius
        'M89.241',  # Other disorders of bone development and growth, right hand
        'M89.242',  # Other disorders of bone development and growth, left hand
        'M89.249',  # Other disorders of bone development and growth, unspecified hand
        'M89.251',  # Other disorders of bone development and growth, right femur
        'M89.252',  # Other disorders of bone development and growth, left femur
        'M89.259',  # Other disorders of bone development and growth, unspecified femur
        'M89.261',  # Other disorders of bone development and growth, right tibia
        'M89.262',  # Other disorders of bone development and growth, left tibia
        'M89.263',  # Other disorders of bone development and growth, right fibula
        'M89.264',  # Other disorders of bone development and growth, left fibula
        'M89.269',  # Other disorders of bone development and growth, unspecified lower leg
        'M89.271',  # Other disorders of bone development and growth, right ankle and foot
        'M89.272',  # Other disorders of bone development and growth, left ankle and foot
        'M89.279',  # Other disorders of bone development and growth, unspecified ankle and foot
        'M89.28',  # Other disorders of bone development and growth, other site
        'M89.29',  # Other disorders of bone development and growth, multiple sites
        'M89.40',  # Other hypertrophic osteoarthropathy, unspecified site
        'M89.411',  # Other hypertrophic osteoarthropathy, right shoulder
        'M89.412',  # Other hypertrophic osteoarthropathy, left shoulder
        'M89.419',  # Other hypertrophic osteoarthropathy, unspecified shoulder
        'M89.421',  # Other hypertrophic osteoarthropathy, right upper arm
        'M89.422',  # Other hypertrophic osteoarthropathy, left upper arm
        'M89.429',  # Other hypertrophic osteoarthropathy, unspecified upper arm
        'M89.431',  # Other hypertrophic osteoarthropathy, right forearm
        'M89.432',  # Other hypertrophic osteoarthropathy, left forearm
        'M89.439',  # Other hypertrophic osteoarthropathy, unspecified forearm
        'M89.441',  # Other hypertrophic osteoarthropathy, right hand
        'M89.442',  # Other hypertrophic osteoarthropathy, left hand
        'M89.449',  # Other hypertrophic osteoarthropathy, unspecified hand
        'M89.451',  # Other hypertrophic osteoarthropathy, right thigh
        'M89.452',  # Other hypertrophic osteoarthropathy, left thigh
        'M89.459',  # Other hypertrophic osteoarthropathy, unspecified thigh
        'M89.461',  # Other hypertrophic osteoarthropathy, right lower leg
        'M89.462',  # Other hypertrophic osteoarthropathy, left lower leg
        'M89.469',  # Other hypertrophic osteoarthropathy, unspecified lower leg
        'M89.471',  # Other hypertrophic osteoarthropathy, right ankle and foot
        'M89.472',  # Other hypertrophic osteoarthropathy, left ankle and foot
        'M89.479',  # Other hypertrophic osteoarthropathy, unspecified ankle and foot
        'M89.48',  # Other hypertrophic osteoarthropathy, other site
        'M89.49',  # Other hypertrophic osteoarthropathy, multiple sites
        'M89.50',  # Osteolysis, unspecified site
        'M89.511',  # Osteolysis, right shoulder
        'M89.512',  # Osteolysis, left shoulder
        'M89.519',  # Osteolysis, unspecified shoulder
        'M89.521',  # Osteolysis, right upper arm
        'M89.522',  # Osteolysis, left upper arm
        'M89.529',  # Osteolysis, unspecified upper arm
        'M89.531',  # Osteolysis, right forearm
        'M89.532',  # Osteolysis, left forearm
        'M89.539',  # Osteolysis, unspecified forearm
        'M89.541',  # Osteolysis, right hand
        'M89.542',  # Osteolysis, left hand
        'M89.549',  # Osteolysis, unspecified hand
        'M89.551',  # Osteolysis, right thigh
        'M89.552',  # Osteolysis, left thigh
        'M89.559',  # Osteolysis, unspecified thigh
        'M89.561',  # Osteolysis, right lower leg
        'M89.562',  # Osteolysis, left lower leg
        'M89.569',  # Osteolysis, unspecified lower leg
        'M89.571',  # Osteolysis, right ankle and foot
        'M89.572',  # Osteolysis, left ankle and foot
        'M89.579',  # Osteolysis, unspecified ankle and foot
        'M89.58',  # Osteolysis, other site
        'M89.59',  # Osteolysis, multiple sites
        'M89.60',  # Osteopathy after poliomyelitis, unspecified site
        'M89.611',  # Osteopathy after poliomyelitis, right shoulder
        'M89.612',  # Osteopathy after poliomyelitis, left shoulder
        'M89.619',  # Osteopathy after poliomyelitis, unspecified shoulder
        'M89.621',  # Osteopathy after poliomyelitis, right upper arm
        'M89.622',  # Osteopathy after poliomyelitis, left upper arm
        'M89.629',  # Osteopathy after poliomyelitis, unspecified upper arm
        'M89.631',  # Osteopathy after poliomyelitis, right forearm
        'M89.632',  # Osteopathy after poliomyelitis, left forearm
        'M89.639',  # Osteopathy after poliomyelitis, unspecified forearm
        'M89.641',  # Osteopathy after poliomyelitis, right hand
        'M89.642',  # Osteopathy after poliomyelitis, left hand
        'M89.649',  # Osteopathy after poliomyelitis, unspecified hand
        'M89.651',  # Osteopathy after poliomyelitis, right thigh
        'M89.652',  # Osteopathy after poliomyelitis, left thigh
        'M89.659',  # Osteopathy after poliomyelitis, unspecified thigh
        'M89.661',  # Osteopathy after poliomyelitis, right lower leg
        'M89.662',  # Osteopathy after poliomyelitis, left lower leg
        'M89.669',  # Osteopathy after poliomyelitis, unspecified lower leg
        'M89.671',  # Osteopathy after poliomyelitis, right ankle and foot
        'M89.672',  # Osteopathy after poliomyelitis, left ankle and foot
        'M89.679',  # Osteopathy after poliomyelitis, unspecified ankle and foot
        'M89.68',  # Osteopathy after poliomyelitis, other site
        'M89.69',  # Osteopathy after poliomyelitis, multiple sites
        'M89.70',  # Major osseous defect, unspecified site
        'M89.711',  # Major osseous defect, right shoulder region
        'M89.712',  # Major osseous defect, left shoulder region
        'M89.719',  # Major osseous defect, unspecified shoulder region
        'M89.721',  # Major osseous defect, right humerus
        'M89.722',  # Major osseous defect, left humerus
        'M89.729',  # Major osseous defect, unspecified humerus
        'M89.731',  # Major osseous defect, right forearm
        'M89.732',  # Major osseous defect, left forearm
        'M89.739',  # Major osseous defect, unspecified forearm
        'M89.741',  # Major osseous defect, right hand
        'M89.742',  # Major osseous defect, left hand
        'M89.749',  # Major osseous defect, unspecified hand
        'M89.751',  # Major osseous defect, right pelvic region and thigh
        'M89.752',  # Major osseous defect, left pelvic region and thigh
        'M89.759',  # Major osseous defect, unspecified pelvic region and thigh
        'M89.761',  # Major osseous defect, right lower leg
        'M89.762',  # Major osseous defect, left lower leg
        'M89.769',  # Major osseous defect, unspecified lower leg
        'M89.771',  # Major osseous defect, right ankle and foot
        'M89.772',  # Major osseous defect, left ankle and foot
        'M89.779',  # Major osseous defect, unspecified ankle and foot
        'M89.78',  # Major osseous defect, other site
        'M89.79',  # Major osseous defect, multiple sites
        'M89.8X0',  # Other specified disorders of bone, multiple sites
        'M89.8X1',  # Other specified disorders of bone, shoulder
        'M89.8X2',  # Other specified disorders of bone, upper arm
        'M89.8X3',  # Other specified disorders of bone, forearm
        'M89.8X4',  # Other specified disorders of bone, hand
        'M89.8X5',  # Other specified disorders of bone, thigh
        'M89.8X6',  # Other specified disorders of bone, lower leg
        'M89.8X7',  # Other specified disorders of bone, ankle and foot
        'M89.8X8',  # Other specified disorders of bone, other site
        'M89.8X9',  # Other specified disorders of bone, unspecified site
        'M89.9',  # Disorder of bone, unspecified
        'M90.80',  # Osteopathy in diseases classified elsewhere, unspecified site
        'M90.811',  # Osteopathy in diseases classified elsewhere, right shoulder
        'M90.812',  # Osteopathy in diseases classified elsewhere, left shoulder
        'M90.819',  # Osteopathy in diseases classified elsewhere, unspecified shoulder
        'M90.821',  # Osteopathy in diseases classified elsewhere, right upper arm
        'M90.822',  # Osteopathy in diseases classified elsewhere, left upper arm
        'M90.829',  # Osteopathy in diseases classified elsewhere, unspecified upper arm
        'M90.831',  # Osteopathy in diseases classified elsewhere, right forearm
        'M90.832',  # Osteopathy in diseases classified elsewhere, left forearm
        'M90.839',  # Osteopathy in diseases classified elsewhere, unspecified forearm
        'M90.841',  # Osteopathy in diseases classified elsewhere, right hand
        'M90.842',  # Osteopathy in diseases classified elsewhere, left hand
        'M90.849',  # Osteopathy in diseases classified elsewhere, unspecified hand
        'M90.851',  # Osteopathy in diseases classified elsewhere, right thigh
        'M90.852',  # Osteopathy in diseases classified elsewhere, left thigh
        'M90.859',  # Osteopathy in diseases classified elsewhere, unspecified thigh
        'M90.861',  # Osteopathy in diseases classified elsewhere, right lower leg
        'M90.862',  # Osteopathy in diseases classified elsewhere, left lower leg
        'M90.869',  # Osteopathy in diseases classified elsewhere, unspecified lower leg
        'M90.871',  # Osteopathy in diseases classified elsewhere, right ankle and foot
        'M90.872',  # Osteopathy in diseases classified elsewhere, left ankle and foot
        'M90.879',  # Osteopathy in diseases classified elsewhere, unspecified ankle and foot
        'M90.88',  # Osteopathy in diseases classified elsewhere, other site
        'M90.89',  # Osteopathy in diseases classified elsewhere, multiple sites
        'N10',  # Acute pyelonephritis
        'N11.0',  # Nonobstructive reflux-associated chronic pyelonephritis
        'N11.1',  # Chronic obstructive pyelonephritis
        'N11.8',  # Other chronic tubulo-interstitial nephritis
        'N11.9',  # Chronic tubulo-interstitial nephritis, unspecified
        'N12',  # Tubulo-interstitial nephritis, not specified as acute or chronic
        'N13.0',  # Hydronephrosis with ureteropelvic junction obstruction
        'N13.6',  # Pyonephrosis
        'N15.1',  # Renal and perinephric abscess
        'N16',  # Renal tubulo-interstitial disorders in diseases classified elsewhere
        'N28.84',  # Pyelitis cystica
        'N28.85',  # Pyeloureteritis cystica
        'N28.86',  # Ureteritis cystica
        'N30.00',  # Acute cystitis without hematuria
        'N30.01',  # Acute cystitis with hematuria
        'N30.10',  # Interstitial cystitis (chronic) without hematuria
        'N30.11',  # Interstitial cystitis (chronic) with hematuria
        'N30.20',  # Other chronic cystitis without hematuria
        'N30.21',  # Other chronic cystitis with hematuria
        'N30.30',  # Trigonitis without hematuria
        'N30.31',  # Trigonitis with hematuria
        'N30.40',  # Irradiation cystitis without hematuria
        'N30.41',  # Irradiation cystitis with hematuria
        'N30.80',  # Other cystitis without hematuria
        'N30.81',  # Other cystitis with hematuria
        'N30.90',  # Cystitis, unspecified without hematuria
        'N30.91',  # Cystitis, unspecified with hematuria
        'N39.0',  # Urinary tract infection, site not specified
        'N41.0',  # Acute prostatitis
        'N41.1',  # Chronic prostatitis
        'N41.2',  # Abscess of prostate
        'N41.3',  # Prostatocystitis
        'N41.4',  # Granulomatous prostatitis
        'N41.8',  # Other inflammatory diseases of prostate
        'N41.9',  # Inflammatory disease of prostate, unspecified
        'N70.01',  # Acute salpingitis
        'N70.02',  # Acute oophoritis
        'N70.03',  # Acute salpingitis and oophoritis
        'N70.11',  # Chronic salpingitis
        'N70.12',  # Chronic oophoritis
        'N70.13',  # Chronic salpingitis and oophoritis
        'N70.91',  # Salpingitis, unspecified
        'N70.92',  # Oophoritis, unspecified
        'N70.93',  # Salpingitis and oophoritis, unspecified
        'N71.0',  # Acute inflammatory disease of uterus
        'N71.1',  # Chronic inflammatory disease of uterus
        'N71.9',  # Inflammatory disease of uterus, unspecified
        'N72',  # Inflammatory disease of cervix uteri
        'N73.0',  # Acute parametritis and pelvic cellulitis
        'N73.1',  # Chronic parametritis and pelvic cellulitis
        'N73.2',  # Unspecified parametritis and pelvic cellulitis
        'N73.3',  # Female acute pelvic peritonitis
        'N73.4',  # Female chronic pelvic peritonitis
        'N73.5',  # Female pelvic peritonitis, unspecified
        'N73.6',  # Female pelvic peritoneal adhesions (postinfective)
        'N73.8',  # Other specified female pelvic inflammatory diseases
        'N73.9',  # Female pelvic inflammatory disease, unspecified
        'N74',  # Female pelvic inflammatory disorders in diseases classified elsewhere
        'N75.0',  # Cyst of Bartholin's gland
        'N75.1',  # Abscess of Bartholin's gland
        'N75.8',  # Other diseases of Bartholin's gland
        'N75.9',  # Disease of Bartholin's gland, unspecified
        'N76.0',  # Acute vaginitis
        'N76.1',  # Subacute and chronic vaginitis
        'N76.2',  # Acute vulvitis
        'N76.3',  # Subacute and chronic vulvitis
        'N76.4',  # Abscess of vulva
        'N76.5',  # Ulceration of vagina
        'N76.6',  # Ulceration of vulva
        'N76.81',  # Mucositis (ulcerative) of vagina and vulva
        'N76.89',  # Other specified inflammation of vagina and vulva
        'N77.0',  # Ulceration of vulva in diseases classified elsewhere
        'N77.1',  # Vaginitis, vulvitis and vulvovaginitis in diseases classified elsewhere
        'Z20.2',  # Contact with and (suspected) exposure to infections with a predominantly sexual mode of transmission
        'Z22.4',  # Carrier of infections with a predominantly sexual mode of transmission
    }
    SNOMEDCT = {
        '151004',  # Gonococcal meningitis (disorder)
        '181007',  # Hemorrhagic bronchopneumonia (disorder)
        '192008',  # Congenital syphilitic hepatomegaly (disorder)
        '714002',  # Abrasion and/or friction burn of toe with infection (disorder)
        '801006',  # Nonvenomous insect bite of foot with infection (disorder)
        '827006',  # Late congenital syphilis, latent (positive serology - cerebrospinal fluid, 2 years OR more) (disorder)
        '1085006',  # Candidiasis of vulva (disorder)
        '1102005',  # Intraerythrocytic parasitosis caused by Nuttallia (disorder)
        '1107004',  # Early latent syphilis, positive serology, negative cerebrospinal fluid, with relapse after treatment (disorder)
        '1282001',  # Perichondritis of larynx (disorder)
        '1515008',  # Gorham's disease (disorder)
        '2129002',  # Edema of pharynx (disorder)
        '2390000',  # Acute gonococcal vulvovaginitis (disorder)
        '2435008',  # Ascariasis (disorder)
        '2513003',  # Tinea capitis caused by Trichophyton (disorder)
        '2523007',  # Salmonella pneumonia (disorder)
        '2622007',  # Infected ulcer of skin (disorder)
        '3261002',  # Migratory osteolysis (disorder)
        '3464007',  # Infection caused by Oesophagostomum dentatum (disorder)
        '3487004',  # Candidiasis of lung (disorder)
        '3589003',  # Syphilitic pericarditis (disorder)
        '3752003',  # Trichuriasis (disorder)
        '4016003',  # Empyema of mastoid (disorder)
        '4082005',  # Syphilitic myocarditis (disorder)
        '4324001',  # Subacute cystitis (disorder)
        '4340003',  # Acrodermatitis atrophicans chronica (disorder)
        '4359001',  # Early congenital syphilis (less than 2 years) (disorder)
        '4483005',  # Syphilitic punched out ulcer (disorder)
        '4499004',  # Disseminated anergic leishmaniasis (disorder)
        '4669001',  # Pinta, primary lesion (disorder)
        '4834000',  # Typhoid fever (disorder)
        '5085001',  # Gonococcemia (disorder)
        '5093001',  # Cystitis with actinomycosis (disorder)
        '5168000',  # Sporotrichotic chancre (disorder)
        '5441008',  # Tinea capitis (disorder)
        '5490008',  # Tinea barbae caused by Trichophyton verrucosum (disorder)
        '5765005',  # Infection caused by Leishmania tropica minor (disorder)
        '6020002',  # Tinea pedis (disorder)
        '6267005',  # Congenital syphilitic meningitis (disorder)
        '6365004',  # Cutaneous actinomycosis (disorder)
        '7063008',  # Gangrenous pneumonia (disorder)
        '7138005',  # Caries of mastoid (disorder)
        '7448003',  # Acute pyonephrosis with renal medullary necrosis (disorder)
        '7493006',  # Infection caused by Plagiorchis (disorder)
        '7548000',  # Rheumatic pneumonia (disorder)
        '7877005',  # Infection caused by Diphyllobothrium latum (disorder)
        '8098009',  # Sexually transmitted infectious disease (disorder)
        '8549006',  # Desquamative interstitial pneumonia (disorder)
        '8555001',  # Syphilis of lung (disorder)
        '8748000',  # Abrasion and/or friction burn of hip with infection (disorder)
        '8776008',  # Amebic cystitis (disorder)
        '9091006',  # Gonococcal iridocyclitis (disorder)
        '9241004',  # Gonococcal heart disease (disorder)
        '9480005',  # Viral papular dermatitis (disorder)
        '9713002',  # Prostatitis (disorder)
        '9723006',  # Hyperphosphatasemia with bone disease (disorder)
        '9802006',  # Abrasion and/or friction burn of buttock with infection (disorder)
        '9866007',  # Infection caused by Trichuris vulpis (disorder)
        '9889004',  # Tinea barbae caused by Trichophyton violaceum (disorder)
        '9941009',  # Congenital syphilitic choroiditis (disorder)
        '10345003',  # Primary syphilis of tonsils (disorder)
        '10662001',  # Nonvenomous insect bite of face with infection (disorder)
        '10671005',  # Blister of scapular region with infection (disorder)
        '10679007',  # Infection caused by Giardia lamblia (disorder)
        '10920005',  # Abrasion and/or friction burn of lip with infection (disorder)
        '10995009',  # Abrasion and/or friction burn of scapular region with infection (disorder)
        '11251000',  # Irradiation cystitis (disorder)
        '11258006',  # Tinea barbae caused by Trichophyton mentagrophytes (disorder)
        '11338007',  # Syphilitic episcleritis (disorder)
        '11625007',  # Torus mandibularis (disorder)
        '11840006',  # Traveler's diarrhea (disorder)
        '11906007',  # Chronic gonococcal vulvovaginitis (disorder)
        '12008008',  # Palmar papilloma of yaws (disorder)
        '12232008',  # Syphilitic aneurysm of aorta (disorder)
        '12373006',  # Chronic gonococcal bartholinitis (disorder)
        '12435008',  # Abrasion and/or friction burn of anus with infection (disorder)
        '12463005',  # Infectious gastroenteritis (disorder)
        '12602008',  # Non-hereditary acrodermatitis enteropathica (disorder)
        '12804006',  # Tinea cruris caused by Trichophyton rubrum (disorder)
        '13087006',  # Necrosis of larynx (disorder)
        '13095005',  # Primary symptomatic early syphilis (disorder)
        '13125003',  # Chronic myometritis (disorder)
        '13177009',  # Cellulitis of nasopharynx (disorder)
        '13285005',  # Cystitis cystica (disorder)
        '13310005',  # Taboparesis (disorder)
        '13338002',  # Verruca palmaris (disorder)
        '13464007',  # Infestation caused by Cheyletiella parasitivorax (disorder)
        '13471002',  # Infection caused by Trichuris ovis (disorder)
        '13731006',  # Secondary symptomatic early syphilis (disorder)
        '14096004',  # Nonvenomous insect bite of scapular region with infection (disorder)
        '14386001',  # Indeterminate leprosy (disorder)
        '14560005',  # Dermal mycosis (disorder)
        '14948001',  # Acute suppurative otitis media without spontaneous rupture of ear drum (disorder)
        '15033003',  # Peritonsillar abscess (disorder)
        '15034009',  # Nonvenomous insect bite of breast with infection (disorder)
        '15207003',  # Zoonotic form of cutaneous leishmaniasis (disorder)
        '15470004',  # Toxic diffuse goiter with acropachy (disorder)
        '15628003',  # Gonorrhea (disorder)
        '15634005',  # Palmar hyperkeratosis of yaws (disorder)
        '15845006',  # Buruli ulcer (disorder)
        '16036000',  # Acute empyema of frontal sinus (disorder)
        '16070004',  # Syphilitic cirrhosis (disorder)
        '16358007',  # Chronic disease of tonsils AND/OR adenoids (disorder)
        '16429005',  # Cutaneous larva migrans caused by Ancylostoma braziliense (disorder)
        '16442005',  # Leontiasis ossium (disorder)
        '16681000',  # Candidal otitis externa (disorder)
        '17039000',  # Blister of abdominal wall with infection (disorder)
        '17198008',  # Nonvenomous insect bite of ear with infection (disorder)
        '17250009',  # Blister of interscapular region with infection (disorder)
        '17273001',  # Mucocutaneous cryptococcosis (disorder)
        '17305005',  # Acute gonorrhea of genitourinary tract (disorder)
        '17384006',  # Nonvenomous insect bite of toe with infection (disorder)
        '17723000',  # Infection caused by Naegleria (disorder)
        '17794004',  # Recurrent cutaneous leishmaniasis (disorder)
        '18081009',  # Enteric campylobacteriosis (disorder)
        '18099001',  # Retropharyngeal abscess (disorder)
        '18229003',  # Suppurative colitis (disorder)
        '18849004',  # Infection caused by Trichostrongylus axei (disorder)
        '18900005',  # Periporitis (disorder)
        '18901009',  # Cutaneous diphtheria (disorder)
        '19108007',  # Nonvenomous insect bite of trunk with infection (disorder)
        '19138001',  # Epidermodysplasia verruciformis (disorder)
        '19206003',  # Syphilitic acoustic neuritis (disorder)
        '19213003',  # Infectious diarrheal disease (disorder)
        '19290004',  # Clutton's joints (disorder)
        '19362000',  # Cutaneous larva migrans (disorder)
        '19824006',  # Infection of skin and/or subcutaneous tissue (disorder)
        '19905009',  # Chronic prostatitis (disorder)
        '20325007',  # Tinea barbae caused by Microsporum (disorder)
        '20547008',  # Ill-defined intestinal infection (disorder)
        '20735004',  # Syphilitic aortitis (disorder)
        '20860008',  # Infestation of intestine caused by fly larvae (disorder)
        '20943002',  # Acute gonococcal cervicitis (disorder)
        '21061004',  # Babesiosis (disorder)
        '21523006',  # Syphilitic gumma of central nervous system (disorder)
        '21560005',  # Lepromatous leprosy (disorder)
        '21955004',  # Trichomonal cervicitis (disorder)
        '22064009',  # Pinta (disorder)
        '22277006',  # Blister of groin with infection (disorder)
        '22386003',  # Syphilitic optic atrophy (disorder)
        '22500005',  # Infection by Oesophagostomum (disorder)
        '22557000',  # Bullous cystitis (disorder)
        '22754005',  # Staphylococcal pneumonia (disorder)
        '22905009',  # Infection caused by Heterophyes heterophyes (disorder)
        '22922006',  # Infection caused by Physaloptera (disorder)
        '23166004',  # Cellulitis of pharynx (disorder)
        '23191004',  # Early yaws (disorder)
        '23196009',  # Frambeside of early yaws (disorder)
        '23502006',  # Lyme disease (disorder)
        '23550005',  # Syphilis of bursa (disorder)
        '23754003',  # Calculous pyelonephritis (disorder)
        '23874000',  # Chronic amebiasis (disorder)
        '23949002',  # Capillaria philippinensis infection (disorder)
        '23975003',  # Chronic gonococcal seminal vesiculitis (disorder)
        '24347001',  # Cellulitis of vocal cords (disorder)
        '24620004',  # Hemolytic anemia caused by babesiosis (disorder)
        '24752008',  # Infantile cortical hyperostosis (disorder)
        '24789006',  # Viral gastroenteritis caused by Norwalk-like agent (disorder)
        '24868007',  # Acute gonococcal cystitis (disorder)
        '25055007',  # Ingrowing nail with infection (disorder)
        '25188002',  # Pediculosis corporis (disorder)
        '25626000',  # Cutaneous infectious disease caused by Mycobacteria (disorder)
        '25726008',  # Nonvenomous insect bite of groin with infection (disorder)
        '25956006',  # Tinea pedis caused by Trichophyton (disorder)
        '25994006',  # Bone island (disorder)
        '26039008',  # Neurosyphilis (disorder)
        '26067004',  # Madura foot caused by Streptomyces (disorder)
        '26135000',  # Syphilitic encephalitis (disorder)
        '26151008',  # Graves' disease with acropachy AND with thyrotoxic crisis (disorder)
        '26249004',  # Intestinal infection caused by Class Cestoda and/or Class Trematoda and/or Phylum Nemata (disorder)
        '26535009',  # Carbuncle of vulva (disorder)
        '26652002',  # Abrasion and/or friction burn of scalp with infection (disorder)
        '26826005',  # Amebic appendicitis (disorder)
        '27023008',  # Infective dermatitis of eyelid (disorder)
        '27075004',  # Venereal disease in mother complicating pregnancy, childbirth AND/OR puerperium (disorder)
        '27278006',  # Acute empyema of sphenoidal sinus (disorder)
        '27460003',  # Primary syphilis of breast (disorder)
        '27648007',  # Congenital syphilitic periostitis (disorder)
        '27681008',  # Chronic gonorrhea (disorder)
        '27836007',  # Pertussis (disorder)
        '27858009',  # Clostridial gastroenteritis (disorder)
        '28198007',  # Primary syphilis of lip (disorder)
        '28438004',  # Gonococcal conjunctivitis neonatorum (disorder)
        '28572009',  # Chronic gonorrhea of genitourinary tract (disorder)
        '28709001',  # Cellulitis of larynx (disorder)
        '28867007',  # Granuloma inguinale (disorder)
        '28951000',  # Infection caused by Trichuris suis (disorder)
        '29604006',  # Infection caused by Trichostrongylus colubriformis (disorder)
        '29864006',  # Acute gonococcal urethritis (disorder)
        '29979000',  # Infection caused by Enteromonas hominis (disorder)
        '30168008',  # Acute gonococcal epididymo-orchitis (disorder)
        '30242009',  # Scarlet fever (disorder)
        '30283007',  # Initial frambesial ulcer (disorder)
        '30358009',  # Acquired stenosis of external ear canal secondary to surgery (disorder)
        '30493003',  # Intestinal infection caused by Proteus mirabilis (disorder)
        '30595008',  # Acute necrotizing cutaneous leishmaniasis (disorder)
        '30716008',  # Infection caused by Oesophagostomum columbianum (disorder)
        '30818003',  # Nonvenomous insect bite of perineum with infection (disorder)
        '31015008',  # Primary anal syphilis (disorder)
        '31137003',  # Early latent syphilis, positive serology, negative cerebrospinal fluid, less than 2 years after infection (disorder)
        '31201001',  # Knight's disease (disorder)
        '31487001',  # Disseminated idiopathic skeletal hyperostosis (disorder)
        '31604004',  # Blister of flank with infection (disorder)
        '31772005',  # Cicatricial skin lesions of pinta (disorder)
        '31999004',  # Chronic gonococcal endometritis (disorder)
        '32204007',  # Pulmonary actinobacillosis (disorder)
        '32286006',  # Pneumonia in Q fever (disorder)
        '32298001',  # Intestinal infection caused by Trichomonas vaginalis (disorder)
        '32527003',  # Staphylococcal enterocolitis (disorder)
        '32580004',  # Enterovirus enteritis (disorder)
        '32735002',  # Congenital syphilitic encephalitis (disorder)
        '33042004',  # Parasitic skin infestation (disorder)
        '33090006',  # Yabapox (disorder)
        '33155001',  # Nonvenomous insect bite of flank with infection (disorder)
        '33261009',  # Abscess of tonsil (disorder)
        '33655002',  # Chronic cystitis (disorder)
        '33666009',  # Black piedra (disorder)
        '33710003',  # Trichostrongyliasis (disorder)
        '33924006',  # Abscess of larynx (disorder)
        '33934002',  # Hemorrhagic otitis externa (disorder)
        '33937009',  # Lyme arthritis (disorder)
        '34020007',  # Pneumonia caused by Streptococcus (disorder)
        '34240005',  # Infection caused by Diphyllobothrium pacificum (disorder)
        '34242002',  # Syphilis in mother complicating pregnancy, childbirth AND/OR puerperium (disorder)
        '34730008',  # Primate erythroparvovirus 1 infection (disorder)
        '34786008',  # Candidal proctitis (disorder)
        '35037009',  # Primary atypical interstitial pneumonia (disorder)
        '35057008',  # Nonvenomous insect bite of penis with infection (disorder)
        '35089004',  # Urogenital infection caused by Trichomonas vaginalis (disorder)
        '35168006',  # Acute empyema of ethmoidal sinus (disorder)
        '35339003',  # Primary pneumonic plague (disorder)
        '35526001',  # Chronic gonococcal epididymo-orchitis (disorder)
        '35586003',  # White piedra (disorder)
        '35742006',  # Congenital syphilis (disorder)
        '35876006',  # Gonococcal infection of eye (disorder)
        '35923002',  # Chronic maxillary sinusitis (disorder)
        '35974005',  # Cryptococcal gastroenteritis (disorder)
        '36001007',  # Capillaria hepatica infection (disorder)
        '36038007',  # Goundou of yaws (disorder)
        '36139001',  # Chronic ulcerating interstitial cystitis (disorder)
        '36188001',  # Shigellosis (disorder)
        '36276008',  # Syphilitic retrobulbar neuritis (disorder)
        '36292003',  # Otitis externa caused by Herpes zoster (disorder)
        '36334003',  # Hymenolepis diminuta infection (disorder)
        '36529003',  # Intestinal infection caused by Morganella morganii (disorder)
        '36689008',  # Acute pyelonephritis (disorder)
        '36789003',  # Acute infective gastroenteritis (disorder)
        '36866003',  # Mucocutaneous-lymphangitic paracoccidioidomycosis (disorder)
        '37160003',  # Pintid (disorder)
        '37426002',  # Ulcerative laryngitis (disorder)
        '37430004',  # Syphilis of synovium (disorder)
        '37702000',  # Hereditary acrodermatitis enteropathica (disorder)
        '37722001',  # Ulceroglandular tularemia (disorder)
        '37754005',  # Asymptomatic neurosyphilis (disorder)
        '37776001',  # Abrasion and/or friction burn of perineum with infection (disorder)
        '37779008',  # Pyeloureteritis cystica (disorder)
        '37832003',  # Infection caused by Metagonimus yokogawai (disorder)
        '37887002',  # Nonvenomous insect bite of neck with infection (disorder)
        '38189003',  # Nonvenomous insect bite of ankle with infection (disorder)
        '38302000',  # Infection caused by Echinostoma lindoense (disorder)
        '38523005',  # Syphilitic parkinsonism (disorder)
        '38573008',  # Diffuse cutaneous leishmaniasis (disorder)
        '38699009',  # Pneumonia caused by Histoplasma capsulatum (disorder)
        '38700005',  # Madura foot caused by Actinomadura pelletieri (disorder)
        '38822007',  # Cystitis (disorder)
        '38961000',  # Chronic sphenoidal sinusitis (disorder)
        '38976008',  # Pneumonic plague (disorder)
        '39084003',  # Tinea cruris caused by Epidermophyton floccosum (disorder)
        '39085002',  # Secondary syphilis of mucous membrane (disorder)
        '39172002',  # Pneumonia caused by Proteus mirabilis (disorder)
        '39341005',  # Infectious colitis (disorder)
        '39439008',  # Wet form of cutaneous leishmaniasis (disorder)
        '39524009',  # Cercarial dermatitis of freshwater mammalian type (disorder)
        '39629007',  # Pyogenic granuloma of skin (disorder)
        '39747007',  # Enteritis necroticans (disorder)
        '39963006',  # Toddler diarrhea (disorder)
        '40055000',  # Chronic sinusitis (disorder)
        '40070004',  # Infection caused by Molluscum contagiosum (disorder)
        '40149008',  # Gonococcal keratitis (disorder)
        '40635000',  # Cystitis caused by Pseudomonas (disorder)
        '40786001',  # Aspiration pneumonia caused by regurgitated gastric secretions (disorder)
        '41080008',  # Cutaneous yaws (less than 5 years) (disorder)
        '41116009',  # Spirochetal infection (disorder)
        '41235001',  # Dry form of cutaneous leishmaniasis (disorder)
        '41381004',  # Pneumonia caused by Pseudomonas (disorder)
        '41545003',  # Whipple's disease (disorder)
        '41804000',  # Blister of axilla with infection (disorder)
        '41958008',  # Nonvenomous insect bite of back with infection (disorder)
        '42004004',  # Aspiration pneumonia caused by inhalation of vomitus (disorder)
        '42032008',  # Nonvenomous insect bite of abdominal wall with infection (disorder)
        '42089007',  # Nonvenomous insect bite of anus with infection (disorder)
        '42402006',  # Kartagener syndrome (disorder)
        '42640003',  # Suppurative appendicitis (disorder)
        '42680007',  # Endogenous lipoid pneumonia (disorder)
        '42746002',  # Gonorrhea of rectum (disorder)
        '42770003',  # Syphilis of tricuspid valve (disorder)
        '42849000',  # Trypanosomal chancre (disorder)
        '43017006',  # Infection caused by Gastrodiscoides hominis (disorder)
        '43021004',  # Periungual wart (disorder)
        '43114002',  # Chronic trigonitis (disorder)
        '43373009',  # Tinea barbae caused by Microsporum canis (disorder)
        '43581009',  # Tinea pedis caused by Epidermophyton (disorder)
        '43591003',  # Exanthematous infectious disease (disorder)
        '44274007',  # Lymphoid interstitial pneumonia (disorder)
        '44352004',  # Nonvenomous insect bite of hand with infection (disorder)
        '44412000',  # Chronic gonococcal urethritis (disorder)
        '44464006',  # Postpartum AND/OR puerperal erysipelas (disorder)
        '44549008',  # Aspiration pneumonia resulting from a procedure (disorder)
        '44568006',  # Syphilitic interstitial keratitis (disorder)
        '44653001',  # Erysipelas (disorder)
        '44697002',  # Melorheostosis (disorder)
        '44743006',  # Gonococcal infection of joint (disorder)
        '44917000',  # Hymenolepiasis (disorder)
        '45058001',  # Syphilis of pulmonary valve (disorder)
        '45072005',  # Acne necrotica miliaris (disorder)
        '45312009',  # Pneumonia in typhoid fever (disorder)
        '45377007',  # Acute gonococcal salpingitis (disorder)
        '45556008',  # Pulmonary tularemia (disorder)
        '45581006',  # Abrasion and/or friction burn of breast with infection (disorder)
        '45629003',  # Abscess of nasopharynx (disorder)
        '45666009',  # Blister of ear with infection (disorder)
        '45816000',  # Pyelonephritis (disorder)
        '45915002',  # Nonvenomous insect bite of upper arm with infection (disorder)
        '46080009',  # Abrasion and/or friction burn of abdominal wall with infection (disorder)
        '46235002',  # Early latent congenital syphilis, positive serology, negative spinal fluid (disorder)
        '46672003',  # Helicobacter-associated proctitis (disorder)
        '46699001',  # Bursitis caused by Neisseria gonorrhoeae (disorder)
        '46727001',  # Disorder of zinc metabolism (disorder)
        '46752004',  # Torus palatinus (disorder)
        '46799006',  # Dysenteric diarrhea (disorder)
        '46970008',  # Pneumonia caused by Mycoplasma pneumoniae (disorder)
        '46999003',  # Tinea barbae caused by Trichophyton (disorder)
        '47027001',  # Nonvenomous insect bite of scrotum with infection (disorder)
        '47277009',  # Izumi fever (disorder)
        '47306003',  # Lobomycosis (disorder)
        '47452006',  # Variola major (disorder)
        '47807003',  # Cutaneous habronemiasis (disorder)
        '47941007',  # Purulent enteritis (disorder)
        '48278001',  # Diphtheritic cystitis (disorder)
        '48631008',  # Pyonephrosis (disorder)
        '48971001',  # Tinea manus (disorder)
        '48982009',  # Bannwarth syndrome (disorder)
        '49442000',  # Initial lesion of yaws (disorder)
        '49454000',  # Abrasion and/or friction burn of flank with infection (disorder)
        '49524007',  # Abrasion and/or friction burn of groin with infection (disorder)
        '49649001',  # Infection caused by Acanthamoeba (disorder)
        '49882001',  # Viral exanthem (disorder)
        '49883006',  # Candidiasis of skin (disorder)
        '49894005',  # Trichomycosis axillaris (disorder)
        '49923008',  # Syphilis of tendon (disorder)
        '50124004',  # Nonvenomous insect bite of buttock with infection (disorder)
        '50242001',  # Tinea barbae caused by Microsporum gypseum (disorder)
        '50293005',  # Abrasion and/or friction burn of penis with infection (disorder)
        '50414000',  # Late cutaneous leishmaniasis (disorder)
        '50528008',  # Syphilis of bone (disorder)
        '50804000',  # Catarrhal pneumonia (disorder)
        '50970007',  # Acute gonorrhea of upper genitourinary tract (disorder)
        '51152000',  # Abscess of vocal cords (disorder)
        '51212009',  # Pitted keratolysis (disorder)
        '51254007',  # Paratyphoid C fever (disorder)
        '51423006',  # Modified smallpox (disorder)
        '51530003',  # Pneumonia caused by Escherichia coli (disorder)
        '51613008',  # Infection caused by Babesia divergens (disorder)
        '51817006',  # Infection caused by Acanthamoeba castellani (disorder)
        '51882009',  # Nonvenomous insect bite of interscapular region with infection (disorder)
        '51928006',  # General paresis - neurosyphilis (disorder)
        '51960003',  # Secondary syphilis of pharynx (disorder)
        '52123000',  # Emphysematous cystitis (disorder)
        '52281009',  # Meningoencephalitis caused by Naegleria (disorder)
        '52404001',  # Mastoiditis (disorder)
        '52414005',  # Secondary syphilis of vulva (disorder)
        '52705001',  # Nonvenomous insect bite of finger with infection (disorder)
        '52914002',  # Anal candidiasis (disorder)
        '52918004',  # Infection caused by Echinochasmus (disorder)
        '52979002',  # Capillaria infection (disorder)
        '53084003',  # Bacterial pneumonia (disorder)
        '53243005',  # Hyperchromic lesions of pinta (disorder)
        '53316003',  # Otomycosis (disorder)
        '53529004',  # Chronic gonococcal salpingitis (disorder)
        '53664003',  # Gonococcal spondylitis (disorder)
        '53734002',  # Hyperkeratosis of yaws (disorder)
        '53991003',  # Cystitis with bilharziasis (disorder)
        '54026009',  # Cystitis with Echinococcus infestation (disorder)
        '54043007',  # Cercarial dermatitis of freshwater avian type (disorder)
        '54069001',  # Congenital syphilitic mucous patches (disorder)
        '54266002',  # Infection caused by Fasciolopsis buski (disorder)
        '54274001',  # Primary syphilis of fingers (disorder)
        '54275000',  # Chronic metritis (disorder)
        '54385001',  # Exanthema subitum (disorder)
        '54825009',  # Acute gonorrhea of lower genitourinary tract (disorder)
        '55023005',  # Amebic nondysenteric colitis (disorder)
        '55184003',  # Infectious enteritis (disorder)
        '55260003',  # Calcaneal spur (finding)
        '55679008',  # Peribronchial pneumonia (disorder)
        '55768006',  # Syphilitic leukoderma (disorder)
        '55799007',  # Tinea capitis caused by Microsporum (disorder)
        '56140006',  # Progressive vaccina (disorder)
        '56200009',  # Infection caused by Trichuris globulosa (disorder)
        '56892000',  # Nonvenomous insect bite of chest wall with infection (disorder)
        '56978007',  # Generalized vaccinia (disorder)
        '57468008',  # Salpingo-oophoritis following molar AND/OR ectopic pregnancy (disorder)
        '57540006',  # Intestinal nematode infection (disorder)
        '57702005',  # Unresolved pneumonia (disorder)
        '57725006',  # Balantidiasis (disorder)
        '58056005',  # Syphilis of mitral valve (disorder)
        '58213005',  # Bacillary angiomatosis (disorder)
        '58227000',  # Secondary syphilis of viscera (disorder)
        '58392004',  # Congenital syphilitic osteochondritis (disorder)
        '58695007',  # Chromoblastomycosis caused by Fonsecaea compacta (disorder)
        '58763001',  # Acute empyema of nasal sinus (disorder)
        '59121004',  # Infection caused by Leishmania aethiopica (disorder)
        '59158004',  # Infection caused by Oesophagostomum stephanostomum (disorder)
        '59233003',  # Secondary syphilis of skin (disorder)
        '59307008',  # Syphilitic alopecia (disorder)
        '59443003',  # Nonvenomous insect bite of thigh with infection (disorder)
        '59475000',  # Pneumonia in pertussis (disorder)
        '59530001',  # Syphilis of kidney (disorder)
        '59707005',  # Abscess of pharynx (disorder)
        '59721007',  # Congenital syphilitic pemphigus (disorder)
        '59763006',  # Hyperphosphatasemia tarda (disorder)
        '59934002',  # Secondary syphilitic adenopathy (disorder)
        '60100005',  # Cyst of pharynx (disorder)
        '60130002',  # Chronic frontal sinusitis (disorder)
        '60298004',  # Nonvenomous insect bite of forearm with infection (disorder)
        '60335002',  # Gonococcal keratosis (disorder)
        '60416001',  # Edema of nasopharynx (disorder)
        '60485005',  # Pleurobronchopneumonia (disorder)
        '60506006',  # Blister of scalp with infection (disorder)
        '60528006',  # Secondary syphilis of mouth (disorder)
        '60570001',  # Infection caused by Trichuris trichiura (disorder)
        '60600009',  # Disorder of the larynx (disorder)
        '60867007',  # Klebsiella cystitis (disorder)
        '60893000',  # Chronic gonococcal prostatitis (disorder)
        '61048000',  # Gonococcal endocarditis (disorder)
        '61099007',  # Rubeola scarlatinosa (disorder)
        '61251002',  # Plantar hyperkeratosis of yaws (disorder)
        '61341000',  # Abrasion and/or friction burn of thigh with infection (disorder)
        '61383005',  # Abrasion and/or friction burn of ear with infection (disorder)
        '61500009',  # Granulomatous prostatitis (disorder)
        '61612001',  # Syphilitic aortic incompetence (disorder)
        '61624008',  # Chronic recurrent streptococcal erysipelas (disorder)
        '61884008',  # Achromobacter pneumonia (disorder)
        '62078006',  # Infection caused by Hypodereum (disorder)
        '62199007',  # Acute trigonitis (disorder)
        '62207008',  # Syphilitic ostial coronary disease (disorder)
        '62558006',  # Dyschromic skin lesions of pinta (disorder)
        '62861003',  # Secondary syphilis, relapse (untreated) (disorder)
        '63302006',  # Chronic pyelonephritis (disorder)
        '63440008',  # Verruca plantaris (disorder)
        '63605003',  # Infection caused by Naegleria gruberi (disorder)
        '63650001',  # Cholera (disorder)
        '63717008',  # Exostosis of orbit (disorder)
        '63751007',  # Secondary syphilis of bone (disorder)
        '63757006',  # Osler's node (disorder)
        '63848007',  # Abrasion and/or friction burn of wrist with infection (disorder)
        '64102008',  # Syphilitic gumma (disorder)
        '64479007',  # Pneumonia caused by Klebsiella pneumoniae (disorder)
        '64650008',  # Cutaneous nocardiosis (disorder)
        '64667001',  # Interstitial pneumonia (disorder)
        '64703005',  # Terminal bronchopneumonia (disorder)
        '64831005',  # Abrasion and/or friction burn of shoulder with infection (disorder)
        '64956000',  # Follicular cystitis (disorder)
        '65049003',  # Acute gonococcal seminal vesiculitis (disorder)
        '65141002',  # Foreign body pneumonia (disorder)
        '65295003',  # Acute gonococcal endometritis (disorder)
        '65332001',  # Diverticulitis of bladder (disorder)
        '65696007',  # Coccidioidal granuloma (disorder)
        '65984002',  # Salpingitis follicularis (disorder)
        '66160001',  # Cryptosporidial gastroenteritis (disorder)
        '66281009',  # Secondary syphilitic chorioretinitis (disorder)
        '66371007',  # Abrasion and/or friction burn of upper arm with infection (disorder)
        '66379009',  # Bacterial overgrowth syndrome (disorder)
        '66429007',  # Unresolved lobar pneumonia (disorder)
        '66578000',  # Abrasion and/or friction burn of cheek with infection (disorder)
        '66887000',  # Late syphilis, latent (positive serology, negative cephalospinal fluid 2 years after) (disorder)
        '67077008',  # Abrasion and/or friction burn of axilla with infection (disorder)
        '67125004',  # Latent syphilis with positive serology (disorder)
        '67167008',  # Urethrotrigonitis (disorder)
        '67391006',  # Syphilitic endocarditis (disorder)
        '67525007',  # Secondary pneumonic plague (disorder)
        '67624004',  # Abscess of Bartholin's gland (disorder)
        '67685000',  # Prostatocystitis (disorder)
        '67862004',  # Nonvenomous insect bite of elbow with infection (disorder)
        '67896006',  # Post-kala-azar dermal leishmaniasis (disorder)
        '67915005',  # Intestinal trichomoniasis (disorder)
        '67924001',  # Smallpox (disorder)
        '67974004',  # Nonvenomous insect bite of nose with infection (disorder)
        '68202005',  # Intermediate lesion of pinta (disorder)
        '68226007',  # Acute cystitis (disorder)
        '68389001',  # Mucopurulent vaginitis (disorder)
        '68409003',  # Organized pneumonia (disorder)
        '68421004',  # Epiphyseal arrest (disorder)
        '68566005',  # Urinary tract infectious disease (disorder)
        '68764005',  # Juvenile taboparesis (disorder)
        '68863007',  # Acute syphilitic meningitis (disorder)
        '69013005',  # Cutaneous draschiasis (disorder)
        '69163003',  # Taenia saginata infection (disorder)
        '69588003',  # Erythema chronica migrans (disorder)
        '69595007',  # Secondary syphilitic periostitis (disorder)
        '69810009',  # Acute hemorrhagic colitis caused by Escherichia coli (disorder)
        '70036007',  # Haemophilus influenzae pneumonia (disorder)
        '70143003',  # Tuberculoid leprosy (disorder)
        '70413005',  # Dermatophilosis caused by Dermatophilus congolensis (disorder)
        '70880006',  # Adenoviral enteritis (disorder)
        '71011005',  # Infestation caused by Phthirus pubis (disorder)
        '71085009',  # Paratyphoid B fever (disorder)
        '71104009',  # Achromic AND hyperchromic skin lesions of pinta (disorder)
        '71186008',  # Croup (disorder)
        '71418005',  # Cyst of nasopharynx (disorder)
        '71450000',  # Contact ulcer of vocal folds (disorder)
        '71572001',  # Infection caused by Enteromonas (disorder)
        '71577007',  # Deformity of eyelid due to infective dermatitis of eyelid (disorder)
        '71590000',  # Trichomonal prostatitis (disorder)
        '71926009',  # Infective pneumonia acquired prenatally (disorder)
        '72074009',  # Infection caused by Trichostrongylus orientalis (disorder)
        '72083004',  # Late syphilis (disorder)
        '72222004',  # Papillary cystitis (disorder)
        '72225002',  # Urethral stricture due to gonococcal infection (disorder)
        '72294005',  # Alastrim (disorder)
        '72463008',  # Chancriform syndrome (disorder)
        '72512006',  # Abrasion and/or friction burn of nose with infection (disorder)
        '72605008',  # Candidal vulvovaginitis (disorder)
        '72656004',  # Granulomatous pneumonia (disorder)
        '72815004',  # Cystitis glandularis (disorder)
        '72854003',  # Aspiration pneumonia due to near drowning (disorder)
        '72934000',  # Candidiasis of vagina (disorder)
        '73237007',  # Chronic ethmoidal sinusitis (disorder)
        '73281004',  # Chronic zinc deficiency (disorder)
        '73375005',  # Ulcer of larynx (disorder)
        '73532000',  # Cutaneous larva migrans caused by Ancylostoma caninum (disorder)
        '73594001',  # Pinta, late lesion (disorder)
        '73908004',  # Infection caused by Babesia microti (disorder)
        '74372003',  # Gonorrhea of pharynx (disorder)
        '74445007',  # Trigonitis (disorder)
        '74578003',  # Pyoderma gangrenosum (disorder)
        '74621002',  # Epidemic vomiting syndrome (disorder)
        '74724001',  # Smallpox without rash (disorder)
        '75073007',  # Malakoplakia of bladder (disorder)
        '75169006',  # Infection caused by Oesophagostomum apiostomum (disorder)
        '75299005',  # Spastic spinal syphilitic paralysis (disorder)
        '75375008',  # Bacterial enteritis (disorder)
        '75426006',  # Aspiration pneumonia caused by regurgitated food (disorder)
        '75498004',  # Acute bacterial sinusitis (disorder)
        '75530006',  # Streptococcal cervicitis (disorder)
        '75656006',  # Ulcerating cutaneous leishmaniasis (disorder)
        '75996005',  # Tinea pedis caused by Epidermophyton floccosum (disorder)
        '76069003',  # Disorder of bone (disorder)
        '76090006',  # Pittsburgh pneumonia (disorder)
        '76122006',  # Hyperkeratosis of pinta (disorder)
        '76272004',  # Syphilis (disorder)
        '76490001',  # Otomycosis externa caused by Fusarium (disorder)
        '76584003',  # Nonvenomous insect bite of hip with infection (disorder)
        '76623002',  # Paratyphoid A fever (disorder)
        '76653009',  # Acute empyema of maxillary sinus (disorder)
        '76802005',  # Chronic gonococcal cervicitis (disorder)
        '76813008',  # Pyogenic granuloma of oral mucosa (disorder)
        '77028001',  # Secondary syphilis of anus (disorder)
        '77412003',  # Hymenolepis nana infection (disorder)
        '77470003',  # Enteroviral exanthematous fever (disorder)
        '77746001',  # Leontiasis (disorder)
        '77798004',  # Systemic chromomycosis (disorder)
        '77863005',  # Lyme carditis (disorder)
        '77874001',  # Nonvenomous insect bite of shoulder with infection (disorder)
        '77939001',  # Syphilitic disseminated retinochoroiditis (disorder)
        '78128001',  # Helicobacter-associated colitis (disorder)
        '78223000',  # Coccidioidomycosis with erythema nodosum (disorder)
        '78360004',  # Abrasion and/or friction burn of ankle with infection (disorder)
        '78420004',  # Viral enteritis (disorder)
        '78487004',  # Granulomatous dermatophytosis (disorder)
        '78895009',  # Congenital pneumonia (disorder)
        '79077003',  # Abrasion and/or friction burn of back with infection (disorder)
        '79340003',  # Abrasion and/or friction burn of interscapular region with infection (disorder)
        '79411002',  # Acute prostatitis (disorder)
        '79739002',  # Infestation caused by Cheyletiella yasguri (disorder)
        '79949009',  # Primary cutaneous coccidioidomycosis (disorder)
        '79974007',  # Cat scratch disease (disorder)
        '80388004',  # Chronic gonorrhea of upper genitourinary tract (disorder)
        '80604007',  # Acute gonococcal bartholinitis (disorder)
        '80645004',  # Chronic mastoiditis (disorder)
        '80770009',  # Secondary syphilis of liver (disorder)
        '80890005',  # Cutaneous sporotrichosis (disorder)
        '81000006',  # Pediculosis capitis (disorder)
        '81020007',  # Cholera caused by Vibrio cholerae El Tor (disorder)
        '81125004',  # Acute hemorrhagic cystitis (disorder)
        '81339006',  # Secondary syphilis of tonsil (disorder)
        '81598001',  # Trichomonal vulvovaginitis (disorder)
        '82054006',  # Hyperostosis interna frontalis (finding)
        '82074004',  # Achromic skin lesions of pinta (disorder)
        '82214002',  # Trench fever (disorder)
        '82323002',  # Late congenital syphilis (2 years OR more) (disorder)
        '82355002',  # Syphilitic aortic stenosis (disorder)
        '82597009',  # Tinea capitis caused by Trichophyton tonsurans (disorder)
        '82930004',  # Intestinal infection caused by Pseudomonas (disorder)
        '82959004',  # Dementia paralytica juvenilis (disorder)
        '83052004',  # Cutaneous screw-worm myiasis (disorder)
        '83070001',  # Friction blister with infection (disorder)
        '83448006',  # Plantar papilloma of yaws (disorder)
        '83587009',  # Nonvenomous insect bite of cheek with infection (disorder)
        '83608006',  # Aspiration pneumonia caused by inhalation of milk (disorder)
        '83883001',  # Cardiovascular syphilis (disorder)
        '83937008',  # Infection caused by Babesia bovis (disorder)
        '84070009',  # Nonvenomous insect bite of axilla with infection (disorder)
        '84204007',  # Abrasion and/or friction burn of scrotum with infection (disorder)
        '84278006',  # Abrasion and/or friction burn of neck with infection (disorder)
        '84753008',  # Pneumonia in systemic mycosis (disorder)
        '84889008',  # Parapharyngeal abscess (disorder)
        '84980006',  # Cutaneous anthrax (disorder)
        '85495007',  # Acute pyonephrosis without renal medullary necrosis (disorder)
        '85857008',  # Secondary syphilis, relapse (treated) (disorder)
        '85904008',  # Paratyphoid fever (disorder)
        '86028001',  # Syphilis of liver (disorder)
        '86279000',  # Acute suppurative otitis media with spontaneous rupture of ear drum (disorder)
        '86333008',  # Subcutaneous phycomycosis caused by Basidiobolus (disorder)
        '86443005',  # Hutchinson's teeth (disorder)
        '86497005',  # Flat-type smallpox (disorder)
        '86615009',  # Epidemic diarrhea (disorder)
        '86981007',  # Infective otitis externa (disorder)
        '87282003',  # Intestinal parasitism (disorder)
        '87318008',  # Congenital syphilis with gumma (disorder)
        '87603003',  # Lymphocutaneous sporotrichosis (disorder)
        '87628006',  # Bacterial infectious disease (disorder)
        '87696004',  # Hemorrhagic cystitis (disorder)
        '88411000',  # Blister of anus with infection (disorder)
        '88558006',  # Cercarial dermatitis of sea water avian type (disorder)
        '88813005',  # Chronic gonococcal cystitis (disorder)
        '88850006',  # Chronic pansinusitis (disorder)
        '88943008',  # Syphilis of muscle (disorder)
        '88981003',  # Pyometra (disorder)
        '89478006',  # Parasitic melanoderma (disorder)
        '89933001',  # Protozoal intestinal disease (disorder)
        '90145001',  # Frontal bossing (disorder)
        '90271007',  # Intestinal infection caused by Arizona group (disorder)
        '90428001',  # Gonococcal pericarditis (disorder)
        '90679001',  # Mixed intestinal infection caused by Class Cestoda and/or Class Trematoda and/or Phylum Nemata (disorder)
        '90886006',  # Infestation of rectum caused by fly larvae (disorder)
        '91060006',  # Nonvenomous insect bite of wrist with infection (disorder)
        '91374005',  # Carbuncle of skin and/or subcutaneous tissue (disorder)
        '91554004',  # Condyloma latum (disorder)
        '95333004',  # Eosinophilic pustular folliculitis (disorder)
        '95340003',  # Nail bed infection (disorder)
        '95374004',  # Application site infection (disorder)
        '95381006',  # Injection site infection (disorder)
        '95547004',  # Ruptured suppurative appendicitis (disorder)
        '95881004',  # Mycetoma of foot (disorder)
        '95889002',  # Mycoplasmal pyelonephritis (disorder)
        '95896000',  # Protozoan infection (disorder)
        '102453009',  # Peritonsillar cellulitis (disorder)
        '105628008',  # Human typhoid AND/OR paratyphoid fever (disorder)
        '105645003',  # Disease caused by Acanthamoebidae (disorder)
        '105656003',  # Disease caused by Enteromonadidae (disorder)
        '105977003',  # Non-infectious pneumonia (disorder)
        '108365000',  # Infection of skin (disorder)
        '109362006',  # Cellulitis of parapharyngeal space (disorder)
        '109436001',  # Moon's molar teeth (disorder)
        '109802000',  # Abscess of uvula of palate (disorder)
        '109909008',  # Acute suppurative mastoiditis (disorder)
        '109910003',  # Chronic suppurative mastoiditis (disorder)
        '111003001',  # Sporotrichotic gumma (disorder)
        '111347003',  # Exostosis of jaw (disorder)
        '111409009',  # Ulcerative cystitis (disorder)
        '111538005',  # Acute mastoiditis with complication (disorder)
        '111705004',  # Blister of trunk with infection (disorder)
        '111707007',  # Abrasion and/or friction burn of chest wall with infection (disorder)
        '111711001',  # Blister of shoulder with infection (disorder)
        '111712008',  # Abrasion and/or friction burn of elbow with infection (disorder)
        '111796005',  # Madura foot caused by Streptomyces somaliensis (disorder)
        '111806005',  # Acute gonococcal prostatitis (disorder)
        '111807001',  # Gonococcal endophthalmia (disorder)
        '111839008',  # Intestinal infection caused by Escherichia coli (disorder)
        '111843007',  # Viral gastroenteritis (disorder)
        '111852003',  # Vaccinia (disorder)
        '111898002',  # Chronic mycotic otitis externa (disorder)
        '111899005',  # Cutaneous aspergillosis (disorder)
        '111900000',  # Pneumonia in aspergillosis (disorder)
        '111901001',  # Bronchocentric granulomatosis (disorder)
        '111911008',  # Chronic intestinal amebiasis without abscess (disorder)
        '111912001',  # Acute amebic dysentery without abscess (disorder)
        '111921000',  # Trichobilharzia cercarial dermatitis (disorder)
        '111923002',  # Gastrodiscoidiasis (disorder)
        '111939009',  # Dysentery (disorder)
        '120639003',  # Hantavirus pulmonary syndrome (disorder)
        '123587001',  # Acute bronchopneumonia (disorder)
        '123588006',  # Confluent bronchopneumonia with abscess formation (disorder)
        '123589003',  # Necrotizing bronchopneumonia (disorder)
        '123590007',  # Focal pneumonia (disorder)
        '123591006',  # Confluent pneumonia (disorder)
        '123611003',  # Subacute pyelonephritis (disorder)
        '123754002',  # Focal pyelonephritis (disorder)
        '123755001',  # Diffuse pyelonephritis (disorder)
        '123812008',  # Zinc excess (disorder)
        '128936008',  # Bacterial infection of skin (disorder)
        '128937004',  # Viral infection of skin (disorder)
        '128938009',  # Parasitic infection of skin (disorder)
        '128939001',  # Infection of subcutaneous tissue (disorder)
        '129451001',  # Respiratory bronchiolitis associated interstitial lung disease (disorder)
        '129452008',  # Nonspecific interstitial pneumonia (disorder)
        '129668006',  # Ulcer caused by Treponema vincentii (disorder)
        '185367005',  # Microsporosis nigra (disorder)
        '186090001',  # Typhoid and paratyphoid fevers (disorder)
        '186105003',  # Shigella boydii (group C) (disorder)
        '186116005',  # Acute amebic dysentery (disorder)
        '186150001',  # Enteritis caused by rotavirus (disorder)
        '186289000',  # Tinea nigra (disorder)
        '186357007',  # Streptococcal sore throat with scarlatina (disorder)
        '186394003',  # Actinomycosis of cecum (disorder)
        '186397005',  # Madura foot - actinomycotic (disorder)
        '186431008',  # Clostridioides difficile infection (disorder)
        '186525007',  # Herpes zoster with dermatitis of eyelid (disorder)
        '186535001',  # Eczema herpeticum (disorder)
        '186668002',  # Gammaherpesviral mononucleosis (disorder)
        '186807008',  # Asian desert cutaneous leishmaniasis (disorder)
        '186825001',  # Systemic bartonellosis (disorder)
        '186826000',  # Cutaneous and mucocutaneous bartonellosis (disorder)
        '186833000',  # Early congenital syphilis - latent (disorder)
        '186842007',  # Late congenital syphilitic oculopathy (disorder)
        '186846005',  # Early symptomatic syphilis (disorder)
        '186847001',  # Primary genital syphilis (disorder)
        '186850003',  # Secondary syphilis of skin and mucous membrane (disorder)
        '186854007',  # Uveitis due to secondary syphilis (disorder)
        '186861006',  # Secondary syphilis relapse (disorder)
        '186863009',  # Acute secondary syphilitic meningitis (disorder)
        '186867005',  # Latent early syphilis (disorder)
        '186868000',  # Serological relapse after treatment of latent early syphilis (disorder)
        '186875004',  # Syphilitic endocarditis of mitral valve (disorder)
        '186877007',  # Syphilitic endocarditis of tricuspid valve (disorder)
        '186878002',  # Syphilitic endocarditis of pulmonary valve (disorder)
        '186893003',  # Rupture of syphilitic cerebral aneurysm (disorder)
        '186903006',  # Late latent syphilis (disorder)
        '186915005',  # Chronic gonorrhea lower genitourinary tract (disorder)
        '186931002',  # Gonococcal anal infection (disorder)
        '186939000',  # Gonococcal peritonitis (disorder)
        '186946009',  # Lymphogranuloma venereum (disorder)
        '186975003',  # Mixed pinta lesions (disorder)
        '187052004',  # Pulmonary African histoplasmosis (disorder)
        '187065008',  # Cutaneous blastomycosis (disorder)
        '187079000',  # Chromoblastomycosis (disorder)
        '187081003',  # Subcutaneous pheohyphomycotic abscess and cyst (disorder)
        '187089001',  # Actinomycotic mycetoma (disorder)
        '187115002',  # Cutaneous schistosomiasis (disorder)
        '187171000',  # Ascariasis with intestinal complications (disorder)
        '187174008',  # Intestinal strongyloidiasis (disorder)
        '187187002',  # Intestinal angiostrongyliasis (disorder)
        '188463006',  # Chlamydial pelvic inflammatory disease (disorder)
        '191727003',  # Post measles pneumonia (disorder)
        '192647003',  # Secondary syphilitic meningitis (disorder)
        '193786000',  # Keratitis caused by syphilis (disorder)
        '193922006',  # Parasitic eyelid infestation (disorder)
        '194202008',  # Infective otitis externa due to erysipelas (disorder)
        '194203003',  # Infective otitis externa due to impetigo (disorder)
        '194208007',  # Chronic otitis externa due to moniliasis (disorder)
        '194281003',  # Acute suppurative otitis media (disorder)
        '194282005',  # Acute suppurative otitis media due to another disease (disorder)
        '194907008',  # Acute syphilitic pericarditis (disorder)
        '194910001',  # Acute gonococcal pericarditis (disorder)
        '194947001',  # Acute myocarditis - syphilitic (disorder)
        '195720006',  # Acute streptococcal bronchitis (disorder)
        '195721005',  # Acute haemophilus influenzae bronchitis (disorder)
        '195756009',  # Woakes' ethmoiditis (disorder)
        '195798007',  # Chronic adenotonsillitis (disorder)
        '195834004',  # Pharynx or nasopharynx edema (disorder)
        '195866009',  # Laryngeal pachydermia (disorder)
        '195878008',  # Pneumonia and influenza (disorder)
        '195886008',  # Group B streptococcal pneumonia (disorder)
        '195888009',  # Proteus pneumonia (disorder)
        '195889001',  # Legionella pneumonia (disorder)
        '195896004',  # Pneumonia caused by pleuropneumonia-like organism (disorder)
        '195902009',  # Anthrax pneumonia (disorder)
        '195904005',  # Pneumonia with coccidioidomycosis (disorder)
        '195908008',  # Actinomycotic pneumonia (disorder)
        '195909000',  # Nocardial pneumonia (disorder)
        '195920000',  # Influenza with pneumonia, influenza virus identified (disorder)
        '197143009',  # Megacolon in Chagas' disease (disorder)
        '197305002',  # Syphilitic portal cirrhosis (disorder)
        '197347003',  # Hepatitis in late syphilis (disorder)
        '197348008',  # Hepatitis in secondary syphilis (disorder)
        '197757004',  # Late syphilis of kidney (disorder)
        '197760006',  # Chronic pyelonephritis without medullary necrosis (disorder)
        '197761005',  # Chronic pyelonephritis with medullary necrosis (disorder)
        '197763008',  # Chronic pyonephrosis (disorder)
        '197764002',  # Non-obstructive reflux-associated chronic pyelonephritis (disorder)
        '197768004',  # Acute pyelonephritis without medullary necrosis (disorder)
        '197769007',  # Acute pyelonephritis with medullary necrosis (disorder)
        '197770008',  # Acute pyonephrosis (disorder)
        '197782004',  # Pyelonephritis associated with another disorder (disorder)
        '197834003',  # Chronic interstitial cystitis (disorder)
        '197837005',  # Submucous cystitis (disorder)
        '197845000',  # Cystitis associated with another disorder (disorder)
        '197848003',  # Gonococcal cystitis (disorder)
        '197849006',  # Cystitis caused by Candida (disorder)
        '197850006',  # Trichomonal cystitis (disorder)
        '197853008',  # Recurrent cystitis (disorder)
        '197963001',  # Prostatitis associated with another disorder (disorder)
        '197964007',  # Actinomycotic prostatitis (disorder)
        '197965008',  # Blastomycotic prostatitis (disorder)
        '197966009',  # Syphilitic prostatitis (disorder)
        '197967000',  # Gonococcal prostatitis (disorder)
        '197968005',  # Prostatitis caused by Candida (disorder)
        '198130006',  # Female pelvic inflammatory disease (disorder)
        '198141008',