# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['spylib', 'spylib.oauth', 'spylib.utils']

package_data = \
{'': ['*']}

install_requires = \
['PyJWT>=2.1.0,<3.0.0',
 'fastapi>=0.65.1,<0.66.0',
 'httpx>=0.18.1,<0.19.0',
 'loguru>=0.5.3,<0.6.0',
 'nest-asyncio>=1.5.1,<2.0.0',
 'shortuuid>=1.0.1,<2.0.0',
 'tenacity>=7.0.0,<8.0.0']

setup_kwargs = {
    'name': 'spylib',
    'version': '0.4',
    'description': "A library to facilitate interfacing with Shopify's API",
    'long_description': '# Shopify Python Library - SPyLib\n\nThe Shopify python library or SPyLib, simplifies the use of the Shopify\nservices such as the REST and GraphQL APIs as well as the OAuth authentication.\nAll of this is done **asynchronously using asyncio**.\n\n![Tests](https://github.com/SatelCreative/satel-spylib/actions/workflows/tests.yml/badge.svg)\n\n## Installation\n\n```bash\npip install spylib\n```\n\n## Contributing\n\nIf you want to contribute a small change/feature, the best is to just create a PR with\nyour changes.\nFor bigger changes/features it\'s best to open an issue first and discuss it to agree\non the code organization and overall implementation before spending too much time on\nthe code, unless you want to keep it in your own forked repo.\n\n### Setting up the development environment\n\nWe use the [python poetry](https://python-poetry.org/) package to manage this package.\nFollow the official instructions to install poetry on your system then once you clone\nthis repository just just need to do the following to install the dependencies from\nthe development environment, as well as install `spylib` in\n[editable mode](https://pip.pypa.io/en/stable/cli/pip_install/#install-editable):\n\n```bash\npoetry install\n```\n\nThen you can start monitoring the code for changes and run the test suite this way:\n\n```bash\npoetry shell\nscripts/test_watch.sh\n```\n\n## Overview\n\n### Token\n\nThe token class contains the majority of the logic for communicating with shopify.\nTo use the token class, you must define a child class for the tokens you are using\nand implement a subset of the `save` and `load` abstract methods. Your option for\nchild classes are `OnlineTokenABC`, `OfflineTokenABC` or `PrivateTokenABC`:\n\n#### Implement Token Classes\n\n```python\nclass OfflineToken(OfflineTokenABC):\n  async def save(self):\n      # Some code to save the token to a database\n\n  @classmethod\n  async def load(cls, store_name: str):\n      # Some code to load the token from the database\n\nclass OnlineToken(OnlineTokenABC):\n  async def save(self):\n      # Some code to save the token to a database\n\n  @classmethod\n  async def load(cls, store_name: str, user_id: str):\n      # Some code to load the token from the database\n\nclass PrivateToken(PrivateTokenABC):\n  @classmethod\n  async def load(cls, store_name: str, user_id: str):\n      # Some code to load the token from the database\n      # No need for save, as it is static.\n```\n\n#### Create Token\n\nOnce you have defined these methods, we can create an instance of a token using\none of the following:\n\n```python\ntoken = OfflineToken(\n  store_name,\n  access_token,\n  scope\n)\n\ntoken = OnlineToken(\n  store_name,\n  access_token,\n  scope,\n  expires_in,\n  associated_user_scope,\n  associated_user_id\n)\n\ntoken = PrivateToken(\n  store_name,\n  access_token,\n  scope\n)\n```\n\n### Querying Shopify\n\n#### REST\n\nWe can query the store using the REST endpoint:\n\n```python\nawait token.execute_rest(\n  request: Request,\n  endpoint: str,\n  json: Optional[Dict[str, Any]],\n  debug: Optional[str],\n)\n```\n\nFor example, if you want to query a product from shopify you can run:\n\n```python\nproduct_json = await token.execute_rest(\n  request = GET,\n  endpoint = f\'/products/{product_id}.json\'\n)\n```\n\nIf you want to update a product in a shop you can run:\n\n```python\nproduct_json = await token.execute_rest(\n  request = PUT,\n  endpoint = f\'/products/{product_id}.json\',\n  json = {\n    "product":\n    {\n      "id": product_id,\n      "title": "New Title"\n    }\n  }\n)\n```\n\nThe `REST` method takes a `request` parameter which is one of the `Request` constants defined in\nthe [rest](./spylib/utils/rest.py) file. The options are `GET`, `POST`, `PUT`, or `DELETE`.\n\nEndpoint is the API endpoint string that we are querying, this should be similar to\nthe following format:\n\n```python\nf\'/{resource}.json?fields={resource.param}&{[params]}\'\n```\n\nThe `debug` parameter is the message that is returned when there is an error. It is optional as it defaults to `""`.\n\n#### GQL\n\nWe can also query Shopify using the GraphQL endpoint:\n\n```python\ntoken.execute_gql(\n  query: str,\n  variables: Dict[str, Any],\n  operation_name: Optional[str]\n)\n```\n\nFor example, if you want to query a product from shopify you can run:\n\n```python\nquery = """\n{\n  product(id: "gid://shopify/Product/1974208299030") {\n  \tid,\n    title\n  }\n}"""\n\nproduct_json = await token.execute_gql(query = query)\n```\n\nIf you want to update a product in a shop you can run:\n\n```python\nquery = """\nmutation productUpdateMutation($id: ID, $title: String) {\n  productUpdate(input: {\n    id: $id,\n    title: $title\n  })\n  {\n    product {\n      id\n    }\n  }\n}"""\n\nvariables = {\n  \'id\': \'gid://shopify/Product/108828309\',\n  \'title\': "Sweet new product - GraphQL Edition"\n}\nproduct_json = await token.execute_gql(\n  query = query,\n  variables = variables\n)\n```\n\nThe `query` is a GraphQL query that will be passed to shopify for execution. You can use the GQL explorer\nfor your shop to create a query. For example, the [shopify demo GQL explorer](https://shopify.dev/apps/tools/graphiql-admin-api).\n\nThe `variables` are a dictionary of variables that will be substituted into the query.\n\nThe `operation_name` is a name for the query you are about to run.\n\n### OAuth\n\n**Notice** there have been considerable changes to the oauth in version 0.3. The core\nof this change has been the move from environment variables to parameters in the init\nscript for the routers. The following describes a very basic implementation.\n\nRather than reimplementing for each app the\n[Shopify OAuth authentication](https://shopify.dev/tutorials/authenticate-with-oauth)\none can simple get a [FastAPI](https://fastapi.tiangolo.com/) router that provides\nthe install and callback endpoints ready to handle the whole OAuth process.\nYou just need to call `init_oauth_router` such that:\n\n```python\nfrom spylib.oauth import OfflineToken, OnlineToken, init_oauth_router\n\n\nasync def my_post_install(storename: str, offline_token: OfflineToken):\n    """Function handling the offline token obtained at the end of the installation"""\n    # Store to database\n    pass\n\nasync def my_post_login(storename: str, online_token: OnlineToken):\n    """Function handling the online token obtained at the end of the user login"""\n    # Store to database\n    pass\n\noauth_router = init_oauth_router(\n    app_scopes=[\'write_orders\', \'write_products\'],\n    user_scopes=[\'read_orders\', \'write_products\'],\n    public_domain=\'my.app.com\',\n    private_key=\'KEY_FOR_OAUTH_JWT\',\n    api_key=\'SHOPIFY_APP_API_KEY\',\n    api_secret_key=\'SHOPIFY_APP_SECRET_KEY\',\n    app_handle=\'SHOPIFY_APP_HANDLE\',\n    post_install=my_post_install,\n    post_login=my_post_login,\n)\n```\n\nThe `app_scopes` are for the offline token and the `user_scopes` for the online token.\nThe `public_domain` is used to set the callback URL used in the OAuth process.\n\nThis library uses a JWT encoded `nonce` to avoid the need for a database or some other\nmechanism to track the `nonce`. This JWT has an expiration time and is unique for each\nOAuth process making it a valid `nonce` mechanism.\nThe `private_key` parameter defines the key used to encode and decode this JWT.\n\nThe api and secret key can be found inside your shopify app main configuration page. The\napp handle can be found in the same spot but needs to be pulled from the url:\n\n1. Go to your shopify app\'s editing page (The url should be `https://partners.shopify.com/<partner_id>/apps/<app_id>/edit`)\n2. Open the console and run `window.RailsData.user.app.handle`. The result is the handle.\n\nThe `post_install` and `post_login` provide a way to inject functions handling the\nresult of the installation and the login processes respectivaly. They are meant in\nparticular to record the offline and online tokens in your app\'s database.\nThey can be synchronous or asynchronous functions taking the storename and the token\nas arguments.\n\n### Session Tokens\n\nThe [session token](https://shopify.dev/apps/auth/session-tokens/authenticate-an-embedded-app-using-session-tokens)\nfunctionality can be used to verify the session for the user. The suggested syntax is to define a partial function:\n\n```python\nfrom spylib.utils import SessionToken\n\ndecode_session_token = partial(SessionToken.decode_token_from_header, api_key=api_key, secret=secret)\n```\n\nThen this can be used as a dependency in FastAPI by doing the following:\n\n```python\n@app.get("/items/")\nasync def read_items(session: SessionToken = Depends(decode_session_token)):\n  # Some api code\n```\n\n### Build and run documentation (lazydocs/mkdocs)\n\nDocumentation for this package is handled by `lazydocs` and so it needs a few steps to generate it locally.\n\nInside `poetry shell`:\n\n```bash\nlazydocs --overview-file="index.md" \\\n--src-base-url="https://github.com/SatelCreative/satel-spylib/tree/main" \\\n--output-path="./docs/api-docs" \\\n--validate spylib\n\nmkdocs build\nmkdocs serve\n```\n\nThe default URL is at `127.0.0.1:8000`.\n\n## For package maintainers\n\nWe use [poetry](https://python-poetry.org/) to manage the dependencies and publish to pypi.\n\n### How to publish\n\n1. Create branch `release/x.y.z`\n2. Change the version in the `pyproject.toml` and `spylib/__init__.py` files\n   - you can use `poetry version XXXXX` to change `pyproject.toml`\n3. Commit to git\n4. Open PR for other codeowners to review\n5. Once approved, squash and merge `release/x.y.z`\n6. Run `poetry publish --build` to create the package and publish to PyPI\n7. Tag the release in git and push it to Github\n\n**Notes**:\n\n- It\'s better to tag after publishing in case there is an issue while publishing\n',
    'author': 'Anthony Hillairet',
    'author_email': 'ant@satel.ca',
    'maintainer': 'Anthony Hillairet',
    'maintainer_email': 'ant@satel.ca',
    'url': 'https://github.com/SatelCreative/satel-spylib',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
