"""
Equinox Day is a public holiday in Japan that usually occurs:
    (in the Spring) on March 20 or 21,
    (in the Autumn) on September 22 or 23,
the date of the equinox in Japan Standard Time.
Due to the necessity of recent astronomical measurements,
the date of the holiday is not officially declared until February of the previous year

We can't easily compute the equinox for a given year, so we pre-compute a list of those
from the Tokyo exchange inauguration through 2099,
using pyephem (http://rhodesmill.org/pyephem/quick.html#equinoxes-solstices).
For a double check, see:  https://aa.usno.navy.mil/data/docs/EarthSeasons.php
"""
import pandas as pd
from pandas.tseries.holiday import sunday_to_monday

vernal_year_to_march_mapping = {
    1875: 21, 1876: 20, 1877: 20, 1878: 21, 1879: 21,
    1880: 20, 1881: 20, 1882: 21, 1883: 21, 1884: 20,
    1885: 20, 1886: 21, 1887: 21, 1888: 20, 1889: 20,
    1890: 21, 1891: 21, 1892: 20, 1893: 20, 1894: 20,
    1895: 21, 1896: 20, 1897: 20, 1898: 20, 1899: 21,

    1900: 21, 1901: 21, 1902: 21, 1903: 22, 1904: 21,
    1905: 21, 1906: 21, 1907: 22, 1908: 21, 1909: 21,
    1910: 21, 1911: 22, 1912: 21, 1913: 21, 1914: 21,
    1915: 22, 1916: 21, 1917: 21, 1918: 21, 1919: 22,
    1920: 21, 1921: 21, 1922: 21, 1923: 22, 1924: 21,
    1925: 21, 1926: 21, 1927: 21, 1928: 21, 1929: 21,
    1930: 21, 1931: 21, 1932: 21, 1933: 21, 1934: 21,
    1935: 21, 1936: 21, 1937: 21, 1938: 21, 1939: 21,
    1940: 21, 1941: 21, 1942: 21, 1943: 21, 1944: 21,
    1945: 21, 1946: 21, 1947: 21, 1948: 21, 1949: 21,
    1950: 21, 1951: 21, 1952: 21, 1953: 21, 1954: 21,
    1955: 21, 1956: 21, 1957: 21, 1958: 21, 1959: 21,
    1960: 20, 1961: 21, 1962: 21, 1963: 21, 1964: 20,
    1965: 21, 1966: 21, 1967: 21, 1968: 20, 1969: 21,
    1970: 21, 1971: 21, 1972: 20, 1973: 21, 1974: 21,
    1975: 21, 1976: 20, 1977: 21, 1978: 21, 1979: 21,
    1980: 20, 1981: 21, 1982: 21, 1983: 21, 1984: 20,
    1985: 21, 1986: 21, 1987: 21, 1988: 20, 1989: 21,
    1990: 21, 1991: 21, 1992: 20, 1993: 20, 1994: 21,
    1995: 21, 1996: 20, 1997: 20, 1998: 21, 1999: 21,

    2000: 20, 2001: 20, 2002: 21, 2003: 21, 2004: 20,
    2005: 20, 2006: 21, 2007: 21, 2008: 20, 2009: 20,
    2010: 21, 2011: 21, 2012: 20, 2013: 20, 2014: 21,
    2015: 21, 2016: 20, 2017: 20, 2018: 21, 2019: 21,
    2020: 20, 2021: 20, 2022: 21, 2023: 21, 2024: 20,
    2025: 20, 2026: 20, 2027: 21, 2028: 20, 2029: 20,
    2030: 20, 2031: 21, 2032: 20, 2033: 20, 2034: 20,
    2035: 21, 2036: 20, 2037: 20, 2038: 20, 2039: 21,
    2040: 20, 2041: 20, 2042: 20, 2043: 21, 2044: 20,
    2045: 20, 2046: 20, 2047: 21, 2048: 20, 2049: 20,
    2050: 20, 2051: 21, 2052: 20, 2053: 20, 2054: 20,
    2055: 21, 2056: 20, 2057: 20, 2058: 20, 2059: 20,
    2060: 20, 2061: 20, 2062: 20, 2063: 20, 2064: 20,
    2065: 20, 2066: 20, 2067: 20, 2068: 20, 2069: 20,
    2070: 20, 2071: 20, 2072: 20, 2073: 20, 2074: 20,
    2075: 20, 2076: 20, 2077: 20, 2078: 20, 2079: 20,
    2080: 20, 2081: 20, 2082: 20, 2083: 20, 2084: 20,
    2085: 20, 2086: 20, 2087: 20, 2088: 20, 2089: 20,
    2090: 20, 2091: 20, 2092: 19, 2093: 20, 2094: 20,
    2095: 20, 2096: 19, 2097: 20, 2098: 20, 2099: 20,
}

autumnal_year_to_september_mapping = {
    1875: 23, 1876: 23, 1877: 23, 1878: 23, 1879: 23,
    1880: 23, 1881: 23, 1882: 23, 1883: 23, 1884: 23,
    1885: 23, 1886: 23, 1887: 23, 1888: 22, 1889: 23,
    1890: 23, 1891: 23, 1892: 22, 1893: 23, 1894: 23,
    1895: 23, 1896: 22, 1897: 23, 1898: 23, 1899: 23,

    1900: 23, 1901: 24, 1902: 24, 1903: 24, 1904: 23,
    1905: 24, 1906: 24, 1907: 24, 1908: 23, 1909: 24,
    1910: 24, 1911: 24, 1912: 23, 1913: 24, 1914: 24,
    1915: 24, 1916: 23, 1917: 24, 1918: 24, 1919: 24,
    1920: 23, 1921: 23, 1922: 24, 1923: 24, 1924: 23,
    1925: 23, 1926: 24, 1927: 24, 1928: 23, 1929: 23,
    1930: 24, 1931: 24, 1932: 23, 1933: 23, 1934: 24,
    1935: 24, 1936: 23, 1937: 23, 1938: 24, 1939: 24,
    1940: 23, 1941: 23, 1942: 24, 1943: 24, 1944: 23,
    1945: 23, 1946: 24, 1947: 24, 1948: 23, 1949: 23,
    1950: 23, 1951: 24, 1952: 23, 1953: 23, 1954: 23,
    1955: 24, 1956: 23, 1957: 23, 1958: 23, 1959: 24,
    1960: 23, 1961: 23, 1962: 23, 1963: 24, 1964: 23,
    1965: 23, 1966: 23, 1967: 24, 1968: 23, 1969: 23,
    1970: 23, 1971: 24, 1972: 23, 1973: 23, 1974: 23,
    1975: 24, 1976: 23, 1977: 23, 1978: 23, 1979: 24,
    1980: 23, 1981: 23, 1982: 23, 1983: 23, 1984: 23,
    1985: 23, 1986: 23, 1987: 23, 1988: 23, 1989: 23,
    1990: 23, 1991: 23, 1992: 23, 1993: 23, 1994: 23,
    1995: 23, 1996: 23, 1997: 23, 1998: 23, 1999: 23,

    2000: 23, 2001: 23, 2002: 23, 2003: 23, 2004: 23,
    2005: 23, 2006: 23, 2007: 23, 2008: 23, 2009: 23,
    2010: 23, 2011: 23, 2012: 22, 2013: 23, 2014: 23,
    2015: 23, 2016: 22, 2017: 23, 2018: 23, 2019: 23,
    2020: 22, 2021: 23, 2022: 23, 2023: 23, 2024: 22,
    2025: 23, 2026: 23, 2027: 23, 2028: 22, 2029: 23,
    2030: 23, 2031: 23, 2032: 22, 2033: 23, 2034: 23,
    2035: 23, 2036: 22, 2037: 23, 2038: 23, 2039: 23,
    2040: 22, 2041: 23, 2042: 23, 2043: 23, 2044: 22,
    2045: 22, 2046: 23, 2047: 23, 2048: 22, 2049: 22,
    2050: 23, 2051: 23, 2052: 22, 2053: 22, 2054: 23,
    2055: 23, 2056: 22, 2057: 22, 2058: 23, 2059: 23,
    2060: 22, 2061: 22, 2062: 23, 2063: 23, 2064: 22,
    2065: 22, 2066: 23, 2067: 23, 2068: 22, 2069: 22,
    2070: 23, 2071: 23, 2072: 22, 2073: 22, 2074: 23,
    2075: 23, 2076: 22, 2077: 22, 2078: 22, 2079: 23,
    2080: 22, 2081: 22, 2082: 22, 2083: 23, 2084: 22,
    2085: 22, 2086: 22, 2087: 23, 2088: 22, 2089: 22,
    2090: 22, 2091: 23, 2092: 22, 2093: 22, 2094: 22,
    2095: 23, 2096: 22, 2097: 22, 2098: 22, 2099: 23,
}


def vernal_equinox_for_year(year):
    day = vernal_year_to_march_mapping.get(year, 20)
    return pd.Timestamp(year, 3, day)


def vernal_equinox(dt):
    year = dt.year
    equinox = vernal_equinox_for_year(year)
    return sunday_to_monday(equinox) if year >= 1973 else equinox


def autumnal_equinox_for_year(year):
    day = autumnal_year_to_september_mapping.get(year, 23)
    return pd.Timestamp(year, 9, day)


def autumnal_equinox(dt):
    year = dt.year
    equinox = autumnal_equinox_for_year(year)
    return sunday_to_monday(equinox) if year >= 1973 else equinox


def autumnal_citizen_dates(start=2003, end=2099):
    dates = []
    for year in range(start, end):
        respect_for_aged = pd.Timestamp(year, 9, 1) + pd.offsets.WeekOfMonth(week=2, weekday=0)
        equinox = autumnal_equinox_for_year(year)
        if (equinox - respect_for_aged).days == 2:
            dates.append(respect_for_aged + pd.offsets.Day())
    return dates
