# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['relycomply_client']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML',
 'Pygments',
 'boto3',
 'gql==3.0.0rc0',
 'littleutils',
 'pandas',
 'pyarrow',
 'requests',
 'requests-toolbelt',
 'tabulate',
 'termcolor',
 'toml']

entry_points = \
{'console_scripts': ['rely = relycomply_client.cli:main']}

setup_kwargs = {
    'name': 'relycomply-client',
    'version': '0.8.1',
    'description': 'A python client for the RelyComply platform',
    'long_description': '# RelyComply Python Client and CLI\n\n**BETA RELEASE**\n\nThis package contains the python client and CLI for the RelyComply platform:\n\n> RelyComply is an end-to-end Anti-Money Laundering (AML) Plaftorm, managing detection, risk management and automation of your AML compliance requirements\n\nThe CLI makes configuration of the system substantially simpler and allows for a full configuration-as-devops experience. \n\nThe python client exposes both a lower level GraphQL client which makes it easy to interact with the GraphQL APi in a pythonic manner. As well as a higher-level integration client that provides useful routines for common integration tasks.\n\n## Credentials\n\nCredentials for the various clients can be loaded in multiple ways. In the following order of precedence:\n\n* Constructor Arguments\n* Environment Variables\n* AWS Secrets\n* Configuration Files (.rely.toml)\n* Defaults\n\nCredentials can stack, so you can define certain credentials in a cofig file for example, and override it with an environment variable.\n\nThe following credentials can be set:\n\n* token: The API token of the user\n* url: The url of the RelyComply application (default: `https://app.relycomply.com`)\n* impersonate: If user impersonation is available for your user this allows you to make a command on behalf of another user. This should be in format `<organisation_name>:<user_email>`, for example `relycomply:james@relycomply.com`.\n\n**Constructor Arguments**\n\nYou can pass credentials when constructing a client as keyword arguments, e.g.\n\n```python\nclient = RelyComplyGQLClient(token="<token>")\n```\n\n**Environment Variables**\n\nThe client will look if there are matching environment variables of the format `RELYCOMPLY_<CREDENTIAL>`, for example: `RELYCOMPLY_TOKEN=<token>`.\n\n**AWS Secrets**\n\nThe client can integrate with the AWS Secrets Manager. This is done by setting appropriate environment variables with the naming convention `RELYCOMPLY_<CREDENTIAL>_AWS_SECRET` and the value being the secrets path, for example:\n\n```\nRELYCOMPLY_TOKEN_AWS_SECRET=path/to/my/secret\n```\n\nFor more information please consult the [AWS Secrets Manager documentation](https://docs.aws.amazon.com/secretsmanager/).\n\n**Config Files**\n\nCredentials can be set in a `.rely.toml` file. The client will search the current working directory, and all parent directories in order, with the most local directory taking precedence. The `.rely.toml` file is a TOML file with the credentials as keys, e.g.\n\n```toml\ntoken="<token>"\nurl="https://relycomply.customer.com"\n```\n\n**Default Credentials**\n\nThe following credentials have default values:\n\n```toml\nurl="https://app.relycomply.com"\n```\n\n## RelyComplyGQLClient\n\nA flexible and intelligent GraphQL client for RelyComply. This client will create methods that match the mutation sand queries of the RelyComply API, and expose them with familiar calling conventions. It also handles paging as well as simplifying the returned structures.\n\nIt can be constructed as below:\n\n```python\nfrom relycomply_client import RelyComplyGQLClient\n\nclient = RelyComplyGQLClient()\n\n# Or with specific credentials\nclient = RelyComplyGQLClient(token="<token>")\n```\n\nQueries can be called with their lowerCase field name and any filter arguments as kwargs, e.g.:\n\n```python\nclient.products(nameContain="ZA") # Will return a list of products\nclient.products(nameContain="ZA", _iter=True) # Will return a lazy generator\nclient.products(name="retailZA", _only=True) # Will return only the first object or None\n```\n\nThe client will automatically collapse edge lists into plain lists of objects to make the output easier to work with.\n\nMutations can be called in a similar way, but arguments will be lifted into the $input variable\n\n```python\nclient.createProduct(name="retailZA", label="South African Retail") # Returns the created product\n```\n\nThe interface is automatically generated from the GQL schema as well as the CLI support templates. Thus it should always be in sync with the latest features on the platform.\n\nThe client also exposes a raw GraphQl call when you need to make a more complex query. No post processing will be done on the results. For example to query the first 10 products.\n\n```python\nclient.graphql(\n    """\n    products(first:$first) {\n        edges {\n            node {\n                id\n                name\n            }\n        }\n    }\n    """, \n    variables=dict(first=10)\n)\n```\n\n## RelyComplyClient\n\nThe RelyComplyCLient contains higher level methods that make common integration tasks simpler. It provides simple integration with various cloud services and common data tools like pandas.\n\n```python\nfrom relycomply_client import RelyComplyGQLClient\nrc = RelyComplyClient()\n\n# Or for quick usage for a standard client\nfrom relycomply_client.rc import rc\n```\n\nA quick overview of a common data integration with the transaction monitoring is shown below:\n\n```python\n# Load a file with pandas\nraw_df = pd.read_csv(file_path)\n\n# Perform some cleaning\ndf = clean_raw_df(raw_df)\n\n# Pull in a datafile from a known source. \n# This will automatically creat a signed URL if an S3 path is passed\nraw_data_file = rc.pull_to_datafile(\n    file_path, "raw/" + file_name, wait_for_ready=True\n)\n\n# Upload a dataframe as parquet datafile\nprocessed_data_file = rc.put_to_datafile(df, "processed/" + file_name)\n\n# Ingest the given files\n\n# Note that the responses from previous calls can be passed as is for the call\n# arguments. Their id\'s will be automatically extracted.\ndata_source_version = rc.ingest_datasource(\n    data_source, data_file=processed_data_file, raw_data_files=[raw_data_file]\n)\n\n# Run a monitor\nrc.run_monitor(monitor_name, source_versions=[data_source_version])\n```\n\nThe underlying `RelyComplyGQLClient` can be accessed with `.gql` property.\n\n```python\nrc.gql.createProduct(name="bank_account", label="My Bank Account)\n```\n\n## Command Line Interface (CLI)\n\nThe command line interface is an important part of our developer first mentality. It acts as a layer on top of the GraphQL API and makes it substantially easier to for power-users to explore and manipulate RelyComply.\n\nGraphQL is excellent as an API for integration, but can be a lot extra overhead to quickly just see what is happening in the system. Primarily this is because the user has to define the output format they want. This greatly improves the flexibility but certainly is not as easy as just using curl on a rest endpoint. The rely CLI makes it easy to perform the standard queries and mutations on the GraphQL API, without extra effort by the user.\n\nThe CLI can be accessed with the `rely` command. The basic format is to call it with a `type` and an `action`. This will automatically be coerced into the appropriate GraphQL calls. \n\nArguments can be passed as keyword arguments of the form `--key=value` additionally a configuration file name can be passed as the final argument. The value can be a json string, which will be parsed correctly for complex arguments. Arguments are merged with the command line arguments taking precedence.\n\nQueries can be performed with the `list` and `retrieve` actions, `retrieve` will return a single item, and `list` will display a table of items.\n\n```bash\nrely product list # Will list all the products\nrely product list --nameContains="za" # Will list all the products with za in their name\nrely product retrieve --id=123 # Will return just the specified product \n```\n\nMutations can called by their name, with the action being prepended to the type. The system is intelligent enough that the case of the action and type do not matter.\n\n```bash\n# Will call createProduct\nrely product create --name="bank_account" --label="Bank Account" \n\n# Will update the given product (updateProduct) based on the given ID and the config file (pr_my_product.toml)\nrely product update --id=10 pr_my_product.toml\n```\n\nCertain aliases are provided for convenience, e.g. \n\n```bash\n# This will call addCaseNote\nrely case addNote --case=123 --note="This is my note"\n```\n\nThe format of the output can be controlled with the `--json`, `--yaml` and `--toml` (default) flags.',
    'author': 'James Saunders',
    'author_email': 'james@relycomply.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://www.relycomply.com',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4',
}


setup(**setup_kwargs)
