# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['aioswitcher', 'aioswitcher.api', 'aioswitcher.device', 'aioswitcher.schedule']

package_data = \
{'': ['*'], 'aioswitcher': ['resources/*']}

setup_kwargs = {
    'name': 'aioswitcher',
    'version': '3.0.3',
    'description': 'Switcher Python Integration.',
    'long_description': '<!-- markdownlint-disable MD033 -->\n# Switcher Python Integration</br>[![pypi-version]][11] [![pypi-downloads]][11] [![license-badge]][4]\n\n[![gh-build-status]][7] [![gh-pages-status]][8] [![codecov]][3]\n\nPyPi module integrating with various [Switcher][12] devices.</br>\nCheck out the [wiki pages][0] for a list of supported devices.\n\n```shell\npip install aioswitcher\n```\n\n<table>\n  <td><a href="https://aioswitcher.tomfi.info/">Documentation</a></td>\n  <td><a href="https://github.com/TomerFi/aioswitcher/wiki">Wiki</a></td>\n  <td><a href="https://github.com/TomerFi/aioswitcher/blob/dev/.github/CONTRIBUTING.md">Contributing</a></td>\n  <td><a href="https://github.com/TomerFi/.github/blob/main/.github/CODE_OF_CONDUCT.md">Code of Conduct</a></td>\n</table>\n\n## Example Usage\n\n<details>\n  <summary>State Bridge</summary>\n\n```python\nasync def print_devices(delay):\n    def on_device_found_callback(device):\n        # a switcher device will broadcast a state message approximately every 4 seconds\n        print(asdict(device))\n\n    async with SwitcherBridge(on_device_found_callback):\n        await asyncio.sleep(delay)\n\n# run the bridge for 60 seconds\nasyncio.get_event_loop().run_until_complete(print_devices(60))\n```\n\n</details>\n\n<details>\n  <summary>Power Plug API</summary>\n\n  ```python\n  async def control_power_plug(device_ip, device_id) :\n      # for connecting to a device we need its id and ip address\n      async with SwitcherType1Api(device_ip, device_id) as api:\n          # get the device current state\n          await api.get_state()\n          # turn the device on\n          await api.control_device(Command.ON)\n          # turn the device off\n          await api.control_device(Command.OFF)\n          # set the device name to \'my new name\'\n          await api.set_device_name("my new name")\n\n  asyncio.get_event_loop().run_until_complete(control_power_plug("111.222.11.22", "ab1c2d"))\n  ```\n\n</details>\n\n<details>\n  <summary>Water Heater API</summary>\n\n  ```python\n  async def control_water_heater(device_ip, device_id) :\n      # for connecting to a device we need its id and ip address\n      async with SwitcherType1Api(device_ip, device_id) as api:\n          # get the device current state\n          await api.get_state()\n          # turn the device on for 15 minutes\n          await api.control_device(Command.ON, 15)\n          # turn the device off\n          await api.control_device(Command.OFF)\n          # set the device name to \'my new name\'\n          await api.set_device_name("my new name")\n          # configure the device for 02:30 auto shutdown\n          await api.set_auto_shutdown(timedelta(hours=2, minutes=30))\n          # get the schedules from the device\n          await api.get_schedules()\n          # delete and existing schedule with id 1\n          await api.delete_schedule("1")\n          # create a new recurring schedule for 13:00-14:30\n          # executing on sunday and friday\n          await api.create_schedule("13:00", "14:30", {Days.SUNDAY, Days.FRIDAY})\n\n  asyncio.get_event_loop().run_until_complete(control_water_heater("111.222.11.22", "ab1c2d"))\n  ```\n\n</details>\n\n<details>\n  <summary>Runner API</summary>\n\n  ```python\n  async def control_runner(device_ip, device_id) :\n      # for connecting to a device we need its id and ip address\n      async with SwitcherType2Api(device_ip, device_id) as api:\n          # get the device current state\n          await api.get_shutter_state()\n          # open the shutter to 30%\n          await api.set_position(30)\n          # stop the shutter if currently rolling\n          await api.stop()\n\n  asyncio.get_event_loop().run_until_complete(control_runner("111.222.11.22", "ab1c2d"))\n  ```\n\n</details>\n\n<details>\n  <summary>Breeze API</summary>\n\n  ```python\n  async def control_breeze(device_ip, device_id, remote_manager, remote_id) :\n      # for connecting to a device we need its id and ip address\n      async with SwitcherType2Api(device_ip, device_id) as api:\n          # get the device current state\n          await api.get_breeze_state()\n          # initialize the Breeze RemoteManager and get the remote\n          remote = remote_manager.get_remote(remote_id)\n          # prepare a control command that turns on the Breeze\n          # set to 24 degree (Celsius) cooling with vertical swing\n          # send command to the device\n          await api.control_breeze_device(\n              remote,\n              DeviceState.ON,\n              ThermostatMode.COOL,\n              24,\n              ThermostatFanLevel.MEDIUM,\n              ThermostatSwing.ON,\n          )\n\n  # create the remote manager outside the context for re-using\n  remote_manager = SwitcherBreezeRemoteManager()\n  asyncio.get_event_loop().run_until_complete(control_breeze("111.222.11.22", "ab1c2d", remote_manager, "DLK65863"))\n  ```\n\n</details>\n\n## Command Line Helper Scripts\n\n- [discover_devices.py](scripts/discover_devices.py) can discover devices and their\n  states.\n- [control_device.py](scripts/control_device.py) can control a device.\n\n## Disclaimer\n\nThis is **NOT** an official module and it is **NOT** officially supported by the vendor.</br>\nThat said, thanks are in order to all the people at [Switcher][12] for their cooperation and general support.\n\n## Contributors\n\nThanks goes to these wonderful people ([emoji key][1]):\n\n<!-- ALL-CONTRIBUTORS-LIST:START - Do not remove or modify this section -->\n<!-- prettier-ignore-start -->\n<!-- markdownlint-disable -->\n<table>\n  <tr>\n    <td align="center"><a href="https://github.com/aviadgolan"><img src="https://avatars.githubusercontent.com/u/17742111?v=4?s=100" width="100px;" alt=""/><br /><sub><b>Aviad Golan</b></sub></a><br /><a href="#data-AviadGolan" title="Data">🔣</a></td>\n    <td align="center"><a href="https://github.com/dolby360"><img src="https://avatars.githubusercontent.com/u/22151399?v=4?s=100" width="100px;" alt=""/><br /><sub><b>Dolev Ben Aharon</b></sub></a><br /><a href="https://github.com/TomerFi/aioswitcher/commits?author=dolby360" title="Documentation">📖</a></td>\n    <td align="center"><a href="http://fabian-affolter.ch/blog/"><img src="https://avatars.githubusercontent.com/u/116184?v=4?s=100" width="100px;" alt=""/><br /><sub><b>Fabian Affolter</b></sub></a><br /><a href="https://github.com/TomerFi/aioswitcher/commits?author=fabaff" title="Code">💻</a></td>\n    <td align="center"><a href="https://github.com/OrBin"><img src="https://avatars.githubusercontent.com/u/6897234?v=4?s=100" width="100px;" alt=""/><br /><sub><b>Or Bin</b></sub></a><br /><a href="https://github.com/TomerFi/aioswitcher/commits?author=OrBin" title="Code">💻</a></td>\n    <td align="center"><a href="http://exploit.co.il"><img src="https://avatars.githubusercontent.com/u/1768915?v=4?s=100" width="100px;" alt=""/><br /><sub><b>Shai rod</b></sub></a><br /><a href="#data-nightrang3r" title="Data">🔣</a></td>\n    <td align="center"><a href="https://github.com/thecode"><img src="https://avatars.githubusercontent.com/u/1858925?v=4?s=100" width="100px;" alt=""/><br /><sub><b>Shay Levy</b></sub></a><br /><a href="https://github.com/TomerFi/aioswitcher/commits?author=thecode" title="Code">💻</a> <a href="#ideas-thecode" title="Ideas, Planning, & Feedback">🤔</a> <a href="#maintenance-thecode" title="Maintenance">🚧</a></td>\n    <td align="center"><a href="https://github.com/dmatik"><img src="https://avatars.githubusercontent.com/u/5577386?v=4?s=100" width="100px;" alt=""/><br /><sub><b>dmatik</b></sub></a><br /><a href="#blog-dmatik" title="Blogposts">📝</a> <a href="#ideas-dmatik" title="Ideas, Planning, & Feedback">🤔</a> <a href="#userTesting-dmatik" title="User Testing">📓</a></td>\n  </tr>\n  <tr>\n    <td align="center"><a href="https://github.com/jafar-atili"><img src="https://avatars.githubusercontent.com/u/19508787?v=4?s=100" width="100px;" alt=""/><br /><sub><b>jafar-atili</b></sub></a><br /><a href="https://github.com/TomerFi/aioswitcher/commits?author=jafar-atili" title="Code">💻</a> <a href="https://github.com/TomerFi/aioswitcher/commits?author=jafar-atili" title="Documentation">📖</a></td>\n  </tr>\n</table>\n\n<!-- markdownlint-restore -->\n<!-- prettier-ignore-end -->\n\n<!-- ALL-CONTRIBUTORS-LIST:END -->\n\n<!-- Real Links -->\n[0]: https://github.com/TomerFi/aioswitcher/wiki\n[1]: https://allcontributors.org/docs/en/emoji-key\n[3]: https://codecov.io/gh/TomerFi/aioswitcher\n[4]: https://github.com/TomerFi/aioswitcher\n[7]: https://github.com/TomerFi/aioswitcher/actions/workflows/stage.yml\n[8]: https://aioswitcher.tomfi.info/\n[11]: https://pypi.org/project/aioswitcher\n[12]: https://www.switcher.co.il/\n<!-- Badges Links -->\n[codecov]: https://codecov.io/gh/TomerFi/aioswitcher/graph/badge.svg\n[gh-build-status]: https://github.com/TomerFi/aioswitcher/actions/workflows/stage.yml/badge.svg\n[gh-pages-status]: https://github.com/TomerFi/aioswitcher/actions/workflows/pages.yml/badge.svg\n[license-badge]: https://img.shields.io/github/license/tomerfi/aioswitcher\n[pypi-downloads]: https://img.shields.io/pypi/dm/aioswitcher.svg?logo=pypi&color=1082C2\n[pypi-version]: https://img.shields.io/pypi/v/aioswitcher?logo=pypi\n',
    'author': 'Tomer Figenblat',
    'author_email': 'tomer.figenblat@gmail.com',
    'maintainer': 'Shay Levy',
    'maintainer_email': 'None',
    'url': 'https://pypi.org/project/aioswitcher/',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.9.0,<4.0.0',
}


setup(**setup_kwargs)
