"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SPADeploy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_patterns_1 = require("@aws-cdk/aws-route53-patterns");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const s3 = require("@aws-cdk/aws-s3");
class SPADeploy extends cdk.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config !== 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false,
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config, isForCloudFront) {
        const bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true,
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true || isForCloudFront === true) {
            bucketConfig.publicReadAccess = false;
            if (typeof config.blockPublicAccess !== 'undefined') {
                bucketConfig.blockPublicAccess = config.blockPublicAccess;
            }
        }
        const bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true && isForCloudFront === false) {
            if (typeof this.globalConfig.ipList === 'undefined') {
                this.node.addError('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(`${bucket.bucketArn}/*`);
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList,
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, accessIdentity, cert) {
        const cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket,
                        originAccessIdentity: accessIdentity,
                    },
                    behaviors: config.cfBehaviors ? config.cfBehaviors : [{ isDefaultBehavior: true }],
                },
            ],
            // We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                },
                {
                    errorCode: 404,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                }],
        };
        if (typeof config.certificateARN !== 'undefined' && typeof config.cfAliases !== 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases,
            };
        }
        if (typeof config.sslMethod !== 'undefined') {
            cfConfig.aliasConfiguration.sslMethod = config.sslMethod;
        }
        if (typeof config.securityPolicy !== 'undefined') {
            cfConfig.aliasConfiguration.securityPolicy = config.securityPolicy;
        }
        if (typeof config.zoneName !== 'undefined' && typeof cert !== 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.subdomain ? `${config.subdomain}.${config.zoneName}` : config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website.
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config, false);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        const cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl,
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName === 'undefined' || config.exportWebsiteUrlName === '') {
                this.node.addError('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new cdk.CfnOutput(this, 'URL', cfnOutputConfig);
        return { websiteBucket };
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution
     It will also setup error forwarding and unauth forwarding back to indexDoc.
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new cdk.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.distributionDomainName,
        });
        return { websiteBucket, distribution };
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto
     configured by using the details in the hosted zone provided.
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        const domainName = config.subdomain ? `${config.subdomain}.${config.zoneName}` : config.zoneName;
        const cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName,
            region: 'us-east-1',
        });
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: domainName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution)),
        });
        if (!config.subdomain) {
            new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
                zone,
                recordNames: [`www.${config.zoneName}`],
                targetDomain: config.zoneName,
            });
        }
        return { websiteBucket, distribution };
    }
}
exports.SPADeploy = SPADeploy;
_a = JSII_RTTI_SYMBOL_1;
SPADeploy[_a] = { fqn: "cdk-spa-deploy.SPADeploy", version: "1.94.1" };
//# sourceMappingURL=data:application/json;base64,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