"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationMultipleTargetGroupsFargateService = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const application_multiple_target_groups_service_base_1 = require("../base/application-multiple-target-groups-service-base");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer.
 *
 * @stability stable
 */
class ApplicationMultipleTargetGroupsFargateService extends application_multiple_target_groups_service_base_1.ApplicationMultipleTargetGroupsServiceBase {
    /**
     * Constructs a new instance of the ApplicationMultipleTargetGroupsFargateService class.
     *
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, props);
        this.assignPublicIp = (_a = props.assignPublicIp) !== null && _a !== void 0 ? _a : false;
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify only one of TaskDefinition or TaskImageOptions.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.FargateTaskDefinition(this, 'TaskDef', {
                memoryLimitMiB: props.memoryLimitMiB,
                cpu: props.cpu,
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            const containerName = (_b = taskImageOptions.containerName) !== null && _b !== void 0 ? _b : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                logging: this.logDriver,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
            });
            if (taskImageOptions.containerPorts) {
                for (const containerPort of taskImageOptions.containerPorts) {
                    container.addPortMappings({
                        containerPort,
                    });
                }
            }
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('At least one essential container must be specified');
        }
        if (this.taskDefinition.defaultContainer.portMappings.length === 0) {
            this.taskDefinition.defaultContainer.addPortMappings({
                containerPort: 80,
            });
        }
        this.service = this.createFargateService(props);
        if (props.targetGroups) {
            this.addPortMappingForTargets(this.taskDefinition.defaultContainer, props.targetGroups);
            this.targetGroup = this.registerECSTargets(this.service, this.taskDefinition.defaultContainer, props.targetGroups);
        }
        else {
            this.targetGroup = this.listener.addTargets('ECS', {
                targets: [this.service],
                port: this.taskDefinition.defaultContainer.portMappings[0].containerPort,
            });
        }
    }
    createFargateService(props) {
        return new aws_ecs_1.FargateService(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
            platformVersion: props.platformVersion,
        });
    }
}
exports.ApplicationMultipleTargetGroupsFargateService = ApplicationMultipleTargetGroupsFargateService;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbXVsdGlwbGUtdGFyZ2V0LWdyb3Vwcy1mYXJnYXRlLXNlcnZpY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcHBsaWNhdGlvbi1tdWx0aXBsZS10YXJnZXQtZ3JvdXBzLWZhcmdhdGUtc2VydmljZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw4Q0FBaUc7QUFHakcsNkhBR2lFOzs7Ozs7QUFpRmpFLE1BQWEsNkNBQThDLFNBQVEsNEZBQTBDOzs7Ozs7SUF5QjNHLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBNEQsRUFBRTs7UUFDdEcsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFeEIsSUFBSSxDQUFDLGNBQWMsU0FBRyxLQUFLLENBQUMsY0FBYyxtQ0FBSSxLQUFLLENBQUM7UUFFcEQsSUFBSSxLQUFLLENBQUMsY0FBYyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLGtFQUFrRSxDQUFDLENBQUM7U0FDckY7YUFBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLEVBQUU7WUFDL0IsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1NBQzVDO2FBQU0sSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDakMsTUFBTSxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUM7WUFDaEQsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLCtCQUFxQixDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7Z0JBQy9ELGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztnQkFDcEMsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO2dCQUNkLGFBQWEsRUFBRSxnQkFBZ0IsQ0FBQyxhQUFhO2dCQUM3QyxRQUFRLEVBQUUsZ0JBQWdCLENBQUMsUUFBUTtnQkFDbkMsTUFBTSxFQUFFLGdCQUFnQixDQUFDLE1BQU07YUFDaEMsQ0FBQyxDQUFDO1lBRUgsTUFBTSxhQUFhLFNBQUcsZ0JBQWdCLENBQUMsYUFBYSxtQ0FBSSxLQUFLLENBQUM7WUFDOUQsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsYUFBYSxFQUFFO2dCQUNoRSxLQUFLLEVBQUUsZ0JBQWdCLENBQUMsS0FBSztnQkFDN0IsT0FBTyxFQUFFLElBQUksQ0FBQyxTQUFTO2dCQUN2QixXQUFXLEVBQUUsZ0JBQWdCLENBQUMsV0FBVztnQkFDekMsT0FBTyxFQUFFLGdCQUFnQixDQUFDLE9BQU87YUFDbEMsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxnQkFBZ0IsQ0FBQyxjQUFjLEVBQUU7Z0JBQ25DLEtBQUssTUFBTSxhQUFhLElBQUksZ0JBQWdCLENBQUMsY0FBYyxFQUFFO29CQUMzRCxTQUFTLENBQUMsZUFBZSxDQUFDO3dCQUN4QixhQUFhO3FCQUNkLENBQUMsQ0FBQztpQkFDSjthQUNGO1NBQ0Y7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELENBQUMsQ0FBQztTQUNyRTtRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO1lBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQUMsb0RBQW9ELENBQUMsQ0FBQztTQUN2RTtRQUNELElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNsRSxJQUFJLENBQUMsY0FBYyxDQUFDLGdCQUFnQixDQUFDLGVBQWUsQ0FBQztnQkFDbkQsYUFBYSxFQUFFLEVBQUU7YUFDbEIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoRCxJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDdEIsSUFBSSxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO1lBQ3hGLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDcEg7YUFBTTtZQUNMLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFO2dCQUNqRCxPQUFPLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDO2dCQUN2QixJQUFJLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYTthQUN6RSxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFTyxvQkFBb0IsQ0FBQyxLQUF5RDtRQUNwRixPQUFPLElBQUksd0JBQWMsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ3pDLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztZQUNyQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDL0IsY0FBYyxFQUFFLElBQUksQ0FBQyxjQUFjO1lBQ25DLGNBQWMsRUFBRSxJQUFJLENBQUMsY0FBYztZQUNuQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtZQUNwRCxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7WUFDbEMsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLG9CQUFvQjtZQUNoRCxlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWU7WUFDdEMsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlO1NBQ3ZDLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQWhHRCxzR0FnR0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBGYXJnYXRlUGxhdGZvcm1WZXJzaW9uLCBGYXJnYXRlU2VydmljZSwgRmFyZ2F0ZVRhc2tEZWZpbml0aW9uIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjcyc7XG5pbXBvcnQgeyBBcHBsaWNhdGlvblRhcmdldEdyb3VwIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWVsYXN0aWNsb2FkYmFsYW5jaW5ndjInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQge1xuICBBcHBsaWNhdGlvbk11bHRpcGxlVGFyZ2V0R3JvdXBzU2VydmljZUJhc2UsXG4gIEFwcGxpY2F0aW9uTXVsdGlwbGVUYXJnZXRHcm91cHNTZXJ2aWNlQmFzZVByb3BzLFxufSBmcm9tICcuLi9iYXNlL2FwcGxpY2F0aW9uLW11bHRpcGxlLXRhcmdldC1ncm91cHMtc2VydmljZS1iYXNlJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBcHBsaWNhdGlvbk11bHRpcGxlVGFyZ2V0R3JvdXBzRmFyZ2F0ZVNlcnZpY2VQcm9wcyBleHRlbmRzIEFwcGxpY2F0aW9uTXVsdGlwbGVUYXJnZXRHcm91cHNTZXJ2aWNlQmFzZVByb3BzIHtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRhc2tEZWZpbml0aW9uPzogRmFyZ2F0ZVRhc2tEZWZpbml0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjcHU/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWVtb3J5TGltaXRNaUI/OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFzc2lnblB1YmxpY0lwPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwbGF0Zm9ybVZlcnNpb24/OiBGYXJnYXRlUGxhdGZvcm1WZXJzaW9uO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEFwcGxpY2F0aW9uTXVsdGlwbGVUYXJnZXRHcm91cHNGYXJnYXRlU2VydmljZSBleHRlbmRzIEFwcGxpY2F0aW9uTXVsdGlwbGVUYXJnZXRHcm91cHNTZXJ2aWNlQmFzZSB7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGFzc2lnblB1YmxpY0lwOiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBzZXJ2aWNlOiBGYXJnYXRlU2VydmljZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB0YXNrRGVmaW5pdGlvbjogRmFyZ2F0ZVRhc2tEZWZpbml0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB0YXJnZXRHcm91cDogQXBwbGljYXRpb25UYXJnZXRHcm91cDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBcHBsaWNhdGlvbk11bHRpcGxlVGFyZ2V0R3JvdXBzRmFyZ2F0ZVNlcnZpY2VQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICB0aGlzLmFzc2lnblB1YmxpY0lwID0gcHJvcHMuYXNzaWduUHVibGljSXAgPz8gZmFsc2U7XG5cbiAgICBpZiAocHJvcHMudGFza0RlZmluaXRpb24gJiYgcHJvcHMudGFza0ltYWdlT3B0aW9ucykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgbXVzdCBzcGVjaWZ5IG9ubHkgb25lIG9mIFRhc2tEZWZpbml0aW9uIG9yIFRhc2tJbWFnZU9wdGlvbnMuJyk7XG4gICAgfSBlbHNlIGlmIChwcm9wcy50YXNrRGVmaW5pdGlvbikge1xuICAgICAgdGhpcy50YXNrRGVmaW5pdGlvbiA9IHByb3BzLnRhc2tEZWZpbml0aW9uO1xuICAgIH0gZWxzZSBpZiAocHJvcHMudGFza0ltYWdlT3B0aW9ucykge1xuICAgICAgY29uc3QgdGFza0ltYWdlT3B0aW9ucyA9IHByb3BzLnRhc2tJbWFnZU9wdGlvbnM7XG4gICAgICB0aGlzLnRhc2tEZWZpbml0aW9uID0gbmV3IEZhcmdhdGVUYXNrRGVmaW5pdGlvbih0aGlzLCAnVGFza0RlZicsIHtcbiAgICAgICAgbWVtb3J5TGltaXRNaUI6IHByb3BzLm1lbW9yeUxpbWl0TWlCLFxuICAgICAgICBjcHU6IHByb3BzLmNwdSxcbiAgICAgICAgZXhlY3V0aW9uUm9sZTogdGFza0ltYWdlT3B0aW9ucy5leGVjdXRpb25Sb2xlLFxuICAgICAgICB0YXNrUm9sZTogdGFza0ltYWdlT3B0aW9ucy50YXNrUm9sZSxcbiAgICAgICAgZmFtaWx5OiB0YXNrSW1hZ2VPcHRpb25zLmZhbWlseSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBjb250YWluZXJOYW1lID0gdGFza0ltYWdlT3B0aW9ucy5jb250YWluZXJOYW1lID8/ICd3ZWInO1xuICAgICAgY29uc3QgY29udGFpbmVyID0gdGhpcy50YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoY29udGFpbmVyTmFtZSwge1xuICAgICAgICBpbWFnZTogdGFza0ltYWdlT3B0aW9ucy5pbWFnZSxcbiAgICAgICAgbG9nZ2luZzogdGhpcy5sb2dEcml2ZXIsXG4gICAgICAgIGVudmlyb25tZW50OiB0YXNrSW1hZ2VPcHRpb25zLmVudmlyb25tZW50LFxuICAgICAgICBzZWNyZXRzOiB0YXNrSW1hZ2VPcHRpb25zLnNlY3JldHMsXG4gICAgICB9KTtcbiAgICAgIGlmICh0YXNrSW1hZ2VPcHRpb25zLmNvbnRhaW5lclBvcnRzKSB7XG4gICAgICAgIGZvciAoY29uc3QgY29udGFpbmVyUG9ydCBvZiB0YXNrSW1hZ2VPcHRpb25zLmNvbnRhaW5lclBvcnRzKSB7XG4gICAgICAgICAgY29udGFpbmVyLmFkZFBvcnRNYXBwaW5ncyh7XG4gICAgICAgICAgICBjb250YWluZXJQb3J0LFxuICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignWW91IG11c3Qgc3BlY2lmeSBvbmUgb2Y6IHRhc2tEZWZpbml0aW9uIG9yIGltYWdlJyk7XG4gICAgfVxuICAgIGlmICghdGhpcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0IGxlYXN0IG9uZSBlc3NlbnRpYWwgY29udGFpbmVyIG11c3QgYmUgc3BlY2lmaWVkJyk7XG4gICAgfVxuICAgIGlmICh0aGlzLnRhc2tEZWZpbml0aW9uLmRlZmF1bHRDb250YWluZXIucG9ydE1hcHBpbmdzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhpcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyLmFkZFBvcnRNYXBwaW5ncyh7XG4gICAgICAgIGNvbnRhaW5lclBvcnQ6IDgwLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgdGhpcy5zZXJ2aWNlID0gdGhpcy5jcmVhdGVGYXJnYXRlU2VydmljZShwcm9wcyk7XG4gICAgaWYgKHByb3BzLnRhcmdldEdyb3Vwcykge1xuICAgICAgdGhpcy5hZGRQb3J0TWFwcGluZ0ZvclRhcmdldHModGhpcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyLCBwcm9wcy50YXJnZXRHcm91cHMpO1xuICAgICAgdGhpcy50YXJnZXRHcm91cCA9IHRoaXMucmVnaXN0ZXJFQ1NUYXJnZXRzKHRoaXMuc2VydmljZSwgdGhpcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyLCBwcm9wcy50YXJnZXRHcm91cHMpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnRhcmdldEdyb3VwID0gdGhpcy5saXN0ZW5lci5hZGRUYXJnZXRzKCdFQ1MnLCB7XG4gICAgICAgIHRhcmdldHM6IFt0aGlzLnNlcnZpY2VdLFxuICAgICAgICBwb3J0OiB0aGlzLnRhc2tEZWZpbml0aW9uLmRlZmF1bHRDb250YWluZXIucG9ydE1hcHBpbmdzWzBdLmNvbnRhaW5lclBvcnQsXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZUZhcmdhdGVTZXJ2aWNlKHByb3BzOiBBcHBsaWNhdGlvbk11bHRpcGxlVGFyZ2V0R3JvdXBzRmFyZ2F0ZVNlcnZpY2VQcm9wcyk6IEZhcmdhdGVTZXJ2aWNlIHtcbiAgICByZXR1cm4gbmV3IEZhcmdhdGVTZXJ2aWNlKHRoaXMsICdTZXJ2aWNlJywge1xuICAgICAgY2x1c3RlcjogdGhpcy5jbHVzdGVyLFxuICAgICAgZGVzaXJlZENvdW50OiB0aGlzLmRlc2lyZWRDb3VudCxcbiAgICAgIHRhc2tEZWZpbml0aW9uOiB0aGlzLnRhc2tEZWZpbml0aW9uLFxuICAgICAgYXNzaWduUHVibGljSXA6IHRoaXMuYXNzaWduUHVibGljSXAsXG4gICAgICBzZXJ2aWNlTmFtZTogcHJvcHMuc2VydmljZU5hbWUsXG4gICAgICBoZWFsdGhDaGVja0dyYWNlUGVyaW9kOiBwcm9wcy5oZWFsdGhDaGVja0dyYWNlUGVyaW9kLFxuICAgICAgcHJvcGFnYXRlVGFnczogcHJvcHMucHJvcGFnYXRlVGFncyxcbiAgICAgIGVuYWJsZUVDU01hbmFnZWRUYWdzOiBwcm9wcy5lbmFibGVFQ1NNYW5hZ2VkVGFncyxcbiAgICAgIGNsb3VkTWFwT3B0aW9uczogcHJvcHMuY2xvdWRNYXBPcHRpb25zLFxuICAgICAgcGxhdGZvcm1WZXJzaW9uOiBwcm9wcy5wbGF0Zm9ybVZlcnNpb24sXG4gICAgfSk7XG4gIH1cbn0iXX0=