"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueueProcessingServiceBase = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * The base class for QueueProcessingEc2Service and QueueProcessingFargateService services.
 *
 * @stability stable
 */
class QueueProcessingServiceBase extends core_2.Construct {
    /**
     * Constructs a new instance of the QueueProcessingServiceBase class.
     *
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        // Create the SQS queue and it's corresponding DLQ if one is not provided
        if (props.queue) {
            this.sqsQueue = props.queue;
        }
        else {
            this.deadLetterQueue = new aws_sqs_1.Queue(this, 'EcsProcessingDeadLetterQueue', {
                retentionPeriod: props.retentionPeriod || core_1.Duration.days(14),
            });
            this.sqsQueue = new aws_sqs_1.Queue(this, 'EcsProcessingQueue', {
                deadLetterQueue: {
                    queue: this.deadLetterQueue,
                    maxReceiveCount: props.maxReceiveCount || 3,
                },
            });
            new core_1.CfnOutput(this, 'SQSDeadLetterQueue', { value: this.deadLetterQueue.queueName });
            new core_1.CfnOutput(this, 'SQSDeadLetterQueueArn', { value: this.deadLetterQueue.queueArn });
        }
        // Setup autoscaling scaling intervals
        const defaultScalingSteps = [{ upper: 0, change: -1 }, { lower: 100, change: +1 }, { lower: 500, change: +5 }];
        this.scalingSteps = (_a = props.scalingSteps) !== null && _a !== void 0 ? _a : defaultScalingSteps;
        // Create log driver if logging is enabled
        const enableLogging = (_b = props.enableLogging) !== null && _b !== void 0 ? _b : true;
        this.logDriver = (_c = props.logDriver) !== null && _c !== void 0 ? _c : (enableLogging ? this.createAWSLogDriver(this.node.id) : undefined);
        // Add the queue name to environment variables
        this.environment = { ...(props.environment || {}), QUEUE_NAME: this.sqsQueue.queueName };
        this.secrets = props.secrets;
        // Determine the desired task count (minimum) and maximum scaling capacity
        this.desiredCount = (_d = props.desiredTaskCount) !== null && _d !== void 0 ? _d : 1;
        this.maxCapacity = props.maxScalingCapacity || (2 * this.desiredCount);
        if (!this.desiredCount && !this.maxCapacity) {
            throw new Error('maxScalingCapacity must be set and greater than 0 if desiredCount is 0');
        }
        new core_1.CfnOutput(this, 'SQSQueue', { value: this.sqsQueue.queueName });
        new core_1.CfnOutput(this, 'SQSQueueArn', { value: this.sqsQueue.queueArn });
    }
    /**
     * Configure autoscaling based off of CPU utilization as well as the number of messages visible in the SQS queue.
     *
     * @param service the ECS/Fargate service for which to apply the autoscaling rules to.
     * @stability stable
     */
    configureAutoscalingForService(service) {
        const scalingTarget = service.autoScaleTaskCount({ maxCapacity: this.maxCapacity, minCapacity: this.desiredCount });
        scalingTarget.scaleOnCpuUtilization('CpuScaling', {
            targetUtilizationPercent: 50,
        });
        scalingTarget.scaleOnMetric('QueueMessagesVisibleScaling', {
            metric: this.sqsQueue.metricApproximateNumberOfMessagesVisible(),
            scalingSteps: this.scalingSteps,
        });
    }
    /**
     * Grant SQS permissions to an ECS service.
     *
     * @param service the ECS/Fargate service to which to grant SQS permissions.
     * @stability stable
     */
    grantPermissionsToService(service) {
        this.sqsQueue.grantConsumeMessages(service.taskDefinition.taskRole);
    }
    /**
     * Returns the default cluster.
     *
     * @stability stable
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Create an AWS Log Driver with the provided streamPrefix
     *
     * @param prefix the Cloudwatch logging prefix
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.QueueProcessingServiceBase = QueueProcessingServiceBase;
//# sourceMappingURL=data:application/json;base64,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