"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkMultipleTargetGroupsServiceBase = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const core_1 = require("@aws-cdk/core");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * The base class for NetworkMultipleTargetGroupsEc2Service and NetworkMultipleTargetGroupsFargateService classes.
 *
 * @stability stable
 */
class NetworkMultipleTargetGroupsServiceBase extends core_2.Construct {
    /**
     * Constructs a new instance of the NetworkMultipleTargetGroupsServiceBase class.
     *
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * @stability stable
         */
        this.listeners = new Array();
        /**
         * @stability stable
         */
        this.targetGroups = new Array();
        this.loadBalancers = new Array();
        this.validateInput(props);
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        this.desiredCount = props.desiredCount || 1;
        if (props.taskImageOptions) {
            this.logDriver = this.createLogDriver(props.taskImageOptions.enableLogging, props.taskImageOptions.logDriver);
        }
        if (props.loadBalancers) {
            for (const lbProps of props.loadBalancers) {
                const lb = this.createLoadBalancer(lbProps.name, lbProps.publicLoadBalancer);
                this.loadBalancers.push(lb);
                for (const listenerProps of lbProps.listeners) {
                    const listener = this.createListener(listenerProps.name, lb, listenerProps.port || 80);
                    this.listeners.push(listener);
                }
                this.createDomainName(lb, lbProps.domainName, lbProps.domainZone);
                new core_1.CfnOutput(this, `LoadBalancerDNS${lb.node.id}`, { value: lb.loadBalancerDnsName });
            }
            // set up default load balancer and listener.
            this.loadBalancer = this.loadBalancers[0];
            this.listener = this.listeners[0];
        }
        else {
            this.loadBalancer = this.createLoadBalancer('LB');
            this.listener = this.createListener('PublicListener', this.loadBalancer, 80);
            this.createDomainName(this.loadBalancer);
            new core_1.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
        }
    }
    /**
     * Returns the default cluster.
     *
     * @stability stable
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs.
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * @stability stable
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
    /**
     * @stability stable
     */
    findListener(name) {
        if (!name) {
            return this.listener;
        }
        for (const listener of this.listeners) {
            if (listener.node.id === name) {
                return listener;
            }
        }
        throw new Error(`Listener ${name} is not defined. Did you define listener with name ${name}?`);
    }
    /**
     * @stability stable
     */
    registerECSTargets(service, container, targets) {
        for (const targetProps of targets) {
            const targetGroup = this.findListener(targetProps.listener).addTargets(`ECSTargetGroup${container.containerName}${targetProps.containerPort}`, {
                port: 80,
                targets: [
                    service.loadBalancerTarget({
                        containerName: container.containerName,
                        containerPort: targetProps.containerPort,
                    }),
                ],
            });
            this.targetGroups.push(targetGroup);
        }
        if (this.targetGroups.length === 0) {
            throw new Error('At least one target group should be specified.');
        }
        return this.targetGroups[0];
    }
    /**
     * @stability stable
     */
    addPortMappingForTargets(container, targets) {
        for (const target of targets) {
            if (!container.findPortMapping(target.containerPort, aws_ecs_1.Protocol.TCP)) {
                container.addPortMappings({
                    containerPort: target.containerPort,
                });
            }
        }
    }
    /**
     * Create log driver if logging is enabled.
     */
    createLogDriver(enableLoggingProp, logDriverProp) {
        const enableLogging = enableLoggingProp !== null && enableLoggingProp !== void 0 ? enableLoggingProp : true;
        const logDriver = logDriverProp !== null && logDriverProp !== void 0 ? logDriverProp : (enableLogging ? this.createAWSLogDriver(this.node.id) : undefined);
        return logDriver;
    }
    validateInput(props) {
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        if (props.loadBalancers) {
            if (props.loadBalancers.length === 0) {
                throw new Error('At least one load balancer must be specified');
            }
            for (const lbProps of props.loadBalancers) {
                if (lbProps.listeners.length === 0) {
                    throw new Error('At least one listener must be specified');
                }
            }
        }
    }
    createLoadBalancer(name, publicLoadBalancer) {
        const internetFacing = publicLoadBalancer !== null && publicLoadBalancer !== void 0 ? publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        return new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(this, name, lbProps);
    }
    createListener(name, lb, port) {
        return lb.addListener(name, {
            port,
        });
    }
    createDomainName(loadBalancer, name, zone) {
        if (typeof name !== 'undefined') {
            if (typeof zone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            new aws_route53_1.ARecord(this, `DNS${loadBalancer.node.id}`, {
                zone,
                recordName: name,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
        }
    }
}
exports.NetworkMultipleTargetGroupsServiceBase = NetworkMultipleTargetGroupsServiceBase;
//# sourceMappingURL=data:application/json;base64,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