"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationMultipleTargetGroupsServiceBase = void 0;
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const core_1 = require("@aws-cdk/core");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * The base class for ApplicationMultipleTargetGroupsEc2Service and ApplicationMultipleTargetGroupsFargateService classes.
 *
 * @stability stable
 */
class ApplicationMultipleTargetGroupsServiceBase extends core_2.Construct {
    /**
     * Constructs a new instance of the ApplicationMultipleTargetGroupsServiceBase class.
     *
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * @stability stable
         */
        this.listeners = new Array();
        /**
         * @stability stable
         */
        this.targetGroups = new Array();
        this.loadBalancers = new Array();
        this.validateInput(props);
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        this.desiredCount = props.desiredCount || 1;
        if (props.taskImageOptions) {
            this.logDriver = this.createLogDriver(props.taskImageOptions.enableLogging, props.taskImageOptions.logDriver);
        }
        if (props.loadBalancers) {
            for (const lbProps of props.loadBalancers) {
                const lb = this.createLoadBalancer(lbProps.name, lbProps.publicLoadBalancer);
                this.loadBalancers.push(lb);
                const protocolType = new Set();
                for (const listenerProps of lbProps.listeners) {
                    const protocol = this.createListenerProtocol(listenerProps.protocol, listenerProps.certificate);
                    if (listenerProps.certificate !== undefined && protocol !== undefined && protocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                        throw new Error('The HTTPS protocol must be used when a certificate is given');
                    }
                    protocolType.add(protocol);
                    const listener = this.configListener(protocol, {
                        certificate: listenerProps.certificate,
                        domainName: lbProps.domainName,
                        domainZone: lbProps.domainZone,
                        listenerName: listenerProps.name,
                        loadBalancer: lb,
                        port: listenerProps.port,
                    });
                    this.listeners.push(listener);
                }
                const domainName = this.createDomainName(lb, lbProps.domainName, lbProps.domainZone);
                new core_1.CfnOutput(this, `LoadBalancerDNS${lb.node.id}`, { value: lb.loadBalancerDnsName });
                for (const protocol of protocolType) {
                    new core_1.CfnOutput(this, `ServiceURL${lb.node.id}${protocol.toLowerCase()}`, { value: protocol.toLowerCase() + '://' + domainName });
                }
            }
            // set up default load balancer and listener.
            this.loadBalancer = this.loadBalancers[0];
            this.listener = this.listeners[0];
        }
        else {
            this.loadBalancer = this.createLoadBalancer('LB');
            const protocol = this.createListenerProtocol();
            this.listener = this.configListener(protocol, {
                listenerName: 'PublicListener',
                loadBalancer: this.loadBalancer,
            });
            const domainName = this.createDomainName(this.loadBalancer);
            new core_1.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
            new core_1.CfnOutput(this, 'ServiceURL', { value: protocol.toLowerCase() + '://' + domainName });
        }
    }
    /**
     * Returns the default cluster.
     *
     * @stability stable
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs.
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * @stability stable
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
    /**
     * @stability stable
     */
    findListener(name) {
        if (!name) {
            return this.listener;
        }
        for (const listener of this.listeners) {
            if (listener.node.id === name) {
                return listener;
            }
        }
        throw new Error(`Listener ${name} is not defined. Did you define listener with name ${name}?`);
    }
    /**
     * @stability stable
     */
    registerECSTargets(service, container, targets) {
        for (const targetProps of targets) {
            const targetGroup = this.findListener(targetProps.listener).addTargets(`ECSTargetGroup${container.containerName}${targetProps.containerPort}`, {
                port: 80,
                targets: [
                    service.loadBalancerTarget({
                        containerName: container.containerName,
                        containerPort: targetProps.containerPort,
                        protocol: targetProps.protocol,
                    }),
                ],
                hostHeader: targetProps.hostHeader,
                pathPattern: targetProps.pathPattern,
                priority: targetProps.priority,
            });
            this.targetGroups.push(targetGroup);
        }
        if (this.targetGroups.length === 0) {
            throw new Error('At least one target group should be specified.');
        }
        return this.targetGroups[0];
    }
    /**
     * @stability stable
     */
    addPortMappingForTargets(container, targets) {
        for (const target of targets) {
            if (!container.findPortMapping(target.containerPort, target.protocol || aws_ecs_1.Protocol.TCP)) {
                container.addPortMappings({
                    containerPort: target.containerPort,
                    protocol: target.protocol,
                });
            }
        }
    }
    /**
     * Create log driver if logging is enabled.
     */
    createLogDriver(enableLoggingProp, logDriverProp) {
        const enableLogging = enableLoggingProp !== null && enableLoggingProp !== void 0 ? enableLoggingProp : true;
        const logDriver = logDriverProp !== null && logDriverProp !== void 0 ? logDriverProp : (enableLogging ? this.createAWSLogDriver(this.node.id) : undefined);
        return logDriver;
    }
    configListener(protocol, props) {
        const listener = this.createListener(props.listenerName, props.loadBalancer, protocol, props.port);
        let certificate;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            certificate = this.createListenerCertificate(props.listenerName, props.certificate, props.domainName, props.domainZone);
        }
        else {
            certificate = undefined;
        }
        if (certificate !== undefined) {
            listener.addCertificateArns(`Arns${props.listenerName}`, [certificate.certificateArn]);
        }
        return listener;
    }
    validateInput(props) {
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        if (props.loadBalancers) {
            if (props.loadBalancers.length === 0) {
                throw new Error('At least one load balancer must be specified');
            }
            for (const lbProps of props.loadBalancers) {
                if (lbProps.listeners.length === 0) {
                    throw new Error('At least one listener must be specified');
                }
            }
        }
    }
    createLoadBalancer(name, publicLoadBalancer) {
        const internetFacing = publicLoadBalancer !== null && publicLoadBalancer !== void 0 ? publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        return new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, name, lbProps);
    }
    createListenerProtocol(listenerProtocol, certificate) {
        return listenerProtocol !== null && listenerProtocol !== void 0 ? listenerProtocol : (certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
    }
    createListenerCertificate(listenerName, certificate, domainName, domainZone) {
        if (typeof domainName === 'undefined' || typeof domainZone === 'undefined') {
            throw new Error('A domain name and zone is required when using the HTTPS protocol');
        }
        if (certificate !== undefined) {
            return certificate;
        }
        else {
            return new aws_certificatemanager_1.Certificate(this, `Certificate${listenerName}`, {
                domainName,
                validation: aws_certificatemanager_1.CertificateValidation.fromDns(domainZone),
            });
        }
    }
    createListener(name, lb, protocol, port) {
        return lb.addListener(name, {
            protocol,
            open: true,
            port,
        });
    }
    createDomainName(loadBalancer, name, zone) {
        let domainName = loadBalancer.loadBalancerDnsName;
        if (typeof name !== 'undefined') {
            if (typeof zone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            const record = new aws_route53_1.ARecord(this, `DNS${loadBalancer.node.id}`, {
                zone,
                recordName: name,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
            domainName = record.domainName;
        }
        return domainName;
    }
}
exports.ApplicationMultipleTargetGroupsServiceBase = ApplicationMultipleTargetGroupsServiceBase;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbXVsdGlwbGUtdGFyZ2V0LWdyb3Vwcy1zZXJ2aWNlLWJhc2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcHBsaWNhdGlvbi1tdWx0aXBsZS10YXJnZXQtZ3JvdXBzLXNlcnZpY2UtYmFzZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw0RUFBbUc7QUFFbkcsOENBRzBCO0FBQzFCLG9GQUFnSjtBQUVoSixzREFBMEU7QUFDMUUsc0VBQWtFO0FBQ2xFLHdDQUEyRDtBQUczRCxnSEFBZ0g7QUFDaEgsMkJBQTJCO0FBQzNCLHdDQUEyRDs7Ozs7O0FBMlQzRCxNQUFzQiwwQ0FBMkMsU0FBUSxnQkFBYTs7Ozs7O0lBK0JwRixZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQXlELEVBQUU7UUFDbkcsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQzs7OztRQVRULGNBQVMsR0FBRyxJQUFJLEtBQUssRUFBdUIsQ0FBQzs7OztRQUM3QyxpQkFBWSxHQUFHLElBQUksS0FBSyxFQUEwQixDQUFDO1FBRXJELGtCQUFhLEdBQUcsSUFBSSxLQUFLLEVBQTJCLENBQUM7UUFRM0QsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUxQixJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLElBQUksSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDeEUsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLENBQUMsQ0FBQztRQUM1QyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDL0c7UUFFRCxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDdkIsS0FBSyxNQUFNLE9BQU8sSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO2dCQUN6QyxNQUFNLEVBQUUsR0FBRyxJQUFJLENBQUMsa0JBQWtCLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsa0JBQWtCLENBQUMsQ0FBQztnQkFDN0UsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7Z0JBQzVCLE1BQU0sWUFBWSxHQUFHLElBQUksR0FBRyxFQUF1QixDQUFDO2dCQUNwRCxLQUFLLE1BQU0sYUFBYSxJQUFJLE9BQU8sQ0FBQyxTQUFTLEVBQUU7b0JBQzdDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxhQUFhLENBQUMsUUFBUSxFQUFFLGFBQWEsQ0FBQyxXQUFXLENBQUMsQ0FBQztvQkFDaEcsSUFBSSxhQUFhLENBQUMsV0FBVyxLQUFLLFNBQVMsSUFBSSxRQUFRLEtBQUssU0FBUyxJQUFJLFFBQVEsS0FBSyxnREFBbUIsQ0FBQyxLQUFLLEVBQUU7d0JBQy9HLE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELENBQUMsQ0FBQztxQkFDaEY7b0JBQ0QsWUFBWSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQztvQkFDM0IsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUU7d0JBQzdDLFdBQVcsRUFBRSxhQUFhLENBQUMsV0FBVzt3QkFDdEMsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVO3dCQUM5QixVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQVU7d0JBQzlCLFlBQVksRUFBRSxhQUFhLENBQUMsSUFBSTt3QkFDaEMsWUFBWSxFQUFFLEVBQUU7d0JBQ2hCLElBQUksRUFBRSxhQUFhLENBQUMsSUFBSTtxQkFDekIsQ0FBQyxDQUFDO29CQUNILElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO2lCQUMvQjtnQkFDRCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxFQUFFLE9BQU8sQ0FBQyxVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO2dCQUNyRixJQUFJLGdCQUFTLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLEVBQUUsQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUM7Z0JBQ3ZGLEtBQUssTUFBTSxRQUFRLElBQUksWUFBWSxFQUFFO29CQUNuQyxJQUFJLGdCQUFTLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRSxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsUUFBUSxDQUFDLFdBQVcsRUFBRSxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLFdBQVcsRUFBRSxHQUFHLEtBQUssR0FBRyxVQUFVLEVBQUUsQ0FBQyxDQUFDO2lCQUNqSTthQUNGO1lBQ0QsNkNBQTZDO1lBQzdDLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUMxQyxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDbkM7YUFBTTtZQUNMLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ2xELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxzQkFBc0IsRUFBRSxDQUFDO1lBQy9DLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLEVBQUU7Z0JBQzVDLFlBQVksRUFBRSxnQkFBZ0I7Z0JBQzlCLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTthQUNoQyxDQUFDLENBQUM7WUFDSCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1lBRTVELElBQUksZ0JBQVMsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUM7WUFDekYsSUFBSSxnQkFBUyxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLFdBQVcsRUFBRSxHQUFHLEtBQUssR0FBRyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1NBQzNGO0lBQ0gsQ0FBQzs7Ozs7O0lBS1MsaUJBQWlCLENBQUMsS0FBZ0IsRUFBRSxHQUFVO1FBQ3RELGdFQUFnRTtRQUNoRSxNQUFNLGtCQUFrQixHQUFHLDZCQUE2QixHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQztRQUNqRixNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE9BQU8sS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsa0JBQWtCLENBQVksSUFBSSxJQUFJLGlCQUFPLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztJQUNuSCxDQUFDOzs7O0lBRVMsa0JBQWtCLENBQUMsTUFBYztRQUN6QyxPQUFPLElBQUksc0JBQVksQ0FBQyxFQUFFLFlBQVksRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO0lBQ3BELENBQUM7Ozs7SUFFUyxZQUFZLENBQUMsSUFBYTtRQUNsQyxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ1QsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDO1NBQ3RCO1FBQ0QsS0FBSyxNQUFNLFFBQVEsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ3JDLElBQUksUUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLEtBQUssSUFBSSxFQUFFO2dCQUM3QixPQUFPLFFBQVEsQ0FBQzthQUNqQjtTQUNGO1FBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxZQUFZLElBQUksc0RBQXNELElBQUksR0FBRyxDQUFDLENBQUM7SUFDakcsQ0FBQzs7OztJQUVTLGtCQUFrQixDQUFDLE9BQW9CLEVBQUUsU0FBOEIsRUFBRSxPQUFpQztRQUNsSCxLQUFLLE1BQU0sV0FBVyxJQUFJLE9BQU8sRUFBRTtZQUNqQyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQyxVQUFVLENBQUMsaUJBQWlCLFNBQVMsQ0FBQyxhQUFhLEdBQUcsV0FBVyxDQUFDLGFBQWEsRUFBRSxFQUFFO2dCQUM3SSxJQUFJLEVBQUUsRUFBRTtnQkFDUixPQUFPLEVBQUU7b0JBQ1AsT0FBTyxDQUFDLGtCQUFrQixDQUFDO3dCQUN6QixhQUFhLEVBQUUsU0FBUyxDQUFDLGFBQWE7d0JBQ3RDLGFBQWEsRUFBRSxXQUFXLENBQUMsYUFBYTt3QkFDeEMsUUFBUSxFQUFFLFdBQVcsQ0FBQyxRQUFRO3FCQUMvQixDQUFDO2lCQUNIO2dCQUNELFVBQVUsRUFBRSxXQUFXLENBQUMsVUFBVTtnQkFDbEMsV0FBVyxFQUFFLFdBQVcsQ0FBQyxXQUFXO2dCQUNwQyxRQUFRLEVBQUUsV0FBVyxDQUFDLFFBQVE7YUFDL0IsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDckM7UUFDRCxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNsQyxNQUFNLElBQUksS0FBSyxDQUFDLGdEQUFnRCxDQUFDLENBQUM7U0FDbkU7UUFDRCxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDOUIsQ0FBQzs7OztJQUVTLHdCQUF3QixDQUFDLFNBQThCLEVBQUUsT0FBaUM7UUFDbEcsS0FBSyxNQUFNLE1BQU0sSUFBSSxPQUFPLEVBQUU7WUFDNUIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxNQUFNLENBQUMsUUFBUSxJQUFJLGtCQUFRLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQ3JGLFNBQVMsQ0FBQyxlQUFlLENBQUM7b0JBQ3hCLGFBQWEsRUFBRSxNQUFNLENBQUMsYUFBYTtvQkFDbkMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRO2lCQUMxQixDQUFDLENBQUM7YUFDSjtTQUNGO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0ssZUFBZSxDQUFDLGlCQUEyQixFQUFFLGFBQXlCO1FBQzVFLE1BQU0sYUFBYSxHQUFHLGlCQUFpQixhQUFqQixpQkFBaUIsY0FBakIsaUJBQWlCLEdBQUksSUFBSSxDQUFDO1FBQ2hELE1BQU0sU0FBUyxHQUFHLGFBQWEsYUFBYixhQUFhLGNBQWIsYUFBYSxHQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdkcsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVPLGNBQWMsQ0FBQyxRQUE2QixFQUFFLEtBQXFCO1FBQ3pFLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWSxFQUFFLFFBQVEsRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbkcsSUFBSSxXQUFXLENBQUM7UUFDaEIsSUFBSSxRQUFRLEtBQUssZ0RBQW1CLENBQUMsS0FBSyxFQUFFO1lBQzFDLFdBQVcsR0FBRyxJQUFJLENBQUMseUJBQXlCLENBQUMsS0FBSyxDQUFDLFlBQVksRUFBRSxLQUFLLENBQUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQ3pIO2FBQU07WUFDTCxXQUFXLEdBQUcsU0FBUyxDQUFDO1NBQ3pCO1FBQ0QsSUFBSSxXQUFXLEtBQUssU0FBUyxFQUFFO1lBQzdCLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPLEtBQUssQ0FBQyxZQUFZLEVBQUUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDO1NBQ3hGO1FBRUQsT0FBTyxRQUFRLENBQUM7SUFDbEIsQ0FBQztJQUVPLGFBQWEsQ0FBQyxLQUFzRDtRQUMxRSxJQUFJLEtBQUssQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUM5QixNQUFNLElBQUksS0FBSyxDQUFDLHFGQUFxRixDQUFDLENBQUM7U0FDeEc7UUFFRCxJQUFJLEtBQUssQ0FBQyxZQUFZLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxZQUFZLEdBQUcsQ0FBQyxFQUFFO1lBQzlELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0RBQWdELENBQUMsQ0FBQztTQUNuRTtRQUVELElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUN2QixJQUFJLEtBQUssQ0FBQyxhQUFhLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtnQkFDcEMsTUFBTSxJQUFJLEtBQUssQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO2FBQ2pFO1lBQ0QsS0FBSyxNQUFNLE9BQU8sSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO2dCQUN6QyxJQUFJLE9BQU8sQ0FBQyxTQUFTLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtvQkFDbEMsTUFBTSxJQUFJLEtBQUssQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO2lCQUM1RDthQUNGO1NBQ0Y7SUFDSCxDQUFDO0lBRU8sa0JBQWtCLENBQUMsSUFBWSxFQUFFLGtCQUE0QjtRQUNuRSxNQUFNLGNBQWMsR0FBRyxrQkFBa0IsYUFBbEIsa0JBQWtCLGNBQWxCLGtCQUFrQixHQUFJLElBQUksQ0FBQztRQUNsRCxNQUFNLE9BQU8sR0FBRztZQUNkLEdBQUcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUc7WUFDckIsY0FBYztTQUNmLENBQUM7UUFFRixPQUFPLElBQUksb0RBQXVCLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMxRCxDQUFDO0lBRU8sc0JBQXNCLENBQUMsZ0JBQXNDLEVBQUUsV0FBMEI7UUFDL0YsT0FBTyxnQkFBZ0IsYUFBaEIsZ0JBQWdCLGNBQWhCLGdCQUFnQixHQUFJLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxnREFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLGdEQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2xHLENBQUM7SUFFTyx5QkFBeUIsQ0FBQyxZQUFvQixFQUFFLFdBQTBCLEVBQUUsVUFBbUIsRUFBRSxVQUF3QjtRQUMvSCxJQUFJLE9BQU8sVUFBVSxLQUFLLFdBQVcsSUFBSSxPQUFPLFVBQVUsS0FBSyxXQUFXLEVBQUU7WUFDMUUsTUFBTSxJQUFJLEtBQUssQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDO1NBQ3JGO1FBRUQsSUFBSSxXQUFXLEtBQUssU0FBUyxFQUFFO1lBQzdCLE9BQU8sV0FBVyxDQUFDO1NBQ3BCO2FBQU07WUFDTCxPQUFPLElBQUksb0NBQVcsQ0FBQyxJQUFJLEVBQUUsY0FBYyxZQUFZLEVBQUUsRUFBRTtnQkFDekQsVUFBVTtnQkFDVixVQUFVLEVBQUUsOENBQXFCLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQzthQUN0RCxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFTyxjQUFjLENBQUMsSUFBWSxFQUFFLEVBQTJCLEVBQUUsUUFBOEIsRUFBRSxJQUFhO1FBQzdHLE9BQU8sRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUU7WUFDMUIsUUFBUTtZQUNSLElBQUksRUFBRSxJQUFJO1lBQ1YsSUFBSTtTQUNMLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxnQkFBZ0IsQ0FBQyxZQUFxQyxFQUFFLElBQWEsRUFBRSxJQUFrQjtRQUMvRixJQUFJLFVBQVUsR0FBRyxZQUFZLENBQUMsbUJBQW1CLENBQUM7UUFDbEQsSUFBSSxPQUFPLElBQUksS0FBSyxXQUFXLEVBQUU7WUFDL0IsSUFBSSxPQUFPLElBQUksS0FBSyxXQUFXLEVBQUU7Z0JBQy9CLE1BQU0sSUFBSSxLQUFLLENBQUMsc0ZBQXNGLENBQUMsQ0FBQzthQUN6RztZQUVELE1BQU0sTUFBTSxHQUFHLElBQUkscUJBQU8sQ0FBQyxJQUFJLEVBQUUsTUFBTSxZQUFZLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRSxFQUFFO2dCQUM3RCxJQUFJO2dCQUNKLFVBQVUsRUFBRSxJQUFJO2dCQUNoQixNQUFNLEVBQUUsMEJBQVksQ0FBQyxTQUFTLENBQUMsSUFBSSx3Q0FBa0IsQ0FBQyxZQUFZLENBQUMsQ0FBQzthQUNyRSxDQUFDLENBQUM7WUFFSCxVQUFVLEdBQUcsTUFBTSxDQUFDLFVBQVUsQ0FBQztTQUNoQztRQUNELE9BQU8sVUFBVSxDQUFDO0lBQ3BCLENBQUM7Q0FDRjtBQXJQRCxnR0FxUEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDZXJ0aWZpY2F0ZSwgQ2VydGlmaWNhdGVWYWxpZGF0aW9uLCBJQ2VydGlmaWNhdGUgfSBmcm9tICdAYXdzLWNkay9hd3MtY2VydGlmaWNhdGVtYW5hZ2VyJztcbmltcG9ydCB7IElWcGMgfSBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCB7XG4gIEF3c0xvZ0RyaXZlciwgQmFzZVNlcnZpY2UsIENsb3VkTWFwT3B0aW9ucywgQ2x1c3RlciwgQ29udGFpbmVyRGVmaW5pdGlvbiwgQ29udGFpbmVySW1hZ2UsIElDbHVzdGVyLCBMb2dEcml2ZXIsIFByb3BhZ2F0ZWRUYWdTb3VyY2UsXG4gIFByb3RvY29sLCBTZWNyZXQsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3MnO1xuaW1wb3J0IHsgQXBwbGljYXRpb25MaXN0ZW5lciwgQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIsIEFwcGxpY2F0aW9uUHJvdG9jb2wsIEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAgfSBmcm9tICdAYXdzLWNkay9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mic7XG5pbXBvcnQgeyBJUm9sZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgQVJlY29yZCwgSUhvc3RlZFpvbmUsIFJlY29yZFRhcmdldCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1yb3V0ZTUzJztcbmltcG9ydCB7IExvYWRCYWxhbmNlclRhcmdldCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1yb3V0ZTUzLXRhcmdldHMnO1xuaW1wb3J0IHsgQ2ZuT3V0cHV0LCBEdXJhdGlvbiwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG4vLyB2MiAtIGtlZXAgdGhpcyBpbXBvcnQgYXMgYSBzZXBhcmF0ZSBzZWN0aW9uIHRvIHJlZHVjZSBtZXJnZSBjb25mbGljdCB3aGVuIGZvcndhcmQgbWVyZ2luZyB3aXRoIHRoZSB2MiBicmFuY2guXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmVcbmltcG9ydCB7IENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBcHBsaWNhdGlvbk11bHRpcGxlVGFyZ2V0R3JvdXBzU2VydmljZUJhc2VQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2x1c3Rlcj86IElDbHVzdGVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjPzogSVZwYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0YXNrSW1hZ2VPcHRpb25zPzogQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRUYXNrSW1hZ2VQcm9wcztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlc2lyZWRDb3VudD86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBoZWFsdGhDaGVja0dyYWNlUGVyaW9kPzogRHVyYXRpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzZXJ2aWNlTmFtZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBsb2FkQmFsYW5jZXJzPzogQXBwbGljYXRpb25Mb2FkQmFsYW5jZXJQcm9wc1tdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByb3BhZ2F0ZVRhZ3M/OiBQcm9wYWdhdGVkVGFnU291cmNlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbmFibGVFQ1NNYW5hZ2VkVGFncz86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNsb3VkTWFwT3B0aW9ucz86IENsb3VkTWFwT3B0aW9ucztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRhcmdldEdyb3Vwcz86IEFwcGxpY2F0aW9uVGFyZ2V0UHJvcHNbXTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VkVGFza0ltYWdlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGltYWdlOiBDb250YWluZXJJbWFnZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbnZpcm9ubWVudD86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlY3JldHM/OiB7IFtrZXk6IHN0cmluZ106IFNlY3JldCB9O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5hYmxlTG9nZ2luZz86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvZ0RyaXZlcj86IExvZ0RyaXZlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZXhlY3V0aW9uUm9sZT86IElSb2xlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGFza1JvbGU/OiBJUm9sZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb250YWluZXJOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbnRhaW5lclBvcnRzPzogbnVtYmVyW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGZhbWlseT86IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFwcGxpY2F0aW9uVGFyZ2V0UHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29udGFpbmVyUG9ydDogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJvdG9jb2w/OiBQcm90b2NvbDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbGlzdGVuZXI/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcmlvcml0eT86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBob3N0SGVhZGVyPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwYXRoUGF0dGVybj86IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlclByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxpc3RlbmVyczogQXBwbGljYXRpb25MaXN0ZW5lclByb3BzW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwdWJsaWNMb2FkQmFsYW5jZXI/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkb21haW5OYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkb21haW5ab25lPzogSUhvc3RlZFpvbmU7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFwcGxpY2F0aW9uTGlzdGVuZXJQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJvdG9jb2w/OiBBcHBsaWNhdGlvblByb3RvY29sO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwb3J0PzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNlcnRpZmljYXRlPzogSUNlcnRpZmljYXRlO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgQXBwbGljYXRpb25NdWx0aXBsZVRhcmdldEdyb3Vwc1NlcnZpY2VCYXNlIGV4dGVuZHMgQ29yZUNvbnN0cnVjdCB7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZGVzaXJlZENvdW50OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgbG9hZEJhbGFuY2VyOiBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGxpc3RlbmVyOiBBcHBsaWNhdGlvbkxpc3RlbmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBjbHVzdGVyOiBJQ2x1c3RlcjtcblxuICBwcm90ZWN0ZWQgbG9nRHJpdmVyPzogTG9nRHJpdmVyO1xuICBwcm90ZWN0ZWQgbGlzdGVuZXJzID0gbmV3IEFycmF5PEFwcGxpY2F0aW9uTGlzdGVuZXI+KCk7XG4gIHByb3RlY3RlZCB0YXJnZXRHcm91cHMgPSBuZXcgQXJyYXk8QXBwbGljYXRpb25UYXJnZXRHcm91cD4oKTtcblxuICBwcml2YXRlIGxvYWRCYWxhbmNlcnMgPSBuZXcgQXJyYXk8QXBwbGljYXRpb25Mb2FkQmFsYW5jZXI+KCk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXBwbGljYXRpb25NdWx0aXBsZVRhcmdldEdyb3Vwc1NlcnZpY2VCYXNlUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnZhbGlkYXRlSW5wdXQocHJvcHMpO1xuXG4gICAgdGhpcy5jbHVzdGVyID0gcHJvcHMuY2x1c3RlciB8fCB0aGlzLmdldERlZmF1bHRDbHVzdGVyKHRoaXMsIHByb3BzLnZwYyk7XG4gICAgdGhpcy5kZXNpcmVkQ291bnQgPSBwcm9wcy5kZXNpcmVkQ291bnQgfHwgMTtcbiAgICBpZiAocHJvcHMudGFza0ltYWdlT3B0aW9ucykge1xuICAgICAgdGhpcy5sb2dEcml2ZXIgPSB0aGlzLmNyZWF0ZUxvZ0RyaXZlcihwcm9wcy50YXNrSW1hZ2VPcHRpb25zLmVuYWJsZUxvZ2dpbmcsIHByb3BzLnRhc2tJbWFnZU9wdGlvbnMubG9nRHJpdmVyKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMubG9hZEJhbGFuY2Vycykge1xuICAgICAgZm9yIChjb25zdCBsYlByb3BzIG9mIHByb3BzLmxvYWRCYWxhbmNlcnMpIHtcbiAgICAgICAgY29uc3QgbGIgPSB0aGlzLmNyZWF0ZUxvYWRCYWxhbmNlcihsYlByb3BzLm5hbWUsIGxiUHJvcHMucHVibGljTG9hZEJhbGFuY2VyKTtcbiAgICAgICAgdGhpcy5sb2FkQmFsYW5jZXJzLnB1c2gobGIpO1xuICAgICAgICBjb25zdCBwcm90b2NvbFR5cGUgPSBuZXcgU2V0PEFwcGxpY2F0aW9uUHJvdG9jb2w+KCk7XG4gICAgICAgIGZvciAoY29uc3QgbGlzdGVuZXJQcm9wcyBvZiBsYlByb3BzLmxpc3RlbmVycykge1xuICAgICAgICAgIGNvbnN0IHByb3RvY29sID0gdGhpcy5jcmVhdGVMaXN0ZW5lclByb3RvY29sKGxpc3RlbmVyUHJvcHMucHJvdG9jb2wsIGxpc3RlbmVyUHJvcHMuY2VydGlmaWNhdGUpO1xuICAgICAgICAgIGlmIChsaXN0ZW5lclByb3BzLmNlcnRpZmljYXRlICE9PSB1bmRlZmluZWQgJiYgcHJvdG9jb2wgIT09IHVuZGVmaW5lZCAmJiBwcm90b2NvbCAhPT0gQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQUykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgSFRUUFMgcHJvdG9jb2wgbXVzdCBiZSB1c2VkIHdoZW4gYSBjZXJ0aWZpY2F0ZSBpcyBnaXZlbicpO1xuICAgICAgICAgIH1cbiAgICAgICAgICBwcm90b2NvbFR5cGUuYWRkKHByb3RvY29sKTtcbiAgICAgICAgICBjb25zdCBsaXN0ZW5lciA9IHRoaXMuY29uZmlnTGlzdGVuZXIocHJvdG9jb2wsIHtcbiAgICAgICAgICAgIGNlcnRpZmljYXRlOiBsaXN0ZW5lclByb3BzLmNlcnRpZmljYXRlLFxuICAgICAgICAgICAgZG9tYWluTmFtZTogbGJQcm9wcy5kb21haW5OYW1lLFxuICAgICAgICAgICAgZG9tYWluWm9uZTogbGJQcm9wcy5kb21haW5ab25lLFxuICAgICAgICAgICAgbGlzdGVuZXJOYW1lOiBsaXN0ZW5lclByb3BzLm5hbWUsXG4gICAgICAgICAgICBsb2FkQmFsYW5jZXI6IGxiLFxuICAgICAgICAgICAgcG9ydDogbGlzdGVuZXJQcm9wcy5wb3J0LFxuICAgICAgICAgIH0pO1xuICAgICAgICAgIHRoaXMubGlzdGVuZXJzLnB1c2gobGlzdGVuZXIpO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGRvbWFpbk5hbWUgPSB0aGlzLmNyZWF0ZURvbWFpbk5hbWUobGIsIGxiUHJvcHMuZG9tYWluTmFtZSwgbGJQcm9wcy5kb21haW5ab25lKTtcbiAgICAgICAgbmV3IENmbk91dHB1dCh0aGlzLCBgTG9hZEJhbGFuY2VyRE5TJHtsYi5ub2RlLmlkfWAsIHsgdmFsdWU6IGxiLmxvYWRCYWxhbmNlckRuc05hbWUgfSk7XG4gICAgICAgIGZvciAoY29uc3QgcHJvdG9jb2wgb2YgcHJvdG9jb2xUeXBlKSB7XG4gICAgICAgICAgbmV3IENmbk91dHB1dCh0aGlzLCBgU2VydmljZVVSTCR7bGIubm9kZS5pZH0ke3Byb3RvY29sLnRvTG93ZXJDYXNlKCl9YCwgeyB2YWx1ZTogcHJvdG9jb2wudG9Mb3dlckNhc2UoKSArICc6Ly8nICsgZG9tYWluTmFtZSB9KTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgLy8gc2V0IHVwIGRlZmF1bHQgbG9hZCBiYWxhbmNlciBhbmQgbGlzdGVuZXIuXG4gICAgICB0aGlzLmxvYWRCYWxhbmNlciA9IHRoaXMubG9hZEJhbGFuY2Vyc1swXTtcbiAgICAgIHRoaXMubGlzdGVuZXIgPSB0aGlzLmxpc3RlbmVyc1swXTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5sb2FkQmFsYW5jZXIgPSB0aGlzLmNyZWF0ZUxvYWRCYWxhbmNlcignTEInKTtcbiAgICAgIGNvbnN0IHByb3RvY29sID0gdGhpcy5jcmVhdGVMaXN0ZW5lclByb3RvY29sKCk7XG4gICAgICB0aGlzLmxpc3RlbmVyID0gdGhpcy5jb25maWdMaXN0ZW5lcihwcm90b2NvbCwge1xuICAgICAgICBsaXN0ZW5lck5hbWU6ICdQdWJsaWNMaXN0ZW5lcicsXG4gICAgICAgIGxvYWRCYWxhbmNlcjogdGhpcy5sb2FkQmFsYW5jZXIsXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IGRvbWFpbk5hbWUgPSB0aGlzLmNyZWF0ZURvbWFpbk5hbWUodGhpcy5sb2FkQmFsYW5jZXIpO1xuXG4gICAgICBuZXcgQ2ZuT3V0cHV0KHRoaXMsICdMb2FkQmFsYW5jZXJETlMnLCB7IHZhbHVlOiB0aGlzLmxvYWRCYWxhbmNlci5sb2FkQmFsYW5jZXJEbnNOYW1lIH0pO1xuICAgICAgbmV3IENmbk91dHB1dCh0aGlzLCAnU2VydmljZVVSTCcsIHsgdmFsdWU6IHByb3RvY29sLnRvTG93ZXJDYXNlKCkgKyAnOi8vJyArIGRvbWFpbk5hbWUgfSk7XG4gICAgfVxuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwcm90ZWN0ZWQgZ2V0RGVmYXVsdENsdXN0ZXIoc2NvcGU6IENvbnN0cnVjdCwgdnBjPzogSVZwYyk6IENsdXN0ZXIge1xuICAgIC8vIG1hZ2ljIHN0cmluZyB0byBhdm9pZCBjb2xsaXNpb24gd2l0aCB1c2VyLWRlZmluZWQgY29uc3RydWN0cy5cbiAgICBjb25zdCBERUZBVUxUX0NMVVNURVJfSUQgPSBgRWNzRGVmYXVsdENsdXN0ZXJNbkwzbU5OWU4ke3ZwYyA/IHZwYy5ub2RlLmlkIDogJyd9YDtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHNjb3BlKTtcbiAgICByZXR1cm4gc3RhY2subm9kZS50cnlGaW5kQ2hpbGQoREVGQVVMVF9DTFVTVEVSX0lEKSBhcyBDbHVzdGVyIHx8IG5ldyBDbHVzdGVyKHN0YWNrLCBERUZBVUxUX0NMVVNURVJfSUQsIHsgdnBjIH0pO1xuICB9XG5cbiAgcHJvdGVjdGVkIGNyZWF0ZUFXU0xvZ0RyaXZlcihwcmVmaXg6IHN0cmluZyk6IEF3c0xvZ0RyaXZlciB7XG4gICAgcmV0dXJuIG5ldyBBd3NMb2dEcml2ZXIoeyBzdHJlYW1QcmVmaXg6IHByZWZpeCB9KTtcbiAgfVxuXG4gIHByb3RlY3RlZCBmaW5kTGlzdGVuZXIobmFtZT86IHN0cmluZyk6IEFwcGxpY2F0aW9uTGlzdGVuZXIge1xuICAgIGlmICghbmFtZSkge1xuICAgICAgcmV0dXJuIHRoaXMubGlzdGVuZXI7XG4gICAgfVxuICAgIGZvciAoY29uc3QgbGlzdGVuZXIgb2YgdGhpcy5saXN0ZW5lcnMpIHtcbiAgICAgIGlmIChsaXN0ZW5lci5ub2RlLmlkID09PSBuYW1lKSB7XG4gICAgICAgIHJldHVybiBsaXN0ZW5lcjtcbiAgICAgIH1cbiAgICB9XG4gICAgdGhyb3cgbmV3IEVycm9yKGBMaXN0ZW5lciAke25hbWV9IGlzIG5vdCBkZWZpbmVkLiBEaWQgeW91IGRlZmluZSBsaXN0ZW5lciB3aXRoIG5hbWUgJHtuYW1lfT9gKTtcbiAgfVxuXG4gIHByb3RlY3RlZCByZWdpc3RlckVDU1RhcmdldHMoc2VydmljZTogQmFzZVNlcnZpY2UsIGNvbnRhaW5lcjogQ29udGFpbmVyRGVmaW5pdGlvbiwgdGFyZ2V0czogQXBwbGljYXRpb25UYXJnZXRQcm9wc1tdKTogQXBwbGljYXRpb25UYXJnZXRHcm91cCB7XG4gICAgZm9yIChjb25zdCB0YXJnZXRQcm9wcyBvZiB0YXJnZXRzKSB7XG4gICAgICBjb25zdCB0YXJnZXRHcm91cCA9IHRoaXMuZmluZExpc3RlbmVyKHRhcmdldFByb3BzLmxpc3RlbmVyKS5hZGRUYXJnZXRzKGBFQ1NUYXJnZXRHcm91cCR7Y29udGFpbmVyLmNvbnRhaW5lck5hbWV9JHt0YXJnZXRQcm9wcy5jb250YWluZXJQb3J0fWAsIHtcbiAgICAgICAgcG9ydDogODAsXG4gICAgICAgIHRhcmdldHM6IFtcbiAgICAgICAgICBzZXJ2aWNlLmxvYWRCYWxhbmNlclRhcmdldCh7XG4gICAgICAgICAgICBjb250YWluZXJOYW1lOiBjb250YWluZXIuY29udGFpbmVyTmFtZSxcbiAgICAgICAgICAgIGNvbnRhaW5lclBvcnQ6IHRhcmdldFByb3BzLmNvbnRhaW5lclBvcnQsXG4gICAgICAgICAgICBwcm90b2NvbDogdGFyZ2V0UHJvcHMucHJvdG9jb2wsXG4gICAgICAgICAgfSksXG4gICAgICAgIF0sXG4gICAgICAgIGhvc3RIZWFkZXI6IHRhcmdldFByb3BzLmhvc3RIZWFkZXIsXG4gICAgICAgIHBhdGhQYXR0ZXJuOiB0YXJnZXRQcm9wcy5wYXRoUGF0dGVybixcbiAgICAgICAgcHJpb3JpdHk6IHRhcmdldFByb3BzLnByaW9yaXR5LFxuICAgICAgfSk7XG4gICAgICB0aGlzLnRhcmdldEdyb3Vwcy5wdXNoKHRhcmdldEdyb3VwKTtcbiAgICB9XG4gICAgaWYgKHRoaXMudGFyZ2V0R3JvdXBzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCBvbmUgdGFyZ2V0IGdyb3VwIHNob3VsZCBiZSBzcGVjaWZpZWQuJyk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLnRhcmdldEdyb3Vwc1swXTtcbiAgfVxuXG4gIHByb3RlY3RlZCBhZGRQb3J0TWFwcGluZ0ZvclRhcmdldHMoY29udGFpbmVyOiBDb250YWluZXJEZWZpbml0aW9uLCB0YXJnZXRzOiBBcHBsaWNhdGlvblRhcmdldFByb3BzW10pIHtcbiAgICBmb3IgKGNvbnN0IHRhcmdldCBvZiB0YXJnZXRzKSB7XG4gICAgICBpZiAoIWNvbnRhaW5lci5maW5kUG9ydE1hcHBpbmcodGFyZ2V0LmNvbnRhaW5lclBvcnQsIHRhcmdldC5wcm90b2NvbCB8fCBQcm90b2NvbC5UQ1ApKSB7XG4gICAgICAgIGNvbnRhaW5lci5hZGRQb3J0TWFwcGluZ3Moe1xuICAgICAgICAgIGNvbnRhaW5lclBvcnQ6IHRhcmdldC5jb250YWluZXJQb3J0LFxuICAgICAgICAgIHByb3RvY29sOiB0YXJnZXQucHJvdG9jb2wsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgbG9nIGRyaXZlciBpZiBsb2dnaW5nIGlzIGVuYWJsZWQuXG4gICAqL1xuICBwcml2YXRlIGNyZWF0ZUxvZ0RyaXZlcihlbmFibGVMb2dnaW5nUHJvcD86IGJvb2xlYW4sIGxvZ0RyaXZlclByb3A/OiBMb2dEcml2ZXIpOiBMb2dEcml2ZXIgfCB1bmRlZmluZWQge1xuICAgIGNvbnN0IGVuYWJsZUxvZ2dpbmcgPSBlbmFibGVMb2dnaW5nUHJvcCA/PyB0cnVlO1xuICAgIGNvbnN0IGxvZ0RyaXZlciA9IGxvZ0RyaXZlclByb3AgPz8gKGVuYWJsZUxvZ2dpbmcgPyB0aGlzLmNyZWF0ZUFXU0xvZ0RyaXZlcih0aGlzLm5vZGUuaWQpIDogdW5kZWZpbmVkKTtcbiAgICByZXR1cm4gbG9nRHJpdmVyO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25maWdMaXN0ZW5lcihwcm90b2NvbDogQXBwbGljYXRpb25Qcm90b2NvbCwgcHJvcHM6IExpc3RlbmVyQ29uZmlnKTogQXBwbGljYXRpb25MaXN0ZW5lciB7XG4gICAgY29uc3QgbGlzdGVuZXIgPSB0aGlzLmNyZWF0ZUxpc3RlbmVyKHByb3BzLmxpc3RlbmVyTmFtZSwgcHJvcHMubG9hZEJhbGFuY2VyLCBwcm90b2NvbCwgcHJvcHMucG9ydCk7XG4gICAgbGV0IGNlcnRpZmljYXRlO1xuICAgIGlmIChwcm90b2NvbCA9PT0gQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQUykge1xuICAgICAgY2VydGlmaWNhdGUgPSB0aGlzLmNyZWF0ZUxpc3RlbmVyQ2VydGlmaWNhdGUocHJvcHMubGlzdGVuZXJOYW1lLCBwcm9wcy5jZXJ0aWZpY2F0ZSwgcHJvcHMuZG9tYWluTmFtZSwgcHJvcHMuZG9tYWluWm9uZSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNlcnRpZmljYXRlID0gdW5kZWZpbmVkO1xuICAgIH1cbiAgICBpZiAoY2VydGlmaWNhdGUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgbGlzdGVuZXIuYWRkQ2VydGlmaWNhdGVBcm5zKGBBcm5zJHtwcm9wcy5saXN0ZW5lck5hbWV9YCwgW2NlcnRpZmljYXRlLmNlcnRpZmljYXRlQXJuXSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGxpc3RlbmVyO1xuICB9XG5cbiAgcHJpdmF0ZSB2YWxpZGF0ZUlucHV0KHByb3BzOiBBcHBsaWNhdGlvbk11bHRpcGxlVGFyZ2V0R3JvdXBzU2VydmljZUJhc2VQcm9wcykge1xuICAgIGlmIChwcm9wcy5jbHVzdGVyICYmIHByb3BzLnZwYykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgY2FuIG9ubHkgc3BlY2lmeSBlaXRoZXIgdnBjIG9yIGNsdXN0ZXIuIEFsdGVybmF0aXZlbHksIHlvdSBjYW4gbGVhdmUgYm90aCBibGFuaycpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5kZXNpcmVkQ291bnQgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5kZXNpcmVkQ291bnQgPCAxKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBtdXN0IHNwZWNpZnkgYSBkZXNpcmVkQ291bnQgZ3JlYXRlciB0aGFuIDAnKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMubG9hZEJhbGFuY2Vycykge1xuICAgICAgaWYgKHByb3BzLmxvYWRCYWxhbmNlcnMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIGxvYWQgYmFsYW5jZXIgbXVzdCBiZSBzcGVjaWZpZWQnKTtcbiAgICAgIH1cbiAgICAgIGZvciAoY29uc3QgbGJQcm9wcyBvZiBwcm9wcy5sb2FkQmFsYW5jZXJzKSB7XG4gICAgICAgIGlmIChsYlByb3BzLmxpc3RlbmVycy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0IGxlYXN0IG9uZSBsaXN0ZW5lciBtdXN0IGJlIHNwZWNpZmllZCcpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVMb2FkQmFsYW5jZXIobmFtZTogc3RyaW5nLCBwdWJsaWNMb2FkQmFsYW5jZXI/OiBib29sZWFuKTogQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIge1xuICAgIGNvbnN0IGludGVybmV0RmFjaW5nID0gcHVibGljTG9hZEJhbGFuY2VyID8/IHRydWU7XG4gICAgY29uc3QgbGJQcm9wcyA9IHtcbiAgICAgIHZwYzogdGhpcy5jbHVzdGVyLnZwYyxcbiAgICAgIGludGVybmV0RmFjaW5nLFxuICAgIH07XG5cbiAgICByZXR1cm4gbmV3IEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyKHRoaXMsIG5hbWUsIGxiUHJvcHMpO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVMaXN0ZW5lclByb3RvY29sKGxpc3RlbmVyUHJvdG9jb2w/OiBBcHBsaWNhdGlvblByb3RvY29sLCBjZXJ0aWZpY2F0ZT86IElDZXJ0aWZpY2F0ZSk6IEFwcGxpY2F0aW9uUHJvdG9jb2wge1xuICAgIHJldHVybiBsaXN0ZW5lclByb3RvY29sID8/IChjZXJ0aWZpY2F0ZSA/IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUFMgOiBBcHBsaWNhdGlvblByb3RvY29sLkhUVFApO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVMaXN0ZW5lckNlcnRpZmljYXRlKGxpc3RlbmVyTmFtZTogc3RyaW5nLCBjZXJ0aWZpY2F0ZT86IElDZXJ0aWZpY2F0ZSwgZG9tYWluTmFtZT86IHN0cmluZywgZG9tYWluWm9uZT86IElIb3N0ZWRab25lKTogSUNlcnRpZmljYXRlIHtcbiAgICBpZiAodHlwZW9mIGRvbWFpbk5hbWUgPT09ICd1bmRlZmluZWQnIHx8IHR5cGVvZiBkb21haW5ab25lID09PSAndW5kZWZpbmVkJykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBIGRvbWFpbiBuYW1lIGFuZCB6b25lIGlzIHJlcXVpcmVkIHdoZW4gdXNpbmcgdGhlIEhUVFBTIHByb3RvY29sJyk7XG4gICAgfVxuXG4gICAgaWYgKGNlcnRpZmljYXRlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybiBjZXJ0aWZpY2F0ZTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIG5ldyBDZXJ0aWZpY2F0ZSh0aGlzLCBgQ2VydGlmaWNhdGUke2xpc3RlbmVyTmFtZX1gLCB7XG4gICAgICAgIGRvbWFpbk5hbWUsXG4gICAgICAgIHZhbGlkYXRpb246IENlcnRpZmljYXRlVmFsaWRhdGlvbi5mcm9tRG5zKGRvbWFpblpvbmUpLFxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVMaXN0ZW5lcihuYW1lOiBzdHJpbmcsIGxiOiBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlciwgcHJvdG9jb2w/OiBBcHBsaWNhdGlvblByb3RvY29sLCBwb3J0PzogbnVtYmVyKTogQXBwbGljYXRpb25MaXN0ZW5lciB7XG4gICAgcmV0dXJuIGxiLmFkZExpc3RlbmVyKG5hbWUsIHtcbiAgICAgIHByb3RvY29sLFxuICAgICAgb3BlbjogdHJ1ZSxcbiAgICAgIHBvcnQsXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZURvbWFpbk5hbWUobG9hZEJhbGFuY2VyOiBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlciwgbmFtZT86IHN0cmluZywgem9uZT86IElIb3N0ZWRab25lKTogc3RyaW5nIHtcbiAgICBsZXQgZG9tYWluTmFtZSA9IGxvYWRCYWxhbmNlci5sb2FkQmFsYW5jZXJEbnNOYW1lO1xuICAgIGlmICh0eXBlb2YgbmFtZSAhPT0gJ3VuZGVmaW5lZCcpIHtcbiAgICAgIGlmICh0eXBlb2Ygem9uZSA9PT0gJ3VuZGVmaW5lZCcpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdBIFJvdXRlNTMgaG9zdGVkIGRvbWFpbiB6b25lIG5hbWUgaXMgcmVxdWlyZWQgdG8gY29uZmlndXJlIHRoZSBzcGVjaWZpZWQgZG9tYWluIG5hbWUnKTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgcmVjb3JkID0gbmV3IEFSZWNvcmQodGhpcywgYEROUyR7bG9hZEJhbGFuY2VyLm5vZGUuaWR9YCwge1xuICAgICAgICB6b25lLFxuICAgICAgICByZWNvcmROYW1lOiBuYW1lLFxuICAgICAgICB0YXJnZXQ6IFJlY29yZFRhcmdldC5mcm9tQWxpYXMobmV3IExvYWRCYWxhbmNlclRhcmdldChsb2FkQmFsYW5jZXIpKSxcbiAgICAgIH0pO1xuXG4gICAgICBkb21haW5OYW1lID0gcmVjb3JkLmRvbWFpbk5hbWU7XG4gICAgfVxuICAgIHJldHVybiBkb21haW5OYW1lO1xuICB9XG59XG5cbi8qKlxuICogUHJvcGVydGllcyB0byBjb25maWd1cmUgYSBsaXN0ZW5lci5cbiAqL1xuaW50ZXJmYWNlIExpc3RlbmVyQ29uZmlnIHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGxpc3RlbmVyXG4gICAqL1xuICByZWFkb25seSBsaXN0ZW5lck5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogTG9hZCBiYWxhbmNlciB0aGUgbGlzdGVuZXIgYXR0YWNoZWQgdG9cbiAgICovXG4gIHJlYWRvbmx5IGxvYWRCYWxhbmNlcjogQXBwbGljYXRpb25Mb2FkQmFsYW5jZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBwb3J0IG9uIHdoaWNoIHRoZSBsaXN0ZW5lciBsaXN0ZW5zIGZvciByZXF1ZXN0cy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZXRlcm1pbmVkIGZyb20gcHJvdG9jb2wgaWYga25vd24uXG4gICAqL1xuICByZWFkb25seSBwb3J0PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBDZXJ0aWZpY2F0ZSBmb3IgdGhlIGxpc3RlbmVyXG4gICAqXG4gICAqIEBkZWZhdWx0IG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGNlcnRpZmljYXRlPzogSUNlcnRpZmljYXRlO1xuXG4gIC8qKlxuICAgKiBUaGUgZG9tYWluIG5hbWUgZm9yIHRoZSBzZXJ2aWNlLCBlLmcuIFwiYXBpLmV4YW1wbGUuY29tLlwiXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZG9tYWluIG5hbWUuXG4gICAqL1xuICByZWFkb25seSBkb21haW5OYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgUm91dGU1MyBob3N0ZWQgem9uZSBmb3IgdGhlIGRvbWFpbiwgZS5nLiBcImV4YW1wbGUuY29tLlwiXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gUm91dGU1MyBob3N0ZWQgZG9tYWluIHpvbmUuXG4gICAqL1xuICByZWFkb25seSBkb21haW5ab25lPzogSUhvc3RlZFpvbmU7XG59XG4iXX0=