import L from "leaflet";
import { GeodesicCore, GeodesicOptions } from "./geodesic-core";
/** detailled information of the current geometry */
export interface Statistics {
    /** Stores the distance for each individual geodesic line */
    distanceArray: number[];
    /** overall distance of all lines */
    totalDistance: number;
    /** number of positions that the geodesic lines are created from */
    points: number;
    /** number vertices that were created during geometry calculation */
    vertices: number;
}
export declare class GeodesicGeometry {
    readonly geodesic: GeodesicCore;
    steps: number;
    constructor(options?: GeodesicOptions);
    /**
     * A geodesic line between `start` and `dest` is created with this recursive function.
     * It calculates the geodesic midpoint between `start` and `dest` and uses this midpoint to call itself again (twice!).
     * The results are then merged into one continuous linestring.
     *
     * The number of resulting vertices (incl. `start` and `dest`) depends on the initial value for `iterations`
     * and can be calculated with: vertices == 1 + 2 ** (initialIterations + 1)
     *
     * As this is an exponential function, be extra careful to limit the initial value for `iterations` (8 results in 513 vertices).
     *
     * @param start start position
     * @param dest destination
     * @param iterations
     * @return resulting linestring
     */
    recursiveMidpoint(start: L.LatLng, dest: L.LatLng, iterations: number): L.LatLng[];
    /**
     * This is the wrapper-function to generate a geodesic line. It's just for future backwards-compatibility
     * if there is another algorithm used to create the actual line.
     *
     * The `steps`-property is used to define the number of resulting vertices of the linestring: vertices == 1 + 2 ** (steps + 1)
     * The value for `steps` is currently limited to 8 (513 vertices) for performance reasons until another algorithm is found.
     *
     * @param start start position
     * @param dest destination
     * @return resulting linestring
     */
    line(start: L.LatLng, dest: L.LatLng): L.LatLng[];
    multiLineString(latlngs: L.LatLng[][]): L.LatLng[][];
    lineString(latlngs: L.LatLng[]): L.LatLng[];
    /**
     *
     * Is much (10x) faster than the previous implementation:
     *
     * ```
     * Benchmark (no split):  splitLine x 459,044 ops/sec ±0.53% (95 runs sampled)
     * Benchmark (split):     splitLine x 42,999 ops/sec ±0.51% (97 runs sampled)
     * ```
     *
     * @param startPosition
     * @param destPosition
     */
    splitLine(startPosition: L.LatLng, destPosition: L.LatLng): L.LatLng[][];
    /**
     * Linestrings of a given multilinestring that cross the antimeridian will be split in two separate linestrings.
     * This function is used to wrap lines around when they cross the antimeridian
     * It iterates over all linestrings and reconstructs the step-by-step if no split is needed.
     * In case the line was split, the linestring ends at the antimeridian and a new linestring is created for the
     * remaining points of the original linestring.
     *
     * @param multilinestring
     * @return another multilinestring where segments crossing the antimeridian are split
     */
    splitMultiLineString(multilinestring: L.LatLng[][]): L.LatLng[][];
    /**
     * Creates a circular (constant radius), closed (1st pos == last pos) geodesic linestring.
     * The number of vertices is calculated with: `vertices == steps + 1` (where 1st == last)
     *
     * @param center
     * @param radius
     * @return resulting linestring
     */
    circle(center: L.LatLng, radius: number): L.LatLng[];
    /**
     * Handles splitting of circles at the antimeridian.
     * @param linestring a linestring that resembles the geodesic circle
     * @return a multilinestring that consist of one or two linestrings
     */
    splitCircle(linestring: L.LatLng[]): L.LatLng[][];
    /**
     * Calculates the distance between two positions on the earths surface
     * @param start 1st position
     * @param dest 2nd position
     * @return the distance in **meters**
     */
    distance(start: L.LatLng, dest: L.LatLng): number;
    multilineDistance(multilinestring: L.LatLng[][]): number[];
    updateStatistics(points: L.LatLng[][], vertices: L.LatLng[][]): Statistics;
}
