from typing import Optional
from selenium import webdriver
from selenium.webdriver.firefox.firefox_binary import FirefoxBinary
from selenium.webdriver.common.proxy import Proxy, ProxyType
from selenium.webdriver.common.keys import Keys
import time
import random

class ExploitBuilder:
    def __init__(
        self, 
        protocol: str, 
        hostname: str, 
        stealth: Optional[bool] = False, 
        firefox_binary_path: Optional[str] = "/opt/firefox/firefox",
        useProxy: Optional[bool] = False,
        proxyAddress: Optional[str] = "127.0.0.1:8080",
    ):
        self.protocol = protocol
        self.hostname = hostname
        binary = FirefoxBinary(firefox_binary_path)
        if useProxy and proxyAddress:
            proxy = Proxy({
                "proxyType": ProxyType.MANUAL,
                "httpProxy": proxyAddress,
                "sslProxy": proxyAddress,
                "noProxy": ''
            })
            proxy.add_to_capabilities(webdriver.DesiredCapabilities.FIREFOX)
            fp = webdriver.FirefoxProfile()
            fp.set_preference("network.proxy.type", 1)
            fp.set_preference("network.proxy.http", proxyAddress.split(":")[0])
            fp.set_preference("network.proxy.http_port", int(proxyAddress.split(":")[1]))
            fp.set_preference('network.proxy.allow_hijacking_localhost', True)
            fp.update_preferences()
            self.driver = webdriver.Firefox(
                firefox_binary=binary,
                proxy=proxy,
                firefox_profile=fp
            )
        else:
            self.driver = webdriver.Firefox(firefox_binary=binary)
        self.stealth = stealth

    def click_by_class(self, button_class: str):
        """click_by_class
        Clicks an element based on CSS class name.

        Parameters:
            button_class (str): The element class name

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element_by_class_name(button_class).click()
        return self

    def click_by_id(self, button_id: str):
        """click_by_id
        Clicks an element based on HTML ID.

        Parameters:
            button_id (str): The element HTML ID

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element_by_id(button_id).click()
        return self

    def dump_cookies(self):
        """dump_cookies
        Dumps the cookies into the terminal.

        Returns:
        exploit_builder.ExploitBuilder"""

        for cookie in self.get_cookies():
            print(cookie)
        return self

    def get(self, endpoint: str):
        """get
        Makes a GET request to the specified endpoint in the browser.

        Parameters:
            endpoint (str): The endpoint to which the browser navigates

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.get(self._format(endpoint))
        return self

    def get_contents_by_id(self, field: str):
        """get_contents
        Get the contents of a specified field, based on the HTML ID

        Parameters:
            field (str): The HTML ID to extract text from

        Returns:
        str: The text in the given field"""

        return self.driver.find_element_by_id(field).text

    def get_contents_by_class(self, field: str):
        """get_contents
        Get the contents of a specified field, based on the CSS class name

        Parameters:
            field (str): The CSS class name to extract text from

        Returns:
        str: The text in the given field"""

        return self.driver.find_element_by_class_name(field).text

    def get_cookie_by_name(self, name: str) -> Optional[dict]:
        """get_cookie_by_name
        Gets a browser cookie by its name.

        Parameters:
            name (str): The name of the cookie

        Returns:
        Optional(dict): The cookie, or None"""

        for cookie in self.get_cookies():
            if cookie["name"] == name:
                return cookie
        return None

    def get_cookies(self):
        """get_cookies
        Get all cookies in the browser for the current page.

        Returns:
        list(dict): A list of cookies in the browser"""
        return self.driver.get_cookies()

    def login(self, endpoint: str, username: str, password: str, username_id: str, password_id: str, submit_id: str):
        """login
        Performs a full login process for a given endpoint. Convenience function.

        Parameters:
            endpoint (str): The location of the login page
            username (str): The username to use
            password (str): The password to use
            username_id (str): The HTML ID of the username field
            password_id (str): The HTML ID of the password field
            submit_id (str): The HTML ID of the submit button

        Returns:
        exploit_builder.ExploitBuilder"""

        (
            self.get(endpoint)
                .type_by_id(username_id, username)
                .type_by_id(password_id, password)
                .click_by_id(submit_id)
        )
        return self

    def send_enter_by_class(self, field: str):
        """send_enter_by_class
        Simulates sending the enter key to a particular element selected by CSS class name.

        Parameters:
            field (str): The CSS Class name of the field to send enter in

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element_by_class_name(field).send_keys(Keys.ENTER)
        return self

    def send_enter_by_id(self, field: str):
        """send_enter_by_id
        Simulates sending the enter key to a particular element selected by HTML ID.

        Parameters:
            field (str): The HTML ID of the field to send enter in

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element_by_id(field).send_keys(Keys.ENTER)
        return self

    def set_cookie(self, name: str, value: str, path: str ="/", secure: bool=False):
        """set_cookie
        Set a cookie in the browser.

        Parameters:
            name (str): The name of the cookie
            value (str): The cookie's value
            path (str): The path to which the cookie belongs
            secure (bool): Whether or not the cookie has the Secure attribute

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.add_cookie({
            "name": name,
            "value": value,
            "path": path,
            "secure": secure
        })
        return self

    def type_by_class(self, field: str, entry: str):
        """type_by_class
        Send text to a particular field selected by CSS class name.

        Parameters:
            field (str): The CSS class name of the field
            entry (str): The text to enter into the field

        Returns:
        exploit_builder.ExploitBuilder"""

        self.wait_for_stealth(entry)
        element = self.driver.find_element_by_class_name(field)
        element.send_keys(entry)
        return self

    def type_by_id(self, field: str, entry: str):
        """type_by_id
        Send text to a particular field selected by HTML ID.

        Parameters:
            field (str): The HTML ID of the field
            entry (str): The text to enter into the field

        Returns:
        exploit_builder.ExploitBuilder"""

        self.wait_for_stealth(entry)
        element = self.driver.find_element_by_id(field)
        element.send_keys(entry)
        return self

    def wait_for_stealth(self, entry: str =""):
        """wait_for_stealth
        Waits for a period of time based on the length of the entry.

        Parameters:
            entry (str): The entry to scale by

        Returns:
        exploit_builder.ExploitBuilder"""

        if self.stealth:
            time.sleep(len(entry) / 5 + random.uniform(0.2, 1))
        return self

    def _format(self, endpoint: str) -> str:
        """_format
        Formats an endpoint into a usable URL with the exploit builder.

        Parameters:
            endpoint (str): The endpoint to format.

        Returns:
        str: The formatted URL"""

        return f"{self.protocol}://{self.hostname}{endpoint}"
